// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Util/hkExtraInplaceLoadUtil.h>

#include <Common/Base/Serialize/Core/hkSerializeCore.h>
#include <Common/Base/Serialize/Format/Tagfile/Detail/hkTagfileDetail.h>
#include <Common/Base/System/Io/Structured/hkStructuredStream.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>

#define DEBUG_LOG_IDENTIFIER "s11n.ExtraInplaceLoadUtil"
#include <Common/Base/System/Log/hkLog.hxx>

#define FAIL_IF( COND, ... ) \
    if( (COND) ) \
    { \
        Log_Warning("IO error"); \
        hff.abort(); \
        return HK_FAILURE; \
    }

namespace
{
    hkReflect::Var extractPointedItem(_In_ const void* self, _In_ const hkReflect::Type* type, hkUint32& countOut)
    {
        if (const hkReflect::PointerType* pointerType = type->asPointer())
        {
            const hkReflect::Detail::PointerImpl* pointerImpl = pointerType->getImpl();
            hkReflect::Var out;
            pointerImpl->getValue(self, pointerType, &out);
            return out;
        }
        else if (const hkReflect::ArrayType* arrayType = type->asArray())
        {
            const hkReflect::Detail::ArrayImpl* arrayImpl = arrayType->getImpl();
            hkReflect::ArrayValue arrayValue;
            arrayImpl->getValue(self, arrayType, &arrayValue);
            countOut = arrayValue.getCount();
            return arrayValue[0];
        }
        else if (const hkReflect::StringType* stringType = type->asString())
        {
            const hkReflect::Detail::StringImpl* stringImpl = stringType->getImpl();
            hkReflect::StringValue sval;
            stringImpl->getValue(self, stringType, &sval);
            return hkReflect::Var(sval.m_value, hkReflect::QualType());
        }
        else
        {
            HK_ERROR(0x69797c98, "Should never get here.");
            return hkReflect::Var();
        }
    }
}

hkResult HK_CALL hkSerialize::ExtraInplaceLoadUtil::unloadAndPrepareForInplaceReload(_Inout_updates_bytes_(bufLen) void* buf, hkUlong bufLen)
{
    HK_ASSERT(0x21ce444e, HK_ENDIAN_BIG == 0, "hkSerialize::ExtraInplaceLoadUtil only supports little-endian platforms.");

    typedef Detail::TagfileTypeSection TagfileTypeSection;
    typedef Detail::TagfileIndexSection TagfileIndexSection;
    typedef Detail::TagfileSection TagfileSection;
    typedef Detail::TagfileRoot TagfileRoot;

    HffMemoryReader hff(buf, bufLen);

    typedef const hkReflect::Type* TypePtr;
    hkArrayView<TypePtr> types;
    if (hff.enter(Detail::TagfileRoot::TAGFILE))
    {
        // Find the DATA section
        // Find the NATIVE_POINTERS section which has the type lookup table
        // Find the ITEMS section which tells us what to destruct
        DLOG_SCOPE("UnloadAndPrepareForInplaceReload");
        hkArrayView<void> data;
        while (hkSerialize::Ident ident = hff.enter())
        {
            switch (ident)
            {
                case TagfileSection::DATA:
                {
                    // Get the data pointer and skip forward.
                    hkIo::ReadBufferView view = hff.view();
                    data = hkArrayViewT::make((void*)view.begin(), (void*)view.end());
                    break;
                }
                case TagfileSection::COMPENDIUM_REFERENCE:
                {
                    // Copy types reference to types and skip forward.
                    hkIo::ReadBufferView view = hff.view();
                    view.skip(sizeof(hkUint64));
                    if (view.remaining() != sizeof(types))
                    {
                        Log_Warning("Invalid COMPENDIUM_REFERENCE");
                        goto fail;
                    }
                    hkMemUtil::memCpyOne(&types, view.access< hkArrayView<TypePtr> >());
                    if (types.getSize() == 0)
                    {
                        Log_Warning("COMPENDIUM_REFERENCE was not initialized. Was this buffer inplace loaded?");
                        goto fail;
                    }
                    break;
                }
                case TagfileSection::TYPES:
                {
                    // Set the types reference and skip forward.
                    while (Ident id = hff.enter(0))
                    {
                        if (id == TagfileTypeSection::NATIVE_POINTERS)
                        {
                            hkIo::ReadBufferView rb = hff.view();
                            types = hkArrayView<TypePtr>((TypePtr*)rb.begin(), (TypePtr*)rb.end());
                        }
                        else if (id == HffBase::IDENT_IO_ERROR)
                        {
                            Log_Warning("IO error");
                            hff.abort();
                            return HK_FAILURE;
                        }
                        else
                        {
                            hff.skipContent();
                        }
                        hff.leave();
                    }
                    break;
                }
                case TagfileSection::INDEX:
                {
                    hkMap<const void*, hkUint32>::Temp itemAddressToIndex;
                    hkArray< hkTuple<void*, hkUint32> >::Temp fixups;
                    hkArrayView<Detail::TagfileItem> items;
                    while (Ident id = hff.enter(0))
                    {
                        switch (id)
                        {
                            case TagfileIndexSection::ITEMS:
                            {
                                // Build the reverse map item address to index.
                                hkIo::ReadBufferView view = hff.view();
                                HK_ASSERT_NO_MSG(0x7495c5d0, (view.remaining() % sizeof(Detail::TagfileItem)) == 0);
                                items = hkArrayViewT::make((Detail::TagfileItem*)view.begin(), (Detail::TagfileItem*)view.end());

                                for (int i = 0; i < items.getSize(); ++i)
                                {
                                    const Detail::TagfileItem& item = items[i];

                                    const void* itemAddress = HK_NULL;
                                    
                                    if (item.isKindType())
                                    {
                                        itemAddress = types[item.getIndex()];
                                    }
                                    else if (item.isKindDecl())
                                    {
                                        const hkReflect::Type* context = types[item.getIndex()];
                                        auto declsArray = hkReflect::TypeDetail::getDeclsArray(context);
                                        HK_ASSERT(0x22c38400, declsArray, "Type must have decls");
                                        hkReflect::Decl decl = declsArray->getDecl(item.count);
                                        itemAddress = decl.getType();
                                    }
                                    else
                                    {
                                        itemAddress = data.ptr(item.offset);
                                    }
                                    hkUint32 itemIndex = i;
                                    itemAddressToIndex.insert(itemAddress, itemIndex);
                                }

                                break;
                            }

                            case TagfileIndexSection::PATCH_DATA:
                            {
                                hkIo::ReadBufferView view = hff.view();
                                HK_ASSERT_NO_MSG(0x20f2cca4, (view.remaining() % sizeof(hkUint32)) == 0);
                                int numInts = hkLosslessCast<int>(view.remaining() / sizeof(hkUint32Le));
                                hkUint32* start = (hkUint32*)view.access<hkUint32>(numInts);

                                int cur = 0;
                                while (cur + 2 < numInts)
                                {
                                    TypeId srcTid = start[cur + 0];
                                    int srcNum = start[cur + 1];
                                    cur += 2;

                                    FAIL_IF(cur + srcNum > numInts, "Corrupt patch index body at index {}", cur);

                                    const hkReflect::Type* srcType = types[srcTid];
                                    DLOG_SCOPE("Type {} ({} instances) //{}", srcTid, srcNum, srcType->getFullName());

                                    for (int i = 0; i < srcNum; ++i)
                                    {
                                        hkUint32 off = start[cur + i];
                                        void* srcAddr = data.ptr(off);

                                        hkUint32 count = 1;
                                        hkReflect::Var pointedVar = extractPointedItem(srcAddr, srcType, count);
                                        hkUint32 index = 0;
                                        if (pointedVar.isValid())
                                        {
                                            const void* itemAddress = pointedVar.getAddress();
                                            index = itemAddressToIndex.getWithDefault(itemAddress, 0);
                                        }

                                        // Restore the index in memory for subsequent reload.
                                        fixups.pushBack(hkTuple<void*, hkUint32>(srcAddr, index));
                                    }

                                    cur += srcNum;
                                }
                                break;
                            }

                            case HffBase::IDENT_NONE:
                            {
                                hff.skipContent();
                                break;
                            }

                            case HffBase::IDENT_IO_ERROR:
                            {
                                FAIL_IF(true, "IO error");
                            }
                        }

                        hff.leave();
                    }

                    // Perform cleanup operations.
                    // 1: Destruct all items that need destruction.
                    for (int i = 1; i < items.getSize(); ++i)
                    {
                        const Detail::TagfileItem& item = items[i];
                        if (item.isKindVar0())
                        {
                            TypeId tid = item.getIndex();
                            const hkReflect::Type* t = types[tid];
                            hkUint32 off = item.offset;
                            void* p = data.ptr(off);
                            hkReflect::TypeDetail::destruct(p, t);
                        }
                    }

                    // 2: Rewrite old indices after destruction so that fixups can happen again on reload.
                    for (int i = 0; i < fixups.getSize(); ++i)
                    {
                        void* addr = fixups[i].m_0;
                        hkUint32 index = fixups[i].m_1;
                        *(hkUint32Le*)addr = index;
                    }

                    break;
                }
                case HffBase::IDENT_IO_ERROR:
                {
                    hff.abort();
                    return HK_FAILURE;
                }
                default:
                {
                    hff.skipContent();
                    break;
                }
            }
            hff.leave();
        }
        hff.leave();

        // Clean up the types in the inplace file buffer.
        hkString::memSet(types.begin(), 0, types.getSize() * sizeof(TypePtr));

        return HK_SUCCESS;
    }
    else if (hff.enter(TagfileRoot::COMPENDIUM))
    {
        //nothing, there are no vars
        return HK_SUCCESS;
    }
    else
    {
        Log_Error("Bad data given to InplaceLoad::unload");
        return HK_FAILURE;
    }
fail:
    hff.abort();
    return HK_FAILURE;
}

hkResult HK_CALL hkSerialize::ExtraInplaceLoadUtil::relocateInplaceLoadBuffer(_In_reads_bytes_(bufLen) const void* oldBuf, _Inout_updates_bytes_(bufLen) void* buf, hkUlong bufLen)
{
    HK_ASSERT(0x142723c9, HK_ENDIAN_BIG == 0, "hkSerialize::ExtraInplaceLoadUtil only supports little-endian platforms.");

    hkLong relocationOffset = hkGetByteOffset(oldBuf, buf);
    hkUlong oldBufAddress = reinterpret_cast<hkUlong>(oldBuf);

    typedef Detail::TagfileTypeSection TagfileTypeSection;
    typedef Detail::TagfileIndexSection TagfileIndexSection;
    typedef Detail::TagfileSection TagfileSection;
    typedef Detail::TagfileRoot TagfileRoot;

    HffMemoryReader hff(buf, bufLen);

    typedef const hkReflect::Type* TypePtr;
    hkArrayView<TypePtr> types;
    if (hff.enter(Detail::TagfileRoot::TAGFILE))
    {
        // Find the DATA section
        // Find the NATIVE_POINTERS section which has the type lookup table
        // Find the ITEMS section which tells us what to destruct
        DLOG_SCOPE("RelocateInplaceLoadBuffer");
        hkArrayView<void> data;
        while (hkSerialize::Ident ident = hff.enter())
        {
            switch (ident)
            {
            case TagfileSection::DATA:
            {
                // Get the data pointer and skip forward.
                hkIo::ReadBufferView view = hff.view();
                data = hkArrayViewT::make((void*)view.begin(), (void*)view.end());
                break;
            }
            case TagfileSection::COMPENDIUM_REFERENCE:
            {
                // Copy types reference to types and skip forward.
                hkIo::ReadBufferView view = hff.view();
                view.skip(sizeof(hkUint64));
                if (view.remaining() != sizeof(types))
                {
                    Log_Warning("Invalid COMPENDIUM_REFERENCE");
                    goto fail;
                }
                hkMemUtil::memCpyOne(&types, view.access< hkArrayView<TypePtr> >());
                if (types.getSize() == 0)
                {
                    Log_Warning("COMPENDIUM_REFERENCE was not initialized. Was this buffer inplace loaded?");
                    goto fail;
                }
                break;
            }
            case TagfileSection::TYPES:
            {
                // Set the types reference and skip forward.
                while (Ident id = hff.enter(0))
                {
                    if (id == TagfileTypeSection::NATIVE_POINTERS)
                    {
                        hkIo::ReadBufferView rb = hff.view();
                        types = hkArrayView<TypePtr>((TypePtr*)rb.begin(), (TypePtr*)rb.end());
                    }
                    else if (id == HffBase::IDENT_IO_ERROR)
                    {
                        Log_Warning("IO error");
                        hff.abort();
                        return HK_FAILURE;
                    }
                    else
                    {
                        hff.skipContent();
                    }
                    hff.leave();
                }
                break;
            }
            case TagfileSection::INDEX:
            {
                while (Ident id = hff.enter(0))
                {
                    switch (id)
                    {

                    case TagfileIndexSection::PATCH_DATA:
                    {
                        hkIo::ReadBufferView view = hff.view();
                        HK_ASSERT_NO_MSG(0x20f2cca4, (view.remaining() % sizeof(hkUint32)) == 0);
                        int numInts = hkLosslessCast<int>(view.remaining() / sizeof(hkUint32Le));
                        hkUint32* start = (hkUint32*)view.access<hkUint32>(numInts);

                        int cur = 0;
                        while (cur + 2 < numInts)
                        {
                            TypeId srcTid = start[cur + 0];
                            int srcNum = start[cur + 1];
                            cur += 2;

                            FAIL_IF(cur + srcNum > numInts, "Corrupt patch index body at index {}", cur);

                            const hkReflect::Type* srcType = types[srcTid];
                            const hkReflect::Detail::Impl* srcImpl = srcType->getImpl();
                            DLOG_SCOPE("Type {} ({} instances) //{}", srcTid, srcNum, srcType->getFullName());

                            for (int i = 0; i < srcNum; ++i)
                            {
                                hkUint32 off = start[cur + i];
                                void* srcAddr = data.ptr(off);

                                hkUint32 count = 1;
                                hkReflect::Var pointedVar = extractPointedItem(srcAddr, srcType, count);
                                if (pointedVar.isValid())
                                {
                                    hkUlong pointedObject = reinterpret_cast<hkUlong>(pointedVar.getAddress());

                                    if (pointedObject >= oldBufAddress && pointedObject < oldBufAddress + bufLen)
                                    {
                                        // Points to a location in the old buffer, relocate.
                                        void* tgtAddr = hkAddByteOffset(pointedVar.getAddress(), relocationOffset);
                                        srcImpl->inplaceFixup(srcAddr, srcType, tgtAddr, pointedVar.getType(), count);
                                    }
                                    // If it's not pointing to the old buffer leave the memory unchanged, cases are:
                                    // - pointing to something external that was allocated during usage of the data, the pointer is still valid.
                                    // - pointing to something in the new buffer because "self" behaves like a relArray, the pointer is still valid.
                                    // - pointing to something in Havok like a type, the pointer is still valid.
                                }
                            }

                            cur += srcNum;
                        }
                        break;
                    }

                    case TagfileIndexSection::ITEMS:
                    case HffBase::IDENT_NONE:
                    {
                        hff.skipContent();
                        break;
                    }

                    case HffBase::IDENT_IO_ERROR:
                    {
                        FAIL_IF(true, "IO error");
                    }
                    }

                    hff.leave();
                }

                break;
            }
            case HffBase::IDENT_IO_ERROR:
            {
                hff.abort();
                return HK_FAILURE;
            }
            default:
            {
                hff.skipContent();
                break;
            }
            }
            hff.leave();
        }
        hff.leave();

        return HK_SUCCESS;
    }
    else if (hff.enter(TagfileRoot::COMPENDIUM))
    {
        //nothing, there are no vars
        return HK_SUCCESS;
    }
    else
    {
        Log_Error("Bad data given to InplaceLoad::unload");
        return HK_FAILURE;
    }
fail:
    hff.abort();
    return HK_FAILURE;
}

#include <Common/Base/Container/PointerMap/hkMap.hxx>
template class hkMapBase<const void*, hkUint32>;
template class hkMap<const void*, hkUint32>;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
