// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Util/Xml/hkXmlObjectWriter.h>
#include <Common/Base/System/Io/OArchive/hkOArchive.h>
#include <Common/Base/System/Io/Writer/hkStreamWriter.h>

static const unsigned char s_base64write_bin2ascii[64] =
{
    'A','B','C','D','E','F','G','H','I','J','K','L','M',
    'N','O','P','Q','R','S','T','U','V','W','X','Y','Z',
    'a','b','c','d','e','f','g','h','i','j','k','l','m',
    'n','o','p','q','r','s','t','u','v','w','x','y','z',
    '0','1','2','3','4','5','6','7','8','9','+','/'
};

int HK_CALL hkXmlObjectWriter::getBase64EncodedLength(int inputLength, bool insertLineBreaks )
{
    int outputLength = (inputLength + 2) / 3 * 4;

    if (insertLineBreaks)
    {
        outputLength += outputLength / 76;
    }

    return outputLength;
}


void HK_CALL hkXmlObjectWriter::base64write(hkArrayView<char> out, hkArrayView<const char> in, bool insertLineBreaks /*= true */)
{
    HK_ASSERT(0xac259, out.getSize() == getBase64EncodedLength(in.getSize(), insertLineBreaks), "Output array size does not match required size");

    int offsetIn = 0;
    int offsetOut = 0;

    int blocksTillNewline = 19; // 19*4 = 76 chars per line
    while (offsetIn < in.getSize())
    {
        unsigned char ibuf[3] = { 0 };

        int ibuflen = hkMath::min2(in.getSize() - offsetIn, 3);

        for (int i = 0; i < ibuflen; ++i)
        {
            ibuf[i] = in[offsetIn++];
        }

        unsigned char obuf[4];
        obuf[0] = s_base64write_bin2ascii[(ibuf[0] >> 2)];
        obuf[1] = s_base64write_bin2ascii[((ibuf[0] << 4) & 0x30) | (ibuf[1] >> 4)];
        obuf[2] = s_base64write_bin2ascii[((ibuf[1] << 2) & 0x3c) | (ibuf[2] >> 6)];
        obuf[3] = s_base64write_bin2ascii[(ibuf[2] & 0x3f)];

        if (ibuflen >= 3)
        {
            out[offsetOut++] = obuf[0];
            out[offsetOut++] = obuf[1];
            out[offsetOut++] = obuf[2];
            out[offsetOut++] = obuf[3];
        }
        else // need to pad up to 4
        {
            switch (ibuflen)
            {
            case 1:
                out[offsetOut++] = obuf[0];
                out[offsetOut++] = obuf[1];
                out[offsetOut++] = '=';
                out[offsetOut++] = '=';
                break;
            case 2:
                out[offsetOut++] = obuf[0];
                out[offsetOut++] = obuf[1];
                out[offsetOut++] = obuf[2];
                out[offsetOut++] = '=';
                break;
            }
        }

        if (--blocksTillNewline == 0)
        {
            if (insertLineBreaks)
            {
                out[offsetOut++] = '\n';
            }
            blocksTillNewline = 19;
        }
    }
}


// save raw data with base64 encoding.
// see rfc2045 Section 6.8 for a description of the format.
hkResult hkXmlObjectWriter::base64write(_Inout_ hkStreamWriter* sw, _In_reads_bytes_(len) const void* buf, int len, bool insertLineBreaks)
{
    const char* cur = static_cast<const char*>(buf);

    hkOArchive oa(sw);

    int blocksTillNewline = 19; // 19*4 = 76 chars per line
    while( len > 0 )
    {
        unsigned char ibuf[3] = { 0 };
        int ibuflen = len >= 3 ? 3 : len;
        hkString::memCpy( ibuf, cur, ibuflen );
        cur += ibuflen;
        len -= ibuflen;
        unsigned char obuf[4];
        obuf[0] = s_base64write_bin2ascii[(ibuf[0] >> 2)];
        obuf[1] = s_base64write_bin2ascii[((ibuf[0] << 4) & 0x30) | (ibuf[1] >> 4 )];
        obuf[2] = s_base64write_bin2ascii[((ibuf[1] << 2) & 0x3c) | (ibuf[2] >> 6 )];
        obuf[3] = s_base64write_bin2ascii[(ibuf[2] & 0x3f)];

        if( ibuflen >= 3 )
        {
            oa.writeRaw(obuf, 4);
        }
        else // need to pad up to 4
        {
            switch( ibuflen )
            {
                case 1:
                    obuf[2] = obuf[3] = '=';
                    oa.writeRaw(obuf, 4);
                    break;
                case 2:
                    obuf[3] = '=';
                    oa.writeRaw(obuf, 4);
            }
            break;
        }

        if( --blocksTillNewline == 0)
        {
            if (insertLineBreaks)
            {
                oa.writeRaw("\n", 1);
            }
            blocksTillNewline = 19;
        }

        if( sw->isOk() == false )
        {
            return HK_FAILURE;
        }
    }
    return HK_SUCCESS;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
