// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/OStream/hkOStream.h>
#include <Common/Base/Container/String/hkUtf8.h>

#ifndef HK_JSON_PP_STRINGIFIED_AS_STRING
#   if ( defined(HK_PLATFORM_DURANGO) || defined(HK_PLATFORM_WINRT) ) && defined(__cplusplus_winrt)
#       define HK_JSON_PP_STRINGIFIED_AS_STRING 1
#   else
#       define HK_JSON_PP_STRINGIFIED_AS_STRING 0
#   endif
#endif

/// JSON node.
struct HK_EXPORT_COMMON hkJsonNode
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_SERIALIZE, hkJsonNode );

    /// Value type.
    enum Type { Invalid, Object, Array, Boolean, String, Number, Null, Element };

    /// Element type.
    struct ElementType
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_SERIALIZE, ElementType );

        const hkJsonNode* m_key;
        const hkJsonNode* m_value;
        const hkJsonNode* m_next;
    };

    /// Compound type.
    struct CompoundType
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_SERIALIZE, CompoundType );

        const hkJsonNode* m_first;
        int m_count;
    };

    /// Children iterator.
    struct ChildrenIterator
    {
        HK_ALWAYS_INLINE ChildrenIterator( const hkJsonNode* compoundNode ) : m_element( compoundNode && compoundNode->isCompound() ? compoundNode->m_compound.m_first : HK_NULL ) {}
        HK_ALWAYS_INLINE bool operator!=( const ChildrenIterator& other ) const { return other.m_element != m_element; }
        HK_ALWAYS_INLINE const hkJsonNode* operator*() const { return m_element ? m_element->m_element.m_value : HK_NULL; }
        HK_ALWAYS_INLINE ChildrenIterator& operator++() { m_element = m_element ? m_element->m_element.m_next : HK_NULL; return *this; }
        HK_ALWAYS_INLINE ChildrenIterator begin() const { return *this; }
        HK_ALWAYS_INLINE ChildrenIterator end() const { return ChildrenIterator( HK_NULL ); }

        const hkJsonNode* m_element;
    };

    /// Key-Value iterator.
    struct KeyValueIterator
    {
        HK_ALWAYS_INLINE KeyValueIterator( const hkJsonNode* objectNode ) : m_element( objectNode && objectNode->m_type == hkJsonNode::Object ? objectNode->m_compound.m_first : HK_NULL ) {}
        HK_ALWAYS_INLINE bool operator!=( const KeyValueIterator& other ) const { return other.m_element != m_element; }
        HK_ALWAYS_INLINE const hkJsonNode* operator*() const { return m_element; }
        HK_ALWAYS_INLINE KeyValueIterator& operator++() { m_element = m_element ? m_element->m_element.m_next : HK_NULL; return *this; }
        HK_ALWAYS_INLINE KeyValueIterator begin() const { return *this; }
        HK_ALWAYS_INLINE KeyValueIterator end() const { return KeyValueIterator( HK_NULL ); }

        const hkJsonNode* m_element;
    };

    /// Constructor.
    HK_ALWAYS_INLINE hkJsonNode( Type type = Invalid ) : m_type( type ) {}

    /// Return true is this value is an array or an object.
    HK_ALWAYS_INLINE bool isCompound() const { return m_type == Array || m_type == Object; }

    /// Return true is this value is an array or an object.
    HK_ALWAYS_INLINE bool isValid() const { return m_type != Invalid; }

    /// Return true is this node has the given value if it is a string and an element.
    HK_ALWAYS_INLINE bool is( const char* text ) const { return select( text ) != HK_NULL; }

    /// Return the key if the node is an element.
    HK_ALWAYS_INLINE const hkJsonNode* key() const { return m_type == Element ? m_element.m_key : HK_NULL; }

    /// Return the value if the node is an element.
    HK_ALWAYS_INLINE const hkJsonNode* value() const { return m_type == Element ? m_element.m_value : HK_NULL; }

    /// Get children iterator.
    HK_ALWAYS_INLINE ChildrenIterator children() const { return ChildrenIterator( this ); }

    /// Get key-value iterator.
    HK_ALWAYS_INLINE KeyValueIterator keyValues() const { return KeyValueIterator( this ); }

    /// Select a child using this value as root.
    /// Path uses '/' to separate keys, zero-based integers are allowed to index both arrays and objects elements.
    _Ret_maybenull_ const hkJsonNode* select(_In_z_ const char* path) const;

    /// Find a pair value in an object by its key.
    _Ret_maybenull_ const hkJsonNode* findValueByKey(_In_z_ const char* key) const;

    /// Find a element or a pair value in an array or object by its index.
    _Ret_maybenull_ const hkJsonNode* findKeyByIndex(int index) const;

    /// Find a element or a pair value in an array or object by its index.
    _Ret_maybenull_ const hkJsonNode* findValueByIndex( int index ) const;

    #if HK_JSON_PP_STRINGIFIED_AS_STRING
    /// Read value from path.
    /// Returns true if the value was found, false otherwise.
    template <typename T>
    HK_INLINE bool  read( Platform::String ^ path, T& valueOut ) const
    {
        const hkUtf8::Utf8FromWide asUtf8( path->Begin() );
        const hkJsonNode* node = select( asUtf8.cString() );
        if ( node ) { node->read( valueOut ); return true; }
        else return false;
    }
    #else
    /// Read value from path.
    /// Returns true if the value was found, false otherwise.
    template <typename T>
    HK_INLINE bool read(_In_z_ const char* path, T& valueOut) const { const hkJsonNode* node = select(path); if (node) { node->read(valueOut); return true; } else return false; }

    /// Read value from path.
    /// Returns true if the value was found, false otherwise.
    template <typename T>
    HK_INLINE bool readWithDefault( const char* path, T& valueOut, const T& defaultValue ) const { valueOut = defaultValue; return read( path, valueOut ); }

    /// Read value from path.
    /// Returns the value if found, defaultValue otherwise.
    template <typename T>
    HK_INLINE T readWithDefault( const char* path, const T& defaultValue ) const { T valueOut; if ( read( path, valueOut ) ) return valueOut; else return defaultValue; }
    #endif

    /// Read a Boolean value.
    /// Returns true if succeeded false otherwise.
    HK_INLINE   bool read( bool& output ) const { if ( m_type == Boolean ) { output = m_boolean; return true; } return false; }

    /// Read a Boolean value.
    /// Returns true if succeeded false otherwise.
    HK_INLINE   bool read( hkBool& output ) const { if ( m_type == Boolean ) { output = m_boolean; return true; } return false; }

    /// Read an hkReal value.
    /// Returns true if succeeded false otherwise.
    HK_INLINE   bool read( hkReal& output ) const { if ( m_type == Number ) { output = m_number; return true; } return false; }

    /// Read an integer value from a number rounded to nearest.
    /// Returns true if succeeded false otherwise.
    HK_INLINE   bool read( int& output ) const { if ( m_type == Number ) { output = m_number < 0 ? int( m_number - hkReal( 0.5 ) ) : int( m_number + hkReal( 0.5 ) ); return true; } return false; }

    /// Read an unsigned integer value from a number rounded to nearest.
    /// Returns true if succeeded false otherwise.
    HK_INLINE   bool read( unsigned& output ) const { if ( m_type == Number ) { output = m_number < 0 ? 0 : unsigned( m_number + hkReal( 0.5 ) ); return true; } return false; }

    /// Read a string.
    /// Returns true if succeeded false otherwise.
    HK_INLINE   bool read( const char*& output ) const { if ( m_type == String ) { output = m_string; return true; } else if ( m_type == Null ) { output = HK_NULL; return true; } else return false; }

    /// Read an hkVector4 from an array of up to 4 numbers (undefined elements are set to zero).
    /// Returns true if succeeded false otherwise.
    bool read( hkVector4& output, int minNumElements = 1 ) const;

    /// Read an hkQuaternion from an array of number of 4 elements interpreted as axis / angle.
    /// Returns true if succeeded false otherwise.
    bool read( hkQuaternion& output ) const;

    /// Read a row major 3x3 matrix from an array of arrays of 4 numbers.
    /// Returns true if succeeded false otherwise.
    bool read( hkMatrix3& output ) const;

    /// Read a row major 4x4 matrix from an array of arrays of 4 numbers.
    /// Returns true if succeeded false otherwise.
    bool read( hkMatrix4& output ) const;

    /// Read a transform from a column major 4x4 matrix from an array of arrays of 4 numbers.
    /// Returns true if succeeded false otherwise.
    bool read( hkTransform& output ) const;

    /// Read an hkAabb from an array of two vector 3 (mins and maxs).
    /// Returns true if succeeded false otherwise.
    bool read( class hkAabb& output ) const;

    /// Read an hkUuid.
    /// Returns true if succeeded false otherwise.
    bool read( class hkUuid& output ) const;

    /// Read a byte array encoded in base64.
    bool readBase64( hkArray<hkUint8>& bytes ) const;

    /// Read a value encoded in base64.
    template <typename T>
    HK_INLINE bool readBase64(_Inout_ T* value) const
    {
        hkArray<hkUint8> bytes; bytes.reserve( sizeof( T ) );
        if ( readBase64( bytes ) )
        {
            if ( bytes.getSize() == sizeof( T ) )
            {
                hkString::memCpy( value, bytes.begin(), sizeof( T ) );
                return true;
            }
        }
        return false;
    }

    /// Read an fixed size array of values, returns true if succeeded false otherwise.
    template <typename T>
    bool readArray( _Inout_updates_bytes_(count) T* output, int count ) const
    {
        if ( isCompound() && m_compound.m_count >= count )
        {
            const hkJsonNode* element = m_compound.m_first;
            for ( int i = 0; i < count; ++i, element = element->m_element.m_next )
            {
                if ( !element->m_element.m_value->read( output[ i ] ) ) return false;
            }
            return true;
        }
        return false;
    }

    /// Read an array of values, returns true if succeeded false otherwise.
    template <typename T>
    bool read( hkArray<T>& output ) const
    {
        if ( isCompound() )
        {
            output.setSize( m_compound.m_count );
            const hkJsonNode* element = m_compound.m_first;
            for ( int i = 0; i < m_compound.m_count; ++i, element = element->m_element.m_next )
            {
                if ( !element->m_element.m_value->read( output[ i ] ) ) return false;
            }
            return true;
        }
        return false;
    }

    /// Read a value using user-defined read function:
    /// bool read(const hkJSON::Value* value).
    template <typename T> bool read( T& output ) const { return output.read( this ); }

    /// Assign a value to a pointer.
    template <typename T> bool read(_Inout_ T* output) const { return read(*output); }

    /// Convert this node to a string.
    void toString( hkStringBuf& stringBuffer ) const;

    /// Type.
    Type    m_type;

    /// Value.
    union
    {
        /// Object or Array.
        CompoundType m_compound;

        /// Object or array element.
        ElementType m_element;

        /// Zero-terminated string.
        const char* m_string;

        /// Boolean.
        bool m_boolean;

        /// Number.
        hkReal m_number;
    };
};

/// JSON document.
/// Implements 'ECMA-404 The JSON Data Interchange Standard' http://www.json.org.
/// Supports JSON5 extensions ( http://json5.org/ ):
/// - Line comments.
/// - Block comments.
/// - Singly quoted strings.
/// - Trailing comma for objects and arrays.
/// - Multi-line strings separated by backslashes (\).
/// - Numbers can begin or end with a (leading or trailing) decimal point.
/// - Numbers can begin with an explicit plus (+) sign.
/// - Object's keys can be identifiers instead of strings.
struct HK_EXPORT_COMMON hkJsonDocument : public hkReferencedObject
{
    HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_SERIALIZE);

    /// Constructor.
    hkJsonDocument() { clear(); }

    /// Virtual destructor.
    virtual ~hkJsonDocument() {}

    /// Clear the document.
    void clear();

    /// Parse a JSON formatted string, return the root node of the document.
    /// Setting mergeStrings to true reduce memory usage by merging duplicated strings but increase parsing time.
    /// Note that the returned value is always a valid pointer but might be a value of type Value::Invalid if the parsing failed.
    _Ret_notnull_ const hkJsonNode* parse(_In_z_ const char* text, bool mergeStrings = false);

    /// Parse a JSON formatted stream, return the root node of the document.
    /// Setting mergeStrings to true reduce memory usage by merging duplicated strings but increase parsing time.
    /// Note that the returned value is always a valid pointer but might be a value of type Value::Invalid if the parsing failed.
    _Ret_notnull_ const hkJsonNode* parse( hkIstream& stream, bool mergeStrings = false );

    /// Return the root node of the document.
    _Ret_notnull_ const hkJsonNode* root() const { return &m_nodes.back(); }

    /// Set this document from an existing JSON node.
    /// Note that the node must come from another document.
    bool                read( const hkJsonNode* json );

    /// Write this document to a JSON stream.
    void                write( struct hkJsonOstream& json ) const;

    int                     m_root;     ///< Root node index.
    hkArray<hkJsonNode>     m_nodes;    ///< Nodes.
    hkArray<char>           m_strings;  ///< Strings.
};

/// JSON formatted output stream.
/// - Automatically add key value separators (':').
/// - Automatically add commas.
struct HK_EXPORT_COMMON hkJsonOstream
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_SERIALIZE, hkJsonOstream );

    /// States.
    enum State { FIRST_OBJ_KEY, OBJ_KEY, OBJ_VALUE, FIRST_ARR_VALUE, ARR_VALUE };

    /// Constructor.
    HK_INLINE   hkJsonOstream( hkOstream& stream ) : m_stream( stream ), m_level( 0 ), m_holdNewLines( false ) {}

    /// Destructor.
    HK_INLINE   ~hkJsonOstream() { if ( m_states.getSize() ) { HK_ERROR( 0x210DB978, "Invalid state, unmatched begin() and end() calls." ); } }

    /// Write a string value or key.
    HK_INLINE void write( _In_opt_z_ const char* value )
    {
        updateState( true );
        if ( value )
        {
            m_stream << '"';
            writeEscapedString( value );
            m_stream << '"';
        }
        else
        {
            m_stream << "null";
        }
    }

    /// Write a integer value.
    HK_INLINE void  write( signed char value ) { updateState(); m_stream << value; }

    /// Write a integer value.
    HK_INLINE void  write( unsigned char value ) { updateState(); m_stream << value; }

    /// Write a integer value.
    HK_INLINE void  write( int value ) { updateState(); m_stream << value; }

    /// Write a unsigned integer value.
    HK_INLINE void  write( unsigned value ) { updateState(); m_stream << value; }

    /// Write a Boolean value.
    HK_INLINE void  write( bool value ) { updateState(); m_stream << ( value ? "true" : "false" ); }

    /// Write a Boolean value.
    HK_INLINE void  write( hkBool value ) { updateState(); m_stream << ( value ? "true" : "false" ); }

    /// Write a hkSimdReal value.
    HK_INLINE void  write( const hkSimdReal& value ) { write( value.getReal() ); }

    /// Write a float value.
    HK_INLINE void  write( float value ) { write( (double)value ); }

    /// Write a double value.
    void    write( double value );

    /// Write an hkVector4 value.
    void    write( const hkVector4& value );

    /// Write an hkQuaternion value.
    void    write( const hkQuaternion& value );

    /// Write an hkMatrix4 value.
    void    write( const hkMatrix4& value );

    /// Write an hkTransform value.
    void    write( const hkTransform& value );

    /// Write an hkAabb value.
    void    write( const class hkAabb& value );

    /// Write an hkUuid value.
    void    write( const class hkUuid& value );

    /// Write a user-defined value.
    /// void write(const hkJsonOstream& json ).
    template <typename T>
    HK_INLINE void      write( const T& value ) { beginObject(); value.write( *this ); end(); }

    /// Write a user-defined value pointer.
    /// void write(const hkJsonOstream& json ).
    template <typename T>
    HK_INLINE void      write( const T* value ) { if ( value ) { beginObject(); value->write( *this ); end(); } else write( "null" ); }

    #if HK_JSON_PP_STRINGIFIED_AS_STRING
    /// Write a key value pair.
    template <typename T>
    inline void         writePair( Platform::String ^ key, const T& value ) { const hkUtf8::Utf8FromWide hasUtf8( key->Begin() ); write( hasUtf8.cString() ); write( value ); }
    #else
    /// Write a key value pair.
    template <typename T>
    HK_INLINE void      writePair(_In_z_ const char* key, const T& value) { write(key); write(value); }
    #endif

    /// Write an array of values.
    template <typename T>
    HK_INLINE void      write( const hkArrayView<T>& values ) { beginArray(); for ( int i = 0; i < values.getSize(); ++i ) write( values[ i ] ); end(); }

    /// Write an object using base64 encoding.
    template <typename T>
    HK_INLINE void      writeBase64(_In_ const T* value) { writeBase64(value, (int) sizeof(T)); }

    /// Write an array using base64 encoding.
    template <typename T>
    HK_INLINE void      writeBase64( const hkArrayView<T>& values ) { writeBase64( values.begin(), int( values.getSize() * sizeof( T ) ) ); }

    /// Write an object using base64 encoding.
    void                writeBase64(_In_reads_bytes_(size) const void* value, int size);

    /// Write an existing node.
    void write( const hkJsonNode* node );

    /// Write a raw value, returns the underlying stream.
    HK_INLINE hkOstream&    rawValue() { updateState(); return m_stream; }

    /// Begin a object value, must be matched with a call to end().
    void beginObject();

    /// Begin a array value, must be matched with a call to end().
    void beginArray();

    /// Close a object or array value.
    void end();

protected:

    /// Update internal state before adding a value.
    void updateState( bool asString = false );

    /// Add 'm_level' tabulations.
    void addTabs();

    /// Write an escaped string.
    void writeEscapedString( _In_z_ const char* str );

    hkArray<State>  m_states;       ///< States stack.
    hkOstream&      m_stream;       ///< Underlying stream.
    int             m_level;        ///< Hierarchy level.
    bool            m_holdNewLines; ///< Prevent new-line insertions.

};

//
// Helpers macros.
//

/// Write a simple member field (strips the 'm_' prefix from the name) into a JSON stream.
#define HKJSON_WRITE_MEMBER( _json_stream_, _member_name_ ) (_json_stream_).writePair( ((#_member_name_)+2), _member_name_ )

/// Read a simple member field (strips the 'm_' prefix from the name) from a JSON node.
#define HKJSON_READ_MEMBER( _json_node_, _member_name_ ) (_json_node_)->read( ((#_member_name_)+2), _member_name_ )

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
