// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Util/Json/hkJson.h>

#include <Common/Base/Container/String/hkStringBuf.h>

#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/Base/Types/Uuid/hkUuid.h>
#include <Common/Base/Container/PointerMap/hkPointerMap.h>

//
namespace hkJSON_Internals
{
    //
    enum SingletonValueIndex
    {
        INVALID_INDEX = 0,
        NULL_INDEX = 1,
        TRUE_INDEX = 2,
        FALSE_INDEX = 3
    };

    //
    struct Context
    {
        const char*         m_stream;
        hkJsonDocument*     m_document;

        bool                        m_mergeStrings;
        hkPointerMap<hkUint32, int> m_dictionary;

        static HK_INLINE int getIndexFromPointer( const char*& pointer )
        {
            return int( *(const hkUlong*)&pointer );
        }

        int mergeLastString( int baseStringSize, bool merge )
        {
            if ( m_mergeStrings || merge )
            {
                const int   stringIndex = getIndexFromPointer( m_document->m_nodes.back().m_string );
                const char* stringBase = m_document->m_strings.begin() + stringIndex;
                const char* stringPtr = stringBase;

                const hkUint32  prime = 16777619;
                hkUint32        hash = 2166136261U;
                while ( stringPtr[ 0 ] )
                {
                    hash = ( hash ^ hkUint32( *stringPtr++ ) ) * prime;
                }

                const int   existingIndex = m_dictionary.getWithDefault( hash, 0 );
                if ( existingIndex && hkString::strCmp( stringBase, m_document->m_strings.begin() + getIndexFromPointer( m_document->m_nodes[ existingIndex ].m_string ) ) == 0 )
                {
                    m_document->m_strings.setSize( baseStringSize );
                    m_document->m_nodes.popBack();
                    return existingIndex;
                }
                else
                {
                    m_dictionary.insert( hash, m_document->m_nodes.getSize() - 1 );
                }
            }
            return m_document->m_nodes.getSize() - 1;
        }
    };

    //
    HK_INLINE bool isAlpha( char c ) { return   ( c >= 'a' && c <= 'z' ) || ( c >= 'A' && c <= 'Z' ) || c == '_'; }

    //
    HK_INLINE bool isDecimal( char c ) { return c >= '0' && c <= '9'; }

    //
    HK_INLINE void skipSpaces( Context& ctx )
    {
        for ( ;;)
        {
            // Character spaces.
            while ( ctx.m_stream[ 0 ] > 0 && ctx.m_stream[ 0 ] <= ' ' ) ctx.m_stream++;

            // Comments.
            if ( ctx.m_stream[ 0 ] == '/' )
            {
                if ( ctx.m_stream[ 1 ] == '/' )
                {
                    // Line comment.
                    ctx.m_stream += 2;
                    do { ctx.m_stream++; } while ( ctx.m_stream[ 0 ] != 0 && ctx.m_stream[ 0 ] != '\n' );
                }
                else if ( ctx.m_stream[ 1 ] == '*' )
                {
                    // Block comment.
                    ctx.m_stream += 2;
                    do { ctx.m_stream++; } while ( ctx.m_stream[ 0 ] != 0 && !( ctx.m_stream[ 0 ] == '*' && ctx.m_stream[ 1 ] == '/' ) );
                    if ( ctx.m_stream[ 0 ] == '*' ) ctx.m_stream += 2;
                }
                else break;
            }
            else break;
        }
    }

    //
    HK_INLINE bool skipChar( Context& ctx, char c )
    {
        if ( ctx.m_stream[ 0 ] == c )
        {
            ctx.m_stream++;
            skipSpaces( ctx );
            return true;
        }
        else
        {
            skipSpaces( ctx );
            return false;
        }
    }

    //
    template <typename T>
    HK_INLINE void setPointerAsIndex( int index, const T*& pointer )
    {
        *(hkUlong*)( &pointer ) = hkUlong( index );
    }

    //
    template <typename T>
    HK_INLINE void convertIndexToPointer(_In_ const T* base, const T*& pointer)
    {
        if ( pointer )
        {
            pointer = &base[ int( *(const hkUlong*)&pointer ) ];
        }
    }

    //
    void prepareDocument( hkJsonDocument& document )
    {
        document.clear();

        // Append singletons nodes.
        document.m_nodes.pushBack( hkJsonNode( hkJsonNode::Invalid ) );
        document.m_nodes.pushBack( hkJsonNode( hkJsonNode::Null ) );
        document.m_nodes.pushBack( hkJsonNode( hkJsonNode::Boolean ) ); document.m_nodes.back().m_boolean = true;
        document.m_nodes.pushBack( hkJsonNode( hkJsonNode::Boolean ) ); document.m_nodes.back().m_boolean = false;

        // Prepare strings.
        document.m_strings.pushBack( 0 );
    }

    //
    int parseValue( Context& ctx );
    int parseString( Context& ctx, bool merge );
    int parseIdentifier( Context& ctx, bool merge );

    //
    int parseKey( Context& ctx )
    {
        if ( ctx.m_stream[ 0 ] == '"' )
            return parseString( ctx, true );
        else if ( isAlpha( ctx.m_stream[ 0 ] ) )
            return parseIdentifier( ctx, true );
        else
            return INVALID_INDEX;
    }

    //
    int parseObjectOrArray( Context& ctx )
    {
        const hkJsonNode::Type type = ctx.m_stream[ 0 ] == '[' ? hkJsonNode::Array : ctx.m_stream[ 0 ] == '{' ? hkJsonNode::Object : hkJsonNode::Invalid;
        if ( type != hkJsonNode::Invalid )
        {
            const char* blockChars = type == hkJsonNode::Array ? "[]" : "{}";

            if ( !skipChar( ctx, blockChars[ 0 ] ) ) return INVALID_INDEX;

            hkJsonNode node( type );

            node.m_compound.m_first = HK_NULL;
            node.m_compound.m_count = 0;

            if ( ctx.m_stream[ 0 ] != blockChars[ 1 ] )
            {
                int lastPair = INVALID_INDEX;
                for ( ; ctx.m_stream[ 0 ] != blockChars[ 1 ];)
                {
                    int k = INVALID_INDEX;
                    if ( type == hkJsonNode::Object )
                    {
                        k = parseKey( ctx ); if ( !k ) return INVALID_INDEX;
                        skipSpaces( ctx );
                        if ( !skipChar( ctx, ':' ) ) return INVALID_INDEX;
                    }

                    const int v = parseValue( ctx ); if ( !v ) return INVALID_INDEX;

                    const int   pairIndex = ctx.m_document->m_nodes.getSize();
                    hkJsonNode  pair( hkJsonNode::Element );
                    setPointerAsIndex( k, pair.m_element.m_key );
                    setPointerAsIndex( v, pair.m_element.m_value );
                    pair.m_element.m_next = HK_NULL;
                    ctx.m_document->m_nodes.pushBack( pair );

                    if ( lastPair == INVALID_INDEX )
                        setPointerAsIndex( pairIndex, node.m_compound.m_first );
                    else
                        setPointerAsIndex( pairIndex, ctx.m_document->m_nodes[ lastPair ].m_element.m_next );

                    lastPair = pairIndex;

                    node.m_compound.m_count++;

                    if ( !skipChar( ctx, ',' ) ) break;
                }
            }

            if ( !skipChar( ctx, blockChars[ 1 ] ) ) return INVALID_INDEX;

            ctx.m_document->m_nodes.pushBack( node );
            return ctx.m_document->m_nodes.getSize() - 1;
        }
        return INVALID_INDEX;
    }

    //
    int parseString( Context& ctx, bool merge )
    {
        const int   stringIndex = ctx.m_document->m_strings.getSize();

        for ( ;;)
        {
            const char closure = ctx.m_stream[ 0 ] == '"' ? '"' : ctx.m_stream[ 0 ] == '\'' ? '\'' : 0;
            if ( !closure ) return INVALID_INDEX; else ++ctx.m_stream;

            while ( ctx.m_stream[ 0 ] && ctx.m_stream[ 0 ] != closure )
            {
                const char c = *ctx.m_stream++;
                if ( c == '\\' )
                {
                    switch ( ctx.m_stream[ 0 ] )
                    {
                        case    '"':    ctx.m_stream++; ctx.m_document->m_strings.pushBack( '"' ); break;
                        case    '\\':   ctx.m_stream++; ctx.m_document->m_strings.pushBack( '\\' ); break;
                        case    '/':    ctx.m_stream++; ctx.m_document->m_strings.pushBack( '/' ); break;
                        case    'b':    ctx.m_stream++; ctx.m_document->m_strings.pushBack( '\b' ); break;
                        case    'f':    ctx.m_stream++; ctx.m_document->m_strings.pushBack( '\f' ); break;
                        case    'n':    ctx.m_stream++; ctx.m_document->m_strings.pushBack( '\n' ); break;
                        case    'r':    ctx.m_stream++; ctx.m_document->m_strings.pushBack( '\r' ); break;
                        case    't':    ctx.m_stream++; ctx.m_document->m_strings.pushBack( '\t' ); break;
                        case    'u':
                            {
                                wchar_t cval = 0;
                                ctx.m_stream++;
                                for ( int i = 0; i < 4; ++i, ++ctx.m_stream )
                                {
                                    const char n = ctx.m_stream[ 0 ];
                                    if ( isDecimal( n ) )               cval = cval << 4 | ( int( n ) - '0' );
                                    else if ( n >= 'a' && n <= 'f' )    cval = cval << 4 | ( 10 + ( int( n ) - 'a' ) );
                                    else if ( n >= 'A' && n <= 'F' )    cval = cval << 4 | ( 10 + ( int( n ) - 'A' ) );
                                    else return INVALID_INDEX;
                                }
                                ctx.m_document->m_strings.popBack( 6 - hkUtf8::utf8FromCodePoint( ctx.m_document->m_strings.expandBy( 6 ), cval ) );
                            }
                            break;
                        default:    return INVALID_INDEX;
                    }
                }
                else ctx.m_document->m_strings.pushBack( c );
            }

            if ( !skipChar( ctx, closure ) ) return INVALID_INDEX;

            if ( !skipChar( ctx, '\\' ) ) break;
        }

        ctx.m_document->m_strings.pushBack( 0 );

        hkJsonNode node( hkJsonNode::String );
        setPointerAsIndex( stringIndex, node.m_string );
        ctx.m_document->m_nodes.pushBack( node );

        return ctx.mergeLastString( stringIndex, merge );
    }

    //
    int parseIdentifier( Context& ctx, bool merge )
    {
        if ( !isAlpha( ctx.m_stream[ 0 ] ) ) return INVALID_INDEX;

        const int stringIndex = ctx.m_document->m_strings.getSize();

        do
        {
            ctx.m_document->m_strings.pushBack( *ctx.m_stream++ );
        } while ( isAlpha( ctx.m_stream[ 0 ] ) || isDecimal( ctx.m_stream[ 0 ] ) );

        ctx.m_document->m_strings.pushBack( 0 );

        hkJsonNode node( hkJsonNode::String );
        setPointerAsIndex( stringIndex, node.m_string );
        ctx.m_document->m_nodes.pushBack( node );

        return ctx.mergeLastString( stringIndex, merge );
    }

    //
    int parseNumber( Context& ctx )
    {
        int     len = 0;
        char    asc[ 256 ];

        for ( ; len < int( HK_COUNT_OF( asc ) - 1 );)
        {
            const char c = ctx.m_stream[ len ];
            if ( isDecimal( c ) || c == '+' || c == '-' || c == '.' || c == 'e' || c == 'E' )
                asc[ len++ ] = c;
            else
                break;
        }

        if ( !len || len == ( HK_COUNT_OF( asc ) - 1 ) ) return INVALID_INDEX;

        ctx.m_stream += len;
        asc[ len ] = 0;

        hkJsonNode node( hkJsonNode::Number );
        node.m_number = hkString::atof( asc );
        ctx.m_document->m_nodes.pushBack( node );

        return ctx.m_document->m_nodes.getSize() - 1;
    }

    //
    int parseValue( Context& ctx )
    {
        switch ( ctx.m_stream[ 0 ] )
        {
            case '{':
            case '[':   return parseObjectOrArray( ctx );
            case '"':   return parseString( ctx, false );
            case 'n':   if ( ctx.m_stream[ 1 ] == 'u' && ctx.m_stream[ 2 ] == 'l' && ctx.m_stream[ 3 ] == 'l' ) { ctx.m_stream += 4; skipSpaces( ctx ); return NULL_INDEX; } break;
            case 't':   if ( ctx.m_stream[ 1 ] == 'r' && ctx.m_stream[ 2 ] == 'u' && ctx.m_stream[ 3 ] == 'e' ) { ctx.m_stream += 4; skipSpaces( ctx ); return TRUE_INDEX; } break;
            case 'f':   if ( ctx.m_stream[ 1 ] == 'a' && ctx.m_stream[ 2 ] == 'l' && ctx.m_stream[ 3 ] == 's' && ctx.m_stream[ 4 ] == 'e' ) { ctx.m_stream += 5; skipSpaces( ctx ); return FALSE_INDEX; } break;
            default:    return parseNumber( ctx );
        }
        return INVALID_INDEX;
    }

    //
    int clone(_In_ const hkJsonNode* node, hkJsonDocument& document, hkPointerMap<const char*,int>& stringMap )
    {
        switch ( node->m_type )
        {
            case    hkJsonNode::Array:
            case    hkJsonNode::Object:
            {
                    hkJsonNode  newNode( node->m_type );
                    int         lastElement = 0;
                    newNode.m_compound.m_first = HK_NULL;
                    newNode.m_compound.m_count = 0;
                    for ( const hkJsonNode* element = node->m_compound.m_first; element; element = element->m_element.m_next )
                    {
                        hkJsonNode newElement( hkJsonNode::Element );

                        newElement.m_element.m_next = HK_NULL;

                        if ( node->m_type == hkJsonNode::Object )
                            setPointerAsIndex( clone( element->m_element.m_key, document, stringMap ), newElement.m_element.m_key );
                        else
                            newElement.m_element.m_key = HK_NULL;

                        setPointerAsIndex( clone( element->m_element.m_value, document, stringMap ), newElement.m_element.m_value );

                        const int index = document.m_nodes.getSize();

                        document.m_nodes.pushBack( newElement );

                        if ( lastElement )
                            setPointerAsIndex( index, document.m_nodes[ lastElement ].m_element.m_next );
                        else
                            setPointerAsIndex( index, newNode.m_compound.m_first );

                        newNode.m_compound.m_count++;
                        lastElement = index;
                    }

                    document.m_nodes.pushBack( newNode );
                    return document.m_nodes.getSize() - 1;
                }
                break;

            case    hkJsonNode::String:
                {
                    hkJsonNode  newNode( hkJsonNode::String );
                    int         stringIndex = stringMap.getWithDefault( node->m_string, 0 );
                    if ( stringIndex == 0 )
                    {
                        const char* stringPtr = node->m_string;
                        stringIndex = document.m_strings.getSize();
                        do { document.m_strings.pushBack( *stringPtr ); } while ( *stringPtr++ );
                        stringMap.insert( node->m_string, stringIndex );
                    }

                    setPointerAsIndex( stringIndex, newNode.m_string );
                    document.m_nodes.pushBack( newNode );
                    return document.m_nodes.getSize() - 1;
                }
                break;

            case    hkJsonNode::Number:
                {
                    hkJsonNode newNode( hkJsonNode::Number );
                    newNode.m_number = node->m_number;
                    document.m_nodes.pushBack( newNode );
                    return document.m_nodes.getSize() - 1;
                }

            case    hkJsonNode::Null:       return NULL_INDEX;
            case    hkJsonNode::Boolean:    return node->m_boolean ? TRUE_INDEX : FALSE_INDEX;

            default:
                HK_ASSERT_NOT_IMPLEMENTED( 0x12F3454D );
                break;
        }
        return INVALID_INDEX;
    }

    //
    void convertIndicesToPointers( hkJsonDocument& document )
    {
        for ( int i = 0; i < document.m_nodes.getSize(); ++i )
        {
            hkJsonNode& node = document.m_nodes[ i ];
            switch ( node.m_type )
            {
                case hkJsonNode::Object:
                case hkJsonNode::Array:
                    convertIndexToPointer( document.m_nodes.begin(), node.m_compound.m_first );
                    break;

                case hkJsonNode::Element:
                    convertIndexToPointer( document.m_nodes.begin(), node.m_element.m_key );
                    convertIndexToPointer( document.m_nodes.begin(), node.m_element.m_value );
                    convertIndexToPointer( document.m_nodes.begin(), node.m_element.m_next );
                    break;

                case hkJsonNode::String:
                    convertIndexToPointer( document.m_strings.begin(), node.m_string );
                    break;
                default:
                    break;
            }
        }
    }

    //
    void base64StringToByteArray(_In_opt_z_ const char* input, hkArray<hkUint8>& output)
    {
        output.clear();
        if ( !input ) return;

        int block[ 4 ] = { 0, 0, 0, 0 };
        int blockSize = 0;
        while ( *input )
        {
            const char  byte = *input++;
            int         value = 0;
            if ( byte >= 'A' && byte <= 'Z' ) value = byte - 'A';
            else if ( byte >= 'a' && byte <= 'z' ) value = 26 + ( byte - 'a' );
            else if ( byte >= '0' && byte <= '9' ) value = 52 + ( byte - '0' );
            else if ( byte == '/' ) value = 63;
            else if ( byte == '+' ) value = 62;
            else if ( byte == '=' ) break;
            else HK_ERROR( 0xfc5230e2, "Found invalid base64 character: " << byte );

            block[ blockSize++ ] = value;

            if ( blockSize == 4 )
            {
                const hkUint8 bytes[] =
                {
                    hkUint8( ( block[ 0 ] << 2 ) | ( block[ 1 ] >> 4 ) ),
                    hkUint8( ( block[ 1 ] << 4 ) | ( block[ 2 ] >> 2 ) ),
                    hkUint8( ( block[ 2 ] << 6 ) | ( block[ 3 ] >> 0 ) )
                };
                output.append( bytes, 3 );
                blockSize = block[ 0 ] = block[ 1 ] = block[ 2 ] = block[ 3 ] = 0;
            }
        }

        if ( blockSize )
        {
            const int numBytes[] = { 0, 0, 1, 2 };
            const hkUint8 bytes[] =
            {
                hkUint8( ( block[ 0 ] << 2 ) | ( block[ 1 ] >> 4 ) ),
                hkUint8( ( block[ 1 ] << 4 ) | ( block[ 2 ] >> 2 ) )
            };
            output.append( bytes, numBytes[ blockSize ] );
        }
    }

    //
    void byteArrayToBase64String(_In_reads_bytes_(size) const void* bytes, int size, hkStringBuf& output )
    {
        const char*     bin2ascii = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
        const hkUint8*  inputs = reinterpret_cast<const hkUint8*>( bytes );
        for ( ; size > 0; )
        {
            hkUint8 ib[] = { 0, 0, 0 };
            const int bufferLength = hkMath::min2( 3, size );
            hkString::memCpy( ib, inputs, bufferLength );

            inputs += bufferLength;
            size -= bufferLength;

            char ob[ 4 ];
            ob[ 0 ] = bin2ascii[ ( ib[ 0 ] >> 2 ) ];
            ob[ 1 ] = bin2ascii[ ( ( ib[ 0 ] << 4 ) & 0x30 ) | ( ib[ 1 ] >> 4 ) ];
            ob[ 2 ] = bin2ascii[ ( ( ib[ 1 ] << 2 ) & 0x3c ) | ( ib[ 2 ] >> 6 ) ];
            ob[ 3 ] = bin2ascii[ ( ib[ 2 ] & 0x3f ) ];

            if ( bufferLength == 1 ) ob[ 2 ] = ob[ 3 ] = '=';
            else if ( bufferLength == 2 ) ob[ 3 ] = '=';

            output.append( ob, 4 );
        }
    }
}

//
// hkJsonNode
//

//
_Ret_maybenull_ const hkJsonNode* hkJsonNode::select(_In_z_ const char* path ) const
{
    const hkJsonNode*           current = this;
    hkInplaceArray<char, 128>   keyOrIndex;

    while ( path[ 0 ] && current )
    {
        bool    isIndex = true;
        keyOrIndex.clear();
        while ( path[ 0 ] && path[ 0 ] != '/' )
        {
            if ( path[ 0 ] < '0' || path[ 0 ] > '9' ) isIndex = false;
            keyOrIndex.pushBack( *path++ );
        }
        if ( path[ 0 ] == '/' ) ++path;
        if ( keyOrIndex.getSize() > 0 )
        {
            keyOrIndex.pushBack( 0 );
            if ( isIndex )
            {
                const int index = hkString::atoi( keyOrIndex.begin(), 10 );
                current = current->findValueByIndex( index );
            }
            else
            {
                current = current->findValueByKey( keyOrIndex.begin() );
            }
        }
    }

    return current;
}

//
_Ret_maybenull_ const hkJsonNode* hkJsonNode::findValueByKey(_In_z_ const char* key) const
{
    const hkJsonNode* element = m_compound.m_first;
    if ( m_type == Object )
    {
        while ( element )
        {
            if ( hkString::strCmp( element->m_element.m_key->m_string, key ) == 0 )
                return element->m_element.m_value;
            else
                element = element->m_element.m_next;
        }
    }
    else if ( m_type == Array )
    {
        while ( element )
        {
            if ( element->m_element.m_value->m_type == String && hkString::strCmp( element->m_element.m_value->m_string, key ) == 0 )
                return element->m_element.m_value;
            else
                element = element->m_element.m_next;
        }
    }
    else if ( m_type == Element )
    {
        if ( hkString::strCmp( m_element.m_key->m_string, key ) == 0 ) return m_element.m_value;
    }
    else if ( m_type == String )
    {
        if ( hkString::strCmp( m_string, key ) == 0 ) return this;
    }
    return HK_NULL;
}

//
_Ret_maybenull_ const hkJsonNode* hkJsonNode::findKeyByIndex(int index) const
{
    if (isCompound())
    {
        if (index < 0 || index >= m_compound.m_count) return HK_NULL;

        const hkJsonNode* element = m_compound.m_first;

        for (int i = 0; i < index && element; ++i) element = element->m_element.m_next;

        if (element) return element->m_element.m_key;
    }
    return HK_NULL;
}

//
const hkJsonNode* hkJsonNode::findValueByIndex( int index ) const
{
    if ( isCompound() )
    {
        if ( index < 0 || index >= m_compound.m_count ) return HK_NULL;

        const hkJsonNode* element = m_compound.m_first;

        for ( int i = 0; i < index && element; ++i ) element = element->m_element.m_next;

        if ( element ) return element->m_element.m_value;
    }
    return HK_NULL;
}

//
bool hkJsonNode::read( hkVector4& output, int minNumElements ) const
{
    if ( isCompound() )
    {
        if ( m_compound.m_count >= minNumElements )
        {
            hkVector4           vector; vector.setZero();
            const hkJsonNode*   element = m_compound.m_first;
            for ( int i = 0, n = hkMath::min2( 4, m_compound.m_count ); i < n; ++i, element = element->m_element.m_next )
            {
                if ( !element->m_element.m_value->read( vector( i ) ) ) return false;
            }
            output = vector;
            return true;
        }
    }
    return false;
}

//
bool hkJsonNode::read( hkQuaternion& output ) const
{
    hkVector4   axisAngle; if ( !read( axisAngle, 4 ) ) return false;
    hkVector4   axis = axisAngle; axis.normalize<3, HK_ACC_FULL, HK_SQRT_SET_ZERO>();
    output.setIdentity();
    if ( axis.isNormalized<3>() )
    {
        output.setAxisAngle( axis, axisAngle( 3 ) );
    }
    return true;
}

//
bool hkJsonNode::read( hkMatrix3& output ) const
{
    output.setIdentity();
    if ( isCompound() && m_compound.m_count == 3 )
    {
        const hkJsonNode*   row = m_compound.m_first;

        hkVector4   row0; if ( !row->m_element.m_value->read( row0, 3 ) ) return false; else row = row->m_element.m_next;
        hkVector4   row1; if ( !row->m_element.m_value->read( row1, 3 ) ) return false; else row = row->m_element.m_next;
        hkVector4   row2; if ( !row->m_element.m_value->read( row2, 3 ) ) return false;

        output.setRows( row0, row1, row2 );

        return true;
    }
    return false;
}

//
bool hkJsonNode::read( hkMatrix4& output ) const
{
    output.setIdentity();
    if ( isCompound() && m_compound.m_count == 4 )
    {
        const hkJsonNode*   row = m_compound.m_first;

        hkVector4   row0; if ( !row->m_element.m_value->read( row0, 4 ) ) return false; else row = row->m_element.m_next;
        hkVector4   row1; if ( !row->m_element.m_value->read( row1, 4 ) ) return false; else row = row->m_element.m_next;
        hkVector4   row2; if ( !row->m_element.m_value->read( row2, 4 ) ) return false; else row = row->m_element.m_next;
        hkVector4   row3; if ( !row->m_element.m_value->read( row3, 4 ) ) return false;

        output.setRows( row0, row1, row2, row3 );

        return true;
    }
    return false;
}

//
bool hkJsonNode::read( hkTransform& output ) const
{
    hkMatrix4 mat;
    if ( read( mat ) )
    {
        output.set4x4ColumnMajor( &mat( 0, 0 ) );
        return true;
    }
    return false;
}

//
bool hkJsonNode::read( class hkAabb& output ) const
{
    output.setEmpty();
    if ( isCompound() && m_compound.m_count == 2 )
    {
        hkVector4 bounds[ 2 ];
        if ( readArray( bounds, 2 ) )
        {
            output.m_min = bounds[ 0 ];
            output.m_max = bounds[ 1 ];
            return true;
        }
    }
    return false;
}

//
bool hkJsonNode::read( class hkUuid& output ) const
{
    if ( m_type == String  && hkString::strLen( m_string ) == 36 )
    {
        if ( output.setFromFormattedString( m_string ).isSuccess() )
        {
            return true;
        }
    }
    output = hkUuid::getNil();
    return false;
}

//
bool hkJsonNode::readBase64( hkArray<hkUint8>& bytes ) const
{
    if ( m_type == String )
    {
        hkJSON_Internals::base64StringToByteArray( m_string, bytes );
        return true;
    }
    return false;
}

//
void hkJsonNode::toString( hkStringBuf& stringBuffer ) const
{
    hkArray<char, hkContainerHeapAllocator> buffer;
    {
        hkOstream       stream( buffer );
        hkJsonOstream   jsonStream( stream );
        jsonStream.write( this );
    }
    stringBuffer.set( buffer.begin() );
}

//
// hkJsonDocument
//

//
void hkJsonDocument::clear()
{
    m_root = 0;
    m_nodes.clear();
    m_strings.clear();
}

//
_Ret_notnull_ const hkJsonNode* hkJsonDocument::parse(_In_z_ const char* text, bool mergeStrings)
{
    // Prepare the document.
    hkJSON_Internals::prepareDocument( *this );

    // Handle BOM if any.
    if ( text[ 0 ] == '\x0ef' && text[ 1 ] == '\x0bb' && text[ 2 ] == '\x0bf' )
    {
        // UTF-8.
        text += 3;
    }
    else if ( text[ 0 ] == '\x0fe' && text[ 1 ] == '\x0ff' )
    {
        // UTF-16 BE.
        
        HK_WARN( 0x793CA465, "UTF-16 BE not implemented" );
    }
    else if ( text[ 0 ] == '\x0ff' && text[ 1 ] == '\x0fe' )
    {
        // UTF-16 LE.
        
        HK_WARN( 0x793CA465, "UTF-16 LE not implemented" );
    }

    // Prepare context.
    hkJSON_Internals::Context ctx;
    ctx.m_stream = text;
    ctx.m_document = this;
    ctx.m_mergeStrings = mergeStrings;

    hkJSON_Internals::skipSpaces( ctx );

    // Parse.
    m_root = hkJSON_Internals::parseValue( ctx );

    // Convert indices to pointers.
    hkJSON_Internals::convertIndicesToPointers( *this );

    return &m_nodes[ m_root ];
}

//
_Ret_notnull_ const hkJsonNode* hkJsonDocument::parse(hkIstream& stream, bool mergeStrings)
{
    hkArray<char>   text;
    while ( stream.isOk() )
    {
        char        block[ 8192 ];
        const int   numRead = stream.read( block, HK_COUNT_OF( block ) );
        text.append( block, numRead );
    }
    text.pushBack( 0 );

    return parse( text.begin(), mergeStrings );
}

//
bool hkJsonDocument::read( const hkJsonNode* json )
{
    if ( m_nodes.getSize() == 0 || json < m_nodes.begin() || json > &m_nodes.back() )
    {
        if ( json->m_type == hkJsonNode::Null )
        {
            clear();
        }
        else
        {
            hkPointerMap<const char*, int> stringMap;
            hkJSON_Internals::prepareDocument( *this );
            m_root = hkJSON_Internals::clone( json, *this, stringMap );
            hkJSON_Internals::convertIndicesToPointers( *this );
        }
        return true;
    }
    return false;
}

//
void hkJsonDocument::write( hkJsonOstream& json ) const
{
    if ( m_root )
        json.write( root() );
    else
        json.write( "null" );
}

//
// hkJsonOstream
//

//
void    hkJsonOstream::write( const hkVector4& value )
{
    updateState();
    const bool h = m_holdNewLines; m_holdNewLines = true;
    m_stream << "[ ";
    m_states.pushBack( FIRST_ARR_VALUE );
    m_level++;

    write( value( 0 ) );
    write( value( 1 ) );
    write( value( 2 ) );
    write( value( 3 ) );

    end();

    m_holdNewLines = h;
}

//
void    hkJsonOstream::write( double value )
{
    updateState();
    char buffer[ 128 ];
    hkString::snPrintf( buffer, HK_COUNT_OF(buffer), HK_COUNT_OF(buffer), "%G", value );
    m_stream << buffer;
}

//
void    hkJsonOstream::write( const hkQuaternion& value )
{
    hkVector4 axisAngle;
    value.getAxis( axisAngle );
    axisAngle( 3 ) = value.getAngle();
    write( axisAngle );
}

//
void    hkJsonOstream::write( const hkMatrix4& value )
{
    hkVector4 row;
    beginArray();
    value.getRow<0>( row ); write( row );
    value.getRow<1>( row ); write( row );
    value.getRow<2>( row ); write( row );
    value.getRow<3>( row ); write( row );
    end();
}

//
void    hkJsonOstream::write( const hkTransform& value )
{
    hkMatrix4 mat;
    value.get4x4ColumnMajor( &mat( 0, 0 ) );
    write( mat );
}

//
void    hkJsonOstream::write( const hkAabb& value )
{
    const bool h = m_holdNewLines; m_holdNewLines = true;
    beginArray();
    write( value.m_min );
    write( value.m_max );
    end();
    m_holdNewLines = h;
}

//
void    hkJsonOstream::write( const hkUuid& value )
{
    hkStringBuf buffer;
    write( value.toString( buffer ) );
}

//
void    hkJsonOstream::writeBase64(_In_reads_bytes_(size) const void* value, int size )
{
    hkStringBuf buffer;
    hkJSON_Internals::byteArrayToBase64String( value, size, buffer );
    write( buffer.cString() );
}

//
void hkJsonOstream::write( const hkJsonNode* node )
{
    switch ( node->m_type )
    {
        case    hkJsonNode::Array:
            {
                beginArray();
                for ( const hkJsonNode* element = node->m_compound.m_first; element; element = element->m_element.m_next )
                {
                    write( element->m_element.m_value );
                }
                end();
            }
            break;
        case    hkJsonNode::Object:
            {
                beginObject();
                for ( const hkJsonNode* element = node->m_compound.m_first; element; element = element->m_element.m_next )
                {
                    write( element->m_element.m_key );
                    write( element->m_element.m_value );
                }
                end();
            }
            break;
        case    hkJsonNode::String:     write( node->m_string );  break;
        case    hkJsonNode::Number:     write( node->m_number );  break;
        case    hkJsonNode::Boolean:    write( node->m_boolean );  break;
        case    hkJsonNode::Null:       write( "null" );  break;
        default: break; // Invalid and Element
    }
}

//
void hkJsonOstream::beginObject()
{
    updateState();
    m_stream << "{\n";
    m_states.pushBack( FIRST_OBJ_KEY );
    m_level++;
}

//
void hkJsonOstream::beginArray()
{
    updateState();
    m_stream << ( m_holdNewLines ? "[ " : "[\n" );
    m_states.pushBack( FIRST_ARR_VALUE );
    m_level++;
}

//
void hkJsonOstream::end()
{
    char c = 0;
    switch ( m_states.back() )
    {
        case    FIRST_OBJ_KEY:
        case    OBJ_KEY:        c = '}'; m_states.popBack(); break;
        case    FIRST_ARR_VALUE:
        case    ARR_VALUE:      c = ']'; m_states.popBack(); break;
        default:                HK_ERROR( 0x6a07dc1, "Invalid state, expecting object value." ); break;

    }
    m_level--;
    if ( !m_holdNewLines )
    {
        m_stream << '\n';
        addTabs();
        m_stream << c;
    }
    else
    {
        m_stream << ' ' << c;
    }
}

//
void hkJsonOstream::updateState( bool asString )
{
    if ( m_states.getSize() == 0 ) return;

    State&      state = m_states.back();
    const State prevState = state;
    bool        holdNewLine = m_holdNewLines;

    switch ( state )
    {
        case FIRST_OBJ_KEY:     if ( !asString ) { HK_ERROR( 0x4A27C758, "Expecting key" ); } state = OBJ_VALUE; break;
        case OBJ_KEY:           if ( !asString ) { HK_ERROR( 0x4A27C758, "Expecting key" ); } m_stream << ",\n"; state = OBJ_VALUE; break;
        case OBJ_VALUE:         m_stream << ": "; state = OBJ_KEY; break;
        case FIRST_ARR_VALUE:   state = ARR_VALUE; break;
        case ARR_VALUE:         m_stream << ( holdNewLine ? ", " : ",\n" ); break;
    }

    if ( prevState != OBJ_VALUE && !holdNewLine ) addTabs();
}

//
void hkJsonOstream::addTabs()
{
    for ( int i = 0; i < m_level; ++i ) m_stream << '\t';
}

void hkJsonOstream::writeEscapedString( _In_z_ const char* str )
{
    // NB: Although JSON extensions allow singly quoted strings as key or value, we always output
    //     doubly quoted ones, so avoid escaping the single quotes here by design.

    char esc[2] = { '\\', ' ' };

    const char* start = str;
    const char* pc;
    for ( pc = str; *pc != '\0'; ++pc )
    {
        char escaped = '\0';
        switch ( *pc )
        {
        case '"':  escaped = '"';  break;
        case '\\': escaped = '\\'; break;
        case '/':  escaped = '/';  break;
        case '\b': escaped = 'b';  break;
        case '\f': escaped = 'f';  break;
        case '\n': escaped = 'n';  break;
        case '\r': escaped = 'r';  break;
        case '\t': escaped = 't';  break;
        default: break;
        }

        if ( escaped != '\0' )
        {
            if ( pc > start )
            {
                m_stream.write( start, hkGetByteOffsetInt( start, pc ) );
            }
            esc[1] = escaped;
            m_stream.write( esc, 2 );
            start = pc + 1;
        }
    }

    if ( pc > start )
    {
        m_stream.write( start, hkGetByteOffsetInt( start, pc ) );
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
