// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once



#include <Common/Base/Container/StringMap/hkStringMap.h>

class hkResourceContainer;

class hkResourceMap;

class HK_EXPORT_COMMON hkResourceBase : public hkReferencedObject
{
    public:

        HK_DECLARE_REFLECTION();
        HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_EXPORT );

            //
        hkResourceBase(): hkReferencedObject() {}

        enum Type
        {
            TYPE_RESOURCE,
            TYPE_CONTAINER
        };

            /// return the type of this object
        virtual Type getType() const = 0;

            /// Returns the name of the resource.
            /// If the name has to be created on the fly the supplied buffer will be used.
        virtual _Ret_z_ const char* getName(hkStringBuf& buffer) const = 0;
};


    /// A virtual interface to a resource which is owned by the hkResourceContainer
class HK_EXPORT_COMMON hkResourceHandle: public hkResourceBase
{
    public:

        HK_DECLARE_REFLECTION();
        HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_EXPORT );

        virtual Type getType() const { return TYPE_RESOURCE; }




            /// An external link
        struct HK_EXPORT_COMMON Link
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkResourceHandle::Link);
            Link() {}

            const _Ret_maybenull_ hkReflect::Type* getLinkType() const;

            const char*     m_memberName;
            const char*     m_externalId;
            hkReflect::Var  m_member;
        };



            /// Set the name of the resource.
            /// A copy of the name will be stored.
        virtual void setName(_In_z_ const char* name) = 0;

            /// Returns a pointer to the object.
        virtual _Ret_notnull_ void* getObject() const = 0;

            /// Returns a pointer to the class.
        virtual _Ret_notnull_ const hkReflect::Type* getObjectType() const = 0;

        _Ret_maybenull_ void* hasA(_In_ const hkReflect::Type* t);

            /// Returns a pointer to the class.
        template<typename T>
        inline _Ret_maybenull_ T* hasA() { return static_cast<T*>(hasA(hkReflect::getType<T>())); }

            /// Returns a pointer to the class.
        virtual void setObject(_In_ void* object, _In_ const hkReflect::Type* klass) = 0;

            /// Adds a new external link, specified by the 'name' of the referencing object as well as the memberName.
            /// If the object uses nested structures, the memberName will look like: xxx.yyy.zzz
        virtual void addExternalLink(_In_z_ const char* memberName, _In_z_ const char* m_externalId) = 0;

            /// removes an internal link
        virtual void removeExternalLink(_In_z_ const char* memberName) = 0;

            /// Returns a list with all unresolved external links.
        virtual void getExternalLinks(hkArray<Link>& linksOut) = 0;

            /// Clears the list with all external links.
        virtual void clearExternalLinks() = 0;

            /// tryToResolveLinks
        virtual void tryToResolveLinks(hkResourceMap& map);


        virtual ~hkResourceHandle() {}

    protected:
        hkResourceHandle() : hkResourceBase() {}
};

    /// The owner of a resource handle
class HK_EXPORT_COMMON hkResourceContainer : public hkResourceBase
{
    public:

        HK_DECLARE_REFLECTION();
        HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_EXPORT );

            // hkResourceBase implementation
        virtual Type getType() const { return TYPE_RESOURCE; }

            /// Create an owned resource
        virtual _Ret_notnull_ hkResourceHandle* createResource(_In_z_ const char* name, _Inout_ void* object, _In_ const hkReflect::Type* klass) = 0;

        /// Create an owned resource
        template<typename T>
        inline _Ret_notnull_ hkResourceHandle* createResource(_In_z_ const char* name, _Inout_ T* object) { return createResource(name, object, hkReflect::getType<T>()); }

            /// Destroys a resource
        virtual void destroyResource(_Inout_ hkResourceHandle* handle) = 0;

            /// Get number of resources
        virtual int getNumResources() = 0;

            /// Tries to find a named resource with given type.
            /// It returns the first object after \a prevObject, with \a resourceName.
            ///   - If \a prevObject is null then the search begins from the start of the container.
            ///   - If \a resourceName is null than all objects will match and the method will return the first match.

        virtual _Ret_maybenull_ hkResourceHandle* findResourceByName(_In_z_ const char* resourceName, _In_opt_ const hkReflect::Type* klass = HK_NULL, _In_opt_ const hkResourceHandle* prevObject = HK_NULL) const = 0;

            /// Tries to find a resource with given type.
            /// It returns the first object after \a prevObject.
            ///   - If \a prevObject is null then the search begins from the start of the container.

        virtual hkResourceHandle* findResourceByType( const hkReflect::Type* klass, const hkResourceHandle* prevObject = HK_NULL ) const = 0;

            /// Tries to find a named resource with given type.
            /// It returns the first object after \a prevObject, with \a resourceName.
            ///   - If \a prevObject is null then the search begins from the start of the container.
            ///   - If \a resourceName is null than all objects will match and the method will return the first match.
        template<typename T>
        inline _Ret_maybenull_ hkResourceHandle* findResourceByName(_In_z_ const char* resourceName, _In_opt_ const hkResourceHandle* prevObject = HK_NULL) const
        {
            return findResourceByName( resourceName, hkReflect::getType<T>(), prevObject );
        }

            /// Tries to find a resource with given type.
            /// It returns the first object after \a prevObject.
            ///   - If \a prevObject is null then the search begins from the start of the container.
        template<typename T>
        inline hkResourceHandle* findResourceByType(const hkResourceHandle* prevObject = HK_NULL) const
        {
            return findResourceByType(hkReflect::getType<T>(), prevObject);
        }

            /// Simple helper function to find a resource which is identified by a path (using '/' to split the path).
        //hkResourceHandle* findResourceRecursively( const char* resourcePath);

            /// Simple helper function to get all terminal resources
        void findAllResourceRecursively( hkArray<hkResourceHandle*>& resourcesOut );


            /// Simple helper function to get all terminal resources
        void findAllContainersRecursively( hkArray<hkResourceContainer*>& resourcesOut );

        void getPath( hkStringBuf& pathOut );

            /// Get my parent container if any
        virtual _Ret_maybenull_ hkResourceContainer* getParent() = 0;

            /// Create child container, if the container already exists, simply return the existing one.
        virtual _Ret_notnull_ hkResourceContainer* createContainer(_In_z_ const char* path) = 0;

            /// Destroys a child container recursively
        virtual void destroyContainer(_Inout_ hkResourceContainer* container) = 0;

            /// Get number of child containers
        virtual int getNumContainers() = 0;

            /// Tries to find a named container.
            /// It returns the first object after \a prevContainer, with \a containerName.
            ///   - If \a prevContainer is null then the search begins from the start of the child containers.
            ///   - If \a containerName is null then all objects will match and the method will return the first match.
        virtual _Ret_maybenull_ hkResourceContainer* findContainerByName(_In_z_ const char* containerName, _In_opt_ const hkResourceContainer* prevContainer = HK_NULL) const = 0;

            /// move this container to a new parent
        virtual hkResult parentTo(_Inout_ hkResourceContainer* newParent) = 0;

        //
        //  Simple helper functions
        //
            /// helper function which tries to resolve external links
        virtual void tryToResolveLinks( hkResourceMap& resourceMap );

        template<typename T>
        _Ret_maybenull_ T* findResource(_In_z_ const char* name, _In_opt_ const hkReflect::Type* klass)
        {
            hkResourceHandle* handle = findResourceByName( name, klass );
            if (!handle)
            {
                return HK_NULL;
            }
            return reinterpret_cast<T*>(handle->getObject());
        }

        template<typename T>
        inline _Ret_maybenull_ T* findResource(_In_z_ const char* name)
        {
            return findResource<T>( name, hkReflect::getType<T>() );
        }

        virtual ~hkResourceContainer() {}

    protected:

            // Empty constructor
        hkResourceContainer() : hkResourceBase() {}
};


    /// Simple class which allows for fast searching of object/classes
class HK_EXPORT_COMMON hkResourceMap
{
    public:
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkResourceMap);

            /// Tries to find a resource with matching name and type
            /// to the first object after prevObject, with a name corresponding to 'objectName'.
            ///   - If prevObject is null then the search begins from the start of the container.
            ///   - If objectName is null than all objects will match.
            ///   = \a klass is only used for extra debugging checks, it cannot be used for searching for a type!
        virtual _Ret_maybenull_ void* findObjectByName(_In_z_ const char* objectName, _In_opt_ const hkReflect::Type** klassOut = HK_NULL) const = 0;

    protected:

        virtual ~hkResourceMap(){}

        hkResourceMap(){}
};




    /// A simple Havok serializable version of a resource
class HK_EXPORT_COMMON hkMemoryResourceHandle: public hkResourceHandle
{
    public:
        HK_DECLARE_CLASS(hkMemoryResourceHandle, Reflect, New, Version(3));

            /// Constructor
        hkMemoryResourceHandle();

            /// Destructor
        virtual ~hkMemoryResourceHandle();

            /// This structure stores information about an external link.
        struct ExternalLink
        {
            //+version(1)
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_EXPORT, hkMemoryResourceHandle::ExternalLink );
            HK_DECLARE_REFLECTION();

            ExternalLink() {}

                /// The member name that stores the link to the external object.
                /// If the name will look like xxx.yyy.zzz
            hkStringPtr m_memberName;

                /// The 'name' of the linked object.
            hkStringPtr m_externalId;
        };

        virtual _Ret_z_ const char* getName(hkStringBuf& buffer) const;

            /// Set the name of the resource.
            /// A copy of the name will be stored.
        void setName(_In_z_ const char* name);

            /// Returns a pointer to the object.
        _Ret_notnull_ void* getObject() const;

            /// Returns a pointer to the class.
        _Ret_notnull_ const hkReflect::Type* getObjectType() const;

            /// Set the object and class.
        void setObject(_In_ void* object, _In_ const hkReflect::Type* klass);

        void addExternalLink(_In_z_ const char* memberName, _In_z_ const char* m_externalId);

            /// removes an external link
        void removeExternalLink(_In_z_ const char* memberName );

            /// Returns a list with all unresolved external links.
        void getExternalLinks(hkArray<Link>& linksOut);

            /// Clears the list with all external links.
        void clearExternalLinks();

    protected:

        hkRefVariant m_variant;
            // Set m_objectIsRerencedObject default to old behavior
        hkStringPtr m_name;

        hkArray<struct hkMemoryResourceHandle::ExternalLink> m_references;
};





    /// Simple Havok serializable version of a hkResourceContainer
class HK_EXPORT_COMMON hkMemoryResourceContainer : public hkResourceContainer
{
    public:
        HK_DECLARE_CLASS(hkMemoryResourceContainer, Reflect, New, Version(1));

        hkMemoryResourceContainer(_In_z_ const char* name = "");

        void afterReflectNew();

        virtual ~hkMemoryResourceContainer();

        virtual _Ret_z_ const char* getName(hkStringBuf& buffer) const;

        _Ret_notnull_ hkResourceHandle* createResource(_In_z_ const char* name, _In_ void* object, _In_ const hkReflect::Type* klass);

        int getNumResources() { return m_resourceHandles.getSize(); }

        _Ret_maybenull_ hkResourceHandle* findResourceByName(_In_opt_z_ const char* resourceName, _In_opt_ const hkReflect::Type* klass = HK_NULL, _In_opt_ const hkResourceHandle* prevObject = HK_NULL) const;

        hkResourceHandle* findResourceByType( const hkReflect::Type* klass, const hkResourceHandle* prevObject = HK_NULL ) const;

        void destroyResource(_Inout_ hkResourceHandle* resourceHandle);

        virtual _Ret_maybenull_ hkResourceContainer* getParent() { return m_parent; }

        virtual _Ret_notnull_ hkResourceContainer* createContainer(_In_opt_z_ const char* name);

        virtual void destroyContainer(_Inout_ hkResourceContainer* container );

        virtual int getNumContainers();

        virtual hkResult parentTo(_Inout_ hkResourceContainer* newParent );

        virtual _Ret_maybenull_ hkResourceContainer* findContainerByName(_In_opt_z_ const char* containerName, _In_opt_ const hkResourceContainer* prevContainer = HK_NULL) const;

    protected:
        hkStringPtr m_name;

        hkMemoryResourceContainer* m_parent;        //+serialized(false)

        hkArray< hkRefPtr<hkMemoryResourceHandle> > m_resourceHandles;

        hkArray< hkRefPtr<hkMemoryResourceContainer> > m_children;
};


    /// A simple hash table to hkResourceHandles
class HK_EXPORT_COMMON hkContainerResourceMap: public hkResourceMap
{
    public:
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkContainerResourceMap);
        hkContainerResourceMap(_Inout_ class hkResourceContainer* container );

        virtual _Ret_maybenull_ void* findObjectByName(_In_z_ const char* objectName, _Outptr_opt_ const hkReflect::Type** klassOut = HK_NULL) const;
    public:

        hkStringMap<hkResourceHandle*> m_resources;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
