// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/ResourceHandle/hkResourceHandle.h>
#include <Common/Base/Container/String/hkStringBuf.h>

_Ret_maybenull_ const hkReflect::Type* hkResourceHandle::Link::getLinkType() const
{
    if (hkReflect::PointerVar ptr = m_member)
    {
        return ptr.getSubType();
    }
    else if (hkReflect::ArrayVar arr = m_member)
    {
        if (const hkReflect::PointerType* ptrType = arr.getSubType()->asPointer())
        {
            return ptrType->getSubType();
        }
    }
    HK_ASSERT_NO_MSG(0x74ad86e5,0);
    return HK_NULL;
}


_Ret_maybenull_ void* hkResourceHandle::hasA(_In_ const hkReflect::Type* t)
{
    const hkReflect::Type* ot = getObjectType();
    return (ot && ot->extendsOrEquals(t)) ? getObject() : HK_NULL;     //SJSJ  COM-4071, getObjectType() legally returns HK_NULL if not found
}


hkMemoryResourceHandle::hkMemoryResourceHandle()
: hkResourceHandle()
{
}


hkMemoryResourceHandle::~hkMemoryResourceHandle()
{
}


void hkMemoryResourceHandle::clearExternalLinks()
{
    m_references.clear();
}


_Ret_z_ const char* hkMemoryResourceHandle::getName(hkStringBuf& buffer) const
{
    if ( m_name )
    {
        return m_name;
    }
    return "null";
}


void hkMemoryResourceHandle::setName(_In_z_ const char* name)
{
    m_name = name;
}


_Ret_notnull_ void* hkMemoryResourceHandle::getObject() const
{
    return m_variant.val();
}


_Ret_notnull_ const hkReflect::Type* hkMemoryResourceHandle::getObjectType() const
{
    return m_variant.getType();
}


void hkMemoryResourceHandle::setObject(_In_ void* object, _In_ const hkReflect::Type* klass)
{
    m_variant.set(object, klass);
}


void hkMemoryResourceHandle::addExternalLink(_In_z_ const char* memberName, _In_z_ const char* externalId)
{
    ExternalLink& link = m_references.expandOne();
    link.m_memberName            = memberName;
    link.m_externalId            = externalId;
}

void hkMemoryResourceHandle::removeExternalLink(_In_z_ const char* memberName )
{
    for (int i =0; i < m_references.getSize(); i++ )
    {
        if ( 0 == hkString::strCmp(m_references[i].m_memberName, memberName) )
        {
            m_references.removeAt(i);
            return;
        }
    }
}


void hkMemoryResourceHandle::getExternalLinks(hkArray<Link>& linksOut)
{
    linksOut.clear();
    linksOut.reserve(m_references.getSize());

    char buffer[512];

    {
        for (int i = 0; i < m_references.getSize(); i++)
        {
            Link& link = *linksOut.expandByUnchecked(1);
            const char* memberName = m_references[i].m_memberName;

            hkReflect::RecordVar record = hkReflect::Var(m_variant.val());
            HK_ASSERT_NO_MSG(0x54453e1, record);
            hkReflect::Var member;

            while(1)
            {
                const char* sep = hkString::strChr( memberName, '.');
                if ( !sep )
                {
                    member = record[memberName];
                    break;
                }
                int len = int(sep - memberName);
                hkString::memCpy( buffer, memberName, len );
                buffer[len] = 0;
                memberName += len + 1;

                member = record[buffer];
                if ( !member )
                {
                    break;
                }

                if ( !member.getType()->asRecord() )
                {
                    HK_WARN_ALWAYS( 0xf032edfe, "Member is not a record : " << buffer );
                    member = hkReflect::Var();
                    break;
                }

                record = member;
            }


            if ( member )
            {
                link.m_memberName = memberName;
                link.m_member = member;
                link.m_externalId = m_references[i].m_externalId;
            }
            else
            {
                linksOut.popBack();
                HK_WARN_ALWAYS( 0xf032edf1, "Cannot find member : " << memberName );
            }
        }
    }
}


void hkResourceHandle::tryToResolveLinks(hkResourceMap& map)
{
    hkArray<Link> links;
    getExternalLinks( links );


    for (int i = links.getSize()-1; i >= 0; i--)
    {
        Link& link = links[i];

        const hkReflect::Type* externalClass;
        void* externalObject = map.findObjectByName(link.m_externalId, &externalClass);

        if ( externalObject == HK_NULL )
        {
            //HK_WARN(0xaf12e114, "Cannot resolve " << link.m_externalId);
            continue;
        }

        hkReflect::PointerVar linkPtr = link.m_member;
        HK_ASSERT_NO_MSG(0x75906492, linkPtr);

        const hkReflect::Type* linkedClass = linkPtr.getSubType();

        // Ignore all objects whose type is not matching.
        if ( !externalClass->extendsOrEquals(linkedClass) )
        {
            HK_WARN_ALWAYS( 0xf034ed21, "Class mismatch, cannot resolve link: " << externalClass->getName() << " != " << linkedClass->getName() );
            continue;
        }

        if ( externalObject == getObject() )
        {
            HK_WARN(0xaf12e113, "Circular dependency in linked object!");
            return;
        }

        linkPtr.setValue(hkReflect::Var(externalObject, externalClass));

        // Remove the reference as we have successfully resolved it.
        removeExternalLink( link.m_memberName ); 
    }
}

void hkResourceContainer::tryToResolveLinks(hkResourceMap& map)
{
    hkArray<hkResourceHandle*> handles; findAllResourceRecursively( handles );

    for ( int i = 0; i < handles.getSize(); i++  )
    {
        hkResourceHandle* handle = handles[i];
        handle->tryToResolveLinks(map);
    }
}

void hkResourceContainer::getPath(hkStringBuf& pathOut)
{
    hkResourceContainer* parent = getParent();
    if ( parent )
    {
        parent->getPath( pathOut );
    }
    hkStringBuf buffer;
    const char* name = getName( buffer );

    pathOut += "/";
    pathOut += name;
}


void hkResourceContainer::findAllResourceRecursively( hkArray<hkResourceHandle*>& resourcesOut )
{
    for (hkResourceContainer* container = findContainerByName(HK_NULL, HK_NULL); container;  container = findContainerByName(HK_NULL, container))
    {
        container->findAllResourceRecursively( resourcesOut );
    }

    for (hkResourceHandle* handle = findResourceByName(HK_NULL, (const hkReflect::Type*)HK_NULL, HK_NULL); handle;  handle = findResourceByName(HK_NULL, (const hkReflect::Type*)HK_NULL, handle))
    {
        resourcesOut.pushBack(handle);
    }
}

void hkResourceContainer::findAllContainersRecursively( hkArray<hkResourceContainer*>& resourcesOut )
{
    resourcesOut.pushBack( this );
    for (hkResourceContainer* container = findContainerByName(HK_NULL, HK_NULL); container;  container = findContainerByName(HK_NULL, container))
    {
        container->findAllContainersRecursively( resourcesOut );
    }
}


hkMemoryResourceContainer::hkMemoryResourceContainer(_In_z_ const char* name)
: hkResourceContainer(), m_name(name), m_parent(HK_NULL)
{
}


void hkMemoryResourceContainer::afterReflectNew()
{
    for( int i = 0; i < m_children.getSize(); ++i )
    {
        m_children[i]->m_parent = this;
    }
}


hkMemoryResourceContainer::~hkMemoryResourceContainer()
{
}


_Ret_notnull_ hkResourceHandle* hkMemoryResourceContainer::createResource(_In_z_ const char* name, _In_ void* object, _In_ const hkReflect::Type* klass)
{
    hkMemoryResourceHandle* handle = new hkMemoryResourceHandle();
    handle->setName( name );
    handle->setObject( object, klass );
    m_resourceHandles.pushBack(handle);
    handle->removeReference();
    return handle;
}

_Ret_maybenull_ hkResourceHandle* hkMemoryResourceContainer::findResourceByName(_In_opt_z_ const char* objectName, _In_opt_ const hkReflect::Type* klass, _In_opt_ const hkResourceHandle* prevObject) const
{
    int index = 0;
    while ((prevObject) && (index < m_resourceHandles.getSize()) && (m_resourceHandles[index++] != prevObject)) {}

    for (int i=index; i < m_resourceHandles.getSize(); i++)
    {
        hkResourceHandle* resourceHandle = m_resourceHandles[i];

        //
        //  Check name
        //
        if ( objectName )
        {
            hkStringBuf nameBuffer;
            if ( hkString::strCmp( objectName, resourceHandle->getName(nameBuffer) ) != 0 )
            {
                continue;
            }
        }

        //
        // check type
        //
        if ( klass )
        {
            const hkReflect::Type* linkedClass = resourceHandle->getObjectType();

            if ( !linkedClass || !linkedClass->extendsOrEquals(klass) )
            {
                if ( objectName )
                {
                    HK_WARN_ALWAYS( 0xf034ed22, "Class mismatch, cannot resolve link: " << klass->getName() << " != " << linkedClass->getName() );
                    return HK_NULL;
                }
                continue;
            }
        }

        return resourceHandle;
    }
    return HK_NULL;
}

hkResourceHandle* hkMemoryResourceContainer::findResourceByType(const hkReflect::Type* klass, const hkResourceHandle* prevObject) const
{
    int index = 0;
    while ((prevObject) && (index < m_resourceHandles.getSize()) && (m_resourceHandles[index++] != prevObject)) {}

    for (int i = index; i < m_resourceHandles.getSize(); i++)
    {
        hkResourceHandle* resourceHandle = m_resourceHandles[i];

        //
        // check type
        //
        const hkReflect::Type* linkedClass = resourceHandle->getObjectType();

        if ( !linkedClass || !linkedClass->extendsOrEquals(klass) )
        {
            continue;
        }

        return resourceHandle;
    }
    return HK_NULL;
}

void hkMemoryResourceContainer::destroyResource(_Inout_ hkResourceHandle* resourceHandle)
{
    hkMemoryResourceHandle* handle = static_cast<hkMemoryResourceHandle*>(resourceHandle);
    int index = m_resourceHandles.indexOf(handle);
    if ( index > -1 )
    {
        m_resourceHandles.removeAtAndCopy(index);
    }
}

_Ret_z_ const char* hkMemoryResourceContainer::getName(hkStringBuf& buffer) const
{
    return m_name;
}

_Ret_notnull_ hkResourceContainer* hkMemoryResourceContainer::createContainer(_In_opt_z_ const char* name)
{
    {
        hkResourceContainer* container = findContainerByName( name );
        if ( container )
        {
            return container;
        }
    }


    hkMemoryResourceContainer* container = new hkMemoryResourceContainer( name );
    m_children.pushBack(container);
    container->m_parent = this;
    container->removeReference();
    return container;
}

void hkMemoryResourceContainer::destroyContainer(_Inout_ hkResourceContainer* container2)
{
    hkMemoryResourceContainer* container = static_cast<hkMemoryResourceContainer*>(container2);
    int index = m_children.indexOf(container);
    if ( index > -1 )
    {
        m_children.removeAt(index);
    }
}

int hkMemoryResourceContainer::getNumContainers()
{
    return m_children.getSize();
}

_Ret_maybenull_ hkResourceContainer* hkMemoryResourceContainer::findContainerByName(_In_opt_z_ const char* containerName, _In_opt_ const hkResourceContainer* prevContainer) const
{
    int index = 0;
    while ((prevContainer) && (index < m_children.getSize()) && (m_children[index++] != prevContainer)) {}

    for (int i=index; i < m_children.getSize(); i++)
    {
        hkMemoryResourceContainer* container = m_children[i];

        //
        //  Check name
        //
        if ( containerName )
        {
            if ( hkString::strCmp( containerName, m_children[i]->m_name ) != 0)
            {
                continue;
            }
        }

        return container;
    }
    return HK_NULL;
}

hkResult hkMemoryResourceContainer::parentTo(_Inout_ hkResourceContainer* newParent)
{
    hkMemoryResourceContainer* newP = static_cast<hkMemoryResourceContainer*>( newParent );

    // check for circular dependency
    {
        for ( hkResourceContainer* p = newParent; p; p = p->getParent() )
        {
            if ( p == this )
            {
                HK_WARN_ALWAYS( 0xabba4554, "Cannot parent '" << m_name << "' to '" << newP->m_name << "' as this would create a circular dependency ");
                return HK_FAILURE;
            }
        }
    }



    // remove from old parent
    this->addReference();
    {
        int index = m_parent->m_children.indexOf( this );
        m_parent->m_children.removeAtAndCopy( index );
    }
    newP->m_children.pushBack( this );
    m_parent = newP;
    this->removeReference();
    return HK_SUCCESS;
}


hkContainerResourceMap::hkContainerResourceMap(_Inout_ class hkResourceContainer* container)
{
    hkArray<hkResourceHandle*> handles; container->findAllResourceRecursively( handles );

    for ( int i = 0; i < handles.getSize(); i++  )
    {
        hkResourceHandle* handle = handles[i];

        hkStringBuf buffer;
        const char* name = handle->getName( buffer );
        HK_ASSERT_NO_MSG( 0xf032f612, name != buffer.cString() );
        m_resources.insert( name, handle );
    }
}

_Ret_maybenull_ void* hkContainerResourceMap::findObjectByName(_In_z_ const char* objectName, _Outptr_opt_ const hkReflect::Type** klassOut) const
{
    if ( klassOut )
    {
        *klassOut = HK_NULL;
    }

    hkResourceHandle* handle = m_resources.getWithDefault( objectName, HK_NULL);
    if ( !handle )
    {
        return HK_NULL;
    }
    if ( klassOut)
    {
        * klassOut = handle->getObjectType();
    }
    return handle->getObject();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
