// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL !OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Format/Xml/hkXmlWriteFormat.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/Serialize/Detail/hkSerializeDetail.h>
#include <Common/Base/Types/hkEndian.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/Container/StringView/hkStringView.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>
#include <Common/Base/Fwd/hkcstdio.h>
#include <Common/Base/Fwd/hkcstdarg.h>
#include <Common/Base/Config/hkOptionalComponent.h>

#include <Common/Base/Container/PointerMap/hkMap.hxx>

#define DEBUG_LOG_IDENTIFIER "s11n.XmlWriteFormat"
#include <Common/Base/System/Log/hkLog.hxx>

static void s_quoteStringForXml(hkStringBuf& b)
{
    b.replace("&", "&amp;");
    b.replace("<", "&lt;");
    b.replace(">", "&gt;");
    b.replace("\"", "&quot;");
    b.replace("'", "&apos;");
}

struct hkSerialize::XmlWriteFormat::XmlFormatter
{
    HK_DECLARE_CLASS(XmlFormatter, New);
    hkIo::WriteBuffer* m_writer;
    int m_tagsOpen;

    XmlFormatter(_Inout_ hkIo::WriteBuffer* writer) : m_writer(writer), m_tagsOpen(0)
    {
    }

    hkResult write(_In_z_ const char* s, ...)
    {
        if (m_writer->writeRaw("\n", 1) != 1)
        {
            return HK_FAILURE;
        }

        char buf[256];

        const int spacesToWrite = (m_tagsOpen * 2) < 256 ? (m_tagsOpen * 2) : 256;
        hkString::memSet(buf, ' ', spacesToWrite);
        if (spacesToWrite)
        {
            if (m_writer->writeRaw(buf, spacesToWrite) != spacesToWrite)
            {
                return HK_FAILURE;
            }
        }

        va_list l;
        va_start(l, s);
        hkResult res = writeInlineV(s, l);
        va_end(l);
        return res;
    }

    hkResult writeInline(_In_z_ const char* s, ...)
    {
        va_list l;
        va_start(l, s);
        hkResult res = writeInlineV(s, l);
        va_end(l);
        return res;
    }

    hkResult writeInlineV(_In_z_ const char* s, va_list l)
    {
        hkInplaceArray<char, 1024>::Temp buf(1024);
        int numChars;
        int loopsLeft = 10;
        bool successfullyWritten = false;
        do
        {
            va_list copy;
            hk_va_copy(copy, l);
            numChars = vsnprintf(buf.begin(), buf.getSize(), s, l);
            va_end(copy);

            if((numChars >= 0) && (numChars < buf.getSize()))
            {
                successfullyWritten = true;
            }
            else
            {
                buf.setSize(buf.getSize() * 2);

                // Only try a max of 10 times, we will have a line length of 1Mb at that stage anyway
                if (loopsLeft-- <= 0)
                {
                    return HK_FAILURE;
                }
            }
        } while(!successfullyWritten);

        if (m_writer->writeRaw(buf.begin(), numChars) != numChars)
        {
            return HK_FAILURE;
        }

        return HK_SUCCESS;
    }

    void indent()
    {
        m_tagsOpen++;
    }

    void outdent()
    {
        m_tagsOpen--;
    }
};

namespace
{
    // Always returns a printable string, even if the type is null
    _Ret_z_ const char* getPrintableTypeName(_In_ const hkReflect::Type* type, hkStringBuf& aux)
    {
        return type ? type->getFullName(aux) : "null";
    }

    hkUint32 asInt(const hkReflect::Detail::InheritanceInfo* info)
    {
        return info ? info->getPreOrder() << 16 | info->getRightMost() : 0;
    }

    struct PrintFormatVisitor : public hkReflect::TypeVisitor<PrintFormatVisitor, void, hkReflect::Format::Value, hkStringBuf&>
    {
        void visit(_In_opt_ const hkReflect::VoidType*, hkReflect::Format::Value, hkStringBuf& out) { out = "void"; }
        void visit(_In_opt_ const hkReflect::PointerType*, hkReflect::Format::Value, hkStringBuf& out) { out = "pointer"; }
        void visit(_In_opt_ const hkReflect::OpaqueType*, hkReflect::Format::Value, hkStringBuf& out) { out = "incomplete"; }
        void visit(_In_opt_ const hkReflect::RecordType*, hkReflect::Format::Value, hkStringBuf& out) { out = "record"; }

        void visit(_In_opt_ const hkReflect::BoolType*, hkReflect::Format::Value value, hkStringBuf& out)
        {
            hkReflect::Format::IntValue iv(value);
            out.printf("bool%d (%s, %s)",
                 iv.numBits(), iv.isSigned() ? "signed" : "unsigned", iv.isBigEndian() ? "BE" : "LE");
        }

        void visit(_In_opt_ const hkReflect::IntType*, hkReflect::Format::Value value, hkStringBuf& out)
        {
            hkReflect::Format::IntValue iv(value);
            out.printf("%sint%d (%s)", iv.isSigned() ? "" : "u", iv.numBits(), iv.isBigEndian() ? "BE" : "LE");
        }

        void visit(_In_opt_ const hkReflect::FloatType*, hkReflect::Format::Value value, hkStringBuf& out)
        {
            hkReflect::Format::FloatValue fv(value);
            if (fv.get() == hkReflect::Format::OfFloat<float>::Value)
            {
                out = "float";
            }
            else if (fv.get() == hkReflect::Format::OfFloat<double>::Value)
            {
                out = "double";
            }
            else
            {
                out.printf("fp %d bits (%s, %d exp, %d significand %s, %s)",
                    fv.sizeOf(), fv.hasSignBit() ? "sign bit" : "", fv.numBitsExponent(),
                    fv.numBitsSignificand(), fv.hasImplicitOneBit() ? ", implicit one" : "",
                    fv.isBigEndian() ? "BE" : "LE");
            }
        }

        void visit(_In_opt_ const hkReflect::StringType*, hkReflect::Format::Value value, hkStringBuf& out)
        {
            hkReflect::Format::StringValue sv(value);
            out.printf("string (%s)", sv.getFixedCount() ? "buffer" : "ptr");
        }

        void visit(_In_opt_ const hkReflect::ArrayType*, hkReflect::Format::Value value, hkStringBuf& out)
        {
            hkReflect::Format::ArrayValue av(value);
            out.printf("array");
            if (av.getFixedCount())
            {
                out.appendPrintf("(%d)", av.getFixedCount());
            }
        }
    };
}

struct hkSerialize::XmlWriteFormat::TypeWriter : public hkSerialize::TypeWriter
{
    hkSerialize::Detail::TypeWriterMap m_typeMap;
    XmlWriteFormat* m_owner;
    TypeId m_highestWrittenId;

    TypeWriter(_In_ XmlWriteFormat* owner) : m_owner(owner), m_highestWrittenId(0) {}

    void writeTypeTag(_Inout_ XmlFormatter* formatter, _In_z_ const char* tag, _In_ const hkReflect::Type* type, TypeId typeId)
    {
        hkStringBuf nameBuf; const char* name = getPrintableTypeName(type, nameBuf);
        formatter->write("<%s id=\"type%u\" /> <!-- %s -->", tag, typeId, name);
        DLOG("{} ${} // {}", tag, typeId, name);
    }

    void writeAllOutstandingTypes(_Inout_ XmlFormatter* formatter)
    {
        if(m_typeMap.isEmpty())
        {
            return;
        }
        while (const hkReflect::Type* type = m_typeMap.nextToWrite())
        {
            using namespace hkReflect;
            TypeId typeId = m_typeMap.lookupType(type);
            if(typeId > m_highestWrittenId)
            {
                m_highestWrittenId = typeId;
            }

            DLOG_SCOPE("Type ${} // {}", typeId, type->getFullName());

            hkUint32 optBits = TypeDetail::localGetOptionalMask(type) &
                (hkReflect::Opt::FORMAT|hkReflect::Opt::SUBTYPE|hkReflect::Opt::NAME|hkReflect::Opt::VERSION|hkReflect::Opt::TEMPLATE|hkReflect::Opt::SIZE_ALIGN|hkReflect::Opt::FLAGS|hkReflect::Opt::DECLS);

            formatter->write("<type id=\"type%d\">", typeId);
            formatter->indent();

            if (optBits & hkReflect::Opt::NAME)
            {
                const char* s = TypeDetail::localGetOptional<hkReflect::Opt::NAME>(type);
                hkStringBuf nb(s);
                s_quoteStringForXml(nb);
                formatter->write("<name value=\"%s\"/>", nb.cString());
                DLOG("Name {}", s);
            }

            if (const hkReflect::Type* parent = type->getParent())
            {
                TypeId parentId = m_typeMap.enqueueForWrite(parent);
                writeTypeTag(formatter, "parent", parent, parentId);
            }

            if( optBits & hkReflect::Opt::FORMAT )
            {
                hkUint32 optFormat = hkLosslessCast<hkUint32>(TypeDetail::localGetOptional<hkReflect::Opt::FORMAT>(type));
                hkReflect::Format::Value format = hkReflect::Format::Value::create(optFormat);
                hkStringBuf printed;
                PrintFormatVisitor().dispatch(type, format, printed);
                formatter->write("<format value=\"%u\"/> <!-- %s -->", optFormat, printed.cString() );
                if( format.getKind() == hkReflect::KIND_STRING )
                {
                    hkReflect::Format::StringValue sf = format;
                    // String extras will need the type of char on load.
                    m_typeMap.enqueueForWrite(sf.isImmutable() ? hkReflect::getType<const char>() : hkReflect::getType<char>());
                }
            }
            if( optBits & hkReflect::Opt::SUBTYPE )
            {
                if( const ArrayType* at = type->asArray() )
                {
                    int count = at->getFixedCount();
                    TypeId tid = m_typeMap.enqueueForWrite(at->getSubType());

                    DLOG("ArrayOf");
                    writeTypeTag(formatter, "subtype", at->getSubType(), hkLosslessCast<hkUint16>(tid));

                    if(count)
                        DLOG("FixedCount {}", count);
                }
                //else if( const ContainerType* ct = type->isContainer() )
                //{
                //    TypeId tid = m_typeMap.lookupType(ct->getSubType());
                //    DLOG("ContainerOf");
                //    writeTypeTag(formatter, "subtype", ct->getSubType(), hkLosslessCast<hkUint16>(tid));
                //}
                else if( const PointerType* pt = type->asPointer() )
                {
                    TypeId tid = m_typeMap.lookupType(pt->getSubType());
                    DLOG("PointerTo");
                    writeTypeTag(formatter, "subtype", pt->getSubType(), hkLosslessCast<hkUint16>(tid));
                }
                else
                {
                    HK_ASSERT_NO_MSG(0x692d19b8,0);
                }
            }
            if( optBits & hkReflect::Opt::VERSION )
            {
                hkUlong optVersion = TypeDetail::localGetOptional<hkReflect::Opt::VERSION>(type);
                formatter->write("<version value=\"%u\"/>", hkLosslessCast<hkUint32>(optVersion));
                DLOG("Version {}", optVersion);
            }
            if( optBits & hkReflect::Opt::TEMPLATE )
            {
                const hkReflect::Template* optParams = type->getTemplate();
                const int numParams = optParams->getNumParams();

                formatter->write("<parameters count=\"%d\">", numParams);
                formatter->indent();

                DLOG_SCOPE("Template np={}", numParams);

                for(int i=0; i < numParams; i++)
                {
                    const hkReflect::Template::Parameter* param = optParams->getParam(i);

                    if(param->isType())
                    {
                        TypeId typeParam = m_typeMap.enqueueForWrite(param->getAsType());
                        writeTypeTag(formatter, "typeparam", param->getAsType(), typeParam);
                    }
                    else
                    {
                        hkUlong valueParam = param->getAsValue();
                        formatter->write("<valueparam value=\"%u\"/>", hkLosslessCast<hkUint32>(valueParam));
                        DLOG("ValueParam {}", valueParam);
                    }
                }
                formatter->outdent();
                formatter->write("</parameters>");
            }
            if( optBits & hkReflect::Opt::FLAGS )
            {
                hkUlong flags = TypeDetail::localGetOptional<hkReflect::Opt::FLAGS>(type);
                formatter->write("<flags value=\"%u\"/>", hkLosslessCast<hkUint32>(flags));
                DLOG("Flags {}", flags);
            }
            if( optBits & hkReflect::Opt::DECLS )
            {
                const hkReflect::Detail::DeclsArray* fieldArray = TypeDetail::localGetOptional<Opt::DECLS>(type);
                int numFields = fieldArray->getNumFields();
                int numSerializableFields = 0;
                for(int i = 0; i < numFields; i++)
                {
                    const hkReflect::FieldDecl field = fieldArray->getField(i);
                    numSerializableFields += field.isSerializable();
                }
                formatter->write("<fields count=\"%d\">", numSerializableFields);
                formatter->indent();

                for(int i = 0; i < numFields; i++)
                {
                    const hkReflect::FieldDecl field = fieldArray->getField(i);
                    if( field.isSerializable() )
                    {
                        TypeId fieldId = m_typeMap.enqueueForWrite(field.getType()->getParent());
                        hkStringBuf fieldName(field.getName());
                        s_quoteStringForXml(fieldName);
                        hkStringBuf fieldTypeBuf; const char* fieldTypeName = getPrintableTypeName(field.getType(), fieldTypeBuf);

                        // properties turn into data fields for serialization
                        int fieldFlags = (field.getFlags().get() & ~hkReflect::Decl::DECL_PROPERTY_FIELD) | hkReflect::Decl::DECL_DATA_FIELD;
                        formatter->write("<field name=\"%s\" typeid=\"type%d\" flags=\"%d\" /> <!-- %s -->",
                            fieldName.cString(), fieldId, fieldFlags, fieldTypeName); // offset=\"%d\", hkLosslessCast<hkUint16>(field.getOffset())
                        DLOG("Field name={} offset={} flags={}, type=${} // {}",
                            field.getName(), field.getOffset(), field.getFlags(), fieldId, fieldTypeName);
                    }
                }
                formatter->outdent();
                formatter->write("</fields>");
            }

            formatter->outdent();
            formatter->write("</type>");

            if (hasVariantArrays(type))
            {
                // Store the inheritance info of the type to bypass decorators and duplicates.
                m_owner->m_withVariantArrays.insert(asInt(hkReflect::TypeDetail::getInheritance(type)));
            }
        }
    }

    hkSerialize::TypeAndId writeType(_In_ const hkReflect::Type* type) HK_OVERRIDE
    {
        TypeId id = m_typeMap.enqueueForWrite(type);
        writeAllOutstandingTypes(m_owner->m_formatter);
        return hkTupleT::make(type, id);
    }

    hkSerialize::TypeId getHighestWrittenId() const HK_OVERRIDE
    {
        return m_highestWrittenId;
    }

};

hkSerialize::XmlWriteFormat::XmlWriteFormat(_In_opt_ hkSerialize::TypeWriter* typeWriter)
    : m_formatter(HK_NULL)
    , m_typeWriter(typeWriter)
{
    if (!m_typeWriter)
    {
        m_typeWriter = hkRefNew<hkSerialize::TypeWriter>(new TypeWriter(this));
    }
}

hkSerialize::XmlWriteFormat::~XmlWriteFormat()
{
}

void hkSerialize::XmlWriteFormat::beginBundle(_In_ hkIo::WriteBuffer* writeBuffer)
{
    HK_ASSERT_NO_MSG(0x1d674687, m_formatter == HK_NULL);
    m_formatter = new XmlFormatter(writeBuffer);
    m_formatter->writeInline("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n<hktagfile version=\"3\">");
    m_formatter->indent();
}

void hkSerialize::XmlWriteFormat::endBundle()
{
    m_formatter->outdent();
    m_formatter->write("</hktagfile>");
    delete m_formatter;
    m_formatter = HK_NULL;
}

bool hkSerialize::XmlWriteFormat::hasVariantArrays(_In_ const hkReflect::Type* type, _In_ const hkReflect::RecordType* curRecord)
{
    if (const hkReflect::RecordType* rec = type->asRecord())
    {
        if(rec == curRecord)
        {
            return false;
        }

        for( hkReflect::DeclIter<hkReflect::FieldDecl> recIt(rec); recIt.advance(); )
        {
            if (hasVariantArrays(recIt.current().getType(), rec))
            {
                return true;
            }
        }
    }
    else if (const hkReflect::ArrayType* arr = type->asArray())
    {
        if (!arr->getSubType())
        {
            return true;
        }
        return hasVariantArrays(arr->getSubType(), curRecord);
    }
    return false;
}

void hkSerialize::XmlWriteFormat::writeVariantArraySubtypes(const hkReflect::Var& var)
{
    if (var.getType()->asRecord() || var.getType()->asArray())
    {
        if (m_withVariantArrays.contains(asInt(hkReflect::TypeDetail::getInheritance(var.getType()))))
        {
            // Visit recursively.
            if (hkReflect::RecordVar rec = var)
            {
                for( hkReflect::DeclIter<hkReflect::FieldDecl> recIt(rec.getType()); recIt.advance(); )
                {
                    writeVariantArraySubtypes( rec[recIt.current()] );
                }
            }
            else if (hkReflect::ArrayVar arr = var)
            {
                if (!arr.getType()->getSubType())
                {
                    // Write out the subtype.
                    m_typeWriter->writeType(arr.getSubType());
                }
                // Visit elements.
                for (int i = 0; i < arr.getCount(); ++i)
                {
                    writeVariantArraySubtypes(arr[i]);
                }
            }
        }
    }
}

hkResult hkSerialize::XmlWriteFormat::writeInternal(const hkReflect::Var& topVar, VarId vid, VarId annotated, _Inout_ IdFromVar* idFromVar)
{
    hkResult res;
    {
        const hkReflect::Type* dstType = topVar.getType();
        TypeId tid = m_typeWriter->writeType(dstType).m_1;
        writeVariantArraySubtypes(topVar);

        hkStringBuf noteBuf;
        if (annotated)
        {
            noteBuf.format("annotates=\"object{}\"", annotated);
        }

        const char* tag = annotated == 0 ? "object" : "note";

        hkStringBuf nameBuf;
        res = m_formatter->write("<%s id=\"object%d\" typeid=\"type%d\" %s> <!-- %s -->",
            tag, vid, tid, noteBuf.cString(), dstType->getFullName(nameBuf));

        m_formatter->indent();

        writeVarContents(topVar, idFromVar, 1);

        m_formatter->outdent();
        m_formatter->write("</%s>", tag);
    }

    return res;
}

hkResult hkSerialize::XmlWriteFormat::write(const hkReflect::Var& topVar, IdFromVar& idFromVar)
{
    VarId vid = idFromVar.lookup(topVar);
    idFromVar.writing(vid);
    return writeInternal(topVar, vid, 0, &idFromVar);
}


hkResult hkSerialize::XmlWriteFormat::writeNote(VarId vid, const hkReflect::Var& note, IdFromVar& idFromVar)
{
    return writeInternal(note, idFromVar.lookup(note), vid, &idFromVar);
}

namespace
{
    struct WriteVarVisitor : public hkReflect::VarVisitor<WriteVarVisitor, void, bool, const hk::Presets*>
    {
        typedef hkReflect::VarVisitor<WriteVarVisitor, void, bool, const hk::Presets*> ParentType;
        void dispatch(const hkReflect::Var& obj, bool inField)
        {
            ParentType::dispatch(obj, inField, obj.getType()->findAttribute<hk::Presets>());
        }

        inline void writeInlineable(bool inField, _In_z_ const char* s, ...)
        {
            va_list l;
            va_start(l, s);
            if (!inField)
            {
                m_formatter->write("");
            }
            m_formatter->writeInlineV(s, l);
            va_end(l);
        }

        inline void writePreset(const hkReflect::Var& obj, _In_opt_ const hk::Presets* presets)
        {
            if (presets)
            {
                if (const char* presetName = presets->getNameByPreset(obj))
                {
                    m_formatter->writeInline("<!-- %s -->", presetName);
                }
            }
        }

        void visit(const hkReflect::RecordVar& rec, bool inField, _In_opt_ const hk::Presets*)
        {
            if (inField)
            {
                m_formatter->indent();
            }

            m_formatter->write("<record> <!-- %s -->", rec.getType()->getName());
            m_formatter->indent();

            for( hkReflect::DeclIter<hkReflect::FieldDecl> fieldIter( rec.getType() ); fieldIter.advance(); )
            {
                const hkReflect::FieldDecl field = fieldIter.current();
                if (field.isSerializable() && field.getName()) 
                {
                    hkStringBuf fieldName(field.getName());
                    s_quoteStringForXml(fieldName);

                    m_formatter->write("<field name=\"%s\">", fieldName.cString());
                    dispatch( rec[field], true);
                    m_formatter->writeInline("</field>", fieldName.cString());
                }
            }

            m_formatter->outdent();
            m_formatter->write("</record>");
            if (inField)
            {
                m_formatter->outdent();
                m_formatter->write("");
            }
        }

        void visit(const hkReflect::ArrayVar& array, bool inField, _In_opt_ const hk::Presets*)
        {
            if (inField)
            {
                m_formatter->indent();
            }
            // Type must already be written
            const int memberTypeId = m_typeWriter->writeType(array.getSubType()).m_1;
            hkStringBuf nameBuf; const char* name = getPrintableTypeName(array.getSubType(), nameBuf);
            m_formatter->write("<array count=\"%d\" elementtypeid=\"type%d\"> <!-- ArrayOf %s -->",
                array.getCount(), memberTypeId, name);
            m_formatter->indent();
            if (array.getSubType())
            {
                const hk::Presets* elemPresets = array.getSubType()->findAttribute<hk::Presets>();
                for (int index = 0; index < array.getCount(); index++)
                {
                    ParentType::dispatch(array[index], false, elemPresets);
                }
            }
            m_formatter->outdent();
            m_formatter->write("</array>");
            if (inField)
            {
                m_formatter->outdent();
                m_formatter->write("");
            }
        }

        void visit(const hkReflect::VoidVar&, bool, _In_opt_ const hk::Presets*) { HK_UNREACHABLE( 0x1aa6a632, "Void field found" ); }

        void visit(const hkReflect::IntVar& iv, bool inField, _In_opt_ const hk::Presets* presets)
        {
            hkReflect::IntValue val = iv.getValue();
            if (iv.getType()->isSigned())
            {
                const hkInt64 v = val.convertTo<hkInt64>();
                writeInlineable(inField, "<integer value=\"%lld\"/>", v);
            }
            else
            {
                const hkUint64 v = val.convertTo<hkUint64>();
                writeInlineable(inField, "<integer value=\"%llu\"/>", v);
            }
            writePreset(iv, presets);
        }

        void visit(const hkReflect::FloatVar& fv, bool inField, _In_opt_ const hk::Presets* presets)
        {
            double realVal = fv.getValue();
            //hkUint64Le hexDoubleLe = *reinterpret_cast<hkUint64*>(&realVal);
            
            hkUint64 hexDoubleLe = *reinterpret_cast<hkUint64*>(&realVal);

            writeInlineable(inField, "<real dec=\"%lg\" hex=\"#%llx\"/>", fv.getValue().m_value, hexDoubleLe);
            writePreset(fv, presets);
        }

        void visit(const hkReflect::StringVar& sv, bool inField, _In_opt_ const hk::Presets* presets)
        {
            hkReflect::StringValue val = sv.getValue();
            if(val)
            {
                hkStringBuf valQuoted(val);
                s_quoteStringForXml(valQuoted);

                writeInlineable(inField, "<string value=\"%s\"/>", valQuoted.cString());
            }
            else
            {
                writeInlineable(inField, "<string/>");
            }
            writePreset(sv, presets);
        }

        void visit(const hkReflect::BoolVar& bv, bool inField, _In_opt_ const hk::Presets*)
        {
            writeInlineable(inField, "<bool value=\"%s\"/>", bv.getValue() ? "true" : "false");
        }

        void visit(const hkReflect::PointerVar& pv, bool inField, _In_opt_ const hk::Presets*)
        {
            auto id = m_idFromVar->pointer(pv);
            DLOG("  //Ptr id=#{}", id);
            writeInlineable(inField, "<pointer id=\"object%d\"/>", id);
        }

        hkSerialize::XmlWriteFormat::XmlFormatter* m_formatter;
        hkSerialize::IdFromVar* m_idFromVar;
        hkSerialize::TypeWriter* m_typeWriter;
    };
}

void hkSerialize::XmlWriteFormat::writeVarContents(const hkReflect::Var& firstObject, _In_ IdFromVar* idFromVar, int numObjects)
{
    WriteVarVisitor visitor;
    visitor.m_formatter = m_formatter;
    visitor.m_idFromVar = idFromVar;
    visitor.m_typeWriter = m_typeWriter;

    hkReflect::Var currentObject(firstObject);
    int stride = firstObject.getType()->getSizeOf();
    for(int i=0; i < numObjects; i++)
    {
        currentObject.setAddress(hkAddByteOffset(firstObject.getAddress(), i*stride));

        visitor.dispatch(currentObject, false);
    }
}

#ifndef HK_DYNAMIC_DLL
static
#endif
hkRefNew<hkSerialize::WriteFormat> xmlTagfileCreateWrite()
{
    HK_OPTIONAL_COMPONENT_MARK_USED(hkWriteFormatXmlTagfile);
    return new hkSerialize::XmlWriteFormat();
}

HK_OPTIONAL_COMPONENT_DEFINE(hkWriteFormatXmlTagfile, hkSerialize::Detail::fileFormatXmlTagfile.m_writeFormatCreateFunc, xmlTagfileCreateWrite);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
