// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL !OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Format/Xml/hkXmlReadFormat.h>
#include <Common/Base/Config/hkOptionalComponent.h>
#include <Common/Base/Container/Set/hkSet.h>
#include <Common/Base/Memory/Allocator/Transient/hkTransientAllocator.h>
#include <Common/Base/Reflect/Builder/hkRecordLayout.h>
#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Reflect/TypeReg/Detail/hkTypeRegDetail.h>
#include <Common/Base/Serialize/Detail/hkSerializeDetail.h>
#include <Common/Base/Serialize/Detail/hkIndexedBundle.h>
#include <Common/Base/Serialize/Util/Xml/hkXmlStreamParser.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/Types/hkEndian.h>
#include <Common/Base/Fwd/hkcstdio.h>

#define DEBUG_LOG_IDENTIFIER "s11n.XmlReadFormat"
#include <Common/Base/System/Log/hkLog.hxx>
#include <Common/Base/Reflect/Util/hkReflectUtil.h>

#define CHECK_READ_INT_ATTRIBUTE(attribName, outInt) if(!streamParser.getIntAttribute(attribName, outInt).isSuccess()) { return HK_FAILURE; }
#define CHECK_READ_STRING_ATTRIBUTE(attribName, outString) do { hkStringView value; if(!streamParser.getValue(attribName, value).isSuccess()) { return HK_FAILURE; } if(!streamParser.decodeString(value, outString).isSuccess()) { return HK_FAILURE; } if(outString.startsWith("\"")) { outString.chompStart(1); outString.chompEnd(1); } } while(0)
#define CHECK_READ_ID(attribName, prefix, outInt) \
    do { \
        hkStringView value; \
        if(!streamParser.getValue(attribName, value).isSuccess()) { return HK_FAILURE; } \
        if(value.beginsWith("\"")) { value = value.ltrim(1).rtrim(1); } \
        if(!value.beginsWith(prefix)) { return HK_FAILURE; } \
        value = value.ltrim(hkString::strLen(prefix)); \
        hkInt64 res; \
        if(!hkXmlStreamParser::parseInt(value, res).isSuccess()) { return HK_FAILURE; } \
        outInt = hkLosslessCast<int>(res); \
    } while (0);
#define BLOCK_NAME_MATCHES(a) (streamParser.getBlockName() == a)

struct hkSerialize::XmlReadFormat::Impl
{
    HK_DECLARE_CLASS(Impl, New);
    Impl();
    hkViewPtr<Bundle> read(hkIo::ReadBuffer& rb);
    hkViewPtr<Bundle> view(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_opt_ hkUlong* usedOut);
protected:
    struct BundleXml : public hkSerialize::Detail::DefaultBundle
    {
        BundleXml() : m_allocator(hkMemHeapAllocator()) {}
        void clear()
        {
            m_allocator.clear();
            hkSerialize::Detail::DefaultBundle::clear();
        }
        hkTransientAllocator m_allocator;
    };

    bool m_readHeader;

    typedef void (Detail::OldDefaultIndexedBundle::*OnRead)(VarId id, _In_ const void* addr, TypeId tid);
    hkResult parseObject(hkXmlStreamParser& streamParser, bool isNote);
    hkResult parseType(hkXmlStreamParser& streamParser);
    hkResult parseArray(hkXmlStreamParser& streamParser);
    hkResult parseImport(hkXmlStreamParser& streamParser);
    hkResult parseExport(hkXmlStreamParser& streamParser);

    hkResult recalcOutstandingTypes();
    hkResult getNextElement(hkXmlStreamParser& streamParser);
    
    hkResult advanceUntilEndBlock(hkXmlStreamParser& streamParser, _In_z_ const char* name);

    int addString(_In_z_ const char* str);

    struct AllocationImpl : public hkReflect::Detail::AllocationImpl
    {
        AllocationImpl(_In_ Impl* self) : m_self(self) {}
        virtual hkReflect::Var allocate(_In_ const hkReflect::Type* type) const HK_OVERRIDE;
        virtual hkReflect::Var allocateForClone(const hkReflect::Var& src, _In_ const hkReflect::Type* type) const HK_OVERRIDE;
        virtual void deallocate(_Inout_ void* target, _In_ const hkReflect::Type* type) const HK_OVERRIDE;
        Impl* m_self;
    };
    friend struct AllocationImpl;
    AllocationImpl m_allocationImpl;
    hkTransientAllocator m_typesAllocator;

    BundleXml m_bundle;

    hkSet<int> m_typesNeedingLayoutCalculation;

    hkResult recursivelyLoad(hkXmlStreamParser& streamParser, hkReflect::Var const& v);
    void pushExtrasIntoBundle();

    struct ArrayImpl : public hkReflect::Detail::ArrayImpl
    {
        ArrayImpl(_In_ const XmlReadFormat::Impl* parent) : m_parent(parent) {}
        const XmlReadFormat::Impl* m_parent;

        virtual hkResult getValue(_In_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const HK_OVERRIDE;
        virtual hkResult setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int nelem) const HK_OVERRIDE;
    };

    struct StringImpl : public hkReflect::Detail::StringImpl
    {
        StringImpl(_In_ const XmlReadFormat::Impl* parent) : m_parent(parent) {}
        const XmlReadFormat::Impl* m_parent;

        virtual hkResult setValue(_Inout_ void* string, _In_ const hkReflect::StringType* type, hkReflect::StringValue newValue) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* string, _In_ const hkReflect::StringType* type, _Out_ hkReflect::StringValue* val) const HK_OVERRIDE;
    };

    ArrayImpl m_arrayImpl;
    StringImpl m_stringImpl;
    typedef hkTuple<const void*, TypeId, int> Extra;
    hkArray<Extra> m_extras;

    // Id of typeof(char) in the Bundle.
    TypeId m_charTypeId;

    // First ID of extras in the final bundle.
    int m_extraStartId;
};

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    static const hkReflect::Detail::VoidImpl s_readType_void_impl;
    static const hkReflect::Detail::OpaqueImpl s_readType_opaque_impl;
}

namespace hkSerialize
{

    HK_DETAIL_DIAG_MSVC_PUSH();

    // Disable warning, it is safe to use 'this' as long as it is not dereferenced
    HK_DETAIL_DIAG_MSVC_OFF(4355);

    XmlReadFormat::Impl::Impl()
        : m_allocationImpl(this)
        , m_typesAllocator(hkMemHeapAllocator())
        , m_arrayImpl(this)
        , m_stringImpl(this)
    {
    }

    HK_DETAIL_DIAG_MSVC_POP();

    hkResult XmlReadFormat::Impl::getNextElement(hkXmlStreamParser& streamParser)
    {
        hkXmlStreamParser::Token tok = streamParser.advance();
        for (; (tok != hkXmlStreamParser::TOKEN_ERROR) && (tok != hkXmlStreamParser::TOKEN_EOF); tok = streamParser.advance())
        {
            if ((tok == hkXmlStreamParser::TOKEN_BLOCK_START) || (tok == hkXmlStreamParser::TOKEN_BLOCK_START_END))
            {
                return HK_SUCCESS;
            }
        }

        return HK_FAILURE;
    }

    hkResult XmlReadFormat::Impl::advanceUntilEndBlock(hkXmlStreamParser& streamParser, _In_z_ const char* name)
    {
        hkXmlStreamParser::Token tok = streamParser.advance();
        for (; (tok != hkXmlStreamParser::TOKEN_ERROR) && (tok != hkXmlStreamParser::TOKEN_EOF); tok = streamParser.advance())
        {
            if (tok == hkXmlStreamParser::TOKEN_BLOCK_START)
            {
                return HK_SUCCESS;
            }
            else if (tok == hkXmlStreamParser::TOKEN_BLOCK_START_END)
            {
                if (streamParser.getBlockName() == name)
                {
                    streamParser.advance();
                    return HK_FAILURE;
                }
                else
                {
                    return HK_SUCCESS;
                }
            }
            else if (tok == hkXmlStreamParser::TOKEN_BLOCK_END)
            {
                if (streamParser.getBlockName() == name)
                {
                    streamParser.advance();
                    return HK_FAILURE;
                }
            }
        }

        return HK_FAILURE;
    }

    hkViewPtr<Bundle> XmlReadFormat::Impl::view(_In_reads_bytes_(len) const void* buf, hkUlong len, _Out_opt_ hkUlong* usedOut)
    {
        hkIo::ReadBuffer stream(buf, hkLosslessCast<int>(len));
        hkViewPtr<Bundle> ret = read(stream);
        if( usedOut )
        {
            *usedOut = stream.tell();
        }
        return ret;
    }

    hkViewPtr<Bundle> XmlReadFormat::Impl::read(hkIo::ReadBuffer& reader)
    {
        m_bundle.clear();
        m_charTypeId = 0;

        m_extras.expandOne(); // We need to keep the zero value for empty extras

        DLOG_SCOPE("hkSerialize::XmlReadFormat::read()");
        hkXmlStreamParser streamParser(reader);

        {
            hkResult res = getNextElement(streamParser);
            if (res.isFailure() || streamParser.getBlockName() != "hktagfile")
            {
                Log_Warning("File does not begin with a 'tagfile' element");
                return HK_NULL;
            }
        }

        int versionInt;
        if(!streamParser.getIntAttribute("version", versionInt).isSuccess() || (versionInt != 3))
        {
            return HK_NULL;
        }

        // If we get this far, it claims to be the correct version
        DLOG("HkTagfile version {}\n", versionInt);

        while(advanceUntilEndBlock(streamParser, "hktagfile").isSuccess())
        {
            if (BLOCK_NAME_MATCHES("type"))
            {
                hkResult res = parseType(streamParser);
                if(!res.isSuccess())
                {
                    return HK_NULL;
                }
            }
            else if (BLOCK_NAME_MATCHES("object"))
            {
                recalcOutstandingTypes();
                hkResult res = parseObject(streamParser, false);
                if(!res.isSuccess())
                {
                    return HK_NULL;
                }
            }
            else if (BLOCK_NAME_MATCHES("import"))
            {
                // Leave these for now
                hkResult res = parseImport(streamParser);
                if(!res.isSuccess())
                {
                    return HK_NULL;
                }
            }
            else if (BLOCK_NAME_MATCHES("export"))
            {
                // Leave these for now
                hkResult res = parseExport(streamParser);
                if(!res.isSuccess())
                {
                    return HK_NULL;
                }
            }
            else if (BLOCK_NAME_MATCHES("note"))
            {
                recalcOutstandingTypes();
                hkResult res = parseObject(streamParser, true);
                if(!res.isSuccess())
                {
                    return HK_NULL;
                }
            }
            else
            {
                // Unknown type?
                return HK_NULL;
            }
        }

        pushExtrasIntoBundle();

        return &m_bundle;
    }

    hkResult XmlReadFormat::Impl::parseType(hkXmlStreamParser& streamParser)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        {
            int typeId;
            int parentId;
            int format = 0;
            CHECK_READ_ID("id", "type", typeId);

            DLOG_SCOPE("Type ${} ", typeId);

            hkReflect::TypeBuilder builder;
            builder.setTypeWorld(this);
            Detail::TypeRelocs reloc;
            reloc.setParent(0);

            while (advanceUntilEndBlock(streamParser,"type").isSuccess() )
            {
                if (BLOCK_NAME_MATCHES("parent"))
                {
                    CHECK_READ_ID("id", "type", parentId);
                    reloc.setParent(parentId);
                    DLOG("Parent ${}", parentId);
                }
                else if (BLOCK_NAME_MATCHES("format"))
                {
                    CHECK_READ_INT_ATTRIBUTE("value", format);

                    builder.addItem<hkReflect::Opt::FORMAT>(format);
                    DLOG("Format {}", format);

                    hkReflect::Detail::SizeAlign sizeAlign(0, 0);
                    builder.addItem<hkReflect::Opt::SIZE_ALIGN>(sizeAlign.asUlong());

                    builder.addItem<hkReflect::Opt::ALLOC_IMPL>(&m_allocationImpl);
                    const hkReflect::Detail::Impl* impl = HK_NULL;
                    int size = 0;
                    int align = 0;
                    switch( format & 0xf )
                    {
                    case hkReflect::KIND_VOID:
                        {
                            impl = &s_readType_void_impl;
                            break;
                        }
                    case hkReflect::KIND_OPAQUE:
                        {
                            impl = &s_readType_opaque_impl;
                            break;
                        }
                    case hkReflect::KIND_BOOL:
                        {
                            impl = &hkReflect::Detail::BoolImplN<hkUint8>::s_instance;
                            size = hkSizeOf(hkUint8);
                            align = HK_ALIGN_OF(hkUint8);
                            break;
                        }
                    case hkReflect::KIND_STRING:
                        {
                            impl = &m_stringImpl;
                            size = hkSizeOf(int);
                            align = HK_ALIGN_OF(int);
                            break;
                        }
                    case hkReflect::KIND_INT:
                    case hkReflect::KIND_FLOAT:
                        {
                            const hkReflect::Type* t = hkReflect::Detail::BuiltinTypeReg::getInstance().builtinFromFormat(format);
                            HK_ASSERT_NO_MSG(0x23b16a62,t);
                            impl = t->getImpl();
                            size = t->getSizeOf();
                            align = t->getAlignOf();
                            break;
                        }
                    case hkReflect::KIND_RECORD:
                        {
                            builder.addItem<hkReflect::Opt::DECLS>(HK_NULL);
                            impl = &hkReflect::Detail::HavokRecordImpl::s_instance;
                            m_typesNeedingLayoutCalculation.insert(typeId);

                            break;
                        }
                    case hkReflect::KIND_POINTER:
                        {
                            impl = m_bundle.pointerImpl();
                            size = hkSizeOf(int);
                            align = HK_ALIGN_OF(int);
                            break;
                        }
                    case hkReflect::KIND_ARRAY:
                        {
                            const int fixedCount = format >> 8;
                            if (fixedCount > 0)
                            {
                                impl = &hkReflect::Detail::RepeatImpl::s_instance;
                                // Need to hold the sizeof calculation until we have the type resolved
                            }
                            else
                            {
                                impl = &m_arrayImpl;
                                size = hkSizeOf(int);
                                align = HK_ALIGN_OF(int);
                            }
                            break;
                        }
                    default:
                        HK_ASSERT_NO_MSG(0x7ea87b27,0);
                    }
                    builder.addItem<hkReflect::Opt::IMPL>(impl);
                    builder.setItem<hkReflect::Opt::SIZE_ALIGN>(hkReflect::Detail::SizeAlign(size, align).asUlong());
                }
                else if (BLOCK_NAME_MATCHES("name"))
                {
                    hkStringBuf name;
                    CHECK_READ_STRING_ATTRIBUTE("value", name);
                    const char* internedName = hkReflectUtil::internCopy(name);
                    builder.addItem<hkReflect::Opt::NAME>(internedName);
                    DLOG("Name {}", internedName);
                }
                else if (BLOCK_NAME_MATCHES("flags"))
                {
                    int flags;
                    CHECK_READ_INT_ATTRIBUTE("value", flags);
                    builder.addItem<hkReflect::Opt::FLAGS>(flags);
                    DLOG("Flags {}", flags);
                }
                else if (BLOCK_NAME_MATCHES("subtype"))
                {
                    int subTypeId;
                    CHECK_READ_ID("id", "type", subTypeId);

                    builder.addItem<hkReflect::Opt::SUBTYPE>(HK_NULL);

                    switch( format & 0xf )
                    {
                    case hkReflect::KIND_ARRAY:
                    //case hkReflect::KIND_CONTAINER:
                    case hkReflect::KIND_POINTER:
                        {
                            reloc.addSubType(subTypeId);
                            DLOG("Subtype ${}", subTypeId);
                            if( ((format & 0xf) ==hkReflect::KIND_ARRAY) && (format>>8) )
                            {
                                
                                DLOG("FixedCount {}", format>>8);
                            }
                            m_typesNeedingLayoutCalculation.insert(subTypeId);
                            break;
                        }
                    default:
                        DLOG("Invalid format for subtype");
                        return HK_FAILURE;
                    }
                }
                else if (BLOCK_NAME_MATCHES("fields"))
                {
                    int fieldCount;
                    CHECK_READ_INT_ATTRIBUTE("count", fieldCount);

                    DLOG_SCOPE("Fields {}", fieldCount);
                    while (advanceUntilEndBlock(streamParser,"fields").isSuccess())
                    {
                        // Fields
                        hkStringBuf fieldName;
                        int fieldType;
                        int fieldFlags;

                        CHECK_READ_STRING_ATTRIBUTE("name", fieldName);
                        CHECK_READ_ID("typeid", "type", fieldType);
                        CHECK_READ_INT_ATTRIBUTE("flags", fieldFlags);

                        DLOG("Field name {} typeid {} flags {} offset {}", fieldName, fieldType, fieldFlags, /*fieldOffset*/0);
                        builder.addMember( hkReflectUtil::internCopy(fieldName), /*fieldOffset*/0, hkLosslessCast<hkUint16>(fieldFlags), HK_NULL);
                        reloc.addField(fieldType);
                    }
                }
                else if (BLOCK_NAME_MATCHES("version"))
                {
                    int version;
                    CHECK_READ_INT_ATTRIBUTE("value", version);
                    builder.addItem<hkReflect::Opt::VERSION>(version);
                    DLOG("Version {}", version);
                }
                else if (BLOCK_NAME_MATCHES("parameters"))
                {
                    int numParams;
                    CHECK_READ_INT_ATTRIBUTE("count", numParams);

                    DLOG_SCOPE("Num Parameters {}", numParams);

                    builder.setItem<hkReflect::Opt::TEMPLATE>(HK_NULL); // force template opt, even if no params

                    while (advanceUntilEndBlock(streamParser,"parameters").isSuccess())
                    {
                        // Parameters, can be type or value
                        if (BLOCK_NAME_MATCHES("typeparam"))
                        {
                            int typeParam;
                            CHECK_READ_ID("id", "type", typeParam);
                            builder.addTypeTemplateParam(HK_NULL, "T"); 
                            reloc.addTemplateParam(typeParam);
                            DLOG("TypeParam ${}", typeParam);
                        }
                        else if (BLOCK_NAME_MATCHES("valueparam"))
                        {
                            int valueParam;
                            CHECK_READ_INT_ATTRIBUTE("value", valueParam);
                            builder.addValueTemplateParam(valueParam, "N"); 
                            DLOG("ValueParam {}", valueParam);
                        }
                        else
                        {
                            return HK_FAILURE;
                        }
                    }
                }
                else
                {
                    DLOG("Unknown element {}", streamParser.getBlockName());
                    return HK_FAILURE;
                }
            }

            hkReflect::Type* type;
            if (builder.allocate(&m_typesAllocator, &type).isFailure())
            {
                Log_Error("Failed to allocate type from builder");
                return HK_FAILURE;
            }

            m_bundle.setType(typeId, type, reloc);

            return HK_SUCCESS;
        }
    }

    hkResult XmlReadFormat::Impl::recalcOutstandingTypes()
    {
        hkArray<int>::Temp typesCalculated;

        hkArrayView<const hkReflect::Type*> types = m_bundle.types();
        for (hkSet<int>::Iterator it = m_typesNeedingLayoutCalculation.getIterator(); m_typesNeedingLayoutCalculation.isValid(it); it = m_typesNeedingLayoutCalculation.getNext(it))
        {
            const int calculateTypeIndex = m_typesNeedingLayoutCalculation.getElement(it);
            if (calculateTypeIndex < types.getSize())
            {
                hkReflect::Type* typeToBeRecomputed = const_cast<hkReflect::Type*>(types[calculateTypeIndex]);
                if (typeToBeRecomputed)
                {
                    hkReflect::RecordLayout::recomputeNative(typeToBeRecomputed);
                }
                typesCalculated.pushBack(calculateTypeIndex);
            }
        }

        for (int i = 0; i < typesCalculated.getSize(); i++)
        {
            m_typesNeedingLayoutCalculation.remove(typesCalculated[i]);
        }

        return HK_SUCCESS;
    }


    hkResult XmlReadFormat::Impl::recursivelyLoad(hkXmlStreamParser& streamParser, hkReflect::Var const& v)
    {
        // This is kind of a type visitor but not really, we are being driven by the text in the file, not the type
        if (BLOCK_NAME_MATCHES("record"))
        {
            // Record
            hkReflect::RecordVar rv(v);

            // Each child element is a member
            // XXX sort this out
            while(advanceUntilEndBlock(streamParser,"record").isSuccess())
            {
                if (BLOCK_NAME_MATCHES("field"))
                {
                    hkStringBuf name;
                    CHECK_READ_STRING_ATTRIBUTE("name", name);
                    if( const hkReflect::FieldDecl rf = rv.getType().get()->findField(name, true) )
                    {
                        hkReflect::Var fieldValue = rv[rf];
                        if (advanceUntilEndBlock(streamParser,"field").isSuccess())
                        {
                            if (recursivelyLoad(streamParser, fieldValue).isFailure())
                            {
                                return HK_FAILURE;
                            }
                        }
                    }
                }
                else
                {
                    return HK_FAILURE;
                }
            }
        }
        else if (BLOCK_NAME_MATCHES("integer"))
        {
            // Integer
            hkReflect::IntVar iv(v);
            hkStringBuf stringVal;
            CHECK_READ_STRING_ATTRIBUTE("value", stringVal);
            bool negative = false;
            hkUint64 magnitude;
            if(stringVal[0] == '-')
            {
                negative = true;
                magnitude = hkString::atoull(stringVal.cString() + 1);
            }
            else
            {
                magnitude = hkString::atoull(stringVal);
            }
            hkReflect::IntValue intVal; intVal.set(magnitude, negative);
            iv.setValue(intVal);
        }
        else if (BLOCK_NAME_MATCHES("real"))
        {
            // Real
            hkReflect::FloatVar fv(v);
            hkStringBuf stringRealVal;
            hkStringBuf stringHexVal;

            CHECK_READ_STRING_ATTRIBUTE("hex", stringHexVal);
            CHECK_READ_STRING_ATTRIBUTE("dec", stringRealVal);

            if (stringHexVal[0] != '#')
            {
                return HK_FAILURE;
            }


            
            double doubleRealVal = 0.0;
            hkInt64 doubleHexVal = 0;
            sscanf(stringRealVal, "%lg", &doubleRealVal);
            sscanf(stringHexVal + 1, "%llx", &doubleHexVal);

            const double doubleRealValFromHex = *reinterpret_cast<double*>(&doubleHexVal);

            // For NaNs, just use the hex
            if( !hkMath::isFinite( doubleRealValFromHex ) )
            {
                fv.setValue( doubleRealValFromHex );
            }
            else if( doubleRealVal > 0.0 )
            {
                if( doubleRealValFromHex != doubleRealVal )
                {
                    // Conversion is not exact
                    if( fv.getType()->getFormat().equalsValue( hkReflect::Format::OfFloat<hkFloat32>::Value ) )
                    {
                // Arbitrary, 0.01% is considered equal -- use exact float val
                        *(hkInt32*)(fv.getAddress()) = hkMath::doubleToFloat( doubleHexVal );
                    }
                    else
                {
                        // Set from hex directly
                        fv.setValue( doubleRealValFromHex );
                    }
                }
                else
                {
                    // String val has been modified, use it
                    fv.setValue( doubleRealVal );
                }
            }
            else
            {
                // Zero, just use zero
                fv.setValue( doubleRealValFromHex );
            }
        }
        else if (BLOCK_NAME_MATCHES("bool"))
        {
            // Bool
            hkReflect::BoolVar bv(v);
            hkStringBuf stringVal;
            CHECK_READ_STRING_ATTRIBUTE("value", stringVal);
            bool val = (hkString::strCasecmp(stringVal, "true") == 0);
            bv.setValue(val);
        }
        else if (BLOCK_NAME_MATCHES("string"))
        {
            // String
            hkReflect::StringVar sv(v);
            hkStringView stringVal;
            if (streamParser.getValue("value", stringVal).isSuccess())
            {
                hkStringBuf tempString(stringVal);
                streamParser.decodeString(stringVal, tempString);
                if (tempString.startsWith("\""))
                {
                    tempString.chompStart(1);
                    tempString.chompEnd(1);
                }
                const int index = addString(tempString);

                if (index < 0)
                {
                    return HK_FAILURE;
                }

                *reinterpret_cast<int*>(sv.getAddress()) = index;
            }
            else
            {
                *reinterpret_cast<int*>(sv.getAddress()) = 0;
            }
        }
        else if (BLOCK_NAME_MATCHES("array"))
        {
            int typeId;
            int count;
            CHECK_READ_ID("elementtypeid", "type", typeId);
            CHECK_READ_INT_ATTRIBUTE("count", count);

            // Array
            hkReflect::ArrayVar av(v);
            hkReflect::ArrayValue arrVal;

            if(av.getType()->getFixedCount())
            {
                // Write directly into the array.
                arrVal = av.getValue();
            }
            else if (count)
            {
                // Write into an extra.
                const hkReflect::Type* type = m_bundle.type(typeId);

                VarId arrayId = m_extras.getSize();
                *reinterpret_cast<hkUint32Le*>(av.getAddress()) = arrayId;

                const int elemSize = type->getSizeOf();
                const int sizeToAllocate = elemSize * count;
                const void* dstAddr = m_bundle.m_allocator.blockAlloc(sizeToAllocate);
                
                m_extras.pushBack(hkTupleT::make(dstAddr, typeId, count));
                arrVal = hkReflect::ArrayValue(dstAddr, count, type);
            }
            else
            {
                *reinterpret_cast<hkUint32Le*>(av.getAddress()) = 0;
            }

            if (arrVal.getCount())
            {
                int i = 0;
                while (advanceUntilEndBlock(streamParser, "array").isSuccess())
                {
                    hkReflect::Var elem = arrVal[i];
                    if (recursivelyLoad(streamParser, elem).isFailure())
                    {
                        return HK_FAILURE;
                    }
                    ++i;
                }
            }
        }
        else if (BLOCK_NAME_MATCHES("pointer"))
        {
            // Pointer
            hkReflect::PointerVar pv(v);

            hkStringBuf target;
            CHECK_READ_STRING_ATTRIBUTE("id", target);

            // The target must be "objectX" or "noteX".
            hkStringView tag;
            if (target.startsWith("object"))
            {
                tag = hkStringView::fromLiteral("object");
            }
            else if (target.startsWith("note"))
            {
                tag = hkStringView::fromLiteral("note");
            }
            else
            {
                // Invalid tag.
                return HK_FAILURE;
            }

            // Extracts the id from the target.
            hkStringView idStr = target;
            idStr = idStr.ltrim(tag.getSize());
            hkInt64 val;
            if (!hkXmlStreamParser::parseInt(idStr, val).isSuccess())
            {
                return HK_FAILURE;
            }
            auto id = hkLosslessCast<VarId>(val);

            // Needs to make space for all pointed objects, even imported ones, so that when extras are added at the
            // end of the Bundle they will not overlap with the pointers indices.
            m_bundle.makeSpace(id);

            *(hkUint32Le*)(v.getAddress()) = id;
        }
        return HK_SUCCESS;
    }

    hkResult XmlReadFormat::Impl::parseObject(hkXmlStreamParser& streamParser, bool isNote)
    {
        int objectId;
        int typeId;

        CHECK_READ_ID("id", "object", objectId);
        CHECK_READ_ID("typeid", "type", typeId);

        int annotated = 0;
        if (isNote)
        {
            if (streamParser.hasAttribute("annotates"))
            {
                CHECK_READ_ID("annotates", "object", annotated);
            }
            else
            {
                // 2015.1 note. Cannot easily load since the id is implicit.
                CHECK_READ_ID("id", "note", annotated);
                Log_Warning("Found deprecated note 'note{}'. Must be assigned an explicit unique id and an 'annotates'"
                    " attribute, e.g. '<note id=\"object99\" annotates=\"object{}\">'", annotated, annotated);
                return HK_FAILURE;
            }
        }

        const char* tag = isNote ? "note" : "object";
        DLOG_SCOPE("{} type=${} id={} annotated={}//", tag, typeId, objectId, annotated);

        while (advanceUntilEndBlock(streamParser, tag).isSuccess())
        {
            // The only things that actually write values are:
            // Float, Int, Bool, String, Pointer
            // Everything else is just storage
            {
                const hkReflect::Type* type = m_bundle.type(typeId);

                hkReflect::Var v = hkReflect::TypeDetail::allocate(type);
                if (!v)
                {
                    DLOG("Cannot allocate instance of {}", type->getFullName());
                }
                if (annotated)
                {
                    m_bundle.setNote(objectId, annotated, v.getAddress(), typeId);
                }
                else
                {
                    m_bundle.setVar(objectId, v.getAddress(), typeId);
                }
                hkMemUtil::memSet(v.getAddress(), 0, type->getSizeOf());

                if (!recursivelyLoad(streamParser,v).isSuccess())
                {
                    return HK_FAILURE;
                }
            }
        }

        return HK_SUCCESS;
    }

    hkResult XmlReadFormat::Impl::parseImport(hkXmlStreamParser& streamParser)
    {
        while (advanceUntilEndBlock(streamParser,"import").isSuccess()) {}

        return HK_SUCCESS;
    }

    hkResult XmlReadFormat::Impl::parseExport(hkXmlStreamParser& streamParser)
    {
        while (advanceUntilEndBlock(streamParser,"export").isSuccess()) {}

        return HK_SUCCESS;
    }

    hkReflect::Var XmlReadFormat::Impl::AllocationImpl::allocate(_In_ const hkReflect::Type* type) const
    {
        void* p = m_self->m_bundle.m_allocator.blockAlloc(type->getSizeOf());
        return hkReflect::Var(p, type);
    }

    hkReflect::Var XmlReadFormat::Impl::AllocationImpl::allocateForClone(const hkReflect::Var& src, _In_ const hkReflect::Type* type) const
    {
        HK_ASSERT_NO_MSG(0x44ae35bc,0);
        return hkReflect::Var();
    }

    void XmlReadFormat::Impl::AllocationImpl::deallocate(_Inout_ void* target, _In_ const hkReflect::Type* type) const
    {
        m_self->m_bundle.m_allocator.blockFree(target, type->getSizeOf());
    }
}

int hkSerialize::XmlReadFormat::Impl::addString(_In_z_ const char* str)
{
    const int index = m_extras.getSize();
    if (m_charTypeId == 0)
    {
        hkArrayView<const hkReflect::Type*> types = m_bundle.types();
        for (int i = 1; i < types.getSize(); i++)
        {
            if (const hkReflect::Type* t = types[i])
            {
                if (t->getFormat().get() == hkReflect::Format::OfInt<char>::Value
                    && t->getName() && hkString::strCmp(t->getName(), "char") == 0)
                {
                    m_charTypeId = i;
                    break;
                }
            }
        }
        if (m_charTypeId == 0)
        {
            // Wrong, file should contain the type of char.
            return -1;
        }
    }
    const void* copy = hkString::strDup(str, m_bundle.m_allocator);
    m_extras.pushBack(hkTupleT::make(copy, m_charTypeId, hkString::strLen(str)));
    return index;
}

void hkSerialize::XmlReadFormat::Impl::pushExtrasIntoBundle()
{
    // Extras are not given an index in the file. They need to be added at the end so that they do not interfere with
    // Vars indexing.
    m_extraStartId = m_bundle.getSize() - 1;
    for (int i = 1; i < m_extras.getSize(); i++)
    {
        int n = m_bundle.getSize();
        m_bundle.setVarN(n, m_extras[i].m_0, m_extras[i].m_1, m_extras[i].m_2);
    }
    m_extras.clear();
}

hkResult hkSerialize::XmlReadFormat::Impl::ArrayImpl::getValue(_In_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const
{
    hkUint32 index = *(hkUint32Le*)arrAddr;
    if (index)
    {
        
        const int bundleIndex = m_parent->m_extraStartId + index;
        hkSerialize::VarN varn = m_parent->m_bundle.varn(bundleIndex);
        HK_ASSERT_NO_MSG(0x3cfbd490, varn.getCount() != 0 || (arrType->getSubType().isNull() && varn.getType()));
        
        
        *val = hkReflect::ArrayValue(varn.getAddress(), varn.getCount(), varn.getType());
        return HK_SUCCESS;
    }
    else
    {
        *val = hkReflect::ArrayValue();
        return HK_SUCCESS;
    }
}

hkResult hkSerialize::XmlReadFormat::Impl::ArrayImpl::setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int nelem) const
{
    HK_ASSERT_NO_MSG(0x73e35517, 0);
    return HK_FAILURE;
}

hkResult hkSerialize::XmlReadFormat::Impl::StringImpl::setValue(_Inout_ void* string, _In_ const hkReflect::StringType* type, hkReflect::StringValue newValue) const
{
    HK_ASSERT_NO_MSG(0x362adba1, 0);
    return HK_FAILURE;
}

hkResult hkSerialize::XmlReadFormat::Impl::StringImpl::getValue(_In_ const void* string, _In_ const hkReflect::StringType* type, _Out_ hkReflect::StringValue* val) const
{
    const int index = *reinterpret_cast<const int*>(string);
    if (index)
    {
        
        const int bundleIndex = m_parent->m_extraStartId + index;
        hkSerialize::VarN bundleExtra = m_parent->m_bundle.varn(bundleIndex);
        HK_ASSERT_NO_MSG(0x98bbd0b, bundleExtra.getType()->getFormat() == hkReflect::getType<char>()->getFormat()); // Is this a string?
        *val = hkReflect::StringValue(reinterpret_cast<const char*>(bundleExtra.getAddress()));
        return HK_SUCCESS;
    }
    *val = hkReflect::StringValue();
    return HK_SUCCESS;
}

hkSerialize::XmlReadFormat::XmlReadFormat()
    : m_impl(new Impl())
{
}

hkSerialize::XmlReadFormat::~XmlReadFormat()
{
    delete m_impl;
}

hkViewPtr<hkSerialize::Bundle> hkSerialize::XmlReadFormat::read(hkIo::ReadBuffer& rb)
{
    return m_impl->read(rb);

}
hkViewPtr<hkSerialize::Bundle> hkSerialize::XmlReadFormat::view(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_opt_ hkUlong* usedOut)
{
    return m_impl->view(buf, bufLen, usedOut);
}

#ifndef HK_DYNAMIC_DLL
static
#endif
hkRefNew<hkSerialize::ReadFormat> xmlTagfileCreateRead()
{
    HK_OPTIONAL_COMPONENT_MARK_USED(hkReadFormatXmlTagfile);
    return new hkSerialize::XmlReadFormat();
}

HK_OPTIONAL_COMPONENT_DEFINE(hkReadFormatXmlTagfile, hkSerialize::Detail::fileFormatXmlTagfile.m_readFormatCreateFunc, xmlTagfileCreateRead);

#include <Common/Base/Container/PointerMap/hkMap.hxx>
template class hkMapBase<int, int>;
template class hkMap<int, int>;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
