// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL !OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Format/Tagfile2014/hkTagfileReadFormat2014.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/Reflect/Builder/hkRecordLayout.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Reflect/Util/hkReflectUtil.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Container/StringMap/hkStringMap.h>
#include <Common/Base/Serialize/Detail/hkIndexedBundle.h>
#include <Common/Base/Serialize/Detail/hkSerializeDetail.h>
#include <Common/Base/Monitor/hkMonitorStream.h>

#include <Common/Base/Serialize/Format/Tagfile2014/hkTagfileCommon2014.h>
#include <Common/Base/Serialize/Format/Tagfile2014/hkLegacyType.h>
#include <Common/Base/Serialize/Format/Tagfile2014/hkTagStringList.h>
#include <Common/Base/Config/hkOptionalComponent.h>

#define DEBUG_LOG_IDENTIFIER "s11n.OldTagfileReadFormat"
#include <Common/Base/System/Log/hkLog.hxx>

// See hkBinaryTagfileWriter.cpp for comments on the various tagfile versions

// This should be an anonymous namespace but MSVC gets confused while debugging
namespace oldTagfileProvider
{
    // We need alignment for these so that vector members work in patches
#define RETURN_OLD_TAGFILE_VEC_TYPE(N) \
    static const hkReflect::Detail::TypeData typeData = \
    { \
        hkReflect::Opt::FORMAT | hkReflect::Opt::SUBTYPE | hkReflect::Opt::IMPL | hkReflect::Opt::SIZE_ALIGN, \
        0, /*parent*/ \
        HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::FORMAT, hkReflect::Format::OfArray<N>::Value), \
        HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::SUBTYPE, HK_REFLECT_GET_TYPE(Real)), \
        HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::IMPL, &hkReflect::Detail::RepeatImpl::s_instance), \
        HK_REFLECT_TYPE_OPTIONAL_SIZE_ALIGN(sizeof(Real[N]), sizeof(Real)*4, 0) \
    }; \
    return reinterpret_cast<const hkReflect::Type*>(&typeData)

    template <typename Real>
    struct VecTypes
    {
        static const hkReflect::Type* vec4() { RETURN_OLD_TAGFILE_VEC_TYPE(4); }
        static const hkReflect::Type* vec8() { RETURN_OLD_TAGFILE_VEC_TYPE(8); }
        static const hkReflect::Type* vec12() { RETURN_OLD_TAGFILE_VEC_TYPE(12); }
        static const hkReflect::Type* vec16() { RETURN_OLD_TAGFILE_VEC_TYPE(16); }
    };

#undef OLD_TAGFILE_VEC_TYPE

    class SwappingStream
    {
    public:
        SwappingStream(hkIo::ReadBuffer& rb, bool swapBytes = HK_ENDIAN_BIG)
            : m_sr(rb)
            , m_swapBytes(swapBytes)
        {
        }

        ~SwappingStream()
        {
        }

        HK_INLINE hkUint8 read8u()
        {
            hkUint8 ret = 0xff;
            m_sr.read(&ret, 1);
            return ret;
        }

        HK_INLINE void readArray8u(_Out_writes_bytes_(numBytes) hkUint8* target, hkUint32 numBytes)
        {
            m_sr.read(target, numBytes);
        }

        HK_INLINE void readRaw(_Out_writes_bytes_(numBytes) hkUint8* target, hkUint32 numBytes) { return readArray8u(target, numBytes); }
        HK_INLINE hkUint32 read32()
        {
            hkUint32 ret;
            readRaw((hkUint8*)&ret, 4);
            if (m_swapBytes)
            {
                hkUint32 temp = ret;
                swap32(&temp, &ret);
            }
            return ret;
        }

        HK_INLINE hkUint16 read16()
        {
            hkUint16 ret;
            readRaw((hkUint8*)&ret, 2);
            if (m_swapBytes)
            {
                hkUint16 temp = ret;
                swap16(&temp, &ret);
            }
            return ret;
        }

        HK_INLINE float readFloat32()
        {
            float ret;
            readRaw((hkUint8*)&ret, 4);
            if (m_swapBytes)
            {
                float temp = ret;
                swap32(&temp, &ret);
            }
            return ret;
        }

        HK_INLINE double readDouble64()
        {
            double ret;
            readRaw((hkUint8*)&ret, 8);
            if (m_swapBytes)
            {
                double temp = ret;
                swap64(&temp, &ret);
            }
            return ret;
        }

        HK_INLINE void readArrayFloat32(_Out_writes_(numFloats) float* dest, int numFloats)
        {
            readRaw((hkUint8*)dest, 4*numFloats);
            if (m_swapBytes)
            {
                for (float* current = dest; current < (dest + numFloats); ++current)
                {
                    float temp = *current;
                    swap32(&temp, current);
                }
            }
        }

        HK_INLINE void readArrayDouble64(_Out_writes_(numFloats) double* dest, int numFloats)
        {
            readRaw((hkUint8*)dest, 8 * numFloats);
            if (m_swapBytes)
            {
                for (double* current = dest; current < (dest + numFloats); ++current)
                {
                    double temp = *current;
                    swap64(&temp, current);
                }
            }
        }

        HK_INLINE void swap32(_In_reads_bytes_(4) const void* src, _Out_writes_bytes_all_(4) void* dst)
        {
            const char* s = reinterpret_cast<const char*>(src);
            char* d = reinterpret_cast<char*>(dst);
            d[0] = s[3];
            d[1] = s[2];
            d[2] = s[1];
            d[3] = s[0];
        }

        HK_INLINE void swap64(_In_reads_bytes_(8) const void* src, _Out_writes_bytes_all_(8) void* dst)
        {
            const char* s = reinterpret_cast<const char*>(src);
            char* d = reinterpret_cast<char*>(dst);
            d[0] = s[7];
            d[1] = s[6];
            d[2] = s[5];
            d[3] = s[4];
            d[4] = s[3];
            d[5] = s[2];
            d[6] = s[1];
            d[7] = s[0];
        }

        HK_INLINE void swap16(_In_reads_bytes_(2) const void* src, _Out_writes_bytes_all_(2) void* dst)
        {
            const char* s = reinterpret_cast<const char*>(src);
            char* d = reinterpret_cast<char*>(dst);
            d[0] = s[1];
            d[1] = s[0];
        }

        hkIo::ReadBuffer& m_sr;
        bool m_swapBytes;
    };

    static _Ret_notnull_ hkReflect::Type* allocNamed(_Inout_ hkMemoryAllocator* a)
    {
        hkReflect::Detail::TypeDataN<6>* mn = a->_blockAlloc< hkReflect::Detail::TypeDataN<6> >(1);
        hkMemUtil::memSet(mn, 0, sizeof(*mn));
        mn->m_bits = hkReflect::Opt::FORMAT | hkReflect::Opt::IMPL | hkReflect::Opt::NAME | hkReflect::Opt::VERSION | hkReflect::Opt::SIZE_ALIGN| hkReflect::Opt::DECLS;
        mn->m_parent = HK_NULL;
        mn->m_opts[0] = hkReflect::Format::OfRecord::Value;

        return reinterpret_cast<hkReflect::Type*>(mn);
    }

    struct Bundle2014 : public hkSerialize::Detail::OldIndexedBundle
    {
        Bundle2014()
            : m_allocator( hkMemHeapAllocator() )
        {
            m_tidFromType.insert(HK_NULL, 0);
            m_types.pushBack(HK_NULL);
        }
        void setVar(hkSerialize::VarId vid, const hkReflect::Var& var)
        {
            makeSpace(vid);
            hkSerialize::TypeId tid = m_tidFromType.getWithDefault(var.getType(), -1);
            HK_ASSERT_NO_MSG(0x64fa3ed3, tid >= 0);
            m_vars[vid].set(var.getAddress(), tid);
        }
        void addNote(hkSerialize::VarId vid, const hkReflect::Var& var)
        {
            // not supported
            HK_ASSERT_NO_MSG(0x19148b3c, 0);
        }
        void setType(hkSerialize::TypeId tid, _In_ const hkReflect::Type* type)
        {
            m_types.setSize(hkMath::max2(m_types.getSize(), tid + 1), HK_NULL);
            m_types[tid] = type;
            HK_ASSERT_NO_MSG(0x1dcf7f50, m_tidFromType.hasKey(type) == false);
            m_tidFromType.insert(type, tid);
        }
        bool hasType(_In_ const hkReflect::Type* type)
        {
            return m_tidFromType.hasKey(type);
        }
        void addType(_In_ const hkReflect::Type* type)
        {
            if(m_tidFromType.hasKey(type) == false)
            {
                setType(m_tidFromType.getSize(), type);
            }
        }
        void setVarN(hkSerialize::VarId vid, const hkSerialize::VarN& var)
        {
            m_extras.setSize(hkMath::max2(m_extras.getSize(), vid + 1), IdxVar());
            hkSerialize::TypeId tid = m_tidFromType.getWithDefault(var.getType(), -1);
            HK_ASSERT_NO_MSG(0x780fa4d2, tid >= 0);
            m_extras[vid].set(var.getAddress(), tid, var.getCount());
        }

        virtual hkReflect::Var getNoteOnPointer(const hkReflect::PointerVar& ptr) const HK_OVERRIDE
        {
            return hkReflect::Var();
        }
        void clear()
        {
            m_allocator.clear();
            hkSerialize::Detail::OldIndexedBundle::clear();
        }
        hkTransientAllocator m_allocator;
    protected:
        void makeSpace(hkSerialize::VarId vid)
        {
            int oldSize = m_vars.getSize();
            int newSize = hkMath::max2(oldSize, vid + 1);
            m_vars.setSize(newSize, IdxVar());
        }
        hkHashMap<const hkReflect::Type*, int> m_tidFromType;
    };

    // see comments in hkBinaryTagfileWriter
    struct Reader
    {
        Reader(hkIo::ReadBuffer& rb, _In_ hkMemoryAllocator* typesAllocator, _In_ hkMemoryAllocator* objectsAllocator, hkViewPtr<Bundle2014> bundle, _In_ hkReflect::Detail::ArrayImpl* arrayImpl, _In_ hkReflect::Detail::PointerImpl* pointerImpl, _In_ hkReflect::Detail::ArrayImpl* homogenousArrayImpl)
            : m_tagfileVersion(-1)
            , m_realIsDouble(false)
            , m_ia(rb)
            , m_numPrevStringsStatic(0)
            , m_typesAllocator(typesAllocator)
            , m_objectsAllocator(objectsAllocator)
            , m_lastTypeSizeCalculated(0)
            , m_arrayImpl(arrayImpl)
            , m_pointerImpl(pointerImpl)
            , m_homogenousArrayImpl(homogenousArrayImpl)
            , m_bundle(bundle)
        {
            m_classes.pushBack(HK_NULL);
            m_prevStrings.pushBack(const_cast<char*>("")); // 0
            m_prevStrings.pushBack(HK_NULL); // -1
            m_numPrevStringsStatic = 2;
        }

        ~Reader()
        {
        }

        struct ClassNameForwardRef
        {
            void** refLocation;
            const char* name;
        };

        template<typename T>
        HK_INLINE T readInt()
        {
            hkUint64 b = m_ia.read8u();
            hkUint64 u = (b & ~0x80) >> 1;
            hkBool32 neg = unsigned(b) & 1;
            unsigned shift = 6;
            while( b & 0x80 )
            {
                b = m_ia.read8u();
                u |= (b & ~0x80) << shift;
                shift += 7;
            }
            const hkInt64 s = neg ? -hkInt64(u) : hkInt64(u);
            //const int sz = sizeof(T);
            // This doesn't work in clang for some reason
            //DLOG("Read int {0} (size: {1})", s, sz);

            return T(s);
        }

        _Ret_z_ const char* readString()
        {
            int len = readInt<int>();
            const char* ret;
            if( len > 0 )
            {
                char* s = (char*)m_typesAllocator->blockAlloc(len + 1);
                m_ia.readRaw( (hkUint8*)s, len );
                s[len] = 0;
                m_prevStrings.pushBack(s);
                ret = s;
                DLOG("Read new string ID = {0}: {1}", m_prevStrings.getSize(), s);
            }
            else
            {
                ret = m_prevStrings[-len]; // index starts at -2. 0 and -1 are reserved for ["", HK_NULL]
                DLOG("Read string backref ID = {0}", len);
            }
            return ret;
        }

        void readFloat(_Out_ void* out)
        {
            if (m_realIsDouble)
            {
                double v = m_ia.readDouble64();
                DLOG("Read double {0}", v);
                *static_cast<double*>(out) = v;
            }
            else
            {
                float v = m_ia.readFloat32();
                DLOG("Read float {0}", v);
                *static_cast<float*>(out) = v;
            }
        }

        void readBitfield( int numMembers, hkLocalArray<hkUint8>& bout)
        {
            int maxBits = HK_NEXT_MULTIPLE_OF(8, numMembers);
            int bitFieldNumBytes = maxBits / 8;
            bout.setSize(maxBits);
            hkUint8 b[64]; HK_ASSERT_NO_MSG(0x17213b79, bitFieldNumBytes < (int)sizeof(b));
            m_ia.readArray8u( b, bitFieldNumBytes );
            for( int byteIndex = 0; byteIndex < bitFieldNumBytes; ++byteIndex )
            {
                unsigned curb = b[byteIndex];
                for( int i = 0; i < 8; ++i)
                {
                    bout[byteIndex*8 + i] = hkUint8(curb & 1);
                    curb >>= 1;
                }
            }
            for( int i = numMembers; i < maxBits; ++i )
            {
                HK_ASSERT_NO_MSG(0x6c2c0450, bout[i] == 0 );
            }
            bout.setSize(numMembers);
        }

        static _Ret_notnull_ const char* getTypeName(_In_ const hkReflect::Type* t, hkStringBuf& nameOut)
        {
            if(const char* name = t->getName())
            {
                nameOut = name;
            }
            else if(t->getKind() == hkReflect::KIND_INT)
            {
                nameOut = "int";
            }
            else if(t->getKind() == hkReflect::KIND_FLOAT)
            {
                nameOut = "float";
            }
            else
            {
                nameOut = "<none>";
            }
            return nameOut.cString();
        }

        template<typename T>
        void readIntArray(_Out_writes_bytes_(nelem * outStride) void* outBuf, _In_range_(>=, 0) int nelem, _In_range_(>=, 8) int outStride)
        {
            hkInt64* arr = (hkInt64*)outBuf;
            for( int i = 0; i < nelem; ++i )
            {
                hkInt64 val = readInt<T>();
                *arr = val;
                arr = hkAddByteOffset(arr, outStride);
            }
        }

        _Ret_notnull_ void* _readArrayItems(_In_ const hkReflect::Type* subType, int nelem, _Out_writes_bytes_(nelem * outStride) void* outBuf, int outStride, int currentTopLevelId)
        {
            HK_ASSERT_NO_MSG(0x66239d45, subType);
            {
                DLOG_IF(hkStringBuf b);
                DLOG("Array type {0}, nelem={1}", getTypeName(subType, b), nelem);
            }

            // The old tagfile code has a fair few special cases
            switch(subType->getKind())
            {
            case hkReflect::KIND_BOOL:
                {
                    HK_ASSERT_NO_MSG(0x3dddb7c0,0);
                }
            case hkReflect::KIND_INT:
                {
                    const hkReflect::IntType* integerType = subType->asInteger();
                    if(integerType->getFormat().equalsValue(hkReflect::Format::OfInt<hkUint8>::Value))
                    {
                        if(outStride == 1)
                        {
                            m_ia.readArray8u( (hkUint8*)outBuf, nelem );
                        }
                        else
                        {
                            for(int i=0;i<nelem;i++)
                            {
                                *(hkUint8*)outBuf = m_ia.read8u();
                                outBuf = hkAddByteOffset(outBuf, outStride);
                            }
                        }
                    }
                    else
                    {
                        HK_ASSERT_NO_MSG(0x5423dd2f, integerType->getFormat().equalsValue(hkReflect::Format::OfInt<hkInt64>::Value));
                        // Pre-version 3, these were serialized as 32 bits anyway
                        int sizeHint = hkSizeOf(hkInt32);
                        if(m_tagfileVersion >= 3)
                        {
                            sizeHint = readInt<int>();
                        }

                        if(sizeHint == hkSizeOf(hkInt32))
                        {
                            readIntArray<hkInt32>(outBuf, nelem, outStride);
                        }
                        else
                        {
                            HK_ASSERT_NO_MSG(0x484fcbd9, sizeHint == sizeof(hkInt64));
                            readIntArray<hkInt64>(outBuf, nelem, outStride);
                        }
                    }
                    break;
                }
            case hkReflect::KIND_FLOAT:
                {
                    if (m_realIsDouble)
                    {
                        if (outStride == 8)
                        {
                            m_ia.readArrayDouble64(reinterpret_cast<double*>(outBuf), nelem);
                        }
                        else
                        {
                            for(int i=0;i<nelem;i++)
                            {
                                *reinterpret_cast<double*>(outBuf) = m_ia.readDouble64();
                                outBuf = hkAddByteOffset(outBuf, outStride);
                            }
                        }
                    }
                    else
                    {
                        if(outStride == 4)
                        {
                            m_ia.readArrayFloat32(reinterpret_cast<float*>(outBuf), nelem);
                        }
                        else
                        {
                            for(int i=0;i<nelem;i++)
                            {
                                *reinterpret_cast<float*>(outBuf) = m_ia.readFloat32();
                                outBuf = hkAddByteOffset(outBuf, outStride);
                            }
                        }
                    }
                    break;
                }
            case hkReflect::KIND_ARRAY:
                {
                    const hkReflect::ArrayType* subArrayType = subType->asArray();
                    HK_ASSERT_NO_MSG(0x3ce41baa, subArrayType);

                    int subFixedSize = subArrayType->getFixedCount();
                    if(subFixedSize > 0)
                    {
                        // It seems arrays of vectors are the only supported array of repeat type
                        const hkReflect::QualType repeatSubType = subArrayType->getSubType();
                        HK_ASSERT_NO_MSG(0xb0468ca, repeatSubType);

                        if(repeatSubType->getKind() == hkReflect::KIND_FLOAT)
                        {
                            HK_ASSERT_NO_MSG(0x2432432a, subFixedSize <= 16);
                            int numReadReal = subFixedSize;
                            if (subFixedSize == 4)
                            {
                                numReadReal = readInt<int>();
                                HK_ASSERT_NO_MSG(0x234a324a, numReadReal == 4 || numReadReal == 3);
                            }

                            if (m_realIsDouble)
                            {
                                double r[16];
                                // Need to initialize buffer, in case we are reading 3, into vec4s
                                r[3] = 0.0;
                                double* arr = (double*)outBuf;
                                for (int item = 0; item < nelem; ++item)
                                {
                                    m_ia.readArrayDouble64(r, numReadReal);
                                    hkMemUtil::memCpy(arr, r, subFixedSize * sizeof(double));
                                    arr = hkAddByteOffset(arr, outStride);
                                }
                            }
                            else
                            {
                                float r[16];
                                // Need to initialize buffer, in case we are reading 3, into vec4s
                                r[3] = 0.0f;
                                float* arr = (float*)outBuf;
                                for (int item = 0; item < nelem; ++item)
                                {
                                    m_ia.readArrayFloat32(r, numReadReal);
                                    hkMemUtil::memCpy(arr, r, subFixedSize * sizeof(float));
                                    arr = hkAddByteOffset(arr, outStride);
                                }
                            }

                            break;
                        }
                        else
                        {
                            for(int i=0;i<nelem;i++)
                            {
                                _readArrayItems(repeatSubType, subFixedSize, hkAddByteOffset(outBuf, i * outStride), repeatSubType->getSizeOf(), currentTopLevelId);
                            }
                        }
                        //HK_ASSERT_NO_MSG(0x23432432, !"Unhandled type");
                        break;
                    }
                    else
                    {
                        const hkReflect::Type* subArrayElementType = subArrayType->getSubType();

                        for (int i = 0; i < nelem; i++)
                        {
                            const int childSize = readInt<int>();

                            void* dstLoc = hkAddByteOffset(outBuf, i * outStride);
                            void* outputBuffer = HK_NULL;
                            if(childSize > 0)
                            {
                                outputBuffer = m_objectsAllocator->blockAlloc(childSize * subArrayElementType->getSizeOf());
                                hkString::memSet(outputBuffer, 0, childSize * subArrayElementType->getSizeOf());

                                // We have set the pointer impl so we know what to write into the location
                                hkUint32 nextId = m_bundle->numExtras();
                                *((hkUlong*)dstLoc) = nextId;
                                m_bundle->addType(subArrayElementType);
                                m_bundle->setVarN(nextId, hkSerialize::VarN::fromArray(outputBuffer, subArrayElementType, childSize));
                            }
                            else
                            {
                                *((hkUlong*)dstLoc) = 0;
                            }

                            DLOG("ARRAY Id = {0} [{1} elems {2}]\n", m_bundle->numExtras() - 1, childSize, subArrayElementType->getName() ? subArrayElementType->getName() : "");

                            _readArrayItems(subArrayElementType, childSize, outputBuffer, subArrayElementType->getSizeOf(), currentTopLevelId);

                            DLOG_IF(hkStringBuf b);
                            DLOG("SET Array {0}@{1} / {2} * {3} \"{4}\"\n", outputBuffer, dstLoc, childSize, subArrayElementType->getSizeOf(), getTypeName(subArrayElementType, b));
                        }
                    }
                    break;
                }
            case hkReflect::KIND_POINTER:
                {
                    void* arr = outBuf;
                    for( int i = 0; i < nelem; ++i )
                    {
                        const hkReflect::Type* pointedType = HK_NULL;
                        DLOG_IF(const int id =) readPointerObject(arr, pointedType);

                        DLOG("POINTER Tag {0} [{1}/{2}]\n", id, i, nelem);
                        arr = hkAddByteOffset(arr, outStride);
                    }
                    break;
                }
            case hkReflect::KIND_STRING:
                {
                    const char** arr = (const char**)outBuf;
                    for( int i = 0; i < nelem; ++i )
                    {
                        const char* s = readString();
                        DLOG("String \"{0}\"", s);
                        *arr = s;
                        arr = hkAddByteOffset(arr, outStride);
                    }
                    break;
                }
            case hkReflect::KIND_RECORD:
                {
                    const hkReflect::RecordType* recordType = subType->asRecord();
                    HK_ASSERT_NO_MSG(0x6db9a6c2, recordType);
                    hkLocalArray<hkUint8> membersPresent(128);
                    {
                        const int numMembers = hkReflectUtil::getRecordNumFieldsWithAncestors(recordType);
                        readBitfield( numMembers, membersPresent );
                    }

                    int memberIndex = 0;
                    for( hkReflect::DeclIter<hkReflect::DataFieldDecl> ci(recordType); ci.advance(); )
                    {
                        const hkReflect::DataFieldDecl field = ci.current();
                        if( membersPresent[memberIndex] )
                        {
                            DLOG_SCOPE("Field name {0}", field.getName());

                            const hkReflect::Type* underlyingFieldType = field.getType();
                            HK_ASSERT_NO_MSG(0x17cd4976, underlyingFieldType);
                            if(underlyingFieldType->asArray() && (underlyingFieldType->asArray()->getFixedCount() > 0)/* && resolveTypeToUnderlying(underlyingFieldType->isRepeat()->getSubType())->asFloat()*/)
                            {
                                const int fieldStride = field.getType()->getSizeOf();
                                hkArray<char>::Temp arrayTemp; arrayTemp.setSize(fieldStride * nelem);

                                // This is just a vector optimisation
                                _readArrayItems(field.getType(), nelem, arrayTemp.begin(), fieldStride, currentTopLevelId);

                                void* src = arrayTemp.begin();
                                void* dst = hkAddByteOffset(outBuf, field.getOffset());
                                for(int objIndex=0; objIndex<nelem; objIndex++)
                                {
                                    hkMemUtil::memCpy(dst, src, fieldStride);
                                    src = hkAddByteOffset(src, fieldStride);
                                    dst = hkAddByteOffset(dst, outStride);
                                }
                            }
                            else
                            {
                                _readArrayItems(field.getType(), nelem, hkAddByteOffset(outBuf, field.getOffset()), outStride, currentTopLevelId);
                            }
                        }
                        else
                        {
                            DLOG("Skip {0}", field.getName());
                        }
                        ++memberIndex;
                    }
                    break;
                }
            default:
                {
                    HK_ASSERT_NO_MSG(0x609d8b09, false); // Unknown kind
                    break;
                }
            }
            return outBuf;
        }

        void readBinaryValue(_Out_writes_bytes_(_Inexpressible_()) void* obj, hkReflect::DataFieldDecl field, int currentTopLevelId)
        {
            const hkReflect::Type* fieldType = field.getType();
            HK_ASSERT_NO_MSG(0x4f3f9ec0, fieldType);
            const hkReflect::Kind fieldKind = fieldType->getKind();

            switch(fieldKind)
            {
                case hkReflect::KIND_BOOL:
                {
                    HK_ASSERT_NO_MSG(0x1a3a0e0b,0);
                }
                case hkReflect::KIND_INT:
                {
                    const hkReflect::IntType* integerType = fieldType->asInteger();
                    if (integerType->getFormat().equalsValue(hkReflect::Format::OfInt<hkUint8>::Value))
                    {
                        int v = m_ia.read8u();
                        DLOG("Byte {0}", v);
                        *(hkUint8*)hkAddByteOffset(obj, field.getOffset()) = hkUint8(v);
                    }
                    else
                    {
                        HK_ASSERT_NO_MSG(0x71bf89dd, integerType->getFormat().equalsValue(hkReflect::Format::OfInt<hkInt64>::Value));
                        hkInt64 v = readInt<hkInt64>();
                        DLOG("Int {0}", v);
                        *(hkInt64*)hkAddByteOffset(obj, field.getOffset()) = v;
                    }
                    break;
                }
                case hkReflect::KIND_FLOAT:
                {
                    readFloat(hkAddByteOffset(obj, field.getOffset()));
                    break;
                }
                case hkReflect::KIND_RECORD:
                {
                    HK_ASSERT_NO_MSG(0x32432423, fieldType->asRecord());
                    if( m_tagfileVersion >= 2 )
                    {
                        // struct handling optimized in version 2+
                        fieldType = field.getType();
                        readObjectTopLevel(hkBinaryTagfile2014::TAG_OBJECT, fieldType, hkAddByteOffset(obj, field.getOffset()));
                        HK_ASSERT_NO_MSG(0xbfb207e, fieldType == field.getType());
                    }
                    else
                    {
                        DLOG("Object");
                        fieldType = HK_NULL;
                        readObjectTopLevel(hkBinaryTagfile2014::TAG_NONE, fieldType, hkAddByteOffset(obj, field.getOffset()));
                    }
                    break;

                    // Old version falls through and structs handled like objects -- we can't mix pointers and objects as the old ver used to
                    //readObjectIntoValue((void**)hkAddByteOffset(obj, field.getOffset()));
                    //break;
                }
                case hkReflect::KIND_POINTER:
                {
                    const hkReflect::Type* pointedType = HK_NULL;
                    DLOG_IF(const int dId =) readPointerObject(hkAddByteOffset(obj, field.getOffset()), pointedType);

                    const hkReflect::PointerType* ptrType = field.getType()->asPointer();
                    DLOG_IF(const char* typeName = ptrType->getSubType()->getName());
                    DLOG_IF(typeName = typeName ? typeName : "<none>");
                    DLOG("READ ID Tag {0} at {1} ({2})\n", dId, hkAddByteOffset(obj, field.getOffset()), typeName);

                    // Old tagfiles have a slightly complicated format
                    // In the old code: if (parentType->isClass() && parentType->getTypeName() == HK_NULL)
                    if(ptrType->getSubType().isNull() )
                    {
                        const int id = *(int*)hkAddByteOffset(obj, field.getOffset());
                        hkReflect::PointerType* nonConstPtrType = const_cast<hkReflect::PointerType*>(ptrType); // hah
                        bool found = false;
                        for(int i=0; i < m_rememberedObjects.getSize(); i++)
                        {
                            if(id == m_rememberedObjects[i])
                            {
                                nonConstPtrType->setSubType(m_bundle->extra(id).getType());
                                found = true;
                            }
                        }
                        if(!found)
                        {
                            m_typeForwardRefs.expandOne().set(nonConstPtrType, currentTopLevelId, id); // Object id
                            //printf("TYPE FORWARD REF %x [%d/%d] (%d)\n", nonConstPtrType, currentTopLevelId, id, m_typeForwardRefs.getSize() - 1);
                        }
                    }
                    break;
                }
                case hkReflect::KIND_STRING:
                {
                    const char* s = readString();
                    DLOG("String \"{0}\"", s);
                    *(const char**)hkAddByteOffset(obj, field.getOffset()) = s;
                    break;
                }
                case hkReflect::KIND_ARRAY:
                {
                    const hkReflect::ArrayType* arrayType = fieldType->asArray();

                    int fixedSize = arrayType->getFixedCount();
                    if(fixedSize > 0)
                    {
                        const hkReflect::QualType subType = arrayType->getSubType();
                        HK_ASSERT_NO_MSG(0x2e00bbb, subType);
                        if (subType->getKind() == hkReflect::KIND_FLOAT)
                        {
                            DLOG("Vector{0}", fixedSize);

                            if (m_realIsDouble)
                            {
                                m_ia.readArrayDouble64((double*)hkAddByteOffset(obj, field.getOffset()), fixedSize);
                            }
                            else
                            {
                                m_ia.readArrayFloat32((float*)hkAddByteOffset(obj, field.getOffset()), fixedSize);
                            }

                            break;
                        }
                        else
                        {
                            _readArrayItems(subType, fixedSize, hkAddByteOffset(obj, field.getOffset()), subType->getSizeOf(), currentTopLevelId);
                        }
                        break;
                    }
                    else
                    {
                        const hkReflect::Type* subType = arrayType->getSubType();
                        int asize = readInt<int>();
                        DLOG("Array size {0}", asize);

                        void* dstLoc = hkAddByteOffset(obj, field.getOffset());
                        hkReflect::ArrayVar array( dstLoc, arrayType );

                        if(subType == HK_NULL)
                        {
                            // Homogenous Array
                            int atype = readInt<int>();
                            // Check if it was already resolved previously
                            for(int i=m_lastTypeSizeCalculated+1; i<=atype; i++)
                            {
                                hkReflect::RecordLayout::recomputeNative(const_cast<hkReflect::Type*>(m_classes[i]));
                                DLOG("CALC type [{0}] {1} -> {2}", m_classes[i]->getName(), m_classes[i], m_classes[i]);
                                m_lastTypeSizeCalculated = i;
                            }
                            subType = m_classes[atype];
                        }
                        m_bundle->addType(subType);
                        HK_ASSERT_NO_MSG(0x27c4257b, subType->getSizeOf() != 0xffff); // Uninitialized

                        void* outputBuffer = m_objectsAllocator->blockAlloc(asize * subType->getSizeOf());
                        hkString::memSet(outputBuffer, 0, asize * subType->getSizeOf());

                        hkUint32 nextId = m_bundle->numExtras();
                        *((hkUlong*)dstLoc) = nextId;

                        m_bundle->setVarN(nextId, hkSerialize::VarN::fromArray(outputBuffer, subType, asize));

                        _readArrayItems(subType, asize, outputBuffer, subType->getSizeOf(), currentTopLevelId);

                        DLOG("ARRAY Id = {0} [{1} elems {2}]\n", nextId, asize, subType->getName() ? subType->getName() : "");

                        break;
                    }
                }
            default:
                {
                    HK_ASSERT_NO_MSG(0x18a9b1ef, false); // Unknown kind
                    break;
                }
            }
        }

        // Read an object and assign it to an lvalue.
        // The value may be an array item or object member reference.

        // Returns a provider id (1-based, 0=null)
        int readPointerObject(_In_ void* location, const hkReflect::Type*& pointedType)
        {
            int* locRefPtr = reinterpret_cast<int*>(location);
            pointedType = HK_NULL;
            if( m_tagfileVersion >= 2 )
            {
                int id = readInt<int>();
                DLOG("Object {0}", id);
                if( id < m_rememberedObjects.getSize() ) // yes, seen it
                {
                    *locRefPtr = m_rememberedObjects[id];
                    return m_rememberedObjects[id];
                }
                m_forwardRefs.pushBack(location);
                *locRefPtr = id; // Zero-based
                return -1; // This is meaningless for a forward ref, we don't know what the tag is going to be
            }
            else // older tagfile version, fully recursive
            {
                DLOG("Object");
                int ret = readObjectTopLevel(hkBinaryTagfile2014::TAG_NONE, pointedType);
                *locRefPtr = ret;
                return ret;
            }
        }

        // Returns a provider id (1-based, 0=null)
        // Can also create array objects, which get added to the output list
        int readObjectTopLevel(hkBinaryTagfile2014::TagType objectTag, const hkReflect::Type*& objectType, _In_opt_ void* obj = HK_NULL)
        {
            if(objectTag == hkBinaryTagfile2014::TAG_NONE) // fetch tag if not supplied
            {
                objectTag = readInt<hkBinaryTagfile2014::TagType>();
            }
            if(objectTag == hkBinaryTagfile2014::TAG_OBJECT_BACKREF)
            {
                int id = readInt<int>();
                DLOG("OBJ backref #{0}",id);
                DLOG("BACKREF Tag {0} -> {1}\n", id, m_rememberedObjects[id]);
                objectType = m_bundle->var(m_rememberedObjects[id]).getType();
                return m_rememberedObjects[id];// + 1;
            }
            else if(objectTag == hkBinaryTagfile2014::TAG_OBJECT_NULL)
            {
                DLOG("OBJ null");
                return 0;
            }

//            const hkReflect::Named* namedObjectType;

            if(!objectType)
            {
                // Make sure we have a real type, with actual pointers. I don't like this
                const int objTag = readInt<int>();
                HK_ASSERT_NO_MSG(0x6ea7f831, objTag < m_classes.getSize());
                for(int i=m_lastTypeSizeCalculated+1; i</*objectTag*/m_classes.getSize(); i++) // calculate all types -- they must be consistent if we are reading an object
                {
                    DLOG("CALC type [{0}] {1} -> {2}", m_classes[i]->getName(), m_classes[i], m_classes[i]);
                    hkReflect::RecordLayout::recomputeNative(const_cast<hkReflect::Type*>(m_classes[i]));
                    m_lastTypeSizeCalculated = i;
                }
                HK_ASSERT_NO_MSG(0x1bc46004, objTag);
                objectType = m_classes[objTag];
                HK_ASSERT_NO_MSG(0x60c7b5f0, objectType);
            }
            //namedObjectType = objectType->isNamed();
            HK_ON_DEBUG(const char* objectName = objectType->getName());
            HK_ASSERT_NO_MSG(0x4119b75e, objectName); // Type must have a name
            const hkReflect::RecordType* objectRecord = objectType->asRecord();
            int returnVal = 0;

            // If obj is set, we are just loading it in-place and it is not a top-level object
            if(!obj)
            {
                obj = m_objectsAllocator->blockAlloc(objectRecord->getSizeOf());
                hkMemUtil::memSet(obj, 0, objectRecord->getSizeOf());
                DLOG("ALLOC Object Tag {0} at {1} ({2})\n", m_bundle->numVars(), obj, objectName);
                DLOG("OBJECT Id = {0} [{1}]\n", returnVal, objectType->getName() ? objectType->getName() : "");

                returnVal = m_bundle->numVars();

                m_bundle->setVar(returnVal, hkReflect::Var(obj, objectType));
            }
            switch( objectTag )
            {
                case hkBinaryTagfile2014::TAG_OBJECT:
                    break;
                case hkBinaryTagfile2014::TAG_OBJECT_REMEMBER:
                {
                    //int objectId = m_rememberedObjects.getSize();
                    DLOG("Obj {0} Remembered as &{1}", objectName, m_rememberedObjects.getSize());
                    m_rememberedObjects.pushBack(returnVal);

                    for(int i=0;i<m_forwardRefs.getSize();i++)
                    {
                        int* refValPtr = reinterpret_cast<int*>(m_forwardRefs[i]);
                        if (*refValPtr == m_rememberedObjects.getSize() - 1)
                        {
                            *refValPtr = returnVal;
                            m_forwardRefs.removeAt(i);
                            i--;
                        }
                    }

                    for(int i=0;i<m_typeForwardRefs.getSize();i++)
                    {
                        if(m_typeForwardRefs[i].m_typeFromObjectId == returnVal)
                        {
                            m_typeForwardRefs[i].m_ptr->setSubType(objectType);
                            m_typeForwardRefs.removeAt(i);
                            i--;
                        }
                    }
                    break;
                }
                default:
                    HK_ERROR(0x484994d5, "corrupt file");
            }

            DLOG_SCOPE("Members");
            hkLocalArray<hkUint8> membersPresent(128);
            {
                int numMembers = hkReflectUtil::getRecordNumFieldsWithAncestors(objectRecord);
                readBitfield(numMembers, membersPresent);
            }

            int memberIndex = 0;
            for( hkReflect::DeclIter<hkReflect::DataFieldDecl> ci(objectRecord); ci.advance(); )
            {
                const hkReflect::DataFieldDecl field = ci.current();
                if( membersPresent[memberIndex] )
                {
                    DLOG_SCOPE("Field {0}", field.getName());
                    readBinaryValue(obj, field, returnVal);
                }
                else
                {
                    DLOG("Skip {0}", field.getName());
                }
                ++memberIndex;
            }
            return returnVal;
        }

        _Ret_notnull_ const hkReflect::Type* findType(hkLegacyType::Type type, _In_z_ const char* typeName, int tuples)
        {
            if(type & hkLegacyType::TYPE_ARRAY)
            {
                hkReflect::ArrayType* rep;
                hkLegacyType::Type arrayElementType = hkLegacyType::Type(int(type) & (~hkLegacyType::TYPE_ARRAY));

                if((!typeName) && (arrayElementType == hkLegacyType::TYPE_STRUCT))
                {
                    // If typeName is null, we really want a homogenous array
                    // Handle this specially, don't go into recursive findType
                    rep = m_typesAllocator->create<hkReflect::BasicArrayType>(
                        (const hkReflect::Type*)HK_NULL, hkSizeOf(hkReflect::Detail::HomogeneousArrayImpl::InMemory), (int)HK_ALIGN_OF(hkReflect::Detail::HomogeneousArrayImpl::InMemory),
                        m_homogenousArrayImpl, "hkVariantArray");
                    //printf("NEW Homogenous Array @ %x, impl = %x\n", rep, rep->getImpl());
                }
                else
                {
                    const hkReflect::Type* elemType = findType(arrayElementType, typeName, 0);
                    HK_ASSERT_NO_MSG(0xf15716c, elemType);
                    rep = m_typesAllocator->create<hkReflect::BasicArrayTypeWithParams>(elemType, hkSizeOf(hkArray<char>), (int)HK_ALIGN_OF(hkArray<char>), m_arrayImpl, "hkArray");
                }

                return rep;
            }

            if(type & hkLegacyType::TYPE_TUPLE)
            {
                const hkReflect::Type* elemType = findType(hkLegacyType::Type(int(type) & (~hkLegacyType::TYPE_TUPLE)), typeName, 0);
                return m_typesAllocator->create<hkReflect::RepeatArrayType>(elemType, tuples, tuples * elemType->getSizeOf(), elemType->getAlignOf(), &hkReflect::Detail::RepeatImpl::s_instance, "T[]");
            }

            switch(type)
            {
            case hkLegacyType::TYPE_BYTE: return hkReflect::getType<hkUint8>();
            case hkLegacyType::TYPE_INT: return hkReflect::getType<hkInt64>();
            case hkLegacyType::TYPE_REAL: return m_realIsDouble ? hkReflect::getType<double>() : hkReflect::getType<float>();
            case hkLegacyType::TYPE_VEC_4: return m_realIsDouble ? VecTypes<double>::vec4() : VecTypes<float>::vec4();
            case hkLegacyType::TYPE_VEC_8: return m_realIsDouble ? VecTypes<double>::vec8() : VecTypes<float>::vec8();
            case hkLegacyType::TYPE_VEC_12: return m_realIsDouble ? VecTypes<double>::vec12() : VecTypes<float>::vec12();
            case hkLegacyType::TYPE_VEC_16: return m_realIsDouble ? VecTypes<double>::vec16() : VecTypes<float>::vec16();
            case hkLegacyType::TYPE_CSTRING: return hkReflect::getType<const char*>();

            case hkLegacyType::TYPE_OBJECT:
                {
                    const hkReflect::Type* subType = HK_NULL;
                    if(typeName != HK_NULL)
                    {
                        subType = m_nameToClassMap.getWithDefault(typeName, HK_NULL);
                        if(!subType) // This will be filled in later
                        {
                            hkReflect::Type* dstRecordType = allocNamed(m_typesAllocator);
                            m_nameToClassMap.insert(typeName, dstRecordType);
                            subType = dstRecordType;

                            DLOG("CLASS REF xx:{0}\n", typeName);
                        }
                    }
                    else
                    {
                        // Type is unknown if name is HK_NULL
                        // Unknown pointer type, this will be filled in later
                        const hkReflect::Type* OpaqueType = hkReflect::getType<hkReflect::Detail::Opaque>();
                        subType = OpaqueType;
                    }
                    hkReflect::PointerType* ret = m_typesAllocator->create<hkReflect::BasicPointerType>(subType, "T*");
                    hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::IMPL>(ret, m_pointerImpl);
                    return ret;
                }
            case hkLegacyType::TYPE_STRUCT:
                {
                    const hkReflect::Type* structType = HK_NULL;
                    if(typeName != HK_NULL)
                    {
                        structType = m_nameToClassMap.getWithDefault(typeName, HK_NULL);
                        if(!structType) // This will be filled in later
                        {
                            hkReflect::Type* dstRecordType = allocNamed(m_typesAllocator);
                            //m_classes.pushBack(dstRecordType);
                            m_nameToClassMap.insert(typeName, dstRecordType);
                            structType = dstRecordType;

                            DLOG("CLASS REF xx:{0}\n", typeName);
                        }
                    }
                    else
                    {
                        HK_ASSERT_NO_MSG(0x7cf3af91, 0);
                    }
                    return structType;
                }
            default:
                {
                    break;
                }
            }

            return hkReflect::getType<void>();
        }

        hkResult readClass()
        {
            const char* recordName = readString();
            int version = readInt<int>();
            hkReflect::Detail::TypeDataN<6>* dstRecord = HK_NULL;

            // We need the name for this
            //if(++m_readClassIndex >= m_classes.getSize())
            hkReflect::Type* passthroughType = m_nameToClassMap.getWithDefault(recordName, HK_NULL);
            if(passthroughType)
            {
                dstRecord = reinterpret_cast<hkReflect::Detail::TypeDataN<6>*>(passthroughType);
                DLOG("CLASS On List {0}:{1} version {2}\n", m_classes.getSize() - 1, recordName, version);
            }
            else
            {
                passthroughType = allocNamed(m_typesAllocator);
                m_nameToClassMap.insert(recordName, passthroughType);
                dstRecord = reinterpret_cast<hkReflect::Detail::TypeDataN<6>*>(passthroughType);
                DLOG("CLASS New {0}:{1} version {2}\n", m_classes.getSize() - 1, recordName, version);
            }

            dstRecord->m_opts[2] = HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::NAME, recordName);
            dstRecord->m_opts[3] = HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::VERSION, version);
            dstRecord->m_opts[4] = HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::SIZE_ALIGN, 0);
            m_bundle->addType(passthroughType);
            m_classes.pushBack( passthroughType );

            int parentIndex = readInt<int>();
            dstRecord->m_parent = ( parentIndex >= 0 ) ? m_classes[parentIndex] : HK_NULL; // PARENT
            int numFields = readInt<int>();

            dstRecord->m_opts[0] = hkReflect::Format::ofRecord();
            dstRecord->m_opts[1] = HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::IMPL, &hkReflect::Detail::HavokRecordImpl::s_instance);
            dstRecord->m_bits = hkReflect::Opt::FORMAT | hkReflect::Opt::IMPL | hkReflect::Opt::NAME | hkReflect::Opt::VERSION | hkReflect::Opt::SIZE_ALIGN | hkReflect::Opt::DECLS;


            DLOG("class {0} {1} members", recordName, numFields);

            hkReflect::Detail::DeclsArray* dstFields = hkReflect::Detail::DeclsArray::create(HK_NULL, numFields, 0, *m_typesAllocator);
            dstRecord->m_opts[5] = HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::DECLS, dstFields);

            for( int fieldIndex = 0; fieldIndex < numFields; ++fieldIndex )
            {
                const char* fieldName = readString();
                const hkLegacyType::Type legacyType = readInt<hkLegacyType::Type>();
                const int tupleCount = ( legacyType & hkLegacyType::TYPE_TUPLE )
                    ? readInt<int>()
                    : 0;
                int bt = legacyType & hkLegacyType::TYPE_MASK_BASIC_TYPES;
                const char* className = ( bt == hkLegacyType::TYPE_STRUCT || bt == hkLegacyType::TYPE_OBJECT )
                    ? readString()
                    : HK_NULL;
                // Work out the type
                DLOG("  {0}: {1}[{2}]", fieldIndex, fieldName, className ? className : "");

                const hkReflect::Type* type = findType(legacyType, className, tupleCount);
                m_bundle->addType(type);
                HK_ASSERT_NO_MSG(0x23432432, type && "Type not found");

                hkReflect::TypeBuilder field;
                field.beginDerived(type); // Add prefix to member names
                field.setField(fieldName, 0xfefe, ((legacyType == hkLegacyType::TYPE_VOID) ? hkReflect::Decl::DECL_NOT_SERIALIZABLE : 0));
                // prevent allocation from field types
                field.addItem<hkReflect::Opt::ALLOC_IMPL>(HK_NULL);

                hkReflect::Type* allocatedType;
                if (field.allocate(m_typesAllocator, &allocatedType).isFailure())
                {
                    Log_Error("Failed to allocate type from builder");
                    return HK_FAILURE;
                }

                dstFields->setField(fieldIndex, allocatedType);
            }

            return HK_SUCCESS;
        }

        hkResult readObjectTree()
        {
            {
                int magic0 = m_ia.read32();
                int magic1 = m_ia.read32();
                if( !hkBinaryTagfile2014::isBinaryMagic(magic0, magic1))
                {
                    HK_WARN_ALWAYS(0x2ab6036f, "This does not look like a binary tagfile (magic number mismatch)");
                    return HK_FAILURE;
                }
            }
            for( hkBinaryTagfile2014::TagType tag = readInt<hkBinaryTagfile2014::TagType>();
                tag != hkBinaryTagfile2014::TagType(-1);
                tag = readInt<hkBinaryTagfile2014::TagType>() )
            {
                switch( tag )
                {
                    case hkBinaryTagfile2014::TAG_FILE_INFO:
                    {
                        m_tagfileVersion = readInt<int>();
                        DLOG("header version {0}", m_tagfileVersion);
                        switch( m_tagfileVersion )
                        {
                            case 0:
                            {
                                break;
                            }
                            case 1:
                            {
                                int tagListVersion = readInt<int>();

                                if(tagListVersion != HK_TAG_LIST_VERSION)
                                {
                                    HK_ASSERT(0x557f6908, 0, "Unknown Taglist version in Tagfile");
                                    return HK_FAILURE;
                                }
                                m_prevStrings.clear();
                                m_prevStrings.pushBack(const_cast<char*>("")); // 0
                                m_prevStrings.pushBack(HK_NULL); // -1
                                m_prevStrings.append(const_cast<char * const *>(HK_TAG_STRING_LIST), HK_NUM_TAG_STRINGS);
                                m_numPrevStringsStatic = 2 + HK_NUM_TAG_STRINGS; // These elements are statically allocated, subsequent elements are freed on destruction
                                break;
                            }
                            case 2:
                            case 3: // 3 is the same as 2 but with array size hints
                            case 4: // 4 adds an sdk version string to the header
                            case 5: // 5 adds an array of predicates verifier by the data in this file
                            case 6: // 6 is double precision 2014 tagfile
                            {
                                HK_ASSERT_NO_MSG(0x2b7ba2cc, m_rememberedObjects.getSize() == 0 );
                                m_rememberedObjects.pushBack(0);

                                const char* versionString = HK_NULL;
                                if( m_tagfileVersion >= 4 )
                                {
                                    versionString = readString();
                                }

                                if(m_tagfileVersion >= 5)
                                {
                                    HK_ON_DEBUG(const hkUint16 maxPredicate = ) m_ia.read16();
                                    HK_ON_DEBUG(maxPredicate);
                                    const hkUint16 numVerifiedPredicates = m_ia.read16();
                                    for (int i = 0; i < numVerifiedPredicates; ++i)
                                    {
                                        m_ia.read16();
                                    }
                                }

                                if (m_tagfileVersion >= 6)
                                {
                                    m_realIsDouble = true;
                                }

                                break;
                            }
                            default:
                            {
                                HK_WARN_ALWAYS(0x2ab6036f, "Unrecognised tagfile version " << m_tagfileVersion);
                                return HK_FAILURE;
                            }
                        }
                        break;
                    }
                    case hkBinaryTagfile2014::TAG_METADATA:
                    {
                        if( readClass().isFailure() )
                        {
                            return HK_FAILURE;
                        }
                        break;
                    }
                    case hkBinaryTagfile2014::TAG_OBJECT:
                    case hkBinaryTagfile2014::TAG_OBJECT_REMEMBER:
                    case hkBinaryTagfile2014::TAG_OBJECT_NULL:
                    {
                        const hkReflect::Type* type = HK_NULL;
                        readObjectTopLevel(tag, type);
                        if( m_tagfileVersion < 2 )
                        {
                            // The old fully recursive way.
                            // When the Top object is read, we're done
                            HK_ASSERT(0x32b2434a, m_forwardRefs.getSize() == 0, "There are still dangling references");
                            HK_ASSERT(0x32b2434a, m_typeForwardRefs.getSize() == 0, "There are still dangling type references");

                            return HK_SUCCESS;
                        }
                        //m_objCount++;

                        // New style, multiple toplevels, loop until TAG_FILE_END
                        break;
                    }
                    case hkBinaryTagfile2014::TAG_FILE_END:
                    {
                        HK_ASSERT(0x32b2434a, m_forwardRefs.getSize() == 0, "There are still dangling references");
                        HK_ASSERT(0x32b2434a, m_typeForwardRefs.getSize() == 0, "There are still dangling type references");
                        HK_ASSERT(0x32b2434b, m_rememberedObjects.getSize() >= 2, "No objects found in this file?");
                        return HK_SUCCESS; // first object is always null, seconds is contents
                    }
                    default:
                    {
                        HK_ASSERT_NO_MSG(0x1aa61754, 0);
                        return HK_FAILURE;
                    }
                }
            }

            // We should have finished with a TAG_END
            return HK_FAILURE;
        }

        int m_tagfileVersion;
        bool m_realIsDouble;
        SwappingStream m_ia;
        hkArray<hkReflect::Type*> m_classes;
        hkStringMap<hkReflect::Type*> m_nameToClassMap;
        hkArray<char*> m_prevStrings;
        int m_numPrevStringsStatic;
        // Remembered objects have a particular meaning in tagfiles
        hkArray<int> m_rememberedObjects; // Tagfile backrefs are to this list only, not the general object list
        hkMemoryAllocator* m_typesAllocator;
        hkMemoryAllocator* m_objectsAllocator;
        //int m_readClassIndex;
        int m_lastTypeSizeCalculated;
        hkArray<void*> m_forwardRefs;
        struct TypeForwardRef
        {
            hkReflect::PointerType* m_ptr;
            int m_refObjectId;
            int m_typeFromObjectId;
            void set(hkReflect::PointerType* p, int refObjectId, int typeFromObjectId) { m_ptr = p; m_refObjectId = refObjectId; m_typeFromObjectId = typeFromObjectId; }
        };
        hkArray<TypeForwardRef> m_typeForwardRefs;
        hkReflect::Detail::ArrayImpl* m_arrayImpl;
        hkReflect::Detail::PointerImpl* m_pointerImpl;
        hkReflect::Detail::ArrayImpl* m_homogenousArrayImpl;
        hkViewPtr<Bundle2014> m_bundle;

        void cleanupClassesForReturn()
        {
            HK_TIME_CODE_BLOCK("hkOldTagfileProvider::cleanupClasses", this);

            for(int classIndex=0; classIndex<m_classes.getSize(); classIndex++)
            {
                if(m_classes[classIndex])
                {
                    if(hkReflect::Type* c = const_cast<hkReflect::Type*>(m_classes[classIndex]))
                    {
                        if(hkReflect::RecordType* recordType = c->asRecord())
                        {
                            // this is basically a writable alias of recordType
                            int nonVoidMemberCount = 0;

                            for(int memberIndex=0; memberIndex < recordType->getNumDataFields(); memberIndex++)
                            {
                                const hkReflect::FieldDecl f = recordType->getField(memberIndex);
                                if(!f.getType()->asVoid())
                                {
                                    nonVoidMemberCount++;
                                }
                            }
                            if( nonVoidMemberCount != recordType->getNumDataFields() )
                            {
                                hkReflect::Detail::DeclsArray* newFields = hkReflect::Detail::DeclsArray::create(HK_NULL, nonVoidMemberCount, 0, *m_typesAllocator);
                                hkArrayView<const hkReflect::FieldDecl> oldFields = recordType->getFields();
                                if(nonVoidMemberCount)
                                {
                                    int newNonVoidMemberIndex=0;
                                    for(int memberIndex=0; memberIndex < recordType->getNumDataFields(); memberIndex++)
                                    {
                                        const hkReflect::FieldDecl f = oldFields[memberIndex];
                                        if(!f.getType()->asVoid())
                                        {
                                            newFields->setField(newNonVoidMemberIndex++, f.getType());
                                        }
                                    }
                                    HK_ASSERT_NO_MSG(0x1c4bfbd0, nonVoidMemberCount == newNonVoidMemberIndex);
                                }
                                hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::DECLS>(recordType, newFields);
                            }
                        }
                    }
                }
            }
        }
    };

    using namespace hkSerialize;
//////////////////////////////////////////////
// Old Tagfile Reader
//////////////////////////////////////////////

class OldTagfileArrayImpl : public hkReflect::Detail::ArrayImpl
{
public:
    OldTagfileArrayImpl(_In_ Detail::OldIndexedBundle* parent) : m_parent(parent)
    {
    }

    ~OldTagfileArrayImpl()
    {
    }

    hkResult getValue(_In_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const HK_OVERRIDE
    {
        const hkUint32 id = *reinterpret_cast<const hkUint32*>(arrAddr);
        if (id > 0)
        {
            hkSerialize::VarN varn = m_parent->extra(id);
            *val = hkReflect::ArrayValue(varn.getAddress(), varn.getCount(), varn.getType());
            return HK_SUCCESS;
        }
        *val = hkReflect::ArrayValue();
        return HK_SUCCESS;
    }

    hkResult setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int len) const HK_OVERRIDE
    {
        HK_ASSERT_NO_MSG(0x45540009, 0);
        return HK_FAILURE;
    }

    Detail::OldIndexedBundle* m_parent;
};

// Homogeneous impl for tf, might not be needed?
class OldTagfileHomogenousArrayImpl : public hkReflect::Detail::ArrayImpl
{
public:
    OldTagfileHomogenousArrayImpl(_In_ Detail::OldIndexedBundle* parent) : m_parent(parent)
    {
    }

    ~OldTagfileHomogenousArrayImpl()
    {
    }

    hkResult getValue(_In_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const HK_OVERRIDE
    {
        const hkUint32 id = *reinterpret_cast<const hkUint32*>(arrAddr);
        if (id > 0)
        {
            hkSerialize::VarN varn = m_parent->extra(id);
            *val = hkReflect::ArrayValue(varn.getAddress(), varn.getCount(), varn.getType());
            return HK_SUCCESS;
        }
        *val = hkReflect::ArrayValue();
        return HK_SUCCESS;
    }

    hkResult setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int len) const HK_OVERRIDE
    {
        HK_ASSERT_NO_MSG(0x753b6d00, 0);
        return HK_FAILURE;
    }

    hkResult spliceInto(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int index, int numToDel, const hkReflect::ArrayValue& val) const HK_OVERRIDE
    {
        HK_ASSERT_NO_MSG(0xbf448bf, 0);
        return HK_FAILURE;
    }

    Detail::OldIndexedBundle* m_parent;
};

// We use our own pointer impl here so it is ok to use ints, these are never loaded in-place and
// are distinct from the types in the file (which are platform-independent)
class OldTagfilePointerImpl : public hkReflect::Detail::PointerImpl
{
public:
    OldTagfilePointerImpl(Detail::OldIndexedBundle* parent) : m_parent(parent)
    {
    }

    virtual hkResult setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* type, const hkReflect::Var& val) const HK_OVERRIDE
    {
        HK_ASSERT_NO_MSG(0x78d1e4fa, 0);
        return HK_FAILURE;
    }
    virtual hkResult getValue(_In_ const void* self, _In_ const hkReflect::PointerType* type, _Out_ hkReflect::Var* val) const HK_OVERRIDE
    {
        const hkUint32 id = *reinterpret_cast<const hkUint32*>(self);
        
        if (id)
        {
            *val = m_parent->var(id);
            return HK_SUCCESS;
        }

        *val = hkReflect::Var();
        return HK_SUCCESS;
    }

    Detail::OldIndexedBundle* m_parent;
};

} // namespace oldTagFileProvider

using namespace oldTagfileProvider;

// New interface

namespace
{
    void recursivelyAddTypes(Bundle2014& bundle, _In_ const hkReflect::Type* t)
    {
        if (t && bundle.hasType(t)==false )
        {
            bundle.addType(t);
            if (const hkReflect::RecordType* rt = t->asRecord())
            {
                for (int i = 0; i < rt->getNumFields(); i++)
                {
                    recursivelyAddTypes(bundle, rt->getField(i).getType());
                }
            }
            else if (const hkReflect::ArrayType* at = t->asArray())
            {
                recursivelyAddTypes(bundle, at->getSubType());
            }
            else if (const hkReflect::PointerType* pt = t->asPointer())
            {
                recursivelyAddTypes(bundle, pt->getSubType());
            }
        }
    }
}


struct hkSerialize::TagfileReadFormat2014::Impl
{
    HK_DECLARE_CLASS(Impl, New);
    Impl();

    hkViewPtr<hkSerialize::Bundle> read(hkIo::ReadBuffer& rb);
    hkViewPtr<hkSerialize::Bundle> view(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_opt_ hkUlong* usedOut);

    Bundle2014 m_bundle;

    // Types allocator
    hkTransientAllocator m_typesAllocator;
    OldTagfileArrayImpl m_arrayImpl;
    OldTagfilePointerImpl m_pointerImpl;
    OldTagfileHomogenousArrayImpl m_homogenousArrayImpl;

};

hkViewPtr<hkSerialize::Bundle> hkSerialize::TagfileReadFormat2014::Impl::read(hkIo::ReadBuffer& rb)
{
    m_bundle.clear();

    // We append things to the Var list based on the size, we need to put in nulls at index zero
    m_bundle.setVar(0, hkReflect::Var());
    m_bundle.setVarN(0, hkSerialize::VarN::invalid());

    Reader reader(rb, &m_typesAllocator, &m_bundle.m_allocator, &m_bundle, &m_arrayImpl, &m_pointerImpl, &m_homogenousArrayImpl);
    if (reader.readObjectTree().isFailure())
    {
        return HK_NULL;
    }

    // Add types to the bundle
    for (int i = 0; i < m_bundle.numVars(); i++)
    {
        if(const hkReflect::Type* rt = m_bundle.var(i).getType())
        {
            recursivelyAddTypes(m_bundle, rt);
        }
    }

    for (int i = 0; i < m_bundle.numExtras(); i++)
    {
        if (const hkReflect::Type* rt = m_bundle.extra(i).getType())
        {
            recursivelyAddTypes(m_bundle, rt);
        }
    }

    return &m_bundle;
}

// Read from buffer
hkViewPtr<hkSerialize::Bundle> hkSerialize::TagfileReadFormat2014::Impl::view(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_opt_ hkUlong* usedOut)
{
    hkIo::ReadBuffer rb(buf, int(bufLen));
    hkViewPtr<hkSerialize::Bundle> ret = read(rb);
    if(usedOut)
    {
        *usedOut = rb.tell();
    }
    return ret;
}

hkSerialize::TagfileReadFormat2014::Impl::Impl()
    : m_typesAllocator(hkMemHeapAllocator())
    , m_arrayImpl(&m_bundle)
    , m_pointerImpl(&m_bundle)
    , m_homogenousArrayImpl(&m_bundle)
{
}

hkSerialize::TagfileReadFormat2014::TagfileReadFormat2014()
    : m_impl(new Impl())
{

}
hkSerialize::TagfileReadFormat2014::~TagfileReadFormat2014()
{
    delete m_impl;
}

hkViewPtr<hkSerialize::Bundle> hkSerialize::TagfileReadFormat2014::read(hkIo::ReadBuffer& rb)
{
    return m_impl->read(rb);

}
hkViewPtr<hkSerialize::Bundle> hkSerialize::TagfileReadFormat2014::view(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_opt_ hkUlong* usedOut)
{
    return m_impl->view(buf, bufLen, usedOut);
}

#ifndef HK_DYNAMIC_DLL
static
#endif
hkRefNew<hkSerialize::ReadFormat> binaryTagfile2014Create()
{
    HK_OPTIONAL_COMPONENT_MARK_USED(hkReadFormatBinaryTagfile2014);
    return hkRefNew<hkSerialize::ReadFormat>( new hkSerialize::TagfileReadFormat2014() );
}

HK_OPTIONAL_COMPONENT_DEFINE(hkReadFormatBinaryTagfile2014, hkSerialize::Detail::fileFormatBinaryTagfile2014.m_readFormatCreateFunc, binaryTagfile2014Create);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
