// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileWriteFormat.h>
#include <Common/Base/Serialize/Format/Tagfile/Detail/hkTagfileDetail.h>
#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/Types/hkEndian.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Reflect/Builder/hkTypeCopier.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/Serialize/Detail/hkUintVle.h>
#include <Common/Base/Serialize/Detail/hkSerializeDetail.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/System/Io/Structured/hkStructuredStream.h>
#include <Common/Base/Container/StringView/hkStringView.h>
#include <Common/Base/Container/Hash/hkHashSet.h>
#include <Common/Base/Container/StringMap/hkStorageStringMap.h>
#include <Common/Base/Reflect/Detail/hkTypeHasher.h>
#include <Common/Base/Types/Uuid/hkUuid.h>
#include <Common/Base/Config/hkConfigVersion.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmFastCopyInterpreter.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompilerPasses.h>
#include <Common/Base/Reflect/Core/Detail/hkPodPunsDetail.h>
#include <Common/Base/Config/hkOptionalComponent.h>

#define DEBUG_LOG_IDENTIFIER "s11n.TagfileWriteFormat"
#include <Common/Base/System/Log/hkLog.hxx>

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    typedef int ExtraId;

    struct VarUint
    {
        VarUint(hkUint64 u) : m_u(u) {}
        int numBytes() const { return m_u.numBytesNeeded(); }
        template<typename WRITER>
        void write(WRITER& writer)
        {
            hkUint8 buf[hkUintVle::MAX_BYTE_COUNT];
            int n = m_u.write(buf);
            writer.writeRaw(buf, n);
        }
        hkUintVle m_u;
    };

    struct TypeIdentity
    {
        TypeIdentity() : nameIndex(0) {}
        int nameIndex;
        hkArray< hkTuple<hkUint64, hkUint64> > params;
    };

    typedef hkTuple<hkSerialize::TypeId, hkUint32> TypeIdAndHash;

    // Context and decl index.
    typedef hkTuple<hkSerialize::TypeId, int> PropDecl;

    template<typename T>
    void growArray(int idx, hkArray<T>& array)
    {
        if(array.getSize() <= idx)
        {
            array.setSize(idx + 1);
        }
    }

    struct StringNull
    {
        const char* m_s;
        StringNull(const char* s) : m_s(s) {}

        template<typename WRITER>
        void write(WRITER& w)
        {
            int len = hkString::strLen(m_s);
            w.writeRaw( m_s, len+1 );
        }
    };

    static _Ret_notnull_ const hkReflect::Type* notfield(_In_ const hkReflect::Type* t)
    {
        while( t->isField() )
        {
            t = t->getParent();
        }
        return t;
    }

    struct Fixups
    {
        virtual ~Fixups() {}
        virtual void writeINP0(_In_ const void* a, _In_ const hkReflect::Type* dt, _In_ const hkReflect::Type* st) = 0;
        virtual void writeINPN(_In_ const void* a, _In_ const hkReflect::Type* dt, _In_ const hkReflect::Type* st) = 0;
        virtual void writeObjectRef(_Inout_ hkUint32Le* a, hkSerialize::VarId id) = 0;
    };

    struct SerializeInterpreter : public hkTypeVm::FastCopyInterpreter
    {
        // check for serializable properties. In the tagfile case, the properties have already been converted to data fields
        // by the type copier. In the packfile case, there may be serializable properties which we don't handle (yet)
        // We allow the types of properties to be serialized as they may be used in template parameters or null pointers, but
        // saving an instance is an error
        inline hkResult execInt(const hkReflect::IntVar& dst, const hkReflect::IntVar& src)
        {
            HK_ASSERT(0x29c46807, !dst.getType()->isProperty(), "Can't save properties in packfiles");
            return hkTypeVm::FastCopyInterpreter::execInt(dst, src);
        }

        inline hkResult execBool(const hkReflect::BoolVar& dst, const hkReflect::BoolVar& src)
        {
            HK_ASSERT(0x63586ea, !dst.getType()->isProperty(), "Can't save properties in packfiles");
            return hkTypeVm::FastCopyInterpreter::execBool(dst, src);
        }

        inline hkResult execFloat(const hkReflect::FloatVar& dst, const hkReflect::FloatVar& src)
        {
            HK_ASSERT(0x648144b2, !dst.getType()->isProperty(), "Can't save properties in packfiles");
            return hkTypeVm::FastCopyInterpreter::execFloat(dst, src);
        }

        inline hkResult execPointer(hkReflect::PointerVar dst, hkReflect::PointerVar src, int dstStride, int srcStride, int numElems)
        {
            HK_ASSERT_EXPR(0x5ddd80c, dst.getType()->getSizeOf(), >=, 4);
            HK_ASSERT(0xd28f3c9, !dst.getType()->isProperty(), "Can't save properties in packfiles");
            hkUint32Le* dstPtr = reinterpret_cast<hkUint32Le*>(dst.getAddress());
            for(int i=0; i < numElems; i++, src = incrementAddress(src, srcStride), dstPtr = hkAddByteOffset(dstPtr, dstStride))
            {
                hkSerialize::VarId value = m_idFromVar->pointer(src);
                DLOG("  //Ptr id=#{}", value);
                HK_ASSERT_EXPR(0x6ac6259, value, >=, 0);
                // Object writer is just filling a buffer, it is safe to just write the int here
                if(m_isPackfile && value)
                {
                    m_fixups->writeINP0(dstPtr, notfield(dst.getType()), notfield(src.getType()));
                }

                *dstPtr = 0;
                m_fixups->writeObjectRef(dstPtr, value);
            }
            return HK_SUCCESS;
        }

        inline void _executeExtra(_In_z_ const char* what, bool cond, const hkReflect::Var& dst, const hkReflect::Var& src)
        {
            HK_ASSERT_EXPR(0x1318736f, dst.getType()->getSizeOf(), >=, 4);
            hkSerialize::VarId value = 0;
            if( cond )
            {
                value = m_idFromVar->newId();
                m_extras.pushBack( hkTupleT::make(src, value));
                HK_ASSERT_EXPR(0x6ac6259, value, >, 0);

                if( m_isPackfile )
                {
                    m_fixups->writeINPN(dst.getAddress(), notfield(dst.getType()), notfield(src.getType()));
                }
            }

            *reinterpret_cast<hkUint32Le*>(dst.getAddress()) = 0;
            m_fixups->writeObjectRef(reinterpret_cast<hkUint32Le*>(dst.getAddress()), value);
            DLOG("  //Extra id=%{}//{}", value, what);
        }

        inline hkResult execString(const hkReflect::StringVar& dst, const hkReflect::StringVar& src)
        {
            HK_ASSERT(0x57ab6eac, !dst.getType()->isProperty(), "Can't save properties in packfiles");
            HK_ASSERT_NO_MSG(0x1318736f, dst.getType()->getSizeOf() >= 4);
            hkSerialize::VarId value = 0;
            if (src.getValue())
            {
                hkReflect::Format::StringValue format(src.getType()->getFormat());

                // For immutable strings, try to find a duplicate string to share content with
                if (format.isImmutable())
                {
                    auto it = m_sharedStrings.findOrInsertKey(src.getValue().m_value, 0);
                    if (m_sharedStrings.getValue(it) == 0) // a new shared string
                    {
                        value = m_idFromVar->newId();
                        m_extras.pushBack(hkTupleT::make(static_cast<const hkReflect::Var&>(src), value));
                        m_sharedStrings.setValue(it, value);
                    }
                    else // reuse
                    {
                        value = m_sharedStrings.getValue(it);
                    }
                }
                else
                {
                    value = m_idFromVar->newId();
                    m_extras.pushBack(hkTupleT::make(static_cast<const hkReflect::Var&>(src), value));
                }

                if (m_isPackfile)
                {
                    m_fixups->writeINPN(dst.getAddress(), notfield(dst.getType()), notfield(src.getType()));
                }
                HK_ASSERT_NO_MSG(0x6ac6259, value > 0);
            }
            *reinterpret_cast<hkUint32Le*>(dst.getAddress()) = value;
            m_fixups->writeObjectRef(reinterpret_cast<hkUint32Le*>(dst.getAddress()), value);
            DLOG("  //Extra id=%{}//Str", value);
            return HK_SUCCESS;
        }

        inline void beginArrayOfArrays( int numElements )
        {
            m_extras.reserve( m_extras.getSize() + numElements );
        }

        inline hkResult execArray(const hkReflect::ArrayVar& dst, const hkReflect::ArrayVar& src)
        {
            hkReflect::ArrayValue aval = src.getValue();
            // execute if there are elements OR a typed variant array
            _executeExtra("Arr", aval.getCount() || (aval.getSubType() && src.getType()->getSubType().isNull()), dst, src);
            return HK_SUCCESS;
        }

        inline void clear()
        {
            m_extras.clear();
            m_sharedStrings.clear();
        }

        hkSerialize::IdFromVar* m_idFromVar;
        hkArray< hkTuple<hkReflect::Var, hkSerialize::VarId> > m_extras;
        Fixups* m_fixups;
        bool m_isPackfile;
        hkHashMap<hkStringPtr, hkSerialize::VarId> m_sharedStrings;
    };

    // Sequential container plus a monotonically increasing "m_firstUnwritten" field
    
    template<typename Container>
    struct ContainerPlusMark : public Container
    {
        ContainerPlusMark() : m_firstUnwritten(0) {}
        bool hasMoreToWrite() const
        {
            return m_firstUnwritten < this->getSize();
        }
        // Specialize this for your element type
        void writeTo(hkIo::WriteBuffer& wb);

        int m_firstUnwritten;
    };

    template<typename T> struct ArrayPlusMark : public HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(ContainerPlusMark)< hkArray<T> > {};

    template<>
    void HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(ContainerPlusMark)< hkArray<char> >::writeTo(hkIo::WriteBuffer& wb)
    {
        wb.writeRaw(begin() + m_firstUnwritten, getSize() - m_firstUnwritten);
        m_firstUnwritten = getSize();
    }

    template<>
    void HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(ContainerPlusMark)< hkArray<HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(TypeIdentity)> >::writeTo(hkIo::WriteBuffer& wb)
    {
        wb.writeObj<VarUint>(m_size); // unusually, we send the high mark, not the count of new items
        for (int idx = m_firstUnwritten; idx < m_size; ++idx)
        {
            const auto& it = m_data[idx];
            wb.writeObj<VarUint>(it.nameIndex);
            wb.writeObj<VarUint>(it.params.getSize());
            for (int i = 0; i < it.params.getSize(); ++i)
            {
                wb.writeObj<VarUint>(it.params[i].m_0);
                wb.writeObj<VarUint>(it.params[i].m_1);
            }
        }
        m_firstUnwritten = getSize();
    }

    template<>
    void HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(ContainerPlusMark)< hkArray<HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(TypeIdAndHash)> >::writeTo(hkIo::WriteBuffer& wb)
    {
        wb.writeObj<VarUint>(getSize());
        for(int i = m_firstUnwritten; i < getSize(); ++i)
        {
            const auto& it = m_data[i];
            wb.writeObj<VarUint>(it.m_0);
            wb.writePod<hkUint32Le>(it.m_1);
        }
        m_firstUnwritten = getSize();
    }

    template<>
    void ContainerPlusMark< hkHashMap<PropDecl, int> >::writeTo( hkIo::WriteBuffer& wb )
    {
        wb.writeObj<VarUint>( getSize() );
        for ( int i = m_firstUnwritten; i < getSize(); ++i )
        {
            wb.writeObj<VarUint>( m_items[i].m_0.m_0 ); // context
            wb.writeObj<VarUint>( m_items[i].m_0.m_1 ); // decl idx

            // Write only the name of the decl (for now) as it is the only info necessary to look it up.
            wb.writeObj<VarUint>( m_items[i].m_1 ); // string id
        }
        m_firstUnwritten = getSize();
    }
}

const hkUint32 hkSerialize::Detail::s_optMapping[8] =
{
    hkReflect::Opt::FORMAT,
    hkReflect::Opt::SUBTYPE,
    hkReflect::Opt::VERSION,
    hkReflect::Opt::SIZE_ALIGN,
    hkReflect::Opt::FLAGS,
    hkReflect::Opt::DECLS,
    hkReflect::Opt::INTERFACES,
    hkReflect::Opt::ATTRIBUTE_STRING,
};

struct hkSerialize::TagfileWriteFormat::Impl : protected HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(Fixups)
{
    HK_DECLARE_CLASS(Impl, New);

    typedef Detail::TagfileRoot TagfileRoot;
    typedef Detail::TagfileSection TagfileSection;
    typedef Detail::TagfileTypeSection TagfileTypeSection;
    typedef Detail::TagfileIndexSection TagfileIndexSection;

    Impl(_Inout_opt_ hkReflect::TypeCopier* copier, bool isPackfile, bool deleteCopier);
    ~Impl();

    static _Ret_notnull_ Impl* make(_In_z_ const char* platform)
    {
        hkReflect::TypeCopier::Options options;
        hkReflect::TypeCopier* tc = HK_NULL;
        if( platform && options.parseTarget(platform).isOk() )
        {
            tc = new hkReflect::TypeCopier(options);
        }
        return new Impl(tc, true, tc);
    }

    void beginBundle(_In_ hkIo::WriteBuffer* writeBuffer);
    void endBundle();

    void beginTypeCompendium(const hkIo::Detail::WriteBufferAdapter& sink);
    void endTypeCompendium();
    hkResult enableMultiBundle();

    hkResult write(const hkReflect::Var& var, IdFromVar& idFromVar);
    hkResult writeNote(VarId vid, const hkReflect::Var& note, IdFromVar& idFromVar);

    hkResult writeInternal(const hkReflect::Var& var, VarId vid, bool isVar0, _Inout_ IdFromVar* idFromVar);
    void writeTypeSection(hkSerialize::HffWriter& hff, hkIo::WriteBuffer& buf);
    void typeHash(TypeAndId tid);

    int findDeclIndex(hkReflect::Decl decl, TypeAndId typeAndId);

    void writeINP0(_In_ const void* a, _In_ const hkReflect::Type* dt, _In_ const hkReflect::Type* st) HK_OVERRIDE
    {
        hkLong off = hkGetByteOffset(m_parts.data.begin(), a);
        TypeAndId tid = m_typeWriter->writeType(st);
        HK_ASSERT_EXPR(0x313c1033, tid.m_0, ==, dt);
        m_parts.index.inplace(tid.m_1, off);
        typeHash(tid);
    }
    void writeObjectRef(_Inout_ hkUint32Le* a, hkSerialize::VarId id) HK_OVERRIDE
    {
        hkLong off = hkGetByteOffset(m_parts.data.begin(), a);
        m_parts.index.objectRef(off, id);
    }
    void writeINPN(_Inout_ const void* a, _In_ const hkReflect::Type* dt, _In_ const hkReflect::Type* st) HK_OVERRIDE
    {
        hkLong off = hkGetByteOffset(m_parts.data.begin(), a);
        TypeAndId tid = m_typeWriter->writeType(st);
        HK_ASSERT_EXPR(0x22362603, tid.m_0, ==, dt);
        m_parts.index.inplace(tid.m_1, off);
        typeHash(tid);
    }

        // This stores the logical data which will be written to the final output (in endBundle).
        // It is organized along the same lines as the file format itself. That is:
        // * a data blob "data"
        // * a struct "type" which has all the type data such as type names, hashes, bodies etc.
        // * a struct "index" which contains indexing information on "data". This is currently an
        //   array of items, plus some extra data for inplace load (pointer & array patches)
    struct Parts
    {
        hkArray<char> data;

        struct Type
        {
            Type()
            {
                typeIdentity.m_firstUnwritten = 1;
            }
            bool isEmpty() const
            {
                return typeHashes.isEmpty() && encoded.isEmpty() && typeIdentity.getSize() <= 1;
            }
            bool hasMoreToWrite() const
            {
                bool more = encoded.hasMoreToWrite();

                HK_ASSERT(0x673f4804, more == typeIdentity.hasMoreToWrite(), "Internal error: type identity and output out of sync.");
                // if there are new strings, encoded MUST have more
                HK_ASSERT_NO_MSG(0x56b2b447, more >= typeNameStrings.hasMoreToWrite());
                HK_ASSERT_NO_MSG(0x714ced7c, more >= fieldNameStrings.hasMoreToWrite());
                HK_ASSERT_NO_MSG(0x2fcb3dec, more >= attributeStrings.hasMoreToWrite());
                // Do not check typeHashes; new hashes can be saved for old types if they get used for top-level
                // instances for the first time.

                bool moreProperties = propertyDecls.hasMoreToWrite();
                HK_ASSERT_NO_MSG(0x1886b790, moreProperties >= propertyHashes.hasMoreToWrite() );

                return more || moreProperties;
            }
            void addHash(TypeId tid, hkUint32 hash)
            {
                typeHashes.pushBack(hkTupleT::make(tid, hash));
            }
            void setName(TypeId tid, _In_z_ const char* name)
            {
                HK_UNITY_USING_ANONYMOUS_NAMESPACE;
                growArray(tid, typeIdentity);
                typeIdentity[tid].nameIndex = hkString::strLen(name) > 0 ? typeNameStrings.intern(name) : -1;
            }
            void setTemplate(TypeId tid, hkArray< hkTuple<hkUint64, hkUint64> >& params)
            {
                HK_UNITY_USING_ANONYMOUS_NAMESPACE;
                growArray(tid, typeIdentity);
                typeIdentity[tid].params.swap(params);
            }
            int internFieldName(_In_z_ const char* s) { return fieldNameStrings.intern(s); }
            int internTypeName(_In_z_ const char* s) { return typeNameStrings.intern(s); }
            int internAttribute(_In_z_ const char* s) { return attributeStrings.intern(s); }

            struct InternTab
            {
                InternTab() : highest(0) { }
                int intern(_In_z_ const char* s)
                {
                    HK_ASSERT_NO_MSG(0x32f516ba, s);
                    return lut.getOrInsertKey(s, lut.getSize());
                }
                int count() const { return lut.getSize(); }
                bool hasMoreToWrite() const { return highest < lut.getSize(); }
                void writeTo(hkIo::WriteBuffer& wb)
                {
                    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
                    hkArrayView< const hkHashMap<const char*, int>::Item> a = lut.viewItems();
                    for(int i = highest; i < a.getSize(); ++i)
                    {
                        wb.writeObj<StringNull>(a[i]);
                    }
                    highest = a.getSize();
                }
                hkUint64 hashAppend(hkUint64 sig) const
                {
                    hkArrayView< const hkHashMap<const char*, int>::Item> a = lut.viewItems();
                    for(int i = 0; i < a.getSize(); ++i)
                    {
                        const char* s = a[i].m_0;
                        sig = hkHash::appendCrc64(sig, s, hkString::strLen(s));
                    }
                    return sig;
                }
            protected:
                hkHashMap<const char*, int> lut;
                int highest;
            };

            struct CompendiumSigCache
            {
                CompendiumSigCache() : encCount(-1) {}
                int encCount;
                int tnameCount;
                int tidCount;
                hkUint64 signature;
            };

            hkUint64 calcCompendiumSignature()
            {
                if(compendiumSignatureCache.encCount == encoded.getSize()
                    && compendiumSignatureCache.tnameCount==typeNameStrings.count()
                    && compendiumSignatureCache.tidCount == typeIdentity.getSize())
                {
                    HK_ASSERT_EXPR(0xe5453ff, compendiumSignatureCache.signature, ==, calcCompendiumSignatureImpl());
                    return compendiumSignatureCache.signature;
                }
                return calcCompendiumSignatureImpl();
            }

            hkUint64 calcCompendiumSignatureImpl()
            {
                HK_UNITY_USING_ANONYMOUS_NAMESPACE;

                hkUint64 sig = 0;
                sig = hkHash::appendCrc64(sig, encoded.begin(), encoded.getSize());
                sig = typeNameStrings.hashAppend(sig);
                for(int tid = 0; tid < typeIdentity.getSize(); ++tid)
                {
                    const TypeIdentity& id = typeIdentity[tid];
                    sig = hkHash::appendCrc64(sig, &id.nameIndex, sizeof(id.nameIndex));
                    for(int i = 0; i < id.params.getSize(); ++i)
                    {
                        sig = hkHash::appendCrc64(sig, id.params.begin(), id.params.getSize()*sizeof(id.params[0]));
                    }
                }

                compendiumSignatureCache.encCount = encoded.getSize();
                compendiumSignatureCache.tnameCount = typeNameStrings.count();
                compendiumSignatureCache.tidCount = typeIdentity.getSize();
                compendiumSignatureCache.signature = sig;
                return sig;
            }

            HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(ArrayPlusMark)<HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(TypeIdAndHash)> typeHashes;
            HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(ArrayPlusMark)<char> encoded; // This is the bytestream produced by writeAllOutstandingTypes
            InternTab typeNameStrings;
            InternTab fieldNameStrings;
            InternTab attributeStrings;
            HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(ArrayPlusMark)<HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(TypeIdentity)> typeIdentity; // name & template params
            CompendiumSigCache compendiumSignatureCache;

            // PropDecl -> decl name id
            ContainerPlusMark< hkHashMap<PropDecl, int> > propertyDecls;
            ArrayPlusMark< TypeIdAndHash > propertyHashes;
        };
        Type type;

        struct Index
        {
            bool isEmpty() const
            {
                return m_items.isEmpty() && m_inplace.isEmpty();
            }
            void addVar0(VarId vid, TypeId tid, hkLong off)
            {
                HK_UNITY_USING_ANONYMOUS_NAMESPACE;

                growArray(vid, m_items);
                m_items[vid].packed = Detail::TagfileItem::KIND_VAR0 | tid;
                m_items[vid].offset = hkLosslessCast<hkUint32>(off);
                m_items[vid].count = 1;
            }
            void addVarN(VarId vid, TypeId tid, hkLong off, int count)
            {
                HK_UNITY_USING_ANONYMOUS_NAMESPACE;
                growArray(vid, m_items);
                m_items[vid].packed = Detail::TagfileItem::KIND_VARN | tid;
                m_items[vid].offset = hkLosslessCast<hkUint32>(off);
                m_items[vid].count = count;
            }
            void addNote(VarId vid, VarId noteId, TypeId tid, hkLong off)
            {
                HK_UNITY_USING_ANONYMOUS_NAMESPACE;
                growArray(hkMath::max2(vid, noteId), m_items);
                m_items[noteId].packed = Detail::TagfileItem::KIND_NOTE | tid;
                m_items[noteId].offset = hkLosslessCast<hkUint32>(off);
                m_items[noteId].count = vid;

                // If the annotated object is an import, add a reference to this note
                if (m_items[vid].isEmpty())
                {
                    m_items[vid].count = noteId;
                }
            }
            void addType(VarId vid, TypeId tid)
            {
                HK_UNITY_USING_ANONYMOUS_NAMESPACE;
                growArray(vid, m_items);
                m_items[vid].packed = Detail::TagfileItem::KIND_TYPE | tid;
                m_items[vid].offset = 0xffffffff;
                m_items[vid].count = 0;
            }
            void addDecl(VarId vid, TypeId tid, int i)
            {
                HK_UNITY_USING_ANONYMOUS_NAMESPACE;
                growArray(vid, m_items);
                m_items[vid].packed = Detail::TagfileItem::KIND_DECL | tid;
                m_items[vid].offset = 0xffffffff;
                m_items[vid].count = i;
            }

            struct Inplace
            {
                bool operator<(const Inplace& i) const
                {
                    if(tid < i.tid) return true;
                    return (tid==i.tid) && (offset < i.offset);
                }
                hkUint32 tid;
                hkUint32 offset;
            };

            struct ObjectReference
            {
                hkUint32 offset;
                VarId vid;
            };

            void inplace(TypeId tid, hkLong off)
            {
                Inplace& i = m_inplace.expandOne();
                i.tid = tid;
                i.offset = hkLosslessCast<hkUint32>(off);
            }

            void objectRef(hkLong off, hkSerialize::VarId id)
            {
                ObjectReference& f = m_objectRef.expandOne();
                f.offset = hkLosslessCast<hkUint32>(off);
                f.vid = id;
            }

            void clear()
            {
                m_items.clearAndDeallocate();
                m_inplace.clearAndDeallocate();
                m_objectRef.clearAndDeallocate();
            }

            hkArray<Detail::TagfileItem> m_items; // index
            hkArray<Inplace> m_inplace; // inplace patches for pointers & arrays
            hkArray<ObjectReference> m_objectRef; // object references for pointer determinism
        };
        Index index;
    };
    Parts m_parts;
    HK_UNITY_ANONYMOUS_NAMESPACE_SYMBOL(SerializeInterpreter) m_interp;

    hkIo::WriteBuffer* m_writeBuffer;

        // Compendium signatures. When writing with a compendium, we have the following
        // beginCompendium()
        // writeFileUsingCompendium(); //A
        // writeFileUsingCompendium(); //B
        // ...
        // endCompendium(); //Z
        // The state of the compendium can change during each bundle write, as new types are added.
        // We want to store a signature so that we can check when loading a file that the correct compendium is loaded.
        // Each file thus has a single type compendium reference (hash of type data up to that point).
        // But the compendium has, not only 1, but possibly many signatures corresponding to each of the intermediate states referenced.
    struct Compendium
    {
        enum State
        {
            NOT_USING = 0,
            WRITING, // after beginTypeCompendium has been called
            CLOSED, // after endTypeCompendium has been called
        };

        Compendium() : m_state(NOT_USING) {}
        hkIo::Detail::WriteBufferAdapter m_adapter;

        State getState() const { return m_state; }
        void setState(State s)
        {
            HK_ASSERT(0x5bd337cc, (m_state==NOT_USING&&s==WRITING) || (m_state==WRITING&&s==CLOSED), "Invalid state transition");
            m_state = s;
        }

        void addSignature(hkUint64 sig)
        {
            if(m_state == CLOSED)
            {
                HK_ASSERT(0x79aace78, m_tcid.getSize() && sig == m_tcid.back(), "Type compendium has been closed");
            }
            else if(m_tcid.getSize() == 0 || m_tcid.back() != sig)
            {
                m_tcid.pushBack(sig);
            }
        }

        hkArray<hkUint64> m_tcid;
    protected:
        State m_state;
    };
    Compendium m_compendium;

    int m_sizeofPointer;
    int m_minArrayAlign; 
    int m_maxDataAlign; 
    hkTypeVm::Compiler m_compiler;
    hkRefPtr<TypeWriter> m_typeWriter;
    hkUint32 m_prevTypeRollingHash;
    bool m_isPackfile;
    bool m_isMultiBundle;
    hkReflect::TypeCopier* m_copierToDelete;
    hkReflect::TypeHasher m_typeHasher;
    hkReflect::PropertyHasher m_propHasher;
    hkHashSet<TypeId> m_typeHashWritten;
};

namespace hkSerialize { namespace Detail {

struct TagfileTypeWriter : public hkSerialize::TypeWriter
{
protected:
    hkSerialize::Detail::TypeWriterMap m_typeMap;
    hkSerialize::TagfileWriteFormat::Impl* m_impl;
    TypeId m_highestWrittenId;

    void writeAllOutstandingTypes()
    {
        if(m_typeMap.isEmpty())
        {
            return;
        }

        hkIo::WriteBuffer buf(&m_impl->m_parts.type.encoded);
        while (const hkReflect::Type* type = m_typeMap.nextToWrite())
        {
            using namespace hkReflect;
            HK_UNITY_USING_ANONYMOUS_NAMESPACE;
            TypeId typeId = m_typeMap.lookupType(type);
            TypeId parentId = m_typeMap.enqueueForWrite(type->getParent());
            DLOG_SCOPE("Type ${} parent=${} tagoff={}// {} {}", typeId, parentId, buf.tell(), type->getName(), type);

            if(typeId > m_highestWrittenId)
            {
                m_highestWrittenId = typeId;
            }
            buf.writeObj<VarUint>( typeId );
            buf.writeObj<VarUint>( parentId );

            hkUlong optbits = TypeDetail::localGetOptionalMask(type);
            {
                // Don't write the opts directly, use an internal mapping which allows changing opts later. Plus it's smaller.
                unsigned mask = 0;
                for(int i = 0; i < HK_COUNT_OF(s_optMapping); ++i)
                {
                    if(optbits & s_optMapping[i])
                    {
                        mask |= (1 << i);
                    }
                }
                buf.writeObj<VarUint>(mask);
            }

            if( optbits & hkReflect::Opt::FORMAT )
            {
                hkUlong format = TypeDetail::localGetOptional<Opt::FORMAT>(type);
                DLOG("Format ${:x}", format);
                buf.writeObj<VarUint>( format );
            }
            if( optbits & hkReflect::Opt::SUBTYPE )
            {
                if( const ArrayType* at = type->asArray() )
                {
                    int count = at->getFixedCount();
                    TypeId tid = count
                        ? m_typeMap.enqueueForWrite(at->getSubType())
                        : m_typeMap.lookupType(at->getSubType());
                    buf.writeObj<VarUint>( tid );
                    DLOG("Subtype ${}//ArrayOf", tid);
                    if(count)
                        DLOG("FixedCount {}", count);
                }
                else if( const PointerType* pt = type->asPointer() )
                {
                    TypeId tid = m_typeMap.lookupType(pt->getSubType());
                    buf.writeObj<VarUint>( tid );
                    DLOG("Subtype ${}//PointerTo", tid);
                }
                else if(type->asRecord())
                {
                    
                    const hkReflect::Type* subType = hkReflect::TypeDetail::globalGetOptional<hkReflect::Opt::SUBTYPE>(type);
                    TypeId tid = m_typeMap.enqueueForWrite(subType); // Always write record subtypes, even if there is no object attached
                    buf.writeObj<VarUint>(tid);
                    DLOG("Subtype ${}//Other", tid);
                }
            }
            if( optbits & hkReflect::Opt::NAME )
            {
                const char* s = TypeDetail::localGetOptional<hkReflect::Opt::NAME>(type);
                DLOG("Name {}", s);
                m_impl->m_parts.type.setName(typeId, s);
            }
            else
            {
                m_impl->m_parts.type.setName(typeId, "");
            }
            if( optbits & hkReflect::Opt::VERSION )
            {
                hkUlong optVersion = TypeDetail::localGetOptional<hkReflect::Opt::VERSION>(type);
                buf.writeObj<VarUint>(optVersion);
                DLOG("Version {}", optVersion);
            }
            if( optbits & hkReflect::Opt::TEMPLATE )
            {
                const hkReflect::Template* optParams = type->getTemplate();
                const int numParams = optParams->getNumParams();
                DLOG_SCOPE("Template np={}", numParams);

                hkArray< hkTuple<hkUint64, hkUint64> > params;
                params.reserve(numParams);
                for(int i=0; i < numParams; i++)
                {
                    const hkReflect::Template::Parameter* param = optParams->getParam(i);

                    hkTuple<hkUint64, hkUint64>& cur = params.expandOne();
                    cur.m_0 = m_impl->m_parts.type.internTypeName(param->m_kindAndName);

                    if(param->isType())
                    {
                        TypeId typeParam = m_typeMap.enqueueForWrite(param->getAsType());
                        cur.m_1 = typeParam;
                        DLOG("TypeParam {} ${}", param->getName(), typeParam);
                    }
                    else
                    {
                        hkUlong valueParam = param->getAsValue();
                        cur.m_1 = valueParam;
                        DLOG("ValueParam {} {}", param->getName(), valueParam);
                    }
                }
                m_impl->m_parts.type.setTemplate(typeId, params);
            }
            if( optbits & hkReflect::Opt::SIZE_ALIGN )
            {
                hkReflect::Detail::SizeAlign sa( TypeDetail::localGetOptional<hkReflect::Opt::SIZE_ALIGN>(type) );
                buf.writeObj<VarUint>(sa.m_sizeOf);
                buf.writeObj<VarUint>(sa.m_alignOf | (sa.m_reqAlignEncoded<<12) );
                DLOG("{}", sa);
            }
            if( optbits & hkReflect::Opt::FLAGS )
            {
                hkUlong flags = TypeDetail::localGetOptional<hkReflect::Opt::FLAGS>(type);
                buf.writeObj<VarUint>(flags);
                DLOG("Flags {}", flags);
            }
            if( optbits & hkReflect::Opt::DECLS )
            {
                const hkReflect::Detail::DeclsArray* fieldArray = TypeDetail::localGetOptional<Opt::DECLS>(type);

                // Save both number of fields and properties. The total number will be used to preallocate the decls
                // array on load.
                int numFields = fieldArray->getNumDataFields();
                int numProps = fieldArray->getNumPropertyFields();
                HK_ASSERT_NO_MSG(0x6dd6f55e, numProps < 0xffff);
                buf.writeObj<VarUint>(numFields | (numProps << 16));

                // Save only data fields. Properties will be saved on-demand in a separate section.
                for(int i = 0; i < numFields; i++)
                {
                    const hkReflect::FieldDecl field = fieldArray->getField(i);

                    int fid = m_impl->m_parts.type.internFieldName(field.getName());
                    buf.writeObj<VarUint>(fid);
                    buf.writeObj<VarUint>(field.getFlags().get());
                    buf.writeObj<VarUint>(field.getOffset());
                    TypeId fieldId = m_typeMap.enqueueForWrite(field.getType()->getParent());
                    buf.writeObj<VarUint>(fieldId);
                    DLOG("Field name={} offset={} flags={}, type=${}", field.getName(), field.getOffset(), field.getFlags(), fieldId);
                }
            }
            if(optbits & Opt::INTERFACES)
            {
                const hkReflect::Detail::InterfaceArray* interfaceArray = TypeDetail::localGetOptional<Opt::INTERFACES>(type);
                HK_ASSERT_NO_MSG(0xe8b910c, interfaceArray);
                hkArrayView<const hkReflect::Detail::Interface> ifaces = interfaceArray->items();
                buf.writeObj<VarUint>(ifaces.getSize());
                for(int i = 0; i < ifaces.getSize(); ++i)
                {
                    TypeId tid = m_typeMap.enqueueForWrite(ifaces[i].m_interfaceType);
                    buf.writeObj<VarUint>(tid);
                    buf.writeObj<VarUint>(ifaces[i].m_offset);
                }
            }
            if (optbits & Opt::ATTRIBUTE_STRING)
            {
                const char* attrString = hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::ATTRIBUTE_STRING>(type);
                int attributeId = m_impl->m_parts.type.internAttribute(attrString);
                buf.writeObj<VarUint>(attributeId);
                DLOG("Attribute string=\"{}\" id={}", attrString, attributeId);
            }
        }
    }

public:

    TagfileTypeWriter(hkSerialize::TagfileWriteFormat::Impl* impl)
        : m_impl(impl), m_highestWrittenId(0)
    {
    }

        // Entry point
    TypeAndId writeType(_In_ const hkReflect::Type* type) HK_OVERRIDE
    {
        TypeId tid = m_typeMap.enqueueForWrite(type);
        writeAllOutstandingTypes();
        return TypeAndId(type, tid);
    }
    TypeId getHighestWrittenId() const HK_OVERRIDE
    {
        return m_highestWrittenId;
    }
};

struct MakePortableTypeWriter : public hkSerialize::TypeWriter
{
    MakePortableTypeWriter(TypeWriter* next)
        : m_copier(hkReflect::TypeCopier::Options())
        , m_next(next)
    {
    }
    TypeAndId writeType(_In_ const hkReflect::Type* srcType) HK_OVERRIDE
    {
        const hkReflect::Type* type = m_copier.copy(srcType);
        return m_next->writeType(type);
    }
    hkReflect::TypeCopier m_copier;
    hkRefPtr<TypeWriter> m_next;
};

struct MakeTargetedTypeWriter : public hkSerialize::TypeWriter
{
    MakeTargetedTypeWriter(_In_ TypeWriter* next, _In_ hkReflect::TypeCopier* copier)
        : m_copier(copier)
        , m_next(next)
    {
    }
    TypeAndId writeType(_In_ const hkReflect::Type* srcType) HK_OVERRIDE
    {
        const hkReflect::Type* type = m_copier->copy(srcType);
        return m_next->writeType(type);
    }
    TypeId getHighestWrittenId() const HK_OVERRIDE
    {
        return m_next->getHighestWrittenId();
    }
    hkReflect::TypeCopier* m_copier;
    hkRefPtr<TypeWriter> m_next;
};

} } // end namespace


hkSerialize::TagfileWriteFormat::Impl::Impl(_Inout_opt_ hkReflect::TypeCopier* copier, bool isPackfile, bool deleteCopier)
    : m_sizeofPointer( copier ? copier->getPointerSize() : sizeof(void*) )
    , m_minArrayAlign(isPackfile ? 4*(copier ? copier->getSizeofReal() : sizeof(hkReal)) : 0)
    , m_maxDataAlign(isPackfile ? m_sizeofPointer : 4)
    , m_prevTypeRollingHash(0)
    , m_isPackfile(isPackfile)
    , m_isMultiBundle(false)
    , m_copierToDelete(deleteCopier ? copier : HK_NULL)
{
    TypeWriter* tw = new Detail::TagfileTypeWriter(this);
    if(copier)
    {
        m_typeWriter.setAndDontIncrementRefCount(new Detail::MakeTargetedTypeWriter(tw,copier));
        tw->removeReference();
    }
    else
    {
        m_typeWriter.setAndDontIncrementRefCount(tw);
    }

    hkTypeVm::addDefaultPasses(m_compiler);
    m_compiler.addPass<hkTypeVm::IntAndFloatConversionPass>();
    m_compiler.excludeNonSerializableFields = true;
    m_compiler.addInstrWithNoSource = false;
}

hkSerialize::TagfileWriteFormat::Impl::~Impl()
{
    if(m_compendium.getState() == Compendium::WRITING)
    {
        Log_Error("beginTypeCompendium called without matching endTypeCompendium");
    }
    delete m_copierToDelete;
}

void hkSerialize::TagfileWriteFormat::Impl::beginBundle(_In_ hkIo::WriteBuffer* writeBuffer)
{
    DLOG_BEGIN("BeginBundle");
    HK_ASSERT(0x51ff02d4, m_parts.data.isEmpty() && m_parts.index.isEmpty(), "Internal error: bundle is not empty.");
    HK_ASSERT(0x5f0ab2d4, m_prevTypeRollingHash==0 || m_isMultiBundle, "Trying to write more than one bundle into a write buffer not enabled for multibundle (streaming)."
        " You must explicitly enable multibundle writing (streaming mode) before writing the first bundle by calling hkSerialize::Save::withMultiBundle() or using a format object"
        " and calling hkSerialize::TagfileWriteFormat::enableMultiBundle() on it.");
    m_writeBuffer = writeBuffer;
    m_maxDataAlign = 1;
}

static int paddingFor( hkUlong off, int align )
{
    HK_ASSERT_NO_MSG(0x22ccdd17, hkMath::isPower2(align));

    if( int r = off % align )
    {
        return align - r;
    }
    return 0;
}

static const signed char s_paddingZero[32] = {};
static const signed char s_paddingMinusOne[32] =
{
    -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1,
    -1, -1, -1, -1, -1, -1, -1, -1,
};
static void padUp( hkIo::WriteBuffer& buf, int reqAlign, int& maxAlign, const signed char (&padding)[32] = s_paddingZero)
{
    int npad = paddingFor( buf.tell(), reqAlign );
    while( npad )
    {
        int w = hkMath::min2(npad, sizeof(padding));
        buf.writeRaw(padding, w);
        npad -= w;
    }
    maxAlign = hkMath::max2(maxAlign, reqAlign);
}
static void padUp(hkArray<char>& buf, int reqAlign, int& maxAlign, char padding = 0)
{
    if( int npad = paddingFor(buf.getSize(), reqAlign) )
    {
        buf.expandBy(npad, padding);
    }
    maxAlign = hkMath::max2(maxAlign, reqAlign);
}

void hkSerialize::TagfileWriteFormat::Impl::writeTypeSection(hkSerialize::HffWriter& hff, hkIo::WriteBuffer& buf)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;

    if (m_parts.type.hasMoreToWrite() == false)
    {
        return;
    }

    hff.openBranch(TagfileSection::TYPES);
    if(m_isPackfile)
    {
        hff.openLeaf(TagfileTypeSection::NATIVE_POINTERS);
        buf.writeZero((m_typeWriter->getHighestWrittenId() + 1) * m_sizeofPointer);
        hff.close();
    }

    if(m_isMultiBundle)
    {
        hkUint32 newHash = hkTruncateCast<hkUint32>(m_parts.type.calcCompendiumSignature());
        hff.openLeaf(TagfileTypeSection::ROLLING_HASH);
        buf.writePod<hkUint32Le>(m_prevTypeRollingHash);
        buf.writePod<hkUint32Le>(newHash);
        hff.close();
        m_prevTypeRollingHash = newHash;
    }
    else
    {
        m_prevTypeRollingHash = 1; // non-zero value for error detection (see assert 0x5f0ab2d4 in beginBundle()).
    }

    if(m_parts.type.typeNameStrings.hasMoreToWrite())
    {
        hff.openLeaf(TagfileTypeSection::NAME_STRINGS);
        m_parts.type.typeNameStrings.writeTo(buf);
        padUp(buf, 4, m_maxDataAlign);
        hff.close();
    }

    if(m_parts.type.typeIdentity.hasMoreToWrite())
    {
        hff.openLeaf(TagfileTypeSection::IDENTITY);
        m_parts.type.typeIdentity.writeTo(buf);
        padUp(buf, 4, m_maxDataAlign);
        hff.close();
    }

    if(m_parts.type.fieldNameStrings.hasMoreToWrite())
    {
        hff.openLeaf(TagfileTypeSection::FIELD_NAME_STRINGS);
        m_parts.type.fieldNameStrings.writeTo(buf);
        padUp(buf, 4, m_maxDataAlign);
        hff.close();
    }

    if (m_parts.type.attributeStrings.hasMoreToWrite())
    {
        hff.openLeaf(TagfileTypeSection::ATTRIBUTE_STRINGS);
        m_parts.type.attributeStrings.writeTo(buf);
        // Pad with 'ff' to allow empty interned strings.
        padUp(buf, 4, m_maxDataAlign, s_paddingMinusOne);
        hff.close();
    }

    if(m_parts.type.encoded.hasMoreToWrite())
    {
        hff.openLeaf(TagfileTypeSection::BODY);
        m_parts.type.encoded.writeTo(buf);
        padUp(buf, 4, m_maxDataAlign);
        hff.close();
    }

    if( m_parts.type.typeHashes.hasMoreToWrite() )
    {
        HK_ASSERT_NO_MSG(0xfc8c6b2, m_isPackfile);
        hff.openLeaf(TagfileTypeSection::HASHES);
        m_parts.type.typeHashes.writeTo(buf);
        padUp(buf, 4, m_maxDataAlign);
        hff.close();
    }

    if ( m_parts.type.propertyDecls.hasMoreToWrite() )
    {
        hff.openLeaf( TagfileTypeSection::PROPERTIES );
        m_parts.type.propertyDecls.writeTo( buf );
        padUp( buf, 4, m_maxDataAlign );
        hff.close();
    }

    if ( m_parts.type.propertyHashes.hasMoreToWrite() )
    {
        HK_ASSERT_NO_MSG(0x6fb25cf1, m_isPackfile);
        hff.openLeaf( TagfileTypeSection::PROPS_HASHES );
        m_parts.type.propertyHashes.writeTo( buf );
        padUp( buf, 4, m_maxDataAlign );
        hff.close();
    }

    {
        hff.openLeaf(TagfileTypeSection::PADDING);
        padUp(buf, 4, m_maxDataAlign);
        hff.close();
    }
    hff.close(); // TagfileSection::TYPE_SECTION;
}

void hkSerialize::TagfileWriteFormat::Impl::endBundle()
{
    hkSerialize::HffWriter hff(*m_writeBuffer);

    hff.openBranch(TagfileRoot::TAGFILE);
    {
        // SDKV goes directly before DATA and it MUST be 8 bytes long.
        // This naturally places DATA at a 32 byte aligned offset.
        // 8 for TAGF, 8 for SDKV, 8 payload, 8 for DATA
        hkStringBuf sb; sb.printf("%i", HAVOK_REFLECTION_VERSION);
        HK_ASSERT_EXPR(0xc6d19b0, sb.getLength(), ==, 8);
        hff.writeLeaf(TagfileSection::SDK_VERSION, sb.cString(), sb.getLength());
    }

    // Adjust pointer ids and compact unused sections
    hkArray<hkUint32>::Temp newIndices; newIndices.reserveExactly(m_parts.index.m_items.getSize());

    // The zero item is always present
    newIndices.pushBackUnchecked(0);
    // Assuming that unreferenced pointers are relatively rare and it is more
    // efficient to occasionally modify the items inplace than make a new array and copy
    // the used elements
    for (int idx = 1; idx < m_parts.index.m_items.getSize(); ++idx)
    {
        // Remove any unused sections
        if (m_parts.index.m_items[idx].isEmpty() && (m_parts.index.m_items[idx].count == 0))
        {
            m_parts.index.m_items.removeAtAndCopy(idx);
            idx--;
            newIndices.pushBackUnchecked(0);
        }
        else
        {
            // Remap any remaining sections from the old index to the new
            newIndices.pushBackUnchecked(idx);
        }
    }

    // Resolve the references to the compacted item ids
    for (int i = 0; i < m_parts.index.m_objectRef.getSize(); i++)
    {
        const hkUint32 remappedId = newIndices[m_parts.index.m_objectRef[i].vid];
        if (remappedId != 0)
        {
            *reinterpret_cast<hkUint32Le*>(m_parts.data.begin() + m_parts.index.m_objectRef[i].offset) = remappedId;
        }
    }

    if(m_parts.data.getSize())
    {
        hff.openLeaf(TagfileSection::DATA);
        m_writeBuffer->writeRaw(m_parts.data.begin(), m_parts.data.getSize());
        padUp(*m_writeBuffer, m_maxDataAlign, m_maxDataAlign);
        hff.close();
    }

    if(m_compendium.getState() == Compendium::NOT_USING)
    {
        writeTypeSection(hff, *m_writeBuffer);
    }
    else
    {
        hff.openLeaf(TagfileSection::COMPENDIUM_REFERENCE);
        hkUint64 sig = m_parts.type.calcCompendiumSignature();
        m_compendium.addSignature(sig);
        m_writeBuffer->writePod<hkUint64Le>(sig);
        if(m_isPackfile)
        {
            // packfile stashes an arrayview of types here for easy unload
            m_writeBuffer->writeZero(2 * m_sizeofPointer);
        }
        hff.close();
    }

    if(m_parts.index.m_items.getSize())
    {
        hff.openBranch(TagfileSection::INDEX);

        {
            hff.openLeaf(TagfileIndexSection::ITEMS);
            for(int idx = 0; idx < m_parts.index.m_items.getSize(); ++idx)
            {
                const Detail::TagfileItem& it = m_parts.index.m_items[idx];
                m_writeBuffer->writePod<hkUint32Le>(it.packed);
                m_writeBuffer->writePod<hkUint32Le>(it.offset);
                if (it.isEmpty() || it.isKindNote())
                {
                    // The note id is encoded in the count
                    const hkUint32 remappedId = newIndices[it.count];
                    m_writeBuffer->writePod<hkUint32Le>(remappedId);
                }
                else
                {
                    m_writeBuffer->writePod<hkUint32Le>(it.count);
                }
            }
            hff.close();
        }

        if(m_isPackfile)
        {
            if( m_parts.index.m_inplace.getSize() )
            {
                hff.openLeaf(TagfileIndexSection::PATCH_DATA);
                hkSort(m_parts.index.m_inplace.begin(), m_parts.index.m_inplace.getSize());
                m_parts.index.inplace(-1, 0); // sentinel
                hkArray<int>::Temp curOffsets;
                hkUint32 curTid = hkUint32(-1);
                for(int idx = 0; idx < m_parts.index.m_inplace.getSize(); ++idx )
                {
                    const Parts::Index::Inplace& it = m_parts.index.m_inplace[idx];
                    if(it.tid != curTid)
                    {
                        if(curOffsets.getSize())
                        {
                            m_writeBuffer->writePod<hkUint32Le>(curTid);
                            m_writeBuffer->writePod<hkUint32Le>(curOffsets.getSize());
                            for(int i = 0; i < curOffsets.getSize(); ++i)
                            {
                                m_writeBuffer->writePod<hkUint32Le>(curOffsets[i]);
                            }
                            curOffsets.clear();
                        }
                        curTid = it.tid;
                    }
                    curOffsets.pushBack(it.offset);
                }
                // current is sentinel
                HK_ASSERT_EXPR(0x374bfe07, curOffsets.getSize(), ==, 1, &&, curOffsets.back(), ==, 0);
                hff.close();
            }
        }

        hff.close();
    }

    hff.close();
    DLOG("EndBundleDataAlign={}", m_maxDataAlign);
    DLOG_END();

    m_writeBuffer = HK_NULL;
    m_parts.data.clearAndDeallocate();
    m_parts.index.clear();
}


void hkSerialize::TagfileWriteFormat::Impl::beginTypeCompendium(const hkIo::Detail::WriteBufferAdapter& sink)
{
    HK_ASSERT(0x3e5a39fb, m_parts.data.isEmpty() && m_parts.type.isEmpty() && m_parts.index.isEmpty(),
        "Writer must be empty before beginning a compendium");
    if(m_compendium.getState() != Compendium::NOT_USING)
    {
        Log_Error("Type Compendium already begun");
        return;
    }
    m_compendium.setState(Compendium::WRITING);
    m_compendium.m_adapter = sink;
}

void hkSerialize::TagfileWriteFormat::Impl::endTypeCompendium()
{
    HK_ASSERT(0x3eca4d2b, m_parts.data.isEmpty() && m_parts.index.isEmpty(), "Bundle not ended.");
    if(m_compendium.getState() != Compendium::WRITING)
    {
        Log_Error("Type Compendium not begun");
        return;
    }
    m_compendium.setState(Compendium::CLOSED);
    hkIo::WriteBuffer buf(m_compendium.m_adapter);
    HffWriter hff(buf);
    hff.openBranch(TagfileRoot::COMPENDIUM);
    {
        {
            hff.openLeaf(TagfileSection::COMPENDIUM_ID);
            for(int i = 0; i < m_compendium.m_tcid.getSize(); ++i)
            {
                buf.writePod<hkUint64Le>(m_compendium.m_tcid[i]);
            }
            hff.close();
        }

        writeTypeSection(hff, buf);
    }
    hff.close();
}

hkResult hkSerialize::TagfileWriteFormat::Impl::enableMultiBundle()
{
    if (m_parts.type.isEmpty())
    {
        m_isMultiBundle = true;
        return HK_SUCCESS;
    }
    Log_Warning("You must enable multibundle before writing any data");
    return HK_FAILURE;
}

// Write a hash into the file for the given type/id pair.
// We do this everywhere we need native types to be valid in the bundle.
// For example we need the type of all top level VAR0 and VARN
// We need native type for the inplace fixups INP0 and INPN and also for TREF.
void hkSerialize::TagfileWriteFormat::Impl::typeHash(TypeAndId tid)
{
    if(m_isPackfile)
    {
        if( m_typeHashWritten.contains(tid.m_1) == false )
        {
            // We keep a separate map of type hashes written.
            // We don't attach a hash to every type (which we could have done in writeType)
            // but instead only hash types which correspond to top-level types.
            m_typeHashWritten.insert(tid.m_1);
            hkUint32 sig = m_typeHasher.calc(tid.m_0);
            m_parts.type.addHash(tid.m_1, sig);
            DLOG("Type {} hash {:x}// {}", tid.m_1, sig, tid.m_0);
        }
    }
}

int hkSerialize::TagfileWriteFormat::Impl::findDeclIndex(hkReflect::Decl decl, TypeAndId contextAndId)
{
    // Look for a decl with the same name in the serialized context type.
    const hkReflect::Type* context = contextAndId.m_0;
    TypeId contextId = contextAndId.m_1;
    auto decls = hkReflect::TypeDetail::getDeclsArray(context);
    HK_ASSERT(0x2a2b092c, decls, "Context must have a decls array");
    for (int i = 0; i < decls->getNumDecls(); ++i)
    {
        const char* declName = decl.getName();
        hkReflect::Decl declInSerializedType = decls->getDecl( i );
        if ( hkString::strCmp( declInSerializedType.getName(), declName) == 0 )
        {
            if ( declInSerializedType.asPropertyField() )
            {
                // The decl is a property and it is serialized on demand. Insert it into the map if it is not already there.
                PropDecl prop = hkTupleT::make( contextId, i );

                int& internedName = m_parts.type.propertyDecls.getOrInsertKey( prop, -1 );
                if ( internedName < 0 )
                {
                    // Decl was new, intern the name and fix it up.
                    internedName = m_parts.type.internFieldName( declName );
                    DLOG( "Added decl ctx=${} idx={} name='{}'", contextId, i, declName );

                    if ( m_isPackfile )
                    {
                        // Write property hash if the type is new.
                        bool typeIsNew;
                        hkUint32 propHash = m_propHasher.calc( context, typeIsNew );
                        if ( typeIsNew )
                        {
                            m_parts.type.propertyHashes.pushBack( TypeIdAndHash( contextId, propHash ) );
                            DLOG( "Type {} property hash {:x}// {}", contextId, propHash, context );
                        }
                    }
                }
            }
            return i;
        }
    }
    HK_ASSERT(0x74dc7f1e, false, "Context does not contain decl");
    return -1;
}

hkResult hkSerialize::TagfileWriteFormat::Impl::writeInternal(const hkReflect::Var& topvar, VarId vid, bool isVar0, _Inout_ IdFromVar* idFromVar)
{
    if(const hkReflect::Type* srcType = topvar.dynCast<hkReflect::Type>())
    {
        if (hkReflect::Decl asDecl = hkReflect::QualType(srcType))
        {
            // Write context
            TypeAndId tid = m_typeWriter->writeType(asDecl.getDeclContext());
            typeHash(tid);

            // Get index of the Decl in the type.
            int declIndex = findDeclIndex(asDecl, tid);

            // Write decl reference.
            m_parts.index.addDecl(vid, tid.m_1, declIndex);
            DLOG_SCOPE("DECL id=#{} context=${} index={} dataoff={} //{}", vid, tid.m_0, declIndex, m_parts.data.getSize(), asDecl.getName());
        }
        else
        {
            TypeAndId tid = m_typeWriter->writeType(srcType);
            DLOG_SCOPE("TYPE id=#{} type=${} dataoff={} //{}", vid, tid.m_0, m_parts.data.getSize(), hkReflect::exactTypeOf(srcType)->getName());
            m_parts.index.addType(vid, tid.m_1);
            typeHash(tid);
        }

    }
    else
    {
        m_interp.clear();
        m_interp.m_idFromVar = idFromVar;
        m_interp.m_fixups = this;
        m_interp.m_isPackfile = m_isPackfile;

        {
            TypeAndId dstTypeAndId = m_typeWriter->writeType(topvar.getType());
            const hkReflect::Type* dstType = dstTypeAndId.m_0;
            TypeId tid = dstTypeAndId.m_1;
            padUp(m_parts.data, dstType->getAlignOf(), m_maxDataAlign);
            DLOG_SCOPE("{} id=#{} type=${} dataoff={} //{}", isVar0 ? "VAR0" : "NOTE", vid, tid, m_parts.data.getSize(), topvar.getType()->getName());

            typeHash(dstTypeAndId);

            if(isVar0)
            {
                m_parts.index.addVar0(vid, tid, m_parts.data.getSize());
            }
            else
            {
                m_parts.index.addNote(vid, idFromVar->lookup(topvar), tid, m_parts.data.getSize());
            }

            const hkTypeVm::Program* program = m_compiler.compile(topvar.getType(), dstType);
            const int dstSize = dstType->getSizeOf();
            void* dstLocation = m_parts.data.expandBy(dstType->getSizeOf());
            hkString::memSet(dstLocation, 0, dstSize);

            HK_VERIFY(0x30a03e4c, hkTypeVm::FastCopyInterpreter::exec1( m_interp, program,
                dstLocation, dstSize,
                topvar.getAddress(), topvar.getType()->getSizeOf() ).isSuccess(), "Object write failed" );
        }

        for( int extraIndex = 0; extraIndex < m_interp.m_extras.getSize(); ++extraIndex )
        {
            hkReflect::Var var = m_interp.m_extras[extraIndex].m_0;
            ExtraId eid = m_interp.m_extras[extraIndex].m_1;
            if(hkReflect::ArrayVar avar = var)
            {
                hkReflect::ArrayValue aval = avar.getValue();
                const hkReflect::Type* srcElemType = aval.getSubType();
                HK_ASSERT_NO_MSG(0xf8b1f9a, srcElemType != HK_NULL);

                TypeAndId dstElemTypeAndId = m_typeWriter->writeType(srcElemType);
                const hkReflect::Type* dstElemType = dstElemTypeAndId.m_0;
                TypeId tid = dstElemTypeAndId.m_1;
                int count = aval.getCount();

                padUp(m_parts.data, hkMath::max2(m_minArrayAlign, dstElemType->getAlignOf()), m_maxDataAlign);
                DLOG_SCOPE("VARN id=#{} elem=${} count={} data={} //(array)", eid, tid, count, m_parts.data.getSize());

                typeHash(dstElemTypeAndId);
                m_parts.index.addVarN(eid, tid, m_parts.data.getSize(), count);

                if( count )
                {
                    int srcElemSize = srcElemType->getSizeOf();
                    int dstElemSize = dstElemType->getSizeOf();
                    int srcTotalSize = count * srcElemSize;
                    int dstTotalSize = count * dstElemSize;

                    const hkTypeVm::Program* program = m_compiler.compile(srcElemType, dstElemType);
                    void* extraLocation = m_parts.data.expandBy(dstTotalSize);
                    hkString::memSet(extraLocation, 0, dstTotalSize);
                    HK_VERIFY(0x69761cf3, hkTypeVm::FastCopyInterpreter::execN( m_interp, program,
                        extraLocation, dstTotalSize,
                        aval.getAddress(), srcTotalSize,
                        dstElemSize, srcElemSize, count ).isSuccess(), "Array write failed" );
                }
            }
            else if( hkReflect::StringVar svar = var )
            {
                const hkReflect::Type* srcElemType = hkReflect::getType<char>();
                TypeAndId typeAndId = m_typeWriter->writeType(srcElemType);
                TypeId tid = typeAndId.m_1;
                hkReflect::StringValue sval = svar.getValue();
                int count = hkString::strLen(sval) + 1; // save the null so we can use it inplace on load
                DLOG_SCOPE("VARN id=#{} elem=${} count={} data={}//(string) {}", eid, tid, count, m_parts.data.getSize(), (const char*)sval);

                typeHash(typeAndId);

                if( m_isPackfile ) 
                {
                    padUp(m_parts.data, 2, m_maxDataAlign);
                }
                m_parts.index.addVarN(eid, tid, m_parts.data.getSize(), count);
                m_parts.data.append(sval, count);
            }
            else
            {
                HK_ASSERT(0x31ba1338,false,"Unknown contents type");
                return HK_FAILURE;
            }
        }
    }
    return HK_SUCCESS;
}

hkResult hkSerialize::TagfileWriteFormat::Impl::write(const hkReflect::Var& topvar, IdFromVar& idFromVar)
{
    VarId vid = idFromVar.lookup(topvar);
    idFromVar.writing(vid);

    return writeInternal( topvar, vid, true, &idFromVar );
}

hkResult hkSerialize::TagfileWriteFormat::Impl::writeNote(VarId vid, const hkReflect::Var& note, IdFromVar& idFromVar)
{
    return writeInternal( note, vid, false, &idFromVar );
}

hkSerialize::TagfileWriteFormat::TagfileWriteFormat()
    : m_impl( new Impl( new hkReflect::TypeCopier(hkReflect::TypeCopier::Options()), false, true ) )
{
}


hkSerialize::TagfileWriteFormat::TagfileWriteFormat(_In_z_ const char* platform)
    : m_impl(Impl::make(platform))
{
}

hkSerialize::TagfileWriteFormat::TagfileWriteFormat(_Inout_ hkReflect::TypeCopier* c)
    : m_impl( new Impl(c,true,false) )
{
}

hkSerialize::TagfileWriteFormat::~TagfileWriteFormat()
{
    delete m_impl;
}

void hkSerialize::TagfileWriteFormat::beginBundle(_In_ hkIo::WriteBuffer* writeBuffer)
{
    m_impl->beginBundle(writeBuffer);
}

void hkSerialize::TagfileWriteFormat::endBundle()
{
    m_impl->endBundle();
}

hkResult hkSerialize::TagfileWriteFormat::write(const hkReflect::Var& topvar, IdFromVar& idFromVar)
{
    return m_impl->write(topvar, idFromVar);
}

hkResult hkSerialize::TagfileWriteFormat::writeNote(VarId vid, const hkReflect::Var& note, IdFromVar& idFromVar)
{
    return m_impl->writeNote(vid, note, idFromVar);
}

void hkSerialize::TagfileWriteFormat::beginTypeCompendium(const hkIo::Detail::WriteBufferAdapter& sink)
{
    m_impl->beginTypeCompendium(sink);
}

void hkSerialize::TagfileWriteFormat::endTypeCompendium()
{
    m_impl->endTypeCompendium();
}

hkResult hkSerialize::TagfileWriteFormat::enableMultiBundle()
{
    return m_impl->enableMultiBundle();
}

hkRefNew<hkSerialize::WriteFormat> hkSerialize::WriteFormat::createDefault()
{
    return new TagfileWriteFormat;
}

#ifndef HK_DYNAMIC_DLL
static
#endif
hkRefNew<hkSerialize::WriteFormat> binaryTagfileCreateWrite()
{
    HK_OPTIONAL_COMPONENT_MARK_USED(hkWriteFormatBinaryTagfile);
    return new hkSerialize::TagfileWriteFormat();
}

HK_OPTIONAL_COMPONENT_DEFINE(hkWriteFormatBinaryTagfile, hkSerialize::Detail::fileFormatBinaryTagfile.m_writeFormatCreateFunc, binaryTagfileCreateWrite);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
