// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileReadFormat.h>
#include <Common/Base/Serialize/Format/Tagfile/Detail/hkTagfileDetail.h>
#include <Common/Base/Algorithm/Find/hkFind.h>
#include <Common/Base/Config/hkOptionalComponent.h>
#include <Common/Base/Memory/Allocator/Transient/hkTransientAllocator.h>
#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Reflect/Detail/hkTypeHasher.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/Reflect/Util/hkReflectionCheckUtil.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>
#include <Common/Base/Serialize/Detail/hkSerializeDetail.h>
#include <Common/Base/Serialize/Detail/hkIndexedBundle.h>
#include <Common/Base/Serialize/Detail/hkUintVle.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/Structured/hkStructuredStream.h>
#include <Common/Base/Types/Uuid/hkUuid.h>
#include <Common/Base/Types/hkBaseDefs.h>
#include <Common/Base/Types/hkEndian.h>


// Define for extra checking on load
//#define TAGFILE_EXTRA_LOAD_CHECKING

#if defined(HK_BUILDING_WITH_ENGINE)
    #include <Common/NewBase/Reflection/Util/hkDynamicTypeManager.h> 
#endif

#define DEBUG_LOG_IDENTIFIER "s11n.TagfileReadFormat"
#include <Common/Base/System/Log/hkLog.hxx>

#define FAIL_IF( COND, ... ) \
    if ( (COND) ) \
    { \
        Log_Error( __VA_ARGS__ ); \
        m_status = HK_FAILURE; \
        return; \
    }

#define FAIL_IF_NO_MSG( ... ) \
    if ( (__VA_ARGS__) ) \
    { \
        Log_Error("{}", #__VA_ARGS__ ); \
        m_status = HK_FAILURE; \
        return; \
    }


#define FAIL_WITH_ID_IF( ID, COND, ... ) \
    if ( (COND) ) \
    { \
        Log_Error( __VA_ARGS__ ).setId( ID ); \
        m_status = HK_FAILURE; \
        return; \
    }

#define FAIL_IF_INVALID( RB ) \
    if(!(RB).isOk()) m_status = HK_FAILURE; \
    if( m_status.isFailure() ) \
    { \
        Log_Error("Error in input stream."); \
        return; \
    }

#define FAIL_ALWAYS( ... ) \
    { \
        Log_Error( __VA_ARGS__ ); \
        m_status = HK_FAILURE; \
        return; \
    }

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    template<typename IDX, typename ARR>
    inline bool indexValid(IDX i, const ARR& arr)
    {
        hkUlong size = arr.getSize();
        return i >= 0 && i < size;
    }

        // Read a full 64 bit integer
    HK_NEVER_INLINE hkUint64 readUintVle64(hkIo::ReadBufferView& reader)
    {
        // If the data is ill formed, we could potentially be at the end of the section/ReadBufferView
        // Seeking past the end would assert, so we need to peek one byte but request 0 instead
        const hkUint8* p = reader.template peekAt<hkUint8>(0, 0);

        // This is where we actually check that there is at least one byte available
        // The current section is guaranteed to be followed by a padding of 8 bytes (END_PADDING_FOR_VLE)
        // So accessing the memory required by the largest possible VLE (hkUintVle::MAX_BYTE_COUNT) is safe
        if (p != reader.end())
        {
            int n = 0;
            hkUintVle ret = hkUintVle::read(p, n);

            // If the VLE is ill formed, hkUintVle::read will report a count of 0 bytes
            // We need to ensure this properly triggers an error condition in the ReadBufferView
            if (n > 0)
            {
                // If the VLE is consistent but goes beyond the end of the ReadBufferView and into the padding section,
                // the call to skip will set the EOF bit which will be tested by the calling code of this function
                reader.skip(n);
                return ret;
            }
        }

        reader.status().setUnexpected();
        return 0;
    }

        // Check that the potentially 64 bit value fits in the given type
    template<typename DST>
    DST readUintVle(hkIo::ReadBufferView& reader)
    {
        hkUint64 src = readUintVle64(reader);
        DST dst = static_cast<DST>(src);
        if ( src > (hkUint64)hkTrait::NumericLimits<DST>::maxValue() )
        {
            reader.status().setUnexpected();
        }
        return dst;
    }

    // The implementation of variable length integer reading needs to be able to perform an unaligned load of
    // hkUintVle::MAX_BYTE_COUNT bytes from the current position. We need to reserve some more space in case
    // it happens to fall at the end of a memory page.
    enum { END_PADDING_FOR_VLE = hkUintVle::MAX_BYTE_COUNT - 1 };

    //
    // FORMAT http://confluence.havok.com/display/com/Tagfile+2015
    //

        /// Copy a section from the stream into a permanent buffer
    hkResult readInto(hkIo::ReadBufferView& rb, hkSerialize::HffStreamReader& reader, hkMemoryAllocator& allocator)
    {
        hkLong size = reader.sizeContent();
        void* p = allocator.blockAlloc(hkLosslessCast<int>(size + END_PADDING_FOR_VLE));
        if (reader.read(p, size) == size)
        {
            rb.init(p, size);
            return HK_SUCCESS;
        }
        else
        {
            Log_Error("Invalid section size");
            reader.getReadBuffer().status().setUnexpected();
            return HK_FAILURE;
        }
    }

        /// Copy a section from the ReadBufferView into a permanent buffer
    hkResult readInto(hkIo::ReadBufferView& rb, hkIo::ReadBufferView sourceRb, hkMemoryAllocator& allocator)
    {
        hkLong n = sourceRb.remaining();
        void* p = allocator.blockAlloc(hkLosslessCast<int>(n + END_PADDING_FOR_VLE));
        hkMemUtil::memCpy(p, sourceRb.begin(), n);
        rb.init(p,n);
        return HK_SUCCESS;
    }


        /// Riaa object to view a range and advance afterwards.
    struct TempReadBuffer : public hkIo::ReadBufferView
    {
        HK_NEVER_INLINE TempReadBuffer(hkSerialize::HffStreamReader& hff, hkResult& status, bool pad = true)
            : m_rb(hff.getReadBuffer())
            , m_size(hff.sizeContent())
        {
            HK_ASSERT_NO_MSG(0x4f3de411, status.isSuccess());
            // prefetch a little more so we can use hkUintVle.read
            hkLong paddedSize = m_size + (pad ? END_PADDING_FOR_VLE : 0);
            if(m_rb.prefetch(paddedSize) >= paddedSize)
            {
                m_cur = m_rb.peekAt<void>(0, m_size);
                m_end = hkAddByteOffset(m_cur, m_size);
                m_start = m_cur;
            }
            else
            {
                Log_Warning("Short read in section {:x}", hff.curIdent());
                status = HK_FAILURE;
            }
        }
        ~TempReadBuffer()
        {
            m_rb.skip(m_size);
        }
        hkIo::ReadBuffer& m_rb;
        hkLong m_size;
    };

    struct LocalTypeHasher
    {
        const hkReflect::TypeReg* m_reg;
        hkReflect::TypeHasher m_hasher;
        LocalTypeHasher(const hkReflect::TypeReg* reg) : m_reg(reg) {}

        _Ret_maybenull_ const hkReflect::Type* findNative(_In_ const hkReflect::Type* src, hkUint32 hash)
        {
            if(const hkReflect::Type* dst = m_reg->typeFromType(src))
            {
                hkUint32 h = m_hasher.calc(dst);
                if(h == hash)
                {
                    return dst;
                }
            }
            return HK_NULL;
        }
    };

    
    template<typename T>
    _Ret_maybenull_ T* findEntry( hkSerialize::Ident ident, hkArrayView<T> table )
    {
        for ( const auto& entry : table )
        {
            if ( entry.m_ident == ident )
            {
                return &entry;
            }
        }
        return HK_NULL;
    }
}

namespace hkSerialize
{
namespace Detail
{
    /// Optimized storage of several TypeRelocs. Note the storage is inverted. TypeRelocs stores
    /// the various types referenced by a type, this stores a list of locations pointing to a given type.
    /// Specialization for Tagfile where placeholder are checked more tightly for security.
    struct TagfileTypeRelocs
    {
        /// Add a fallback type for a given type ID.
        /// This fallback is used when applying relocs if the type doesn't have a body yet (streaming).
        void addFallbackType( TypeId typeId, _In_ const hkReflect::Type* fallbackType )
        {
            HK_ASSERT_NO_MSG(0x69557bdf, fallbackType );
            RelocInfo defVal;
            auto const it = m_typeRelocs.findOrInsertKey( typeId, defVal );
            auto& relocInfo = m_typeRelocs.getValue( it );
            HK_ASSERT(0x75dee20c, relocInfo.m_relocs.isEmpty(), "Cannot add fallback type for type ID {} after some relocations were already added.", typeId );
            relocInfo.m_placeholderType = fallbackType;
        }

        /// Notify that we now have the given type with the given typeId.
        /// Resolve existing pending relocations with the given type as target after it was loaded.
        void resolve( TypeId typeId, _In_ const hkReflect::Type* type );

        /// Apply relocations on a newly loaded type, either registering pending relocations if the source type
        /// is not loaded yet, or resolving them if the source type is already available.
        void flatten( _In_ hkReflect::Type* type, const TypeRelocs& relocs, hkArrayView<const hkReflect::Type*> types );

    protected:

        /// Attempt to apply a relocation to address @addr of type @ownerType with type ID @typeId.
        /// The list of loaded types is passed in @types, and if @typeId is present the relocation is applied.
        /// Otherwise the fallback collection is used, or if empty a NULL pointer is left at @addr.
        void applyRelocation( _In_ const hkReflect::Type* ownerType, _Inout_ const hkReflect::Type** addr, TypeId id, hkArrayView<const hkReflect::Type*> types );

        /// Single relocation.
        struct Reloc
        {
            /// Target to reloc.
            const hkReflect::Type** m_target;

            /// Owner type where the reloc target sits.
            const hkReflect::Type* m_ownerType;
        };

        /// Relocation information for a single type.
        struct RelocInfo
        {
            /// Placeholder type when actual type not loaded yet.
            /// This can be either the type identity for the given type ID if available, or s_noIdentity if not.
            const hkReflect::Type* m_placeholderType;

            /// Relocations.
            hkArray<Reloc> m_relocs;
        };

        /// Helper to find a Reloc by its address.
        struct FindRelocByAddress
        {
            FindRelocByAddress( const hkReflect::Type** addr )
                : m_addr( addr )
            {
            }

            bool operator()( Reloc& reloc ) const
            {
                return ( reloc.m_target == m_addr );
            }

            const hkReflect::Type** m_addr;
        };

    protected:

        /// Map of typeId to relocations "waiting" to be assigned to this type.
        hkHashMap< TypeId, RelocInfo > m_typeRelocs;
    };

    void TagfileTypeRelocs::applyRelocation( _In_ const hkReflect::Type* ownerType, _Inout_ const hkReflect::Type** addr, TypeId id, hkArrayView<const hkReflect::Type*> types )
    {
        HK_ASSERT( 0x7b7f204a, *addr == nullptr, "Relocation target was expected to be a NULL pointer." );
        HK_ASSERT_NO_MSG( 0x7b7f204b, id > 0 );
        DLOG( "Relocate {} -> TypeId={}", addr, id );

        if ( ( id < types.getSize() ) && types[id] )
        {
            // Type was loaded since relocation was registered; apply the relocation
            *addr = types[id];
        }
        else
        {
            // Type ID not encountered yet, leave a placeholder (fallback type)
            auto const it = m_typeRelocs.find( id );
            HK_ASSERT( 0x607fc9c6, m_typeRelocs.isValid( it ), "Invalid type ID out of fallback types array bounds." );
            auto& relocInfo = m_typeRelocs.getValue( it );
            *addr = relocInfo.m_placeholderType;

            // Add a relocation entry for later resolve
            auto& relocs = relocInfo.m_relocs;
            HK_ASSERT_SLOW( 0, hkAlgorithm::findIf( relocs.begin(), relocs.end(), FindRelocByAddress( addr ) ) == relocs.end(), "Found duplicate relocation." );
            auto& r = relocs.emplaceBack();
            r.m_target = addr;
            r.m_ownerType = ownerType;
        }
    }

    void TagfileTypeRelocs::resolve( TypeId id, _In_ const hkReflect::Type* type )
    {
        DLOG_SCOPE( "TypeId {} resolves to {} '{}'.", id, type, type->getName() );
        auto const it = m_typeRelocs.find( id );
        if ( m_typeRelocs.isValid( it ) )
        {
            // New loaded type is a target for pending relocs; resolve them now.
            auto& relocInfo = m_typeRelocs.getValue( it );
            hkArray<Reloc>& pendingRelocs = relocInfo.m_relocs;
            for ( auto& reloc : pendingRelocs )
            {
                auto& target = *( reloc.m_target );
                HK_ASSERT( 0x607fc9c5, target == relocInfo.m_placeholderType, "Expected to find placeholder type before applying type relocation." );
                // At this point we would want to compare the full name, but the type has not been checked yet so template parameters can contain all sorts of errors => check the short name only.
                HK_ASSERT( 0x607fc9c4, (!type->getName() && !target->getName()) || (type->getName() && target->getName() && hkString::strCmp( type->getName(), target->getName() ) == 0), "Relocation target should be a type with the same name as the source, or both should be unnamed." );
                target = type;
                DLOG( "Resolving {} from type {} '{}'.", reloc.m_target, reloc.m_ownerType, reloc.m_ownerType->getName() );
            }

            // Delete the reloc entries associated to this type ID.
            m_typeRelocs.remove( it );
        }
    }
    
    void TagfileTypeRelocs::flatten( _In_ hkReflect::Type* type, const TypeRelocs& relocs, hkArrayView<const hkReflect::Type*> types )
    {
        DLOG_AUTO( "Flattening type relocations in {} '{}'.", type, type->getName() );

        if ( relocs.m_parent > 0 )
        {
            applyRelocation( type, ( const hkReflect::Type** )( hkReflect::TypeDetail::addressParent( type ) ), relocs.m_parent, types );
        }

        if ( relocs.m_subType > 0 )
        {
            applyRelocation( type, reinterpret_cast<const hkReflect::Type**>( hkReflect::TypeDetail::localAddressOptional<hkReflect::Opt::SUBTYPE>( type ) ), relocs.m_subType, types );
        }

        if (const hkReflect::Detail::DeclsArray* fa = hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::DECLS>( type ) )
        {
            HK_ASSERT_NO_MSG(0x27d0ed1a, fa->getNumDataFields() == relocs.m_fields.getSize());
            for ( int i = 0; i < fa->getNumDataFields(); ++i )
            {
                const hkReflect::Type* fieldType = fa->getField( i ).getType();
                applyRelocation( type, ( const hkReflect::Type** )hkReflect::TypeDetail::addressParent( const_cast<hkReflect::Type*>( fieldType ) ), relocs.m_fields[i], types );
            }
#ifdef HK_DEBUG_SLOW
            for (int i = fa->getNumDataFields(); i < fa->getNumFields(); ++i)
            {
                HK_ASSERT(0x6459e7f5, fa->getField(i).getType()->getParent() == hkReflect::getType<hkReflect::Detail::Opaque>(),
                    "All property fields should have been initialized to Opaque");
            }
#endif
        }

        if ( relocs.m_template.getSize() )
        {
            auto* templateParams = const_cast<hkReflect::Template::Parameter*>( type->getTemplate()->getParams() );
            for ( int i = 0; i < relocs.m_template.getSize(); )
            {
                HK_ASSERT_NO_MSG( 0x532dad0d, ( templateParams - type->getTemplate()->getParams() ) < type->getTemplate()->getNumParams() );
                if ( templateParams->isType() )
                {
                    applyRelocation( type, reinterpret_cast<const hkReflect::Type**>( &templateParams->m_storage ), relocs.m_template[i], types );
                    ++i;
                }
                templateParams++;
            }
        }

        if ( relocs.m_interfaces.getSize() )
        {
            auto ifaceArray = const_cast<hkReflect::Detail::InterfaceArray*>( hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::INTERFACES>( type ) )->items();
            HK_ASSERT_NO_MSG( 0x532dad0e, relocs.m_interfaces.getSize() == ifaceArray.getSize() );
            for ( int i = 0; i < relocs.m_interfaces.getSize(); ++i )
            {
                applyRelocation( type, &ifaceArray[i].m_interfaceType, relocs.m_interfaces[i], types );
            }
        }
    }

    inline bool isConvertibleToVar(const TagfileItem& item)
    {
        switch (item.getKind())
        {
        case TagfileItem::KIND_VAR0:
        case TagfileItem::KIND_TYPE:
        case TagfileItem::KIND_DECL:
            return true;
        default:
            return false;
        }
    }

    struct TagfileBundle : public hkSerialize::Bundle
    {
        struct NamedType : public hkReflect::Type
        {
            HK_DECLARE_CLASS(NamedType, New);
            static NamedType* s_named0; // storage for NULL for TypeId #0
            static NamedType s_noIdentity; // for SUBTYPE without loaded type identity

            NamedType( _In_opt_z_ const char* name = nullptr, TypeId id = 0 )
                : m_format( hkReflect::Format::OfOpaque::Value )
                , m_name( name )
                , m_template( nullptr )
                , m_sizeAlign( 0 )
                , m_id( id )
            {
                m_optional = ( name ? hkReflect::Opt::NAME : 0 ) | hkReflect::Opt::FORMAT | hkReflect::Opt::SIZE_ALIGN;
                m_parent = nullptr;
            }

            void setName(_In_opt_z_ const char* s)
            {
                m_optional |= hkReflect::Opt::NAME;
                m_name = s;
            }

            void setTemplate(_In_ hkReflect::Template* t)
            {
                m_optional |= hkReflect::Opt::TEMPLATE;
                m_template = t;
            }

            // Opt storage - Order matters
            hkUlong m_format;
            const char* m_name;
            void* m_template;
            hkUlong m_sizeAlign;

            TypeId m_id;
        };

        TagfileBundle(_In_ const struct TagfilePointerImpl* pointerImpl)
            : m_typeAllocator(hkMemHeapAllocator())
            , m_sequenceNumber(hkUint32(-1))
            , m_pointerImpl(pointerImpl)
        {
            // The first slot is reserved to be null. The following is equivalent to m_typeIdentity.setSize(1, nullptr) except it makes no allocations.
            m_typeIdentity.setDataUserFree(&NamedType::s_named0, 1, 1);
        }

        hkReflect::Var getContents() const HK_OVERRIDE
        {
            return m_items.getSize() > 1
                ? getVar(1)
                : hkReflect::Var();
        }

        virtual int getItems(hkArray<Item>& out) const HK_OVERRIDE
        {
            out.setSize(m_items.getSize());

            for(int i = 1; i < m_items.getSize(); ++i)
            {
                const Detail::TagfileItem& item = m_items[i];

                auto addrAndType = getAddrAndType(item);
                out[i].m_addr = addrAndType.m_addr;
                out[i].m_type = addrAndType.m_type;

                if(isConvertibleToVar(item))
                {
                    out[i].m_kind = Item::VAR0;
                    out[i].m_extra = VarN::SINGLE;
                }
                else if (item.isKindNote())
                {
                    out[i].m_kind = Item::NOTE;
                    out[i].m_extra = item.count;
                }
                else if (item.isKindVarN())
                {
                    out[i].m_kind = Item::VARN;
                    out[i].m_extra = item.count;
                }
                // else leave a "hole" in out
            }

            return out.getSize();
        }

        virtual hkReflect::Var getNoteOnPointer(const hkReflect::PointerVar& ptr) const HK_OVERRIDE
        {
            HK_ASSERT_NO_MSG(0x318b20a2, ptr.getImpl() == reinterpret_cast<const hkReflect::Detail::PointerImpl*>(m_pointerImpl));
            hkUint32 id = *(const hkUint32Le*)ptr.getAddress();
            if (id < (hkUint32)m_items.getSize())
            {
                if (m_items[id].isKindNote())
                {
                    return getVar(m_items[id]);
                }
                else if (m_items[id].isEmpty())
                {
                    // If the item is imported m_items[id].count will be the index of the import note.
                    hkUint32 nid = m_items[id].count;
                    return getVar(m_items[nid]);
                }
            }
            return hkReflect::Var();
        }

        void setType(int tid, _Inout_ hkReflect::Type* ty, const Detail::TypeRelocs& relocs)
        {
            m_types[tid] = ty;
            m_typeRelocs.resolve(tid, ty);
            m_typeRelocs.flatten(ty, relocs, m_types);
        }

        inline hkReflect::Var getVar(int vid) const
        {
            return getVar(m_items[vid]);
        }

        inline hkReflect::Var getVar(const Detail::TagfileItem& item) const
        {
            HK_ASSERT(0x65465632, !item.isKindVarN(), "Item is not a single object");
            hkReflect::Detail::AddrAndType addrAndType = getAddrAndType(item);
            return hkReflect::Var(addrAndType.m_addr, addrAndType.m_type);
        }

        inline hkReflect::Detail::AddrAndType getAddrAndType(const Detail::TagfileItem& item) const
        {
            if (item.isEmpty())
            {
                return hkReflect::Detail::AddrAndType(HK_NULL, HK_NULL);
            }
            else if (item.isKindVarN() && item.count == 0)
            {
                // empty variant array
                return hkReflect::Detail::AddrAndType(HK_NULL, m_types[item.getIndex()]);
            }
            else if (item.isKindType())
            {
                const hkReflect::Type* type = m_types[item.getIndex()];
                return hkReflect::exactObj(type);
            }
            else if (item.isKindDecl())
            {
                const hkReflect::Type* context = m_types[item.getIndex()];
                auto declsArray = hkReflect::TypeDetail::getDeclsArray(context);
                HK_ASSERT(0x46570e0, declsArray, "Type must have decls");
                HK_ASSERT(0x7f1551bc, item.count < (unsigned)declsArray->getNumDecls(),
                    "Found invalid Decl index in loaded file. The file might be corrupt and must be re-saved" );
                hkReflect::Decl decl = declsArray->getDecl(item.count);
                HK_ASSERT(0x1127ae44, decl.getName(), "Found reference to missing decl ({}, {})", context, item.count );
                return hkReflect::exactObj(decl.getType());
            }
            else
            {
                return hkReflect::Detail::AddrAndType(hkAddByteOffset(m_data.begin(), item.offset), m_types[item.getIndex()]);
            }
        }

        void clear()
        {
            m_dataBuf.clear();
            m_indexBuf.clear();
        }

        hkArray<const hkReflect::Type*>& accessTypes() { return m_types; }

        hkArray<char> m_dataBuf;
        hkArray<char> m_indexBuf;

        hkArrayView<void> m_data;
        hkArrayView<Detail::TagfileItem> m_items;

        hkArray< NamedType*> m_typeIdentity;
        hkArray<const char *> m_typeStrings;
        hkArray<const char *> m_fieldStrings;
        hkArray<const char*> m_attributeStrings;
        Detail::TagfileTypeRelocs m_typeRelocs;
        hkTransientAllocator m_typeAllocator;
        hkUint32 m_sequenceNumber;
        const TagfilePointerImpl* m_pointerImpl;
    };

    TagfileBundle::NamedType* TagfileBundle::NamedType::s_named0;
    TagfileBundle::NamedType TagfileBundle::NamedType::s_noIdentity( "[no-type-identity]", -0x42 );

    struct TagfileArrayImpl : public hkReflect::Detail::ArrayImpl
    {
        HK_DECLARE_CLASS(TagfileArrayImpl, New);

        const TagfileBundle* m_bundle;
        TagfileArrayImpl(const TagfileBundle* b) : m_bundle(b) {}

        virtual hkResult getValue(_In_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const HK_OVERRIDE
        {
            
            if(int count = arrType->getFixedCount())
            {
                *val = hkReflect::ArrayValue(arrAddr, count, arrType->getSubType());
                return HK_SUCCESS;
            }
            hkUint32 id = *(const hkUint32Le*)arrAddr;
            if (id >= (hkUint32)m_bundle->m_items.getSize())
            {
                *val = hkReflect::ArrayValue();
                return HK_FAILURE;
            }
            const TagfileItem& item = m_bundle->m_items[id];
            if(item.isKindVarN())
            {
                if(item.count > 0)
                {
                    *val = hkReflect::ArrayValue(m_bundle->m_data.ptr(item.offset), item.count, m_bundle->type(item.getIndex()));
                    return HK_SUCCESS;
                }
                else
                {
                    *val = hkReflect::ArrayValue(HK_NULL, item.count, m_bundle->type(item.getIndex()));
                    return HK_SUCCESS;
                }

            }
            *val = hkReflect::ArrayValue();
            return HK_SUCCESS;
        }
        virtual hkResult setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int nelem) const HK_OVERRIDE
        {
            HK_ASSERT_NO_MSG(0x73e35517, 0);
            return HK_FAILURE;
        }
    };

    struct TagfilePointerImpl : public hkReflect::Detail::PointerImpl
    {
        HK_DECLARE_CLASS(TagfilePointerImpl, New);

        const TagfileBundle* m_bundle;
        TagfilePointerImpl(const TagfileBundle* b) : m_bundle(b) {}

        virtual hkResult setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* type, const hkReflect::Var& var) const HK_OVERRIDE
        {
            HK_ASSERT_NO_MSG(0x73e35517, 0);
            return HK_FAILURE;
        }
        virtual hkResult getValue(_In_ const void* self, _In_ const hkReflect::PointerType*, _Out_ hkReflect::Var* val) const HK_OVERRIDE
        {
            *val = hkReflect::Var();

            hkUint32 id = *(const hkUint32Le*)self;
            if (id >= (hkUint32)m_bundle->m_items.getSize()) { return HK_FAILURE; }

            const TagfileItem& item = m_bundle->m_items[id];
            if (item.isKindNote())
            {
                // Return annotated object
                *val = m_bundle->getVar(item.count);
                return HK_SUCCESS;
            }
            if (!item.isKindVarN())
            {
                *val = m_bundle->getVar(item);
                return HK_SUCCESS;
            }

            return HK_FAILURE;
        }
        virtual void* queryInterfaceImpl(const hkReflect::Type* type, const hkReflect::Var& self) const HK_OVERRIDE
        {
            hkUint32 id = *(const hkUint32Le*)self.getAddress();
            if(id && type->extendsOrEquals<Detail::IdFromPointer>())
            {
                return reinterpret_cast<Detail::IdFromPointer*>(self.getAddress());
            }
            return HK_NULL;
        }
    };

    struct TagfileStringImpl : public hkReflect::Detail::StringImpl
    {
        HK_DECLARE_CLASS(TagfileStringImpl, New);

        const TagfileBundle* m_bundle;
        TagfileStringImpl(const TagfileBundle* b) : m_bundle(b) {}

        virtual hkResult setValue(_Inout_ void* string, _In_ const hkReflect::StringType* type, hkReflect::StringValue newValue) const HK_OVERRIDE
        {
            HK_ASSERT_NO_MSG(0x1b5781e5, 0);
            return HK_FAILURE;
        }
        virtual hkResult getValue(_In_ const void* self, _In_ const hkReflect::StringType* type, _Out_ hkReflect::StringValue* val) const HK_OVERRIDE
        {
            *val = hkReflect::StringValue();

            hkUint32 id = *(const hkUint32Le*)self;
            if (id >= (hkUint32)m_bundle->m_items.getSize()) { return HK_FAILURE; }

            const TagfileItem& item = m_bundle->m_items[id];
            if(item.isKindVarN())
            {
                if ( indexValid(item.offset, m_bundle->m_data)==false )
                {
                    return HK_FAILURE;
                }
                *val = reinterpret_cast<const char*>( m_bundle->m_data.ptr(item.offset) );
                return HK_SUCCESS;
            }

            return item.isEmpty() ? HK_SUCCESS : HK_FAILURE;
        }
    };
} }

namespace
{
    /// Callback to check during type consistency checks that a type is reachable, i.e. it is indeed present in the bundle.
    bool HK_CALL s_isTypeInBundle(_In_ const hkReflect::Type* type, _In_opt_ void* userData )
    {
        auto* const bundle = static_cast<hkSerialize::Detail::TagfileBundle*>( userData );
        if ( !bundle || !type )
        {
            return false;
        }

        // Opaque is used as a placeholder for non-serializable Decls.
        if (type == hkReflect::getType<hkReflect::Detail::Opaque>())
        {
            return true;
        }

        // Check that the type is present in the bundle. Types defined in the type section must only
        // reference other types (e.g. through parent or sub-type) that are too in the bundle.
        
        if ( bundle->types().indexOf( type ) >= 0 )
        {
            return true;
        }

        // Type not relocated yet. Look in the type identities.
        const bool hasIdentity = ( bundle->m_typeIdentity.indexOf( const_cast<hkSerialize::Detail::TagfileBundle::NamedType*>( static_cast<const hkSerialize::Detail::TagfileBundle::NamedType*>( type ) ) ) >= 0 );
        if ( !hasIdentity )
        {
            return false;
        }
#ifdef HK_DEBUG_SLOW
        // Check there is indeed no body.
        for ( auto* const loadedType : bundle->types() )
        {
            HK_ASSERT( 0x4ea0977d, !loadedType || ( loadedType->getFullName() != type->getFullName() ), "Found type pointer not relocated to type body." );
        }
#endif
        return true;
    }

    /// Check if a type ID is out of bounds, or is zero (NULL type).
    bool typeIdIsNullOrOutOfBounds( hkSerialize::TypeId typeId, const hkSerialize::Detail::TagfileBundle& bundle )
    {
        return ( typeId <= 0 ) || ( typeId >= bundle.m_typeIdentity.getSize() );
    }

    /// Check if a type ID is out of bounds.
    bool typeIdIsOutOfBounds( hkSerialize::TypeId typeId, const hkSerialize::Detail::TagfileBundle& bundle )
    {
        return ( typeId < 0 ) || ( typeId >= bundle.m_typeIdentity.getSize() );
    }

    struct ImplBase
    {
        enum Mode { READ, VIEW, UNLOAD };
        template<Mode MODE> struct Traits;

        struct ViewConfig
        {
            ViewConfig() : m_isInplace(false), m_typeReg(nullptr) {}
            bool m_isInplace;
            const hkReflect::TypeReg* m_typeReg;
        };

        struct UnloadState
        {
            UnloadState() : m_status(HK_SUCCESS) {}
            hkResult& status() { return m_status; }
            hkResult m_status;
            hkArrayView<const hkReflect::Type*> m_types;
            hkIo::ReadBufferView m_data;
        };

        ImplBase() : m_status(HK_SUCCESS) {}
        hkResult m_status;
    };

    template<>
    struct ImplBase::Traits<ImplBase::READ>
    {
        typedef hkSerialize::HffStreamReader Reader;
        struct Data
        {
            Data(ImplBase* i) : m_impl(i) {}
            ImplBase* const m_impl;
            hkResult& status() const { return m_impl->m_status; }
        };
    };

    template<>
    struct ImplBase::Traits<ImplBase::VIEW>
    {
        typedef hkSerialize::HffMemoryReader Reader;
        struct Data
        {
            Data(ImplBase* i, ViewConfig c) : m_impl(i), m_config(c) {}
            ImplBase* const m_impl;
            hkResult& status() const { return m_impl->m_status; }
            const ViewConfig m_config;
        };
    };
    template<>
    struct ImplBase::Traits<ImplBase::UNLOAD>
    {
        typedef hkSerialize::HffMemoryReader Reader;
        typedef ImplBase::UnloadState& Data;
    };
}

struct hkSerialize::TagfileReadFormat::Impl : public ImplBase
{
    HK_DECLARE_CLASS(Impl, New);

    typedef Detail::TagfileRoot TagfileRoot;
    typedef Detail::TagfileSection TagfileSection;
    typedef Detail::TagfileTypeSection TagfileTypeSection;
    typedef Detail::TagfileIndexSection TagfileIndexSection;

    Impl();

    hkViewPtr<Bundle> read(hkIo::ReadBuffer& rb);
    hkViewPtr<Bundle> view(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_ hkUlong* usedOut, ViewConfig config);

    // Some handlers can optimize if they know whether data comes from a stream or from a memory buffer, so we keep two
    // handlers for each section even if they share most of the code.
    typedef void ( Impl::*StreamHandler )( HffStreamReader& reader );
    typedef void ( Impl::*ViewHandler )(hkIo::ReadBufferView view, ViewConfig config );
    typedef void( *UnloadHandler )(hkIo::ReadBufferView, UnloadState& state );
    // These traits abstract over the various loading/unloading mode so that a single parse function can be used.
    


    // Map a section to a pair of handler methods.
    struct HandlerEntry
    {
        Ident m_ident;
        StreamHandler m_streamHandler;
        ViewHandler m_viewHandler;
        UnloadHandler m_unloadHandler;

        template<Mode MODE> bool isNop() const;

        HK_INLINE void handle( HffStreamReader& reader, Traits<READ>::Data data ) const
        {
            ( static_cast<Impl*>(data.m_impl)->*m_streamHandler )( reader );
        }

        HK_INLINE void handle( HffMemoryReader& reader, Traits<VIEW>::Data data ) const
        {
            ( static_cast<Impl*>(data.m_impl)->*m_viewHandler )( reader.view(), data.m_config );
        }

        HK_INLINE void handle( HffMemoryReader& reader, Traits<UNLOAD>::Data state ) const
        {
            ( *m_unloadHandler )( reader.view(), state );
        }
    };
    typedef hkArrayView< const HandlerEntry > HandlerTable;

    static const HandlerEntry s_root[];
    static const HandlerEntry s_types[];
    static const HandlerEntry s_index[];

    template<Mode MODE>
    static void parse( typename Traits<MODE>::Reader& reader, typename Traits<MODE>::Data data, HandlerTable table );

    void readType(hkIo::ReadBufferView&);

    void readStrings(hkIo::ReadBufferView rb, hkArray<const char*>& strings, bool allowEmpty = false);

    void handleTypeSequenceNumber(hkIo::ReadBufferView rb, ViewConfig);
    void handleTypeRollingHash(hkIo::ReadBufferView rb, ViewConfig);
    void handleCompendiumId(hkIo::ReadBufferView rb, ViewConfig);
    void handleCompendiumReference(hkIo::ReadBufferView rb, ViewConfig config);

    // IDENTITY_V0 (2015.1) has a bug where all type identities are sent again and again in streaming mode.
    // But otherwise it's essentially the same as IDENTITY. Same for BODY_V0 and BODY.
    template <int VERSION>
    void handleTypeIdentity(hkIo::ReadBufferView rb, ViewConfig);
    void handleTypeIdentityImpl(hkIo::ReadBufferView rb, ViewConfig, int version);
    template <int VERSION>
    void handleTypeBody(hkIo::ReadBufferView rb, ViewConfig);
    void handleTypeBodyImpl(hkIo::ReadBufferView rb, ViewConfig, int version);

    void handlePropertyDecls(hkIo::ReadBufferView rb, ViewConfig);

    template<typename HASHER>
    void inplaceTypeHashesImpl(hkIo::ReadBufferView reader, HASHER& hasher);
    void inplaceTypeHashes(hkIo::ReadBufferView thsh, ViewConfig config);
    template<typename HASHER>
    void inplacePropertyHashesImpl(hkIo::ReadBufferView rb, HASHER& hasher);
    void inplacePropertyHashes(hkIo::ReadBufferView rb, ViewConfig config);

    void handleDataStream( HffStreamReader& reader );
    void handleData( hkIo::ReadBufferView view, ViewConfig);
    void handleItemsStream( HffStreamReader& reader );
    void handleItems( hkIo::ReadBufferView view, ViewConfig config );
    void handlePatchData( hkIo::ReadBufferView view, ViewConfig config );
    void handleNativePointers( hkIo::ReadBufferView view, ViewConfig config);

    static void unloadData( hkIo::ReadBufferView view, UnloadState& state );
    static void unloadCompendiumReference( hkIo::ReadBufferView view, UnloadState& state );
    static void unloadNativePointers( hkIo::ReadBufferView view, UnloadState& state );
    static void unloadItems( hkIo::ReadBufferView view, UnloadState& state );


    template<typename READER> Ident enterRootScope(READER& reader);
    template<typename READER> void exitRootScope(READER& reader, Ident root);

    hkResult resetBundle();

    Detail::TagfileBundle m_bundle;
    hkUint32 m_typeRollingHash;

    struct Compendium
    {
        enum State{
            NOT_USING = 0,
            USING,
            READING,
        };
        Compendium() : m_state(NOT_USING) {}
        hkArray<hkUint64> m_sigs;
        State m_state;
    };
    Compendium m_compendium;

    Detail::TagfilePointerImpl m_pointerImpl;
    Detail::TagfileArrayImpl m_arrayImpl;
    Detail::TagfileStringImpl m_stringImpl;
    hkLong m_hackTypeBodyOffset;
    hkUint32 m_hackTypeBodyHash; // HK_DEBUG_SLOW
    hkArray<bool> m_propHashVerified;

    hkReflect::Detail::CheckUtil::IncrementalChecker m_checker;

    NewTypeCallback m_newTypeCallback;

    template<ViewHandler viewHandler>
    void _runOnTempView(HffStreamReader& reader, bool padded)
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;
        TempReadBuffer rb(reader, m_status, padded);
        if (m_status.isSuccess()) (this->*viewHandler)(rb, ViewConfig());
    }
    template<ViewHandler viewHandler>
    void runOnTempView(HffStreamReader& reader)
    {
        return _runOnTempView<viewHandler>(reader, true);
    }
    template<ViewHandler viewHandler>
    void runOnTempViewNoPad(HffStreamReader& reader)
    {
        return _runOnTempView<viewHandler>(reader, false);
    }

    template< hkArray<const char*> hkSerialize::Detail::TagfileBundle::*stringArray>
    void _readStringsHff( HffStreamReader& reader, bool allowEmpty )
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        hkIo::ReadBufferView rb;
        FAIL_IF(readInto(rb, reader, m_bundle.m_typeAllocator).isFailure(), "Failed to read string section");
        readStrings(rb, this->m_bundle.*stringArray, allowEmpty);
        FAIL_IF_INVALID(rb);
    }
    template< hkArray<const char*> hkSerialize::Detail::TagfileBundle::*stringArray>
    void readStringsHffNonEmpty(HffStreamReader& reader)
    {
        return _readStringsHff<stringArray>(reader, false);
    }
    template< hkArray<const char*> hkSerialize::Detail::TagfileBundle::*stringArray>
    void readStringsHffAllowEmpty( HffStreamReader& reader )
    {
        return _readStringsHff<stringArray>(reader, true);
    }

    template< hkArray<const char*> hkSerialize::Detail::TagfileBundle::*stringArray>
    void _readStringsRbv( hkIo::ReadBufferView view, ViewConfig config, bool allowEmpty )
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        // We have to copy type and field strings in case they are used in a subsequent bundle.
        // It's a pity to do this for the single bundle case, but we can't tell that in advance.
        if ( config.m_isInplace )
        {
            readStrings( view, this->m_bundle.*stringArray, allowEmpty);
        }
        else
        {
            hkIo::ReadBufferView rb;
            FAIL_IF(readInto(rb, view, m_bundle.m_typeAllocator).isFailure(), "Failed to read string section");
            readStrings(rb, this->m_bundle.*stringArray, allowEmpty);
            FAIL_IF_INVALID(rb);
        }
    }
    template< hkArray<const char*> hkSerialize::Detail::TagfileBundle::*stringArray>
    void readStringsRbvNonEmpty(hkIo::ReadBufferView view, ViewConfig config)
    {
        return _readStringsRbv<stringArray>(view, config, false);
    }
    template< hkArray<const char*> hkSerialize::Detail::TagfileBundle::*stringArray>
    void readStringsRbvAllowEmpty(hkIo::ReadBufferView view, ViewConfig config)
    {
        return _readStringsRbv<stringArray>(view, config, true);
    }

    template<int N, const HandlerEntry* TABLE>
    void parseSubsection( HffStreamReader& reader )
    {
        parse<READ>( reader, Traits<READ>::Data(this), hkArrayViewT::make( TABLE, N ) );
    }

    template<int N, const HandlerEntry* TABLE>
    void parseSubsection( hkIo::ReadBufferView view, ViewConfig config )
    {
        HffMemoryReader reader( view.begin(), view.remaining() );
        parse<VIEW>(reader, Traits<VIEW>::Data(this, config), hkArrayViewT::make(TABLE, N));
        view.skip(view.remaining());
    }

    template<int N, const HandlerEntry* TABLE>
    static void parseSubsection( hkIo::ReadBufferView view, UnloadState& state )
    {
        HffMemoryReader reader( view.begin(), view.remaining() );
        parse<UNLOAD>( reader, state, hkArrayViewT::make( TABLE, N ) );
        view.skip(view.remaining());
    }
};

namespace hkSerialize
{
    template<> HK_ALWAYS_INLINE bool TagfileReadFormat::Impl::HandlerEntry::isNop<ImplBase::READ>() const { return m_streamHandler == HK_NULL; }
    template<> HK_ALWAYS_INLINE bool TagfileReadFormat::Impl::HandlerEntry::isNop<ImplBase::VIEW>() const { return m_viewHandler == HK_NULL; }
    template<> HK_ALWAYS_INLINE bool TagfileReadFormat::Impl::HandlerEntry::isNop<ImplBase::UNLOAD>() const { return m_unloadHandler == HK_NULL; }

    const TagfileReadFormat::Impl::HandlerEntry hkSerialize::TagfileReadFormat::Impl::s_types[] =
    {
        { TagfileTypeSection::NAME_STRINGS, &hkSerialize::TagfileReadFormat::Impl::readStringsHffNonEmpty<&hkSerialize::Detail::TagfileBundle::m_typeStrings>, &hkSerialize::TagfileReadFormat::Impl::readStringsRbvNonEmpty<&hkSerialize::Detail::TagfileBundle::m_typeStrings>, HK_NULL },
        { TagfileTypeSection::FIELD_NAME_STRINGS, &hkSerialize::TagfileReadFormat::Impl::readStringsHffNonEmpty<&hkSerialize::Detail::TagfileBundle::m_fieldStrings>, &hkSerialize::TagfileReadFormat::Impl::readStringsRbvNonEmpty<&hkSerialize::Detail::TagfileBundle::m_fieldStrings>, HK_NULL },
        { TagfileTypeSection::ATTRIBUTE_STRINGS, &hkSerialize::TagfileReadFormat::Impl::readStringsHffAllowEmpty<&hkSerialize::Detail::TagfileBundle::m_attributeStrings>, &hkSerialize::TagfileReadFormat::Impl::readStringsRbvAllowEmpty<&hkSerialize::Detail::TagfileBundle::m_attributeStrings>, HK_NULL },
        { TagfileTypeSection::IDENTITY, &hkSerialize::TagfileReadFormat::Impl::runOnTempView<&hkSerialize::TagfileReadFormat::Impl::handleTypeIdentity<1>>, &hkSerialize::TagfileReadFormat::Impl::handleTypeIdentity<1>, HK_NULL },
        { TagfileTypeSection::BODY, &hkSerialize::TagfileReadFormat::Impl::runOnTempView<&hkSerialize::TagfileReadFormat::Impl::handleTypeBody<1>>, &hkSerialize::TagfileReadFormat::Impl::handleTypeBody<1>, HK_NULL },
        { TagfileTypeSection::ROLLING_HASH, &hkSerialize::TagfileReadFormat::Impl::runOnTempView<&hkSerialize::TagfileReadFormat::Impl::handleTypeRollingHash>, &hkSerialize::TagfileReadFormat::Impl::handleTypeRollingHash, HK_NULL },
        { TagfileTypeSection::NATIVE_POINTERS, HK_NULL, &hkSerialize::TagfileReadFormat::Impl::handleNativePointers, &hkSerialize::TagfileReadFormat::Impl::unloadNativePointers },
        { TagfileTypeSection::HASHES, HK_NULL, &hkSerialize::TagfileReadFormat::Impl::inplaceTypeHashes, HK_NULL },
        { TagfileTypeSection::PROPERTIES, &hkSerialize::TagfileReadFormat::Impl::runOnTempView<&hkSerialize::TagfileReadFormat::Impl::handlePropertyDecls>, &hkSerialize::TagfileReadFormat::Impl::handlePropertyDecls, HK_NULL },
        { TagfileTypeSection::PROPS_HASHES, HK_NULL, &hkSerialize::TagfileReadFormat::Impl::inplacePropertyHashes, HK_NULL },
        { TagfileTypeSection::PADDING, HK_NULL, HK_NULL, HK_NULL },
        { TagfileTypeSection::SEQUENCE_NUMBER_V0, &hkSerialize::TagfileReadFormat::Impl::runOnTempView<&hkSerialize::TagfileReadFormat::Impl::handleTypeSequenceNumber>, &hkSerialize::TagfileReadFormat::Impl::handleTypeSequenceNumber, HK_NULL },
        { TagfileTypeSection::IDENTITY_V0, &hkSerialize::TagfileReadFormat::Impl::runOnTempView<&hkSerialize::TagfileReadFormat::Impl::handleTypeIdentity<0>>, &hkSerialize::TagfileReadFormat::Impl::handleTypeIdentity<0>, HK_NULL },
        { TagfileTypeSection::BODY_V0, &hkSerialize::TagfileReadFormat::Impl::runOnTempView<&hkSerialize::TagfileReadFormat::Impl::handleTypeBody<0>>, &hkSerialize::TagfileReadFormat::Impl::handleTypeBody<0>, HK_NULL },
    };

    const TagfileReadFormat::Impl::HandlerEntry hkSerialize::TagfileReadFormat::Impl::s_index[] =
    {
        { TagfileIndexSection::ITEMS, &hkSerialize::TagfileReadFormat::Impl::handleItemsStream, &hkSerialize::TagfileReadFormat::Impl::handleItems, &hkSerialize::TagfileReadFormat::Impl::unloadItems },
        { TagfileIndexSection::PATCH_DATA, HK_NULL, &hkSerialize::TagfileReadFormat::Impl::handlePatchData, HK_NULL },
    };

    const TagfileReadFormat::Impl::HandlerEntry hkSerialize::TagfileReadFormat::Impl::s_root[] =
    {
        { TagfileSection::SDK_VERSION, HK_NULL, HK_NULL, HK_NULL },
        { TagfileSection::DATA, &hkSerialize::TagfileReadFormat::Impl::handleDataStream, &hkSerialize::TagfileReadFormat::Impl::handleData, &hkSerialize::TagfileReadFormat::Impl::unloadData },
        { TagfileSection::TYPES, &hkSerialize::TagfileReadFormat::Impl::parseSubsection<HK_COUNT_OF( s_types ), s_types>, &hkSerialize::TagfileReadFormat::Impl::parseSubsection<HK_COUNT_OF( s_types ), s_types>, &hkSerialize::TagfileReadFormat::Impl::parseSubsection<HK_COUNT_OF( s_types ), s_types> },
        { TagfileSection::INDEX, &hkSerialize::TagfileReadFormat::Impl::parseSubsection<HK_COUNT_OF( s_index ), s_index>, &hkSerialize::TagfileReadFormat::Impl::parseSubsection<HK_COUNT_OF( s_index ), s_index>, &hkSerialize::TagfileReadFormat::Impl::parseSubsection<HK_COUNT_OF( s_index ), s_index> },
        { TagfileSection::COMPENDIUM_REFERENCE, &hkSerialize::TagfileReadFormat::Impl::runOnTempViewNoPad<&Impl::handleCompendiumReference>, &hkSerialize::TagfileReadFormat::Impl::handleCompendiumReference, &hkSerialize::TagfileReadFormat::Impl::unloadCompendiumReference },
        { TagfileSection::COMPENDIUM_ID, &hkSerialize::TagfileReadFormat::Impl::runOnTempViewNoPad<&Impl::handleCompendiumId>, &hkSerialize::TagfileReadFormat::Impl::handleCompendiumId, HK_NULL },
    };

    template<TagfileReadFormat::Impl::Mode MODE>
    void TagfileReadFormat::Impl::parse( typename Traits<MODE>::Reader& reader, typename Traits<MODE>::Data data, HandlerTable table )
    {
        HK_UNITY_USING_ANONYMOUS_NAMESPACE;

        // In loading the status sticks to the Impl; in unloading it's local to this call only.
        hkResult& status = data.status();

        hkInplaceArray<bool, 16>::Temp visited;
        visited.setSize( table.getSize(), false );

        while ( status.isSuccess() )
        {
            Ident ident = reader.enter();

            if ( ident == HffBase::IDENT_IO_ERROR )
            {
                Log_Warning( "IO error" );
                status = HK_FAILURE;
            }
            else if ( ident == HffBase::IDENT_NONE )
            {
                // Success.
                return;
            }
            else if ( const HandlerEntry* entry = findEntry( ident, table ) )
            {
                
                // Check that the section is not duplicate.
                int entryIdx = hkLosslessCast<int>( entry - table.begin() );
                if ( !visited[entryIdx] )
                {
                    visited[entryIdx] = true;

                    if ( entry->isNop<MODE>() )
                    {
                        status = reader.skipContent();
                    }
                    else
                    {
                        
                        entry->handle( reader, data );
                    }

                    if ( status.isSuccess() )
                    {
                        reader.leave();
                    }
                }
                else
                {
                    Log_Warning( "Duplicate section" );
                    status = HK_FAILURE;
                }
            }
            else
            {
                char identString[] = { char( ident >> 24 ), char( ident >> 16 ), char( ident >> 8 ), char( ident >> 0 ), 0 };
                Log_Warning( "Unexpected section {}", identString );
                status = HK_FAILURE;
            }
        }
        // status is failure
        reader.abort();
    }
}
typedef int ExtraId;

hkSerialize::TagfileReadFormat::Impl::Impl()
    : m_bundle(&m_pointerImpl)
    , m_typeRollingHash(0)
    , m_pointerImpl(&m_bundle)
    , m_arrayImpl(&m_bundle)
    , m_stringImpl(&m_bundle)
    , m_hackTypeBodyOffset(0)
    , m_hackTypeBodyHash(0)
    , m_newTypeCallback(HK_NULL)
{
    hkReflect::Detail::CheckUtil::Config config;
    config.m_earlyOutOnFirstFailure = true;
    config.m_allowedFlags = (
        hkReflect::Type::TYPE_NOT_SERIALIZABLE // yes, ... T_T
        | hkReflect::Type::TYPE_ABSTRACT
        | hkReflect::Type::TYPE_DECORATOR_HACK
        | hkReflect::Type::TYPE_PROPERTY
        | hkReflect::Type::TYPE_INTERFACE
        | hkReflect::Type::TYPE_DYNAMIC
        | hkReflect::Type::TYPE_POINTER_WEAK
        | hkReflect::Type::TYPE_OWN_VTABLE
        | hkReflect::Type::TYPE_IS_ATTRIBUTE
        | hkReflect::Type::TYPE_NOT_RETARGETABLE
        //| hkReflect::Type::TYPE_SYNTHETIC
        | hkReflect::Type::TYPE_FOREIGN
        | hkReflect::Type::TYPE_TEMPORARY
        );
    config.m_allowedOptMask = (
        hkReflect::Opt::FORMAT
        | hkReflect::Opt::SUBTYPE
        | hkReflect::Opt::IMPL // set by the reader, not loaded from the bundle (but can't discriminate at the moment)
        | hkReflect::Opt::NAME
        | hkReflect::Opt::VERSION
        //| hkReflect::Opt::ATTRIBUTES
        //| hkReflect::Opt::DEF_CONSTRUCTOR
        //| hkReflect::Opt::COPY_CONSTRUCTOR
        //UNUSED = 8 - always invalid -
        //| hkReflect::Opt::DESTRUCTOR
        //| hkReflect::Opt::COPY_ASSIGNMENT
        //| hkReflect::Opt::FUNCTIONS
        //| hkReflect::Opt::EXACT_TYPE
        //| hkReflect::Opt::MESSAGING
        //| hkReflect::Opt::DEFAULT
        //| hkReflect::Opt::INHERITANCE
        //| hkReflect::Opt::POINTER_IMPL
        | hkReflect::Opt::INTERFACES
        | hkReflect::Opt::TEMPLATE
        //| hkReflect::Opt::DECL_NAME
        //| hkReflect::Opt::DECL_FORMAT
        //| hkReflect::Opt::DECL_CONTEXT
        //| hkReflect::Opt::ALLOC_IMPL
        | hkReflect::Opt::SIZE_ALIGN
        | hkReflect::Opt::FLAGS
        //| hkReflect::Opt::REFLECT_CONSTRUCTOR
        | hkReflect::Opt::DECLS
        //| hkReflect::Opt::AFTER_REFLECT_NEW
        | hkReflect::Opt::ATTRIBUTE_STRING
        //| hkReflect::Opt::TYPE_WORLD
        //| hkReflect::Opt::VALIDATE
        );
    config.m_isTypeReachableCallback = s_isTypeInBundle;
    config.m_isTypeReachableUserData = &m_bundle;
    config.m_origin = &DEBUG_LOG_OBJECT;
    config.m_level = hkLog::Level::Error;
    config.m_earlyOutOnFirstFailure = true;
    config.m_allowedPlaceholderDecls = true;
    m_checker.setConfig( config );
}

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    static const hkReflect::Detail::VoidImpl s_readType_void_impl;
    static const hkReflect::Detail::OpaqueImpl s_readType_opaque_impl;
}

void hkSerialize::TagfileReadFormat::Impl::readType( hkIo::ReadBufferView& rb )
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;

    // Read serialized type head
    const auto typeId = readUintVle<TypeId>(rb);
    const auto parentTypeId = readUintVle<TypeId>(rb);
    auto optbits = readUintVle<hkUint32>(rb);
    FAIL_IF_INVALID(rb);

    // Type ID cannot be zero, this would mark the beginning of padding and would have been caught in the caller.
    // This is consistent with the fact that 0 is the NULL type, which is never serialized and so should not appear here.
    FAIL_WITH_ID_IF( 0x71b95bd8, typeIdIsNullOrOutOfBounds( typeId, m_bundle ), "Invalid type ID (out of bounds)." );
    FAIL_WITH_ID_IF( 0x71b95bd9, typeId >= m_bundle.types().getSize(), "Invalid type ID (out of bounds).");
    FAIL_IF(m_bundle.type(typeId)!=nullptr, "TypeId {} already read", typeId);
    auto const* const typeIdentity = m_bundle.m_typeIdentity[typeId];

    // Parent type ID can be zero if NULL type.
    FAIL_IF( typeIdIsOutOfBounds( parentTypeId, m_bundle ), "Invalid parent type ID (out of bounds) for type '{}'.", typeIdentity->getName() );

    // Check optionals
    // This is a limit set of Opt bits (see Detail::s_optMapping). In particular it excludes those
    // already stored in the type identity (NAME and TEMPLATE), which have already been validated.
    {
        hkUint32 out = 0;
        for ( int i = 0; i < HK_COUNT_OF( Detail::s_optMapping ); ++i )
        {
            const hkUint32 optMask = ( 1 << i );
            if ( optbits & optMask )
            {
                out |= Detail::s_optMapping[i];
                optbits ^= optMask; // clear once processed
            }
        }

        // Check the Opt bits are only the ones expected as per Detail::s_optMapping, which are the only valid one in a Tagfile.
        FAIL_IF( optbits != 0, "Invalid Opt bits 0x{:x} on content of type '{}'.", optbits, typeIdentity->getName() );

        // Further check that Opt bits are valid according to the type checker. This checks in particular invalid bits (e.g. bit #8).
        FAIL_IF( m_checker.areOptsValid( out ) == false, "Forbidden Opt bits 0x{:x} on content of type '{}'.", out, typeIdentity->getName() );

        optbits = out;
    }

    DLOG_SCOPE( "Type ${} parent=${} tagoff={}//", typeId, parentTypeId, rb.tell() );

    // Start building the output type
    hkReflect::TypeBuilder builder;
    //builder.beginDerived( nullptr ); // relocated later
    Detail::TypeRelocs reloc;
    reloc.setParent( parentTypeId );

    // Process Opt bits and add items to TypeBuilder
    hkUint32 format = 0;
    if ( optbits & hkReflect::Opt::FORMAT )
    {
        format = readUintVle<hkUint32>(rb);
        FAIL_IF_INVALID(rb);
        builder.addItem<hkReflect::Opt::FORMAT>( format );
        DLOG( "Format ${:x}", format );

        const hkReflect::Detail::Impl* impl = HK_NULL;
        switch ( format & hkReflect::Format::KIND_MASK )
        {
        case hkReflect::KIND_VOID:
            {
                impl = &s_readType_void_impl;
                break;
            }
        case hkReflect::KIND_OPAQUE:
            {
                impl = &s_readType_opaque_impl;
                break;
            }
        case hkReflect::KIND_STRING:
            {
                impl = &m_stringImpl;
                break;
            }
        case hkReflect::KIND_BOOL:
        case hkReflect::KIND_FLOAT:
        case hkReflect::KIND_INT:
            {
                const hkReflect::Type* t = hkReflect::Detail::builtinFromFormat( hkReflect::Format::Value::create( format ) );
                FAIL_WITH_ID_IF( 0x775182a1, t == nullptr, "Cannot find a built-in type corresponding to format 0x{:x8} of type '{}' (invalid format?).", format, typeIdentity->getName() );
                impl = t->getImpl();
                HK_ASSERT_NO_MSG( 0x4ea0977c, impl );
                break;
            }
        case hkReflect::KIND_RECORD:
            {
                builder.addItem<hkReflect::Opt::DECLS>( HK_NULL );
                impl = &hkReflect::Detail::HavokRecordImpl::s_instance;
                break;
            }
        case hkReflect::KIND_POINTER:
            {
                impl = &m_pointerImpl;
                break;
            }
        case hkReflect::KIND_ARRAY:
            {
                impl = &m_arrayImpl;
                break;
            }
        default:
            {
                FAIL_ALWAYS( "Unknown format kind 0x{:x8} for type '{}'.", format & hkReflect::Format::KIND_MASK, typeIdentity->getName() );
            }
        }
        builder.addItem<hkReflect::Opt::IMPL>( impl );
    }
    if ( optbits & hkReflect::Opt::SUBTYPE )
    {
        FAIL_WITH_ID_IF( 0x1aff8e1a, format == 0, "Invalid type #{} with Opt::SUBTYPE optional but no Opt::FORMAT.", typeId );

        const auto subtypeId = readUintVle<TypeId>(rb);
        FAIL_IF_INVALID(rb);
        FAIL_IF( subtypeId < 0, "Invalid sub-type ID (out of bounds) for type '{}'.", typeIdentity->getName() );
        // Subtype ID can be >= m_typeIdentity.getSize() for non-root types because SUBTYPE identity not enqueued in hkTagFileWriteFormat.cpp (by design).
        if ( subtypeId >= m_bundle.m_typeIdentity.getSize() )
        {
            // Update the fallback for the subtype before the type relocs are resolved/flattened in setType() below.
            m_bundle.m_typeRelocs.addFallbackType( subtypeId, &hkSerialize::Detail::TagfileBundle::NamedType::s_noIdentity );
        }

        builder.addItem<hkReflect::Opt::SUBTYPE>( nullptr ); // relocated later
        reloc.addSubType( subtypeId );
        DLOG( "Subtype ${}", subtypeId );

        if ( ( ( format & hkReflect::Format::KIND_MASK ) == hkReflect::KIND_ARRAY ) && ( format >> 8 ) )
        {
            DLOG( "FixedCount {}", format >> 8 );
        }
    }
    if ( optbits & hkReflect::Opt::VERSION )
    {
        const auto rawVersion = readUintVle64(rb);
        FAIL_IF_INVALID(rb);

        // We write a signed int cast to a hkUlong. The top 32 bits should be all one or all zero
        FAIL_WITH_ID_IF( 0xfc913ab0, ( ( rawVersion & 0xffffffff00000000 ) != 0 ) && ( ( rawVersion & 0xffffffff00000000 ) != 0xffffffff00000000 ), "Invalid version number {} for type #{}.", rawVersion, typeId );
        const hkInt32 version = static_cast<hkInt32>( rawVersion );
        FAIL_IF( version < 0, "Invalid negative version for type #{}.", typeId );

        builder.addItem<hkReflect::Opt::VERSION>( version );
        DLOG( "Version {}", version );
    }
    if ( optbits & hkReflect::Opt::SIZE_ALIGN )
    {
        const auto size = readUintVle<hkUint32>(rb);
        const auto align = readUintVle<hkUint32>(rb);
        FAIL_IF_INVALID(rb);

        hkReflect::Detail::SizeAlign sa;
        sa.m_sizeOf = size;
        sa.m_alignOf = align & 0xfff;
        sa.m_reqAlignEncoded = align >> 12;
        const hkUlong sizeAlign = sa.asUlong();
        builder.addItem<hkReflect::Opt::SIZE_ALIGN>( sizeAlign );
        DLOG( "{}", hkReflect::Detail::SizeAlign( sizeAlign ) );
    }
    if ( optbits & hkReflect::Opt::FLAGS )
    {
        const auto flags = readUintVle<hkUint16>(rb);
        FAIL_IF_INVALID(rb);

        builder.addItem<hkReflect::Opt::FLAGS>( flags );
        DLOG( "Flags {}", flags );
    }
    if ( optbits & hkReflect::Opt::DECLS )
    {
        const auto encoded = readUintVle<hkUint32>(rb);
        FAIL_IF_INVALID(rb);

        const int numFields = encoded & 0xffff;
        const int numProperties = encoded >> 16;

        FAIL_IF(numFields > rb.remaining() / 4/*see loop below*/, "Too many fields");

        for ( int i = 0; i < numFields; i++ )
        {
            const auto sid = readUintVle<hkUint16>(rb);
            const auto flags = readUintVle<hkUint16>(rb);
            const auto offset = readUintVle<hkUint16>(rb);
            const auto fieldId = readUintVle<TypeId>(rb);
            FAIL_IF_INVALID(rb);
            FAIL_IF( sid >= m_bundle.m_fieldStrings.getSize(), "Invalid field name string ID (out of bounds)." );
            FAIL_IF( typeIdIsNullOrOutOfBounds( fieldId, m_bundle ), "Invalid field type ID (out of bounds) for type '{}'.", typeIdentity->getName() );
            const char* const name = m_bundle.m_fieldStrings[sid];
            FAIL_IF( (flags & (hkReflect::Decl::DECL_PROPERTY_FIELD | hkReflect::Decl::DECL_DATA_FIELD)) != hkReflect::Decl::DECL_DATA_FIELD,
                "Invalid flags on {}::{}", typeIdentity->getName(), name);
            builder.addMember( name, offset, flags, nullptr ); // relocated later
            reloc.addField( fieldId );
            DLOG( "Field name={} offset={} flags={}, type=${}", name, offset, hkReflect::FieldDecl::DeclFlags( flags ), fieldId );
        }

        // Leave placeholders for the properties.
        for (int i = 0; i < numProperties; i++)
        {
            builder.addMember("$PLACEHOLDER$", 0,
                hkReflect::Decl::DECL_PROPERTY_FIELD | hkReflect::Decl::DECL_NOT_SERIALIZABLE,
                hkReflect::getType<hkReflect::Detail::Opaque>());
        }
    }
    if ( optbits & hkReflect::Opt::INTERFACES )
    {
        const auto numIface = readUintVle<int>(rb);
        FAIL_IF_INVALID(rb);
        FAIL_IF(numIface > rb.remaining() / 2/*see loop below*/, "Too many fields"); // 2 ->

        for ( int i = 0; i < numIface; ++i )
        {
            const auto tid = readUintVle<TypeId>(rb);
            const auto off = readUintVle<int>(rb);
            FAIL_IF_INVALID(rb);
            FAIL_IF( typeIdIsNullOrOutOfBounds( tid, m_bundle ), "Invalid interface type ID (out of bounds) for type '{}'.", typeIdentity->getName() );

            builder.addInterface( nullptr, off ); // relocated later
            reloc.addInterface( tid );
            DLOG( "Interface type=${} offset={} ", tid, off );
        }
    }
    if ( optbits & hkReflect::Opt::ATTRIBUTE_STRING )
    {
        const auto attributeId = readUintVle<int>(rb);
        FAIL_IF_INVALID(rb);
        FAIL_IF( attributeId >= m_bundle.m_attributeStrings.getSize(), "Invalid attribute string ID (out of bounds) for type '{}'.", typeIdentity->getName() );

        const char* const attributes = m_bundle.m_attributeStrings[attributeId];
        builder.addItem<hkReflect::Opt::ATTRIBUTE_STRING>( attributes );
        DLOG( "Attribute string=\"{}\" id={}", attributes, attributeId );
    }

    // Append type identity.
    // name and template are a bit special. They have been saved separately so that we can
    // still identify the types, even if the rest of type data has been stripped.
    if ( hkReflect::TypeDetail::localHasOptional( typeIdentity, hkReflect::Opt::NAME ) )
    {
        const char* const name = typeIdentity->getName();
        builder.addItem<hkReflect::Opt::NAME>( name );
        DLOG( "Name {}", name );
    }
    if ( hkReflect::TypeDetail::localHasOptional( typeIdentity, hkReflect::Opt::TEMPLATE ) )
    {
        if ( const hkReflect::Template* tplate = typeIdentity->getTemplate() )
        {
            DLOG_SCOPE("Template np={}", tplate->getNumParams());
            builder.addItem<hkReflect::Opt::TEMPLATE>( tplate );

            for ( int i = 0; i < tplate->getNumParams(); i++ )
            {
                const hkReflect::Template::Parameter* param = tplate->getParam(i);
                DLOG_IF(const char* pname = param->getName());
                if(param->isType())
                {
                    auto asNamedType = static_cast<const hkSerialize::Detail::TagfileBundle::NamedType*>( param->getAsType() );
                    HK_ASSERT(0x46f45e31, m_bundle.m_typeIdentity[asNamedType->m_id] == asNamedType,
                        "Template parameter has mismatching type pointer, inconsistent with its expected type identity (ID #{}).", asNamedType->m_id );
                    reloc.addTemplateParam( asNamedType->m_id );
                    const_cast<hkReflect::Template::Parameter*>( param )->m_storage = 0; // relocated later
                    DLOG( "TypeParam {} ${}", pname, param->getAsType() );
                }
                else
                {
                    DLOG("ValueParam {} {}", pname, param->getAsValue());
                }
            }
        }
    }

    hkReflect::Type* type;
    FAIL_IF(builder.allocate(&m_bundle.m_typeAllocator, &type).isFailure(), "Failed to allocate type from builder");

    m_bundle.setType( typeId, type, reloc );
    if (m_newTypeCallback)
    {
        m_newTypeCallback(type);
    }
}


template<typename READER>
hkSerialize::Ident hkSerialize::TagfileReadFormat::Impl::enterRootScope(READER& reader)
{
    if(reader.enter(TagfileRoot::TAGFILE) == TagfileRoot::TAGFILE)
    {
        DLOG("BeginBundle");
        return TagfileRoot::TAGFILE;
    }
    else if(reader.enter(TagfileRoot::COMPENDIUM) == TagfileRoot::COMPENDIUM)
    {
        DLOG("BeginCompendium");
        m_compendium.m_state = Compendium::READING;
        return TagfileRoot::COMPENDIUM;
    }
    Log_Warning("Input does not appear to be a tagfile");
    return HffBase::IDENT_NONE;
}


template<typename READER>
void hkSerialize::TagfileReadFormat::Impl::exitRootScope(READER& reader, Ident id)
{
    reader.leave(id);
    switch( id )
    {
        case TagfileRoot::TAGFILE:
            DLOG("EndBundle");
            break;
        case TagfileRoot::COMPENDIUM:
            m_compendium.m_state = Compendium::USING;
            DLOG("EndCompendium");
            break;
        default: // enterRootScope shouldn't allow this
            HK_ASSERT_NO_MSG(0x3dd8d7a4,0);
    }
}

void hkSerialize::TagfileReadFormat::Impl::handleDataStream( HffStreamReader& reader )
{
    int size = hkLosslessCast<int>( reader.sizeContent() );
    void* data = m_bundle.m_dataBuf.expandBy( size );
    hkLong readNum = reader.read( data, size );
    FAIL_IF( readNum != size, "Short read while reading DATA" );
    m_bundle.m_data = hkArrayViewT::make( data, size );
}

void hkSerialize::TagfileReadFormat::Impl::handleData( hkIo::ReadBufferView view, ViewConfig )
{
    m_bundle.m_data = hkArrayViewT::make( const_cast<void*>( view.begin() ), view.remaining() );
}

void hkSerialize::TagfileReadFormat::Impl::handleItemsStream( HffStreamReader& reader )
{
    int size = hkLosslessCast<int>( reader.sizeContent() );
    void* buf = m_bundle.m_indexBuf.expandBy( size );
    hkLong readNum = reader.read( buf, size );
    FAIL_IF( readNum != size, "Short read while reading DATA" );
    handleItems( hkIo::ReadBufferView( buf, size ), ViewConfig());
}

namespace
{
    struct RaiiReadLimiter
    {
        RaiiReadLimiter(hkIo::ReadBuffer& rbuf, hkLong limit)
            : m_rbuf(rbuf)
            , m_oldLimit(rbuf.getReadLimit())
        {
            m_rbuf.setReadLimit(limit);
        }
        ~RaiiReadLimiter()
        {
            m_rbuf.setReadLimit(m_oldLimit);
        }
        hkIo::ReadBuffer& m_rbuf;
        hkLong m_oldLimit;
    };
}

hkResult hkSerialize::TagfileReadFormat::Impl::resetBundle()
{
    if (m_bundle.types().isEmpty()) // first time through always succeeds
    {
        m_bundle.clear();
        return HK_SUCCESS;
    }
    else if (m_bundle.m_sequenceNumber != -1) // or if we're multibundle
    {
        m_bundle.clear();
        return HK_SUCCESS;
    }
    else if (m_compendium.m_state == Compendium::USING) // or compendium
    {
        m_bundle.clear();
        return HK_SUCCESS;
    }

    HK_ASSERT(0x2fb4e942, false, "Format is being used twice, but not initialized");
    return HK_FAILURE;
}

hkViewPtr<hkSerialize::Bundle> hkSerialize::TagfileReadFormat::Impl::read(hkIo::ReadBuffer& stream)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    HK_INTERNAL_TIME_CODE_BLOCK("TagfileReadFormat.Read", this);

    if (resetBundle().isFailure())
    {
        return nullptr;
    }

    HffStreamReader reader(stream);
    Ident rootScope = enterRootScope(reader);
    if(rootScope != HffBase::IDENT_NONE)
    {
        RaiiReadLimiter readLimiter(stream, stream.tell() + reader.sizeContent());
        m_bundle.m_sequenceNumber += 1;

        parse<READ>( reader, Traits<READ>::Data(this), hkArrayViewT::make( s_root ) );

        if ( m_status.isSuccess() )
        {
            exitRootScope( reader, rootScope );
            return &m_bundle;
        }
    }

    stream.status().setUnexpected();
    return HK_NULL;
}

void hkSerialize::TagfileReadFormat::Impl::readStrings(hkIo::ReadBufferView rb, hkArray<const char*>& strings, bool allowEmpty)
{
    const char* cur = static_cast<const char*>( rb.begin() );
    const char* const end = static_cast<const char*>( rb.end() );
    const char* s = cur;
    for( ; cur != end; ++cur )
    {
        if( *cur == 0 ) // found a null, we have a string
        {
            if( s != cur || allowEmpty ) // unless it's empty
            {
                strings.pushBack(s);
            }
            s = cur+1;
        }
    }
    // End-of-section padding is dropped (not null-terminated)
}

template <int VERSION>
void hkSerialize::TagfileReadFormat::Impl::handleTypeBody(hkIo::ReadBufferView rb, ViewConfig config)
{
    handleTypeBodyImpl(rb, config, VERSION);
}

void hkSerialize::TagfileReadFormat::Impl::handleTypeBodyImpl(hkIo::ReadBufferView rb, ViewConfig config, int version)
{
    const bool multiBundleWorkaround = (version == 0);

    if ( config.m_isInplace == false )
    {
        FAIL_IF( m_compendium.m_state == Compendium::USING, "Types cannot be added to when using a compendium." );

        // Resize type body array to match type identity array size
        const int prevNumTypes = hkMath::max2<int>( 1, m_bundle.accessTypes().getSize() ); // NULL type is always implicitly present at index #0
        const int nextNumTypes = m_bundle.m_typeIdentity.getSize();
        FAIL_IF( nextNumTypes < prevNumTypes, "Cannot remove types while deserializing." );
        m_bundle.accessTypes().setSize( nextNumTypes, nullptr );

        // Read body for new types
        int numRead = 0;
        const int numReadMax = ( nextNumTypes - prevNumTypes );
        if ( multiBundleWorkaround )
        {
            // Skip body of types already read in a previous bundle.
            rb.skip( m_hackTypeBodyOffset );
            FAIL_IF_INVALID(rb);
        }
        while ( ( rb.remaining() > 0 ) && ( *rb.peekAt<hkUint8>( 0 ) != 0 ) ) // 0 at the read position is padding
        {
            FAIL_IF( numRead >= numReadMax, "Too many type bodies actually present in bundle compared to what expected ({}).", numReadMax );
            readType( rb );
            FAIL_IF_INVALID(rb);
            ++numRead;
        }

        // Skip padding
        FAIL_WITH_ID_IF( 0x16964161, rb.remaining() >= 4, "Type body section (TBOD/TBDY) has oversized padding: max padding of 3 bytes expected, found {} instead.", rb.remaining() );
        const hkLong newEnd = rb.tell();
        while ( rb.remaining() ) // remaining() == 0 if stream end is encountered by readPod()
        {
            FAIL_WITH_ID_IF( 0x68bf09d3, rb.readPod<hkUint8>() != 0, "Found non-zero byte in padding of type body section (TBOD/TBDY)." );
        }
        FAIL_IF_INVALID(rb);

        if (multiBundleWorkaround)
        {
#if defined(HK_DEBUG_SLOW)
            
            // We need to fix multibundle handling properly so rather than change the format for 2016.1 and then change it again,
            // we just handle the duplication here by ignoring the repeated prefix. We hash the previously seen parts to check we're not skipping anything new.
            hkUint32 hashOfSkipped = hkHash::computeCrc32( rb.begin(), m_hackTypeBodyOffset );
            hkUint32 newHash = hkHash::appendCrc32( hashOfSkipped, hkAddByteOffset( rb.begin(), m_hackTypeBodyOffset ), newEnd - m_hackTypeBodyOffset );
            HK_ASSERT( 0x237fbc27, m_hackTypeBodyOffset == 0 || hashOfSkipped == m_hackTypeBodyHash, "Internal error in type body" );
            m_hackTypeBodyHash = newHash;
#endif
            m_hackTypeBodyOffset = newEnd;
        }

        // Check new types
#if defined(TAGFILE_EXTRA_LOAD_CHECKING)
        FAIL_IF( m_checker.addAndCheckTypes( m_bundle.types().ltrim( prevNumTypes ) ).isFailure(), "New types failed internal consistency checks." );
#endif
    }
}

static bool isPlaceholderDecl(hkReflect::Decl decl)
{
    return decl.getName()[0] == '$';
}

void hkSerialize::TagfileReadFormat::Impl::handlePropertyDecls(hkIo::ReadBufferView rb, ViewConfig config)
{
    if (!config.m_isInplace)
    {
        const auto numDecls = readUintVle<int>(rb);
        FAIL_IF_INVALID(rb);
        for (int i = 0; i < numDecls; ++i)
        {
            const auto ctxId = readUintVle<TypeId>(rb);
            const auto declIdx = readUintVle<hkUint32>(rb);
            const auto sid = readUintVle<hkUint16>(rb);
            FAIL_IF_INVALID(rb);

            // Get the decl context.
            FAIL_IF(typeIdIsNullOrOutOfBounds(ctxId, m_bundle), "Invalid context ID (out of bounds) when reading decl");
            const hkReflect::Type* context = m_bundle.type(ctxId);

            // Get the actual decl name.
            FAIL_IF(sid >= m_bundle.m_fieldStrings.getSize(), "Invalid field name string ID (out of bounds)");
            const char* declName = m_bundle.m_fieldStrings[sid];

            // Get the correct decl in the context.
            auto decls = hkReflect::TypeDetail::getDeclsArray(context);
            FAIL_IF(!decls, "Cannot add decl to '{}' as it does not have Opt::DECLS", context);
            hkReflect::FieldDecl decl = decls->getField(declIdx);
            FAIL_IF(!isPlaceholderDecl(decl), "Decl {} of '{}' ('{}') has already been read",
                declIdx, context, declName);

            // Set the name on the decl.
            hkReflect::TypeDetail::setFieldName(decl, declName);

            DLOG("Added decl ctx=${} idx={} name='{}'", ctxId, declIdx, declName);
        }
    }
}

void hkSerialize::TagfileReadFormat::Impl::handleTypeSequenceNumber(hkIo::ReadBufferView rb, ViewConfig)
{
    const hkUint32 seqn = rb.readPod<hkUint32Le>();
    FAIL_IF_INVALID(rb);
    FAIL_IF( seqn != m_bundle.m_sequenceNumber, "Sequence number incorrect. This reader didn't read the stream as it was written" );
    FAIL_WITH_ID_IF( 0x6e8537db, rb.remaining() != 0, "Oversized type sequence section (TSEQ)." );
}

void hkSerialize::TagfileReadFormat::Impl::handleTypeRollingHash(hkIo::ReadBufferView rb, ViewConfig)
{
    const hkUint32 oldHash = rb.readPod<hkUint32Le>();
    const hkUint32 newHash = rb.readPod<hkUint32Le>();
    FAIL_IF_INVALID(rb);
    FAIL_IF( oldHash != m_typeRollingHash, "Read format has lost sync: invalid previous type rolling hash (expected 0x{:x8}, got 0x{:x8}).", oldHash, m_typeRollingHash );
    m_typeRollingHash = newHash;
    FAIL_WITH_ID_IF( 0x6e8537db, rb.remaining() != 0, "Oversized type rolling hash section (TSHA)." );
}

void hkSerialize::TagfileReadFormat::Impl::handleCompendiumId(hkIo::ReadBufferView rb, ViewConfig)
{
    FAIL_IF( m_compendium.m_state != Compendium::READING, "Unexpected compendium ID section (TCID)." );
    FAIL_IF( rb.remaining() % sizeof( hkUint64Le ), "Type compendium ID has the wrong size." );
    while ( rb.remaining() )
    {
        const hkUint64 sig = rb.readPod<hkUint64Le>();
        m_compendium.m_sigs.pushBack( sig );
    }
}

//Type Compendium Reference - a 64 bit hash of types
void hkSerialize::TagfileReadFormat::Impl::handleCompendiumReference(hkIo::ReadBufferView rb, ViewConfig config)
{
    FAIL_IF(m_compendium.m_state != Compendium::USING, "Encountered compendium reference, but no compendium loaded" );
    FAIL_IF(rb.remaining() < hkSizeOf(hkUint64), "TCRF chunk is the wrong size ({}, expected at least {})", rb.remaining(), sizeof(hkUint64));
    hkUint64 sig = rb.readPod<hkUint64Le>();
    FAIL_IF_INVALID(rb);
    FAIL_IF(m_compendium.m_sigs.indexOf(sig) == -1,
        "Compendium with signature {} needed to load ({} are loaded)", sig, m_compendium.m_sigs);

    if ( config.m_isInplace )
    {
        FAIL_IF( rb.remaining() != sizeof( void* ) * 2 , "COMPENDIUM_REFERENCE wrong size" );

        typedef const hkReflect::Type* TypePtr;
        typedef hkArrayView<TypePtr> Array;
        Array* a = rb.accessRW<Array>( 1 );
        FAIL_IF( a->begin() || a->end() , "Buffer already inplace loaded" );
        new ( a ) Array( m_bundle.types() );
    }
}

template <int VERSION>
void hkSerialize::TagfileReadFormat::Impl::handleTypeIdentity(hkIo::ReadBufferView rb, ViewConfig config)
{
    handleTypeIdentityImpl(rb, config, VERSION);
}

void hkSerialize::TagfileReadFormat::Impl::handleTypeIdentityImpl(hkIo::ReadBufferView rb, ViewConfig, int version)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;

    const bool multiBundleWorkAround = (version == 0);

    // Note: the null type pointer at index #0 isn't explicitly stored

    // Allocate storage for new type identities
    int prevNumTypes;
    for ( prevNumTypes = m_bundle.m_typeIdentity.getSize() - 1; prevNumTypes >= 1; --prevNumTypes )
    {
        // Type identity array is padded with s_noIdentity for all types referenced by ID (e.g. from SUBTYPE)
        // which don't actually have a type identity loaded yet from any bundle (current or previous). Ignore those.
        if ( m_bundle.m_typeIdentity[prevNumTypes] != &Detail::TagfileBundle::NamedType::s_noIdentity )
        {
            break;
        }
    }
    ++prevNumTypes;
    const auto nextNumTypes = readUintVle<int>(rb);
    FAIL_IF_INVALID(rb);
    FAIL_IF( nextNumTypes < prevNumTypes, "Cannot remove types." );
    if ( nextNumTypes == prevNumTypes )
    {
        return;
    }
    {
        const int numNewTypes = nextNumTypes - prevNumTypes;
        FAIL_IF(numNewTypes > rb.remaining(), "More types than would fit in the section");
        FAIL_IF(m_bundle.m_typeIdentity.trySetSize(nextNumTypes, nullptr).isFailure(), "Failed to allocate storage for {} new type identities.", numNewTypes);
        Detail::TagfileBundle::NamedType* buf = m_bundle.m_typeAllocator._blockAlloc<Detail::TagfileBundle::NamedType>(numNewTypes); 
        for (int i = prevNumTypes; i < nextNumTypes; ++i)
        {
            new ( buf ) Detail::TagfileBundle::NamedType();
            m_bundle.m_typeIdentity[i] = buf;
            ++buf;
        }
    }

    if ( multiBundleWorkAround )
    {
        // 2015.1 has a bug with multibundle writing - it sends over ALL the type identities, even ones we've seen before.
        // Skip the ones we already have.
        for ( int tid = 1; tid < prevNumTypes; ++tid )
        {
            readUintVle64(rb);
            const auto numTemplateParams = readUintVle<int>(rb);
            FAIL_IF_INVALID(rb);
            for ( int i = 0; i < numTemplateParams; ++i )
            {
                readUintVle64(rb); // kindAndName
                readUintVle64(rb); // value
            }
        }
        FAIL_IF_INVALID(rb);
    }

    // Parse the new ones
    for (int tid = prevNumTypes; tid < nextNumTypes; ++tid)
    {
        auto* const typeIdentity = m_bundle.m_typeIdentity[tid];
        typeIdentity->m_id = tid;

        // Read name
        {
            // The "no name" value is a hack. We should change to reserve 0 for no name.
            
            const auto nid = readUintVle64(rb);
            FAIL_IF_INVALID(rb);
            if ( nid <= (hkUint64)hkTrait::NumericLimits<int>::maxValue() ) // negative indices (usually -1) mean unnamed
            {
                FAIL_IF( nid >= m_bundle.m_typeStrings.getSize(), "Invalid type name string ID #{} (out of bounds).", nid );
                typeIdentity->setName( m_bundle.m_typeStrings[ static_cast<int>(nid) ] );
            }
        }

        // Read template parameters
        const auto numParams = readUintVle<int>(rb);
        FAIL_IF_INVALID(rb);
        if ( numParams > 0 )
        {
            FAIL_IF( numParams > rb.remaining()/2 /*see loop below*/, "Invalid template" );

            using namespace hkReflect;

            Template* const tpl = Template::create(numParams, m_bundle.m_typeAllocator);
            typeIdentity->setTemplate(tpl);

            Template::Parameter* const params = const_cast<Template::Parameter*>( tpl->getParams() );
            for( int i = 0; i < numParams; ++i )
            {
                const auto kindAndNameId = readUintVle<int>(rb);
                const auto value = readUintVle<hkUlong>(rb);
                FAIL_IF_INVALID(rb);

                FAIL_IF( ( kindAndNameId < 0 ) || ( kindAndNameId >= m_bundle.m_typeStrings.getSize() ), "Invalid template parameter name string ID #{} (out of bounds) for type '{}'.",
                    kindAndNameId, typeIdentity->getName() );
                FAIL_IF( hkString::strLen( m_bundle.m_typeStrings[kindAndNameId] ) <= 1, "Invalid template parameter string ID #{} with empty name for type '{}'.",
                    kindAndNameId, typeIdentity->getName() );
                FAIL_IF( hkReflect::Template::Parameter::isValidKind( m_bundle.m_typeStrings[kindAndNameId][0] ) == false, "Invalid template parameter name '{}' for type '{}'"
                    " (expected 't' for type, 'v' for value; found '{}' instead).", m_bundle.m_typeStrings[kindAndNameId], typeIdentity->getName(), m_bundle.m_typeStrings[kindAndNameId][0] );

                params[i].m_kindAndName = m_bundle.m_typeStrings[kindAndNameId];
                if ( params[i].isType() )
                {
                    FAIL_IF( value==0 || value >= hkUlong(m_bundle.m_typeIdentity.getSize()), "Invalid template type ID #{} (out of bounds) for type '{}'.", value, typeIdentity->getName());
                    const int paramTypeId = int( value ); // safe as per above
                    params[i].m_storage = hkUlong( m_bundle.m_typeIdentity[paramTypeId] );
                }
                else
                {
                    params[i].m_storage = value;
                }
            }
        }

        // Set fallback for relocs in case this type does not have a body.
        m_bundle.m_typeRelocs.addFallbackType( tid, typeIdentity );
    }
}

hkViewPtr<hkSerialize::Bundle> hkSerialize::TagfileReadFormat::Impl::view(
    _In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_ hkUlong* lenUsed, ViewConfig config)
{
    HK_INTERNAL_TIME_CODE_BLOCK(config.m_isInPlace ? "TagfileReadFormat.Inplace" : "TagfileReadFormat.View", this);
    *lenUsed = 0;
    if (resetBundle().isFailure())
    {
        return nullptr;
    }

    HffMemoryReader reader( buf,bufLen );
    Ident rootScope = enterRootScope(reader);
    if ( rootScope != HffBase::IDENT_NONE )
    {
        m_bundle.m_sequenceNumber += 1;
        const hkLong entireSize = reader.sizeScope();

        parse<VIEW>( reader, Traits<VIEW>::Data(this, config), hkArrayViewT::make( s_root ) );

        if ( m_status.isSuccess() )
        {
            *lenUsed = entireSize;
            exitRootScope( reader, rootScope );
            return &m_bundle;
        }
    }
    return HK_NULL;
}


template<typename HASHER>
void hkSerialize::TagfileReadFormat::Impl::inplaceTypeHashesImpl(hkIo::ReadBufferView rb, HASHER& hasher)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    using namespace hkReflect;
    auto numHashes = readUintVle<int>(rb);
    FAIL_IF_INVALID(rb);
    hkArray<const hkReflect::Type*>& types = m_bundle.accessTypes();
    FAIL_IF_NO_MSG(numHashes > rb.remaining() / 5 /*see loop below*/);
    for(int i = 0; i < numHashes; ++i)
    {
        auto tid = readUintVle<TypeId>(rb);
        hkUint32 hsh = rb.readPod<hkUint32Le>();
        FAIL_IF_INVALID(rb);
        FAIL_IF_NO_MSG(typeIdIsNullOrOutOfBounds(tid, m_bundle));
        DLOG("Type {} hash {:x}// {}", tid, hsh, m_bundle.m_typeIdentity[tid]);
        const hkReflect::Type* t = hasher.findNative(m_bundle.m_typeIdentity[tid], hsh);
        FAIL_IF(t == HK_NULL, "Type {} with hash {:x} does not match a native type", m_bundle.m_typeIdentity[tid], hsh);
        types[tid] = t;
    }
}


void hkSerialize::TagfileReadFormat::Impl::inplaceTypeHashes( hkIo::ReadBufferView rb, ViewConfig config )
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;

    if ( config.m_isInplace )
    {
        // When we get the type hashes, this is the point where we can lookup the native types.
        if ( config.m_typeReg == HK_NULL )
        {
            // null means use the global one
            hkReflect::Detail::TypeHashCache hasher;
            inplaceTypeHashesImpl(rb, hasher );
        }
        else
        {
            // non-null means we can't use the global cache
            LocalTypeHasher hasher( config.m_typeReg );
            inplaceTypeHashesImpl(rb, hasher );
        }
    }
}

template<typename HASHER>
void hkSerialize::TagfileReadFormat::Impl::inplacePropertyHashesImpl(hkIo::ReadBufferView rb, HASHER& hasher )
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    using namespace hkReflect;
    const auto numHashes = readUintVle<int>(rb);
    FAIL_IF_INVALID(rb);

    hkArray<const hkReflect::Type*>& types = m_bundle.accessTypes();

    for ( int i = 0; i < numHashes; ++i )
    {
        const auto tid = readUintVle<TypeId>(rb);
        hkUint32 hsh = rb.readPod<hkUint32Le>();
        FAIL_IF_INVALID(rb);

        // Get the native type corresponding to tid.
        FAIL_IF(typeIdIsNullOrOutOfBounds(tid, m_bundle), "Invalid type ID (out of bounds) when reading property hash");
        const hkReflect::Type* t = types[tid];
        FAIL_IF( t == HK_NULL, "Type ${} is not initialized; HASHES section must have been loaded before PROPS_HASHES", tid );

        DLOG("Type {} property hash {:x}// {}", tid, hsh, m_bundle.m_typeIdentity[tid]);

        // Check the property hash.
        hkUint32 nativeHash = hasher.calc( t );
        FAIL_IF( hsh != nativeHash, "Property hash in file for type ${}, '{}' ({}) does not match native hash ({})",
            tid, m_bundle.m_typeIdentity[tid], hsh, nativeHash );

        m_propHashVerified[tid] = true;
    }
}

void hkSerialize::TagfileReadFormat::Impl::inplacePropertyHashes(hkIo::ReadBufferView reader, ViewConfig config )
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;

    if (config.m_isInplace)
    {
        // When we get the type hashes, this is the point where we can lookup the native types.
        if (config.m_typeReg == HK_NULL)
        {
            // null means use the global one
            hkReflect::Detail::PropertyHashCache hasher;
            inplacePropertyHashesImpl(reader, hasher);
        }
        else
        {
            // non-null means we can't use the global cache
            hkReflect::PropertyHasher hasher;
            inplacePropertyHashesImpl(reader, hasher);
        }
    }
}

void hkSerialize::TagfileReadFormat::Impl::handleNativePointers(hkIo::ReadBufferView view, ViewConfig config )
{
    if ( config.m_isInplace )
    {
        FAIL_WITH_ID_IF(0x574328eb, view.remaining() % sizeof(void*), "NATIVE_POINTERS section has the wrong size");
        int num = hkLosslessCast<int>( view.remaining() / sizeof( void* ) );
        const hkReflect::Type** types = ( const hkReflect::Type** )view.begin();
        for ( int i = 0; i < num; ++i )
        {
            FAIL_IF( types[i], "Buffer already inplace loaded" );
        }
        // use the memory inside the file
        m_bundle.accessTypes().setDataUserFree( types, num, num );
        m_propHashVerified.setSize(num, false);
    }
}

static _Ret_maybenull_z_ const char* s_kindName(hkUint32 /*TagfileItem::Enum*/ packed)
{
    using namespace hkSerialize::Detail;
    switch(packed & TagfileItem::KIND_MASK)
    {
        case TagfileItem::KIND_VAR0: return "VAR0";
        case TagfileItem::KIND_VARN: return "VARN";
        case TagfileItem::KIND_NOTE: return "NOTE";
        case TagfileItem::KIND_TYPE: return "TYPE";
        case TagfileItem::KIND_DECL: return "DECL";
        default:
            HK_ASSERT_NO_MSG(0x38d7a363, 0);
            return HK_NULL;
    }
}

void hkSerialize::TagfileReadFormat::Impl::handleItems( hkIo::ReadBufferView view, ViewConfig config )
{
    FAIL_WITH_ID_IF(0x66e1b3ab, view.remaining() % sizeof(Detail::TagfileItem), "ITEMS section has the wrong size");
    int numItems = hkLosslessCast<int>( view.remaining() / sizeof( Detail::TagfileItem ) );
    m_bundle.m_items = hkArrayViewT::make( ( Detail::TagfileItem* )view.begin(), numItems );
    if ( HK_ENDIAN_BIG )
    {
        for ( int i = 0; i < m_bundle.m_items.getSize(); ++i )
        {
            Detail::TagfileItem& item = m_bundle.m_items[i];
            item.packed = hkEndian::swap( item.packed );
            item.offset = hkEndian::swap( item.offset );
            item.count = hkEndian::swap( item.count );
        }
    }

#if defined(TAGFILE_EXTRA_LOAD_CHECKING)
    // sanity check index
    {
        const hkUint32 dataSize = m_bundle.m_data.getSize();
        const hkUint32 typeCount = m_bundle.types().getSize();
        if(m_bundle.m_items.getSize())
        {
            // first item is the empty sentinel
            FAIL_IF_NO_MSG(m_bundle.m_items[0].isEmpty() == false);
            FAIL_IF_NO_MSG(m_bundle.m_items[0].count != 0); //plus additional checks in the loop below
            if (m_bundle.m_items.getSize() >= 2)
            {
                
                // hkSerialize expects a var0 for cloning
                FAIL_IF_NO_MSG(isConvertibleToVar(m_bundle.m_items[1]) == false);
            }
        }
        for (const auto& item : m_bundle.m_items)
        {
            FAIL_IF_NO_MSG( item.getIndex() >= typeCount ); // item.packed.tid24 valid
            const auto* type = m_bundle.type(item.getIndex());
            FAIL_IF_NO_MSG(item.getKind() != Detail::TagfileItem::KIND_EMPTY && type == nullptr);
            hkUint32 count = item.count; // if nonzero do data range check
            switch (item.getKind()) // packed.kind4 valid
            {
                case Detail::TagfileItem::KIND_EMPTY:
                    FAIL_IF_NO_MSG(item.packed != 0);
                    FAIL_IF_NO_MSG(item.offset != 0);
                    FAIL_IF_NO_MSG(item.count >= (unsigned)m_bundle.m_items.getSize());
                    count = 0;
                    break;
                case Detail::TagfileItem::KIND_VAR0:
                    FAIL_IF_NO_MSG(item.count != 1);
                    break;
                case Detail::TagfileItem::KIND_VARN:
                    FAIL_IF_NO_MSG(item.offset > dataSize);
                    break;
                case Detail::TagfileItem::KIND_NOTE:
                    count = 1;
                    if( item.count < (unsigned)m_bundle.m_items.getSize()) // past-the-end items are implicitly empty
                    {
                        auto& v = m_bundle.m_items[item.count];
                        FAIL_IF_NO_MSG( v.isKindVar0() == false && v.isEmpty() == false);
                    }
                    break;
                case Detail::TagfileItem::KIND_TYPE:
                    FAIL_IF_NO_MSG(item.offset != -1);
                    FAIL_IF_NO_MSG(item.count != 0);
                    break;
                case Detail::TagfileItem::KIND_DECL:
                    FAIL_IF_NO_MSG(item.offset != -1);
                    FAIL_IF_NO_MSG(count >= (unsigned)type->getNumFields());
                    count = 0;
                    break;
                default:
                    FAIL_ALWAYS("Unexpected item kind {}", (int)item.getKind());
            }
            if (count)
            {
                hkUint32 typeSize = type->getSizeOf();
                hkUint32 typeAlign = type->getAlignOf();
                FAIL_IF_NO_MSG(typeSize == 0 || typeAlign == 0);
                // item is aligned and in bounds
                FAIL_IF_NO_MSG(item.offset % typeAlign);
                hkUint64 itemSize = hkUint64(typeSize) * count;
                FAIL_IF_NO_MSG( itemSize > dataSize);
                hkUint64 endOff = item.offset + itemSize;
                FAIL_IF_NO_MSG(endOff > dataSize);
            }
        }
    }
#endif

    if ( config.m_isInplace )
    {
        hkReflect::Detail::VtableInitializer vtinit;
        hkArray<hkReflect::Detail::VtableInitializer::PreCalc> preCalcTab;
        preCalcTab.setSize( m_bundle.types().getSize() );
        FAIL_IF( preCalcTab.isEmpty(), "No types found. Was this input created for inplace loading? (Save.withTarget())" );
        for ( int i = 1; i < m_bundle.m_items.getSize(); ++i )
        {
            const Detail::TagfileItem& item = m_bundle.m_items[i];
            //if( item.kindAndTid & Detail::ItemBits::FLAG_VTINIT ) 
            {
                TypeId tid = item.getIndex();
                DLOG( "{} id={} tid={} dataoff={}", s_kindName( item.packed ), i, tid, item.offset );
                hkReflect::Detail::VtableInitializer::PreCalc& preCalc = preCalcTab[tid];
                if ( preCalc.isInitialized() == false )
                {
                    preCalc = vtinit.preCalc( m_bundle.type( tid ) );
                }
                if ( preCalc.hasSome() )
                {
                    vtinit.init( preCalc, m_bundle.m_data.ptr( item.offset ), item.count );
                }
            }
        }
    }
}

void hkSerialize::TagfileReadFormat::Impl::handlePatchData( hkIo::ReadBufferView view, ViewConfig config )
{
    if ( config.m_isInplace )
    {
        {
            
            HK_INTERNAL_TIME_CODE_BLOCK( "Patch", this );
            DLOG_SCOPE( "Patch" );
            FAIL_WITH_ID_IF(0x20f2cca4, view.remaining() % sizeof(hkUint32), "PATCH_DATA section has the wrong size");
            int numInts = hkLosslessCast<int>( view.remaining() / sizeof( hkUint32Le ) );
            hkUint32* start = (hkUint32*)view.access<hkUint32>( numInts );
            if ( HK_ENDIAN_BIG )
            {
                for ( int i = 0; i < numInts; ++i )
                {
                    start[i] = hkEndian::swap( start[i] );
                }
            }

            int cur = 0;
            while ( cur + 2 < numInts )
            {
                TypeId srcTid = start[cur + 0];
                int srcNum = start[cur + 1];
                cur += 2;
                FAIL_IF( cur + srcNum > numInts, "Corrupt patch index body at index {}", cur );

                const hkReflect::Type* srcType = m_bundle.type( srcTid );
                const hkReflect::Detail::Impl* srcImpl = srcType->getImpl();
                DLOG_SCOPE( "Type {} ({} instances) //{}", srcTid, srcNum, srcType->getName() );

                for ( int i = 0; i < srcNum; ++i )
                {
                    hkUint32 off = start[cur + i];
                    void* srcAddr = m_bundle.m_data.ptr( off );
                    hkUint32 tgtIdx = *(hkUint32Le*)srcAddr;
                    DLOG( "srcOff={} tgtIdx={}", off, tgtIdx );
                    if ( tgtIdx )
                    {
                        const Detail::TagfileItem& targetItem = m_bundle.m_items[tgtIdx];
                        auto addrAndType = m_bundle.getAddrAndType( targetItem );
                        if ( targetItem.isKindDecl() )
                        {
                            // Verify that the property hash for this type has been received and checked.
                            hkReflect::PropertyFieldDecl decl(static_cast<const hkReflect::Type*>(addrAndType.m_addr));
                            FAIL_IF(decl && !m_propHashVerified[targetItem.getIndex()],
                                "Found pointer to decl in type ${} ({}) but no property hash has been loaded for the type",
                                targetItem.getIndex(), m_bundle.type(targetItem.getIndex()));
                        }
                        srcImpl->inplaceFixup( srcAddr, srcType, addrAndType.m_addr, addrAndType.m_type, targetItem.count );
                    }
                    else
                    {
                        srcImpl->inplaceFixup( srcAddr, srcType, HK_NULL, HK_NULL, 0 );
                    }
                }

                cur += srcNum;
            }
        }

        {
            HK_INTERNAL_TIME_CODE_BLOCK( "After", this );
            hkReflect::Detail::AfterInitializer afterinit;
            hkArray<hkReflect::Detail::AfterInitializer::PreCalc> preCalcTab;
            preCalcTab.setSize( m_bundle.types().getSize() );
            for ( int i = 1; i < m_bundle.m_items.getSize(); ++i )
            {
                const Detail::TagfileItem& item = m_bundle.m_items[i];
                //if( item.kindAndTid & Detail::ItemBits::FLAG_AFTERINIT ) 
                {
                    TypeId tid = item.getIndex();
                    hkReflect::Detail::AfterInitializer::PreCalc& preCalc = preCalcTab[tid];
                    if ( preCalc.isInitialized() == false )
                    {
                        preCalc = afterinit.preCalc( m_bundle.type( tid ) );
                    }
                    if ( preCalc.hasSome() )
                    {
                        afterinit.init( preCalc, m_bundle.m_data.ptr( item.offset ), m_bundle.type( tid ), item.count );
                    }
                }
            }
        }
    }
}

void hkSerialize::TagfileReadFormat::Impl::unloadData( hkIo::ReadBufferView view, UnloadState& state )
{
    // Take a view on the data buffer.
    state.m_data = view;
}

void hkSerialize::TagfileReadFormat::Impl::unloadCompendiumReference( hkIo::ReadBufferView view, UnloadState& state )
{
    // Take a view on the native type pointers in the compendium.
    view.skip( sizeof( hkUint64 ) );
    if ( view.remaining() != sizeof( state.m_types ) )
    {
        Log_Warning( "Invalid COMPENDIUM_REFERENCE" );
        state.m_status = HK_FAILURE;
    }
    else
    {
        hkMemUtil::memCpyOne( &state.m_types, view.access< hkArrayView<const hkReflect::Type*> >() );
        if ( state.m_types.isEmpty() )
        {
            Log_Warning( "COMPENDIUM_REFERENCE was not initialized. Was this buffer inplace loaded?" );
            state.m_status = HK_FAILURE;
        }
    }
}

void hkSerialize::TagfileReadFormat::Impl::unloadNativePointers( hkIo::ReadBufferView view, UnloadState& state )
{
    // Take a view on the native type pointers.
    state.m_types = hkArrayViewT::make( (const hkReflect::Type**)view.begin(), (const hkReflect::Type**)view.end() );
}

void hkSerialize::TagfileReadFormat::Impl::unloadItems( hkIo::ReadBufferView view, UnloadState& state )
{
    HK_ASSERT_NO_MSG( 0x7495c5d0, ( view.remaining() % sizeof( Detail::TagfileItem ) ) == 0 );
    hkArrayView<Detail::TagfileItem> items = hkArrayViewT::make( ( Detail::TagfileItem* )view.begin(), ( Detail::TagfileItem* )view.end() );

    // Go over the items and destruct the objects.
    for ( int i = 1; i < items.getSize(); ++i )
    {
        const Detail::TagfileItem& item = items[i];
        //if( item.kindAndTid & Detail::ItemBits::FLAG_VTINIT ) 
        if ( item.isKindVar0() )
        {
            TypeId tid = item.getIndex();
            const hkReflect::Type* t = state.m_types[tid];
            hkUlong off = item.offset;
            const void* p = state.m_data.peekAt<void>( off, 1 );
            hkReflect::TypeDetail::destruct( const_cast<void*>( p ), t );
        }
    }
}

void hkSerialize::TagfileReadFormat::registerCallback(NewTypeCallback cb)
{
    m_impl->m_newTypeCallback = cb;
}

hkResult hkSerialize::TagfileReadFormat::unload(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen)
{
    HK_INTERNAL_TIME_CODE_BLOCK( "TagfileReadFormat.Unload", HK_NULL );
    hkSerialize::HffMemoryReader hff(buf, bufLen);
    if(hff.enter(Detail::TagfileRoot::TAGFILE)== Detail::TagfileRoot::TAGFILE)
    {
        Impl::UnloadState state;
        Impl::parse<Impl::UNLOAD>( hff, state, hkArrayViewT::make( Impl::s_root ) );
        if ( state.m_status.isSuccess() )
        {
            hff.leave();
        }
        return state.m_status;
    }
    else if(hff.enter(Detail::TagfileRoot::COMPENDIUM)== Detail::TagfileRoot::COMPENDIUM)
    {
        //nothing, there are no vars
        hff.leave();
        return HK_SUCCESS;
    }
    else
    {
        Log_Error("Bad data given to InplaceLoad::unload");
        return HK_FAILURE;
    }
}

hkSerialize::TagfileReadFormat::TagfileReadFormat()
    : m_impl(new Impl())
{
}

hkSerialize::TagfileReadFormat::~TagfileReadFormat()
{
    delete m_impl;
}

hkViewPtr<hkSerialize::Bundle> hkSerialize::TagfileReadFormat::read(hkIo::ReadBuffer& rb)
{
    return m_impl->read(rb);
}

hkViewPtr<hkSerialize::Bundle> hkSerialize::TagfileReadFormat::view(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_ hkUlong* usedOut)
{
    return m_impl->view( buf, bufLen, usedOut, Impl::ViewConfig());
}

hkViewPtr<hkSerialize::Bundle> hkSerialize::TagfileReadFormat::inplace(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_ hkUlong* usedOut, _In_ const hkReflect::TypeReg* typeReg)
{
    Impl::ViewConfig conf;
    conf.m_isInplace = true;
    conf.m_typeReg = typeReg;
    return m_impl->view( buf, bufLen, usedOut, conf );
}

#ifndef HK_DYNAMIC_DLL
static
#endif
hkRefNew<hkSerialize::ReadFormat> binaryTagfileCreateRead()
{
    HK_OPTIONAL_COMPONENT_MARK_USED(hkReadFormatBinaryTagfile);
    return new hkSerialize::TagfileReadFormat();
}

HK_OPTIONAL_COMPONENT_DEFINE(hkReadFormatBinaryTagfile, hkSerialize::Detail::fileFormatBinaryTagfile.m_readFormatCreateFunc, binaryTagfileCreateRead);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
