// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : WIN32 X64 !OSINTERNAL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Container/Hash/hkHashMap.h>

namespace hkSerialize { class ReadFormat; }
namespace hkReflect { class Any; }

    /// Wrapper for hkCompatFormats DLL, enables loading of pre-2015 Havok file formats.
    /// Default initialization looks for the hkCompatFormats library in the current path. In order to override this
    /// behavior:
    /// * load the DLL through loadLibrary() passing a custom path;
    /// * pass the result to hkCompatFormats::replaceInstance().
    
    
    
class HK_EXPORT_COMMON hkCompatFormats : public hkReferencedObject
{
    public:

        HK_DECLARE_CLASS(hkCompatFormats, New, Singleton);

            /// Load the library from the given path. If dir is HK_NULL the library will be searched in the current
            /// executable path.
        hkCompatFormats(_In_opt_z_ const char* dir = HK_NULL);

            /// Release the library.
        ~hkCompatFormats();

            // Returns false if the DLL was not found/did not export the necessary symbols
        bool isLoaded() const { return m_isLoaded; }

        

            /// Converts a filter configuration set from the old HKO format to a 2015.1 XML file patching the filters
            /// options to the 2015.1 version.
        void convertConfigurationSet(_In_reads_bytes_(optionSize) const void* optionData, int optionSize, _Inout_ hkStreamWriter* optionsOut) const;

            /// Converts a Cloth Setup options set from the old XML packfile format to a 2015.1 XML file patching the
            /// plugins options to the 2015.1 version.
        void convertClothSetupOptions(_Inout_ class hkStreamReader* optionsIn, _Inout_ class hkStreamWriter* optionsOut) const;

            /// Converts an attribute descriptions set from the old XML packfile format to a 2015.1 XML file.
        void convertAttributeDescriptionOptions(_Inout_ class hkStreamReader* optionsIn, _Inout_ class hkStreamWriter* optionsOut) const;

    public:
            /// Callback for custom options loading. Will be called when loading options for custom filters/Cloth Setup
            /// plugins from a 2014 filter configuration set/Cloth Setup file.
            /// Must read the options from the optionData buffer, version them up if necessary and store them as a
            /// reflected, serializable object in optionsOut. The output object will be eventually passed to the
            /// corresponding filter/plugin.
        typedef void(*OptionsCb)(_In_reads_bytes_(optionSize) const void* optionData, int optionSize, unsigned int version, hkReflect::Any& optionsOut);

            /// Register a callback for the filter with the given id.
        void registerFilterOptionsCallback(unsigned filterId, OptionsCb callback);
            /// Unregister a callback for the filter with the given id.
        void unregisterFilterOptionsCallback(unsigned filterId);

            /// Register a callback for the Cloth Setup plugin with the given name.
        void registerClothSetupOptionsCallback(_In_z_ const char* pluginName, OptionsCb callback);
            /// Unregister a callback for the Cloth Setup plugin with the given name.
        void unregisterClothSetupOptionsCallback(_In_z_ const char* pluginName);

    public:

            /// Read an instance of klass saved in the old HKO format. The loaded object and its extra contents
            /// (strings, arrays) will be written in buffer. Returns HK_NULL if the read fails.
            /// Only string and array types supported by the old hkClass reflection are allowed (char*, hkStringPtr,
            /// hkArray, hkRelArray). Support for pointers, variants and variant arrays is not implemented.
        template <typename T>
        static T* readOptionsXml(_In_reads_bytes_(optionSize) const void* optionData, const int optionDataSize, hkArray<char>& buffer)
        {
            return static_cast<T*>(readOptionsXml(optionData, optionDataSize, buffer, hkReflect::getType<T>()).getAddress());
        }

            /// Read an instance of klass saved in the old HKO format.
            /// Returns a null Var if the read fails.
            /// See readOptionsXml(const void*, const int, hkArray<char>&) for details.
        static hkReflect::Var HK_CALL readOptionsXml(_In_reads_bytes_(optionSize) const void* optionData, const int optionDataSize,
            hkArray<char>& buffer, const hkReflect::Type* klass);

    public:
        // Internal use.
        static hkRefNew<hkSerialize::ReadFormat> HK_CALL createBinaryPackfile2014();
        static hkRefNew<hkSerialize::ReadFormat> HK_CALL createXmlTagfile2014();
        static hkRefNew<hkSerialize::ReadFormat> HK_CALL createXmlPackfile2010();

    protected:
        class ReadFormat;
        void clearPtrs();
        hkUlong loadClothSetupOptions(_In_reads_bytes_(optionSize) const char* optionData, int optionSize,
            class hctClothSetup20151Options& optionsOut,
            hkArray<hkReflect::Any>::Temp& customOptions,
            hkArray<hkReflect::Type*>::Temp& convertedTypes) const;
        void releaseClothSetupOptions(class hctClothSetup20151Options& options, hkUlong handle) const;

    protected:
        hkRefPtr<class hkDynamicLibrary> m_library;

        hkHashMap<unsigned, OptionsCb> m_filterCbs;
        hkHashMap<hkStringPtr, OptionsCb> m_clothSetupCbs;

        void* m_initFunc;
        void* m_quitFunc;

        void* m_loadFunc;
        void* m_freeFunc;

        void* m_loadFilterConfigSetFunc;
        void* m_getFilterConfigSetFunc;
        void* m_getFilterConfigFunc;
        void* m_getFilterStageFunc;

        void* m_loadClothSetupFunc;
        void* m_freeClothSetupFunc;
        void* m_getClothSetupDataNumFunc;
        void* m_getClothSetupDataFunc;
        void* m_getClothSetupObjectFunc;

        void* m_loadAttributeDescriptionFunc;
        void* m_getAttributeDescriptionGroupsNumFunc;
        void* m_getAttributeDescriptionGroupFunc;
        void* m_getAttributeDescriptionFunc;
        void* m_getAttributeDescriptionEnumItemFunc;

        bool m_isLoaded;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
