// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

namespace hkIo
{
    namespace Detail
    {
            /// Views an in-memory window of an output stream.
        struct HK_EXPORT_COMMON WriteBufferState
        {
            HK_DECLARE_CLASS(WriteBufferState, NewPlacement, NonCopyable);
            WriteBufferState();
            void clear() { m_cur = m_end = m_start = HK_NULL; m_attached = false; m_offsetAtStart = 0; }

            void setValid(bool valid)
            {
                m_okay = valid;
                if (!m_okay)
                {
                    clear();
                }
            }

            void* m_cur; // Write position in the buffer.
            void* m_end; // One past the last byte in the buffer.
            void* m_start; // Start of the buffer.
            hkLong m_attached;
            hkLong m_offsetAtStart; // Offset at start of buffer (nonzero for streams)
            bool m_okay; // Flag for if write buffer is okay
        };
            /// Interface to update a moving window over an output stream.
        struct HK_EXPORT_COMMON WriteBufferImpl : public hkReferencedObject
        {
            HK_DECLARE_CLASS(WriteBufferImpl, New);
            virtual void _attachTo(WriteBufferState& wb) = 0;
            virtual void _detachFrom(WriteBufferState& wb) = 0;
            virtual hkLong _writeRaw(WriteBufferState& wb, _In_reads_bytes_(n) const void* p, hkLong n) = 0;
            virtual hkLong _writeRawAt(WriteBufferState& wb, hkLong off, _In_reads_bytes_(n) const void* p, hkLong n) = 0;
            virtual _Ret_notnull_ void* _expandBy(WriteBufferState& wb, hkLong n) = 0;
            virtual void _flush(WriteBufferState& wb) = 0;

            static _Ret_notnull_ Detail::WriteBufferImpl* HK_CALL createFromArray(_Inout_ hkArrayBase<char>* buf, hkMemoryAllocator& a);
        };

        HK_EXPORT_COMMON _Ret_maybenull_ WriteBufferImpl* HK_CALL createWriterImpl(_In_opt_ hkStreamWriter* sw);
        HK_EXPORT_COMMON _Ret_maybenull_ WriteBufferImpl* HK_CALL createWriterImpl(_In_z_ const char* filename);
        template<typename A>
        _Ret_notnull_ WriteBufferImpl* HK_CALL createWriterImpl(hkArray<char, A>* a)
        {
            return WriteBufferImpl::createFromArray(a, a->getAllocator());
        }

        HK_ALWAYS_INLINE _Ret_notnull_ WriteBufferImpl* createWriterImpl(WriteBufferImpl* impl)
        {
            return impl;
        }

            /// Helper class to provide a backend which can be used by a WriteBuffer.
            /// This class is intended to be constructed implicitly by passing the address of an adaptable output object.
            /// E.g. void func(const Detail::WriteBufferAdapter& s); hkArray<char> buf; func(&buf);
            /// See overloads of Detail::createWriterImpl for the types which are adaptable.
        class HK_EXPORT_COMMON WriteBufferAdapter
        {
            public:
                WriteBufferAdapter()
                    : m_data(HK_NULL)
                    , m_dataSize(0)
                    , m_dataUsedSizeOut(HK_NULL)
                {
                }

                /// Note that smart pointers may need to be explicitly dereferenced for templates to be deduced correctly.
                template<typename T> WriteBufferAdapter(const T& t) { set(t); }
                template<typename T> WriteBufferAdapter(_In_bytecount_(dataSize) T* data, hkLong dataSize, _Out_opt_ hkLong* usedSizeOut = HK_NULL) { set(data, dataSize, usedSizeOut); }

                template<typename T> void set(const T& t)
                {
                    m_impl = createWriterImpl(t);
                    m_data = HK_NULL;
                    m_dataSize = 0;
                    m_dataUsedSizeOut = HK_NULL;
                }

                template<typename T> void set(_In_reads_bytes_(size) T* data, hkLong size, _In_opt_ hkLong* usedSizeOut = HK_NULL)
                {
                    m_impl = HK_NULL;
                    m_data = data;
                    m_dataSize = size;
                    m_dataUsedSizeOut = usedSizeOut;
                }

                operator hkBoolOperator::BoolType() const { return hkBoolOperator::cast(m_impl.val()); }
                _Ret_maybenull_ WriteBufferImpl* get() const { return m_impl; }
                HK_ALWAYS_INLINE void clear() { m_impl = HK_NULL; }

                _Ret_maybenull_ void* getData() const { return m_data; }
                hkLong getDataSize() const { return m_dataSize; }
                _Ret_maybenull_ hkLong* getUsedSizeOut() const { return m_dataUsedSizeOut; }

            protected:
                hkRefPtr<WriteBufferImpl> m_impl;
                void* m_data;
                hkLong m_dataSize;
                hkLong* m_dataUsedSizeOut;
        };
    };

        /// Interface to an append-only buffer.
        /// This class should be attach()ed to a WriteBufferImpl before writing. Similary, an explicit detach() call
        /// or an implicit detach() via the destructor will flush any change to the WriterImpl.
        /// Note that the storage behind the WriteBufferImpl may be inconsistent between attach() and detach() calls and should not be accessed.
    class HK_EXPORT_COMMON WriteBuffer
    {
        public:

            HK_DECLARE_CLASS(WriteBuffer, NewPlacement, NonCopyable);

                ///
            WriteBuffer()
                : m_usedSizeOut(HK_NULL)
            {
            }

                ///
            WriteBuffer(const Detail::WriteBufferAdapter& sink)
                : m_usedSizeOut(HK_NULL)
            {
                attach(sink);
            }

                /// Initializes the WriteBuffer using the given memory block. This
                /// is equivalent constructing a WriteBuffer using its default
                /// constructor and then calling \ref attach on it. See \ref attach
                /// for documentation of the specific parameters.
            WriteBuffer(_In_reads_bytes_(len) void* buf, hkLong len, _In_ hkLong* usedSizeOut)
                : m_usedSizeOut(HK_NULL)
            {
                attach(buf, len, usedSizeOut);
            }

                /// Implicitly detaches if not already detached.
            ~WriteBuffer();

                /// Attach to a given sink
            void attach(const Detail::WriteBufferAdapter& sink);

                /// Attach to the given memory block. All subsequent write calls
                /// will write to the given memory block.
                /// \param buf The memory block to write to.
                /// \param len The length of the memory block pointed to by \p buf.
                /// \param userSizeOut A pointer to an hkLong to which the actual
                ///   number of bytes written will be stored. The value can not be
                ///   relied upon before the WriteBuffer is detached. Can be HK_NULL
                ///   if not needed.
            void attach(_In_reads_bytes_(len) void* buf, hkLong len, _In_ hkLong* usedSizeOut);

                /// Ensure the writes are present in the sink.
            void detach();

                /// Is this write buffer attached to a sink?
            bool isAttached() const { return m_impl != HK_NULL; }

                /// Write the given buffer.
                /// \param p A pointer to the beginning of the data to write.
                /// \param n The size in bytes of the data to write.
                /// \returns The number of bytes written is returned. This number
                ///   of bytes written can be fewer than the requested number when
                ///   there's a maxmimum on the length of the stream and the end has
                ///   been reached. If this happens, then it's guaranteed that all
                ///   subsequent writes will fail and won't write anything at all.
            inline hkLong writeRaw(_In_reads_bytes_(n) const void* p, hkLong n);
                /// Forwards to writeRaw(void*, hkLong)
            inline hkLong writeRaw(hkArrayView<const void> a);

                /// Write the given buffer at the given offset. The number of  bytes
                /// written is returned.
                ///
                /// Calling this function does not move the write pointer of the
                /// stream, so the next call to one of the normal write functions
                /// (those which don't take a location) will continue at the same
                /// location they would have if this function hadn't been called.
            inline hkLong writeRawAt(hkLong off, _In_reads_bytes_(n) const void* p, hkLong n);

                /// Return a writable memory block which is valid until any mutating method of this object is called.
            inline _Ret_notnull_ void* expandBy(hkLong n);

                /// Convert SRC into a DST and memcpy it into the stream.
            template<typename DST, typename SRC>
            void writePod(const SRC& s);

                /// Convert SRC into a DST and memcpy it into the stream at the given offset.
            template<typename DST, typename SRC>
            void writePodAt(hkLong off, const SRC& s);

                /// Calls a write() method on an object.
                /// Constructs a DST d from s. Then calls d.write(*this)
            template<typename DST, typename SRC>
            void writeObj(const SRC& s);

                /// Write n bytes of zeros.
            void writeZero(hkLong n);

                /// Report the current offset.
            hkLong tell() const { return (hkLong(m_state.m_cur)-hkLong(m_state.m_start)) + m_state.m_offsetAtStart; }

                /// Report the offset of addr in the buffer.
            hkLong tellAt(_In_ const void* addr) const
            {
                hkLong off = hkGetByteOffset(m_state.m_start, addr);
                HK_ASSERT_NO_MSG(0x5cf60670, addr>=m_state.m_start);
                HK_ASSERT_NO_MSG(0x54f51233, addr< m_state.m_cur);
                return off + m_state.m_offsetAtStart;
            }

                /// Safe rewind of data if possible
            inline hkResult rewind(hkLong amount);

                /// Flush any buffered output.
            void flush();

                /// Query write buffer failure state
            inline bool isOk() const;

                /// Force the write buffer to be flagged as bad
            inline void forceNotOk();

            void write8(hkUint8 u) { writePod<hkUint8>(u); }
            void write32i(hkInt32 i) { writePod<hkInt32Le>(i); }
            void write32u(hkUint32 u) { writePod<hkUint32Le>(u); }
            void write64i(hkInt64 i) { writePod<hkInt64Le>(i); }
            void write64u(hkUint64 u) { writePod<hkUint64Le>(u); }

        protected:

            hkLong implWriteRaw(_In_reads_bytes_(n) const void* p, hkLong n);
            hkLong implWriteRawAt(hkLong off, _In_reads_bytes_(n) const void* p, hkLong n);
            _Ret_maybenull_ void* implExpandBy(hkLong n);

            Detail::WriteBufferState m_state;
            hkRefPtr<Detail::WriteBufferImpl> m_impl;
            hkLong* m_usedSizeOut;
    };
}

#include <Common/Base/Serialize/Detail/hkWriteBuffer.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
