// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>

#define DEBUG_LOG_IDENTIFIER "io.WriteBuffer"
#include <Common/Base/System/Log/hkLog.hxx>

namespace hkIo { namespace Detail {

    class HK_EXPORT_COMMON StreamWriteBufferImpl : public Detail::WriteBufferImpl
    {
        public:

            StreamWriteBufferImpl(hkStreamWriter* sw)
                : m_stream(sw), m_seekTellSupported(false)
            {
            }

        protected:

            virtual void _flush(Detail::WriteBufferState& wb) HK_OVERRIDE
            {
                // flush existing
                int numInBuf = hkGetByteOffsetInt(m_buf.begin(), wb.m_cur);
                HK_ON_DEBUG(hkLong written = )m_stream->write( m_buf.begin(), numInBuf );
                HK_ASSERT_NO_MSG(0x1264fba, written==numInBuf || (!m_stream->isOk()));
                wb.m_offsetAtStart += numInBuf;
                wb.m_cur = m_buf.begin();
            }

            virtual void _attachTo(Detail::WriteBufferState& wb) HK_OVERRIDE
            {
                m_seekTellSupported = m_stream->seekTellSupported();
                hkLong offAtStart = m_seekTellSupported ? m_stream->tell() : 0;
                wb.m_offsetAtStart = hkMath::max2( offAtStart, 0 );

                m_buf.setSize( hkMath::max2(m_buf.getCapacity(), 1024) );
                wb.m_start = wb.m_cur = m_buf.begin();
                wb.m_end = m_buf.begin() + m_buf.getSize();
            }

            virtual void _detachFrom(Detail::WriteBufferState& wb) HK_OVERRIDE
            {
                _flush(wb);
                HK_ASSERT_NO_MSG(0xdf1db3c, m_seekTellSupported==false || (wb.m_offsetAtStart==m_stream->tell()));
                wb.clear();
            }

            virtual hkLong _writeRaw(Detail::WriteBufferState& wb, _In_reads_bytes_(n) const void* p, hkLong n) HK_OVERRIDE
            {
                _flush(wb);

                // heuristic - small writes are buffered, larger writes written
                if( n*2 < m_buf.getSize() )
                {
                    hkMemUtil::memCpy(wb.m_cur, p, n);
                    wb.m_cur = hkAddByteOffset(wb.m_cur, n);
                    return n;
                }
                else
                {
                    hkLong r = m_stream->write(p, hkLosslessCast<int>(n));
                    wb.m_offsetAtStart += r;
                    return r;
                }
            }

            virtual hkLong _writeRawAt(Detail::WriteBufferState& wb, hkLong off, _In_reads_bytes_(n) const void* p, hkLong n) HK_OVERRIDE
            {
                if( m_seekTellSupported )
                {
                    HK_ASSERT_NO_MSG(0x40cb2db4, off < wb.m_offsetAtStart);
                    hkLong prev = m_stream->tell();
                    m_stream->seek( hkLosslessCast<int>(off), hkStreamWriter::STREAM_SET);
                    hkLong written = m_stream->write(p, hkLosslessCast<int>(n));
                    m_stream->seek(hkLosslessCast<int>(prev), hkStreamWriter::STREAM_SET);
                    return written;
                }
                Log_Warning("_writeRawAt failed - stream does not support seeking");
                return -1;
            }


            virtual _Ret_notnull_ void* _expandBy(Detail::WriteBufferState& wb, hkLong n) HK_OVERRIDE
            {
                _flush(wb);
                m_buf.setSize( hkMath::max2(hkLosslessCast<int>(n), m_buf.getSize()) );
                wb.m_start = m_buf.begin();
                wb.m_cur = m_buf.begin() + n;
                wb.m_end = m_buf.begin() + m_buf.getSize();
                return m_buf.begin();
            }

            hkRefPtr<hkStreamWriter> m_stream;
            hkArray<char> m_buf;
            hkBool m_seekTellSupported;
    };


    struct ArrayWriteBufferImpl : public Detail::WriteBufferImpl
    {
        public:

            ArrayWriteBufferImpl(hkArrayBase<char>& arr, hkMemoryAllocator& mem)
                : m_buf(arr), m_mem(mem), m_attached(HK_NULL)
            {
            }

        protected:

            // Invariants:
            // [0 .. m_cur] is written part, [m_cur .. m_end] is the available buffer
            // m_end == m_buf.end()
            // m_buf.getSize() == m_buf.getCapacity() (when unlocked)

            virtual void _attachTo(Detail::WriteBufferState& wb) HK_OVERRIDE
            {
                wb.m_offsetAtStart = 0;
                attach2(wb);
            }

            virtual void _detachFrom(Detail::WriteBufferState& wb) HK_OVERRIDE
            {
                detach2(wb);
                wb.clear();
            }

            virtual hkLong _writeRaw(Detail::WriteBufferState& wb, _In_reads_bytes_(n) const void* p, hkLong n) HK_OVERRIDE
            {
                detach2(wb);
                m_buf._append(m_mem, static_cast<const char*>(p),hkLosslessCast<int>(n));
                attach2(wb);
                return n;
            }
            virtual hkLong _writeRawAt(Detail::WriteBufferState& wb, hkLong off, _In_reads_bytes_(n) const void* p, hkLong n) HK_OVERRIDE
            {
                hkMemUtil::memCpy(m_buf.begin()+off, p, n);
                return n;
            }
            virtual _Ret_notnull_ void* _expandBy(Detail::WriteBufferState& wb, hkLong n) HK_OVERRIDE
            {
                detach2(wb);
                void* p = m_buf._expandBy(m_mem, hkLosslessCast<int>(n));
                attach2(wb);
                return p;
            }
            virtual void _flush(Detail::WriteBufferState& wb) HK_OVERRIDE
            {
            }

            // prepares m_buf for modification using hkArray methods
            inline void detach2(Detail::WriteBufferState& wb)
            {
                int actualSize = hkGetByteOffsetInt(m_buf.begin(), wb.m_cur);
                HK_ASSERT_NO_MSG(0x665e694d, actualSize <= m_buf.getSize());
                m_buf.setSizeUnchecked( actualSize ); // we may have unused space
                m_attached = HK_NULL;
            }
            // End of hkArray access, prepares for WriterBuffer methods
            inline void attach2(Detail::WriteBufferState& wb)
            {
                HK_ASSERT_NO_MSG(0x761cfd5d, m_attached == HK_NULL);
                m_attached = &wb;
                wb.m_start = m_buf.begin();
                wb.m_cur = m_buf.begin() + m_buf.getSize();
                m_buf.setSizeUnchecked( m_buf.getCapacity() ); //use full capacity
                wb.m_end = m_buf.begin() + m_buf.getCapacity();
            }

            hkArrayBase<char>& m_buf;
            hkMemoryAllocator& m_mem;
            const Detail::WriteBufferState* m_attached;
        };
} }

hkIo::Detail::WriteBufferState::WriteBufferState()
    : m_cur(HK_NULL), m_end(HK_NULL), m_start(HK_NULL), m_attached(false), m_offsetAtStart(0), m_okay(true)
{
}


_Ret_maybenull_ hkIo::Detail::WriteBufferImpl* HK_CALL hkIo::Detail::createWriterImpl(_In_opt_ hkStreamWriter* sw)
{
    if (sw)
    {
        WriteBufferImpl* impl = new StreamWriteBufferImpl(sw);
        impl->setReferenceCount(0);
        return impl;
    }
    return HK_NULL;
}

_Ret_maybenull_ hkIo::Detail::WriteBufferImpl* HK_CALL hkIo::Detail::createWriterImpl(_In_z_ const char* filename)
{
    return createWriterImpl(hkOstream(filename).getStreamWriter());
}

_Ret_notnull_ hkIo::Detail::WriteBufferImpl* hkIo::Detail::WriteBufferImpl::createFromArray(_Inout_ hkArrayBase<char>* arr, hkMemoryAllocator& mem)
{
    Detail::WriteBufferImpl* impl = new Detail::ArrayWriteBufferImpl(*arr, mem);
    impl->setReferenceCount(0);
    return impl;
}


hkIo::WriteBuffer::~WriteBuffer()
{
    detach();
}

void hkIo::WriteBuffer::attach(const hkIo::Detail::WriteBufferAdapter& sink)
{
    detach();
    if (hkIo::Detail::WriteBufferImpl* w = sink.get())
    {
        m_impl = w;
        w->_attachTo(m_state);
        m_usedSizeOut = HK_NULL;
    }
    else
    {
        void* buf = sink.getData();
        hkLong len = sink.getDataSize();

        m_state.m_cur = buf;
        m_state.m_end = hkAddByteOffset(buf, len);
        m_state.m_start = buf;
        m_state.m_attached = true;
        m_state.m_offsetAtStart = 0;
        m_usedSizeOut = sink.getUsedSizeOut();
    }

    m_state.m_attached = true;

}

void hkIo::WriteBuffer::attach(_In_reads_bytes_(len) void* buf, hkLong len, _In_ hkLong* usedSizeOut)
{
    detach();

    m_state.m_cur = buf;
    m_state.m_end = hkAddByteOffset(buf, len);
    m_state.m_start = buf;
    m_state.m_attached = true;
    m_state.m_offsetAtStart = 0;
    m_usedSizeOut = usedSizeOut;
}

void hkIo::WriteBuffer::detach()
{
    if (m_impl)
    {
        m_impl->_detachFrom(m_state);
        m_impl = HK_NULL;
    }
    else if(m_state.m_attached && m_usedSizeOut)
    {
        // We don't have an impl, yet we are attached which means we're writing
        // to a memory block. Since we have an m_usedSizeOut (which is only
        // possible when we're using a memory block), we need to write the
        // number of bytes written to it.
        *m_usedSizeOut = hkGetByteOffset(m_state.m_start, m_state.m_cur);
    }

    m_state.m_attached = false;
}


void hkIo::WriteBuffer::writeZero(hkLong n)
{
    while(n > 0)
    {
        hkLong s = hkMath::min2(n, 4096);
        void* p = expandBy(s);
        hkMemUtil::memSet(p, 0, s);
        n -= s;
    }
}

void hkIo::WriteBuffer::flush()
{
    if (m_impl)
    {
        if (m_state.m_cur != m_state.m_start)
        {
            m_impl->_flush(m_state);
        }
    }
}

hkLong hkIo::WriteBuffer::implWriteRaw(_In_reads_bytes_(n) const void* p, hkLong n)
{
    hkLong avail = hkGetByteOffset(m_state.m_cur, m_state.m_end);
    HK_ASSERT(0x30fec1ac, avail < n, "implWriteRaw should only be called when we're "
        "trying to write past the end of m_state's current buffer.");

    if (m_impl)
    {
        return m_impl->_writeRaw(m_state, p, n);
    }
    else
    {
        // There's no implementation, so we have to do with the buffer in
        // m_state. We know this buffer isn't large enough, so we write the part
        // which fits and return the number of bytes written.

        hkMemUtil::memCpy(m_state.m_cur, p, avail);
        m_state.m_cur = m_state.m_end;
        return avail;
    }
}

hkLong hkIo::WriteBuffer::implWriteRawAt(hkLong off, _In_reads_bytes_(n) const void* p, hkLong n)
{
    if (m_impl)
    {
        return m_impl->_writeRawAt(m_state, off, p, n);
    }
    else
    {
        HK_ASSERT(0x292694e, m_state.m_offsetAtStart == 0,
            "m_state.m_offsetAtStart should be null if there's no m_impl "
            "(that is, if we're writing to a memory buffer).");

        void* writeBegin = hkAddByteOffset(m_state.m_cur, off);
        hkLong avail = hkGetByteOffset(writeBegin, m_state.m_end);

        HK_ASSERT(0x35f45377, avail < n, "implWriteRawAt should only be called when we're "
            "trying to write past the end of m_state's buffer.");

        hkMemUtil::memCpy(writeBegin, p, avail);
        return avail;
    }
}

_Ret_maybenull_ void* hkIo::WriteBuffer::implExpandBy(hkLong n)
{
    if (m_impl)
    {
        return m_impl->_expandBy(m_state, n);
    }
    return nullptr;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
