// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/Serialize/Core/hkSerializeCore.h>
#include <Common/Base/Reflect/Util/hkReflectClone.h>
#include <Common/Base/Serialize/Resource/hkResource.h>
#include <Common/Base/Container/PointerMap/hkPointerMap.h>

namespace hkSerialize {

class ReadFormat;

namespace Detail {

        /// Utility class to keep track of the written types.
    class HK_EXPORT_COMMON TypeWriterMap
    {
    public:
        TypeWriterMap();

            /// Get the id of the type WITHOUT necessarily writing it. (i.e. like a weak reference)
            /// This is used for e.g. pointer types where the pointed-to object is not saved.
        TypeId lookupType(_In_ const hkReflect::Type* type);

            /// Get the id of the type AND enqueue the type for writing if it is not already written/in the queue.
        TypeId enqueueForWrite(_In_ const hkReflect::Type* type);

            ///
        bool isEmpty() const { return m_nextToWrite == m_writeQueue.getSize(); }

            /// Get the next type in the write queue.
        _Ret_maybenull_ const hkReflect::Type* nextToWrite();

    private:
        typedef hkPointerMap<const hkReflect::Type*, TypeId> ItemFromType;
        ItemFromType m_itemFromType;
        hkArray<bool> m_written;
        hkArray<const hkReflect::Type*> m_writeQueue;
        int m_nextToWrite;
    };

        /// Clones objects on instances of the types in the provided registry.
    struct HK_EXPORT_COMMON CloneToRegistered : public hkReflect::Detail::CloneOnHeap
    {
        public:
            CloneToRegistered(_In_opt_ const hkReflect::TypeReg* typeReg = HK_NULL);
            virtual _Ret_maybenull_ const hkReflect::Type* typeFromType(_In_ const hkReflect::Type* type) HK_OVERRIDE;

            const hkReflect::TypeReg* m_typeReg;
    };

        /// Keeps alive all loaded objects and destroys them all at once on destruction.
        /// Reference count of loaded objects is disabled.
    struct OwnedResource : public hkResource
    {
        public:
            OwnedResource();
            OwnedResource(const hkArrayView<hkReflect::Var>& objects);
            ~OwnedResource();

            virtual void getImportsExports(hkArray<hkResource::Import>& impOut, hkArray<hkResource::Export>& expOut) const HK_OVERRIDE;
            virtual hkReflect::Var getContents() const HK_OVERRIDE;
            virtual void cleanup() HK_OVERRIDE;
            virtual void callDestructors() HK_OVERRIDE;
            virtual _Ret_opt_z_ const char* getName() const HK_OVERRIDE;

            void addObject(hkReflect::Var obj);
            void addExport(const hkReflect::Var& var, _In_z_ const char* name);
            void addImport(const hkReflect::PointerVar& pvar, _In_z_ const char* name);
            void clearContents();

        private:
            hkArray<hkResource::Import> m_imports;
            hkArray<hkResource::Export> m_exports;
            hkArray<hkReflect::Var> m_contents;
            // Needed to free objects
            hkPointerMap<void*, int> m_actualObjectSizes;
            bool m_destructorsCalled;
    };


    /// Error ids which can be issued during de-serialization.
    enum ErrorID
    {
        /// Failed to read from stream.
        ERRORID_READ_FAILED = 0xc8920000,
        /// Unsupported file format.
        ERRORID_UNSUPPORTED_FORMAT = 0xc8920001,
        /// Packfile cannot be read on this platform.
        ERRORID_PACKFILE_PLATFORM = 0xc8920002,
        /// Versioning cannot be applied to loaded objects.
        ERRORID_VERSIONING_FAILED = 0xc8920003,
        /// Stream was ok but some error in the data.
        ERRORID_LOAD_FAILED = 0xc8920005,
    };

    struct FileFormatRegNode;

        /// Detect format.
    typedef bool (HK_CALL*ReadFormatDetectFunc)(_In_reads_bytes_(bufSize) const void* buf, hkUlong bufSize);

        /// The type of functions which create a ReadFormat to load the format.
    typedef hkRefNew<hkSerialize::ReadFormat> (HK_CALL*ReadFormatCreateFunc)();

        /// The type of functions which create a WriteFormat to write the format.
    typedef hkRefNew<hkSerialize::WriteFormat>(HK_CALL*WriteFormatCreateFunc)();

        /// Detect the format, using the registered detect format functions.
    _Ret_maybenull_ const FileFormatRegNode* detectFormat(_In_reads_bytes_(bufSize) const void* buf, hkUlong bufSize );

        /// Create a registered ReadFormat class to the buffer, if there is one.
    hkRefNew<hkSerialize::ReadFormat> createReadFormat(_In_reads_bytes_(bufSize) const void* buf, hkUlong bufSize );

    enum
    {
        /// A number of bytes which is sufficient to identify formats. (Hopefully XML formats do not have lots of initial whitespace.)
        FORMAT_HEADER_PEEK_LENGTH = 128
    };

        /// Register a factory which can create a ReadFormat object for a format.
    struct HK_EXPORT_COMMON FileFormatRegNode
    {
        public:

            FileFormatRegNode(_In_z_ const char* name, ReadFormatDetectFunc detect, ReadFormatCreateFunc createRead, WriteFormatCreateFunc createWrite);

            bool detect(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen) const { return (*m_detectFunc)(buf,bufLen); }
            hkRefNew<ReadFormat> createReadFormat() const { if(m_readFormatCreateFunc) { return (m_readFormatCreateFunc)(); } return HK_NULL; }
            hkRefNew<WriteFormat> createWriteFormat() const { if (m_writeFormatCreateFunc) { return (m_writeFormatCreateFunc)(); } return HK_NULL; }
            _Ret_z_ const char* getName() const { return m_name; }

                /// Name of the format
            const char* m_name;
                /// Detect a format
            ReadFormatDetectFunc m_detectFunc;
                /// Create a read format
            ReadFormatCreateFunc m_readFormatCreateFunc;

                /// Create a write format
            WriteFormatCreateFunc m_writeFormatCreateFunc;
    };

    HK_EXPORT_COMMON extern FileFormatRegNode fileFormatBinaryTagfile;
    HK_EXPORT_COMMON extern FileFormatRegNode fileFormatBinaryTagfile2015;
    HK_EXPORT_COMMON extern FileFormatRegNode fileFormatBinaryTagfileNT6;
    HK_EXPORT_COMMON extern FileFormatRegNode fileFormatBinaryTagfile2014;
    HK_EXPORT_COMMON extern FileFormatRegNode fileFormatBinaryPackfile2014;
    HK_EXPORT_COMMON extern FileFormatRegNode fileFormatXmlTagfile;
    HK_EXPORT_COMMON extern FileFormatRegNode fileFormatXmlTagfile2014;
    HK_EXPORT_COMMON extern FileFormatRegNode fileFormatXmlPackfile2010;
    HK_EXPORT_COMMON extern FileFormatRegNode fileFormatYamlfile;
    HK_EXPORT_COMMON extern FileFormatRegNode fileFormatCompressedTagfile;
} }

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
