// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Detail/hkSerializeDetail.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/Serialize/Format/Tagfile2014/hkTagfileCommon2014.h>
#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/Serialize/Util/Xml/hkXmlStreamParser.h>
#include <Common/Base/Reflect/TypeReg/Detail/hkTypeRegDetail.h>

#include <Common/Base/Container/PointerMap/hkMap.hxx>

#define DEBUG_LOG_IDENTIFIER "s11n.SerializeDetail"
#include <Common/Base/System/Log/hkLog.hxx>

#if defined(HK_BUILDING_WITH_ENGINE)
    #include <Common/NewBase/Reflection/Util/hkDynamicTypeManager.h> 

    static hkResult s_typeNotFound(_In_ const hkReflect::Type* srcType, _In_ const hkReflect::TypeReg* typeReg, _Outptr_ hkReflect::Type const** typeOut);

    static hkResult s_copyIfSubtypeIsGood(_In_ const hkReflect::Type* srcType, _In_ const hkReflect::Type* srcSub, _In_ const hkReflect::TypeReg* typeReg, _Outptr_ hkReflect::Type const** typeOut)
    {
        if (srcSub == nullptr)
        {
            return HK_FAILURE; // no variant types
        }

        if (auto dstSub = typeReg->typeFromType(srcSub))
        {
            // ok known
        }
        else if (s_typeNotFound(srcSub, typeReg, &dstSub).isFailure())
        {
            return HK_FAILURE;
        }

        *typeOut = hkDynamicTypeManager::getInstance().addCopyType(srcType);
        return *typeOut ? HK_SUCCESS : HK_FAILURE;
    }

    static hkResult s_typeNotFound(_In_ const hkReflect::Type* srcType, _In_ const hkReflect::TypeReg* typeReg, _Outptr_ hkReflect::Type const** typeOut)
    {
        *typeOut = nullptr;
        
        if (srcType->isDynamicType() || srcType->isForeign())
        {
            *typeOut = hkDynamicTypeManager::getInstance().addCopyType(srcType);
            return *typeOut ? HK_SUCCESS : HK_FAILURE;
        }
        else if (const hkReflect::PointerType* srcPt = srcType->asPointer())
        {
            return s_copyIfSubtypeIsGood(srcType, srcPt->getSubType(), typeReg, typeOut);
        }
        else if (const hkReflect::ArrayType* srcAt = srcType->asArray())
        {
            return s_copyIfSubtypeIsGood(srcType, srcAt->getSubType(), typeReg, typeOut);
        }
        return HK_FAILURE;
    }

#else
    static hkResult s_typeNotFound(_In_ const hkReflect::Type*, const hkReflect::TypeReg*, _Outptr_ hkReflect::Type const** out)
    {
        *out = nullptr;
        return HK_FAILURE;
    }
#endif


hkSerialize::Detail::TypeWriterMap::TypeWriterMap()
    : m_nextToWrite(0)
{
    m_written.pushBack(true);
    m_itemFromType.insert(HK_NULL, 0);
}

namespace
{
    // Remap tracker types on Opaque so that they are never saved out.
    
    _Ret_maybenull_ const hkReflect::Type* s_remapOpaque(const hkReflect::Type* type)
    {
        if (type && hkReflect::Detail::isOpaqueWrapper(type))
        {
            return hkReflect::getType<hkReflect::Detail::Opaque>();
        }
        return type;
    }
}

hkSerialize::TypeId hkSerialize::Detail::TypeWriterMap::lookupType(_In_ const hkReflect::Type* type)
{
    type = s_remapOpaque(type);

    TypeId id = m_written.getSize();
    m_itemFromType.get(type, &id);
    if (id == m_written.getSize())
    {
        // The type is not in the map.
        // Check if it is a duplicate, in that case use the id of the original (check the registry only if it is a
        // registered builtin type, i.e. it has a local name and an inheritance value).
        if (hkReflect::TypeDetail::localHasOptional(type, hkReflect::Opt::NAME) &&
            hkReflect::TypeDetail::getInheritance(type) && !type->isDynamicType())
        {
            const hkReflect::Type* orig = hkReflect::typeFromType(type);
            if (orig && orig != type)
            {
                ItemFromType::Iterator it = m_itemFromType.findOrInsertKey(orig, m_written.getSize());
                id = m_itemFromType.getValue(it);
            }
        }

        // Add the type to the map.
        m_itemFromType.insert(type, id);
    }

    if (id == m_written.getSize())
    {
        // A unique type has been added to the map, add a new entry.
        m_written.pushBack(false);
    }

    return id;
}

hkSerialize::TypeId hkSerialize::Detail::TypeWriterMap::enqueueForWrite(_In_ const hkReflect::Type* type)
{
    TypeId id = lookupType(type);
    if (m_written[id] == false)
    {
        m_written[id] = true;
        m_writeQueue.pushBack(s_remapOpaque(type));
    }
    return id;
}

_Ret_maybenull_ const hkReflect::Type* hkSerialize::Detail::TypeWriterMap::nextToWrite()
{
    if (m_nextToWrite == m_writeQueue.getSize())
    {
        m_writeQueue.clear();
        m_nextToWrite = 0;
        return HK_NULL;
    }

    return m_writeQueue[m_nextToWrite++];
}


hkSerialize::Detail::FileFormatRegNode::FileFormatRegNode(_In_z_ const char* name, ReadFormatDetectFunc detectFunc, ReadFormatCreateFunc createRead, WriteFormatCreateFunc createWrite)
    : m_name(name)
    , m_detectFunc(detectFunc)
    , m_readFormatCreateFunc(createRead)
    , m_writeFormatCreateFunc(createWrite)
{
}

static hkSerialize::Detail::FileFormatRegNode const * s_readFormats[] =
{
    &hkSerialize::Detail::fileFormatBinaryTagfile,
    &hkSerialize::Detail::fileFormatBinaryTagfile2015,
    &hkSerialize::Detail::fileFormatBinaryTagfileNT6,
    &hkSerialize::Detail::fileFormatBinaryTagfile2014,
    &hkSerialize::Detail::fileFormatBinaryPackfile2014,
    &hkSerialize::Detail::fileFormatXmlTagfile,
    &hkSerialize::Detail::fileFormatXmlTagfile2014,
    &hkSerialize::Detail::fileFormatXmlPackfile2010,
    &hkSerialize::Detail::fileFormatYamlfile,
    &hkSerialize::Detail::fileFormatCompressedTagfile,
};

hkRefNew<hkSerialize::ReadFormat> hkSerialize::Detail::createReadFormat(_In_reads_bytes_(bufSize) const void* buf, hkUlong bufSize)
{
    for (int i = 0; i < HK_COUNT_OF(s_readFormats); ++i)
    {
        const FileFormatRegNode* reg = s_readFormats[i];
        if (reg->detect(buf, bufSize))
        {
            return reg->createReadFormat();
        }
    }
    return HK_NULL;
}

namespace
{
    struct BinaryPackfile2014
    {
        enum
        {
            // both endian agnostic (byte symmetric)
            MAGIC_0 = 0x57e0e057,
            MAGIC_1 = 0x10c0c010,
        };

        /// Magic file identifier. See constructor for values.
        hkInt32 m_magic[2];

        /// This is a user settable tag.
        hkInt32 m_userTag;

        /// Binary file version. Currently 11.
        hkInt32 m_fileVersion;
    };


    static _Ret_z_ const char* skipSpaces(_In_z_ const char* text)
    {
        while (*text && *text == ' ')
        {
            ++text;
        }
        return text;
    }

}

static bool detectBinaryPackfile2014(_In_reads_bytes_(bufSize) const void* buf, hkUlong bufSize)
{
    if (bufSize >= sizeof(BinaryPackfile2014))
    {
        const BinaryPackfile2014* packfileLoaded = reinterpret_cast<const BinaryPackfile2014*>(buf);
        if ((packfileLoaded->m_magic[0] == BinaryPackfile2014::MAGIC_0)
            && (packfileLoaded->m_magic[1] == BinaryPackfile2014::MAGIC_1))
        {
            return true;
        }
    }
    return false;
}

static bool detectXmlPackfile2014(_In_reads_bytes_(bufSize) const void* vbuf, hkUlong bufSize)
{
    hkStringView buf(static_cast<const char*>(vbuf), bufSize);
    if (buf.strStr("<hkpackfile ") || buf.strStr("<hkpackfile>"))
    {
        return true;
    }
    return false;
}

static bool detectXmlTagfile(_In_reads_bytes_(bufSize) const void* vbuf, hkUlong bufSize)
{
    hkIo::ReadBuffer sr(vbuf, bufSize);
    hkXmlStreamParser parser(sr);

    hkXmlStreamParser::Token token = parser.advance();
    while (token != hkXmlStreamParser::TOKEN_BLOCK_START &&
        token != hkXmlStreamParser::TOKEN_EOF &&
        token != hkXmlStreamParser::TOKEN_ERROR)
    {
        token = parser.advance();
    }

    if (token == hkXmlStreamParser::TOKEN_BLOCK_START)
    {
        if (parser.getBlockName() == "hktagfile")
        {
            int version;
            if (parser.getIntAttribute("version", version).isSuccess() && version == 3)
            {
                return true;
            }
        }
    }

    return false;
}

static bool detectXmlTagfile2014(_In_reads_bytes_(bufSize) const void* vbuf, hkUlong bufSize)
{
    hkIo::ReadBuffer sr(vbuf, bufSize);
    hkXmlStreamParser parser(sr);

    hkXmlStreamParser::Token token = parser.advance();
    while (token != hkXmlStreamParser::TOKEN_BLOCK_START &&
        token != hkXmlStreamParser::TOKEN_EOF &&
        token != hkXmlStreamParser::TOKEN_ERROR)
    {
        token = parser.advance();
    }

    if (token == hkXmlStreamParser::TOKEN_BLOCK_START)
    {
        if (parser.getBlockName() == "hktagfile")
        {
            int version;
            if (!parser.getIntAttribute("version", version).isSuccess() || version < 3)
            {
                return true;
            }
        }
    }
    return false;
}

static bool detectBinaryTagfile(_In_reads_bytes_(bufSize) const void* buf, hkUlong bufSize)
{
    if (bufSize >= 8)
    {
        const char* cbuf = reinterpret_cast<const char*>(buf);
        if (hkString::memCmp(cbuf + 4, "TAG0", 4) == 0) return true;
        if (hkString::memCmp(cbuf + 4, "TCM0", 4) == 0) return true;
    }
    return false;
}

static bool detectBinaryTagfile2015(_In_reads_bytes_(bufSize) const void* buf, hkUlong bufSize)
{
    if (bufSize >= 8)
    {
        const char* cbuf = reinterpret_cast<const char*>(buf);
        if (hkString::memCmp(cbuf + 4, "TAGF", 4) == 0) return true;
        if (hkString::memCmp(cbuf + 4, "TCMP", 4) == 0) return true;
    }
    return false;
}

static bool detectBinaryTagfileNT6(_In_reads_bytes_(bufSize) const void* buf, hkUlong bufSize)
{
    const hkUint32 BINARY_MAGIC_0 = 0xCAB00D1E;
    const hkUint32 BINARY_MAGIC_1 = 0xDEADC0DE;
    if (bufSize >= 8)
    {
        const hkUint32Le *const number = reinterpret_cast<const hkUint32Le*>(buf);
        if ((static_cast<hkUint32>(number[0]) == BINARY_MAGIC_0) && (static_cast<hkUint32>(number[1]) == BINARY_MAGIC_1))
        {
            return true;
        }
    }
    return false;
}

static bool HK_CALL detectBinaryTagfile2014(_In_reads_bytes_(bufSize) const void* buf, hkUlong bufSize)
{
    const hkUint32* tagfileLoaded = reinterpret_cast<const hkUint32*>(buf);
    if ((bufSize >= 8) && hkBinaryTagfile2014::isBinaryMagic(tagfileLoaded[0], tagfileLoaded[1]))
    {
        return true;
    }
    return false;
}

static bool detectYamlFile(_In_reads_bytes_(bufSize) const void* vbuf, hkUlong bufSize)
{
    hkStringView buf(static_cast<const char*>(vbuf), bufSize);

    if (buf.beginsWith("---"))
    {
        return true;
    }
    return false;
}

static bool detectCompressedTagfile(_In_reads_bytes_(bufSize) const void* vbuf, hkUlong bufSize)
{
    hkStringView buf(static_cast<const char*>(vbuf), bufSize);

    if (buf.beginsWith("<compr>"))
    {
        return true;
    }
    return false;
}

// m_createFunc is null by default, initialized as a hkOptionalComponent
hkSerialize::Detail::FileFormatRegNode hkSerialize::Detail::fileFormatBinaryTagfile("Binary Tagfile >=2015.1", detectBinaryTagfile, HK_NULL, HK_NULL);
hkSerialize::Detail::FileFormatRegNode hkSerialize::Detail::fileFormatBinaryTagfile2015("Binary Tagfile pre 2015.1", detectBinaryTagfile2015, HK_NULL, HK_NULL);
hkSerialize::Detail::FileFormatRegNode hkSerialize::Detail::fileFormatBinaryTagfileNT6("Binary Tagfile NT6", detectBinaryTagfileNT6, HK_NULL, HK_NULL);
hkSerialize::Detail::FileFormatRegNode hkSerialize::Detail::fileFormatBinaryTagfile2014("Binary Tagfile <=2014.2", detectBinaryTagfile2014, HK_NULL, HK_NULL);
hkSerialize::Detail::FileFormatRegNode hkSerialize::Detail::fileFormatBinaryPackfile2014("Binary Packfile <=2014.2", detectBinaryPackfile2014, HK_NULL, HK_NULL);
hkSerialize::Detail::FileFormatRegNode hkSerialize::Detail::fileFormatXmlTagfile("Xml Tagfile >=2015.1", detectXmlTagfile, HK_NULL, HK_NULL);
hkSerialize::Detail::FileFormatRegNode hkSerialize::Detail::fileFormatXmlTagfile2014("Xml Tagfile <=2014.2", detectXmlTagfile2014, HK_NULL, HK_NULL);
hkSerialize::Detail::FileFormatRegNode hkSerialize::Detail::fileFormatXmlPackfile2010("Xml Packfile <=2014.2", detectXmlPackfile2014, HK_NULL, HK_NULL);
hkSerialize::Detail::FileFormatRegNode hkSerialize::Detail::fileFormatYamlfile("Yaml File", detectYamlFile, HK_NULL, HK_NULL);
hkSerialize::Detail::FileFormatRegNode hkSerialize::Detail::fileFormatCompressedTagfile("Compressed Tagfile", detectCompressedTagfile, HK_NULL, HK_NULL);

_Ret_maybenull_ const hkSerialize::Detail::FileFormatRegNode* hkSerialize::Detail::detectFormat(_In_reads_bytes_(bufSize) const void* buf, hkUlong bufSize)
{
    for (int i = 0; i < HK_COUNT_OF(s_readFormats); ++i)
    {
        const FileFormatRegNode* reg = s_readFormats[i];
        if (reg->detect(buf, bufSize))
        {
            return reg;
        }
    }
    return HK_NULL;
}

hkSerialize::Detail::CloneToRegistered::CloneToRegistered(_In_opt_ const hkReflect::TypeReg* typeReg)
    : m_typeReg(typeReg ? typeReg : hkReflect::getTypeReg())
{
}

_Ret_maybenull_ const hkReflect::Type* hkSerialize::Detail::CloneToRegistered::typeFromType(_In_ const hkReflect::Type* type)
{
    if (const hkReflect::Type* regType = m_typeReg->typeFromType(type))
    {
#if defined(HK_BUILDING_WITH_ENGINE)
        if (const char* attributes = hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::ATTRIBUTE_STRING>(type))
        {
            return hkDynamicTypeManager::getInstance().getAttributedTypeOf(regType, attributes);
        }
#endif

        return regType;
    }
    const hkReflect::Type* t;
    return s_typeNotFound(type, m_typeReg, &t).isSuccess() ? t : nullptr;
}

hkSerialize::Detail::OwnedResource::OwnedResource()
    : hkResource(), m_destructorsCalled(false)
{
}
hkSerialize::Detail::OwnedResource::OwnedResource(const hkArrayView<hkReflect::Var>& objects)
    : hkResource(), m_destructorsCalled(false)
{
    for (int i = 0; i < objects.getSize(); ++i)
    {
        addObject(objects[i]);
    }
}

hkSerialize::Detail::OwnedResource::~OwnedResource()
{
    callDestructors();
    clearContents();
}

void hkSerialize::Detail::OwnedResource::getImportsExports(hkArray<hkResource::Import>& impOut, hkArray<hkResource::Export>& expOut) const
{
    impOut.append(m_imports);
    expOut.append(m_exports);
}

void hkSerialize::Detail::OwnedResource::addExport(const hkReflect::Var& var, _In_z_ const char* name)
{
    m_exports.expandOne().set(var, name);
}

void hkSerialize::Detail::OwnedResource::addImport(const hkReflect::PointerVar& pvar, _In_z_ const char* name)
{
    m_imports.expandOne().set(pvar, name);
}

hkReflect::Var hkSerialize::Detail::OwnedResource::getContents() const
{
    if (m_contents.getSize() > 0)
    {
        return m_contents[0];
    }
    return hkReflect::Var();
}

void hkSerialize::Detail::OwnedResource::cleanup()
{
    m_imports.clearAndDeallocate();
    m_exports.clearAndDeallocate();
}

void hkSerialize::Detail::OwnedResource::callDestructors()
{
    if (!m_destructorsCalled)
    {
        for (int i = 0; i < m_contents.getSize(); i++)
        {
            if (m_contents[i].getType() && m_contents[i].getAddress())
            {
                // We use extendsOrEquals as the object itself may already be dead
                if (m_contents[i].getType()->extendsOrEquals<hkReferencedObject>())
                {

                    // If non refcounted, the resource owns every object
                    hkReferencedObject* ro = const_cast<hkReferencedObject*>(reinterpret_cast<const hkReferencedObject*>(m_contents[i].getAddress()));

                    // Map of objects that are not their natural size
                    if (!m_actualObjectSizes.hasKey(ro))
                    {
                        m_actualObjectSizes.insert(ro, m_contents[i].getType()->getSizeOf());
                    }
                    // If we are reference counted (loaded on heap), do nothing, as the top
                    // level object should own these references. Parent destructor will remove
                    // the top level object
                    m_contents[i].destroy(hkReflect::Var::FLAG_DESTRUCT);
                }
                else
                {
                    // Deallocate the object directly
                    m_contents[i].destroy(hkReflect::Var::FLAG_DESTRUCT);
                }
            }
        }
        m_destructorsCalled = true;
    }
}

void hkSerialize::Detail::OwnedResource::clearContents()
{
    // Destructors have already been called, this is just freeing storage
    for (int i = 0; i < m_contents.getSize(); i++)
    {
        if (const hkReflect::Type* objType = m_contents[i].getType())
        {
            if (void* addr = m_contents[i].getAddress())
            {
                int sizeToFree = m_actualObjectSizes.getWithDefault(addr, -1);
                if (sizeToFree == -1)
                {
                    sizeToFree = objType->getSizeOf();
                }

                HK_MEMORY_TRACKER_REMOVE(addr);
                hkMemHeapBlockFree(addr, sizeToFree);
            }
        }
    }
    m_contents.clear();
}

void hkSerialize::Detail::OwnedResource::addObject(hkReflect::Var obj)
{
    m_contents.pushBack(obj);

    if (obj.getType())
    {
        // We use extendsOrEquals as the object itself may already be dead
        if (obj.getType()->extendsOrEquals<hkReferencedObject>())
        {
            hkReferencedObject* ro = const_cast<hkReferencedObject*>(reinterpret_cast<const hkReferencedObject*>(obj.getAddress()));
            // If any object has explicitly set its size, we need to remember
            // so we can free it correctly. We then need to set memSizeAndFlags to 0 to get
            // correct reference count behavior
            if ((ro->getMemorySizeAndFlags() != 0) && (ro->getMemorySizeAndFlags() != hkReferencedObject::AUTO_MEMSIZE))
            {
                m_actualObjectSizes.insert(ro, ro->getMemorySizeAndFlags());
            }
            ro->setMemorySizeFlagsAndReferenceCount(0, 0);
        }
    }
}

_Ret_opt_z_ const char* hkSerialize::Detail::OwnedResource::getName() const
{
    return HK_NULL;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
