// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Types/hkScopedPtr.h>

class hkStreamReader;

namespace hkIo
{
    struct HK_EXPORT_COMMON ReadBufferStatus
    {
        ReadBufferStatus()
        {
            reset();
        }

        void reset()
        {
            m_flags.clear();
        }

        bool isOk() const { return m_flags.get() == 0; }

        void setEnd() { m_flags.orWith(STATUS_END); }
        void setUnexpected() { m_flags.orWith(STATUS_UNEXPECTED); }
        void setDefective() { m_flags.orWith(STATUS_DEFECTIVE); }

    private:

        enum ReadBufferStatusBits
        {
            STATUS_END = (1 << 0), // End of buffer reached
            STATUS_UNEXPECTED = (1 << 1), // Encountered inconsistent data
            STATUS_DEFECTIVE = (1 << 2), // Low level IO error
        };

        hkFlags<ReadBufferStatus::ReadBufferStatusBits, hkUint8> m_flags;
    };

    /// Interface to a fully in-memory read buffer.
    class ReadBufferView
    {
    public:

        HK_DECLARE_CLASS(ReadBufferView, NewPlacement);

        ReadBufferView() : m_cur(HK_NULL), m_end(HK_NULL), m_start(HK_NULL) { }
        ReadBufferView(_In_ const void* c, _In_ const void* e)
        {
            m_cur = c;
            m_end = e;
            m_start = c;
        }
        ReadBufferView(_In_reads_bytes_(len) const void* c, hkLong len)
        {
            m_cur = c;
            m_end = hkAddByteOffset(c, len);
            m_start = c;
        }

        inline bool isOk() const
        {
            return m_status.isOk();
        }

        /// Memcpy a DST from the stream.
        template<typename DST>
        DST readPod()
        {
            DST d = 0;
            const void* c = hkAddByteOffset(m_cur, sizeof(DST));
            if (c <= m_end)
            {
                hkMemUtil::memCpy<1>(&d, m_cur, sizeof(DST));
                m_cur = c;
            }
            else
            {
                m_cur = m_end;
                m_status.setEnd();
                m_status.setUnexpected();
            }
            return d;
        }

        /// Try to make n bytes available for reading.
        /// Returns the actual number of bytes available (which may be greater than the number requested)
        hkLong prefetch(hkLong n) const
        {
            return hkGetByteOffset(m_cur, m_end);
        }

        /// Skip ahead n bytes.
        void skip(hkLong n)
        {
            const void* c = hkAddByteOffset(m_cur, n);
            if (c <= m_end)
            {
                m_cur = c;
            }
            else
            {
                m_cur = m_end;
                m_status.setEnd();
            }
        }

        /// Peek into the buffer at an optional offset without advancing the read position.
        template<typename DST>
        _Ret_notnull_ const DST* peekAt(hkLong off = 0, hkLong count = 1) const
        {
            const void* s = hkAddByteOffset(m_cur, off);
            HK_ASSERT(0x0990e92b, hkAddByteOffset(s, count * hkSizeOfTypeOrVoid<DST>::val) <= m_end, "peekAt({}, {}) going past the end of ReadBufferView, ensure prefetch() was called and its return value checked", off, count);
            return static_cast<const DST*>(s);
        }

        /// Access count items of type DST at the current offset and advance the read position.
        template<typename DST>
        _Ret_notnull_ const DST* access(hkLong count = 1)
        {
            const DST* r = peekAt<DST>(0, count);
            skip(count * hkSizeOfTypeOrVoid<DST>::val);
            return r;
        }

        template<typename DST>
        HK_ALWAYS_INLINE _Ret_notnull_ DST* accessRW(hkLong count = 1)
        {
            return const_cast<DST*>(access<DST>(count));
        }

        /// Return the current read position.
        inline hkLong tell() const
        {
            return hkGetByteOffset(m_start, m_cur);
        }

        /// Number of bytes remaining in the buffer.
        inline hkLong remaining() const
        {
            return hkGetByteOffset(m_cur, m_end);
        }

        inline _Ret_maybenull_ const void* begin() const { return m_start; }
        inline _Notvalid_ const void* end() const { return m_end; }

        void init(_In_ const void* c, _Notvalid_ const void* e)
        {
            m_status.reset();
            m_cur = c; m_end = e; m_start = c;
        }
        void init(_In_reads_bytes_(len) const void* c, hkLong len)
        {
            m_status.reset();
            m_cur = c; m_end = hkAddByteOffset(c, len); m_start = c;
        }

        ReadBufferStatus& status() { return m_status; }
        const ReadBufferStatus& status() const { return m_status; }

    protected:

        const void* m_cur; // Read position in the buffer.
        const void* m_end; // One past the last byte in the buffer.
        const void* m_start; // Start of the buffer.

        ReadBufferStatus m_status;
    };

    namespace Detail
    {
        struct HK_EXPORT_COMMON ReadBufferState
        {
            ReadBufferState() { clear(); }
            void clear() { m_cur = m_end = m_start = m_rewind = HK_NULL; m_offsetAtStart = 0; }
            const void* m_cur; // Read position in the buffer.
            const void* m_end; // One past the last byte in the buffer.
            const void* m_start; // Start of the buffer.
            const void* m_rewind; // Rewind point of the buffer, if set it will prevent data after that point from being freed
            ReadBufferStatus m_status;
            hkLong remaining() const { return hkGetByteOffset(m_cur, m_end); }
            hkLong m_offsetAtStart; // Offset of stream at m_start (not offset at m_cur so we don't have to bump on every read)
        };

        struct HK_EXPORT_COMMON ReadBufferImpl
        {
            HK_DECLARE_CLASS(ReadBufferImpl, New);
            virtual ~ReadBufferImpl() {}

            virtual void _attachTo(ReadBufferState& rs) = 0;
            virtual void _detachFrom(ReadBufferState& rs) = 0;

            /// Request that at least "n" bytes are available in the buffered
            /// part. The return value should be the actual number of bytes
            /// available, which may be more than or equal to the number of
            /// requested bytes, or fewer if we've reached the end of the stream.
            ///
            /// Note that the passed in ReadBufferState may still contain some
            /// buffered data, in which case this existing data should be kept
            /// as part of the new buffer. In this case, the return value should
            /// include the number of bytes which already were in the buffer as well.
            /// If the rewind position is set, buffered data should be kept from the rewind
            /// position, instead of the from the current position (rewind should be < cur)
            virtual hkLong _prefetch(ReadBufferState& rs, hkLong n) = 0;

            /// Request the entire stream be read into a contiguous block.
            virtual hkLong _prefetchAll(ReadBufferState& rs) = 0;
            /// Skip n bytes of input.
            virtual hkLong _skip(ReadBufferState& rs, hkLong n) = 0;

            /// Tries to read n bytes into buf and returns the number of bytes
            /// read. The actual number of bytes read can be less than the
            /// requested number if we've reached the end of the stream.
            ///
            /// When this function is called, there may still be some data
            /// remaining in the ReadBufferState passed in, in which case this
            /// data should be used before the newly read data. Implementations
            /// can assume that the amount of data remaining in this buffer is
            /// less than the requested amount. This function is allowed (and
            /// encourage) to read more than the requested amount of data and
            /// leave the rest in the ReadBufferState.
            virtual hkLong _read(ReadBufferState& rs, _Out_writes_bytes_(n) void* buf, hkLong n) = 0;
            /// Set a virtual end-of-file at the given absolute offset.
            virtual hkResult _setReadLimit(ReadBufferState& rs, hkLong n) = 0;
            /// Get a virtual end-of-file offset or -1 if not set.
            virtual hkLong _getReadLimit(const ReadBufferState& rs) const = 0;

            /// Rewind read to a particular offset off the read pointer
            virtual hkResult _rewind(ReadBufferState& rs, hkLong offset);
        };

        HK_EXPORT_COMMON _Ret_maybenull_ ReadBufferImpl* HK_CALL createReaderImpl(_In_z_ const char* filename);
        HK_EXPORT_COMMON _Ret_notnull_ ReadBufferImpl* HK_CALL createReaderImpl(_Inout_ hkStreamReader* reader);

        HK_ALWAYS_INLINE _Ret_notnull_ ReadBufferImpl* createReaderImpl(_In_ ReadBufferImpl* impl)
        {
            return impl;
        }
            /// Helper class to provide a backend which can be used by a ReadBuffer.
            /// This class is intended to be constructed implicitly by passing the address of an adaptable output object.
            /// Creating one of these explicitly is almost certainly an error.
            /// E.g. void func(const Detail::ReadBufferAdapter& s); hkArray<char> buf; func(&buf);
            /// See overloads of Detail::createReaderImpl for the types which are adaptable.
        class HK_EXPORT_COMMON ReadBufferAdapter
        {
            public:
                /// Note that smart pointers may need to be explicitly dereferenced for templates to be deduced correctly.
                template<typename T> ReadBufferAdapter(const T& t) { set(t); }
                ReadBufferAdapter(_In_bytecount_(l) const void* p, hkLong l) { set(p, l); }

                template<typename T> void set(hkArrayView<T> arr) { set(arr.begin(), arr.getSize() * sizeof(T)); }
                template<typename T> void set(const hkArray<T>& arr) { set(arr.begin(), arr.getSize() * sizeof(T)); }

                template<typename T>
                HK_INLINE void set(const T& t)
                {
                    m_impl.reset(createReaderImpl(t));
                    m_data = 0;
                    m_dataSize = 0;
                }

                HK_INLINE void set(_In_reads_bytes_(l) const void* b, hkLong l)
                {
                    m_impl.reset();
                    m_data = b;
                    m_dataSize = l;
                }

                _Ret_maybenull_ ReadBufferImpl* steal() const { return m_impl.steal(); }
                HK_ALWAYS_INLINE void clear() { m_impl.reset(); }

                /// Returns whether this ReadBufferAdapter has a non-null
                /// implementation. If false, then it must hold a memory block
                /// instead (ie. getData() and getdataSize() return valid values.
                bool hasNonNullImpl() const { return m_impl; }

                _Ret_maybenull_ const void* getData() const { return m_data; }
                hkLong getDataSize() const { return m_dataSize; }

            protected:
                /// Each instance either has a non-null m_impl or a non-null
                /// m_data and m_dataSize member.
                mutable hkScopedPtr<ReadBufferImpl> m_impl;
                const void* m_data;
                hkLong m_dataSize;
        };
    };


        /// Interface to a windowed read buffer.
    class HK_EXPORT_COMMON ReadBuffer
    {
        public:

            HK_DECLARE_CLASS(ReadBuffer, NewPlacement, NonCopyable);

                ///
            ReadBuffer() : m_attached(false) {}

                /// Create from a memory region.
            ReadBuffer(_In_reads_bytes_(len) const void* buf, hkLong len);

                ///
            ReadBuffer(const Detail::ReadBufferAdapter& source) : m_attached(false) { attach(source); }

            ~ReadBuffer();

                /// Attach to an impl
            void attach(const Detail::ReadBufferAdapter& source);
                ///
            void attach(_In_reads_bytes_(len) const void* buf, hkLong len);
                /// Detach from an impl.
            void detach();
                /// Is this read buffer attached to a source?
            bool isAttached() const { return m_attached; }

            template<typename DST>
            DST readPod()
            {
                DST d = 0;
                read(&d, sizeof(DST));
                return d;
            }

                /// Try to make n bytes available for reading.
                /// Returns the actual number of bytes available (which may be greater than the number requested)
            hkLong prefetch(hkLong n)
            {
                HK_ASSERT(0x3ad0f0a1, m_attached, "Can't prefetch from an unattached ReadBuffer.");

                hkLong avail = hkGetByteOffset(m_state.m_cur, m_state.m_end);
                if( avail >= n || !m_impl)
                {
                    return avail;
                }
                else
                {
                    return implPrefetch(n);
                }
            }

                /// Prefetch the entire contents.
            hkLong prefetchAll();

                /// Skip ahead n bytes.
            hkLong skip(hkLong n)
            {
                HK_ASSERT(0x63056df0, m_attached, "Can't skip using an unattached ReadBuffer.");

                const void* e = hkAddByteOffset(m_state.m_cur, n);
                if(e <= m_state.m_end)
                {
                    m_state.m_cur = e;
                    return n;
                }
                else
                {
                    return implSkip(n);
                }
            }

                /// Peek into the buffer at an optional offset without advancing the read position.
                /// The data MUST already be in the buffer before calling this method. I.e. use prefetch() before peeking.
            template<typename DST>
            _Ret_notnull_ const DST* peekAt(hkLong off=0, hkLong count=1) const
            {
                HK_ASSERT(0x7e3b7398, m_attached, "Can't peekAt using an unattached ReadBuffer.");
                HK_ASSERT(0x7a69d195, (m_state.remaining() - off) >= (count * hkSizeOfTypeOrVoid<DST>::val),
                    "peekAt called with an offset and count which fall outside the current buffer. "
                    "Use prefetch to make sure the buffer holds enough data.");
                const void* s = hkAddByteOffset(m_state.m_cur, off);
                return static_cast<const DST*>(s);
            }

                /// Return the current read position.
            hkLong tell() const
            {
                HK_ASSERT(0x32a83d15, m_attached, "Can't tell() using an unattached ReadBuffer.");
                return m_state.m_offsetAtStart + hkGetByteOffset(m_state.m_start, m_state.m_cur);
            }

                /// Read "len" bytes into buf.
                /// Return the actual number of bytes read.
            hkLong read(_Out_writes_bytes_(len) void* buf, hkLong len)
            {
                HK_ASSERT(0x94a4050, m_attached, "Can't read from an unattached ReadBuffer.");

                hkLong avail = hkGetByteOffset(m_state.m_cur, m_state.m_end);
                if( avail >= len )
                {
                    hkMemUtil::memCpy(buf, m_state.m_cur, len);
                    m_state.m_cur = hkAddByteOffset(m_state.m_cur, len);
                    return len;
                }
                else
                {
                    return implRead(buf, len);
                }
            }

                /// Set current position as rewind position, buffered data wont be dropped after the rewind position,
                /// even when data is read so that we can rewind back to that position. This will force the buffer to
                /// grow to accomodate the needed buffered, so be sure to clear the rewind position as soon as possible.
                /// Note that, in the current implementation, rewind points do not nest.
            void setRewindPosition()
            {
                HK_ASSERT(0x28c2f6e0, m_attached, "Can't rewind using an unattached ReadBuffer.");
                HK_ASSERT(0x2e4edc51, m_state.m_rewind == HK_NULL, "Rewind position is already set");

                m_state.m_rewind = m_state.m_cur;
            }

                /// Clear rewind position, allowing back the buffering to drop data until the current position
                /// even when data is read so that we rewind back to that position.
            void clearRewindPosition()
            {
                HK_ASSERT(0x6fb46e8e, m_attached, "Can't rewind using an unattached ReadBuffer.");

                m_state.m_rewind = HK_NULL;
            }

            /// Rewind to the given offset past the rewind position, offset must be in the prefetched area
            hkResult rewindTo(hkLong offset)
            {
                HK_ASSERT(0x3481460e, m_attached, "Can't rewind using an unattached ReadBuffer.");
                HK_ASSERT(0x5ee925d7, m_state.m_rewind, "No rewind point, has setSeekPosition been called");
                HK_ASSERT(0x7916eafd, offset >= 0, "Seeking does not support negative offsets");

                const void* offsetPtr = hkAddByteOffset(m_state.m_rewind, offset);
                if (m_state.m_start <= offsetPtr && offsetPtr <= m_state.m_end)
                {
                    m_state.m_cur = hkAddByteOffset(m_state.m_rewind, offset);
                    return HK_SUCCESS;
                }
                else
                {
                    return implRewind(offset);
                }
            }

                /// Used by getReadLimit/setReadLimit
            enum { READ_UNLIMITED=-1 };
                /// Don't read past absolute offset "n".
                /// Disable the limit by passing READ_UNLIMITED.
                /// It is an error if the limit has already been reached.
            hkResult setReadLimit(hkLong n);
                /// Get the readlimit or READ_UNLIMITED if not set.
            hkLong getReadLimit() const;

            hkUint8 read8u() { return readPod<hkUint8Le>(); }
            hkUint32 read32u() { return readPod<hkUint32Le>(); }
            hkUint64 read64u() { return readPod<hkUint64Le>(); }
            hkInt32 read32i() { return readPod<hkInt32Le>(); }
            hkInt64 read64i() { return readPod<hkInt64Le>(); }

            ReadBufferStatus& status() { return m_state.m_status; }
            const ReadBufferStatus& status() const { return m_state.m_status; }

        protected:

            hkLong implPrefetch(hkLong n);
            hkLong implSkip(hkLong n);
            hkLong implRead(_Out_writes_bytes_(n) void* buf, hkLong n);
            hkResult implRewind(hkLong offset);

            Detail::ReadBufferState m_state;
            hkBool m_attached;
            hkScopedPtr<Detail::ReadBufferImpl> m_impl;
    };
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
