// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>

#define DEBUG_LOG_IDENTIFIER "io.ReadBuffer"
#include <Common/Base/System/Log/hkLog.hxx>

namespace hkIo { namespace Detail
{
    enum
    {
        SENSIBLE_PREFETCH_SIZE = 64 * 1024
    };

    hkResult ReadBufferImpl::_rewind(ReadBufferState& rs, hkLong offset)
    {
        /// No rewind implement by default
        HK_ASSERT(0x1b76410c, 0, "No rewind implemented by default");
        return HK_FAILURE;
    }

    struct StreamReaderImpl : public ReadBufferImpl
    {
        HK_DECLARE_CLASS(StreamReaderImpl, New);

            // Loop several times to handle read() returning less than "len"
            // Note that the readLimit handling is done by the caller!
        inline hkLong _streamRead(_Out_writes_bytes_(len) void* buf, hkLong len)
        {
            hkLong left = len;
            while(left)
            {
                hkLong r = m_stream->read(buf, hkLosslessCast<int>(left));
                if(r > 0)
                {
                    buf = hkAddByteOffset(buf, r);
                    left -= r;
                }
                else
                {
                    return len - left;
                }
            }
            return len;
        }

        virtual void _attachTo(ReadBufferState& rs) HK_OVERRIDE
        {
            rs.m_offsetAtStart = 0;

            if(hkSeekableStreamReader* sr = m_stream->isSeekTellSupported())
            {
                rs.m_offsetAtStart = sr->tell();
            }
        }

        virtual void _detachFrom(ReadBufferState& rs) HK_OVERRIDE
        {
            if (hkLong rem = rs.remaining())
            {
                if(hkSeekableStreamReader* sr = m_stream->isSeekTellSupported())
                {
                    sr->seek(hkLosslessCast<int>(-rem), hkSeekableStreamReader::STREAM_CUR);
                }
                else
                {
                    rs.m_status.setDefective();
                    Log_Warning("Could not detach cleanly from stream because it is not seekable.");
                }
            }

            rs.clear();
        }

        virtual hkLong _prefetch(ReadBufferState& rs, hkLong n) HK_OVERRIDE
        {
            // Note that we don't necessarily read exactly "n" bytes since we may have some in the buffer already

            // if rewind is set, we want to keep data from rewind instead of cur
            HK_ASSERT(0x1227bf91, !rs.m_rewind || (rs.m_rewind <= rs.m_cur && rs.m_rewind >= rs.m_start), "ReadBufferState is in an invalid state");
            const void* keep = rs.m_rewind ? rs.m_rewind : rs.m_cur;
            const hkLong prefix = hkGetByteOffset(keep, rs.m_cur);

            // move the bit to keep to the start of the buffer
            const hkLong have = hkGetByteOffset(keep, rs.m_end);
            if(rs.m_start != keep)
            {
                rs.m_offsetAtStart += hkGetByteOffset(rs.m_start, keep);
                hkMemUtil::memMove(const_cast<void*>(rs.m_start), keep, have);
            }

            // want == How many we want in the buffer at the end.
            hkLong want = prefix + n;
            if( m_readLimit!=ReadBuffer::READ_UNLIMITED) // we have a readlimit, it's safe to prefetch a bit more than requested
            {
                want = hkMath::max2( want, (hkLong)SENSIBLE_PREFETCH_SIZE ); // maybe take a bit more
                want = hkMath::min2( want, m_readLimit - rs.m_offsetAtStart ); // but not too much
            }
            // Expand buffer if needed
            if(want > m_buf.getSize())
            {
                m_buf.setSize( hkLosslessCast<int>(want) );
                rs.m_start = m_buf.begin();
                HK_ASSERT(0x29b8e65f, rs.m_start, "Failed to allocate buffer");
            }

            // now do the read
            hkLong nr = _streamRead(m_buf.begin() + have, hkLosslessCast<int>(want - have));

            if (!nr)
            {
                rs.m_status.setEnd();
            }

            rs.m_cur = m_buf.begin() + prefix;
            rs.m_end = m_buf.begin() + have + nr;
            if (rs.m_rewind != nullptr)
            {
                rs.m_rewind = m_buf.begin();
            }

            HK_ASSERT_EXPR(0x40925bfa, m_readLimit, ==, hkLong(ReadBuffer::READ_UNLIMITED), ||, rs.m_offsetAtStart + rs.remaining(), <=, m_readLimit);

            return hkGetByteOffset(rs.m_cur, rs.m_end);
        }

        virtual hkLong _prefetchAll(hkIo::Detail::ReadBufferState& rs) HK_OVERRIDE
        {
            hkLong available = 0;

            if(hkSeekableStreamReader* sr = m_stream->isSeekTellSupported())
            {
                int pos = sr->tell();
                sr->seek(0, hkSeekableStreamReader::STREAM_END);
                int size = sr->tell();
                sr->seek(pos, hkSeekableStreamReader::STREAM_SET);
                available = _prefetch(rs, rs.remaining() + size - pos);
            }
            else // grow
            {
                if (m_buf.getCapacity() == 0)
                {
                    m_buf.reserve(SENSIBLE_PREFETCH_SIZE);
                }

                for(hkLong n = m_buf.getCapacity(); true; n *= 2)
                {
                    available = _prefetch(rs, n);
                    if(available < n)
                    {
                        break;
                    }
                }
            }

            rs.m_status.setEnd();
            return available;
        }

        virtual hkLong _skip(ReadBufferState& rs, hkLong n) HK_OVERRIDE
        {
            if(n <= rs.remaining())
            {
                HK_ASSERT_NO_MSG(0x2695df21, false); // shouldn't happen. be paranoid
                rs.m_cur = hkAddByteOffset(rs.m_cur, n);
                return n;
            }
            if(rs.m_rewind) // we have a rewind point, we actually need to read the data into the buffer
            {
                hkLong skipped = hkMath::min2(n, _prefetch(rs, n));
                rs.m_cur = hkAddByteOffset(rs.m_cur, skipped);

                if (skipped < n)
                {
                    rs.m_status.setEnd();
                }

                return skipped;
            }

            // skip the ones in the buffer
            hkLong bufSkip = rs.remaining();
            rs.m_cur = rs.m_end;

            // forward the rest to stream.skip
            hkLong wantSkip = n - bufSkip;
            hkLong reqStreamSkip = wantSkip;
            if(m_readLimit != ReadBuffer::READ_UNLIMITED)
            {
                hkLong curStreamPos = rs.m_offsetAtStart + hkGetByteOffset(rs.m_start, rs.m_end);
                HK_ASSERT_NO_MSG(0x1f509a56, curStreamPos <= m_readLimit );
                reqStreamSkip = hkMath::min2(m_readLimit - curStreamPos, wantSkip);
            }

            hkLong actualStreamSkip = m_stream->skip(hkLosslessCast<int>(reqStreamSkip));
            rs.m_offsetAtStart += actualStreamSkip;

            if (actualStreamSkip < wantSkip)
            {
                rs.m_status.setEnd();
            }

            return bufSkip + actualStreamSkip;
        }

        virtual hkLong _read(ReadBufferState& rs, _Out_writes_bytes_(n) void* buf, hkLong n) HK_OVERRIDE
        {
            HK_ASSERT_EXPR(0x17caa618, rs.remaining(), <, n, "StreamReaderImpl::_read "
                "shouldn't be called when there is still more data remaining "
                "in the ReadBufferState's buffer than we're trying to read.");
            hkLong numTotal = 0;
            hkLong numStillRequired = n;
            while( 1 )
            {
                HK_ASSERT_NO_MSG(0x5a571c60, numStillRequired > 0);
                // copy some from the buffer
                hkLong num = hkMath::min2( rs.remaining(), numStillRequired);
                hkMemUtil::memCpy(buf, rs.m_cur, num);
                rs.m_cur = hkAddByteOffset(rs.m_cur, num);
                numTotal += num;

                if (num == numStillRequired)
                {
                    return numTotal;
                }

                buf = hkAddByteOffset(buf, num);
                numStillRequired -= num;

                if (_prefetch(rs, numStillRequired) == 0)
                {
                    return numTotal;
                }

                
//              else if( n > m_buf.getCapacity() ) // bypass buffer if request is big enough
//              {
//                  HK_ASSERT_NO_MSG(0x723feda1, rs.remaining()==0);
//                  rs.m_offsetAtStart += hkGetByteOffset(rs.m_start, rs.m_cur);
//                  rs.m_start = rs.m_end;
//                  hkLong r = _streamRead(buf, n);
//                  rs.m_offsetAtStart += r;
//                  return numTotal + r;
//              }
            }
        }

        virtual hkResult _setReadLimit(ReadBufferState& rs, hkLong n) HK_OVERRIDE
        {
            if (n == ReadBuffer::READ_UNLIMITED )
            {
                m_readLimit = ReadBuffer::READ_UNLIMITED;
                return HK_SUCCESS;
            }
            HK_ASSERT(0x1c361aca, n >= 0, "Invalid readlimit");
            hkLong alreadyRead = rs.m_offsetAtStart + hkGetByteOffset(rs.m_start, rs.m_end);
            if( alreadyRead <= n )
            {
                m_readLimit = n;
                return HK_SUCCESS;
            }
            // oops, we've already gone too far
            return HK_FAILURE;
        }

        virtual hkLong _getReadLimit(const ReadBufferState& rs) const HK_OVERRIDE
        {
            return m_readLimit;
        }

        static _Ret_maybenull_  StreamReaderImpl* create(_Inout_ hkStreamReader* sr)
        {
            return sr ? new StreamReaderImpl(sr) : nullptr;
        }

    private:

        StreamReaderImpl(_In_ hkStreamReader* sr)
            : m_stream(sr)
            , m_readLimit(ReadBuffer::READ_UNLIMITED)
        {
        }

        hkRefPtr<hkStreamReader> m_stream;
        hkArray<char> m_buf;
        hkLong m_readLimit;
    };

    _Ret_maybenull_ ReadBufferImpl* HK_CALL createReaderImpl(_In_z_  const char* filename) 
    {
        if(hkRefPtr<hkStreamReader> sr = hkFileSystem::getInstance().openReader(filename))
        {
            return createReaderImpl(sr);
        }

        return HK_NULL;
    }

    _Ret_notnull_ ReadBufferImpl* HK_CALL createReaderImpl(_Inout_ hkStreamReader* reader)
    {
        return StreamReaderImpl::create(reader);
    }
} }

hkIo::ReadBuffer::ReadBuffer(_In_reads_bytes_(len) const void* buf, hkLong len) : m_attached(false)
{
    attach(buf, len);
}

void hkIo::ReadBuffer::attach(_In_reads_bytes_(len) const void* buf, hkLong len)
{
    detach();

    m_state.m_cur = buf;
    m_state.m_end = hkAddByteOffset(buf, len);
    m_state.m_start = buf;
    m_attached = true;
    m_state.m_offsetAtStart = 0;
}

hkIo::ReadBuffer::~ReadBuffer()
{
    HK_ASSERT(0x3f3725bd, m_state.m_rewind == HK_NULL, "the rewind position was never cleared, its likely to be a bug");
    detach();
}

void hkIo::ReadBuffer::attach(const Detail::ReadBufferAdapter& source)
{
    detach();

    m_impl.reset(source.steal());
    if(m_impl)
    {
        m_impl->_attachTo(m_state);
    }
    else
    {
        m_state.m_cur = m_state.m_start = source.getData();
        m_state.m_end = hkAddByteOffset(m_state.m_start, source.getDataSize());
    }

    m_attached = true;
}

void hkIo::ReadBuffer::detach()
{
    if (!m_attached)
    {
        return;
    }

    if(m_impl)
    {
        m_impl->_detachFrom(m_state);
        m_impl.reset();
    }

    m_attached = false;
}

hkLong hkIo::ReadBuffer::prefetchAll()
{
    HK_ASSERT(0x332bfede, m_attached, "Can't prefetch from an unattached ReadBuffer.");

    if(m_impl)
    {
        return m_impl->_prefetchAll(m_state);
    }
    else
    {
        return hkGetByteOffset(m_state.m_cur, m_state.m_end);
    }
}

hkResult hkIo::ReadBuffer::setReadLimit(hkLong n)
{
    HK_ASSERT(0x1c98e11a, m_attached, "You shouldn't call setReadLimit on an unattached ReadBuffer.");
    return m_impl
        ? m_impl->_setReadLimit(m_state, n)
        : HK_FAILURE;
}

hkLong hkIo::ReadBuffer::getReadLimit() const
{
    HK_ASSERT(0x78c74353, m_attached, "You shouldn't call getReadLimit on an unattached ReadBuffer.");
    return m_impl
        ? m_impl->_getReadLimit(m_state)
        : -1;
}

hkLong hkIo::ReadBuffer::implPrefetch(hkLong n)
{
    return m_impl
        ? m_impl->_prefetch(m_state, n)
        : 0;
}

hkLong hkIo::ReadBuffer::implSkip(hkLong n)
{
    return m_impl
        ? m_impl->_skip(m_state, n)
        : 0;
}

hkLong hkIo::ReadBuffer::implRead(_Out_writes_bytes_(n) void* buf, hkLong n)
{
    if(m_impl)
    {
        return m_impl->_read(m_state, buf, n);
    }
    else
    {
        HK_ASSERT_NO_MSG(0x448dd197, m_attached);

        // We're attached but don't have an impl, so we must be using a memory buffer.
        hkLong avail = hkGetByteOffset(m_state.m_cur, m_state.m_end);
        HK_ASSERT_EXPR(0, avail, <, n, "implRead should only be called when the number "
            "of available bytes is less than the number of requested bytes. "
            "Otherwise it should be handled by the inline part (ie. ReadBuffer::read).");

        _Analysis_assume_(avail < n);

        hkMemUtil::memCpy(buf, m_state.m_cur, avail);
        m_state.m_cur = m_state.m_end;
        return avail;
    }
}

hkResult hkIo::ReadBuffer::implRewind(hkLong offset)
{
    return m_impl
        ? m_impl->_rewind(m_state, offset)
        : HK_FAILURE;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
