// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Serialize/Core/hkSerializeCore.h>
#include <Common/Base/Types/hkEndian.h>

namespace hkSerialize
{
namespace Detail
{
    struct HK_EXPORT_COMMON IndexedArrayImpl : public hkReflect::Detail::ArrayImpl
    {
        IndexedArrayImpl(const hkArray<Bundle::Item>& items);

        virtual hkResult getValue(_In_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const HK_OVERRIDE;
        virtual hkResult setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int nelem) const HK_OVERRIDE;

        const hkArray<Bundle::Item>& m_items;
    };

    struct HK_EXPORT_COMMON IndexedPointerImpl : public hkReflect::Detail::PointerImpl
    {
        IndexedPointerImpl(const hkArray<Bundle::Item>& items);

        virtual hkResult setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* type, const hkReflect::Var& var) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* self, _In_ const hkReflect::PointerType* type, _Out_ hkReflect::Var* val) const HK_OVERRIDE;
        inline hkUint32 getIndex(_In_ const void* self) const;
        const hkArray<Bundle::Item>& m_items;
    };

    struct HK_EXPORT_COMMON IndexedStringImpl : public hkReflect::Detail::StringImpl
    {
        IndexedStringImpl(const hkArray<Bundle::Item>& items);

        virtual hkResult setValue(_Inout_ void* string, _In_ const hkReflect::StringType* type, hkReflect::StringValue newValue) const HK_OVERRIDE;
        virtual hkResult getValue(_Inout_ const void* string, _In_ const hkReflect::StringType* type, _Out_ hkReflect::StringValue* value) const HK_OVERRIDE;

        const hkArray<Bundle::Item>& m_items;
    };

    /// A bundle which is created from a source which is index based (e.g. tagfiles).
    /// Assumes that all pointers/variable arrays contain the id of the pointed object/array buffer in the first
    /// sizeof(void*) bytes. A pointer is considered annotated if the id it contains corresponds to a note Item.
    /// All items share the same indexing.
    /// Invariants:
    /// * vars(0) and extra(0) MUST be empty values (null var & empty array).
    /// * each type used in a var or varN MUST be in the types array.
    class HK_EXPORT_COMMON IndexedBundle : public Bundle
    {
    public:
        HK_DECLARE_CLASS(IndexedBundle, New);

        IndexedBundle();

        virtual hkReflect::Var getContents() const HK_OVERRIDE;
        virtual int getItems(hkArray<Item>& out) const HK_OVERRIDE;
        virtual hkReflect::Var getNoteOnPointer(const hkReflect::PointerVar& ptr) const HK_OVERRIDE;

        hkReflect::Var var(int i) const;
        hkSerialize::VarN varn(int i) const;
        const Item& item(int i) const;

        void setVar(VarId vid, _In_opt_ const void* addr, TypeId tid);
        void setVarToType(VarId vid, TypeId tid);
        void setNote(VarId vid, VarId annotated, _In_opt_ const void* addr, TypeId tid);
        void setType(TypeId tid, _In_ const hkReflect::Type* type);
        void setVarN(VarId vid, _In_opt_ const void* addr, TypeId tid, int count);
        int getSize() const;

        _Ret_notnull_ const hkReflect::Detail::ArrayImpl* arrayImpl() const;
        _Ret_notnull_ const hkReflect::Detail::PointerImpl* pointerImpl() const;
        _Ret_notnull_ const hkReflect::Detail::StringImpl* stringImpl() const;

        void makeSpace(VarId vid);

    protected:
        void clear();
        hkArray<Item> m_items;

        Detail::IndexedArrayImpl m_arrayImpl;
        Detail::IndexedPointerImpl m_pointerImpl;
        Detail::IndexedStringImpl m_stringImpl;
    };


    
    class OldIndexedBundle;

    struct HK_EXPORT_COMMON OldIndexedArrayImpl : public hkReflect::Detail::ArrayImpl
    {
        const OldIndexedBundle* m_bundle;

        virtual hkResult getValue(_In_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const HK_OVERRIDE;
        virtual hkResult setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int nelem) const HK_OVERRIDE;
    };

    struct HK_EXPORT_COMMON OldIndexedPointerImpl : public hkReflect::Detail::PointerImpl
    {
        const OldIndexedBundle* m_bundle;

        inline hkUint32 getIndex(const void* self) const;

        virtual hkResult setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* type, const hkReflect::Var& var) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* self, _In_ const hkReflect::PointerType* type, _Out_ hkReflect::Var* val) const HK_OVERRIDE;
        virtual _Ret_maybenull_ void* queryInterfaceImpl(_In_ const hkReflect::Type* type, const hkReflect::Var& self) const HK_OVERRIDE;
    };

    struct HK_EXPORT_COMMON OldIndexedStringImpl : public hkReflect::Detail::StringImpl
    {
        const OldIndexedBundle* m_bundle;

        virtual hkResult setValue(_Inout_ void* string, _In_ const hkReflect::StringType* type, hkReflect::StringValue newValue) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* string, _In_ const hkReflect::StringType* type, _Out_ hkReflect::StringValue* val) const HK_OVERRIDE;
    };

    
    
    /// A bundle which is created from a source which is index based (e.g. tagfiles)
    /// Invariants:
    /// * vars(0) and extra(0) MUST be empty values (null var & empty array).
    /// * each type used in a var or varN MUST be in the types array.
    class HK_EXPORT_COMMON OldIndexedBundle : public Bundle
    {
        public:
            HK_DECLARE_CLASS(OldIndexedBundle, New);

            OldIndexedBundle();

            virtual hkReflect::Var getContents() const HK_OVERRIDE;

            hkReflect::Var var(int i) const;
            VarN extra(int i) const;
            hkReflect::Var note(int i) const;

            int numNotes() const { return m_notes.getSize(); }
            int numVars() const { return m_vars.getSize(); }
            int numExtras() const { return m_extras.getSize(); }

            virtual int getItems(hkArray<Item>& out) const HK_OVERRIDE;

            virtual hkReflect::Var getNoteOnPointer(const hkReflect::PointerVar& ptr) const HK_OVERRIDE;

            HK_ALWAYS_INLINE const hkReflect::Detail::PointerImpl* pointerImpl() const { return &m_pointerImpl; }
            HK_ALWAYS_INLINE const hkReflect::Detail::ArrayImpl* arrayImpl() const { return &m_arrayImpl; }
            HK_ALWAYS_INLINE const hkReflect::Detail::StringImpl* stringImpl() const { return &m_stringImpl; }
        protected:
            void clear();
            struct IdxVar
            {
                IdxVar() : m_addr(HK_NULL), m_tid(0), m_count(-1) {}
                IdxVar(const void* addr, TypeId tid, int count) : m_addr(addr), m_tid(tid), m_count(count) {}
                void set(_In_ const void* addr, TypeId tid)
                {
                    m_addr = addr;
                    m_tid = tid;
                }
                void set(_In_ const void* addr, TypeId tid, int count)
                {
                    set(addr, tid);
                    m_count = count;
                }
                const void* m_addr;
                TypeId m_tid;
                int m_count;
            };
            hkArray<IdxVar> m_vars;
            hkArray<IdxVar> m_notes;
            hkArray<IdxVar> m_extras;

            Detail::OldIndexedPointerImpl m_pointerImpl;
            Detail::OldIndexedArrayImpl m_arrayImpl;
            Detail::OldIndexedStringImpl m_stringImpl;
    };

    class HK_EXPORT_COMMON OldDefaultIndexedBundle : public OldIndexedBundle
    {
        public:
            HK_DECLARE_CLASS(OldDefaultIndexedBundle, New);
            OldDefaultIndexedBundle() : OldIndexedBundle() {}
            void setVar(VarId vid, _In_ const void* addr, TypeId tid);
            void addNote(VarId vid, _In_ const void* addr, TypeId tid);
            void setType(TypeId tid, _In_ const hkReflect::Type* type);
            void setVarN(VarId vid, _In_ const void* addr, TypeId tid, int count);
        protected:
            void makeSpace(VarId vid);
    };

    /// Keeps track of the types referenced from a given type.
    /// This structure is used while parsing a type. See also FlatTypeRelocs
    struct TypeRelocs
    {
        TypeRelocs() : m_parent(0), m_subType(0) {}
        void setParent(TypeId t) { m_parent = t; }
        void addSubType(TypeId t) { m_subType = t; }
        void addField(TypeId t) { HK_ASSERT_NO_MSG(0x16cf1a68, t > 0 ); m_fields.pushBack(t); }
        void addOpaqueField() { m_fields.pushBack(-1); }
        void addTemplateParam(TypeId t) { m_template.pushBack(t); }
        void addInterface(TypeId t) { m_interfaces.pushBack(t); }

        TypeId m_parent;
        TypeId m_subType;
        hkInplaceArray<TypeId, 32> m_fields;
        hkInplaceArray<TypeId, 32> m_template;
        hkInplaceArray<TypeId, 2> m_interfaces;
    };

    /// Optimized storage of several TypeRelocs. Note the storage is inverted. TypeRelocs stores
    /// the various types referenced by a type, this stores a list of locations pointing to a given type.
    struct HK_EXPORT_COMMON FlatTypeRelocs
    {
        /// Notify that we now have the given type with the given typeId.
        /// Resolve existing pending relocations with the given type as target after it was loaded.
        void resolve( TypeId typeId, _In_ const hkReflect::Type* type );

        /// Apply relocations on a newly loaded type, either registering pending relocations if the source type
        /// is not loaded yet, or resolving them if the source type is already available.
        void flatten( _In_ hkReflect::Type* type, const TypeRelocs& relocs, hkArrayView<const hkReflect::Type*> types );

    protected:

        /// Attempt to apply a relocation to address @addr with type ID @typeId.
        /// The list of loaded types is passed in @types, and if @typeId is present the relocation is applied.
        /// Otherwise the fallback collection is used, or if empty a NULL pointer is left at @addr.
        void applyRelocation(_Outptr_ const hkReflect::Type** addr, TypeId id, hkArrayView<const hkReflect::Type*> types );

        /// Map of typeId to array of pointers "waiting" to be assigned.
        hkHashMap< TypeId, hkArray<const hkReflect::Type**> > m_typeRelocs;
    };

        /// Keeps an array of id-type relocations. Whenever a new type is added, the corresponding relocations are fixed.
    template<typename BaseBundle>
    class BundleWithRelocs : public BaseBundle
    {
    public:
        BundleWithRelocs() {}

        void setType(int tid, _In_ hkReflect::Type* _type, const TypeRelocs& relocs);
        void setType(int tid, _In_ const hkReflect::Type* _type);
    protected:
        FlatTypeRelocs m_relocs;
    };

    typedef BundleWithRelocs<OldDefaultIndexedBundle> OldBundleWithRelocs;
    typedef BundleWithRelocs<IndexedBundle> DefaultBundle;
} }

#include <Common/Base/Serialize/Detail/hkIndexedBundle.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
