// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Detail/hkIndexedBundle.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>

#define DEBUG_LOG_IDENTIFIER "s11n.detail.IndexedBundle"
#include <Common/Base/System/Log/hkLog.hxx>

namespace hkSerialize {
namespace Detail {

IndexedArrayImpl::IndexedArrayImpl(const hkArray<Bundle::Item>& items) : m_items(items) {}

hkResult IndexedArrayImpl::getValue(_In_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const
{
    if (int count = arrType->getFixedCount())
    {
        *val = hkReflect::ArrayValue(arrAddr, count, arrType->getSubType());
        return HK_SUCCESS;
    }
    hkUint32 eid = *(hkUint32Le*)arrAddr;
    if (eid) //&& m_items[eid].isVarN())
    {
        VarN varn = m_items[eid].varn();
        HK_ASSERT_NO_MSG(0x3cfbd490, varn.getCount() != 0 || (arrType->getSubType().isNull() && varn.getType()));
        
        
        *val = hkReflect::ArrayValue(varn.getAddress(), varn.getCount(), varn.getType());
        return HK_SUCCESS; 
    }
    else
    {
        *val = hkReflect::ArrayValue();
        return HK_SUCCESS; 
    }
}

hkResult IndexedArrayImpl::setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int nelem) const
{
    HK_ASSERT_NO_MSG(0x73e35517, 0);
    return HK_FAILURE;
}

IndexedPointerImpl::IndexedPointerImpl(const hkArray<Bundle::Item>& items) : m_items(items) {}

hkResult IndexedPointerImpl::setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* type, const hkReflect::Var& var) const
{
    HK_ASSERT_NO_MSG(0x41bd7b9, 0);
    return HK_FAILURE;
}

inline hkUint32 IndexedPointerImpl::getIndex(_In_ const void* self) const
{
    hkUint32 i = *(hkUint32Le*)self;
    // We may have IDs to vars that haven't been written to the file (addEmpty). If so, just redirect
    // those IDs to the null var (id=0).
    if (i >= hkLosslessCast<hkUint32>(m_items.getSize()))
    {
        i = 0;
    }
    return i;
}

hkResult IndexedPointerImpl::getValue(_In_ const void* self, _In_ const hkReflect::PointerType* type, _Out_ hkReflect::Var* val) const
{
    hkUint32 i = getIndex(self);
    const auto& item = m_items[i];
    //TODO.SECURITY - check type
    if (item.isNote())
    {
        *val = m_items[item.getAnnotated()].var();
        return HK_SUCCESS;
    }
    *val = item.var();
    return HK_SUCCESS;
}

IndexedStringImpl::IndexedStringImpl(const hkArray<Bundle::Item>& items) : m_items(items) {}

hkResult IndexedStringImpl::setValue(_Inout_ void* string, _In_ const hkReflect::StringType* type, hkReflect::StringValue newValue) const
{
    HK_ASSERT_NO_MSG(0x1b5781e5, 0);
    return HK_FAILURE;
}
hkResult IndexedStringImpl::getValue(_Inout_ const void* string, _In_ const hkReflect::StringType* type, _Out_ hkReflect::StringValue* val) const
{
    if (hkUint32 eid = *(hkUint32Le*)string)
    {
        auto& item = m_items[eid];
        HK_ASSERT_NO_MSG(0x3a546001, item.isVarN());
        
        *val = (const char*)item.varn().getAddress();
        return HK_SUCCESS;
    }
    else
    {
        *val = hkReflect::StringValue();
        return HK_SUCCESS;
    }
}

IndexedBundle::IndexedBundle()
    : m_arrayImpl(m_items)
    , m_pointerImpl(m_items)
    , m_stringImpl(m_items)
{
    m_types.pushBack( nullptr ); // TypeId #0 is always the NULL type - never explicitly serialized
}

hkReflect::Var IndexedBundle::getContents() const
{
    return m_items.getSize() > 1 ? m_items[1].var() : hkReflect::Var();
}

void IndexedBundle::clear()
{
    m_items.clearAndDeallocate();
}

int IndexedBundle::getItems(hkArray<hkSerialize::Bundle::Item>& out) const
{
    out = m_items;
    return out.getSize();
}

hkReflect::Var IndexedBundle::getNoteOnPointer(const hkReflect::PointerVar& ptr) const
{
    HK_ASSERT_NO_MSG(0x2813d5b0, ptr.getImpl() == &m_pointerImpl);
    hkUint32 index = m_pointerImpl.getIndex(ptr.getAddress());
    if (m_items[index].isNote())
    {
        return m_items[index].var();
    }
    else if (m_items[index].m_addr == HK_NULL && m_items[index].m_extra > 0)
    {
        
        // If the index corresponds to an import with a note, m_vars[index].m_count will contain the note id
        const Item& importNote = m_items[m_items[index].m_extra];
        HK_ASSERT_NO_MSG(0xf24be95, importNote.isNote());
        return importNote.var();
    }
    return hkReflect::Var();
}

void OldDefaultIndexedBundle::setVar(VarId vid, _In_ const void* addr, TypeId tid)
{
    makeSpace(vid);
    m_vars[vid].set(addr, tid);
}

void OldDefaultIndexedBundle::addNote(VarId vid, _In_ const void* addr, TypeId tid)
{
    makeSpace(vid);

    // If the id is an import, add a reference for getPointerOnNote
    if (m_vars[vid].m_addr == HK_NULL)
    {
        m_vars[vid].m_count = m_notes.getSize();
    }

    m_notes.expandOne().set(addr, tid, vid);
}

void OldDefaultIndexedBundle::setType(int tid, _In_ const hkReflect::Type* _type)
{
    m_types.setSize(hkMath::max2(m_types.getSize(), tid + 1), HK_NULL);
    m_types[tid] = _type;
}

void OldDefaultIndexedBundle::setVarN(VarId vid, _In_ const void* addr, TypeId tid, int count)
{
    m_extras.setSize(hkMath::max2(m_extras.getSize(), vid + 1), IdxVar(HK_NULL, 0, 0));
    m_extras[vid].set(addr, tid, count);
}

void OldDefaultIndexedBundle::makeSpace(VarId vid)
{
    int oldSize = m_vars.getSize();
    int newSize = hkMath::max2(oldSize, vid + 1);
    m_vars.setSize(newSize, IdxVar());
}

hkReflect::Var OldIndexedBundle::getContents() const
{
    return m_vars.getSize() > 1 ? var(1) : hkReflect::Var();
}

void OldIndexedBundle::clear()
{
    m_vars.clearAndDeallocate();
    m_extras.clearAndDeallocate();
    m_notes.clearAndDeallocate();
}

int OldIndexedBundle::getItems(hkArray<Item>& out) const
{
    int offset = m_extras.isEmpty() ? 0 : 1;
    out.setSize(m_vars.getSize() + m_notes.getSize() + m_extras.getSize() - offset);

    for(int i = 1; i < m_vars.getSize(); ++i)
    {
        out[i].m_addr = var(i).getAddress();
        out[i].m_type = var(i).getType();
        out[i].m_kind = Item::VAR0;
        out[i].m_extra = VarN::SINGLE;
    }

    for (int i = 0; i < m_notes.getSize(); ++i)
    {
        int outIdx = m_vars.getSize() + i;
        out[outIdx].m_addr = m_notes[i].m_addr;
        out[outIdx].m_type = type(m_notes[i].m_tid);
        out[outIdx].m_kind = Item::NOTE;
        out[outIdx].m_extra = m_notes[i].m_count;
    }

    for (int i = 1; i < m_extras.getSize(); ++i)
    {
        int outIdx = m_vars.getSize() + m_notes.getSize() + i - 1;
        out[outIdx].m_addr = m_extras[i].m_addr;
        out[outIdx].m_type = type(m_extras[i].m_tid);
        out[outIdx].m_kind = Item::VARN;
        out[outIdx].m_extra = m_extras[i].m_count;
    }
    return out.getSize();
}

hkReflect::Var OldIndexedBundle::getNoteOnPointer(const hkReflect::PointerVar& ptr) const
{
    HK_ASSERT_NO_MSG(0x5effd96c, ptr.getImpl() == &m_pointerImpl);
    hkUint32 index = m_pointerImpl.getIndex(ptr.getAddress());
    if (m_vars[index].m_addr == HK_NULL && m_vars[index].m_count >= 0)
    {
        // If the index corresponds to an import with a note, m_vars[index].m_count will contain the note id
        return note(m_vars[index].m_count);
    }
    return hkReflect::Var();
}

void FlatTypeRelocs::applyRelocation( _Outptr_ const hkReflect::Type** addr, TypeId id, hkArrayView<const hkReflect::Type*> types )
{
    HK_ASSERT( 0x7b7f204a, *addr == nullptr, "Relocation target was expected to be a NULL pointer." );
    HK_ASSERT_NO_MSG( 0x7b7f204b, ( id == -1 ) || ( id > 0 ) );
    DLOG("Relocate {} -> TypeId={}", addr, id);

    if ( id == -1 )
    {
        // Special value for YAML (NULL type -> Opaque)
        *addr = hkReflect::getType<hkReflect::Detail::Opaque>().get();
    }
    else if ( ( id < types.getSize() ) && types[id] )
    {
        // Type was loaded since relocation was registered; apply the relocation
        *addr = types[id];
    }
    else
    {
        // Type ID not encountered yet, leave a placeholder (fallback type)
        *addr = hkReflect::getType<hkReflect::Detail::Opaque>().get();

        // Add a relocation entry for later resolve
        hkArray<const hkReflect::Type**> defVal;
        auto& relocs = m_typeRelocs.getValue( m_typeRelocs.findOrInsertKey( id, defVal) );
        HK_ASSERT(0x763a432b, relocs.indexOf(addr) < 0, "Found duplicate relocation." );
        relocs.pushBack(addr);
    }
}

void FlatTypeRelocs::resolve( TypeId id, _In_ const hkReflect::Type* type )
{
    DLOG_SCOPE( "TypeId {} resolves to {} '{}'.", id, type, type->getName() );
    auto const it = m_typeRelocs.find( id );
    if ( m_typeRelocs.isValid( it ) )
    {
        // New loaded type is a target for pending relocs; resolve them now.
        hkArray<const hkReflect::Type**>& pendingRelocs = m_typeRelocs.getValue( it );

        // These branches do the same thing, except for error checking.
        // In both cases, we check that the existing pointer is what we expect before overwriting it.
        for ( auto reloc : pendingRelocs )
        {
            HK_ASSERT( 0x607fc9c7, *reloc == hkReflect::getType<hkReflect::Detail::Opaque>().get(), "Expected to find the Detail::Opaque type before applying type relocation." );
            *reloc = type;
            DLOG( "Resolving {}", reloc );
        }

        // Delete the reloc entries associated to this type ID.
        m_typeRelocs.remove( it );
    }
}

void FlatTypeRelocs::flatten( _In_ hkReflect::Type* type, const TypeRelocs& relocs, hkArrayView<const hkReflect::Type*> types )
{
    DLOG_AUTO( "Flattening type relocations in {} '{}'.", type, type->getName() );

    if ( relocs.m_parent > 0 )
    {
        applyRelocation( const_cast<const hkReflect::Type**>(hkReflect::TypeDetail::addressParent( type )), relocs.m_parent, types );
    }

    if ( relocs.m_subType > 0 )
    {
        applyRelocation( hkReflect::TypeDetail::localAddressOptional<hkReflect::Opt::SUBTYPE>(type), relocs.m_subType, types );
    }

    if (const hkReflect::Detail::DeclsArray* fa = hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::DECLS>(type) )
    {
        HK_ASSERT_NO_MSG(0x61173216, fa->getNumDataFields() == relocs.m_fields.getSize());
        for ( int i = 0; i < fa->getNumDataFields(); ++i )
        {
            const hkReflect::Type* fieldType = fa->getField( i ).getType();

            applyRelocation((const hkReflect::Type**)hkReflect::TypeDetail::addressParent(const_cast<hkReflect::Type*>(fieldType)), relocs.m_fields[i], types);
        }
#ifdef HK_DEBUG_SLOW
        for (int i = fa->getNumDataFields(); i < fa->getNumFields(); ++i)
        {
            HK_ASSERT(0x30cb2669, fa->getField(i).getType()->getParent() == hkReflect::getType<hkReflect::Detail::Opaque>(),
                "All property fields should have been initialized to Opaque");
        }
#endif
    }

    if ( relocs.m_template.getSize() )
    {
        auto* templateParams = const_cast<hkReflect::Template::Parameter*>( type->getTemplate()->getParams() );
        for ( int i = 0; i < relocs.m_template.getSize(); )
        {
            HK_ASSERT_NO_MSG( 0x532dad0d, ( templateParams - type->getTemplate()->getParams() ) < type->getTemplate()->getNumParams() );
            if ( templateParams->isType() )
            {
                applyRelocation( reinterpret_cast<const hkReflect::Type**>( &templateParams->m_storage ), relocs.m_template[i], types );
                ++i;
            }
            templateParams++;
        }
    }

    if ( relocs.m_interfaces.getSize() )
    {
        auto ifaceArray = const_cast<hkReflect::Detail::InterfaceArray*>( hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::INTERFACES>( type ) )->items();
        HK_ASSERT_NO_MSG( 0x532dad0e, relocs.m_interfaces.getSize() == ifaceArray.getSize() );
        for ( int i = 0; i < relocs.m_interfaces.getSize(); ++i )
        {
            applyRelocation( &ifaceArray[i].m_interfaceType, relocs.m_interfaces[i], types );
        }
    }
}

hkResult OldIndexedArrayImpl::getValue(_In_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const
{
    if(int count = arrType->getFixedCount())
    {
        *val = hkReflect::ArrayValue(arrAddr, count, arrType->getSubType());
        return HK_SUCCESS;
    }
    hkUint32 eid = *(hkUint32Le*)arrAddr;
    
    if(eid)
    {
        // arrays in the var list are {i32 size, packed_data...}
        VarN varn = m_bundle->extra(eid);
        HK_ASSERT_NO_MSG(0x3cfbd490, varn.getCount()!=0 || (arrType->getSubType().isNull() && varn.getType()));
        
        
        *val = hkReflect::ArrayValue(varn.getAddress(), varn.getCount(), varn.getType());
        return HK_SUCCESS;
    }
    else
    {
        *val = hkReflect::ArrayValue();
        return HK_SUCCESS;
    }
}

hkResult OldIndexedArrayImpl::setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int nelem) const
{
    HK_ASSERT_NO_MSG(0x73e35517, 0);
    return HK_FAILURE;
}

hkResult OldIndexedPointerImpl::setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* type, const hkReflect::Var& var) const
{
    HK_ASSERT_NO_MSG(0x41bd7b9, 0);
    return HK_FAILURE;
}

hkResult OldIndexedPointerImpl::getValue(_In_ const void* self, _In_ const hkReflect::PointerType* type, _Out_ hkReflect::Var* val) const
{
    hkUint32 i = getIndex(self);
    
    *val = m_bundle->var(i);
    return HK_SUCCESS;
}

_Ret_maybenull_ void* OldIndexedPointerImpl::queryInterfaceImpl(_In_ const hkReflect::Type* type, const hkReflect::Var& self) const
{
    if(getIndex(self.getAddress()) && type->extendsOrEquals<Detail::IdFromPointer>())
    {
        return reinterpret_cast<Detail::IdFromPointer*>(self.getAddress());
    }
    return HK_NULL;
}

hkResult OldIndexedStringImpl::setValue(_Inout_ void* string, _In_ const hkReflect::StringType* type, hkReflect::StringValue newValue) const
{
    HK_ASSERT_NO_MSG(0x1b5781e5, 0);
    return HK_FAILURE;
}

hkResult OldIndexedStringImpl::getValue(_In_ const void* string, _In_ const hkReflect::StringType* type, _Out_ hkReflect::StringValue* val) const
{
    if (hkUint32 eid = *(hkUint32Le*)string)
    {
        
        *val = (const char*)m_bundle->extra(eid).getAddress();
        return HK_SUCCESS;
    }
    else
    {
        *val = hkReflect::StringValue();
        return HK_SUCCESS;
    }
}

} // namespace Detail
} // namespace hkSerialize

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
