// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Serialize/Deprecated/hkSerializeUtil.h>
#include <Common/Base/Config/hkConfigVersion.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Memory/Allocator/Pooled/hkPooledAllocator.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/System/Log/hkLog.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/Reflect/Builder/hkTypeCopier.h>
#if defined(HK_BUILDING_WITH_ENGINE)
#include <Common/NewBase/Reflection/Util/hkDynamicTypeManager.h> 
#endif

#include <Common/Base/Serialize/Core/hkSerializeCore.h>
#include <Common/Base/Serialize/Detail/hkSerializeDetail.h>
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileWriteFormat.h>

#define RAISE_ERROR(err, m_id, msg) if( err ) err->raiseError( m_id, msg )

namespace
{

static const char loadVersion0FailedBoilerPlate[] =
    "No information about this class exists in the serialization system. "
    "This could happen if:\n"
    "* You have provided a custom classes file which does not include this class\n"
    "* You have not specified keycodes before including hkProductFeatures.cxx\n"
    "* Custom classes have not been registered.";

static const char versioningFailedBoilerPlate[] =
    "Patching to latest version failed. Have you registered the necessary patches? " \
    "Patches may be missing because your assets are older than HK_SERIALIZE_MIN_COMPATIBLE_VERSION (if defined). " \
    "Make sure that the patches are registered properly (see hkRegisterPatches.cxx, included by hkProductFeatures.cxx). "
    "See the hkError output for more details.";

static const char loadingFailedBoilerPlate[] =
    "Loading tagfile failed. Is it from a newer version of the SDK?  "\
    "See the hkError output for more details";

struct ErrorMapEntry
{
    hkSerialize::Detail::ErrorID m_serializeDetailError;
    hkSerializeUtil::ErrorDetails::ErrorID m_serializeUtilError;
};

// Compatibility.
const ErrorMapEntry errorMap[] = {
    { hkSerialize::Detail::ERRORID_READ_FAILED, hkSerializeUtil::ErrorDetails::ERRORID_READ_FAILED },
    { hkSerialize::Detail::ERRORID_UNSUPPORTED_FORMAT, hkSerializeUtil::ErrorDetails::ERRORID_UNSUPPORTED_FORMAT },
    { hkSerialize::Detail::ERRORID_VERSIONING_FAILED, hkSerializeUtil::ErrorDetails::ERRORID_VERSIONING_FAILED },
    { hkSerialize::Detail::ERRORID_LOAD_FAILED, hkSerializeUtil::ErrorDetails::ERRORID_LOAD_FAILED },
    { hkSerialize::Detail::ERRORID_UNSUPPORTED_FORMAT, hkSerializeUtil::ErrorDetails::ERRORID_UNSUPPORTED_FORMAT }
};

hkSerialize::Load::VersioningOptions convertSerializeUtilToSerializeOptions(hkFlags<hkSerializeUtil::LoadOptionBits, hkUint32> serializeUtilOptions)
{
    if (serializeUtilOptions.anyIsSet(hkSerializeUtil::LOAD_NOVERSIONING))
    {
        return hkSerialize::Load::VERSIONING_DISABLED;
    }
    else if (serializeUtilOptions.anyIsSet(hkSerializeUtil::LOAD_FAIL_IF_VERSIONING))
    {
        return hkSerialize::Load::FAIL_IF_VERSIONING_REQUIRED;
    }
    else
    {
        return hkSerialize::Load::VERSIONING_DEFAULT;
    }
}

}

//
// Load into resource
//

_Ret_maybenull_ hkResource* HK_CALL hkSerializeUtil::load(_Inout_ hkStreamReader* streamIn, _Inout_opt_ ErrorDetails* errorOut, LoadOptions options)
{
    if( streamIn == HK_NULL )
    {
        RAISE_ERROR( errorOut, ErrorDetails::ERRORID_READ_FAILED, "Stream is not ok");
        return HK_NULL;
    }

    hkRefPtr<hkResource> ret;
    hkLog::Raii::CaptureOutput capture;
    {
        ret = hkSerialize::Load().withTypeReg(options.getTypeLookup())
            .withValidation(false) 
            .withVersioning(convertSerializeUtilToSerializeOptions(options.getOptions())).withIgnoreInvalid()
            .toOwnedResource(streamIn);
    }

    if (ret)
    {
        ret->addReference();
    }
    else if (errorOut)
    {
        hkStringBuf text; capture.getText(text);
        errorOut->raiseError(ErrorDetails::ERRORID_LOAD_FAILED, text.cString());
    }

    return ret;
}

//
// Load into heap objects
//
_Ret_maybenull_ hkObjectResource* HK_CALL hkSerializeUtil::loadOnHeap(_Inout_ hkStreamReader* streamIn, _Inout_opt_ ErrorDetails* errorOut, LoadOptions options)
{
    if( streamIn == HK_NULL )
    {
        RAISE_ERROR( errorOut, ErrorDetails::ERRORID_READ_FAILED, "Stream pointer is null");
        return HK_NULL;
    }

    if ( !streamIn->isOk() )
    {
        RAISE_ERROR( errorOut, ErrorDetails::ERRORID_READ_FAILED, "Stream is not ok");
        return HK_NULL;
    }

    hkRefPtr<hkObjectResource> ret;
    hkLog::Raii::CaptureOutput capture;
    {
        ret = hkSerialize::Load().withTypeReg(options.getTypeLookup())
            .withVersioning(convertSerializeUtilToSerializeOptions(options.getOptions())).withIgnoreInvalid()
            .toResource(streamIn);
    }

    if (ret)
    {
        ret->addReference();
    }
    else if (errorOut)
    {
        hkStringBuf text; capture.getText(text);
        errorOut->raiseError(ErrorDetails::ERRORID_LOAD_FAILED, text.cString());
    }

    return ret;
}

_Ret_maybenull_ hkResource* HK_CALL hkSerializeUtil::load(_In_z_ const char* filename, _Inout_opt_ ErrorDetails* resultOut, LoadOptions options)
{
    hkIstream stream(filename);
    return load(stream.getStreamReader(), resultOut, options);
}

_Ret_maybenull_ hkResource* HK_CALL hkSerializeUtil::load(_In_reads_bytes_(buflen) const void* buf, int buflen, _Inout_opt_ ErrorDetails* resultOut, LoadOptions options)
{
    return load(hkIstream(buf, buflen).getStreamReader(), resultOut, options);
}

_Ret_maybenull_ hkObjectResource* HK_CALL hkSerializeUtil::loadOnHeap(_In_z_ const char* filename, _Inout_opt_ ErrorDetails* errorOut, LoadOptions options)
{
    return loadOnHeap(hkIstream(filename).getStreamReader(), errorOut, options);
}

_Ret_maybenull_ hkObjectResource* HK_CALL hkSerializeUtil::loadOnHeap(_In_reads_bytes_(buflen) const void* buf, int buflen, _Inout_opt_ ErrorDetails* errorOut, LoadOptions options)
{
    return loadOnHeap(hkIstream(buf, buflen).getStreamReader(), errorOut, options);
}

hkResult HK_CALL hkSerializeUtil::saveTagfile(_In_ const void* object, _In_ const hkReflect::Type* klass, _Inout_ hkStreamWriter* writer, SaveOptions options)
{
    if (writer)
    {
        hkReflect::Var toSave(hkReflect::exactObj(object, klass));
        if (options.anyIsSet(SAVE_TEXT_FORMAT))
        {
            hkRefPtr<hkSerialize::WriteFormat> format = hkSerialize::Detail::fileFormatXmlTagfile.createWriteFormat();
            if ( format )
            {
                return hkSerialize::Save().withFormatState( format ).withIgnoreInvalid().contentsVar( toSave, writer );
            }
        }
        else
        {
            return hkSerialize::Save().withIgnoreInvalid().contentsVar(toSave, writer);
        }
    }
    return HK_FAILURE;
}

hkResult HK_CALL hkSerializeUtil::save(_In_ const void* object, _In_ const hkReflect::Type* klass, _Inout_ hkStreamWriter* writer, SaveOptions options)
{
    
    return saveTagfile(object, klass, writer, options);
}

hkBool32 HK_CALL hkSerializeUtil::isLoadable(_Inout_opt_ hkStreamReader* sr)
{
    if (!sr || !sr->isOk())
    {
        return false;
    }

    if( detectFormat( sr ) > FORMAT_UNKNOWN )
    {
        return true;
    }

    return false;
}

hkEnum<hkSerializeUtil::FormatType, hkInt32> HK_CALL hkSerializeUtil::detectFormat(_Inout_ hkStreamReader* reader, _Inout_opt_ ErrorDetails* errorOut)
{
    FormatDetails details;
    detectFormat(reader, details, errorOut);
    return details.m_formatType;
}

namespace
{
    struct KnownDetectFormatEntries
    {
        hkSerialize::Detail::FileFormatRegNode* m_entry;
        hkSerializeUtil::FormatType m_formatType;
    };
    static const KnownDetectFormatEntries knownDetectFormatEntries[] =
    {
        { &hkSerialize::Detail::fileFormatBinaryTagfile2014, hkSerializeUtil::FORMAT_TAGFILE_BINARY },
        { &hkSerialize::Detail::fileFormatBinaryPackfile2014, hkSerializeUtil::FORMAT_PACKFILE_BINARY },
        { &hkSerialize::Detail::fileFormatXmlTagfile2014, hkSerializeUtil::FORMAT_TAGFILE_XML },
        { &hkSerialize::Detail::fileFormatXmlPackfile2010, hkSerializeUtil::FORMAT_PACKFILE_XML },
    };
}
hkSerializeUtil::FormatDetails::FormatDetails()
    : m_formatType( FORMAT_ERROR )
    , m_formatVersion( 0 )
{
}

void HK_CALL hkSerializeUtil::detectFormat(_Inout_ hkStreamReader* reader, FormatDetails& details, _Inout_opt_ ErrorDetails* errorOut )
{
    // Peek at first few bytes to check if the file is binary
    char buf[hkSerialize::Detail::FORMAT_HEADER_PEEK_LENGTH + 1] = {};
    buf[hkSerialize::Detail::FORMAT_HEADER_PEEK_LENGTH] = 0;
    int bufSize = reader->peek( buf, sizeof(buf));
    if( bufSize <= 0 )
    {
        HK_WARN( 0x2fbceba8, "Could not peek at file header.");
        return;
    }

    details.m_formatType = FORMAT_UNKNOWN;

    for ( int i = 0; i < HK_COUNT_OF(knownDetectFormatEntries); ++i )
    {
        if ( knownDetectFormatEntries[i].m_entry->detect( buf, bufSize ) )
        {
            details.m_formatType = knownDetectFormatEntries[i].m_formatType;
            return;
        }
    }

    if ( hkSerialize::Detail::detectFormat( buf, bufSize ) )
    {
        details.m_formatType = FORMAT_OTHER;
    }
}

hkResult HK_CALL hkSerializeUtil::savePackfile(
    _In_ const void* object,
    _In_ const hkReflect::Type* klass,
    _Inout_ hkStreamWriter* writer,
    SaveOptions options)
{
    HK_ASSERT( 0x22440730, options == SAVE_DEFAULT, "savePackfile is deprecated and only supports SAVE_DEFAULT. Use hkSerialize::Save instead." );
    hkReflect::TypeCopier::Options platformCopyOptions;
    platformCopyOptions.setHostTarget();
    hkReflect::TypeCopier copier( platformCopyOptions );
    hkSerialize::TagfileWriteFormat writeFormat( &copier );
    return hkSerialize::Save().withFormatState( &writeFormat ).withIgnoreInvalid().contentsVar( hkReflect::Var( object, klass ), writer );
}

namespace
{
    class DefaultTypeCallbackType : public hkReflect::TypeReg
    {
        virtual _Ret_maybenull_ const hkReflect::Type* typeFromName(_In_z_ const char* n) const HK_OVERRIDE
        {
            return hkReflect::typeFromName(n);
        }

        virtual _Ret_maybenull_ const hkReflect::Type* typeFromType(_In_ const hkReflect::Type* tp) const HK_OVERRIDE
        {
            const hkReflect::Type* regType = hkReflect::typeFromType(tp);
            if(regType)
            {
                return regType;
            }
#if defined(HK_BUILDING_WITH_ENGINE)
            if(tp->isDynamicType())
            {
                return hkDynamicTypeManager::getInstance().addCopyType(tp);
            }
#endif
            // Default only copies dynamic types
            return HK_NULL;
        }
    } s_instanceDefaultTypeCallbackType;
}

namespace hkSerializeUtil
{
    HK_EXPORT_COMMON hkReflect::TypeReg* s_defaultLookupTypeCallback = &s_instanceDefaultTypeCallbackType;
}

hkSerializeUtil::LoadOptions::LoadOptions(hkSerializeUtil::LoadOptionBits opts, _Inout_opt_ hkReflect::TypeReg* lookup)
: m_options(opts)
{
    if(lookup == HK_NULL)
    {
        if(m_options.anyIsSet(LOAD_FAIL_IF_DYNAMIC))
        {
            lookup = hkReflect::getTypeReg();
        }
        else
        {
            lookup = s_defaultLookupTypeCallback;
        }
    }
    m_typeLookup = lookup;
}

hkUint32 hkSerializeUtil::calcCrc32(const hkReflect::Var& var, hkUint32 seed)
{
    hkArray<char> buf;
    hkSerialize::Save().withIgnoreInvalid().contentsVar(var, &buf);
    return hkHash::appendCrc32(seed, buf.begin(), buf.getSize());
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
