// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>
#include <Common/Base/Types/hkEndian.h>
#include <Common/Base/Container/Hash/hkHashMap.h>
#include <Common/Base/Container/Hash/hkHashSet.h>

namespace hkSerialize
{
    typedef int VarId;
    typedef int TypeId;

    namespace Note
    {
        struct HK_EXPORT_COMMON Import
        {
            HK_DECLARE_CLASS(Import, New, Reflect, BypassCtor);
            Import(const char* n, int l = -1);
            hkStringPtr m_name;
        };

        struct HK_EXPORT_COMMON Export
        {
            HK_DECLARE_CLASS(Export, New, Reflect, BypassCtor);
            Export(const char* n, int l = -1);
            hkStringPtr m_name;
        };
    }

        /// Equivalent of Var except for a variable number of contiguous objects.
        /// This class uses the named constructor idiom, see fromSingle and fromArray.
    class HK_EXPORT_COMMON VarN
    {
        public:

            HK_DECLARE_CLASS(VarN, New);
            enum { SINGLE = -1, /*count if >= 0*/ };

            static VarN fromSingle(_In_ const void* addr, hkReflect::QualType type);
            static VarN fromSingle(const hkReflect::Var& var);
            static VarN fromArray(_In_ const void* addr, hkReflect::QualType type, int count);

            hkReflect::QualType getType() const;

            template<typename T>
            _Ret_maybenull_ T* isInstanceOf() const;

            _Ret_maybenull_ void* getAddress() const;

            int getCount() const;

            operator hkBoolOperator::BoolType() const;
            hkReflect::Var getFirst() const;
            hkReflect::Var get(int id) const;
            static VarN invalid();
        protected:

            VarN(_In_ const void* addr, hkReflect::QualType type, int count);

            static const VarN s_invalid;
            hkReflect::Var m_first;
            int m_count; ///< >=0 means an array, -1 means single, -2 is invalid
    };

        /// A bundle is a self-contained unit of instances and types.
        /// Bundles are typically produced from loading a serialized file or network message.
        /// The base Bundle interface only provides read access. Derived classes populate the base
        /// members in various different ways.
        /// Invariants:
        /// * the types array MAY contain other types which are indirectly referenced (e.g. types used by fields)
    class HK_EXPORT_COMMON Bundle
    {
        public:
            HK_DECLARE_CLASS(Bundle, New);

            Bundle();
            virtual ~Bundle();

            virtual hkReflect::Var getContents() const = 0;

            _Ret_notnull_ const hkReflect::Type* type(int i) const;
            hkArrayView<const hkReflect::Type*> types() const;

            
            class Item
            {
                public:
                    HK_DECLARE_CLASS(Item, New);

                    enum Kind
                    {
                        VAR0,
                        VARN,
                        NOTE
                    };

                    Item();

                    void set(_In_ const void* addr, hkReflect::QualType type, Kind kind, int extra);

                    bool isVar() const;
                    bool isVarN() const;
                    int getCount() const;

                    bool isNote() const;
                    int getAnnotated() const;

                    hkReflect::Var var() const;
                    VarN varn() const;

                    const void* m_addr;
                    hkReflect::QualType m_type;
                    hkEnum<Kind, hkInt8> m_kind;
                    int m_extra;
            };

                /// Get all the items in the Bundle.
            virtual int getItems(hkArray<Item>& out) const = 0;

                /// Return the note attached to the pointer, if any. If the pointed object is an import and it has a
                /// note, must return that note.
            virtual hkReflect::Var getNoteOnPointer(const hkReflect::PointerVar& ptr) const = 0;

        protected:

            hkArray<const hkReflect::Type*> m_types;
    };

    namespace Detail
    {
        struct HK_EXPORT_COMMON IdFromPointer
        {
            HK_DECLARE_CLASS(IdFromPointer, New, Reflect);
            IdFromPointer() {}
            hkUint32Le id;
        };
    }

        /// Interface to provide stable ids from vars.
    class HK_EXPORT_COMMON IdFromVar
    {
        public:
                /// Request the id for a given var.
                /// If it is new, an new id is generated, else the previous id is returned.
            virtual VarId lookup(const hkReflect::Var& var) = 0;
                /// Request the id for the pointed-var.
                /// If it is new, an new id is generated, else the previous id is returned.
            virtual VarId pointer(const hkReflect::PointerVar& ptr) = 0;
                /// Notify that the named id is being written.
                /// This allows us to track which vars have been referenced but not yet written.
            virtual void writing(VarId id) = 0;
                /// Assign a new id.
            virtual VarId newId() = 0;

                /// Trivial dtor.
            virtual ~IdFromVar();
    };

        /// Interface to provide stringified identifiers for vars that are persistent across multiple
        /// serializations and deserializations, such as GUIDs or unique names from a pre-existing source.
        ///
        /// The stringified identifiers have no inherent meaning, other than providing a stable
        /// human-readable moniker for a variable in text serialization formats.
    class HK_EXPORT_COMMON PersistentIdProvider
    {
    public:
        HK_DECLARE_CLASS(PersistentIdProvider, ReflectVirtualBase);

            /// Retrieves the persistent ID for a var, if one exists; otherwise, the buffer idOut is left empty.
        virtual void getPersistentId(const hkReflect::Var& var, hkStringBuf& idOut) const = 0;
    };


    typedef hkTuple<const hkReflect::Type*, TypeId> TypeAndId;

        /// Interface to write type data
    class HK_EXPORT_COMMON TypeWriter : public hkReferencedObject
    {
        public:
                /// Given a source type, return the type to be written and an id.
                /// The returned type may be different to the the input type when writing
                /// a packfile for a foreign target for instance.
            virtual TypeAndId writeType(_In_ const hkReflect::Type* type) = 0;
                /// Return the highest id which has been written.
            virtual TypeId getHighestWrittenId() const = 0;
    };

        /// Interface to writing a serialized stream.
        /// Normally this class is used only indirectly via hkSerialize::Save.
        /// Streams consist of one or more bundles. Each bundle contains zero or more
        /// types and zero or more objects. Types are shared across bundles and are only written once.
        /// Objects within a bundle cannot reference outside its bundle except indirectly. E.g. via
        /// an export or import note attached to the object.
    class HK_EXPORT_COMMON WriteFormat : public hkReferencedObject
    {
    public:
        

        virtual ~WriteFormat();

            /// Begin a new bundle.
        virtual void beginBundle(_In_ hkIo::WriteBuffer* writeBuffer) = 0;
            /// End a bundle.
        virtual void endBundle() =0;

            /// Write the given object into the current bundle.
        virtual hkResult write(const hkReflect::Var& var, IdFromVar& idFromVar) = 0;
            /// Attach the given note to the object with the given id.
        virtual hkResult writeNote(VarId vid, const hkReflect::Var& note, IdFromVar& idFromVar) = 0;

            /// Request writing types to a separate stream.
            /// If the format supports this feature, instead of storing types within the normal output
            /// stream, the types are stored externally. If several bundles (files) are written between
            /// calls to beginTypeCompendium and endTypeCompendium, this can considerably reduce the
            /// sizes of the bundles.
        virtual void beginTypeCompendium(const hkIo::Detail::WriteBufferAdapter& sink);
            /// Finish the type compendium.
        virtual void endTypeCompendium();

            /// Enable multiple bundles to be written. Each bundle reuses the types, but no
            /// data from the previous bundles.
        virtual hkResult enableMultiBundle();

            /// Create the default write format.
        static hkRefNew<WriteFormat> createDefault();
    };

        /// Interface to reading serialized data.
    class HK_EXPORT_COMMON ReadFormat : public hkReferencedObject
    {
        public:
                /// Read the data from the given stream.
                /// The returned bundle is valid for the lifetime of the ReadFormat instance.
                /// The data underlying in the returned bundle must be assumed to be opaque and does not
                /// match the native types. It is only safe to access it through the hkReflect::Var interface
            virtual hkViewPtr<Bundle> read(hkIo::ReadBuffer& rb) = 0;

                /// The same as "read()" except from a buffer.
                /// In this case, as much of the buffer as possible is used without copying (generally the data contents for binary files).
                /// Thus the bundle is dependent on the lifetimes of both the ReadFormat AND the buffer.
            virtual hkViewPtr<Bundle> view(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_opt_ hkUlong* usedOut) = 0;

                /// Attempt to cast the objects in the buffer as native objects.
                /// This will only work if the buffer has been created by a WriteFormat which supports it
                /// AND if the endianness, pointer size, structure layouts are an exact match.
                /// The ReadFormat may be destroyed after this method returns.
            virtual hkViewPtr<Bundle> inplace(_In_reads_bytes_(bufLen) const void* buf, hkUlong bufLen, _Out_ hkUlong* usedOut, _In_ const hkReflect::TypeReg* typeReg);
    };


        /// Utility class to build a bundle (self-contained tree of objects) for serialization.
    class HK_EXPORT_COMMON BundleBuilder : private IdFromVar
    {
    public:
            /// Create a builder with a format and an initial contents and an output.
        BundleBuilder(_In_ WriteFormat* format, _In_ hkIo::WriteBuffer* buf);

        virtual ~BundleBuilder();

            /// Write the given object.
        template<typename T>
        void add(_In_ const T* t);

            /// Write the given object.
        void add(const hkReflect::Var& var);

            /// Write the given object instead of orig
        void addSubstitution(const hkReflect::Var& var, const hkReflect::Var& orig);

            /// Attach a note to an object.
            /// A note can be any object, the fact that it is a note will be encoded in the Bundle when calling this
            /// method. The note may contain arrays and strings, but must not contain pointers.
        void addNote(const hkReflect::Var& var, const hkReflect::Var& note);

            /// Label a previously written object
        void addExport(const hkReflect::Var& var, _In_z_ const char* name);

            /// Write a symbolic reference.
            /// The object will not be written in the file (unless it has already been written).
        void addImport(const hkReflect::Var& var, _In_z_ const char* name);

            /// Associate an ImportType with a var
            /// The object will not be written in the file (unless it has already been written).
        template<typename ImportType>
        void addGenericImport(const hkReflect::Var& var, const ImportType& import);

            /// Write an empty value for the given var.
        void addEmpty(const hkReflect::Var& var);

            ///
        bool nextUnresolved();

            /// Get the next unresolved item.
        hkReflect::Var currentUnresolved();

            /// Register callbacks. CbData is passed through to the callback functions.
            /// See hkSerialize::Save::withCallbacks.
        void setCallbacks(_In_ void* cbdata, Save::VarCallback vcb, Save::PtrCallback pcb);

            /// Recursively add until no more unresolved references.
        hkResult recursiveAdd();

    private:

        BundleBuilder(const BundleBuilder&);
        void operator=(const BundleBuilder&);

        virtual VarId pointer(const hkReflect::PointerVar& ptr) HK_OVERRIDE;
        virtual VarId lookup(const hkReflect::Var& var) HK_OVERRIDE;
        virtual void writing(VarId id) HK_OVERRIDE;
        virtual VarId newId() HK_OVERRIDE;
        void init(const hkReflect::Var& top);
        void write(VarId id);

        typedef hkHashMap<hkReflect::Var, int> IdFromKey;
        WriteFormat* m_format;
        IdFromKey m_idFromKey;
        int m_idxUnresolved;
        Save::PtrCallback m_ptrCallback;
        Save::VarCallback m_varCallback;
        void* m_cbData;

        struct Item
        {
            void set(const hkReflect::Var& var, bool weak);
            hkReflect::Var m_var;
            bool m_weak;
            bool m_written;
            VarId m_annotated;
        };
        hkArray<Item> m_items; // All items encountered so far.
        hkArray<VarId> m_writeQueue;

        hkArray< hkScopedPtr<Note::Export> > m_exports;
        hkArray< hkReflect::Var > m_imports;
        hkHashSet<void*> m_imported; // All the imported objects so far.
    };

        /// Collects notes on objects and pointers.
    class NoteHandler
    {
        public:
                /// Collect all the notes on a deserialized object. Called before cloning the object to native.
                /// dst and src will be empty in case of import notes.
                /// Gives ownership of the note objects to the NoteHandler. The NoteHandler should keep the notes alive
                /// until the current Bundle has been deserialized (they might be passed to subsequent calls to
                /// atPointerNote).
                
            virtual void addNotes(const hkArrayView<hkReflect::Var>& notes, hkReflect::Var& dst, hkReflect::Var& src) = 0;

                /// Called when a note is found on a pointer about to be cloned to native. Can change dst, src and
                /// their contents. If true is returned the pointer will be cloned, otherwise it will be skipped.
            virtual bool atPointerNote(const hkReflect::Var& note, hkReflect::PointerVar& dst, hkReflect::PointerVar& src) = 0;
    };
}

#include <Common/Base/Serialize/Core/hkSerializeCore.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
