// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Thread/ReadersWriterLock/hkReadersWriterLock.h>
#include <Common/Base/Container/StringMap/hkStringMap.h>
#include <Common/Base/Container/Set/hkSet.h>
#include <Common/Base/Container/PointerMap/hkPointerMap.h>

namespace hkSerialize { class Bundle; }

namespace hkReflect
{
    namespace Version
    {
        class PatchAdder;
        class PatchRemover;
        struct PatchInfo;
        class PatchSet;
        class PatchInfoCache;
        // This should be accessed through hkReflect::Version::PatchAdder or PatchSet, not directly. They will handle the necessary locking

        class HK_EXPORT_COMMON PatchRegistry : public hkReferencedObject
        {
        public:
            HK_DECLARE_CLASS(PatchRegistry, New, Singleton);

            PatchRegistry();
            virtual ~PatchRegistry();

            void addPatchAndInvalidate(_In_ const PatchInfo* p);
            void removePatchAndInvalidate(_In_ const PatchInfo* p);

            hkResult recomputePatchDependencies() const;
            hkBool32 hasUncomputedDependencies() const;

            hkReadersWriterLock& getLock() { return m_lock; }

            // All of these can only be done from within the read lock
            hkArrayView<const PatchInfo*> getPatches() const { return m_patchInfos; }
            hkUint64 getUid(_In_z_ const char* name, int ver) const;
            int getPatchAddedIndex(_In_z_ const char* name) const;
            int getPatchAddedIndexFromLaterVersion(hkUint64 uid) const;
            _Ret_z_ const char* getClassName(hkUint64 uid) const;
            hkInt32 getClassVersion(hkUint64 uid) const;

            HK_INLINE int getPatchIndex(hkUint64 uid) const
            {
                return m_patchIndexFromUid.getWithDefault(uid, -1);
            }

            // TODO: Why is this necessary?? Seems over complicated
            HK_INLINE int getPatchAddedIndex(hkUint64 uid) const
            {
                return m_classAddedPatchIndexFromUid.getWithDefault(uid, -1);
            }

            HK_INLINE _Ret_notnull_ const PatchInfo* getPatch(int patchIndex) const
            {
                return m_patchInfos[patchIndex];
            }

            void clear();
            void clearPatchesStartingWith(_In_z_ const char* prefix);

        private:

            static hkBool32 isValidPatch(_In_ const PatchInfo* patch);
            int findLastPatchIndexForUid(hkUint64 uid, hkBool32 allowRenames = false) const;

            hkReadersWriterLock m_lock;
            class UidFromClassVersion;
            UidFromClassVersion* m_uidFromClassVersion;
            mutable hkArray<const PatchInfo*> m_patchInfos;
            mutable hkPointerMap<hkUint64, hkInt32> m_patchIndexFromUid;
            mutable hkPointerMap<hkUint64, hkInt32> m_classAddedPatchIndexFromUid;
            mutable hkStringMap<hkInt32> m_firstPatchIndexFromName;
            mutable hkPointerMap<hkUint64, int> m_classAddedPatchIndexFromLaterPatchUid; 
            // Related to patching though not strictly patches:
            mutable hkStringMap<const char*> m_versioningTypeNameFixup;
        };

        // Add or remove patches to a type registry, by default the compiled-in one
        // This is a RIIA object, it will write-lock the registry
        class HK_EXPORT_COMMON PatchHandler
        {
        public:
            PatchHandler(hkReflect::Version::PatchRegistry& reg);
            virtual ~PatchHandler();

            virtual void handlePatch(_In_ const PatchInfo* p) = 0;

        protected:
            hkReflect::Version::PatchRegistry& m_reg;
            hkWriteLock m_lock;
        };

        class HK_EXPORT_COMMON PatchAdder : public PatchHandler
        {
        public:
            PatchAdder(hkReflect::Version::PatchRegistry& reg = hkReflect::Version::PatchRegistry::getInstance()); // XXX Pointer and generate in cpp
            virtual ~PatchAdder() {}

            virtual void handlePatch(_In_ const PatchInfo* p);
        };

        class HK_EXPORT_COMMON PatchRemover : public PatchHandler
        {
        public:
            PatchRemover(hkReflect::Version::PatchRegistry& reg = hkReflect::Version::PatchRegistry::getInstance()); // XXX Pointer and generate in cpp
            virtual ~PatchRemover() {}

            virtual void handlePatch(_In_ const PatchInfo* p);
        };

        // Caches patch information for a certain set of types
        // If you modify any source types, this is invalid
        class HK_EXPORT_COMMON PatchInfoCache : public hkReferencedObject
        {
        public:
            PatchInfoCache();

            void addPatch(int index);

            // Returns null if we are actually entering the type, the indices concerned if we are not
            _Ret_maybenull_ hkArray<int>* enterTypePointerIfUnseen(_In_ const hkReflect::Type* type);
            void leaveTypePointer();

        protected:
            hkArray< hkArray< int> > m_patchesNeeded;
            hkPointerMap<const hkReflect::Type*, int> m_indexFromTypePointer;
            // As we add patch indices, all of these types get them
            hkArray<int> m_currentActiveTypes;
        };

        // This is a RIIA object, it will read-lock the registry
        // It describes a subset of the patches in the registry
        class HK_EXPORT_COMMON PatchSet : public hkReferencedObject
        {
        public:

            HK_DECLARE_CLASS(PatchSet, New);

            enum NamedVersionBehavior
            {
                USE_OLDEST_VERSION,
                USE_GIVEN_TYPES_ONLY
            };

            PatchSet(hkReflect::Version::PatchRegistry& reg = hkReflect::Version::PatchRegistry::getInstance()
                , _In_opt_ hkReflect::TypeReg* typeReg = HK_NULL
                , _In_opt_ hkReflect::Version::PatchInfoCache* patchInfoCache = HK_NULL
                , NamedVersionBehavior b = USE_OLDEST_VERSION);
            virtual ~PatchSet();

            _Ret_maybenull_ const PatchInfo* addPatchFor(_In_ const hkReflect::Type* t) { return addPatchFor(t->getName(), t->getVersion()); }
            _Ret_maybenull_ const PatchInfo* addPatchFor(_In_opt_z_ const char* name, int ver);

            // Name is not sufficient, a class can have two independent existences with a different set of version numbers
            _Ret_maybenull_ const PatchInfo* addClassAddedPatchFor(_In_z_ const char* name, int version);

            hkResult addPatchesNeededFor(const hkArrayView<const hkReflect::Type*>& types);
            hkResult addPatchesNeededFor(_In_z_ const char* name, int version);

            // Sorts the patch list if needed and returns it
            hkArrayView<const hkReflect::Version::PatchInfo*> getPatchList();

            bool isNotEmpty() const { return /*(m_patchIndices.getSize() > 0)*/m_containsCompatPatches || m_containsV2Patches; }
            bool containsCompatPatches() const { return m_containsCompatPatches; }
            bool containsV2Patches() const { return m_containsV2Patches; }

            // For debug checks, normally this should only add the patches actively in use
            void addAllPatchesFromRegistry();
            hkUint64 getUid(_In_z_ const char* typeName, int typeVersion) const;
            const PatchInfo* getPatch(hkUint64 uid) const;

            // Temp
            hkReflect::Version::PatchRegistry& getReg() const { return m_reg; }

        private:
            hkResult addPatchesNeededFor(_In_opt_ const hkReflect::Type* t, hkSet<const hkReflect::Type*>& alreadyDone);

            hkSet<int> m_patchIndices;
            hkStringMap<int> m_oldestVersionEncountered;
            hkArray<const hkReflect::Version::PatchInfo*> m_patchList;

            hkRefPtr<hkReflect::Version::PatchInfoCache> m_cache;
            hkReflect::Version::PatchRegistry& m_reg;
            hkReflect::TypeReg* m_typeLookup;
            hkReadLock m_lock;
            hkArray<hkStringPtr> m_names;
            bool m_containsCompatPatches;
            bool m_containsV2Patches;
            NamedVersionBehavior m_namedVersionBehavior;
        };

    }
}

typedef hkReflect::Version::PatchHandler hkVersionPatchManager;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
