// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Version/hkReflectPatchRegistry.h>
#include <Common/Base/Serialize/Util/hkSerializeMultiMap.h> // Shouldn't do this but it seems common
#include <Common/Base/Container/Set/hkSet.h>
#include <Common/Base/Serialize/Core/hkSerializeCore.h>
#include <Common/Base/Reflect/Version/hkReflectVersionPatches.h>
#include <Common/Base/Reflect/Version/hkModifiableTypeSet.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/Object/hkSingleton.h>

#define DEBUG_LOG_IDENTIFIER "ver.PatchRegistry"
#include <Common/Base/System/Log/hkLog.hxx>

#include <stdio.h>

HK_SINGLETON_IMPLEMENTATION(hkReflect::Version::PatchRegistry);


#   define LOG(A)
#   define TRACE(A)

namespace
{
    hkBool32 findPatchIndexInDependencies(int indexToFind, int startIndex, const hkSerializeMultiMap<int, int>& incoming, hkPointerMap<int, int>& done)
    {
        if (done.hasKey(startIndex) == false)
        {
            done.insert(startIndex, 0);
            for (int it = incoming.getFirstIndex(startIndex); it != -1; it = incoming.getNextIndex(it))
            {
                int foundIndex = incoming.getValue(it);
                if (indexToFind == foundIndex)
                {
                    return true;
                }
                if (findPatchIndexInDependencies(indexToFind, foundIndex, incoming, done))
                {
                    return true;
                }
            }
        }
        return false;
    }

    _Ret_z_ const char* reportPendingDependencies(const hkArray<int>& pending, const hkArray<const hkReflect::Version::PatchInfo*>& patchInfos, hkStringBuf& dependencies)
    {
#define NAME(IDX) patchInfos[(IDX)]->oldName ? patchInfos[(IDX)]->oldName : "ADD", patchInfos[(IDX)]->newName ? patchInfos[(IDX)]->newName : "REMOVE"
#define VERS(IDX) patchInfos[(IDX)]->oldVersion, patchInfos[(IDX)]->newVersion

        dependencies.clear();
        for (int i = 0; i < pending.getSize(); ++i)
        {
            dependencies.appendPrintf("%s-%s(%x-%x)...", NAME(pending[i]), VERS(pending[i]));
        }
        return dependencies;
#undef NAME
#undef VERS
    }

    int walkDependencies(int curIndex, hkArray<int>& order, const hkSerializeMultiMap<int, int>& incoming, int counter, hkArray<int>& pending, const hkArray<const hkReflect::Version::PatchInfo*>& patchInfos)
    {
        enum Status { UNSEEN = -1, PENDING = -2 /*DONE >= 0 */ };

        if (order[curIndex] == UNSEEN)
        {
            order[curIndex] = PENDING;
            pending.pushBack(curIndex);
            for (int it = incoming.getFirstIndex(curIndex);
                it != -1;
                it = incoming.getNextIndex(it))
            {
                int precedingIndex = incoming.getValue(it);
                switch (order[precedingIndex])
                {
                case UNSEEN:
                    counter = walkDependencies(precedingIndex, order, incoming, counter, pending, patchInfos);
                    break;
                case PENDING:
                {
                    pending.pushBack(precedingIndex);
                    hkStringBuf deps;
                    HK_ASSERTV(0x6171e278, false, "Circular: {}", reportPendingDependencies(pending, patchInfos, deps));
                    pending.popBack();
                    break;
                }
                default:
                    break;
                }
            }
            pending.popBack();
            order[curIndex] = counter++;
        }
        return counter;
    }
}

namespace hkReflect
{
    namespace Version
    {
        class PatchRegistry::UidFromClassVersion
        {
        public:
            HK_DECLARE_CLASS(UidFromClassVersion, NewManual);
            ~UidFromClassVersion();
            hkUint64 get(_In_z_ const char* name, int ver) const;
            _Ret_z_ const char* getName(hkUint64 uid) const;
            int getVersion(hkUint64 uid) const;

        private:
            _Ret_maybenull_z_ const char* cache(_In_z_ const char* name) const;
            mutable hkStringMap<int> m_indexFromName;
            mutable hkArray<const char*> m_names;
            mutable hkStringMap<char*> m_cachedNames;
            mutable hkCriticalSection m_lock;
        };

        PatchRegistry::UidFromClassVersion::~UidFromClassVersion()
        {
            for (hkStringMap<const char*>::Iterator it = m_cachedNames.getIterator(); m_cachedNames.isValid(it); it = m_cachedNames.getNext(it))
            {
                char* value = m_cachedNames.getValue(it);
                HK_MEMORY_TRACKER_REMOVE_RAW(value);
                hkDeallocate<char>(value);
            }
        }

        hkUint64 PatchRegistry::UidFromClassVersion::get(_In_z_ const char* name, int ver) const
        {
            hkCriticalSectionLock l(&m_lock);
            HK_ASSERT_NO_MSG(0x345e3567, m_indexFromName.getSize() == m_names.getSize());
            const char* cachedName = cache(name);
            int nid = m_indexFromName.getOrInsert(cachedName, m_names.getSize());
            if (nid == m_names.getSize())
            {
                m_names.pushBack(cachedName);
            }
            return (hkUint64(ver) << 32) | unsigned(nid);
        }

        _Ret_z_ const char* PatchRegistry::UidFromClassVersion::getName(hkUint64 uid) const
        {
            hkCriticalSectionLock l(&m_lock);

            int idx = int(uid);
            return m_names[idx];
        }

        int PatchRegistry::UidFromClassVersion::getVersion(hkUint64 uid) const
        {
            return int(uid >> 32);
        }

        _Ret_maybenull_z_ const char* PatchRegistry::UidFromClassVersion::cache(_In_z_ const char* name) const
        {
            hkCriticalSectionLock l(&m_lock);

            if (name)
            {
                char* cached = m_cachedNames.getWithDefault(name, HK_NULL);
                if (cached == HK_NULL)
                {
                    cached = hkString::strDup(name);
                    HK_MEMORY_TRACKER_ADD_RAW("buffer_hkVersionPatchSet::UidFromClassVersion", cached, hkString::strLen(cached)+1);
                    m_cachedNames.insert(cached, cached);
                }
                return cached;
            }
            return HK_NULL;
        }

        HK_MEMORY_TRACKER_MANUAL_BEGIN(3, PatchRegistry::UidFromClassVersion);
            HK_MEMORY_TRACKER_MANUAL_FIELD(m_indexFromName, hkStringMap<int>);
            HK_MEMORY_TRACKER_MANUAL_FIELD(m_names, hkArray<const char*>);
            HK_MEMORY_TRACKER_MANUAL_FIELD(m_cachedNames, hkStringMap<char*>);
        HK_MEMORY_TRACKER_MANUAL_END();


        
        PatchRegistry::PatchRegistry() : m_uidFromClassVersion(HK_NULL)
        {
            m_uidFromClassVersion = new UidFromClassVersion();
        }

        PatchRegistry::~PatchRegistry()
        {
            delete m_uidFromClassVersion;
        }

        void PatchRegistry::addPatchAndInvalidate(_In_ const PatchInfo* p)
        {
            if (!isValidPatch(p))
            {
                return;
            }

            m_patchInfos.pushBack(p);
            m_patchIndexFromUid.clear();
            m_firstPatchIndexFromName.clear();
        }

        namespace
        {
            inline bool stringsMatchOrNull(_In_opt_z_ const char* strA, _In_opt_z_ const char* strB)
            {
                if (strA && strB)
                {
                    return hkString::strCmp(strA, strB) == 0;
                }
                else
                {
                    return strA == strB;
                }
            }

            inline bool patchesMatch(_In_ const hkReflect::Version::PatchInfo* pA, _In_ const hkReflect::Version::PatchInfo* pB)
            {
                return (pA->oldVersion == pB->oldVersion) && (pA->newVersion == pB->newVersion) &&
                    stringsMatchOrNull(pA->oldName, pB->oldName) && stringsMatchOrNull(pA->newName, pB->newName);
            }
        }

        void PatchRegistry::removePatchAndInvalidate(_In_ const PatchInfo* p)
        {
            // The patches are local statics, we can't just compare addresses, the remove ones will have different
            // addresses to the add ones
            if (!isValidPatch(p))
            {
                return;
            }

            for (int i = 0; i < m_patchInfos.getSize(); i++)
            {
                if (patchesMatch(m_patchInfos[i], p))
                {
                    m_patchInfos.removeAt(i);
                    i--;
                }
            }
            m_patchIndexFromUid.clear();
            m_firstPatchIndexFromName.clear();
        }


        hkBool32 PatchRegistry::isValidPatch(_In_ const PatchInfo* patch)
        {
            if (patch->oldVersion == -1/*HK_CLASS_ADDED*/)
            {
                if (patch->oldName != HK_NULL)
                {
                    HK_ASSERTV(0x21be641e, false, "Found incorrectly defined patch. Add new class patch (HK_CLASS_ADDED) has 'source' class name  set to '{}', but must be HK_NULL.", patch->oldName);
                    return false;
                }
                if (patch->newVersion == -1/*HK_CLASS_REMOVED*/)
                {
                    HK_ASSERT(0x66920794, false, "Found incorrectly defined patch. Add new class patch (HK_CLASS_ADDED) has 'destination' class version set to HK_CLASS_REMOVED.");
                    return false;
                }
                if (patch->newName == HK_NULL)
                {
                    HK_ASSERT(0x60dc0f50, false, "Found incorrectly defined patch. Add new class patch (HK_CLASS_ADDED) has 'destination' class name set to null.");
                    return false;
                }
            }
            else if (patch->newVersion == -1/*HK_CLASS_REMOVED*/)
            {
                if (patch->oldName == HK_NULL)
                {
                    HK_ASSERT(0x7178f8cf, false, "Found incorrectly defined patch. Remove class patch (HK_CLASS_REMOVED) has 'source' class name set to null.");
                    return false;
                }
                if (patch->newName != HK_NULL)
                {
                    HK_ASSERTV(0x69e2d0b3, false, "Found incorrectly defined patch. Remove class patch (HK_CLASS_REMOVED) has 'destination' class name  set to '{}', but must be HK_NULL.", patch->newName);
                    return false;
                }
            }
            else
            {
                if (patch->oldName == HK_NULL)
                {
                    HK_ASSERT(0x42f53922, false, "Found incorrectly defined patch. Update class patch has 'source' class name set to HK_NULL.");
                    return false;
                }
            }
            return true;
        }

        hkBool32 PatchRegistry::hasUncomputedDependencies() const
        {
            return m_patchIndexFromUid.getSize() == 0 && m_patchInfos.getSize();
        }

        hkResult PatchRegistry::recomputePatchDependencies() const
        {
            hkSerializeMultiMap<hkUint64, hkInt32> idxDstFromUid;
            const UidFromClassVersion* uidFromTuple = m_uidFromClassVersion;
            hkSerializeMultiMap<const char*, hkInt32> lastClassPatchIndexesFromClassName;

            {
                m_patchIndexFromUid.clear();
                m_classAddedPatchIndexFromUid.clear();

                for (int i = 0; i < m_patchInfos.getSize(); ++i)
                {
                    const PatchInfo* p = m_patchInfos[i];
                    if(p)
                    {
                        if (!isValidPatch(p))
                        {
                            return HK_FAILURE;
                        }
                        if (p->oldVersion != -1 /*HK_CLASS_ADDED*/)
                        {
                            hkUint64 uid = m_uidFromClassVersion->get(p->oldName, p->oldVersion);
                            if (m_patchIndexFromUid.hasKey(uid) == false)
                            {
                                m_patchIndexFromUid.insert(uid, i);
                            }
                            else
                            {
                                HK_ASSERTV(0x6a12698e, false, "Found duplicated patch for class '{}' (version {}).\n"
                                    "Make sure you register the patch once with hkVersionPatchManager.",  p->oldName, p->oldVersion);
                                return HK_FAILURE;
                            }
                            if (p->newVersion == -1 /*HK_CLASS_REMOVED*/ || (p->newName && hkString::strCmp(p->oldName, p->newName) != 0 /*renamed*/))
                            {
                                lastClassPatchIndexesFromClassName.insert(p->oldName, i); // last known patch index for p->oldName class
                            }
                        }
                        else
                        {
                            hkUint64 uid = m_uidFromClassVersion->get(p->newName, p->newVersion);
                            if (m_classAddedPatchIndexFromUid.hasKey(uid) == false)
                            {
                                m_classAddedPatchIndexFromUid.insert(uid, i);
                            }
                            else
                            {
                                HK_ASSERTV(0x6b12698f, false, "Found duplicated patch for new class '{}' (version {}).\n"\
                                    "Make sure you register the new class patch once with hkVersionPatchManager.", p->newName, p->newVersion);
                                return HK_FAILURE;
                            }
                        }
                        if (p->newVersion != -1/*HK_CLASS_REMOVED*/)
                        {
                            hkUint64 newUid = uidFromTuple->get(p->newName ? p->newName : p->oldName, p->newVersion);
                            idxDstFromUid.insert(newUid, i);
                        }
                    }
                }
            }

            TRACE(("There are %i patches to consider", m_patchInfos.getSize() - 1));
            hkStringMap<int> patchIndexFromClassName; // used to check conflicts among patch groups with the same class name
            hkSerializeMultiMap<int, int> incoming; // patch index key comes after patch index value
            hkSerializeMultiMap<int, int> multiIncoming; // patch index key comes after patch index value for patch groups with the same class name

            for (int patchIndex = 0; patchIndex < m_patchInfos.getSize(); ++patchIndex)
            {
                const PatchInfo* pinfo = m_patchInfos[patchIndex];
                const char* curName = pinfo->oldName;
                int curVersion = pinfo->oldVersion;
                if (pinfo->oldVersion == -1 /*HK_CLASS_ADDED*/)
                {
                    // new class patch (may have dependencies)
                    curName = pinfo->newName;
                    curVersion = pinfo->newVersion;
                    if (!curName || curVersion == -1 /*HK_CLASS_REMOVED*/)
                    {
                        HK_ASSERT(0x36b2fc1e, false, "Found incorrectly defined patch. Class cannot be added and removed using one patch.");
                        return HK_FAILURE;
                    }
                }
                hkUint64 curUid = uidFromTuple->get(curName, curVersion);
                int curIdx;
                if (pinfo->oldVersion == -1 /*HK_CLASS_ADDED*/)
                {
                    curIdx = m_classAddedPatchIndexFromUid.getWithDefault(curUid, -1);
                    HK_ASSERT_NO_MSG(0x14f5428f, curIdx == patchIndex);
                }
                else
                {
                    curIdx = m_patchIndexFromUid.getWithDefault(curUid, -1);
                    // update class patch
                    HK_ASSERT_NO_MSG(0x14f5427f, curIdx == patchIndex);
#if defined(TRACE)
#   define NAME(IDX) (m_patchInfos[(IDX)]->oldName ? m_patchInfos[(IDX)]->oldName : m_patchInfos[(IDX)]->newName)
#   define VERS(IDX) (m_patchInfos[(IDX)]->oldName ? m_patchInfos[(IDX)]->oldVersion : m_patchInfos[(IDX)]->newVersion)
#endif
                    for (int it = idxDstFromUid.getFirstIndex(curUid);
                        it != -1;
                        it = idxDstFromUid.getNextIndex(it))
                    {
                        int dstIdx = idxDstFromUid.getValue(it);
                        // patch producing me comes first
                        incoming.insert(curIdx, dstIdx);
                        //TRACE(("\t%s_%x -> %s_%x [color=\"direct\" constraint=false]\n", NAME(dstIdx), VERS(dstIdx), NAME(curIdx), VERS(curIdx) ));
                        TRACE(("\t%d (%d)\n", curIdx, dstIdx));
                    }
                }

                // check dependecies
                for (int ci = 0; ci < pinfo->numComponent; ++ci)
                {
                    const PatchInfo::Component& pc = pinfo->component[ci];
                    if (const PatchInfo::DependsPatch* dp = pc.asDependsPatch())
                    {
                        hkUint64 dpUid = uidFromTuple->get(dp->name, dp->version);
                        {
                            // patch modifying my dependency must come after me
                            int dpIdx = m_patchIndexFromUid.getWithDefault(dpUid, -1);
                            if (dpIdx != -1)
                            {
                                // I must come before patch for my dependency
                                incoming.insert(dpIdx, curIdx);
                                //TRACE(("\t%s_%x -> %s_%x [label=\"dep comes after\"]\n", NAME(curIdx), VERS(curIdx), NAME(dpIdx), VERS(dpIdx) ));
                                TRACE(("\t%d (%d)\n", dpIdx, curIdx));
                            }
                        }
                        // I must come after all patches producing my dependency UNLESS they are combines
                        for (int it = idxDstFromUid.getFirstIndex(dpUid);
                            it != -1;
                            it = idxDstFromUid.getNextIndex(it))
                        {
                            int dpOutputIdx = idxDstFromUid.getValue(it);
                            if(!m_patchInfos[dpOutputIdx]->isCombine())
                            {
                                incoming.insert(curIdx, dpOutputIdx); // I come after patches producing my dependency
                                //TRACE(("\t%s_%x -> %s_%x [color=\"he helps before\" constraint=false]\n", NAME(dpOutputIdx), VERS(dpOutputIdx), NAME(curIdx), VERS(curIdx) ));
                                TRACE(("\t%d (%d)\n", curIdx, dpOutputIdx));
                            }
                            else
                            {
                                TRACE(("Skipping %s-%x as it is a combine", m_patchInfos[dpOutputIdx]->oldName, m_patchInfos[dpOutputIdx]->oldVersion));
                            }
                        }
                    }
                }
                // check groups
                if (pinfo->oldVersion == -1 /*HK_CLASS_ADDED*/ && lastClassPatchIndexesFromClassName.getFirstIndex(curName) != -1)
                {
                    int it = lastClassPatchIndexesFromClassName.getFirstIndex(curName);
                    if (it != -1)
                    {
                        int curLastIdx = findLastPatchIndexForUid(curUid);
                        if (curLastIdx == -1) // -1 = class still exists, so this patch must be after all other group patches with the same class name
                        {
                            int lastPatchIndex = patchIndexFromClassName.getWithDefault(curName, -1);
                            if (lastPatchIndex != -1)
                            {
                                HK_ASSERTV(0x4a21c0fc, false, "Found conflict in registered patches for class '{}'.\n"
                                    "The class patch for version {} conflicts with patch for new class version {}.",
                                    curName, m_patchInfos[lastPatchIndex]->newVersion, curVersion);
                                return HK_FAILURE;
                            }
                            patchIndexFromClassName.insert(curName, patchIndex);
                        }
                        hkSerializeMultiMap<int, int>& dependencyMap = curLastIdx == -1 ? incoming : multiIncoming;
                        for (; it != -1; it = lastClassPatchIndexesFromClassName.getNextIndex(it))
                        {
                            int prevIdx = lastClassPatchIndexesFromClassName.getValue(it);
                            if (curLastIdx != prevIdx)
                            {
                                dependencyMap.insert(curIdx, prevIdx); // I come after all patches for previously removed class
                                if (curLastIdx == -1)
                                {
                                    TRACE((".\t%d (%d)\n", curIdx, prevIdx));
                                }
                            }
                        }
                    }
                }
            }
            // update 'incoming' with patch group dependencies
            for (hkPointerMap<int, int>::Iterator it = multiIncoming.m_indexMap.getIterator(); multiIncoming.m_indexMap.isValid(it); it = multiIncoming.m_indexMap.getNext(it))
            {
                int beginPatchIndex = multiIncoming.m_indexMap.getKey(it);
                for (int mi = multiIncoming.getFirstIndex(beginPatchIndex); mi != -1; mi = multiIncoming.getNextIndex(mi))
                {
                    hkPointerMap<int, int> done;
                    // make sure there is no circular dependency walking backward
                    int startIndex = multiIncoming.getValue(mi);
                    if (!findPatchIndexInDependencies(beginPatchIndex, startIndex, incoming, done))
                    {
                        // add dependency
                        incoming.insert(beginPatchIndex, startIndex);
                        TRACE(("+\t%d (%d)\n", beginPatchIndex, startIndex));
                    }
                }
            }
            TRACE(("}\n"));


            hkArray<const PatchInfo*> allPatchInfos; allPatchInfos.swap(m_patchInfos);
            // 0=< done
            // -1 pending
            // -2 done
            hkArray<int> order; order.setSize(allPatchInfos.getSize(), -1);

            hkArray<int> pending;
            int counter = 0;
            for (int patchIndex = 0; patchIndex < allPatchInfos.getSize(); ++patchIndex)
            {
                counter = walkDependencies(patchIndex, order, incoming, counter, pending, allPatchInfos);
            }
            m_patchInfos.setSize(allPatchInfos.getSize());
            for (int patchIndex = 0; patchIndex < allPatchInfos.getSize(); ++patchIndex)
            {
                int correctIdx = order[patchIndex];
                if (correctIdx < 0)
                {
                    HK_ASSERT(0x3fecb5e3, false, "Found circular dependency in patches.");
                    m_patchInfos.swap(allPatchInfos);
                    return HK_FAILURE;
                }
                m_patchInfos[correctIdx] = allPatchInfos[patchIndex];
            }

            hkArray<int> classAddedPatches;

            {
                m_patchIndexFromUid.clear();
                for (int i = 0; i < m_patchInfos.getSize(); ++i)
                {
                    const PatchInfo* p = m_patchInfos[i];
                    if (p->oldName)
                    {
                        hkUint64 uid = m_uidFromClassVersion->get(p->oldName, p->oldVersion);
                            HK_ASSERT_NO_MSG(0x6c12698e, m_patchIndexFromUid.hasKey(uid) == false);
                        m_patchIndexFromUid.insert(uid, i);
                    }
                    else
                    {
                        // Patch added
                            HK_ASSERT_NO_MSG(0x3754c508, p->newName);
                        classAddedPatches.pushBack(i);
                    }

                    // This stores the oldest patch known for a given type. Patches are now
                    // in a consistent order, so the first patch we see is always the oldest patch
                    if (p->oldName || p->newName)
                    {
                        const char* checkName = p->newName ? p->newName : p->oldName;

                        hkStringMap<hkInt32>::Iterator it = m_firstPatchIndexFromName.findOrInsertKey(checkName, i);
                        // A rename and not an add or remove
                        if ((p->newName != p->oldName) && (p->oldVersion != -1) && (p->newVersion != -1))
                        {
                            // The first patch is the start of the old name, not this one
                            int newVal = m_firstPatchIndexFromName.getWithDefault(p->oldName, i);
                            m_firstPatchIndexFromName.setValue(it, newVal);
                        }
                    }
                }
            }

            for (int i = 0; i < classAddedPatches.getSize(); i++)
            {
                const int startPatchIndex = classAddedPatches[i];
                const PatchInfo* currentPatch = m_patchInfos[startPatchIndex];

                while (currentPatch && currentPatch->newName && (currentPatch->newVersion != -1))
                {
                    hkUint64 uid = m_uidFromClassVersion->get(currentPatch->newName, currentPatch->newVersion);
                    m_classAddedPatchIndexFromLaterPatchUid.insert(uid, startPatchIndex);
                    const int nextIndex = m_patchIndexFromUid.getWithDefault(uid, -1);
                    currentPatch = (nextIndex >= 0) ? m_patchInfos[nextIndex] : HK_NULL;
                }

                // Deleted
                //if (currentPatch && (currentPatch->newVersion == -1))
                //{
                //  hkUint64 uid = m_uidFromClassVersion->get(currentPatch->oldName, currentPatch->newVersion);
                //  m_classAddedPatchIndexFromLaterPatchUid.insert(uid, startPatchIndex);
                //}
            }

            if(m_patchInfos.getSize() && (m_patchInfos.back() == HK_NULL))
            {
                // Don't need this anymore and it confuses it if we later register more patches
                m_patchInfos.popBack();
            }
            return HK_SUCCESS;
        }

        int PatchRegistry::findLastPatchIndexForUid(hkUint64 uid, hkBool32 allowRenames) const
        {
            for (int index = m_patchIndexFromUid.getWithDefault(uid, -1);
                index != -1;
                index = m_patchIndexFromUid.getWithDefault(uid, -1))
            {
                const PatchInfo* p = m_patchInfos[index];
                HK_ASSERT_NO_MSG(0x7f2cd63e, p && p->oldName);
                if (p->newVersion == -1/*HK_CLASS_REMOVED*/
                    || (!allowRenames && p->newName && hkString::strCmp(p->oldName, p->newName) != 0))
                {
                    return index;
                }
                uid = m_uidFromClassVersion->get(p->newName, p->newVersion);
            }
            return -1;
        }

        hkUint64 PatchRegistry::getUid(_In_z_ const char* name, int ver) const
        {
            return m_uidFromClassVersion->get(name, ver);
        }

        int PatchRegistry::getPatchAddedIndex(_In_z_ const char* name) const
        {
            int firstPatchIndex = m_firstPatchIndexFromName.getWithDefault(name, -1);
            return firstPatchIndex;
        }

        int PatchRegistry::getPatchAddedIndexFromLaterVersion(hkUint64 uid) const
        {
            int firstPatchIndex = m_classAddedPatchIndexFromLaterPatchUid.getWithDefault(uid, -1);
            return firstPatchIndex;
        }

        _Ret_z_ const char* PatchRegistry::getClassName(hkUint64 uid) const
        {
            return m_uidFromClassVersion->getName(uid);
        }

        hkInt32 PatchRegistry::getClassVersion(hkUint64 uid) const
        {
            return m_uidFromClassVersion->getVersion(uid);
        }

        void PatchRegistry::clear()
        {
            m_patchInfos.clear();
            m_patchIndexFromUid.clear();

            delete m_uidFromClassVersion;
            m_uidFromClassVersion = new UidFromClassVersion();
        }

        void PatchRegistry::clearPatchesStartingWith(_In_z_ const char* prefix)
        {
            PatchRemover rem(*this);
            const int prefixLen = hkString::strLen(prefix);

            for (int thisPatchIndex = 0; thisPatchIndex < getPatches().getSize(); thisPatchIndex++)
            {
                const hkReflect::Version::PatchInfo* patch = getPatch(thisPatchIndex);
                if (patch->newName && !hkString::strNcmp(prefix, patch->newName, prefixLen))
                {
                    rem.handlePatch(patch);
                    thisPatchIndex--;
                }
                else if (patch->oldName && !hkString::strNcmp(prefix, patch->oldName, prefixLen))
                {
                    rem.handlePatch(patch);
                    thisPatchIndex--;
                }
            }
        }

        PatchHandler::PatchHandler(hkReflect::Version::PatchRegistry& reg) : m_reg(reg), m_lock(m_reg.getLock())
        {
        }

        PatchAdder::PatchAdder(hkReflect::Version::PatchRegistry& reg) : PatchHandler(reg)
        {
        }

        PatchRemover::PatchRemover(hkReflect::Version::PatchRegistry& reg) : PatchHandler(reg)
        {
        }

        PatchHandler::~PatchHandler()
        {
            if (m_reg.hasUncomputedDependencies())
            {
                m_reg.recomputePatchDependencies();
            }
        }

        void PatchAdder::handlePatch(_In_ const PatchInfo* p)
        {
            // Default behaviour is to ignore patches from native types
            // Skip adds and deletes
            if (p->oldName && p->newName)
            {
                if (const hkReflect::Type* t = hkReflect::typeFromName(p->oldName))
                {
                    if (t->getVersion() == p->oldVersion)
                    {
                        Log_Error("Ignoring patch from {}/{} to {}/{} as it attempts to version from a registered type", p->oldName, p->oldVersion, p->newName, p->newVersion);
                        return;
                    }
                }
            }

            m_reg.addPatchAndInvalidate(p);
        }

        void PatchRemover::handlePatch(_In_ const PatchInfo* p)
        {
            m_reg.removePatchAndInvalidate(p);
        }

        PatchSet::PatchSet(hkReflect::Version::PatchRegistry& reg, _In_opt_ hkReflect::TypeReg* typeLookup, _In_opt_ hkReflect::Version::PatchInfoCache* cache, NamedVersionBehavior b)
            : m_cache(cache)
            , m_reg(reg)
            , m_typeLookup(typeLookup)
            , m_lock(reg.getLock())
            , m_containsCompatPatches(false)
            , m_containsV2Patches(false)
            , m_namedVersionBehavior(b)
        {
            if (!m_typeLookup)
            {
                m_typeLookup = hkReflect::getTypeReg();
            }
            if (!m_cache)
            {
                m_cache.setAndDontIncrementRefCount(new hkReflect::Version::PatchInfoCache());
            }
        }

        PatchSet::~PatchSet()
        {
        }

        _Ret_maybenull_ const PatchInfo* PatchSet::addPatchFor(_In_opt_z_ const char* name, int ver)
        {
            if (name)
            {
                const hkUint64 uid = m_reg.getUid(name, ver);
                const int index = m_reg.getPatchIndex(uid);
                if (index >= 0)
                {
                    m_cache->addPatch(index);
                    if (m_patchIndices.insert(index))
                    {
                        // New patch, sorted list is no longer valid
                        const hkReflect::Version::PatchInfo* patch = m_reg.getPatch(index);
                        m_containsCompatPatches = m_containsCompatPatches || patch->isCompatPatch();
                        m_containsV2Patches = m_containsV2Patches || patch->isV2Patch();
                        m_patchList.clear();
                    }
                    return m_reg.getPatch(index);
                }
            }
            return HK_NULL;
        }

        _Ret_maybenull_ const PatchInfo* PatchSet::addClassAddedPatchFor(_In_z_ const char* name, int version)
        {
            if (name)
            {
                hkUint64 uid = m_reg.getUid(name, version);
                const int index = m_reg.getPatchAddedIndexFromLaterVersion(uid);
                if (index >= 0)
                {
                    m_cache->addPatch(index);
                    if (m_patchIndices.insert(index))
                    {
                        // New patch, sorted list is no longer valid
                        const hkReflect::Version::PatchInfo* patch = m_reg.getPatch(index);
                        m_containsCompatPatches = m_containsCompatPatches || patch->isCompatPatch();
                        m_containsV2Patches = m_containsV2Patches || patch->isV2Patch();
                        m_patchList.clear();
                    }
                    return m_reg.getPatch(index);
                }
            }
            return HK_NULL;
        }

        hkArrayView<const hkReflect::Version::PatchInfo*> PatchSet::getPatchList()
        {
            if (m_patchList.getSize())
            {
                return m_patchList;
            }

            m_patchList.clear();

            // Given an array of patch pointers, sort them according to the registry's order
            const int totalNumPatches = m_reg.getPatches().getSize();
            for (int thisPatchIndex = 0; thisPatchIndex < totalNumPatches; thisPatchIndex++)
            {
                if (m_patchIndices.contains(thisPatchIndex))
                {
                    const hkReflect::Version::PatchInfo* patch = m_reg.getPatch(thisPatchIndex);
                    if (patch->isCompatPatch())
                    {
                        m_patchList.pushBack(patch);
                    }
                }
            }

            return (m_patchList.getSize()/* == m_patchIndices.getSize()*/) ?
                hkArrayView<const hkReflect::Version::PatchInfo*>(m_patchList) :
                hkArrayView<const hkReflect::Version::PatchInfo*>();
        }

        hkResult PatchSet::addPatchesNeededFor(const hkArrayView<const hkReflect::Type*>& types)
        {
            hkSet<const hkReflect::Type*> typesAlreadyProcessed;
            hkResult res = HK_SUCCESS;

            for (int i = 0; i < types.getSize(); i++)
            {
                if(const hkReflect::Type* thisType = types[i])
                {
                    addPatchesNeededFor(thisType, typesAlreadyProcessed);
                }
            }

            return res;
        }

        hkUint64 PatchSet::getUid(_In_z_ const char* typeName, int typeVersion) const
        {
            return m_reg.getUid(typeName, typeVersion);
        }

        hkResult PatchSet::addPatchesNeededFor(_In_opt_ const hkReflect::Type* thisType, hkSet<const hkReflect::Type*>& typesAlreadyProcessed)
        {
            if (thisType)
            {
                while (thisType->isDecorator())
                {
                    thisType = thisType->getParent();
                }
            }

            if (hkArray<int>* patches = m_cache->enterTypePointerIfUnseen(thisType))
            {
                // We already know the patches for this name
                for (int i = 0; i < patches->getSize(); i++)
                {
                    const int index = (*patches)[i];
                    if (m_patchIndices.insert(index))
                    {
                        // New patch, sorted list is no longer valid
                        const hkReflect::Version::PatchInfo* patch = m_reg.getPatch(index);
                        m_containsCompatPatches = m_containsCompatPatches || patch->isCompatPatch();
                        m_containsV2Patches = m_containsV2Patches || patch->isV2Patch();

                        m_patchList.clear();
                    }
                }
                return HK_SUCCESS;
            }

            if (const hkReflect::RecordType* rt = thisType->asRecord()) // Temporary maybe
            {
                if (typesAlreadyProcessed.insert(thisType))
                {
                    hkStringBuf fullName;
                    const char* className = thisType->getFullName(fullName);
                    m_names.pushBack(className);
                    const int version = (m_namedVersionBehavior == USE_OLDEST_VERSION ? m_oldestVersionEncountered.getOrInsert(m_names.back(), thisType->getVersion()) : thisType->getVersion());

                    DLOG("Looking at {} version {} [{}]", className, thisType->getVersion(), version);

                    if (addPatchesNeededFor(className, version).isFailure())
                    {
                        DLOG("Patch lookup failed");
                        m_cache->leaveTypePointer();
                        return HK_FAILURE;
                    }

                    for (hkReflect::DeclIter<hkReflect::DataFieldDecl> fieldIterator(rt); fieldIterator.advance();)
                    {
                        const hkReflect::DataFieldDecl f = fieldIterator.current();
                        DLOG("Field {}", f);
                        if (addPatchesNeededFor(f.getType(), typesAlreadyProcessed).isFailure())
                        {
                            DLOG("Patch lookup failed");
                            m_cache->leaveTypePointer();
                            return HK_FAILURE;
                        }
                    }

                    if (const hkReflect::Type* thisTypeParent = thisType->getParent())
                    {
                        DLOG("Parent {}", thisTypeParent);
                        if (addPatchesNeededFor(thisTypeParent, typesAlreadyProcessed).isFailure())
                        {
                            m_cache->leaveTypePointer();
                            return HK_FAILURE;
                        }
                    }
                }
            }

            m_cache->leaveTypePointer();
            return HK_SUCCESS;
        }

        hkResult PatchSet::addPatchesNeededFor(_In_z_ const char* className, int version)
        {
            hkResult result = HK_SUCCESS;

            DLOG("addPatchesNeededFor {}/{}", className, version);
            const hkReflect::Version::PatchInfo* patch = HK_NULL;
            HK_ASSERT_NO_MSG(0xa1e4d33, version >= 0);

            patch = addPatchFor(className, version);

            int lastVersionApplied = version;
            const char* currentTypeName = className;

            while (patch)
            {
                DLOG("Found patch {}[{}] -> {}[{}]", patch->oldName, patch->oldVersion, patch->newName, patch->newVersion);
                if (!patch->isClassAdded())
                // Run dependencies first, on the forward walk
                {
                    for (int componentIndex = patch->numComponent - 1; componentIndex >= 0; componentIndex--)
                    {
                        const hkReflect::Version::PatchInfo::Component& component = patch->component[componentIndex];
                        if (const hkReflect::Version::PatchInfo::DependsPatch* dependsPatch = component.asDependsPatch())
                        {
                            // Run the patches to calculate the size
                            addPatchesNeededFor(dependsPatch->name, dependsPatch->version);
                        }
                    }

                    if (patch->newName && currentTypeName && hkString::strCmp(patch->newName, currentTypeName)) // name has changed
                    {
                        // Need to transform TemplateOldName<X> to TemplateNewName<X> where the patch is just TemplateOldName -> TemplateNewName
                        const char* startArgs = hkString::strStr(currentTypeName, "<");
                        if (startArgs)
                        {
                            hkStringBuf tempStr(patch->newName);
                            tempStr.append(startArgs);
                            m_names.pushBack(tempStr.cString());
                            currentTypeName = m_names.back();
                        }
                        else
                        {
                            currentTypeName = patch->newName;
                        }
                    }
                }

                lastVersionApplied = patch->newVersion;
                patch = patch->newName ? addPatchFor(patch->newName, patch->newVersion) : HK_NULL;
            }

            DLOG("No more patches, last version {}", lastVersionApplied);

            return result;
        }

        void PatchSet::addAllPatchesFromRegistry()
        {
            bool anyPatchesActuallyAdded = false;
            for (int i = 0; i < m_reg.getPatches().getSize(); i++)
            {
                if (m_patchIndices.insert(i))
                {
                    // New patch, sorted list is no longer valid
                    const hkReflect::Version::PatchInfo* patch = m_reg.getPatch(i);
                    m_containsCompatPatches = m_containsCompatPatches || patch->isCompatPatch();
                    m_containsV2Patches = m_containsV2Patches || patch->isV2Patch();

                    anyPatchesActuallyAdded = true; // was false??!
                }
            }
            if (anyPatchesActuallyAdded)
            {
                m_patchList.clear();
            }
        }

        const PatchInfo* PatchSet::getPatch(hkUint64 uid) const
        {
            const int index = m_reg.getPatchIndex(uid);
            if (index >= 0)
            {
                return m_reg.getPatch(index);
            }
            return HK_NULL;
        }

        void PatchInfoCache::addPatch(int index)
        {
            for (int i = 0; i < m_currentActiveTypes.getSize(); i++)
            {
                m_patchesNeeded[m_currentActiveTypes[i]].pushBack(index);
            }
        }

        // Returns null if we are actually entering the type, the indices concerned if we are not
        _Ret_maybenull_ hkArray<int>* PatchInfoCache::enterTypePointerIfUnseen(_In_ const hkReflect::Type* type)
        {
            const int indexOfNewElement = m_patchesNeeded.getSize();
            hkStringMap<int>::Iterator it = m_indexFromTypePointer.findOrInsertKey(type, indexOfNewElement);
            int retrievedIndex = m_indexFromTypePointer.getValue(it);
            if (retrievedIndex == indexOfNewElement)
            {
                // This is a new type
                m_patchesNeeded.expandOne();
                m_currentActiveTypes.pushBack(indexOfNewElement);
                return HK_NULL;
            }
            else
            {
                return &m_patchesNeeded[retrievedIndex];
            }
        }
        void PatchInfoCache::leaveTypePointer()
        {
            m_currentActiveTypes.popBack();
        }

        PatchInfoCache::PatchInfoCache()
        {

        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
