// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Memory/Allocator/Transient/hkTransientAllocator.h>
#include <Common/Base/Container/Set/hkSet.h>
#include <Common/Base/Container/StringMap/hkStringMap.h>
#include <Common/Base/Reflect/Version/hkReflectVersionPatches.h>
#include <Common/Base/Reflect/Version/hkReflectPatchRegistry.h>

class hkModifiableTypeSet;
struct hkPatchDependencies;

class HK_EXPORT_COMMON hkVersionedRecordType
{
public:
    HK_DECLARE_CLASS(hkVersionedRecordType, New);
    enum Status
    {
        STATUS_INVALID = -1,
        STATUS_NOTCOPIED = 0, // Placeholder for a type, name and parent info but nothing else
        STATUS_INPROGRESS = 1, // Started
        STATUS_LOCALLY_COMPLETE = 2, // Local members are correct but not parents
        STATUS_RESOLVED = 3, // Complete type, ready to use
    };

    Status getStatus() const { return m_status; }
    void setStatus(Status s) { m_status = s; }

    hkVersionedRecordType(_In_ hkModifiableTypeSet* owner, _In_ hkReflect::Type* type, int startVersion, Status status, _In_ const hkReflect::Type* sourceType, hkUint64 startingUid);
    ~hkVersionedRecordType();

    hkReflect::Type* getCompletedType() const { HK_ASSERT_NO_MSG(0x41866087, getStatus() == STATUS_RESOLVED); return m_type; }
    // After we have run all of the patches, get the type ready for use
    hkResult finalize(_Inout_ hkMemoryAllocator* allocator, bool copyParentMembers);

    int getNumChildren() const { return m_children.getSize(); }
    _Ret_maybenull_ hkVersionedRecordType* getChild(int index) const { return m_children[index]; }
    void addChild(_In_ hkVersionedRecordType* child) { m_children.pushBack(child); }
    void removeChild(_In_ hkVersionedRecordType* child) { m_children.removeAllAndCopy(child); }
    void removeParent(_In_ hkVersionedRecordType* parent);

    int getNumParents() const { return m_parents.getSize(); }
    _Ret_maybenull_ hkVersionedRecordType* getParent(int index) const { return m_parents[index]; }
    _Ret_maybenull_ hkVersionedRecordType* getCurrentParent() const { return m_parents.getSize() ? m_parents[m_currentParentIndex] : HK_NULL; }
    void addParent(_In_ hkVersionedRecordType* parent);

    // Most of these are only ever called from the type set?
    void applyReversedPatchToPatchedRecord(_In_ const hkReflect::Version::PatchInfo* patch);
    void applyPatchToPatchedRecord(_In_ const hkReflect::Version::PatchInfo* patch);

    void addPatchedFieldFromDescription(_In_z_ const char* fieldName, hkReflect::Version::LegacyType type, _In_z_ const char* typeName, int tuples, _In_opt_ const void* defaultPtr, int insertPoint, const hkPatchDependencies& patchDeps);
    void addPatchedFieldFromTypeInformation(_In_z_ const char* fieldName, _In_ const hkReflect::Type* type);

    _Ret_notnull_ hkReflect::Type* getTypeInternal() const { return m_type; }
    void addTemplateParameter(_In_z_ const char* kindAndName, hkUlong value);
    bool isTemplate() const { return m_templateParameters.getSize() > 0; }
    void renameType(_In_z_ const char* oldName, _In_z_ const char* newName);

    static _Ret_notnull_ hkReflect::Type* copyAndAddOptional(_In_ const hkReflect::Type* type, hkMemoryAllocator& allocator, hkReflect::Optional opt, _In_ const void* val);
    _Ret_z_ const char* getName() const { return m_type->getName(); }
    _Ret_z_ const char* getFullName(hkStringBuf& buf) const { return m_type->getFullName(buf); }
    _Ret_notnull_ const hkReflect::Type* getSourceType() const { return m_sourceType; }
    int getStartVersion() const { return m_startVersion; }
    void invalidateType();

protected:
    struct OldRecordFieldPod;
    _Ret_maybenull_ OldRecordFieldPod* findField(_In_z_ const char* name);

    hkReflect::Type* m_type;
    const hkReflect::Type* m_sourceType;
    int m_startVersion;
    int m_currentParentIndex; // Index into m_parents
    class hkModifiableTypeSet* m_owner;

    struct OldRecordFieldPod
    {
        HK_DECLARE_CLASS(OldRecordFieldPod, New, Pod);

        const char* m_name;
        hkReflect::Decl::DeclFlags m_flags;
        hkUint16 m_offset;
        const hkReflect::Type* m_type;
    };

    hkArray<OldRecordFieldPod*> m_fields;
    // This is the storage for the fields when finalized
    hkArray<hkVersionedRecordType*> m_parents; // Parent records
    hkArray<hkVersionedRecordType*> m_children; // Derived records
    hkArray<hkReflect::Template::Parameter> m_templateParameters; // template parameters while building
    hkReflect::Template* m_templateParametersObject; // final template parameters object
    Status m_status;
    hkUint64 m_startingUid;
};

// Store the dependencies from a patch in a convenient form to look up
// versions from within the patch
struct HK_EXPORT_COMMON hkPatchDependencies
{
    hkPatchDependencies() {}
    hkPatchDependencies(_In_ const hkReflect::Version::PatchInfo* patch);
    void addPatch(_In_ const hkReflect::Version::PatchInfo* patch);
    void clear();

    // Will assert if a name is given that doesn't exist, -1 if no name
    int getVersion(_In_opt_z_ const char* typeName) const;
    // Returns -1 if it doesn't exist or no name given
    int tryGetVersion(_In_opt_z_ const char* typeName) const;
    // For checking
    bool nameIsOk(_In_opt_z_ const char* typeName) const;

    hkStringMap<int> m_versions;
};

// This stores a set of types, and allows them to be modified via patches and hkDataObject
class HK_EXPORT_COMMON hkModifiableTypeSet
{
    friend class hkVersionedRecordType;
public:
    enum TypeOptions
    {
        FAIL_ON_INCORRECT_TYPES = 0, // Default behaviour. If we return a type, it should work. If the patches are impossible or incomplete, we give up.
        BEST_EFFORT_ON_INCOMPLETE_TYPES = 1, // For utilities and debug, emit as much type information as we can. The created type may not be usable.
        GENERATE_TYPES_FROM_PATCHES_ONLY = 2, // For checking, don't take any information from the compiled-in types, only patches. This requires a class added patch for every reflected class
    };

    hkModifiableTypeSet(
        hkReflect::Version::PatchSet& patches,
        _In_ const hkReflect::TypeReg* reg,
        _In_ hkMemoryAllocator* allocator,
        _In_ const hkReflect::Detail::ArrayImpl* arrayImpl,
        _In_ const hkReflect::Detail::ArrayImpl* homogenousArrayImpl,
        _In_ const hkReflect::Detail::PointerImpl* m_pointerImpl,
        _In_ const hkReflect::Detail::StringImpl* m_stringImpl,
        hkFlags<TypeOptions, hkUint16> opts = FAIL_ON_INCORRECT_TYPES);
    ~hkModifiableTypeSet();

    _Ret_maybenull_ hkVersionedRecordType* getParent(_In_ const hkReflect::Type* type);

    void getAllRecordTypes(_In_z_ const char* name, int version, hkArray<hkVersionedRecordType*>::Temp& typesOut);

    void resolveAndFinalize(bool copyParentMembers = true);
    void setStartingVersions();

    // Search for a type, return HK_NULL if it does not exist
    _Ret_maybenull_ hkVersionedRecordType* findType(_In_z_ const char* typeName, int typeVersion);

    // Search for a type, create a new one of it does not exist, using the compiled-in types
    _Ret_maybenull_ hkVersionedRecordType* createType(_In_z_ const char* name, int version = 0, _In_opt_ const hkReflect::Type* type = HK_NULL); // Use type if given, else use name
    // Search for a type, create a new one of it does not exist, using only the information in the patches
    _Ret_maybenull_ hkVersionedRecordType* createFinalTypeFromPatches(_In_z_ const char* name, int version = 0, _In_opt_ const hkReflect::Type* type = HK_NULL); // Use type if given, else use name

    /// Utility to allocate from the types allocator. Needed for some data that must be persistent
    template<typename T> T* allocateFromTypes() { return reinterpret_cast<T*>(m_typesAllocator.blockAlloc(sizeof(T))); }
    hkTransientAllocator& getTypesAllocator() { return m_typesAllocator; }

    // Which patches have been visited while constructing types? The same patches should be applied to objects
    hkArrayView<const hkReflect::Version::PatchInfo*> getPatchesApplied() { return m_patches.getPatchList(); }

    void addAllDependentTodos(_In_ const hkReflect::Type* sourceType);
    void addDeferredLayout(_In_ hkReflect::Type* type); // Layout needs deferred calculation
    void addTypeTodo(_In_ const hkReflect::Type* r, _In_ hkReflect::QualType* typeOut);

    bool anyPatchesNeededFor(_In_ const hkReflect::Type* t);
    _Ret_maybenull_ hkVersionedRecordType* versionedTypeFromSourceType(_In_ const hkReflect::Type* sourceType) const;

    void appendAllTypesToArray(hkArray<hkVersionedRecordType*>& a) const;

    // Create all of the types that are possible from the current patch set. This is mostly for checking, usually
    // the set should only contain the set of types actively in use
    void createAllTypes();

protected:
    hkTransientAllocator m_typesAllocator;
    hkArray<hkVersionedRecordType*> m_recordTypes;
    hkPointerMap<hkUint64, int> m_typeIndexFromUid;
    hkArray<hkStringPtr> m_names;

    hkReflect::Detail::CustomAllocImpl m_allocImpl;
    const hkReflect::Detail::ArrayImpl* m_arrayImpl;
    const hkReflect::Detail::ArrayImpl* m_homogenousArrayImpl;
    const hkReflect::Detail::StringImpl* m_stringImpl;
    const hkReflect::Detail::PointerImpl* m_pointerImpl;

    typedef hkPointerMap<const hkReflect::Type*, const hkReflect::Type*> CopiedTypeFromSourceMap;
    CopiedTypeFromSourceMap m_copiedTypeFromSourceMap;

    struct TypeTodo
    {
        const char* m_sourceTypeName;
        int m_sourceTypeVersion;
        const hkReflect::Type* m_type;
        hkReflect::QualType* m_location;

        void set(_In_z_ const char* tn, int tv, _In_ hkReflect::QualType* loc) { m_sourceTypeName = tn; m_sourceTypeVersion = tv; m_type = HK_NULL; m_location = loc; }
        void set(_In_ const hkReflect::Type* type, _In_ hkReflect::QualType* loc) { m_sourceTypeName = HK_NULL; m_sourceTypeVersion = -1; m_type = type; m_location = loc; }

        void resolve(_In_ const hkReflect::Type* t);
    };
    hkArray<TypeTodo> m_typeTodos;

    struct ParentTodo
    {
        const char* m_parentRecordName;
        int m_parentRecordVersion;
        hkVersionedRecordType* m_childRecord;
        const hkReflect::Type* m_type;

        void set(_In_z_ const char* pn, int pv, _In_ hkVersionedRecordType* childRecord) { m_parentRecordName = pn; m_parentRecordVersion = pv; m_childRecord = childRecord; m_type = HK_NULL; }
        void set(_In_ const hkReflect::Type* type, _In_ hkVersionedRecordType* childRecord) { m_parentRecordName = HK_NULL; m_parentRecordVersion = -1; m_childRecord = childRecord; m_type = type; }
    };
    hkArray<ParentTodo> m_parentTodos;

    struct AddDependentTodoVisitor;

    hkArray<hkReflect::Type*> m_deferredLayouts;

    void createPatchedTypeFromDescription(hkReflect::Version::LegacyType type, _In_z_ const char* typeName, int tuples, _In_opt_ const void* defaultPtr, _Inout_ hkReflect::QualType* typeOut, const hkPatchDependencies& patchDeps);

    void resolveRecordTypeTodo(int index);
    void resolveParentRecordTodo(int index);
    void createTypeFromName(_Inout_ hkReflect::QualType* typeOut, _In_z_ const char* typeName, const hkPatchDependencies& patchDeps);

    // Not sure about this data structure
    typedef hkStringMap<hkPointerMap<int, int>* > PatchTypeFromNameMap;
    PatchTypeFromNameMap m_patchTypeFromNameMap;
    hkReflect::Version::PatchSet& m_patches;

    const hkReflect::TypeReg* m_lookupType;
    int m_lastTypeResolved;

    // We may see multiple versions of a type when looking through compiled-in types, patches etc.
    // This stores the oldest version we have seen (which is where we need to start patching)
    // We could do better here, multiple types can have the same name at different points
    hkStringMap<int> m_oldestVersionEncountered;

    // Internal tracking of source type patch needs
    enum
    {
        PATCH_NEEDED,
        NO_PATCH_NEEDED,
        IN_PROGRESS
    };

    hkPointerMap<const hkReflect::Type*, int> m_patchesNeededForType;
    hkPointerMap<const hkReflect::Type*, int> m_versionedTypeFromSourceType;
    hkFlags<TypeOptions, hkUint16> m_typeOptions;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
