// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Version/hkModifiableTypeSet.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Reflect/Builder/hkRecordLayout.h>
#include <Common/Base/Reflect/Version/hkReflectVersionPatches.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/Serialize/Core/hkSerializeCore.h>
#include <Common/Base/Serialize/Version/hkVersionBundle.h>
#include <Common/Base/Fwd/hkcstring.h>

#define DEBUG_LOG_IDENTIFIER "s11n.ModifiableTypeSet"
#include <Common/Base/System/Log/hkLog.hxx>

namespace
{
    void resolveTypeLoc(_In_opt_ hkReflect::QualType* location, _In_ const hkReflect::Type* target)
    {
        if (location) // We sometimes todo to nothing, just to create a dependency
        {
            HK_ASSERT_NO_MSG(0x2c6311bf, (location->get() == reinterpret_cast<const hkReflect::Type*>(0x22222222)) || (location->get() == 0));
            location->set(target, false);
        }
    }

    void calculateRecordLayout(_Inout_ hkReflect::Type* t)
    {
        hkReflect::RecordLayout::recompute(t, hkReflect::RecordLayout::Options().setNative().withMinimumRecordSize(HK_POINTER_SIZE));
        // We require this for alignment purposes in other places
        HK_ASSERT_NO_MSG(0x658bf685, t->getSizeOf() >= HK_POINTER_SIZE);
    }

    enum
    {
        INCOMPLETE_TYPE_MARKER = 0x50000000
    };

    _Ret_maybenull_ const hkReflect::Version::PatchInfo* stripV2Patches(const hkReflect::Version::PatchInfo* p) { if (p && !p->isV2Patch()) { return p; } return HK_NULL; }
}


hkModifiableTypeSet::hkModifiableTypeSet(hkReflect::Version::PatchSet& patches
    , _In_ const hkReflect::TypeReg* reg
    , _In_ hkMemoryAllocator* allocator
    , _In_ const hkReflect::Detail::ArrayImpl* arrayImpl
    , _In_ const hkReflect::Detail::ArrayImpl* homogenousArrayImpl
    , _In_ const hkReflect::Detail::PointerImpl* pointerImpl
    , _In_ const hkReflect::Detail::StringImpl* stringImpl
    , hkFlags<TypeOptions, hkUint16> typeOptions)
    : m_typesAllocator(allocator)
    , m_allocImpl(&m_typesAllocator)
    , m_arrayImpl(arrayImpl)
    , m_homogenousArrayImpl(homogenousArrayImpl)
    , m_stringImpl(stringImpl)
    , m_pointerImpl(pointerImpl)
    , m_patches(patches)
    , m_lookupType(reg)
    , m_lastTypeResolved(0)
    , m_typeOptions(typeOptions)
{
    DLOG_BEGIN("**** TYPESET START ****");
}

namespace
{
    void recursivelyAddChildren(_Inout_ hkVersionedRecordType* root, hkArray<hkVersionedRecordType*>::Temp& typesOut)
    {
        if (typesOut.indexOf(root) < 0)
        {
            typesOut.pushBack(root);

            for (int childIdx = 0; childIdx < root->getNumChildren(); childIdx++)
            {
                hkVersionedRecordType* childType = root->getChild(childIdx);
                HK_ASSERT_NO_MSG(0x2fb28367, childType->getStatus() == hkVersionedRecordType::STATUS_RESOLVED);
                recursivelyAddChildren(childType, typesOut);
            }
        }
    }
}

void hkModifiableTypeSet::getAllRecordTypes(_In_z_ const char* name, int version, hkArray<hkVersionedRecordType*>::Temp& typesOut)
{
    for (int i = 0; i < m_recordTypes.getSize(); i++)
    {
        hkStringBuf fullName;
        if ((hkString::strCmp(m_recordTypes[i]->getFullName(fullName), name) == 0))
        {
            if ((m_recordTypes[i]->getCompletedType()->getVersion() == version))
            {
                HK_ASSERT_NO_MSG(0x491e043b, m_recordTypes[i]->getStatus() == hkVersionedRecordType::STATUS_RESOLVED);
                recursivelyAddChildren(m_recordTypes[i], typesOut);
            }
        }
    }
}

void hkModifiableTypeSet::setStartingVersions()
{
    for (int i = 0; i < m_recordTypes.getSize(); i++)
    {
        hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::VERSION>(m_recordTypes[i]->getCompletedType(), m_recordTypes[i]->getStartVersion());
    }
}

void hkModifiableTypeSet::resolveRecordTypeTodo(int index)
{
    // New todos will be added at the end
    TypeTodo currentTodo = m_typeTodos[index];

    const hkVersionedRecordType* ti;
    if (currentTodo.m_sourceTypeName)
    {
        ti = findType(currentTodo.m_sourceTypeName, currentTodo.m_sourceTypeVersion);
    }
    else
    {
        // See have we already done this type
        if(const hkReflect::Type* lookedUp = m_lookupType->typeFromType(currentTodo.m_type))
        {
            currentTodo.m_type = lookedUp;
        }
        ti = findType(currentTodo.m_type->getName(), currentTodo.m_type->getVersion());
    }

    if (ti)
    {
        currentTodo.resolve(ti->getTypeInternal());
        DLOG("RESOLVE todo @{} from {} to {} ({})", hkUlong(currentTodo.m_location), currentTodo.m_sourceTypeName, ti->getTypeInternal(), ti->getTypeInternal()->getFullName());
    }
    else
    {
        // Else this is a new type, so we need to copy it
        // This will only copy the current type, it will never recurse. Todos are added for any linked types

        const char* typeName = currentTodo.m_sourceTypeName;
        const int typeVersion = currentTodo.m_sourceTypeVersion;
        const hkReflect::Type* sourceType = currentTodo.m_type;
        hkVersionedRecordType* newType = createType(typeName, typeVersion, sourceType);
        const hkReflect::Type* generatedType = HK_NULL;

        if (newType)
        {
            generatedType = newType->getTypeInternal();
        }
        else if (typeName)
        {
            if (const hkReflect::Type* lookupType = m_lookupType->typeFromName(typeName))
            {
                if (!lookupType->asRecord())
                {
                    generatedType = lookupType;
                }
            }
        }

        currentTodo.resolve(generatedType);
        hkStringBuf sb;
        DLOG("RESOLVE todo @{} from {}[{}] to {} ({})", hkUlong(currentTodo.m_location), typeName, typeVersion, sourceType, sourceType ? sourceType->getFullName(sb) : "<none>");
    }
}

void hkModifiableTypeSet::resolveParentRecordTodo(int index)
{
    hkVersionedRecordType* childRecord = m_parentTodos[index].m_childRecord;
    if (m_typeOptions.allAreSet(hkModifiableTypeSet::GENERATE_TYPES_FROM_PATCHES_ONLY))
    {
        childRecord->addParent(createFinalTypeFromPatches(m_parentTodos[index].m_parentRecordName, m_parentTodos[index].m_parentRecordVersion, m_parentTodos[index].m_type));
    }
    else
    {
        if(!m_parentTodos[index].m_parentRecordName && !m_parentTodos[index].m_type)
        {
            childRecord->addParent(HK_NULL);
        }
        else
        {
            childRecord->addParent(createType(m_parentTodos[index].m_parentRecordName, m_parentTodos[index].m_parentRecordVersion, m_parentTodos[index].m_type));
        }
    }
    //DLOG("RESOLVE parent of {},{} ({}) to {} ({})", m_parentTodos[index].m_childRecord, m_parentTodos[index].m_childRecord->m_parents.getSize() - 1, m_recordTypes[m_parentTodos[index].m_childRecordIndex].m_type->getName(), m_recordTypes[m_parentTodos[index].m_childRecordIndex].m_parents.back(), m_parentTodos[index].m_parentRecordName);

    m_parentTodos.removeAtAndCopy(index);

}


void hkModifiableTypeSet::resolveAndFinalize(bool copyParentMembers)
{
    while (m_typeTodos.getSize() || m_parentTodos.getSize())
    {
        for (int index = 0; index < m_typeTodos.getSize(); index++)
        {
            resolveRecordTypeTodo(index);
        }
        m_typeTodos.clear();

        // Parent todo may add new field type todos so resolve them before trying more parents
        if (m_parentTodos.getSize())
        {
            resolveParentRecordTodo(0);
        }
    }

    for (int currentId = m_lastTypeResolved; currentId < m_recordTypes.getSize(); currentId++)
    {
        if (m_recordTypes[currentId])
        {
            if (m_recordTypes[currentId]->finalize(&m_typesAllocator, copyParentMembers).isFailure())
            {
                m_recordTypes[currentId]->invalidateType();
                m_recordTypes.removeAt(currentId);
                currentId--;
            }
        }

        // Copy from above, copyAndFinalize could add more
        while (m_typeTodos.getSize() || m_parentTodos.getSize())
        {
            for (int index = 0; index < m_typeTodos.getSize(); index++)
            {
                resolveRecordTypeTodo(index);
            }
            m_typeTodos.clear();

            // Parent todo may add new field type todos so resolve them before trying more parents
            if (m_parentTodos.getSize())
            {
                resolveParentRecordTodo(0);
            }
        }
    }


    for (int currentId = m_lastTypeResolved; currentId < m_recordTypes.getSize(); currentId++)
    {
        hkVersionedRecordType& currentTypeInfo = *m_recordTypes[currentId];
        hkReflect::RecordType* targetRecord = currentTypeInfo.getCompletedType()->asRecord();

        // If we are not copying parent members we need to set the pointer
        if (!copyParentMembers)
        {
            if (currentTypeInfo.getNumParents())
            {
                hkReflect::TypeDetail::setParent(targetRecord, currentTypeInfo.getCurrentParent()->getCompletedType());
            }
        }

        calculateRecordLayout(targetRecord);
    }

    m_lastTypeResolved = m_recordTypes.getSize() - 1;

    for (int i = 0; i < m_deferredLayouts.getSize(); i++)
    {
        calculateRecordLayout(m_deferredLayouts[i]);
    }
    m_deferredLayouts.clear();

    DLOG_SCOPE("Generated Types::");
    hkStringBuf sb;
    DLOG_IF(for (int i = 0; i < m_recordTypes.getSize(); i++))
    {
        DLOG("{0}: {1} ({2})", i, m_recordTypes[i]->getFullName(sb), m_recordTypes[i]->getCompletedType()->getSizeOf());
    }
}

hkModifiableTypeSet::~hkModifiableTypeSet()
{
    for (int i = 0; i < m_recordTypes.getSize(); i++)
    {
        m_recordTypes[i]->~hkVersionedRecordType();
    }
    DLOG_END();
}

hkVersionedRecordType::hkVersionedRecordType(_In_ hkModifiableTypeSet* owner, _In_ hkReflect::Type* type, int startVersion, Status status, _In_ const hkReflect::Type* sourceType, hkUint64 startingUid)
    : m_type(type)
    , m_sourceType(sourceType)
    , m_startVersion(startVersion)
    , m_currentParentIndex(0)
    , m_owner(owner)
    , m_templateParametersObject(HK_NULL)
    , m_status(status)
    , m_startingUid(startingUid)
{
}

hkVersionedRecordType::~hkVersionedRecordType()
{
    HK_ON_DEBUG(setStatus(STATUS_INVALID));
    if (m_templateParametersObject)
    {
        hkMemHeapBlockFree<void>(m_templateParametersObject, hkReflect::Template::getSizeOf(m_templateParametersObject->getNumParams()));
    }
}

void hkVersionedRecordType::addParent(_In_ hkVersionedRecordType* parent)
{
    m_parents.pushBack(parent);
}

hkResult hkVersionedRecordType::finalize(_Inout_ hkMemoryAllocator* allocator, bool copyParentMembers)
{
    if ((getStatus() == hkVersionedRecordType::STATUS_RESOLVED) || (getStatus() == hkVersionedRecordType::STATUS_LOCALLY_COMPLETE))
    {
        return HK_SUCCESS;
    }
#if defined(HK_DEBUG_SLOW)
    DLOG_SCOPE("finalize {} ({})", this, m_type->getFullName());
#endif
    setStatus(STATUS_LOCALLY_COMPLETE);

    // Copy in parent members and do the layout calculation
    // No need to recurse, as we have a complete view of all types we can encounter here already

    for (int parentIndex = 0; parentIndex < getNumParents(); parentIndex++)
    {
        if(hkVersionedRecordType* parentType = getParent(parentIndex))
        {
            if (parentType->finalize(allocator, copyParentMembers).isFailure())
            {
                return HK_FAILURE;
            }

            const hkReflect::RecordType* currentParentRecord = parentType->m_type->asRecord();

            DLOG("{} ({}) copying from parent {} ({}) {} fields ({})", (void*)this, m_type->getFullName(), parentIndex, currentParentRecord->getFullName(), currentParentRecord->getNumFields(), m_type->asRecord()->getNumFields());
            // Locally complete -> parent of itself, we don't add a backlink to ourself
            if (parentType->getStatus() != STATUS_LOCALLY_COMPLETE)
            {
                parentType->addChild(this);
            }

            if (copyParentMembers)
            {
                int addedFieldIndex = 0;
                for (hkReflect::DeclIter<hkReflect::FieldDecl> fieldIter(currentParentRecord); fieldIter.advance();)
                {
                    const hkReflect::FieldDecl f = fieldIter.current();
                    if (f.isSerializable())
                    {
                        // Finalize type happens after todos are resolved so this doesn't mess up the todo list
                        OldRecordFieldPod* newField = m_owner->m_typesAllocator.create<OldRecordFieldPod>();
                        m_fields.insertAt(addedFieldIndex, &newField, 1);
                        if ((parentIndex < (getNumParents() - 1)) && (f.getName()[0] != '#'))
                        {
                            hkStringBuf newName("#");
                            newName.append(f.getName());
                            m_owner->m_names.pushBack(newName.cString());
                            newField->m_name = m_owner->m_names.back();
                        }
                        else
                        {
                            newField->m_name = f.getName();
                        }
                        newField->m_flags = f.getFlags().get();
                        newField->m_type = f.getType();
                        newField->m_offset = 0;

                        addedFieldIndex++;
                    }
                }
            }
        }
    }

    if (m_owner->m_typeOptions.allAreSet(hkModifiableTypeSet::GENERATE_TYPES_FROM_PATCHES_ONLY))
    {
        m_currentParentIndex = getNumParents() - 1;
    }

    hkReflect::RecordType* targetRecord = m_type->asRecord();
    hkArray<hkReflect::Detail::RecordFieldPod> fieldTypes;
    hkStringMap<bool> membersAlreadyFound;

    for (int i = 0; i < m_fields.getSize(); ++i)
    {
        OldRecordFieldPod& fieldData = *m_fields[i];

        const char* fieldNameForRecord = fieldData.m_name;
        if (fieldNameForRecord && (fieldNameForRecord[0] != '#'))
        {
            if (membersAlreadyFound.hasKey(fieldNameForRecord))
            {
                const int totalLength = hkString::strLen(fieldNameForRecord) + 2;
                char* newFieldName = reinterpret_cast<char*>(allocator->blockAlloc(totalLength));
                newFieldName[0] = '#';
                hkString::strNcpy(newFieldName + 1, totalLength, fieldNameForRecord, totalLength - 1);
                fieldNameForRecord = newFieldName; // This is the value we are going to write below
            }
            else
            {
                membersAlreadyFound.insert(fieldNameForRecord, true);
            }
        }

        if (fieldData.m_type)
        {
            hkReflect::TypeBuilder field;
            field.setTypeWorld(fieldData.m_type->getTypeWorld());
            field.beginDerived(fieldData.m_type);
            field.setField(fieldNameForRecord, fieldData.m_offset, fieldData.m_flags);
            field.addItem<hkReflect::Opt::ALLOC_IMPL>(HK_NULL); // prevent allocation of field types
            field.setItem<hkReflect::Opt::DECL_CONTEXT>(targetRecord);

            hkReflect::Type* allocatedType;
            if (field.allocate(allocator, &allocatedType).isFailure())
            {
                Log_Error("Failed to allocate type from builder");
                return HK_FAILURE;
            }

            hkReflect::Detail::RecordFieldPod fieldPod = { allocatedType };
            fieldTypes.pushBack(fieldPod);
        }
        else
        {
            // For now we will just drop the failing fields
            Log_Warning("Dropping field {} as it has a null type", fieldNameForRecord).setId(0x2dba8010);
            if (m_owner->m_typeOptions.noneIsSet(hkModifiableTypeSet::BEST_EFFORT_ON_INCOMPLETE_TYPES))
            {
                return HK_FAILURE;
            }
        }
    }

    hkReflect::Detail::DeclsArray* fieldArray = hkReflect::Detail::DeclsArray::create(fieldTypes.begin(), fieldTypes.getSize(), 0, *allocator);
    hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::DECLS>(targetRecord, fieldArray);

    // Set the template parameters
    if (hkReflect::TypeDetail::localHasOptional(targetRecord, hkReflect::Opt::TEMPLATE))
    {
        m_templateParametersObject = hkReflect::Template::create(m_templateParameters.getSize(), *hkMemHeapAllocator());
        hkString::memCpy(const_cast<hkReflect::Template::Parameter*>(m_templateParametersObject->getParams()), m_templateParameters.begin(), m_templateParameters.getSize() * hkSizeOf(hkReflect::Template::Parameter));
        hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::TEMPLATE>(targetRecord, m_templateParametersObject);
    }

    DLOG("COMPLETED {} ({}) {}[{}]", (void*)this, targetRecord->getFullName(), targetRecord->getSizeOf(), targetRecord->getAlignOf());

    setStatus(hkVersionedRecordType::STATUS_RESOLVED);
    return HK_SUCCESS;
}

_Ret_maybenull_ hkVersionedRecordType* hkModifiableTypeSet::versionedTypeFromSourceType(_In_ const hkReflect::Type* sourceType) const
{
    if(int index = m_versionedTypeFromSourceType.getWithDefault(sourceType, 0))
    {
        return m_recordTypes[index];
    }
    return HK_NULL;
}

// This uses the type and the version
_Ret_maybenull_ hkVersionedRecordType* hkModifiableTypeSet::findType(_In_z_ const char* typeName, int typeVersion)
{
    HK_ASSERT_NO_MSG(0x7f3c9534, typeVersion != -1); // Reserved value
    const hkUint64 uid = m_patches.getUid(typeName, typeVersion);
    const int newTypeId = m_typeIndexFromUid.getWithDefault(uid, -1);
    if ((newTypeId >= 0) && (newTypeId != INCOMPLETE_TYPE_MARKER))
    {
        return m_recordTypes[newTypeId];
    }
    return HK_NULL;
}

// This is called from within patch functions and should return the current parent, according to the patching system
_Ret_maybenull_ hkVersionedRecordType* hkModifiableTypeSet::getParent(_In_ const hkReflect::Type* type)
{
    if (const hkVersionedRecordType* ti = findType(type->getName(), type->getVersion()))
    {
        return ti->getCurrentParent();
    }

    return HK_NULL;
}

void hkModifiableTypeSet::addDeferredLayout(_In_ hkReflect::Type* type)
{
    m_deferredLayouts.pushBack(type);
}

void hkModifiableTypeSet::addTypeTodo(_In_ const hkReflect::Type* r, _In_ hkReflect::QualType* typeOut)
{
    m_typeTodos.expandOne().set(r, typeOut);
}

namespace
{
    bool isWhiteSpace(char c)
    {
        return ((c == ' ') || (c == '\t'));
    }

    void stripLeadingAndTrailingSpace(hkStringBuf& sb)
    {
        int numAtStart = 0;
        int numAtEnd = 0;
        const int length = sb.getLength();
        while (isWhiteSpace(sb[numAtStart]) && (numAtStart < length))
        {
            numAtStart++;
        }
        while (isWhiteSpace(sb[length - numAtEnd - 1]) && ((numAtStart + numAtEnd) < length))
        {
            numAtEnd++;
        }
        sb.chompStart(numAtStart);
        sb.chompEnd(numAtEnd);
    }

    void stripTemplateBrackets(hkStringBuf& sb)
    {
        stripLeadingAndTrailingSpace(sb);
        HK_ASSERT_NO_MSG(0x6394d93b, sb.startsWith("<") && sb.endsWith(">"));
        sb.chompStart(1);
        sb.chompEnd(1);
    }

    int removeCount(hkStringBuf& sb)
    {
        sb.chompEnd(1); // "]"
        stripLeadingAndTrailingSpace(sb); // "   ]"
        const char* currentIntStart = &sb.cString()[sb.getLength() - 1]; // "0    ]"

        while (currentIntStart > (sb.cString() + 1))
        {
            if (((*(currentIntStart - 1)) >= '0') && ((*(currentIntStart - 1)) <= '9'))
            {
                currentIntStart--;
            }
            else
            {
                break;
            }
        }

        const int ret = hkString::atoi(currentIntStart); // "1230   ]"
        stripLeadingAndTrailingSpace(sb);   // "   1230   ]";
        sb.chompEnd(1); // "[   1230   ]"
        stripLeadingAndTrailingSpace(sb);   // "  [   1230   ]"

        return ret;
    }
}
void hkModifiableTypeSet::createTypeFromName(_Inout_ hkReflect::QualType* typeOut, _In_z_ const char* typeNameStr, const hkPatchDependencies& patchDeps)
{
    // This is basically a free-form type described by a string
    // Ideally any combination is supported

    hkStringBuf typeName(typeNameStr);
    stripLeadingAndTrailingSpace(typeName);

    // SOMETHING[n]
    if (typeName.endsWith("]"))
    {
        int count = removeCount(typeName);
        hkReflect::RepeatArrayType* rep = m_typesAllocator.create<hkReflect::RepeatArrayType>((const hkReflect::Type*)HK_NULL, count, 0, 0, &hkReflect::Detail::RepeatImpl::s_instance, "T[]");

        createTypeFromName(&rep->m_elemType, typeName, patchDeps);

        HK_ASSERT_NO_MSG(0x320a42a8, rep->m_elemType); // Tuple can't be a forward ref type
        hkReflect::TypeDetail::localSetSizeAlignPreserveReq(rep,
            count * rep->m_elemType->getSizeOf(),
            rep->m_elemType->getAlignOf());

        const_cast<hkReflect::Template::Parameter*>(rep->getTemplate()->getParam(0))->m_storage = hkUlong(rep->m_elemType.get());

        resolveTypeLoc(typeOut, rep);
    }
    // hkArray< SOMETHING >
    else if (typeName.startsWithCase("hkArray"))
    {
        typeName.chompStart(7);
        stripTemplateBrackets(typeName);

        hkReflect::BasicArrayTypeWithParams* rep = m_typesAllocator.create<hkReflect::BasicArrayTypeWithParams>((const hkReflect::Type*)0x22222222, hkSizeOf(hkReflect::Detail::HomogeneousArrayImpl::InMemory), (int)HK_ALIGN_OF(hkReflect::Detail::HomogeneousArrayImpl::InMemory), m_homogenousArrayImpl, "hkArray");
        createTypeFromName(&rep->m_elemType, typeName, patchDeps);
        const_cast<hkReflect::Template::Parameter*>(rep->getTemplate()->getParam(0))->m_storage = hkUlong(rep->m_elemType.get());
        resolveTypeLoc(typeOut, rep);
    }
    // hkPtr< SOMETHING >
    else if (typeName.startsWith("hkPtr"))
    {
        HK_WARN(0x736692a5, "hkPtr has been removed.  Please update dynamic type to use hkViewPtr instead.");

        typeName.chompStart(5);
        stripTemplateBrackets(typeName);

        hkReflect::BasicPointerType* newP = m_typesAllocator.create<hkReflect::BasicPointerType>(reinterpret_cast<const hkReflect::Type*>(0x22222222), "hkPtr");
        newP->m_impl = m_pointerImpl;

        createTypeFromName(&newP->m_next, typeName, patchDeps);
        const_cast<hkReflect::Template::Parameter*>(newP->getTemplate()->getParam(0))->m_storage = hkUlong(newP->m_next.get());
        resolveTypeLoc(typeOut, newP);
    }
    // hkViewPtr< SOMETHING >
    else if (typeName.startsWith("hkViewPtr"))
    {
        typeName.chompStart(9);
        stripTemplateBrackets(typeName);

        hkReflect::BasicPointerType* newP = m_typesAllocator.create<hkReflect::BasicPointerType>(reinterpret_cast<const hkReflect::Type*>(0x22222222), "hkViewPtr");
        newP->m_impl = m_pointerImpl;

        createTypeFromName(&newP->m_next, typeName, patchDeps);
        const_cast<hkReflect::Template::Parameter*>(newP->getTemplate()->getParam(0))->m_storage = hkUlong(newP->m_next.get());
        resolveTypeLoc(typeOut, newP);
    }
    else if (typeName.startsWith("hkReflect::Type*"))
    {
        resolveTypeLoc(typeOut, hkReflect::getType<hkReflect::Type>());
    }
    else
    {
        // Check for versioned types first
        const int version = patchDeps.tryGetVersion(typeName); // This may not be a record
        if (version != -1)
        {
            if (hkVersionedRecordType* vrt = findType(typeName, version))
            {
                resolveTypeLoc(typeOut, vrt->getTypeInternal());
                return;
            }
        }

        // It should be a dependency already, so if we don't find it, assume it's builtin
        if (const hkReflect::Type* lookedUpType = m_lookupType->typeFromName(typeName))
        {
            resolveTypeLoc(typeOut, lookedUpType);
        }
        else
        {
            HK_ASSERT_NO_MSG(0x27fa4823, 0); // Couldn't find type
        }
    }
}
void hkModifiableTypeSet::createPatchedTypeFromDescription(hkReflect::Version::LegacyType type, _In_z_ const char* typeName, int tuples, _In_opt_ const void* defaultPtr, _Inout_ hkReflect::QualType* typeOut, const hkPatchDependencies& patchDeps)
{
    if (type & hkReflect::Version::TYPE_ARRAY)
    {
        hkReflect::BasicArrayType* rep;
        //if((type == hkReflect::Version::TYPE_ARRAY_STRUCT) && !typeName)
        {
            rep = m_typesAllocator.create<hkReflect::BasicArrayType>((const hkReflect::Type*)HK_NULL, hkSizeOf(hkReflect::Detail::HomogeneousArrayImpl::InMemory), (int)HK_ALIGN_OF(hkReflect::Detail::HomogeneousArrayImpl::InMemory), m_homogenousArrayImpl, "hkArray");
            DLOG("Generated_1 HA @{}", hkUlong(rep));
        }
        //      else
        //      {
        //          rep = m_typesAllocator.create<hkReflect::BasicArrayType>((const hkReflect::Type*)HK_NULL, hkSizeOf(hkArray<char>), (int)HK_ALIGN_OF(hkArray<char>), m_arrayImpl);
        //      }
        createPatchedTypeFromDescription(hkReflect::Version::LegacyType(int(type) & (~hkReflect::Version::TYPE_ARRAY)), typeName, 0, 0, &rep->m_elemType, patchDeps);
        resolveTypeLoc(typeOut, rep);
        return;
    }

    if (type & hkReflect::Version::TYPE_TUPLE)
    {
        hkReflect::RepeatArrayType* rep = m_typesAllocator.create<hkReflect::RepeatArrayType>((const hkReflect::Type*)HK_NULL, tuples, 0, 0, &hkReflect::Detail::RepeatImpl::s_instance, "T[]");

        createPatchedTypeFromDescription(hkReflect::Version::LegacyType(int(type) & (~hkReflect::Version::TYPE_TUPLE)), typeName, 0, 0, &rep->m_elemType, patchDeps);
        HK_ASSERT_NO_MSG(0x320a42a8, rep->m_elemType); // Tuple can't be a forward ref type
        hkReflect::TypeDetail::localSetSizeAlignPreserveReq(rep,
            tuples * rep->m_elemType->getSizeOf(),
            rep->m_elemType->getAlignOf());
        const_cast<hkReflect::Template::Parameter*>(rep->getTemplate()->getParam(0))->m_storage = hkUlong(rep->m_elemType.get());

        resolveTypeLoc(typeOut, rep);
        return;
    }

    switch (type)
    {
        case hkReflect::Version::TYPE_BYTE: resolveTypeLoc(typeOut, hkReflect::getType<hkUint8>().get()); return;
        case hkReflect::Version::TYPE_INT: resolveTypeLoc(typeOut, hkReflect::getType<hkInt32>().get()); return;
        case hkReflect::Version::TYPE_REAL: resolveTypeLoc(typeOut, hkReflect::getType<float>().get()); return;
        case hkReflect::Version::TYPE_VEC_4: resolveTypeLoc(typeOut, hkReflect::getType< hkVector4 >().get()); return;
        case hkReflect::Version::TYPE_VEC_8: resolveTypeLoc(typeOut, hkReflect::getType<hkReal[8]>().get()); return;
        case hkReflect::Version::TYPE_VEC_12: resolveTypeLoc(typeOut, hkReflect::getType< hkMatrix3 >().get()); return;
        case hkReflect::Version::TYPE_VEC_16: resolveTypeLoc(typeOut, hkReflect::getType< hkMatrix4 >().get()); return;
        // The memory is tied up inside the versioned allocator so we can treat all strings as char*
        case hkReflect::Version::TYPE_CSTRING: resolveTypeLoc(typeOut, hkReflect::getType< const char* >().get()); return;

        case hkReflect::Version::TYPE_OBJECT:
        {
            hkReflect::BasicPointerType* newP = m_typesAllocator.create<hkReflect::BasicPointerType>(reinterpret_cast<const hkReflect::Type*>(0x22222222), "T*");
            newP->m_impl = m_pointerImpl;

            if (typeName)
            {
                const int newTypeVersion = patchDeps.getVersion(typeName);
                const hkVersionedRecordType* ti = findType(typeName, newTypeVersion);

                if (ti)
                {
                    newP->m_next = ti->getTypeInternal();
                    const_cast<hkReflect::Template::Parameter*>(newP->getTemplate()->getParam(0))->m_storage = hkUlong(newP->m_next.get());
                }
                else
                {
                    DLOG("Add TODO1 {}({})", typeName, -1); // This one from patches, it really is just a string
                    m_typeTodos.expandOne().set(typeName, newTypeVersion, &newP->m_next);
                    m_typeTodos.expandOne().set(typeName, newTypeVersion, reinterpret_cast<hkReflect::QualType*>(&(const_cast<hkReflect::Template::Parameter*>(newP->getTemplate()->getParam(0))->m_storage)));
                }
            }
            else
            {
                newP->m_next = hkReflect::getType<void>();
                *reinterpret_cast<hkReflect::QualType*>(&(const_cast<hkReflect::Template::Parameter*>(newP->getTemplate()->getParam(0))->m_storage)) = hkReflect::getType<void>();
            }

            resolveTypeLoc(typeOut, newP);
            return;
        }
        case hkReflect::Version::TYPE_STRUCT:
        {
            if (typeName)
            {
                const int newTypeVersion = patchDeps.getVersion(typeName);
                const hkVersionedRecordType* ti = findType(typeName, newTypeVersion);

                if (ti)
                {
                    resolveTypeLoc(typeOut, ti->getTypeInternal());
                }
                else
                {
                    resolveTypeLoc(typeOut, reinterpret_cast<const hkReflect::Type*>(0x22222222));
                    DLOG("Add TODO2 {}({})", typeName, -1); // From patches, really just a string
                    HK_ASSERT_NO_MSG(0x1f4521b3, typeName[0] != '$'); // Templates not allowed here
                    m_typeTodos.expandOne().set(typeName, newTypeVersion, typeOut);
                }
            }
            else
            {
                resolveTypeLoc(typeOut, hkReflect::getType<void>());
            }
            return;
        }
        case hkReflect::Version::TYPE_VOID:
        {
            resolveTypeLoc(typeOut, hkReflect::getType<void>()); // These are sometimes added in patches
            return;
        }
        case hkReflect::Version::TYPE_FROMNAME:
        {
            createTypeFromName(typeOut, typeName, patchDeps); 
            return;
        }
        default:
        {
            HK_ASSERT_NO_MSG(0x5d4ff4d3, false); // Unknown type
        }
    }
}

static int hkReflect_Type_getTypeSizeInBytes(hkUlong optional)
{
    int count = 0;
    for (hkUlong c = optional; c; c &= c - 1)
    {
        count += 1;
    }

    return sizeof(hkReflect::Type) + sizeof(hkUlong) * count;
}

/*static*/ _Ret_notnull_ hkReflect::Type* hkVersionedRecordType::copyAndAddOptional(_In_ const hkReflect::Type* srcType, hkMemoryAllocator& allocator, hkReflect::Optional opt, _In_ const void* val)
{
    // Enlarge the type size for the new opt if it isn't present
    hkUlong newOptMask = hkReflect::TypeDetail::getOptMask(srcType) | opt;
    const int newTypeSize = hkReflect_Type_getTypeSizeInBytes(hkReflect::TypeDetail::getOptMask(srcType) | opt);
    hkReflect::Type* newType = reinterpret_cast<hkReflect::Type*>(allocator.blockAlloc(newTypeSize));

    hkString::memCpy(newType, srcType, sizeof(hkReflect::Type));
    hkReflect::TypeDetail::setOptMask(newType, newOptMask);

    for (int i = 0; i < 32; ++i)
    {
        const hkReflect::Optional o = static_cast<hkReflect::Optional>(1 << i);
        if (hkReflect::TypeDetail::localHasOptional(srcType, o))
        {
            hkUlong v = hkReflect::TypeDetail::localGetOptionalUlong(srcType, o);
            hkReflect::TypeDetail::localSetOptionalUlong(newType, o, v);
        }
    }

    // Insert or overwrite the new opt
    hkReflect::TypeDetail::localSetOptionalUlong(newType, opt, hkUlong(val));

    return newType;
}

void hkVersionedRecordType::addPatchedFieldFromDescription(_In_z_ const char* fieldName, hkReflect::Version::LegacyType type, _In_z_ const char* typeName, int tuples, _In_opt_ const void* defaultPtr, int insertPosition, const hkPatchDependencies& patchDeps)
{
    insertPosition = (insertPosition == -1) ? m_fields.getSize() : insertPosition;
    OldRecordFieldPod* newField = m_owner->m_typesAllocator.create<OldRecordFieldPod>();
    m_fields.insertAt(insertPosition, &newField, 1);

    newField->m_name = fieldName;
    newField->m_flags = 0;
    newField->m_type = reinterpret_cast<const hkReflect::Type*>(0x22222222);
    newField->m_offset = 0;

    if (typeName && typeName[0] == '$')
    {
        HK_ASSERT_NO_MSG(0x4e06b9d2, isTemplate());
        for (int i = 0; i < m_templateParameters.getSize(); i++)
        {
            if (hkString::strCmp(typeName + 1, m_templateParameters[i].getName()) == 0)
            {
                typeName = m_templateParameters[i].getAsType()->getName();
                break;
            }
        }
    }

    m_owner->createPatchedTypeFromDescription(type, typeName, tuples, defaultPtr, reinterpret_cast<hkReflect::QualType*>(&newField->m_type), patchDeps);
    if (defaultPtr)
    {
        HK_ASSERT_NO_MSG(0x2d097e3d, newField->m_type);
        newField->m_type = copyAndAddOptional(newField->m_type, m_owner->getTypesAllocator(), hkReflect::Opt::DEFAULT, defaultPtr); // Only pointers get todo'ed, and they can't have defaults
    }

}

struct CopyPatchedTypeVisitor : public hkReflect::TypeVisitor < CopyPatchedTypeVisitor, void, hkReflect::QualType* >
{
    CopyPatchedTypeVisitor(hkModifiableTypeSet& typeSet, hkTransientAllocator& typesAllocator, _In_ const hkReflect::Detail::ArrayImpl* arrayImpl, _In_ const hkReflect::Detail::StringImpl* stringImpl, _In_ const hkReflect::Detail::ArrayImpl* homogenousArrayImpl, _In_ const hkReflect::Detail::PointerImpl* pointerImpl)
        : m_typeSet(typeSet)
        , m_typesAllocator(typesAllocator)
        , m_arrayImpl(arrayImpl)
        , m_stringImpl(stringImpl)
        , m_homogenousArrayImpl(homogenousArrayImpl)
        , m_pointerImpl(pointerImpl)
    {
    }

    void visit(_In_ const hkReflect::VoidType* v, _Inout_ hkReflect::QualType* typeOut)
    {
        resolveTypeLoc(typeOut, v);
    }

    void visit(_In_ const hkReflect::ValueType* v, _Inout_ hkReflect::QualType* typeOut)
    {
        resolveTypeLoc(typeOut, v);
    }

    void visit(_In_ const hkReflect::StringType* s, _Inout_ hkReflect::QualType* typeOut)
    {
        resolveTypeLoc(typeOut, m_typesAllocator.create<hkReflect::BasicStringType>(hkSizeOf(char*), (unsigned int)HK_ALIGN_OF(char*), m_stringImpl, "const char*", false));
    }

    void visit(_In_ const hkReflect::RecordType* r, _Inout_ hkReflect::QualType* typeOut)
    {
        if (const hkVersionedRecordType* ti = m_typeSet.findType(r->getName(), r->getVersion()))
        {
            resolveTypeLoc(typeOut, ti->getTypeInternal());
        }
        else
        {
            resolveTypeLoc(typeOut, reinterpret_cast<const hkReflect::Type*>(0x22222222));
            hkStringBuf sb;
            DLOG("Add TODO3 {}({})", r->getFullName(sb), r->getVersion());
            m_typeSet.addTypeTodo(r, typeOut); // This must be a todo to preserve the version structure of the file
        }
    }

    void visit(_In_ const hkReflect::PointerType* p, _Inout_ hkReflect::QualType* typeOut)
    {
        const hkReflect::Type* subType = p->getSubType();
        if (subType)
        {
            hkReflect::BasicPointerType* p2 = m_typesAllocator.create<hkReflect::BasicPointerType>(reinterpret_cast<const hkReflect::Type*>(0x22222222), "T*");
            p2->m_impl = m_pointerImpl;
            if (subType->asRecord())
            {
                m_typeSet.addTypeTodo(p->getSubType(), &p2->m_next); // From the file, this is actually a type
                m_typeSet.addTypeTodo(p->getSubType(), reinterpret_cast<hkReflect::QualType*>(const_cast<hkUlong*>(&p2->getTemplate()->getParam(0)->m_storage))); // From the file, this is actually a type
            }
            else
            {
                dispatch(p->getSubType(), &p2->m_next);
                dispatch(p->getSubType(), reinterpret_cast<hkReflect::QualType*>(const_cast<hkUlong*>(&p2->getTemplate()->getParam(0)->m_storage)));
            }
            resolveTypeLoc(typeOut, p2);
        }
        else
        {
            // Leave the subtype as null
            hkReflect::BasicPointerType* p2 = m_typesAllocator.create<hkReflect::BasicPointerType>(HK_NULL, "T*");
            p2->m_impl = m_pointerImpl;
            resolveTypeLoc(typeOut, p2);
        }
    }

    _Ret_maybenull_ const hkReflect::Type* checkForSpecialBuiltinType(_In_ const hkReflect::ArrayType* a)
    {
        // Single
        if (a->extendsOrEquals<hkVector4f>()) { return hkReflect::getType<hkVector4f>(); }
        else if (a->extendsOrEquals<hkQuaternionf>()) { return hkReflect::getType<hkQuaternionf>(); }
        else if (a->extendsOrEquals<hkMatrix3f>()) { return hkReflect::getType<hkMatrix3f>(); }
        else if (a->extendsOrEquals<hkQsTransformf>()) { return hkReflect::getType<hkQsTransformf>(); }
        else if (a->extendsOrEquals<hkMatrix4f>()) { return hkReflect::getType<hkMatrix4f>(); }
        else if (a->extendsOrEquals<hkTransformf>()) { return hkReflect::getType<hkTransformf>(); }
        // Double versions
        else if (a->extendsOrEquals<hkVector4d>()) { return hkReflect::getType<hkVector4d>(); }
        else if (a->extendsOrEquals<hkQuaterniond>()) { return hkReflect::getType<hkQuaterniond>(); }
        else if (a->extendsOrEquals<hkMatrix3d>()) { return hkReflect::getType<hkMatrix3d>(); }
        else if (a->extendsOrEquals<hkQsTransformd>()) { return hkReflect::getType<hkQsTransformd>(); }
        else if (a->extendsOrEquals<hkMatrix4d>()) { return hkReflect::getType<hkMatrix4d>(); }
        else if (a->extendsOrEquals<hkTransformd>()) { return hkReflect::getType<hkTransformd>(); }

        return HK_NULL;
    }

    void visit(_In_ const hkReflect::ArrayType* a, _Inout_ hkReflect::QualType* typeOut)
    {
        // We special-case these because the native alignment has to be preserved
        if (const hkReflect::Type* specialBuiltin = checkForSpecialBuiltinType(a))
        {
            resolveTypeLoc(typeOut, specialBuiltin);
            return;
        }

        const hkReflect::Detail::ArrayImpl* impl = HK_NULL;
        hkReflect::ArrayType* newArr = HK_NULL;

        if (a->getFixedCount() > 0)
        {
            impl = &hkReflect::Detail::RepeatImpl::s_instance;
            newArr = m_typesAllocator.create<hkReflect::RepeatArrayType>(reinterpret_cast<const hkReflect::Type*>(a->getSubType() ? hkUlong(0x22222222) : 0), a->getFixedCount(), 0, 0, impl, "T[]");
        }
        else
        {
            impl = a->getSubType() ? m_arrayImpl : m_homogenousArrayImpl;
            if(a->getSubType())
            {
                newArr = m_typesAllocator.create<hkReflect::BasicArrayTypeWithParams>(reinterpret_cast<const hkReflect::Type*>(hkUlong(0x22222222)), 0, 0, impl, "hkArray");
            }
            else
            {
                newArr = m_typesAllocator.create<hkReflect::BasicArrayType>(reinterpret_cast<const hkReflect::Type*>(0), 0, 0, impl, "hkArray");
            }
        }

        int size = 0;
        int align = 0;

        if (a->getSubType())
        {
            HK_COMPILE_TIME_ASSERT(HK_OFFSET_OF(hkReflect::BasicArrayType, m_elemType) == HK_OFFSET_OF(hkReflect::RepeatArrayType, m_elemType));
            dispatch(a->getSubType(), &static_cast<hkReflect::BasicArrayType*>(newArr)->m_elemType);
            dispatch(a->getSubType(), reinterpret_cast<hkReflect::QualType*>(&static_cast<hkReflect::BasicArrayTypeWithParams*>(newArr)->m_template->item(0).m_storage));
            const int fixedSize = a->getFixedCount();
            if (fixedSize > 0)
            {
                // RecordLayout will pick these up later
                size = 0;
                align = 0;
            }
            else
            {
                size = hkSizeOf(hkReflect::Detail::HomogeneousArrayImpl::InMemory);
                align = HK_ALIGN_OF(hkReflect::Detail::HomogeneousArrayImpl::InMemory);
            }
        }
        else
        {
            size = hkSizeOf(hkReflect::Detail::HomogeneousArrayImpl::InMemory);
            align = HK_ALIGN_OF(hkReflect::Detail::HomogeneousArrayImpl::InMemory);
        }
        hkReflect::TypeDetail::localSetSizeAlignPreserveReq(newArr, size, align);

        if (impl == m_homogenousArrayImpl)
        {
            DLOG("Generated_2 HA @{}", newArr);
        }

        if (size == 0)
        {
            m_typeSet.addDeferredLayout(newArr);
        }
        resolveTypeLoc(typeOut, newArr);
    }

    void visit(_In_ const hkReflect::OpaqueType* i, _Inout_ hkReflect::QualType* typeOut)
    {
        resolveTypeLoc(typeOut, i);
    }

    hkModifiableTypeSet& m_typeSet;
    hkTransientAllocator& m_typesAllocator;
    const hkReflect::Detail::ArrayImpl* m_arrayImpl;
    const hkReflect::Detail::StringImpl* m_stringImpl;
    const hkReflect::Detail::ArrayImpl* m_homogenousArrayImpl;
    const hkReflect::Detail::PointerImpl* m_pointerImpl;
};

void hkVersionedRecordType::addPatchedFieldFromTypeInformation(_In_z_ const char* fieldName, _In_ const hkReflect::Type* type)
{
    // This is a field with a builtin type that we are adding to a patched type
    // Any builtins we encounter need to be recreated from patching, not from the type that we can see

    OldRecordFieldPod* newField = m_owner->m_typesAllocator.create<OldRecordFieldPod>();
    m_fields.pushBack(newField);
    newField->m_name = fieldName;
    newField->m_flags = 0;
    newField->m_offset = 0;
    newField->m_type = reinterpret_cast<const hkReflect::Type*>(0x22222222);

    
    CopyPatchedTypeVisitor(*m_owner, m_owner->m_typesAllocator, m_owner->m_arrayImpl, m_owner->m_stringImpl, m_owner->m_homogenousArrayImpl, m_owner->m_pointerImpl).dispatch(type, reinterpret_cast<hkReflect::QualType*>(&newField->m_type));
}

void hkVersionedRecordType::applyReversedPatchToPatchedRecord(_In_ const hkReflect::Version::PatchInfo* patch)
{
    hkPatchDependencies dependencies(patch);

    for (int componentIndex = patch->numComponent - 1; componentIndex >= 0; componentIndex--)
    {
        const hkReflect::Version::PatchInfo::Component& component = patch->component[componentIndex];
        if (const hkReflect::Version::PatchInfo::SetParentPatch* setParentPatch = component.asSetParentPatch())
        {
            // Again, patch up from zero since it is not in the file
            // And set the parent to this new type
            if (setParentPatch->oldParent)
            {
                const int depVersion = dependencies.getVersion(setParentPatch->oldParent);
                m_owner->m_parentTodos.expandOne().set(setParentPatch->oldParent, depVersion, this);
            }
        }
        else if (const hkReflect::Version::PatchInfo::MemberAddedPatch* memberAddedPatch = component.asMemberAddedPatch())
        {
            for (int i = 0; i < m_fields.getSize(); i++)
            {
                if (!hkString::strCmp(memberAddedPatch->name, m_fields[i]->m_name))
                {
                    hkStringBuf newName("#");
                    newName.append(memberAddedPatch->name);
                    m_owner->m_names.pushBack(newName.cString());
                    m_fields[i]->m_name = m_owner->m_names.back();
                    break;
                }
            }
        }
        else if (const hkReflect::Version::PatchInfo::MemberRemovedPatch* memberRemovedPatch = component.asMemberRemovedPatch())
        {
            // Insert at the START of the object, as this member is from an older version
            // This is significant if the member is removed and re-added with the same name, the older version
            // must come before the newer in the file
            addPatchedFieldFromDescription(memberRemovedPatch->name, memberRemovedPatch->type, memberRemovedPatch->typeName, memberRemovedPatch->tuples, HK_NULL, 0, dependencies);
        }
        else if (const hkReflect::Version::PatchInfo::TypedMemberRemovedPatch* typedMemberRemovedPatch = component.asTypedMemberRemovedPatch())
        {
            // Insert at the START of the object, as this member is from an older version
            // This is significant if the member is removed and re-added with the same name, the older version
            // must come before the newer in the file

            addPatchedFieldFromTypeInformation(typedMemberRemovedPatch->name, typedMemberRemovedPatch->type);
        }
        else if (component.asDefaultChangedPatch())
        {
            // Intermediate patch functions will not get the default unless we set it in reverse here
            // In practice, this seems to be ok
        }
        else if (const hkReflect::Version::PatchInfo::MemberRenamedPatch* memberRenamedPatch = component.asMemberRenamedPatch())
        {
            for (int i = 0; i < m_fields.getSize(); i++)
            {
                if (!hkString::strCmp(memberRenamedPatch->newName, m_fields[i]->m_name))
                {
                    m_fields[i]->m_name = memberRenamedPatch->oldName;
                    break;
                }
            }
        }
        else if (const hkReflect::Version::PatchInfo::TypeChangedFunctionPatch* typeChangedPatch = component.asTypeChangedFunctionPatch())
        {
            // This is basically rename / add / function / remove
            // We don't bother with the first rename since we haven't added it yet, add and function does nothing here so it is just the remove that matters
            HK_ASSERTV(0x9ac2b711, findField(typeChangedPatch->m_name) != 0,
                "Removing member {} from type {} but the element does not exist",
                typeChangedPatch->m_name, patch->oldName);

            for (int i = 0; i < m_fields.getSize(); i++)
            {
                if (!hkString::strCmp(typeChangedPatch->m_name, m_fields[i]->m_name))
                {
                    hkStringBuf newName("#");
                    newName.append(typeChangedPatch->m_name);
                    m_owner->m_names.pushBack(newName.cString());
                    m_fields[i]->m_name = m_owner->m_names.back();
                    break;
                }
            }
            addPatchedFieldFromDescription(typeChangedPatch->m_name, typeChangedPatch->m_oldType, typeChangedPatch->m_oldTypeName, typeChangedPatch->m_oldTuples, HK_NULL, 0, dependencies);
        }

        else if (const hkReflect::Version::PatchInfo::TypeTemplateParameterRemovedPatch* typeRemoved = component.asTypeTemplateParameterRemovedPatch())
        {
            HK_ASSERT_NO_MSG(0x1b86a869, 0);
        }
        else if (const hkReflect::Version::PatchInfo::ValueTemplateParameterRemovedPatch* valueRemoved = component.asValueTemplateParameterRemovedPatch())
        {
            HK_ASSERT_NO_MSG(0x43a64fc5, 0);
        }
    }

    if (patch->newVersion == -1)
    {
        const hkUint32 VERSION_DELETED = 0x80000000U;
        hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::VERSION>(m_type, VERSION_DELETED);
    }
    else
    {
        hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::VERSION>(m_type, patch->newVersion);
    }
}

_Ret_maybenull_ hkVersionedRecordType::OldRecordFieldPod* hkVersionedRecordType::findField(_In_z_ const char* queryName)
{
    for (int i = 0; i < m_fields.getSize(); i++)
    {
        const char* thisFieldName = m_fields[i]->m_name;
        if (thisFieldName)
        {
            bool hashFound = false;
            while (*thisFieldName == '#')
            {
                hashFound = true;
                thisFieldName++;
            }

            if (hkString::strCmp(thisFieldName, queryName) == 0)
            {
                if (hashFound)
                {
                    m_fields[i]->m_name = thisFieldName;
                }
                return m_fields[i];
            }
        }
    }

    return HK_NULL;
}


void hkVersionedRecordType::applyPatchToPatchedRecord(_In_ const hkReflect::Version::PatchInfo* patch)
{
    hkPatchDependencies dependencies(patch);

    for (int componentIndex = 0; componentIndex < patch->numComponent; componentIndex++)
    {
        const hkReflect::Version::PatchInfo::Component& component = patch->component[componentIndex];
        if (const hkReflect::Version::PatchInfo::SetParentPatch* setParentPatch = component.asSetParentPatch())
        {
            if (setParentPatch->newParent)
            {
                const int depVersion = dependencies.getVersion(setParentPatch->newParent);
                m_owner->m_parentTodos.expandOne().set(setParentPatch->newParent, depVersion, this);
            }
        }
        else if (const hkReflect::Version::PatchInfo::MemberAddedPatch* memberAddedPatch = component.asMemberAddedPatch())
        {
            // Insert at the START of the object, as this member is from an older version
            // This is significant if the member is removed and re-added with the same name, the older version
            // must come before the newer in the file

            addPatchedFieldFromDescription(memberAddedPatch->name, memberAddedPatch->type, memberAddedPatch->typeName, memberAddedPatch->tuples, HK_NULL, 0, dependencies);
        }
        else if (const hkReflect::Version::PatchInfo::TypedMemberAddedPatch* typedMemberAddedPatch = component.asTypedMemberAddedPatch())
        {
            // Insert at the START of the object, as this member is from an older version
            // This is significant if the member is removed and re-added with the same name, the older version
            // must come before the newer in the file

            addPatchedFieldFromTypeInformation(typedMemberAddedPatch->name, typedMemberAddedPatch->type);
        }
        else if (const hkReflect::Version::PatchInfo::MemberRemovedPatch* memberRemovedPatch = component.asMemberRemovedPatch())
        {
            // Insert at the START of the object, as this member is from an older version
            // This is significant if the member is removed and re-added with the same name, the older version
            // must come before the newer in the file

            // Need to use this->m_fields, we are not copied out yet
            if (OldRecordFieldPod* f = findField(memberRemovedPatch->name))
            {
                f->m_name = HK_NULL;
            }
        }
        else if (const hkReflect::Version::PatchInfo::TypedMemberRemovedPatch* typedMemberRemovedPatch = component.asTypedMemberRemovedPatch())
        {
            // Insert at the START of the object, as this member is from an older version
            // This is significant if the member is removed and re-added with the same name, the older version
            // must come before the newer in the file

            if (OldRecordFieldPod* f = findField(typedMemberRemovedPatch->name))
            {
                f->m_name = HK_NULL;
            }
        }
        else if (component.asDefaultChangedPatch())
        {
            // Intermediate patch functions will not get the default unless we set it in reverse here
            // In practice, this seems to be ok
        }
        else if (const hkReflect::Version::PatchInfo::MemberRenamedPatch* memberRenamedPatch = component.asMemberRenamedPatch())
        {
            if (OldRecordFieldPod* f = findField(memberRenamedPatch->oldName))
            {
                f->m_name = memberRenamedPatch->newName;
            }
        }
        else if (const hkReflect::Version::PatchInfo::TypeChangedFunctionPatch* typeChangedPatch = component.asTypeChangedFunctionPatch())
        {
            // This is basically rename / add / function / remove
            // Since we don't care about the function here we null the old and then add the new member

            if (OldRecordFieldPod* f = findField(typeChangedPatch->m_name))
            {
                f->m_name = HK_NULL;
            }
            else
            {
                HK_ASSERTV(0x9ac2b710, 0, "Removing member {} from type {} but the element does not exist", typeChangedPatch->m_name, patch->oldName);
            }
            addPatchedFieldFromDescription(typeChangedPatch->m_name, typeChangedPatch->m_newType, typeChangedPatch->m_newTypeName, typeChangedPatch->m_newTuples, HK_NULL, -1, dependencies);

        }
        else if (const hkReflect::Version::PatchInfo::TypeTemplateParameterAddedPatch* tplTypeParamAdded = component.asTypeTemplateParameterAddedPatch())
        {
            HK_ASSERT_NO_MSG(0x672b9eef, 0);
        }
        else if (const hkReflect::Version::PatchInfo::ValueTemplateParameterAddedPatch* tplValueParamAdded = component.asValueTemplateParameterAddedPatch())
        {
            HK_ASSERT_NO_MSG(0x2625d7c7, 0);
        }
    }

    if (patch->newVersion == -1)
    {
        const hkUint32 VERSION_DELETED = 0x80000000U;
        hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::VERSION>(m_type, VERSION_DELETED);
    }
    else
    {
        hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::VERSION>(m_type, patch->newVersion);
        hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::NAME>(m_type, patch->newName);
    }
}

static void stripParamsFrom(hkStringBuf& sb)
{
    int idx = sb.indexOf("<");
    if (idx >= 0)
    {
        sb.chompEnd(sb.getLength() - idx);
    }
}

_Ret_maybenull_ hkVersionedRecordType* hkModifiableTypeSet::createType(_In_z_ const char* name, int startingVersion, _In_opt_ const hkReflect::Type* type)
{
    int version = startingVersion;
    // Create a (new) patched object from the name or return any cached version, patches up to compiled in or removed type.
    hkStringBuf reflectedName;
    if (name)
    {
        reflectedName = name;
        m_names.expandOne() = name;

        version = m_oldestVersionEncountered.getOrInsert(m_names.back(), version);

    }
    else
    {
        const char* n = type->getFullName(reflectedName);
        m_names.expandOne() = n;
        version = m_oldestVersionEncountered.getOrInsert(m_names.back(), type->getVersion());
    }
    const char* className;
    hkVersionedRecordType* thisRecord = HK_NULL;
    int newTypeIndex = -1;

    const hkReflect::Version::PatchInfo* firstPatch = HK_NULL;
    HK_ASSERT_NO_MSG(0x7ac4d567, version >= 0);
    const hkUint64 startingUid = m_patches.getUid(reflectedName, version);
    {
        HK_ASSERT_NO_MSG(0x29791035, version != -1); // Reserved value
        hkPointerMap<hkUint64, int>::Iterator it = m_typeIndexFromUid.findKey(startingUid);
        if (m_typeIndexFromUid.isValid(it))
        {
            const int newTypeId = m_typeIndexFromUid.getValue(it);
            if (newTypeId == INCOMPLETE_TYPE_MARKER)
            {
                return HK_NULL;
            }
            else
            {
                return m_recordTypes[newTypeId];
            }
        }
        else
        {
            // insert
            m_typeIndexFromUid.insert(startingUid, INCOMPLETE_TYPE_MARKER);
        }

        m_names.expandOne() = reflectedName;
        className = m_names.back().cString();
    }

    firstPatch = stripV2Patches(m_patches.addPatchFor(reflectedName, version));

    int lastVersionApplied = version;

    hkArray<const hkReflect::Version::PatchInfo*> patchesToApply; 

    const char* currentTypeName = className;
    bool typeRenamed = false;

    // Every name this type has been called
    hkArray<const char*> allTypeNames;
    hkArray<hkUint64> allUids;
    allTypeNames.pushBack(className);
    allUids.pushBack(startingUid);

    // Need to set incomplete for all versions of this type, not just the starting one to avoid loops
    for (const hkReflect::Version::PatchInfo* thisPatch = firstPatch; thisPatch; thisPatch = stripV2Patches(m_patches.addPatchFor(thisPatch->newName, thisPatch->newVersion)))
    {
        if (!thisPatch->isClassAdded())
        {
            patchesToApply.pushBack(thisPatch);
        }

        if (thisPatch->newName)
        {
            const hkUint64 thisUid = m_patches.getUid(thisPatch->newName, thisPatch->newVersion);
            m_typeIndexFromUid.insert(thisUid, INCOMPLETE_TYPE_MARKER);
        }
    }

    for (int pi = 0; pi < patchesToApply.getSize(); pi++)
    {
        const hkReflect::Version::PatchInfo* patch = patchesToApply[pi];
        {
            // Run dependencies first, on the forward walk
            for (int componentIndex = patch->numComponent - 1; componentIndex >= 0; componentIndex--)
            {
                const hkReflect::Version::PatchInfo::Component& component = patch->component[componentIndex];
                if (const hkReflect::Version::PatchInfo::DependsPatch* dependsPatch = component.asDependsPatch())
                {
                    const hkUint64 dependsUid = m_patches.getUid(dependsPatch->name, dependsPatch->version);
                    const int dependsIndex = m_typeIndexFromUid.getWithDefault(dependsUid, -1);
                    // It is either incomplete (in progress) or done, either way we don't need to start it
                    if (dependsIndex == -1)
                    {
                        // Run the patches to calculate the size
                        if (createType(dependsPatch->name, dependsPatch->version, HK_NULL) == HK_NULL)
                        {
                            // A dependency failed -> we fail
                            return HK_NULL;
                        }
                    }
                }
            }

            if (patch->newName && currentTypeName && hkString::strCmp(patch->newName, currentTypeName)) // name has changed
            {
                DLOG_IF(const char* oldName = currentTypeName);
                {
                    currentTypeName = patch->newName;
                }
                // Store the type rename
                allTypeNames.pushBack(currentTypeName);

                DLOG("Inserting type from rename {} -> {}({}) at {}", oldName, currentTypeName, version, newTypeIndex);

                typeRenamed = true;
            }

            if (patch->newName)
            {
                const hkUint64 thisUid = m_patches.getUid(patch->newName, patch->newVersion);
                allUids.pushBack(thisUid);
            }
        }

        lastVersionApplied = patch->newVersion;
    }

    // m_patchesApplied is never modified after this

    const hkReflect::Type* currentType = HK_NULL;
    if (lastVersionApplied >= 0)
    {
        className = currentTypeName;

        // Class was not deleted, so it should be in the currently loaded classes

        // If the type has been renamed, we can't use the templated version. Template patching when supported will handle
        // Changing the name or parameters
        if (type && !typeRenamed)
        {
            currentType = m_lookupType->typeFromType(type);
        }
        else
        {
            currentType = m_lookupType->typeFromName(className);
        }

        if (!currentType)
        {
            if (m_typeOptions.noneIsSet(BEST_EFFORT_ON_INCOMPLETE_TYPES))
            {
                Log_Warning("Patches are trying to create class {} but it is not registered", className).setId(0x2db84748);
                return HK_NULL;
            }
        }

        if (currentType)
        {
            if (currentType->getVersion() != lastVersionApplied)
            {
                if (m_typeOptions.noneIsSet(BEST_EFFORT_ON_INCOMPLETE_TYPES))
                {
                    Log_Warning("Patches say class {} version {} should be present, but version {} is registered",
                        className, lastVersionApplied, currentType->getVersion()).setId(0x6a320fb);
                    return HK_NULL;
                }
            }
        }
    }

    // If we get here, our type creation is likely to work
    hkReflect::RecordType* record = HK_NULL;
    {
        hkReflect::TypeBuilder builder;
        builder.setTypeWorld(this);
        hkStringBuf nameWithoutParams(allTypeNames[0]);
        stripParamsFrom(nameWithoutParams);
        m_names.pushBack(nameWithoutParams.cString());
        builder.beginRecord(m_names.back(), HK_NULL); // Start from the FIRST type name, not the last
        builder.addItem<hkReflect::Opt::ALLOC_IMPL>(&m_allocImpl);
        builder.addItem<hkReflect::Opt::VERSION>(version);

        if(currentType)
        {
            hkSerialize::CompatTypeParentInfo::Parent* firstParent = nullptr;
            hkSerialize::CompatTypeParentInfo::Parent* currentParent = nullptr;

            for (const hkReflect::Type* parentIn = currentType->getParent(); parentIn; parentIn = parentIn->getParent())
            {
                if (const hkReflect::Type* t = m_lookupType->typeFromType(parentIn))
                {
                    hkStringBuf sbTemp;
                    const char* nameTemp = t->getFullName(sbTemp);

                    hkSerialize::CompatTypeParentInfo::Parent* newParent = m_typesAllocator.create<hkSerialize::CompatTypeParentInfo::Parent>(hkString::strDup(nameTemp, m_typesAllocator), t->getVersion());
                    if (currentParent)
                    {
                        currentParent->m_next = newParent;
                        currentParent = newParent;
                    }
                    else
                    {
                        currentParent = newParent;
                        firstParent = newParent;
                    }
                }
            }

            if (firstParent)
            {
                // Store the parent info for version bundle
                hkSerialize::CompatTypeParentInfo* info = m_typesAllocator.create<hkSerialize::CompatTypeParentInfo>();
                info->m_firstParent = firstParent;
                builder.addAttribute(info);
            }
        }

        if (reflectedName.indexOf('<') >= 0)
        {
            // This is a template
            builder.addItem<hkReflect::Opt::TEMPLATE>(HK_NULL);
        }

        hkReflect::Type* allocatedType;
        if (builder.allocate(&m_typesAllocator, &allocatedType).isFailure())
        {
            Log_Error("Failed to allocate type from builder");
            return nullptr;
        }
        record = allocatedType->asRecord();
    }

    newTypeIndex = m_recordTypes.getSize();
    thisRecord = m_typesAllocator.create<hkVersionedRecordType>(this, (hkReflect::Type*)record, version, hkVersionedRecordType::STATUS_INPROGRESS, type, startingUid);

    m_recordTypes.pushBack(thisRecord);

    HK_ASSERT_NO_MSG(0x618c563f, hkString::strLen(className) > 0);

    for (int i = 0; i < allUids.getSize(); i++)
    {
        const hkUint64 n = allUids[i];
        m_typeIndexFromUid.insert(n, newTypeIndex);
    }

    DLOG("Inserting type {}({}) at {}", className, version, newTypeIndex);

    if ((lastVersionApplied >= 0) && currentType)
    {
        const hkReflect::RecordType* currentRecord = currentType->asRecord();

        HK_ASSERT_NO_MSG(0x4f83f1c3, currentRecord);

        // Iterate over our fields without parent records
        for (int i = 0; i < currentRecord->getNumFields(); i++)
        {
            const hkReflect::FieldDecl f = currentRecord->getField(i);
            if (f.isSerializable())
            {
                thisRecord->addPatchedFieldFromTypeInformation(f.getName(), f.getType());
            }
        }

        if (const hkReflect::Template* params = currentRecord->getTemplate())
        {
            for (int i = 0; i < params->getNumParams(); i++)
            {
                const hkReflect::Template::Parameter* param = params->getParam(i);
                thisRecord->addTemplateParameter(param->m_kindAndName, param->m_storage);
            }
        }

        // Add parent todo
        if (const hkReflect::RecordType* parent = currentRecord->getParentRecord())
        {
            m_parentTodos.expandOne().set(m_lookupType->typeFromType(parent), thisRecord);
        }
    }
    // We only do this when we are NOT going to build the type entirely from patches.
    for (int patchIndex = patchesToApply.getSize() - 1; patchIndex >= 0; patchIndex--)
    {
        //      patchIndex = patchesToApply.back();
        //      patchesToApply.popBack();
        //
        //      const hkVersionPatchManager::PatchInfo* patch = m_patches.getPatch(patchIndex);

        const hkReflect::Version::PatchInfo* patch = patchesToApply[patchIndex];

        DLOG("PATCH visit {} {}->{} {} (at ---)", patch->oldName ? patch->oldName : "<none>", patch->oldVersion, patch->newName ? patch->newName : "<none>", patch->newVersion/*, patchIndex*/);

        thisRecord->applyReversedPatchToPatchedRecord(patch); // Reversed as we want to ADD any members REMOVED by patches

        lastVersionApplied = patch->newVersion;
        if (patch->newName)
        {
            if (patch->newName != patch->oldName)
            {
                HK_ON_DEBUG(hkStringBuf reflectedNameDebug(patch->newName));
                HK_ASSERT_NO_MSG(0x47d2fec1, reflectedNameDebug.getLength() > 0);

            }
            // If newname is null, keep the old name just so we can check it for deleted classes
            className = patch->newName;
        }

        //uid = m_patches.getUid(patch->newName, patch->newVersion);
        //patchIndex = m_patches.getPatchIndex(uid);
        //patch = m_patches.getPatchFor(patch->newName, patch->newVersion); // Not needed???
    }

    return thisRecord;
}

_Ret_maybenull_ hkVersionedRecordType* hkModifiableTypeSet::createFinalTypeFromPatches(_In_z_ const char* name, int startingVersion, _In_opt_ const hkReflect::Type* type)
{
    int version = startingVersion;
    hkStringBuf reflectedName;

    if (startingVersion == -1)
    {
        startingVersion = 0;
        version = startingVersion;
    }
    if (name)
    {
        reflectedName = name;
        m_names.expandOne() = name;

        
        if (const hkReflect::Type* lookupType = m_lookupType->typeFromName(name))
        {
            if ((hkString::strCmp(lookupType->getName(), "hkHalf16") == 0) || hkString::strCmp(lookupType->getName(), "hkPropertyBag") == 0)
            {
                // This is horrible but hkHalf16 has been a record and is now a float, this confuses things
            }
            else
            {
                if (!lookupType->asRecord())
                {
                    // Not a record
                    return HK_NULL;
                }
            }
        }

    }
    else
    {
        const char* n = type->getFullName(reflectedName);
        reflectedName = n;
        m_names.expandOne() = reflectedName;
        version = m_oldestVersionEncountered.getOrInsert(m_names.back(), type->getVersion());
    }
    const char* className;
    hkVersionedRecordType* thisRecord = HK_NULL;
    int newTypeIndex = -1;
    // We need to remove :: etc

    {
        if (hkVersionedRecordType* ret = findType(reflectedName.cString(), version))
        {
            return ret;
        }

        m_names.expandOne() = reflectedName;
        className = m_names.back().cString();
    }

    const char* currentTypeName = className;
    bool typeRenamed = false;

    // Every name this type has been called
    hkArray<const char*> allTypeNames;
    allTypeNames.pushBack(className);
    hkArray<hkUint64> allUids;
    const hkUint64 startingUid = m_patches.getUid(className, version);
    allUids.pushBack(startingUid);

    hkReflect::RecordType* record = HK_NULL;
    {
        hkReflect::TypeBuilder builder;
        builder.setTypeWorld(this);
        hkStringBuf nameWithoutParams(allTypeNames[0]);
        //stripParamsFrom(nameWithoutParams); 
        m_names.pushBack(nameWithoutParams.cString());
        builder.beginRecord(m_names.back(), HK_NULL); // Start from the FIRST type name, not the last
        builder.addItem<hkReflect::Opt::ALLOC_IMPL>(&m_allocImpl);
        builder.addItem<hkReflect::Opt::VERSION>(version);

        if (reflectedName.indexOf('<') >= 0)
        {
            // This is a template
            builder.addItem<hkReflect::Opt::TEMPLATE>(HK_NULL);
        }

        hkReflect::Type* allocatedType;
        if (builder.allocate(&m_typesAllocator, &allocatedType).isFailure())
        {
            Log_Error("Failed to allocate type from builder");
            return nullptr;
        }
        record = allocatedType->asRecord();
    }

    thisRecord = m_typesAllocator.create<hkVersionedRecordType>(this, record, version, hkVersionedRecordType::STATUS_INPROGRESS, type, startingUid);
    newTypeIndex = m_recordTypes.getSize();
    m_recordTypes.pushBack(thisRecord);

    m_typeIndexFromUid.insert(allUids[0], newTypeIndex);

    const hkReflect::Version::PatchInfo* patch = m_patches.addClassAddedPatchFor(className, version);
    hkArray<const hkReflect::Version::PatchInfo*> patchesToApply;

    if (!patch)
    {
        patch = stripV2Patches(m_patches.addPatchFor(className, version));
    }

    while (patch)
    {
        patchesToApply.pushBack(patch);
        // Run dependencies first, on the forward walk
        for (int componentIndex = patch->numComponent - 1; componentIndex >= 0; componentIndex--)
        {
            const hkReflect::Version::PatchInfo::Component& component = patch->component[componentIndex];
            if (const hkReflect::Version::PatchInfo::DependsPatch* dependsPatch = component.asDependsPatch())
            {
                // Run the patches to calculate the size
                if (createFinalTypeFromPatches(dependsPatch->name, dependsPatch->version, HK_NULL) == HK_NULL) // was createType
                {
                    // A dependency failed -> we fail
                    return HK_NULL;
                }
            }
        }

        if (patch->newName && currentTypeName && hkString::strCmp(patch->newName, currentTypeName)) // name has changed
        {
            DLOG_IF(const char* oldName = currentTypeName);
            {
                currentTypeName = patch->newName;
            }
            // Store the type rename
            allTypeNames.pushBack(currentTypeName);

            DLOG("Inserting type from rename {} -> {}({}) at {}", oldName, currentTypeName, version, newTypeIndex);

            typeRenamed = true;
        }

        if (patch->newName)
        {
            const hkUint64 thisUid = m_patches.getUid(patch->newName, patch->newVersion);
            allUids.pushBack(thisUid);
        }
        patch = stripV2Patches(m_patches.addPatchFor(patch->newName, patch->newVersion));
    }

    // m_patchesApplied is never modified after this
    HK_ASSERT_NO_MSG(0x59788e36, hkString::strLen(className) > 0);

    for (int i = 0; i < allUids.getSize(); i++)
    {
        const hkUint64 n = allUids[i];
        m_typeIndexFromUid.insert(n, newTypeIndex);
    }
    DLOG("Inserting type {}({}) at {}", className, version, newTypeIndex);

    // Run forward from start to here to get the types not mentioned in the chain we have made
    for (int i = 0; i < patchesToApply.getSize(); i++)
    {
        const hkReflect::Version::PatchInfo* fromStartPatch = patchesToApply[i];

        thisRecord->applyPatchToPatchedRecord(fromStartPatch);
    }

    return thisRecord;
}

struct hkModifiableTypeSet::AddDependentTodoVisitor : public hkReflect::TypeVisitor < hkModifiableTypeSet::AddDependentTodoVisitor, void >
{
    AddDependentTodoVisitor(hkArray<TypeTodo>& todos)
    : m_todos(todos)
    {
    }

    void visit(_In_ const hkReflect::VoidType* v) {}
    void visit(_In_ const hkReflect::ValueType* v) {}
    void visit(_In_ const hkReflect::StringType* s) {}
    void visit(_In_ const hkReflect::OpaqueType* i) {}
    void visit(_In_ const hkReflect::PointerType* p)
    {
        if(const hkReflect::Type* subType = p->getSubType())
        {
            dispatch(subType);
        }
    }

    void visit(_In_ const hkReflect::RecordType* r)
    {
        if (r->isDynamicType())
        {
            return;
        }

        bool found = false;
        for (int i = 0; i < m_todos.getSize(); i++)
        {
            if ((m_todos[i].m_sourceTypeName && (hkString::strCmp(r->getName(), m_todos[i].m_sourceTypeName) == 0)) || (r == m_todos[i].m_type))
            {
                found = true;
                break;
            }
        }
        if (!found)
        {
            DLOG("Add TODO4 {}({})", r->getFullName(), r->getVersion()); // From the file, actually a type

            DLOG("File contains {}({})", r->getFullName(), r->getVersion());

            // We don't support dependent types here (yet)
            m_todos.expandOne().set(r, HK_NULL);

            m_olderVersions->insert(r->getName(), r->getVersion());
            for( hkReflect::DeclIter<hkReflect::FieldDecl> it(r); it.advance(); )
            {
                dispatch(it.current().getType());
            }

            // We need to dispatch on the parent so we get the correct version
            if (const hkReflect::Type* parent = r->getParent())
            {
                dispatch(parent);
            }
        }
    }

    void visit(_In_ const hkReflect::ArrayType* a)
    {
        if (a->getSubType())
        {
            dispatch(a->getSubType());
        }
    }

    hkArray<TypeTodo>& m_todos;
    hkStringMap<int>* m_olderVersions;
};

void hkModifiableTypeSet::addAllDependentTodos(_In_ const hkReflect::Type* sourceType)
{
    AddDependentTodoVisitor vis(m_typeTodos);
    vis.m_olderVersions = &m_oldestVersionEncountered;
    vis.dispatch(sourceType);
}

void hkModifiableTypeSet::TypeTodo::resolve(_In_ const hkReflect::Type* t)
{
    if (m_location) // We sometimes todo to nothing, just to create a dependency
    {
        HK_ASSERT_NO_MSG(0x2c6311bf, (m_location->get() == reinterpret_cast<const hkReflect::Type*>(0x22222222)) || (m_location->get() == 0));
        m_location->set(t, false);
    }
}

void hkVersionedRecordType::addTemplateParameter(_In_z_ const char* kindAndName, hkUlong value)
{
    for (int i = 0; i < m_templateParameters.getSize(); i++)
    {
        if (hkString::strCmp(m_templateParameters[i].m_kindAndName, kindAndName) == 0)
        {
            return;
        }
    }
    m_templateParameters.expandOne();
    m_templateParameters.back().m_kindAndName = kindAndName;
    m_templateParameters.back().m_storage = value;
}

void hkVersionedRecordType::renameType(_In_z_ const char* oldPatchName, _In_z_ const char* newPatchName)
{
    const char* oldTypeName = getCompletedType()->getName();
    hkStringBuf newName;
    if (hkString::strCmp(oldTypeName, oldPatchName) == 0)
    {
        newName = newPatchName;
    }
    else
    {
        HK_ASSERT_NO_MSG(0x190bf798, hkString::strStr(oldTypeName, oldPatchName) == oldTypeName);
        newName = newPatchName;
        newName.append(oldTypeName + hkString::strLen(oldPatchName));
    }

    m_owner->m_names.pushBack(newName.cString());
    hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::NAME>(getCompletedType(), m_owner->m_names.back().cString());
}

bool hkModifiableTypeSet::anyPatchesNeededFor(_In_ const hkReflect::Type* t)
{
    int valOut;
    if (m_patchesNeededForType.get(t, &valOut).isSuccess())
    {
        if ((valOut == NO_PATCH_NEEDED) || (valOut == IN_PROGRESS))
        {
            return false;
        }
        return true;
    }

    m_patchesNeededForType.insert(t, IN_PROGRESS);

    if (const hkReflect::RecordType* rt = t->asRecord())
    {
        if (m_patches.addPatchFor(t->getName(), t->getVersion()))
        {
            m_patchesNeededForType.insert(t, PATCH_NEEDED);
            return true;
        }

        if (const hkReflect::Type* pt = t->getParent())
        {
            if (anyPatchesNeededFor(pt))
            {
                m_patchesNeededForType.insert(t, PATCH_NEEDED);
                return true;
            }
        }
        for (int i = 0; i < rt->getNumFields(); i++)
        {
            hkReflect::FieldDecl fd = rt->getField(i);
            if(anyPatchesNeededFor(fd.getType()))
            {
                m_patchesNeededForType.insert(t, PATCH_NEEDED);
                return true;
            }
        }
    }
    else if (const hkReflect::ArrayType* at = t->asArray())
    {
        if (at->getSubType() && anyPatchesNeededFor(at->getSubType()))
        {
            m_patchesNeededForType.insert(t, PATCH_NEEDED);
            return true;
        }
    }
    // Some patches seem to rely on this happening, in theory it shouldn't be needed
    else if (const hkReflect::PointerType* pt = t->asPointer())
    {
        if (pt->getSubType() && anyPatchesNeededFor(pt->getSubType()))
        {
            m_patchesNeededForType.insert(t, PATCH_NEEDED);
            return true;
        }
    }

    m_patchesNeededForType.insert(t, NO_PATCH_NEEDED);
    return false;
}

void hkModifiableTypeSet::createAllTypes()
{
    m_patches.addAllPatchesFromRegistry();
    hkArrayView<const hkReflect::Version::PatchInfo*> patchList = m_patches.getPatchList();

    for (int i = patchList.getSize() - 1; i >= 0; i--)
    {
        const hkReflect::Version::PatchInfo* thisPatch = patchList[i];
        if (thisPatch && thisPatch->newName)
        {
            // Only do this if the patch is the final patch in a chain
            if (!m_patches.addPatchFor(thisPatch->newName, thisPatch->newVersion))
            {
                createFinalTypeFromPatches(thisPatch->newName, thisPatch->newVersion);
            }
        }
    }
}

void hkModifiableTypeSet::appendAllTypesToArray(hkArray<hkVersionedRecordType*>& a) const
{
    for (int i = 0; i < m_recordTypes.getSize(); i++)
    {
        if (m_recordTypes[i])
        {
            a.pushBack(m_recordTypes[i]);
        }
    }
}

void hkVersionedRecordType::removeParent(_In_ hkVersionedRecordType* parent)
{
    const int parentIndex = m_parents.indexOf(parent);
    if(parentIndex >= 0)
    {
        m_parents.removeAtAndCopy(parentIndex);
    }
}

void hkVersionedRecordType::invalidateType()
{
    const hkUint64 startingUid = m_startingUid;
    m_owner->m_typeIndexFromUid.insert(startingUid, INCOMPLETE_TYPE_MARKER);

    const hkReflect::Version::PatchInfo* firstPatch = m_owner->m_patches.getPatch(startingUid);

    // Need to set incomplete for all versions of this type, not just the starting one to avoid loops
    for (const hkReflect::Version::PatchInfo* thisPatch = firstPatch; thisPatch; thisPatch = m_owner->m_patches.addPatchFor(thisPatch->newName, thisPatch->newVersion))
    {
        if (thisPatch->newName)
        {
            const hkUint64 thisUid = m_owner->m_patches.getUid(thisPatch->newName, thisPatch->newVersion);
            m_owner->m_typeIndexFromUid.insert(thisUid, INCOMPLETE_TYPE_MARKER);
        }
    }
}

hkPatchDependencies::hkPatchDependencies(_In_ const hkReflect::Version::PatchInfo* patch)
{
    addPatch(patch);
}

void hkPatchDependencies::clear()
{
    m_versions.clear();
}

void hkPatchDependencies::addPatch(_In_ const hkReflect::Version::PatchInfo* patch)
{
    // Our own version is an implicit dependency
    if (patch->newName)
    {
        m_versions.insert(patch->newName, patch->newVersion);
    }
    else
    {
        // Remove patch
        m_versions.insert(patch->oldName, patch->oldVersion);
    }
    for (int componentIndex = patch->numComponent - 1; componentIndex >= 0; componentIndex--)
    {
        const hkReflect::Version::PatchInfo::Component& component = patch->component[componentIndex];
        if (const hkReflect::Version::PatchInfo::DependsPatch* depends = component.asDependsPatch())
        {
            m_versions.insert(depends->name, depends->version);
        }
    }
}

int hkPatchDependencies::getVersion(_In_opt_z_ const char* typeName) const
{
    if (typeName)
    {
        const int ret = m_versions.getWithDefault(typeName, -1);
        HK_ASSERT(0xdf02cc87, ret != -1, "no version for {}", typeName);
        return ret;
    }
    return -1;
}

int hkPatchDependencies::tryGetVersion(_In_opt_z_ const char* typeName) const
{
    if (typeName)
    {
        const int ret = m_versions.getWithDefault(typeName, -1);
        return ret;
    }
    return -1;
}

bool hkPatchDependencies::nameIsOk(_In_opt_z_ const char* typeName) const
{
    if (typeName)
    {
        return m_versions.hasKey(typeName);
    }
    return true;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
