// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : COMMON
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/System/Log/hkLog.h>
#include <Common/Base/Container/Hash/hkHashSet.h>

namespace hkReflect
{
namespace Detail
{
    namespace CheckUtil
    {
        /// Callback to check for type reachability.
        typedef bool (HK_CALL *IsTypeReachableCallback)( const hkReflect::Type*, void* /* userData */ );

        /// Special value of type reachable callback returning true if the type is present in the incremental checker already.
        const IsTypeReachableCallback TYPE_REACHABLE_IF_PRESENT_IN_BATCH = (IsTypeReachableCallback)0x1;

        /// Checks configuration.
        struct Config
        {
            enum
            {
                // Bits 0-12 are valid flags; everything else is invalid.
                DEFAULT_ALLOWED_FLAGS = hkUint32( 0x1FFF ),

                // Bits 8 and 31 are invalid. All other Opt bits are allowed by default.
                DEFAULT_ALLOWED_OPT = hkUlong( -1 ) & ~( ( hkUlong( 1 ) << 8 ) | ( hkUlong( 1 ) << 31 ) )
            };


            //
            // Execution control
            //

            /// Stop checks on first failure?
            bool m_earlyOutOnFirstFailure;

            /// Mask of allowed hkReflect::Type flags.
            hkReflect::Type::Flags m_allowedFlags;

            /// Mask of allowed hkReflect::Opt bits.
            hkUlong m_allowedOptMask;


            //
            // Consistency checks
            //

            /// Callback to check if a type is reachable.
            /// Can be NULL to skip test, or TYPE_REACHABLE_IF_PRESENT_IN_BATCH (default) to use the built-in callback
            /// which returns that a type is reachable if already added to the incremental checker.
            IsTypeReachableCallback m_isTypeReachableCallback;

            /// User data passed to the type reachable callback.
            void* m_isTypeReachableUserData;

            /// Allow duplicate placeholder decls in a type. A decl is considered a placeholder if its name starts
            /// with '$'.
            bool m_allowedPlaceholderDecls;

            //
            // Logging
            //

            /// Log origin.
            const hkLog::Origin* m_origin;

            /// Log level.
            hkLog::Level::Enum m_level;


            /// Default constructor.
            Config()
                : m_earlyOutOnFirstFailure( true )
                , m_allowedFlags( DEFAULT_ALLOWED_FLAGS )
                , m_allowedOptMask( DEFAULT_ALLOWED_OPT )
                , m_isTypeReachableCallback( TYPE_REACHABLE_IF_PRESENT_IN_BATCH )
                , m_isTypeReachableUserData( nullptr )
                , m_allowedPlaceholderDecls( false )
                , m_origin( nullptr )
                , m_level( hkLog::Level::Error )
            {
            }
        };

        /// Incremental check utility.
        /// Allows incrementally adding types to an internal collection, validating new types as they are added
        /// and checking them for consistency by making sure they reference only types in the collection.
        class HK_EXPORT_COMMON IncrementalChecker
        {
        public:

            /// Constructor.
            IncrementalChecker( const Config& config = Config() );

            /// Set the execution configuration.
            void setConfig( const Config& config );

            /// Reset and clear all internal states.
            void reset();

            /// Check some optionals against the allowed mask set in the configuration.
            bool areOptsValid( hkUint32 opts ) const;

            /// Add new types to the collection of types to check.
            /// Those types are immediately checked for validity, and consistency with existing types in the collection.
            /// Return the result for this batch only. The general accumulated result for all addAndCheckTypes() calls
            /// since last reset() is available via getStatus().
            _Must_inspect_result_
            hkResult addAndCheckTypes( hkArrayView<const hkReflect::Type* const> types );

            /// Add new types to the collection of types to check.
            /// Variant taking a TypeReg linked list.
            _Must_inspect_result_
            hkResult addAndCheckTypes( _In_ const struct TypeRegNode* head );

            /// Invalidate a type, forcing it to be checked again.
            /// This is used e.g. by tagfile to check for loops again after a type is relocated (bundle streaming).
            void invalidateType( _In_ const hkReflect::Type* type );

            /// Get current status for all previous check operations since last reset.
            _Must_inspect_result_
            hkResult getStatus() const
            {
                return m_status;
            }

            /// Remove some types from the checker.
            void removeTypes( hkArrayView<const hkReflect::Type*> types );

        protected:

            /// Helper to ensure that a type is reachable if and only if it is in a given type collection.
            struct ReachabilityHelper
            {
                ReachabilityHelper(const hkHashSet<const hkReflect::Type*>& validTypes) : m_validTypes(validTypes) {}

                    /// Callback for type reachable check.
                _Check_return_
                static bool HK_CALL s_isTypeReachable( _In_ const hkReflect::Type* type, _In_ void* helper );

                /// Collection of valid types.
                const hkHashSet<const hkReflect::Type*>& m_validTypes;

                /// Current batch of new types being checked.
                hkArrayView<const hkReflect::Type* const> m_newTypes;
            };

            /// Helper to do outer checks and loop checks.
            _Must_inspect_result_
            hkResult doOuterAndLoopChecks( hkArrayView<const hkReflect::Type* const> types );

        protected:

            /// Collection of valid types to use for consistency check with new types.
            hkHashSet<const hkReflect::Type*> m_validTypes;

            /// Checks configuration.
            Config m_config;

            /// Global check status since last reset.
            hkResult m_status;

            /// Helper for type reachability check.
            ReachabilityHelper m_reachabilityHelper;
        };
    }
}
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
