// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : COMMON
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Util/hkReflectionCheckUtil.h>

#include <Common/Base/Container/Hash/hkHashSet.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Container/RelArray/hkRelArray.h>
#include <Common/Base/Container/Set/hkSet.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/System/Log/hkLog.h>

//Inherited == c++ inheritance
//Extended == type inheritance (adding optionals etc)
//"present" means in the current type
//"available" means in the parent chain


// Types must have a name available
// Records must have a name present (unless extended)
// Records parent type can only be record
// Non-abstract types must have nonzero size and align




// Checks in order of increasing completeness:
// 1) "reachable"  : a type pointer points to an address that is expected to contain a hkReflect::Type, as defined by the caller's callback.
// 2) "valid"      : the type has passed all "inner" checks on its own data, excluding any dereferencing to other types (e.g. parent or sub-type).
// 3) "consistent" : the type is valid, and has passed all "outer" checks, that is all other types referenced (e.g. parent or sub-type) are reachable
//                   and contain compatible data (e.g. the parent of a record is itself a record).


namespace
{
    using namespace hkReflect::Detail::CheckUtil;

    HK_ALWAYS_INLINE bool isFailure( bool cond ) { return !cond; }
    HK_ALWAYS_INLINE bool isFailure( hkResult ret ) { return ret.isFailure(); }

    /// Helper to catch erroneous FAILURE -> SUCCESS transition.
    class Status
    {
    public:
        HK_ALWAYS_INLINE Status( hkResult result ) : m_result( result ) {}
        HK_ALWAYS_INLINE void reset() { m_result = HK_SUCCESS; }
        HK_ALWAYS_INLINE operator hkResult() const { return m_result; }
        HK_ALWAYS_INLINE Status& operator=( hkResult ret )
        {
            HK_ASSERT( 0xA62E68D7, ret.isFailure() || m_result.isSuccess(), "Tried to change the global status of a check run from HK_FAILURE to HK_SUCCESS." );
            m_result = ret;
            return ( *this );
        }
    private:
        hkResult m_result;
    };

    // Utility for debugging; put a breakpoint here to catch deserialization errors.
    HK_NEVER_INLINE void setFail( Status& status )
    {
        status = HK_FAILURE;
    }

    HK_ALWAYS_INLINE _Ret_z_ const char* getExpectedMessage( bool expected )
    {
        return ( expected ? "expected but not found" : "found but not expected" );
    }

    /// Log a message and fail if the condition is false.
#define ENSURE_SUCCESS_OR_RETURN_WITH_ID( cond, id, ... ) \
    if ( isFailure( cond ) ) \
    { \
        setFail( m_status ); \
        hkLog_Log( (*m_config.m_origin), m_config.m_level, __VA_ARGS__ ).setId( id ); \
        return HK_FAILURE; \
    }

    /// Log a message and fail if the condition is false.
#define ENSURE_SUCCESS_OR_RETURN( cond, ... ) \
    ENSURE_SUCCESS_OR_RETURN_WITH_ID( cond, 0, __VA_ARGS__ )

    /// Validate a condition, and if the condition is false then log a message and possibly early out (if early out is enabled).
#define CHECK_SUCCESS( cond, ... ) \
    if ( isFailure( cond ) ) \
    { \
        setFail( m_status ); \
        hkLog_Log( (*m_config.m_origin), m_config.m_level, __VA_ARGS__ ); \
        if ( !keepGoing() ) \
        { \
            return HK_FAILURE; \
        } \
    }

    /// Validate a condition, and if the condition is false then log a message and possibly early out (if early out is enabled).
    /// Otherwise do a 'continue'. Useful inside a loop.
#define CHECK_SUCCESS_OR_CONTINUE( cond, ... ) \
    if ( isFailure( cond ) ) \
    { \
        setFail( m_status ); \
        hkLog_Log( (*m_config.m_origin), m_config.m_level, __VA_ARGS__ ); \
        if ( !keepGoing() ) \
        { \
            return HK_FAILURE; \
        } \
        continue; \
    }

    /// Validate a condition, and if the condition is false then possibly early out (if early out is enabled).
#define CHECK_SUCCESS_NO_LOG( cond ) \
    if ( isFailure( cond ) ) \
    { \
        setFail( m_status ); \
        if ( !keepGoing() ) \
        { \
            return HK_FAILURE; \
        } \
    }

    enum TrileanEnum
    {
        triMaybe = -1,
        triFalse = 0,
        triTrue = 1
    };

    /// Tri-state "boolean" with "maybe" option.
    class Trilean
    {
    public:
        HK_ALWAYS_INLINE Trilean( TrileanEnum value ) : m_value( value ) {}
        HK_ALWAYS_INLINE Trilean( bool value ) : m_value( TrileanEnum( int( value ) ) ) {}
        HK_ALWAYS_INLINE Trilean& operator=( TrileanEnum value )
        {
            m_value = value;
            return ( *this );
        }
        HK_ALWAYS_INLINE Trilean& operator=( bool value )
        {
            m_value = TrileanEnum( int( value ) );
            return ( *this );
        }
        HK_ALWAYS_INLINE operator TrileanEnum() const
        {
            return m_value;
        }
        HK_ALWAYS_INLINE operator bool() const
        {
            HK_ASSERT( 0xC8EC4001, m_value != triMaybe, "Cannot convert trilean to boolean when value is 'maybe'." );
            return bool( m_value );
        }
    protected:
        TrileanEnum m_value;
    };

    /// Bits 8 and 31 are always invalid - keep this in sync with hkReflect::Opt::Values.
    const hkUlong INVALID_OPT_BITS = hkUlong( ( 1 << 8 ) | ( 1 << 31 ) );

    /// Dummy hkLog origin which is always disabled, discards all messages.
    const hkLog::RegisteredOrigin s_disabledOrigin( "reflect.checkUtil.disabled", hkLog::Level::Disabled );

    /// Base visitor for hReflect::Type checks.
    class CheckVisitorBase
    {
    public:

        /// Constructor without logging.
        CheckVisitorBase( const Config& config )
            : m_status( HK_SUCCESS )
            , m_config( config )
        {
            if ( !m_config.m_origin )
            {
                m_config.m_origin = &s_disabledOrigin;
            }
        }

        bool keepGoing() const
        {
            if (m_config.m_earlyOutOnFirstFailure && isFailure(m_status))
            {
                return false;
            }

            return true;
        }

    protected:

        /// Current check status.
        Status m_status;

        /// Checks configuration.
        Config m_config;

        /// Check that a type is reachable. This implies not NULL.
        HK_ALWAYS_INLINE bool isTypeReachable(_In_opt_ const hkReflect::Type* type) const
        {
            HK_ASSERT_NO_MSG(0x442a1238, m_config.m_isTypeReachableCallback != TYPE_REACHABLE_IF_PRESENT_IN_BATCH);
            return (type != nullptr) && ((m_config.m_isTypeReachableCallback == nullptr) || (*m_config.m_isTypeReachableCallback)(type, m_config.m_isTypeReachableUserData));
        }
    };

    /// Visitor for hReflect::Type to check if there is no loop in the linked list of parents, much simpler than the full LoopCheckVisitor.
    /// This test is required to guarantee that all the hkReflect::Type::address functions (used by the other checks) won't end up in an infinite loop.
    class ParentLoopVisitor : public CheckVisitorBase
    {
    public:

        /// Constructor without logging.
        ParentLoopVisitor(const Config& config)
            : CheckVisitorBase(config)
        {
        }

        /// Visitor dispatcher implementation for a generic type.
        hkResult dispatch(_In_ const hkReflect::Type* type)
        {
            // This dispatch() is called once per type. This is the highest level call, so reset the status for this type here
            // and return it at the end of the function. The caller is in charge of accumulating the results for several types
            // if wanted. This visitor accumulates the results only during the checks for this single type, if early out is disabled.
            m_status.reset();

            if (!type)
            {
                // Skip NULL types.
                return HK_SUCCESS;
            }

            const hkReflect::Type* tortoise = type;
            const hkReflect::Type* hare = type;

            for (;;)
            {
                ENSURE_SUCCESS_OR_RETURN(reassignToParent(&hare), "Type parents linked list points to an unreachable type, or reached a loop");

                if (!hare)
                {
                    break;
                }
                if (hare == tortoise)
                {
                    return HK_FAILURE;
                }

                ENSURE_SUCCESS_OR_RETURN(reassignToParent(&hare), "Type parents linked list points to an unreachable type, or reached a loop");

                if (!hare)
                {
                    break;
                }
                if (hare == tortoise)
                {
                    return HK_FAILURE;
                }

                tortoise = tortoise->getParent();
            }

            return m_status;
        }

    protected:

        hkResult reassignToParent(_Outptr_result_maybenull_ const hkReflect::Type** type) const
        {
            if (!isTypeReachable(*type))
            {
                return HK_FAILURE;
            }

            *type = (*type)->getParent();

            return HK_SUCCESS;
        }
    };

    /// Visitor for hReflect::Type inner checks to validate each type's validity.
    /// Inner checks only access the type's own data. No data from other types (e.g. parent type) is accessed.
    class InnerCheckVisitor : public CheckVisitorBase
    {
    public:

        /// Constructor without logging.
        InnerCheckVisitor( const Config& config )
            : CheckVisitorBase( config )
        {
        }

        /// Visitor dispatcher implementation for a generic type.
        hkResult dispatch( _In_ const hkReflect::Type* type )
        {
            // This dispatch() is called once per type. This is the highest level call, so reset the status for this type here
            // and return it at the end of the function. The caller is in charge of accumulating the results for several types
            // if wanted. This visitor accumulates the results only during the checks for this single type, if early out is disabled.
            m_status.reset();

            if ( !type )
            {
                // Skip NULL types.
                return HK_SUCCESS;
            }

            ENSURE_SUCCESS_OR_RETURN( checkFormat( *type ), "Type has invalid format." );

            if ( hkReflect::Detail::isOpaqueWrapper( type ) )
            {
                // Skip opaque wrappers.
                return HK_SUCCESS;
            }

            // Check the generic type.
            ENSURE_SUCCESS_OR_RETURN(checkType(*type), "Type {} failed reflection checks.", type->getName());

            // Check DECLS
            if ( type->isDecorator() == false )
            {
                CHECK_SUCCESS_NO_LOG( checkFields( *type ) );
                CHECK_SUCCESS_NO_LOG( checkDataFields( *type ) );
                //CHECK_SUCCESS_NO_LOG( checkProperties( *type ) );
            }

            return m_status;
        }

    protected:

        //
        // Generic checks - Apply to all types, independently of their kind.
        //

        /// Check the format (and kind) of a type.
        /// This is the very first test, without which nothing else can work.
        hkResult checkFormat( const hkReflect::Type& type )
        {
            // Raw access to m_optional without checks.
            const hkUlong opt = hkReflect::TypeDetail::getOptMask( &type );
            ENSURE_SUCCESS_OR_RETURN( ( opt & INVALID_OPT_BITS ) == 0, "Type has invalid optionals." );
            ENSURE_SUCCESS_OR_RETURN( ( opt & m_config.m_allowedOptMask ) == opt, "Type has invalid optionals {}.", static_cast<hkReflect::Opt::Values>( opt & ( ~m_config.m_allowedOptMask ) ) );

            return HK_SUCCESS;
        }

        /// Check a single type.
        hkResult checkType( const hkReflect::Type& type )
        {
            CHECK_SUCCESS_NO_LOG( checkVersion( type ) );
            CHECK_SUCCESS_NO_LOG( checkSizeAndAlign( type ) );
            CHECK_SUCCESS_NO_LOG( checkFlags( type ) );
            CHECK_SUCCESS_NO_LOG( checkStructure( type ) );

            // Check the template parameters.
            if ( const hkReflect::Template* const params = type.getTemplate() )
            {
                const int numParams = params->getNumParams();
                CHECK_SUCCESS( numParams >= 0, "Template type '{}' has invalid negative number of template parameters.", type.getName() );
                CHECK_SUCCESS( numParams <= 255, "Template type '{}' has invalid number of template parameters greater than 255.", type.getName() ); 
                for ( int i = 0; i < numParams; ++i )
                {
                    auto* const param = params->getParam( i );
                    if ( param->isType() )
                    {
                        CHECK_SUCCESS( param->getAsType() != nullptr, "Template type '{}' has invalid NULL template type parameter #{}.", type.getName(), i );
                    }
                }
            }

            return HK_SUCCESS;
        }

        /// Check that a type has a valid VERSION, if any.
        hkResult checkVersion( const hkReflect::Type& type )
        {
            if ( hkReflect::TypeDetail::localHasOptional( &type, hkReflect::Opt::VERSION ) )
            {
                auto const value = hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::VERSION>( &type );
                CHECK_SUCCESS( value <= hkTrait::NumericLimits<hkUint32>::maxValue(), "Overflow in VERSION of type '{}' (should fit on 32 bits).", type.getName() );
            }

            return HK_SUCCESS;
        }

        /// Check that a type (apart from Void and Incomplete) has non-zero size and alignment.
        hkResult checkSizeAndAlign( const hkReflect::Type& type )
        {
            if ( type.asVoid() || type.asOpaque() )//|| type.isProperty() )
            {
                // Void, incomplete types, and properties, are all unsized
                ENSURE_SUCCESS_OR_RETURN( !type.getSizeOf() && !type.getAlignOf(), "Void/opaque type '{}' has non-zero size and/or alignment.", type.getName() );
            }
            else
            {
                auto const size = type.getSizeOf();
                auto const align = type.getAlignOf();

                // Other types are required to have non-zero size
                ENSURE_SUCCESS_OR_RETURN( size > 0, "Type '{}' has zero size.", type.getName() );

                // Non-abstract types are required to have non-zero align
                
                if ( !type.isAbstract() )
                {
                    ENSURE_SUCCESS_OR_RETURN( align > 0, "Non-abstract type '{}' has zero alignment.", type.getName() );
                    ENSURE_SUCCESS_OR_RETURN( ( align & (align - 1) ) == 0, "Non-abstract type '{}' has non-power-of-two alignment ({} bytes).", type.getName(), align );
                    ENSURE_SUCCESS_OR_RETURN( ( size % align ) == 0, "Non-abstract type '{}' has size ({} bytes) not divisible by its alignment ({} bytes).", type.getName(), type, align );
                }
            }
            return HK_SUCCESS;
        }

        /// Check that a type has the expected flags.
        hkResult checkFlags( const hkReflect::Type& type )
        {
            auto const flags = hkReflect::TypeDetail::getFlags( &type );
            if ( type.getKind() == hkReflect::Kind::KIND_VOID )
            {
                return ( flags == hkReflect::TypeDetail::getFlags( hkReflect::getType<void>() ) ? HK_SUCCESS : HK_FAILURE ); 
            }
            auto validFlags = flags;
            validFlags.andWith( m_config.m_allowedFlags );
            ENSURE_SUCCESS_OR_RETURN_WITH_ID( validFlags == flags, 0xA62E68D8, "Type '{}' has invalid forbidden flag(s) 0x{:x8}.", type.getName(), flags.get() & ~m_config.m_allowedFlags.get() );
            return HK_SUCCESS;
        }

        /// Helper to check the structure of a hkReflect::Type.
        /// Stores whether to check for hkReflect::Opt bits, and what value to expect for them.
        ///
        ///   Table of expected hkReflect::Opt value:
        ///                | m_optValue bit:       |
        ///   m_optEnforce | TRUE  | FALSE         |
        ///   bit:         |-------|---------------|
        ///     TRUE       | TRUE  | FALSE         |
        ///     FALSE      | MAYBE | NOT SPECIFIED |
        ///
        struct TypeStructure
        {
            /// Mask of bits to enforce a check of. If a bit is set, m_optValue contains the expected value of the bit.
            /// Otherwise if the bit of m_optEnforce is not set, the bits of m_optValue simply indicate if the caller expected
            /// the Opt to be maybe present, or didn't specify anything.
            hkUint32 m_optEnforce;

            /// Values to expect for Opt bits if corresponding m_optEnforce bit is set.
            hkUint32 m_optValue;

            /// Constructor.
            TypeStructure()
                : m_optEnforce( 0 )
                , m_optValue( 0 )
            {
            }

            /// Require an Opt to be present (true) or absent (false).
            void mustHave( hkReflect::Opt::Values opt, bool value )
            {
                m_optEnforce |= hkUint32( opt );
                if ( value )
                {
                    m_optValue |= hkUint32( opt );
                }
                else
                {
                    m_optValue &= ~hkUint32( opt );
                }
            }

            /// Require an Opt to be present (true) or absent (false).
            template <hkReflect::Opt::Values OPT>
            void mustHave( bool value )
            {
                mustHave( OPT, value );
            }

            /// Signal an Opt may or may not be present depending on types, so both are valid.
            void mayHave( hkReflect::Opt::Values opt )
            {
                m_optEnforce &= ~hkUint32( opt );
                m_optValue |= opt;
            }

            /// Signal an Opt may or may not be present depending on types, so both are valid.
            template <hkReflect::Opt::Values OPT>
            void mayHave()
            {
                mayHave( OPT );
            }

            TypeStructure operator|( hkReflect::Opt::Values value )
            {
                mustHave( value, true );
                return ( *this );
            }

            TypeStructure operator|( TypeStructure flags )
            {
                HK_ASSERT( 0xC8EC4002, ( ( m_optEnforce | m_optValue ) & ( flags.m_optValue | flags.m_optEnforce ) ) == 0, "Cannot combine incompatible type structures." );
                m_optEnforce |= flags.m_optEnforce;
                m_optValue |= flags.m_optValue;
                return ( *this );
            }

            /// Is an Opt expected to be present (triTrue) or absent (triFalse), or maybe (triMaybe)?
            template <hkReflect::Opt::Values OPT>
            Trilean isExpected() const
            {
                if ( ( m_optEnforce & OPT ) == 0 )
                {
                    return triMaybe;
                }
                return Trilean( static_cast<bool>( ( m_optValue & OPT ) != 0 ) );
            }

            /// Is an actual present/absent value valid compared to what is expected?
            template <hkReflect::Opt::Values OPT>
            hkResult isValid( bool expectedHasOpt ) const
            {
                const bool optNotEnforced = ( ( m_optEnforce & hkUint32( OPT ) ) == 0 );
                const bool optBitMatches = ( ( ( m_optValue & hkUint32( OPT ) ) != 0 ) == expectedHasOpt );
                return ( optNotEnforced || optBitMatches ? HK_SUCCESS : HK_FAILURE );
            }

            template <hkReflect::Opt::Values OPT>
            hkResult checkLocalOpt( const hkReflect::Type& type ) const
            {
                return isValid<OPT>( hkReflect::TypeDetail::localHasOptional( &type, OPT ) );
            }

            template <hkReflect::Opt::Values OPT>
            hkResult checkGlobalOpt( const hkReflect::Type& type ) const
            {
                return isValid<OPT>( hkReflect::TypeDetail::globalHasOptional( &type, OPT ) );
            }
        };

        HK_ALWAYS_INLINE TypeStructure checkHas( hkReflect::Opt::Values value )
        {
            TypeStructure typeStructure;
            typeStructure.mustHave( value, true );
            return typeStructure;
        };

        HK_ALWAYS_INLINE TypeStructure checkMaybe( hkReflect::Opt::Values value )
        {
            TypeStructure typeStructure;
            typeStructure.mayHave( value );
            return typeStructure;
        };

        HK_ALWAYS_INLINE TypeStructure checkHasNot( hkReflect::Opt::Values value )
        {
            TypeStructure typeStructure;
            typeStructure.mustHave( value, false );
            return typeStructure;
        };

        /// Check the type structure, that is the list of Opt against its kind.
        hkResult checkStructure( const hkReflect::Type& type )
        {
            using namespace hkReflect;

            if ( type.isDecorator() )
            {
                CHECK_SUCCESS( TypeDetail::localHasOptional( &type, Opt::FORMAT ) == false, "Decorator type '{}' has local Opt::FORMAT.", type.getName() );
                CHECK_SUCCESS( TypeDetail::localHasOptional( &type, Opt::SUBTYPE ) == false, "Decorator type '{}' has local Opt::SUBTYPE.", type.getName() );
                CHECK_SUCCESS( TypeDetail::localHasOptional( &type, Opt::FUNCTIONS ) == false, "Decorator type '{}' has local Opt::FUNCTIONS.", type.getName() );
                CHECK_SUCCESS( TypeDetail::localHasOptional( &type, Opt::INTERFACES ) == false, "Decorator type '{}' has local Opt::INTERFACES.", type.getName() );
                CHECK_SUCCESS( TypeDetail::localHasOptional( &type, Opt::INHERITANCE ) == false, "Decorator type '{}' has local Opt::INHERITANCE.", type.getName() );
                CHECK_SUCCESS( TypeDetail::localHasOptional( &type, Opt::DEF_CONSTRUCTOR ) == false, "Decorator type '{}' has local Opt::DEF_CONSTRUCTOR.", type.getName() );
                CHECK_SUCCESS( TypeDetail::localHasOptional( &type, Opt::DESTRUCTOR ) == false, "Decorator type '{}' has local Opt::DESTRUCTOR.", type.getName() );
                CHECK_SUCCESS( TypeDetail::localHasOptional( &type, Opt::DECLS ) == false, "Decorator type '{}' has local Opt::DECLS.", type.getName() );
                return HK_SUCCESS;
            }

            TypeStructure typeStructure;
            switch ( type.getKind() )
            {
            case KIND_BOOL:
            case KIND_INT:
            case KIND_FLOAT:
            case KIND_STRING:
                {
                    typeStructure = checkHas( Opt::FORMAT ) | checkHasNot( Opt::SUBTYPE ) | checkHas( Opt::IMPL ) | checkMaybe( Opt::DECLS );
                    break;
                }
            case KIND_VOID:
            case KIND_OPAQUE:
                {
                    // Built-in static 'void' and opaque types have no IMPL, but the ones coming from a Tagfile have one.
                    
                    typeStructure = checkHas( Opt::FORMAT ) | checkHasNot( Opt::SUBTYPE ) | checkMaybe( Opt::IMPL ) | checkMaybe( Opt::DECLS );
                    break;
                }
            case KIND_POINTER:
                {
                    typeStructure = checkHas( Opt::FORMAT ) | checkHas( Opt::SUBTYPE ) | checkHas( Opt::IMPL ) | checkMaybe( Opt::DECLS );
                    break;
                }
            case KIND_ARRAY:
                {
                    typeStructure = checkHas( Opt::FORMAT ) | checkHas( Opt::SUBTYPE ) | checkHas( Opt::IMPL ) | checkMaybe( Opt::DECLS );
                    break;
                }
            case KIND_RECORD:
                {
                    typeStructure = checkHas( Opt::FORMAT ) | checkHasNot( Opt::SUBTYPE ) | checkHas( Opt::IMPL ) | checkHas( Opt::DECLS );
                    break;
                }
            default:
                setFail( m_status );
                return HK_FAILURE;
            }

            CHECK_SUCCESS( typeStructure.isValid<Opt::IMPL>( type.getImpl() != nullptr ), "Invalid Impl for type '{}'.", type.getName() );

            // Opt::TEMPLATE implies Opt::NAME
            CHECK_SUCCESS( ( TypeDetail::localHasOptional( &type, Opt::TEMPLATE ) == false ) || TypeDetail::localHasOptional( &type, Opt::NAME ), "Type '{}' has local Opt::TEMPLATE but no local Opt::NAME.", type.getName() );

            CHECK_SUCCESS( typeStructure.checkLocalOpt<Opt::FORMAT>( type ), "Type '{}' has unexpected Opt::FORMAT ({}).", type.getName(), getExpectedMessage( typeStructure.isExpected<Opt::FORMAT>() ) );
            CHECK_SUCCESS( typeStructure.checkLocalOpt<Opt::SUBTYPE>( type ), "Type '{}' has unexpected Opt::SUBTYPE ({}).", type.getName(), getExpectedMessage( typeStructure.isExpected<Opt::SUBTYPE>() ) );
            CHECK_SUCCESS( typeStructure.checkLocalOpt<Opt::DECLS>( type ), "Type '{}' has unexpected Opt::DECLS ({}).", type.getName(), getExpectedMessage( typeStructure.isExpected<Opt::DECLS>() ) );

            return HK_SUCCESS;
        }

        /// Check fields.
        hkResult checkFields( const hkReflect::Type& type )
        {
            for ( auto field : type.getFields() )
            {
                CHECK_SUCCESS(field.getType()->getParent(), "Record type '{}' has a field with no parent.", type.getName());

                // All fields must have a name
                const char* const fieldName = field.getName();
                CHECK_SUCCESS( fieldName != nullptr, "Record type '{}' has a field with no name.", type.getName() );

                // Fields should have a DECL context pointing back to the type
                auto* const declContext = field.getDeclContext();
                CHECK_SUCCESS( declContext == &type, "Record type '{}' has field '{}' with DECL context missing or pointing to another type.", type.getName(), fieldName );
            }

            return HK_SUCCESS;
        }

        /// Check data fields.
        hkResult checkDataFields( const hkReflect::Type& type )
        {
            
            if ( type.asRecord() == nullptr )
            {
                return HK_SUCCESS;
            }

            auto const typeSize = type.getSizeOf();
            int lastOffset = -1;
            int lastSizeOf = 0;

            for ( auto dataField : type.getDataFields() )
            {
                // All data fields must have an offset, 0 <= offset < size
                auto const offset = dataField.getOffset();
                CHECK_SUCCESS( offset >= 0, "Record type '{}' has field '{}' with negative offset.", type.getName(), dataField.getName() );
                CHECK_SUCCESS( offset + dataField.getType()->getSizeOf() <= typeSize, "Record type '{}' has field '{}' with offset + size {} greater than total type size {}.", type.getName(), dataField.getName(), offset, typeSize );

                // Offsets must be sorted
                CHECK_SUCCESS( offset >= lastOffset + lastSizeOf, "Record type '{}' has field '{}' with offset {} less than the previous one + size {}. This means that fields either overlap or are not sorted.", type.getName(), dataField.getName(), offset, lastOffset + lastSizeOf );
                lastOffset = offset;
                lastSizeOf = dataField.getType()->getSizeOf();
            }

            return HK_SUCCESS;
        }

        ///// Check properties.
        //hkResult checkProperties( const hkReflect::Type& type )
        //{
        //    auto const& fields = type.getFields();
        //    for ( auto i = type.getNumDataFields(); i < type.getNumFields(); ++i )
        //    {
        //        auto const& propertyField = static_cast<hkReflect::PropertyFieldDecl const&>( fields[i] );

        //        // All properties must have zero offset
        //        
        //        CHECK_SUCCESS( propertyField.getOffset() == 0, "Record type '{}' has property '{}' with non-zero offset ({} bytes).", type.getName(), propertyField.getName(), propertyField.getOffset() );
        //    }

        //    return HK_SUCCESS;
        //}
    };

    /// Visitor for hReflect::Type outer checks to validate inter-type consistency.
    /// Outer checks access the type's own data as well as data from other types (e.g. parent type).
    /// Those checks assume that inner checks all passed on all references types, so e.g. this type as well as
    /// its parent type both passed their inner checks.
    class OuterCheckVisitor : public CheckVisitorBase
    {
    public:

        /// Constructor without logging.
        OuterCheckVisitor( const Config& config )
            : CheckVisitorBase( config )
        {
        }

        /// Visitor dispatcher implementation for a generic type.
        hkResult dispatch( _In_ const hkReflect::Type* type )
        {
            // This dispatch() is called once per type. This is the highest level call, so reset the status for this type here
            // and return it at the end of the function. The caller is in charge of accumulating the results for several types
            // if wanted. This visitor accumulates the results only during the checks for this single type, if early out is disabled.
            m_status.reset();

            if ( !type )
            {
                // Skip NULL types.
                return HK_SUCCESS;
            }

            if ( hkReflect::Detail::isOpaqueWrapper( type ) )
            {
                // Skip opaque wrappers.
                return HK_SUCCESS;
            }

            // Check the parent.
            if ( const hkReflect::Type* parentType = type->getParent() )
            {
                CHECK_SUCCESS( isTypeReachable( parentType ), "Type '{}' has unreachable parent type (not in scope).", type->getName() );
            }

            // Check the interfaces.
            if ( const hkReflect::Detail::InterfaceArray* ifacesPtr = hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::INTERFACES>( type ) )
            {
                hkArrayView<const hkReflect::Detail::Interface> ifaces = ifacesPtr->items();
                for ( int i = 0; i < ifaces.getSize(); ++i )
                {
                    // Check the interface is reachable
                    const hkReflect::Type* const interfaceType = ifaces[i].m_interfaceType;
                    CHECK_SUCCESS_OR_CONTINUE( isTypeReachable( interfaceType ), "Type '{}' has unreachable interface type (not in scope).", type->getFullName() );

                    // Check the interface is a record type
                    CHECK_SUCCESS_OR_CONTINUE( interfaceType->asRecord(), "Type '{}' has interface '{}' which is not a record type.", type->getFullName(), interfaceType->getFullName() );

                    // Check the interface has no decl (so no field)
                    // NB: The interface type does not need to be an abstract type (this is not a "pure" interface like e.g. C#).
                    //     We simply want to avoid multiple inheritance from the point of view of the fields.
                    hkReflect::DeclIter<hkReflect::DataFieldDecl> it( ifaces[i].m_interfaceType );
                    CHECK_SUCCESS_OR_CONTINUE( false == it.advance(), "Type '{}' has interface '{}' with non-empty decl array.", type->getFullName(), interfaceType->getFullName() );
                }
            }

            // Check the fields
            if ( type->isDecorator() == false )
            {
                // Gather field names from ancestors
                hkArray<hkStringPtr> memberNames;   
                auto* parent = type->getParent();
                while ( parent )
                {
                    for ( auto field : parent->getFields() )
                    {
                        memberNames.pushBack( field.getName() );
                    }
                    parent = parent->getParent();
                }

                // Check fields of current type, excluding ancestors
                for ( auto const& field : type->getFields() )
                {
                    // Field names must be unique
                    const char* const fieldName = field.getName();
                    const bool newlyInserted = ( memberNames.indexOf( fieldName ) == -1 );
                    memberNames.pushBack( fieldName );
                    CHECK_SUCCESS( newlyInserted || (m_config.m_allowedPlaceholderDecls && fieldName[0] == '$'),
                        "Type '{}' has duplicate field with name '{}' (possibly from ancestor).", type->getName(), fieldName );

                    // Check field type is reachable, so can be subsequently accessed safely for other checks.
                    const hkReflect::Type* const fieldType = field.getType();
                    CHECK_SUCCESS_OR_CONTINUE( fieldType != nullptr, "Field '{}' of type '{}' has invalid NULL type.", fieldName, type->getFullName() );
                    // NB: Field types are non-serialized decorator types created by the deserialization code, so are considered always valid (as long as the data they're created from is).
                    //     So we do *not* check for reachability on those types, as they are not serialized.

                    // No field can have VOID format
                    CHECK_SUCCESS( fieldType->asVoid() == nullptr, "Type '{}' has field '{}' with VOID type.", type->getName(), fieldName );

                    // Field decorators are unique to a record, never shared, so inner-check them directly here.
                    CHECK_SUCCESS_OR_CONTINUE( checkType( *fieldType ), "Field decorator type '{}' of decl '{}' of type '{}' failed inner checks.", fieldType->getFullName(), fieldName, type->getFullName() );
                    CHECK_SUCCESS_OR_CONTINUE( fieldType->isDecorator(), "Field '{}' of type '{}' has non-decorator type.", fieldName, type->getFullName() ); // after checkType() as it relies on OpT::FORMAT begin already validated

                    // Check the actual type of the field is reachable.
                    CHECK_SUCCESS_OR_CONTINUE( isTypeReachable( fieldType->getParent() ), "Type of field '{}' is not reachable.", field );
                }
            }

            // Check the template parameters.
            if ( const hkReflect::Template* params = type->getTemplate() )
            {
                for ( int i = 0; i < params->getNumParams(); ++i )
                {
                    if ( params->getParam( i )->isType() )
                    {
                        const hkReflect::Type* const paramType = params->getParam( i )->getAsType();
                        CHECK_SUCCESS_OR_CONTINUE( isTypeReachable( paramType ), "Template parameter '{}' of type '{}' has unreachable type.", params->getParam( i )->getName(), type->getFullName() );
                        CHECK_SUCCESS_NO_LOG( checkInheritance( *paramType ) );
                    }
                }
            }

            // Specialized checks for pointers.
            if ( auto* const ptr = type->asPointer() )
            {
                CHECK_SUCCESS_NO_LOG( checkPointerType( *ptr ) );
            }

            // Specialized checks for arrays.
            if ( auto* const arr = type->asArray() )
            {
                CHECK_SUCCESS_NO_LOG( checkArrayType( *arr ) );
            }

            return HK_SUCCESS;
        }

        /// Specialized checks for a pointer type.
        hkResult checkPointerType( const hkReflect::PointerType& ptr )
        {
            if ( const hkReflect::Type* const pointeeType = ptr.getSubType() )
            {
                CHECK_SUCCESS( isTypeReachable( pointeeType ), "Pointer type '{}' has unreachable pointee type.", ptr.getName() );
            }
            return HK_SUCCESS;
        }

        /// Specialized checks for an array type.
        hkResult checkArrayType( const hkReflect::ArrayType& arr )
        {
            if ( const hkReflect::Type* const elemType = arr.getSubType() )
            {
                // Arrays with opaque element type are valid; they are in particular produced if all instances have zero size
                // so the element type is not actually used in the serialized data. In that case FlatTypeRelocs::applyRelocation() will leave
                // the type pointing to the type identity pending possible future relocation if the actual element type is later loaded.
                if ( !elemType->asOpaque() )
                {
                    CHECK_SUCCESS( isTypeReachable( elemType ), "Array type '{}' has unreachable element type.", arr.getName() );

                    if (arr.getFixedCount() > 0)
                    {
                        CHECK_SUCCESS(arr.getSizeOf() == elemType->getSizeOf() * arr.getFixedCount(),
                            "Fixed size array type '{}' ({} bytes) contains {} '{}' ({} bytes each), sizes mismatch.",
                            arr.getName(), arr.getSizeOf(), arr.getFixedCount(), elemType->getName(), elemType->getSizeOf());
                    }
                }
            }
            else
            {
                CHECK_SUCCESS(arr.getFixedCount()==0, "Fixed array type {} must have a subtype", arr.getFullName())
            }

            return HK_SUCCESS;
        }

    protected:

        //
        // Generic checks - Apply to all types, independently of their kind.
        //

        /// Check Opt::INHERITANCE for a single type.
        /// [pre ] The type is reachable and valid.
        /// [post] The type has reachable and valid canonical resolved type, and valid Opt::INHERITANCE.
        hkResult checkInheritance( const hkReflect::Type& type )
        {
            // Resolve typedefs and get the resolved "canonical" type.
            // This e.g. resolve hkArray<hkInt32> to hkArray<int>, so not necessarily a typedef of the type itself, but also on any
            // of a templated type's template arguments. It also resolves any typedef of the type itself by skipping decorators.
            ENSURE_SUCCESS_OR_RETURN( hkReflect::TypeDetail::skipDecorators( &type ), "Type '{} is decorator without concrete parent.", type.getFullName() );
            const hkReflect::Type* const canonicalType = hkReflect::TypeDetail::getUndecorated( &type );
            ENSURE_SUCCESS_OR_RETURN( isTypeReachable( canonicalType ), "Type '{}' resolved to an unreachable type after resolving typedefs.", type.getFullName() );
            CHECK_SUCCESS( false == canonicalType->isDecorator(), "Type '{}' resolved to type '{}' which is a decorator, so not canonical.", type.getFullName(), canonicalType->getFullName() );

            // Check Opt::INHERITANCE
            if ( const hkReflect::Detail::InheritanceInfo* inheritance = hkReflect::TypeDetail::localAddressOptional<hkReflect::Opt::INHERITANCE>(&type) )
            {
                CHECK_SUCCESS( inheritance->asUlong() != 0, "Type '{}' has empty Opt::INHERITANCE.", type.getFullName() );

                const hkReflect::Detail::InheritanceInfo* canonInheritance = hkReflect::TypeDetail::getInheritance( canonicalType );
                CHECK_SUCCESS( inheritance->equals(*canonInheritance), "Type '{}' has inheritance {} but the corresponding resolved type '{}' has a different inheritance {}.",
                    type.getFullName(), inheritance->asUlong(), canonicalType->getFullName(), canonInheritance->asUlong());
            }

            return HK_SUCCESS;
        }

        /// Check a single type.
        hkResult checkType( const hkReflect::Type& type )
        {
            CHECK_SUCCESS_NO_LOG( checkRecordParentIsRecord( type ) );
            CHECK_SUCCESS_NO_LOG( checkFlaggedIfHasRelArrayField( type ) );
            return HK_SUCCESS;
        }

        /// Check that if a record type has a parent, then it is of record kind, which is the only possibly kind for parents.
        hkResult checkRecordParentIsRecord( const hkReflect::Type& type )
        {
            if ( const hkReflect::RecordType* recordType = type.asRecord() )
            {
                const hkReflect::Type* parentType = recordType->getParent();
                ENSURE_SUCCESS_OR_RETURN( isTypeReachable( parentType ), "Type '{}' is a record type, but has unreachable parent.", type.getFullName() );

                const hkReflect::RecordType* parentRecordType = parentType->asRecord();
                CHECK_SUCCESS( parentRecordType != nullptr, "Registered record type '{}' has as parent type '{}', which is a non-record type.", type.getName(), parentType->getName() );
            }
            return HK_SUCCESS;
        }

        /// Check that a record type containing rel-array decls has a flag set in it so that we can allocate it specially in allocateForClone().
        hkResult checkFlaggedIfHasRelArrayField( const hkReflect::Type& type )
        {
            if ( const hkReflect::RecordType* recordType = type.asRecord() )
            {
                const char* firstRelArray = nullptr;
                hkArrayView<const hkReflect::FieldDecl> fields = recordType->getFields();
                for ( int i = 0; i < fields.getSize(); ++i )
                {
                    const hkReflect::FieldDecl f = fields[i];

                    const hkReflect::Type* const ftype = f.getType();
                    CHECK_SUCCESS_OR_CONTINUE( ftype != nullptr, "Field '{}' of record type '{}' has invalid NULL type.", f.getName(), type.getName() );

                    if ( const hkReflect::ArrayType* atype = ftype->asArray() )
                    {
                        // quick rejection
                        typedef hkRelArray<char> RelArray;
                        if ( atype->getSizeOf() == sizeof( RelArray ) )
                        {
                            if ( const char* name = atype->getName() )
                            {
                                if ( hkString::strCmp( name, hkReflect::getType<RelArray>()->getName() ) == 0 )
                                {
                                    firstRelArray = f.getName();
                                    break;
                                }
                            }
                        }
                    }
                }

                if ( firstRelArray )
                {
                    const hk::ContainsRelArrays* cra = recordType->findAttribute<hk::ContainsRelArrays>();
                    CHECK_SUCCESS( ( cra != nullptr ) && ( cra->m_value == true ), "Record type '{}' contains a relarray '{}' but is not marked with a hk::ContainsRelArrays attribute.", recordType->getName(), firstRelArray );
                }
            }
            return HK_SUCCESS;
        }
    };

    /// Visitor for hReflect::Type checks to validate the absence of loops in:
    /// - parent / child linear hierarchy;
    /// - interface tree hierarchy;
    /// - fields / decls context.
    class LoopCheckVisitor : public CheckVisitorBase
    {
    public:

        /// Constructor without logging.
        LoopCheckVisitor( const Config& config )
            : CheckVisitorBase( config )
        {
        }

        /// Dispatch to the given test batch.
        _Must_inspect_result_
        hkResult dispatch( _In_ hkArrayView<const hkReflect::Type* const> testBatch, _Inout_ hkHashSet<const hkReflect::Type*>& validTypes )
        {
            m_status.reset();

            validTypes.reserve( validTypes.getSize() + testBatch.getSize() );

            // Types are generally already pseudo-sorted, in that parents appear before children.
            // Try to preserve this order as much as possible, as this prevents restarts of the algorithm.
            // Because we use a stack and pop elements from back, enqueue incoming types from the test batch
            // in reverse order so first in batch is last in stack vector, so on top of stack.
            for ( auto it = testBatch.rbegin(); it != testBatch.rend(); ++it )
            {
                hkArray<const hkReflect::Type*> stack;
                HK_ASSERT_NO_MSG(0x594f6bf4, *it );

                stack.pushBack( *it );
                if ( checkSingleType( stack, validTypes ).isFailure() )
                {
                    return HK_FAILURE;
                }

                if (auto* const params = (*it)->getTemplate())
                {
                    hkInplaceArray<const hkReflect::Type*, 8> templateStack;
                    templateStack.pushBack((*it));
                    if (checkSingleParams(templateStack).isFailure())
                    {
                        return HK_FAILURE;
                    }
                }
            }

            return m_status;
        }

    protected:

        template<typename T>
        struct StackPopRaii
        {
            StackPopRaii(hkArray<T>& stack) : m_stack(stack) {}
            hkArray<T>& m_stack;
            ~StackPopRaii() { m_stack.popBack(); }
        };

        /// Check a single type for loops.
        hkResult checkSingleType( _Inout_ hkArray<const hkReflect::Type*>& stack, _Inout_ hkHashSet<const hkReflect::Type*>& validTypes )
        {
            StackPopRaii<const hkReflect::Type*> stackPopRaii(stack);

            {
                auto* const type = stack.back();

                if ( validTypes.contains( type ) )
                {
                    return HK_SUCCESS;
                }

                // Recursively check fixed arrays subtypes for loops.
                {
                    const hkReflect::Type* fixedArraySubType = (type->asArray() && (type->asArray()->getFixedCount() > 0)) ? type->asArray()->getSubType() : nullptr;

                    if (fixedArraySubType && !validTypes.contains(fixedArraySubType))
                    {
                        ENSURE_SUCCESS_OR_RETURN(stack.indexOf(fixedArraySubType) < 0, "Found loop in subtype '{}' of array type '{}'", fixedArraySubType->getName(), type->getName());
                        stack.pushBack(fixedArraySubType);

                        if (checkSingleType(stack, validTypes).isFailure())
                        {
                            return HK_FAILURE;
                        }
                    }
                }

                // Recurse into the hierarchy of decl types, starting with the current type as root.
                // Make sure that all fields have validated loop-free types before moving on to check the current type itself and its hierarchy.
                {
                    auto const fields = type->getFields();
                    for ( auto const field : fields )
                    {
                        // fieldType is the undecorated type
                        const hkReflect::Type* const fieldType = field.getType()->getParent();

                        if (!validTypes.contains(fieldType))
                        {
                            ENSURE_SUCCESS_OR_RETURN(stack.indexOf(fieldType) < 0, "Found loop in type '{}' with field '{}' of type '{}'.", type->getName(), field.getName(), fieldType->getName());
                            stack.pushBack(fieldType);

                            if (checkSingleType(stack, validTypes).isFailure())
                            {
                                return HK_FAILURE;
                            }
                        }
                    }
                }

                // Check interfaces (multiple inheritance)
                if ( const hkReflect::Detail::InterfaceArray* ifacesPtr = hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::INTERFACES>( type ) )
                {
                    hkArrayView<const hkReflect::Detail::Interface> ifaces = ifacesPtr->items();
                    for ( auto const& iface : ifaces )
                    {
                        auto* const ifaceType = iface.m_interfaceType;
                        if ( !validTypes.contains( ifaceType ) )
                        {
                            // Interface was not checked yet; push it to the stack for check now.
                            ENSURE_SUCCESS_OR_RETURN( stack.indexOf( ifaceType ) < 0, "Found loop in type '{}' with interface type '{}'.", type->getName(), ifaceType->getName() );
                            HK_ASSERT_NO_MSG(0x3f219e66, ifaceType );
                            stack.pushBack( ifaceType );
                            if ( checkSingleType( stack, validTypes ).isFailure() )
                            {
                                return HK_FAILURE;
                            }
                        }
                    }
                }

                // Check ancestors (single inheritance).
                if ( auto* const parentType = type->getParent() )
                {
                    if ( !validTypes.contains( parentType ) )
                    {
                        // Parent was not checked yet; push it to the stack for check now.
                        ENSURE_SUCCESS_OR_RETURN( stack.indexOf( parentType ) < 0, "Found loop in type '{}' with parent type '{}'.", type->getName(), parentType->getName() );
                        HK_ASSERT_NO_MSG(0x30d142e5, parentType );
                        stack.pushBack( parentType );
                        if ( checkSingleType( stack, validTypes ).isFailure() )
                        {
                            return HK_FAILURE;
                        }
                    }
                }

                validTypes.insert( type );
            }
            return HK_SUCCESS;
        }

        hkResult checkSingleParams(_Inout_ hkArray<const hkReflect::Type*>& templateStack)
        {
            StackPopRaii<const hkReflect::Type*> stackPopRaii(templateStack);

            auto* const params = templateStack.back()->getTemplate();
            auto* const paramsEnd = params->getParams() + params->getNumParams();

            for (auto* param = params->getParams(); param != paramsEnd; ++param)
            {
                if (!param->isType()) continue;

                auto* const subType = param->getAsType();

                if(subType->getTemplate())
                {
                    ENSURE_SUCCESS_OR_RETURN(templateStack.indexOf(subType) < 0, "Found loop in type '{}' with type template parameter '{}'.", templateStack.back()->getName(), subType->getName());

                    templateStack.pushBack(subType);
                    if (checkSingleParams(templateStack).isFailure())
                    {
                        return HK_FAILURE;
                    }
                }
            }

            return HK_SUCCESS;
        }
    };

#undef ENSURE_SUCCESS_OR_RETURN
#undef CHECK_SUCCESS
#undef CHECK_SUCCESS_OR_CONTINUE
#undef CHECK_SUCCESS_NO_LOG
}

namespace hkReflect {
namespace Detail {
namespace CheckUtil {

IncrementalChecker::IncrementalChecker( const Config& config )
    : m_status(HK_SUCCESS)
    , m_reachabilityHelper(m_validTypes)
{
    setConfig(config);
}

void IncrementalChecker::setConfig( const Config& config )
{
    HK_ASSERT( 0xC8EC4000, ( m_validTypes.getSize() == 0 ) && m_status.isSuccess(), "Cannot change CheckUtil configuration after some types were added. Call reset() first." );
    m_config = config;
    if ( config.m_isTypeReachableCallback == TYPE_REACHABLE_IF_PRESENT_IN_BATCH )
    {
        m_config.m_isTypeReachableCallback = &IncrementalChecker::ReachabilityHelper::s_isTypeReachable;
        m_config.m_isTypeReachableUserData = &m_reachabilityHelper;
    }
    m_validTypes.clear();
    m_status = HK_SUCCESS;
}

void IncrementalChecker::reset()
{
    m_validTypes.clear();
    m_status = HK_SUCCESS;
}

bool IncrementalChecker::areOptsValid( hkUint32 opts ) const
{
    const bool allOptsAreValid = ( ( opts & INVALID_OPT_BITS ) == 0 );
    const bool allOptsAreAllowed = ( ( opts & m_config.m_allowedOptMask ) == opts );
    return ( allOptsAreValid && allOptsAreAllowed );
}

_Must_inspect_result_
hkResult IncrementalChecker::addAndCheckTypes(hkArrayView<const hkReflect::Type* const> types)
{
    hkResult ret = HK_SUCCESS;

    // Parent loop check on types
    {
        ParentLoopVisitor visitor(m_config);
        const int numTypes = types.getSize();
        for (int i = 0; i < numTypes; ++i)
        {
            if (types[i] && visitor.dispatch(types[i]).isFailure())
            {
                if (!visitor.keepGoing())
                {
                    return HK_FAILURE;
                }
                ret = HK_FAILURE;
            }
        }
    }

    if (ret.isFailure())
    {
        return HK_FAILURE;
    }

    // Inner checks on new types
    {
        InnerCheckVisitor visitor( m_config );
        const int numTypes = types.getSize();
        for ( int i = 0; i < numTypes; ++i )
        {
            if (types[i] && visitor.dispatch(types[i]).isFailure())
            {
                if (!visitor.keepGoing())
                {
                    return HK_FAILURE;
                }
                ret = HK_FAILURE;
            }
        }
    }

    if (ret.isFailure())
    {
        return HK_FAILURE;
    }

    // Save new types before outer checks, so reachability callback takes them into account
    hkArray<const hkReflect::Type*>::Temp newTypes;
    newTypes.reserve( types.getSize() );
    for ( auto* const type : types )
    {
        if ( type )
        {
            newTypes.pushBackUnchecked( type );
        }
    }

    if ( doOuterAndLoopChecks( newTypes ).isFailure() )
    {
        return HK_FAILURE;
    }
    return ret;
}

void IncrementalChecker::invalidateType( const hkReflect::Type* type )
{
    m_validTypes.remove( type );
}

_Must_inspect_result_
hkResult IncrementalChecker::addAndCheckTypes( _In_ const TypeRegNode* head )
{
    hkResult ret = HK_SUCCESS;

    // Parent loop check on types
    {
        ParentLoopVisitor visitor(m_config);
        for (const TypeRegNode* node = head; node; node = node->getNext())
        {
            auto* const type = node->getType();
            if (type && visitor.dispatch(type).isFailure())
            {
                if (!visitor.keepGoing())
                {
                    return HK_FAILURE;
                }
                ret = HK_FAILURE;
            }
        }
    }

    if (ret.isFailure())
    {
        return HK_FAILURE;
    }

    // Inner checks on new types
    int numNewTypes = 0;
    {
        InnerCheckVisitor visitor( m_config );
        for ( const TypeRegNode* node = head; node; node = node->getNext() )
        {
            auto* const type = node->getType();
            ++numNewTypes;
            if ( type && visitor.dispatch( type ).isFailure() )
            {
                if (!visitor.keepGoing())
                {
                    return HK_FAILURE;
                }
                ret = HK_FAILURE;
            }
        }
    }

    if (ret.isFailure())
    {
        return HK_FAILURE;
    }

    // Save new types before outer checks, so reachability callback takes them into account
    hkArray<const hkReflect::Type*>::Temp newTypes;
    newTypes.reserve( numNewTypes );
    for ( const TypeRegNode* node = head; node; node = node->getNext() )
    {
        if ( node->getType() )
        {
            newTypes.pushBackUnchecked( node->getType() );
        }
    }

    if ( doOuterAndLoopChecks( newTypes ).isFailure() )
    {
        return HK_FAILURE;
    }
    return ret;
}

_Must_inspect_result_
hkResult IncrementalChecker::doOuterAndLoopChecks( hkArrayView<const hkReflect::Type* const> types )
{
    if ( types.isEmpty() )
    {
        return HK_SUCCESS;
    }

    hkResult ret = HK_SUCCESS;

    m_reachabilityHelper.m_newTypes = types;

    // Outer checks on new types using all types for reachability
    {
        OuterCheckVisitor visitor( m_config );
        for ( auto* const type : types )
        {
            HK_ASSERT_NO_MSG(0x371d5024, type);
            if (type && visitor.dispatch(type).isFailure())
            {
                if (!visitor.keepGoing())
                {
                    return HK_FAILURE;
                }
                ret = HK_FAILURE;
            }
        }
    }

    // Loop checks on new types, and add them to valid types as processed
    m_validTypes.reserve( m_validTypes.getSize() + types.getSize() );
    {
        LoopCheckVisitor visitor( m_config );
        if ( visitor.dispatch( types, m_validTypes ).isFailure() )
        {
            if (!visitor.keepGoing())
            {
                return HK_FAILURE;
            }
            ret = HK_FAILURE;
        }
    }

    // Return the result for this batch of new types only.
    return ret;
}

void IncrementalChecker::removeTypes( hkArrayView<const hkReflect::Type*> types )
{
    for ( const hkReflect::Type** it = types.begin(); it != types.end(); ++it )
    {
        m_validTypes.remove( *it );
    }
}

_Check_return_
bool IncrementalChecker::ReachabilityHelper::s_isTypeReachable( _In_ const hkReflect::Type* type, _In_ void* helper )
{
    auto* const self = static_cast<IncrementalChecker::ReachabilityHelper*>( helper );
    if ( self->m_validTypes.contains( type ) )
    {
        return true;
    }
    return ( self->m_newTypes.indexOf( type ) >= 0 );
}

}   }   }

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
