// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Util/hkReflectCrc.h>
#include <Common/Base/System/Io/Writer/Crc/hkCrcStreamWriter.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>

namespace
{       // -------------------------------------------------------------------------------------------------------------
    // Visitor to compute the CRC used for correct access checking.
    // -------------------------------------------------------------------------------------------------------------

    template <typename CRCStreamWriter>
    struct ComputeCrcVisitor : public hkReflect::VarVisitor< ComputeCrcVisitor<CRCStreamWriter> >
    {
        HK_DECLARE_CLASS(ComputeCrcVisitor, New);

        CRCStreamWriter& m_crcStreamWriter;

        ComputeCrcVisitor(CRCStreamWriter& crcStreamWriter, hkReflect::FilteredDeclIter<hkReflect::DataFieldDecl>::FilterFunc fn = HK_NULL)
            : m_crcStreamWriter(crcStreamWriter), m_fn(fn)
        {}

        void visit(const hkReflect::BoolVar& bt)
        {
            m_crcStreamWriter.write(bt.getAddress(), bt.getType()->getSizeOf());
        }

        void visit(const hkReflect::IntVar& it)
        {
            m_crcStreamWriter.write(it.getAddress(), it.getType()->getSizeOf());
        }

        void visit(const hkReflect::FloatVar& fpt)
        {
            m_crcStreamWriter.write(fpt.getAddress(), fpt.getType()->getSizeOf());
        }

        void visit(const hkReflect::StringVar& st)
        {
            if(const char* string = st.getValue())
            {
                m_crcStreamWriter.write(string, hkString::strLen(string));
            }
        }

        void visit(const hkReflect::VoidVar&)
        {
            HK_ASSERT_NO_MSG(0x31024bbc, false); // Should never be encountered.
        }

        void visit(const hkReflect::OpaqueVar&)
        {
            HK_ASSERT_NO_MSG(0x7930ee03, false); // Should never be encountered.
        }

        void visit(const hkReflect::ArrayVar& at)
        {
            hkReflect::ArrayValue val = at.getValue();
            const hkReflect::QualType elemType = val.getSubType();

            // the sub-type of an array could change, so here we accommodate that by adding that sub-type to the CRC
            m_crcStreamWriter.write(&elemType, sizeof(hkReflect::QualType));

            if (elemType.get())
            {
                if (elemType->asInteger() || elemType->asFloat() || elemType->asBool())
                {
                    m_crcStreamWriter.write(val.getAddress(), val.getCount() * val.getStride());
                }
                else
                {
                    for (int index = 0; index < val.getCount(); ++index)
                    {
                        this->dispatch(val[index]);
                    }
                }
            }
        }

        void visit(const hkReflect::RecordVar& rv)
        {
            HK_ASSERT(0x4193e62f, rv.getType()->isSerializable(), "Non serializable types are not allowed here.");
            if(m_fn == HK_NULL)
            {
                for(hkReflect::DeclIter<hkReflect::DataFieldDecl> fieldIter(rv.getType()); fieldIter.advance(); )
                {
                    const hkReflect::DataFieldDecl& f = fieldIter.current();
                    if(f.isSerializable())
                    {
                        this->dispatch(rv[f]);
                    }
                }
            }
            else
            {
                for(hkReflect::FilteredDeclIter<hkReflect::DataFieldDecl> fieldIter(rv.getType(), m_fn); fieldIter.advance(); )
                {
                    const hkReflect::DataFieldDecl& f = fieldIter.current();
                    if(f.isSerializable())
                    {
                        this->dispatch(rv[f]);
                    }
                }
            }
        }

        void visit(const hkReflect::PointerVar& pt)
        {
            m_crcStreamWriter.write(pt.getAddress(), pt.getType()->getSizeOf());
        }


        hkReflect::FilteredDeclIter<hkReflect::DataFieldDecl>::FilterFunc m_fn;
    };
}

hkUint32 hkReflectUtil::computeCrc32(hkReflect::Var obj)
{
    hkCrc32StreamWriter crcStreamWriter;
    ComputeCrcVisitor<hkCrc32StreamWriter> crcVisitor(crcStreamWriter);
    crcVisitor.dispatch(obj);
    return crcStreamWriter.getCrc();
}

hkUint64 hkReflectUtil::computeCrc64(hkReflect::Var obj)
{
    hkCrc64StreamWriter crcStreamWriter;
    ComputeCrcVisitor<hkCrc64StreamWriter> crcVisitor(crcStreamWriter);
    crcVisitor.dispatch(obj);
    return crcStreamWriter.getCrc();
}

hkUint64 hkReflectUtil::computeCrc64(hkReflect::Var obj, hkReflect::FilteredDeclIter<hkReflect::DataFieldDecl>::FilterFunc fn)
{
    hkCrc64StreamWriter crcStreamWriter;
    ComputeCrcVisitor<hkCrc64StreamWriter> crcVisitor(crcStreamWriter, fn);
    crcVisitor.dispatch(obj);
    return crcStreamWriter.getCrc();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
