// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

namespace hkSerialize
{
    class VarN;
}

    /// Clone objects using hkType information.
namespace hkReflect
{
        /// Utility class for cloning reflected data.
        /// This class is commonly used in two ways. It can be used as an unnamed object similar to a function call.
        /// e.g. Obj* clone = hkReflect::Cloner().cloneRecursive(&obj);
        /// Alternatively a cloner object can be reused several times. e.g.
        /// hkReflect::Cloner cloner; cloner.cloneRecursive(&obj1); cloner.cloneRecursive(&obj2);
        /// The cloner maintains a cache of previously copied objects so repeated calls my reuse objects
        /// from a previous clone. It is the callers responsibility to ensure that previously copied objects
        /// have not been destroyed. This cache can be cleared with clearObjectCache().
        /// The cloner also maintains a cache of rules for copying each type. Thus types also need to be persistent
        /// for the lifetime of the cloner.
    struct HK_EXPORT_COMMON Cloner
    {
        HK_DECLARE_CLASS(Cloner, New, NonCopyable);

        struct CloneImpl;

        struct PointerAction
        {
            enum Enum
            {
                SKIP,
                CLONE,
            };
        };

            /// Callbacks which can customize the clone behavior.
        struct HK_EXPORT_COMMON Callback
        {
                /// Called when a new top-level var is entered.
                /// If dst is null, the callback should initialize it.
                /// Also src may be mutated to selectively replace objects.
                /// If HK_FAILURE is returned, the clone fails.
            virtual hkResult beginVar(Var& dst, Var& src) = 0;
                /// Called before cloning the contents of an array. Must initialize dst for clone by calling
                /// ArrayImpl::allocateElements on it. Also src may be mutated to selectively replace objects.
                /// If HK_FAILURE is returned, the clone fails.
            virtual hkResult beginArrayElements(ArrayVar& dst, ArrayVar& src) = 0;
                /// Called at each pointer. Returns true in mustBeCloned if the pointer target must be cloned.
                /// If HK_FAILURE is returned, the clone fails.
            virtual hkResult atPointer(PointerVar& dst, PointerVar& src, _Out_ PointerAction::Enum* action);
                /// Map a type in the source to the corresponding type in the destination.
            virtual _Ret_maybenull_ const hkReflect::Type* typeFromType(_In_ const hkReflect::Type* srcType);

                /// Called before the first object is cloned.
            virtual void cloneStart();
                /// If clone failed, the callback must clean up.
            virtual void cloneEnd(hkResult r);
                /// virtual dtor
            virtual ~Callback() {}
        };

            /// A container which holds all the objects which need to have
            /// afterReflectNew called on them.
        class HK_EXPORT_COMMON AfterReflectNewHandler
        {
            friend struct CloneImpl;

        public:
            HK_DECLARE_CLASS(AfterReflectNewHandler, New, NonCopyable);

            AfterReflectNewHandler();
            void callAfterReflectNews();
            ~AfterReflectNewHandler();

        private:

            void addVar(const hkReflect::Var& v);
            void addArray(const hkReflect::ArrayValue& v);

            hkBool isEmpty() const { return m_vars.isEmpty(); }

            hkArray< hkSerialize::VarN > m_vars;
        };

        template<typename T0, typename T1 = T0>
        struct Pair
        {
            void set(const T0& d, const T1& s) { dst = d; src = s; }
            T0 dst;
            T1 src;
        };

        Cloner();
        ~Cloner();

            /// If this is called with true, cloning will not fail when a type is not found or an object/array cannot be
            /// allocated/constructed. Instead, the failed object/array will be ignored (pointers to invalid objects
            /// will be set to HK_NULL in the cloned tree; failed arrays will be left empty).
            /// False by default.
        void setIgnoreInvalid(bool ignore);

        void setAfterReflectNewHandler(_In_ hkReflect::Cloner::AfterReflectNewHandler* afterReflectNewHandler);
        void setDyingObjectsSalvageArray(_In_ hkArray<hkRefPtr<hkReferencedObject>>* array);

        void setPerformValidation(bool val = true);

            /// If orig has been cloned by this Cloner object, returns its clone, otherwise returns an empty Var.
        hkReflect::Var getCachedClone(const hkReflect::Var& orig) const;

            /// Resets the cache of cloned objects.
            /// When an object is cloned, it is added to an internal cache. This way, any subsequent references to this
            /// object will reuse the previous clone. The cache expects the objects to remain valid between clone requests.
            /// Use this method to clear this cache if the cached objects are no longer valid (e.g. have been destroyed).
        void clearObjectCache();

            /// Clone a var.
        Var HK_CALL cloneVar(const Var& var, Callback& callback);
            /// Clone a full object tree allocating each new object on the heap.
        Var HK_CALL cloneVarRecursive(const Var& var);

            /// Clone a value.
        template<typename T>
        T* HK_CALL clone(_In_ const T* obj, Callback& callback)
        {
            return cloneVar(Var(obj), callback).dynCast<T>();
        }
            /// Clone a full object tree allocating each new object on the heap.
        template<typename T>
        T* HK_CALL cloneRecursive(_In_ const T* obj)
        {
            return cloneVarRecursive(Var(obj)).dynCast<T>();
        }

        /// Create a shallow clone of a var.
        /// Caution: Repeatedly calling this method with the same cloner object will produce clones which refer to each other.
        /// You can call clearObjectCache in between calls to avoid this behavior.
        Var cloneOne( const Var& src, _In_opt_ AfterReflectNewHandler* afrnHandler = HK_NULL, _In_opt_ hkArray<hkRefPtr<hkReferencedObject>>* salvageArray = HK_NULL);

        /// Create a shallow clone of an object.
        /// Caution: Repeatedly calling this method with the same cloner object will produce clones which refer to each other.
        /// You can call clearObjectCache in between calls to avoid this behavior.
        template<typename T>
        _Ret_notnull_ T* HK_CALL cloneOne(_In_ const T* obj)
        {
            return cloneOne(Var(obj)).dynCast<T>();
        }

    protected:
        CloneImpl* m_impl;
    };

    namespace Detail
    {
            /// Clone a var into a preallocated var.
        HK_EXPORT_COMMON Var HK_CALL cloneVarInto(Var& dst, const Var& src, Cloner::Callback& callback);

            /// Clone a value into a preallocated destination.
        template<typename T>
        T* HK_CALL cloneInto(_Inout_ T* dst, _In_ const T* src, Cloner::Callback& callback)
        {
            Var d(dst);
            return cloneVarInto(d, Var(src), callback).dynCast<T>();
        }

        struct HK_EXPORT_COMMON ClonerAllocatedVars
        {
            int getSize() const { return m_vars.getSize(); }
            Var operator[](int i) { return m_vars[i]; }
            void pushBack(const Var& var) { m_vars.pushBack(var); }

            void start();
            void end(hkResult r);

            hkArray< Var > m_vars;
        };

        /// Clones an object hierarchy on the heap. If the clone fails, deallocates the created objects.
        /// Uses typeFromType to get the type of new objects (default returns the original type but can be overridden).
        struct HK_EXPORT_COMMON CloneOnHeap : public hkReflect::Cloner::Callback
        {
        public:
            virtual hkResult beginVar(hkReflect::Var& dst, hkReflect::Var& src) HK_OVERRIDE;
            virtual hkResult beginArrayElements(hkReflect::ArrayVar& dst, hkReflect::ArrayVar& src) HK_OVERRIDE;
            virtual void cloneStart() HK_OVERRIDE;
            virtual void cloneEnd(hkResult r) HK_OVERRIDE;

            hkReflect::Detail::ClonerAllocatedVars m_vars;
        };

        // Strip the setter Impl from the Var if present.
        template<typename MyVar>
        static HK_INLINE MyVar getPlain(const MyVar& dst)
        {
            if (FieldDecl asField = QualType(dst.getType()))
            {
                if (asField.hasCustomSetter())
                {
                    const typename MyVar::MyType* plainType = hkDynCast(dst.getType()->getParent());
                    HK_ASSERT_NO_MSG(0x75bbf241, plainType);
                    return MyVar(dst.getAddress(), plainType);
                }
            }
            return dst;
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
