// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Util/hkReflectClone.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>
#include <Common/Base/Reflect/Util/hkReflectVarMapOperations.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Serialize/Core/hkSerializeCore.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompiler.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmFastCopyInterpreter.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompilerPasses.h>
#include <Common/Base/Types/Properties/hkDefaultPropertyBag.h>
#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>
#include <Common/Base/Reflect/Util/hkReflectUtil.h>
#include <Common/Base/Object/hkSingleton.h>

#include <Common/Base/Container/PointerMap/hkMap.hxx>

#define DEBUG_LOG_IDENTIFIER "reflect.Clone"
#include <Common/Base/System/Log/hkLog.hxx>

#define HK_FAIL_IF(COND, ...) { \
    if (COND) { \
        Log_Warning(__VA_ARGS__); \
        m_ok = false; \
        return; \
    } }

namespace
{
    using namespace hkReflect;

    HK_INLINE void checkForNoSetters(const Var& dst)
    {
        HK_ASSERT(0x5c007777, !dst.getType()->isField() || !dst.getType()->isField().hasCustomSetter(),
            "Setters should have been removed at this point");
    }

    struct CloneInterpreter : public hkTypeVm::FastCopyInterpreter
    {
        hkArray< Cloner::Pair<Var> >* newVars;
        hkInplaceArray< Cloner::Pair<PointerVar>, 64 >* pointers;

        HK_NEVER_INLINE hkResult execString(StringVar dst, StringVar src)
        {
            
            DLOG("String dst={} src={}", dst, src);
            newVars->expandOne().set(dst,src);
            return HK_SUCCESS;
        }

        HK_NEVER_INLINE hkResult execPointer(PointerVar dst, PointerVar src, int dstStride, int srcStride, int repeat)
        {
            for(int i=0; i < repeat; i++, dst = incrementAddress(dst, dstStride), src = incrementAddress(src, srcStride))
            {
                DLOG("Pointer dst={} src={}", dst, src);
                pointers->expandOne().set(dst, src);
            }
            return HK_SUCCESS;
        }

        HK_NEVER_INLINE hkResult execArray(const ArrayVar& dst, const ArrayVar& src)
        {
            DLOG("Array dst={} src={}", dst, src);
            newVars->expandOne().set(dst, src);
            return HK_SUCCESS;
        }

        HK_INLINE hkResult execBool(const hkReflect::BoolVar& dst, const hkReflect::BoolVar& src)
        {
            checkForNoSetters(dst);
            return FastCopyInterpreter::execBool(dst, src);
        }
        HK_INLINE hkResult execInt(const hkReflect::IntVar& dst, const hkReflect::IntVar& src)
        {
            checkForNoSetters(dst);
            return FastCopyInterpreter::execInt(dst, src);
        }
        HK_INLINE hkResult execFloat(const hkReflect::FloatVar& dst, const hkReflect::FloatVar& src)
        {
            checkForNoSetters(dst);
            return FastCopyInterpreter::execFloat(dst, src);
        }
    };
}

struct hkReflect::Cloner::CloneImpl : public hkReflect::VarVisitor<CloneImpl, void, Var&>
{
    HK_DECLARE_CLASS(CloneImpl, New);

    enum CloneFlags
    {
        FLAG_NONE = 0,
        FLAG_NEEDS_CONSTRUCTOR = 1
    };

    CloneImpl()
        : m_callback(HK_NULL)
        , m_ok(true)
        , m_ignoreInvalid(false)
        , m_afterReflectNewHandler(HK_NULL)
        , m_dyingObjectsSalvageArray(HK_NULL)
    {
        hkTypeVm::addDefaultPasses(m_compiler);
        m_compiler.addPass<hkTypeVm::IntAndFloatConversionPass>();
        m_compiler.addPass<hkTypeVm::CheckTypeKindsPass>();
        m_compiler.addInstrWithNoSource = false;
        m_compiler.excludeNonSerializableFields = true;
    }

    ~CloneImpl()
    {
    }

    void setIgnoreInvalid(bool ignore) { m_ignoreInvalid = ignore; }

    void setAfterReflectNewHandler(_In_ hkReflect::Cloner::AfterReflectNewHandler* cb) { m_afterReflectNewHandler = cb; }
    void setDyingObjectsSalvageArray(_In_ hkArray<hkRefPtr<hkReferencedObject>>* array)
    {
        // We should be either setting it from null to a value or from a value to null
        // For ease of use, setting it to the already set value is allowed
        HK_ASSERT_NO_MSG(0x93b22de1, (m_dyingObjectsSalvageArray == array) || !array || !m_dyingObjectsSalvageArray);
        m_dyingObjectsSalvageArray = array;
    }
    void setPerformValidation(bool val) { m_validation.setEnabled(val); }

    hkReflect::Var getCachedClone(const hkReflect::Var& orig) const
    {
        return orig ? m_cloneFromOriginal.getWithDefault(orig, hkReflect::Var()) : hkReflect::Var();
    }

    void clearObjectCache()
    {
        m_ok = true;
        m_cloneFromOriginal.clearAndDeallocate();
        m_newVars.clearAndDeallocate();
        m_newObjs.clearAndDeallocate();
        m_compiler.clearTemporaryTypes();
    }

    void clone(Var& dst, const Var& src, Callback& cb)
    {
        Log_Auto("clone dst={} src={}", dst, src);
        cb.cloneStart();
        m_callback = &cb;

        hkReflect::Cloner::AfterReflectNewHandler localAfterReflectNewHandler;
        hkBool hasLocalAfterReflectNewHandler = m_afterReflectNewHandler == HK_NULL;
        if(hasLocalAfterReflectNewHandler)
        {
            m_afterReflectNewHandler = &localAfterReflectNewHandler;
        }

        clone1(dst, src, FLAG_NEEDS_CONSTRUCTOR);
        while( m_newVars.getSize() && m_ok )
        {
            Pair<Var> newVar = m_newVars.back();
            m_newVars.popBack();
            clone1(newVar.dst, newVar.src, FLAG_NONE);
        }
        if (m_ok)
        {
            m_ok = m_validation.callAndClearValidation();
        }
        if( m_ok )
        {
            if( hasLocalAfterReflectNewHandler )
            {
                localAfterReflectNewHandler.callAfterReflectNews();
                m_afterReflectNewHandler = HK_NULL;
            }

            cb.cloneEnd(HK_SUCCESS);
        }
        else
        {
            if( hasLocalAfterReflectNewHandler )
            {
                m_afterReflectNewHandler = HK_NULL;
            }

            cb.cloneEnd(HK_FAILURE);
            if (src.getType())
            {
                Log_Warning("Unable to clone object {} or one of its dependencies", src.getType()->getName());
            }
            dst = hkReflect::Var();
        }

        if (m_ok)
        {
            // Remove one reference from all the cloned hkReferencedObjects so that their refcount matches the number of
            // pointers referencing them.
            for (const auto& pair : m_newObjs)
            {
                if (hkReferencedObject* refObj = hkDynCast(pair.dst))
                {
                    HK_ASSERT(0x69308067, refObj->getReferenceCount() >= 1, "One too many references were removed from a cloned object" );
                    if (refObj->getReferenceCount() == 1)
                    {
                        // TODO: Get rid of m_dyingObjectsSalvageArray hack
                        if (m_dyingObjectsSalvageArray)
                        {
                            m_dyingObjectsSalvageArray->expandOne() = refObj;
                        }
                        else
                        {
                            // Some afterReflectNew/callback has removed the last reference to this object. Remove it from the cache.
                            m_cloneFromOriginal.remove(pair.src);
                        }
                    }
                    refObj->removeReference();
                }
            }
            m_newObjs.clear();

            // Do the same for the root object (but do not delete it).
            if (hkReferencedObject* refObj = hkDynCast(dst))
            {
                refObj->setReferenceCount(refObj->getReferenceCount() - 1);
            }
        }

        m_callback = HK_NULL;
    }

    void cloneInto(Var& dst, const Var& src, Callback& cb)
    {
        m_cloneFromOriginal.insert(src, dst);
        clone(dst, src, cb);
    }


    template<typename T>
    void visit(const T& src, Var& dst )
    {
        T tdst = dst;
        HK_FAIL_IF( !tdst, "Cannot clone an object of type {} onto an object of type {} (different kinds, {} != {})",
            src.getType(), dst.getType(), src.getType()->getKind(), dst.getType()->getKind() );
        visit2(src, tdst);
    }

    void visit2(const PointerVar& src, PointerVar& dst)
    {
        Pair<PointerVar> p = { dst, src };
        handlePointers(&p, 1);
    }

    void visit2( const RecordVar& src, RecordVar& dst )
    {
        hkInplaceArray< Pair<PointerVar>, 64 > pointers;

        // If a record is a property, iterate over its elements individually
        
        if (dst.getType()->isProperty())
        {
            for (hkReflect::DeclIter<FieldDecl> it(src.getType()); it.advance();)
            {
                hkReflect::Var srcElem = src[it.current()];
                hkReflect::Var dstElem = dst[it.current().getName()];

                if (dstElem)
                {
                    switch (dstElem.getType()->getKind())
                    {
                    case KIND_STRING:
                    case KIND_ARRAY:
                    {
                        m_newVars.expandOne().set(dstElem, srcElem);
                        break;
                    }
                    case KIND_POINTER:
                    {
                        pointers.expandOne().set(dstElem, srcElem);
                        break;
                    }
                    default:
                        dispatch(srcElem, dstElem);
                    }
                }
            }
        }
        else
        {
            
            if (src.getType()->isForeign() && src.getType()->isDynamicType() && dst.getType()->isDynamicType() == false)
            {
                hkReflect::ArrayVar fieldArray = src["foreignUtil"]["fields"];
                HK_FAIL_IF( !fieldArray, "No ForeignUtil found in foreign type" );

                hkReflect::ArrayValue fieldValues = fieldArray.getValue();
                for (int i = 0; i < fieldValues.getCount(); ++i)
                {
                    hkReflect::StringVar name = fieldValues[i]["name"];
                    HK_FAIL_IF( !name, "Name not found in dynamic field" );
                    hkReflect::ArrayVar contents = fieldValues[i]["contents"];
                    HK_FAIL_IF( !contents, "Contents not found in dynamic field" );
                    HK_FAIL_IF( contents.getCount() == 0, "Empty dynamic field in foreign object" );

                    hkReflect::Var s = contents[0];
                    Var d = dst[name.getValue()];

                    if (d)
                    {
                        
                        
                        
                        auto dt = d.getType(), st = s.getType();
                        if ( (dt->asValue() && dt->getKind()==st->getKind()) // special case valuetypes, we can try to copy them even if the name has changed
                            || (dt->getFullName() == st->getFullName()) )
                        {
                            switch ( st->getKind() )
                            {
                            case KIND_STRING:
                            case KIND_ARRAY:
                            {
                                m_newVars.expandOne().set( d, s );
                                break;
                            }
                            case KIND_POINTER:
                            {
                                pointers.expandOne().set( d, s );
                                break;
                            }
                            default:
                                dispatch( s, d );
                            }
                        }
                        else
                        {
                            Log_Warning( "Property '{}.{}' has type '{}', but a value of type '{}' has been found "
                                "in the file. The value for the property must be reset manually.",
                                dst.getType()->getFullName(), (const char*)name.getValue(), dt->getFullName(),
                                st->getFullName() );
                        }
                    }
                }
            }
            const hkTypeVm::Program* program = m_compiler.compile( src.getType(), dst.getType() );

            HK_FAIL_IF( program == HK_NULL, "Cannot compile type {}", (dst.getType() ? dst.getType()->getName() : "<No name>") );

            CloneInterpreter machine;
            machine.pointers = &pointers;
            machine.newVars = &m_newVars;
            hkResult res = hkTypeVm::FastCopyInterpreter::exec1( machine, program,
                dst.getAddress(), dst.getType()->getSizeOf(),
                src.getAddress(), src.getType()->getSizeOf() );
            HK_FAIL_IF( res.isFailure(), "Cannot copy object" );

            if (hkReflect::Detail::hasAfterReflectNew( dst.getType() ))
            {
                m_afterReflectNewHandler->addVar( dst );
            }
        }

        handlePointers(pointers.begin(), pointers.getSize());
    }

    void visit2( const ArrayVar& srcOrig, ArrayVar& dst, bool hackInTransientPropertyMap = false )
    {
        ArrayVar src = srcOrig;

        // PROPERTY BAG CLONING HACK
        // During native-to-native cloning we want to a) clone hkDefaultPropertyBag::m_transientPropertyMap even if it
        // is hk::Serialize(false), b) skip properties marked as NO_CLONE, c) preserve NO_CLONE properties already on
        // the destination (hack currently needed by pointers in WinRt objects).
        
        {
            // Detect native cloning by checking that both src and dst are the m_items field in a native property map.
            typedef hkHashMapDetail::MapTuple<hkPropertyId, Any> PropertyEntry;
            auto dstAsNativeProperties = dst.getType()->getTypeWorld() == HK_NULL ?
                dst.dynCast< hkArray< PropertyEntry > >() : HK_NULL;
            auto srcAsNativeProperties = src.getType()->getTypeWorld() == HK_NULL ?
                src.dynCast< hkArray< PropertyEntry > >() : HK_NULL;
            if ( srcAsNativeProperties && dstAsNativeProperties )
            {
                // HACK - Force visit of m_transientPropertyMap even if the compiler ignored it.
                if ( !hackInTransientPropertyMap )
                {
#ifdef HK_DEBUG_SLOW
                    // Confirm that the hack is up-to-date.
                    {
                        const hkReflect::Type* defaultPropertyBagType = hkReflect::getType<hkDefaultPropertyBag>();
                        // The fields are protected, so have to use names.
                        hkReflect::DataFieldDecl srcDecl = defaultPropertyBagType->findDecl( "propertyMap" ).asDataField();
                        hkReflect::DataFieldDecl nextDecl = defaultPropertyBagType->findDecl( "transientPropertyMap" ).asDataField();
                        HK_ASSERT(0x306d335e, srcDecl && nextDecl, "hkDefaultPropertyBag has changed. Check the special-casing is still valid." );
                        HK_ASSERT(0x3afbd04d, ( nextDecl.getOffset() - srcDecl.getOffset() ) == sizeof( hkDefaultPropertyBag::HashMap ), "hkDefaultPropertyBag has changed. Check the special-casing is still valid." );
                        HK_ASSERT(0x77d3b00a, srcDecl.getType()->equals( nextDecl.getType() ), "hkDefaultPropertyBag has changed. Check the special-casing is still valid." );
                    }
#endif
                    // Bump the offset by the size of a HashMap.
                    hkReflect::ArrayVar srcNext( hkAddByteOffset( src.getAddress(), sizeof( hkDefaultPropertyBag::HashMap ) ), src.getType() );
                    hkReflect::ArrayVar dstNext( hkAddByteOffset( dst.getAddress(), sizeof( hkDefaultPropertyBag::HashMap ) ), dst.getType() );

                    // Execute this method again on the transient map.
                    visit2( srcNext, dstNext, /*hackInTransientPropertyMap =*/ true );
                }

                // Count the number of clonable properties in src.
                int numToCloneFromSrc = 0;
                for ( const auto& element : *srcAsNativeProperties )
                {
                    if ( element.m_0.desc()->isCloneable() )
                    {
                        ++numToCloneFromSrc;
                    }
                }


                if ( numToCloneFromSrc == 0 )
                {
                    // Early out, leave dst unchanged.
                    return;
                }

                // Skip the NO_CLONE properties already in dst.
                int numInDst = dstAsNativeProperties->getSize();
#ifdef HK_DEBUG_SLOW
                for ( const auto& element : *dstAsNativeProperties )
                {
                    HK_ASSERT_DEBUG(0x3578be8e, !element.m_0.desc()->isCloneable() && !element.m_0.desc()->isSerialized(),
                        "A property bag has been initialized with clonable/serializable properties before being filled by "
                        "cloning, this is not allowed" );
                }
#endif

                if ( numInDst || numToCloneFromSrc != src.getCount() )
                {
                    // Merge clonable properties in src with the ones already in dst. Rely on hkHashMap::afterReflectNew
                    // to fixup the index correctly at the end.
                    dstAsNativeProperties->setSize( numInDst + numToCloneFromSrc );
                    CloneInterpreter machine;
                    hkInplaceArray< Pair<PointerVar>, 64 > pointers;
                    machine.pointers = &pointers;
                    machine.newVars = &m_newVars;
                    ArrayValue valSrc = src.getValue();
                    auto cloneBuf = dstAsNativeProperties->view().ltrim( numInDst );
                    const hkTypeVm::Program* program = m_compiler.compile( hkReflect::getType<PropertyEntry>() );
                    int dstIdx = 0;
                    for ( const auto& element : *srcAsNativeProperties )
                    {
                        if ( element.m_0.desc()->isCloneable() )
                        {
                            // Clone properties one by one.
                            hkResult res = hkTypeVm::FastCopyInterpreter::exec1( machine, program,
                                &cloneBuf[dstIdx], sizeof( PropertyEntry ),
                                &element, sizeof( PropertyEntry ) );
                            HK_FAIL_IF( res.isFailure(), "Cannot copy object" );
                            ++dstIdx;
                        }
                    }
                    HK_ASSERT_NO_MSG(0x5fbfd37b, Detail::hasAfterReflectNew( hkReflect::getType<PropertyEntry>() ) );
                    m_afterReflectNewHandler->addArray( ArrayValue( cloneBuf.begin(), cloneBuf.getSize(), hkReflect::getType<PropertyEntry>() ) );
                    handlePointers( pointers.begin(), pointers.getSize() );
                    return;
                }
                // else fall through and copy array normally.
            }
        }

        hkResult res = m_callback->beginArrayElements(dst, src);
        if (!res.isSuccess())
        {
            Log_Warning("Cannot allocate array elements of type '{}', element type '{}', count={}",
                srcOrig.getType()->getFullName(), src.getSubType() ? src.getSubType()->getFullName() : hkReflect::TypeName("NULL", nullptr),
                srcOrig.getCount());
            HK_FAIL_IF(!m_ignoreInvalid, "");
        }

        hkReflect::ArrayValue valDst;
        HK_FAIL_IF(Detail::getPlain(dst).getValue(&valDst).isFailure(), "Cannot access array value");

        if( int count = valDst.getCount() )
        {
            HK_ASSERT_NO_MSG( 0x67d72b15, src.getCount() == count );
            hkReflect::ArrayValue valSrc;
            HK_FAIL_IF(src.getValue(&valSrc).isFailure(), "Cannot access array value");
            hkInplaceArray< Pair<PointerVar>, 64 > pointers;

            if (valDst.getAddress() == valSrc.getAddress())
            {
                HK_FAIL_IF(valDst.getSubType() != valSrc.getSubType() && !valDst.getSubType()->equals(valSrc.getSubType()),
                    "Clone on self with different subtype is not supported");
                HK_FAIL_IF(valDst.getCount() != valSrc.getCount(),
                    "Clone on self with different count is not supported");
            }
            else
            {
                HK_FAIL_IF(hkRangesOverlap(valDst.getAddress(), valDst.getEndAddress(),
                    valSrc.getAddress(), valSrc.getEndAddress()),
                    "Cannot clone on a buffer overlapping with source array");

                // If an array is a property, iterate over its elements individually
                if (dst.getType()->isProperty() || (dst.hasDynamicImpl() && dst.getImpl()->isProperty()))
                {
                    for (int i = 0; i < valDst.getCount(); ++i)
                    {
                        hkReflect::Var srcElem = valSrc[i];
                        hkReflect::Var dstElem = valDst[i];

                        {
                            switch (dstElem.getType()->getKind())
                            {
                            case KIND_STRING:
                            case KIND_ARRAY:
                            {
                                m_newVars.expandOne().set(dstElem, srcElem);
                                break;
                            }
                            case KIND_POINTER:
                            {
                                pointers.expandOne().set(dstElem, srcElem);
                                break;
                            }
                            default:
                                dispatch(srcElem, dstElem);
                            }
                        }
                    }
                }
                else
                {
                    const hkTypeVm::Program* program = m_compiler.compile(valSrc.getSubType(), valDst.getSubType());

                    HK_FAIL_IF(!program, "Cannot compile array subtype {}", (valDst.getSubType() ? valDst.getSubType()->getName() : "<No name>"));

                    CloneInterpreter machine;
                    machine.pointers = &pointers;
                    machine.newVars = &m_newVars;

                    hkResult r = hkTypeVm::FastCopyInterpreter::execN(machine, program,
                        valDst.getAddress(), count*valDst.getStride(),
                        valSrc.getAddress(), count*valSrc.getStride(),
                        valDst.getStride(), valSrc.getStride(), valSrc.getCount() );
                    HK_FAIL_IF( r.isFailure(), "Cannot copy array" );

                    if(count)
                    {
                        m_validation.addArray(valDst[0], count, valDst.getStride());
                    }

                    if (hkReflect::Detail::hasAfterReflectNew(dst.getType()))
                    {
                        m_afterReflectNewHandler->addVar(dst);
                    }
                    if (hkReflect::Detail::hasAfterReflectNew(valDst.getSubType()))
                    {
                        m_afterReflectNewHandler->addArray(valDst);
                    }

                }

                handlePointers(pointers.begin(), pointers.getSize());
            }
        }
    }

    template<typename KVar>
    void copyValue( const KVar& src, const KVar& dst )
    {
        typename hkReflect::ValueFromType<typename KVar::MyType>::Type value;
        HK_FAIL_IF( src.getValue( &value ).isFailure(), "Failed to get value from Var '{}'", src );
        HK_FAIL_IF( dst.setValue( value ).isFailure(),
            "Failed to set value '{:*}' ('{}') on Var of type '{}'", src, src.getType(), dst.getType() );
    }

    void visit2( const IntVar& src, IntVar& dst )
    {
        checkForNoSetters(dst);
        copyValue( src, dst );
    }
    void visit2( const FloatVar& src, FloatVar& dst )
    {
        checkForNoSetters(dst);
        copyValue( src, dst );
    }

    void visit2( const StringVar& src, StringVar& dst )
    {
        copyValue( src, Detail::getPlain( dst ) );
    }
    void visit2( const BoolVar& src, BoolVar& dst )
    {
        checkForNoSetters(dst);
        copyValue( src, dst );
    }

    void visit2( const OpaqueVar& src, OpaqueVar& dst )
    {
        HK_FAIL_IF( true, "Cannot clone an object of type Opaque" );
    }

    void visit2( const VoidVar& src, VoidVar& dst )
    {
        HK_FAIL_IF( true, "Cannot clone an object of type void" );
    }

protected:

    HK_NEVER_INLINE void handlePointers(_Inout_updates_(numPointers) Pair<PointerVar>* pointers, int numPointers)
    {
        for( int i = 0; i < numPointers; ++i)
        {
            Pair<PointerVar>& p = pointers[i];
            PointerAction::Enum action = PointerAction::Enum::SKIP;
            hkResult cbRes = m_callback->atPointer( p.dst, p.src, &action );
            HK_FAIL_IF( cbRes.isFailure(), "Cloner callback failed on {} pointing to {}", p.dst.getType(), p.dst.getSubType()) ;
            if(action == PointerAction::Enum::CLONE)
            {
                PointerVar plainDst = Detail::getPlain(p.dst);
                Var src;
                HK_FAIL_IF(p.src.getValue(&src).isFailure(), "Can't get var from {} pointing to {}", p.src.getType(), p.src.getSubType());
                if(src)
                {
                    Var dst = m_cloneFromOriginal.getWithDefault(src, Var());
                    if (dst.isValid() == false)
                    {
                        #if 0
                        // This is very slow so enable it only when debugging this.
                        // Make sure that we didn't miss the value in the map because of a non-exact type.
                        for (hkMap<Var, Var>::Iterator it = m_cloneFromOriginal.getIterator(); m_cloneFromOriginal.isValid(it); it = m_cloneFromOriginal.getNext(it))
                        {
                            hkReflect::Var cur = m_cloneFromOriginal.getValue(it);
                            if (cur.getAddress() == src.getAddress())
                            {
                                // It's OK to have different Vars in the map with the same address as long as their types are unrelated.
                                HK_ASSERT_NO_MSG(0x31debc8, !cur.getType()->extendsOrEquals(src.getType()) && !src.getType()->extendsOrEquals(cur.getType()));
                            }
                        }
                        #endif

                        clone1(dst, src, FLAG_NEEDS_CONSTRUCTOR);

                        if(!m_ok)
                        {
                            return;
                        }

                        Pair<Var> pair = { dst, src };
                        m_newObjs.pushBack(pair);
                    }
                    hkResult res = plainDst.setValue( dst );
                    HK_FAIL_IF( res.isFailure(), "Cannot assign '{:*}' ('{}') to pointer of type {} at {}",
                        dst, dst.getType(), plainDst, p.dst.getType()->isField() );
                }
                else
                {
                    hkResult res = plainDst.setValue( Var() );
                    HK_FAIL_IF( res.isFailure(), "Cannot assign null to pointer of type {} at {}",
                        plainDst, p.dst.getType()->isField() );
                }
            }
        }
    }

    void clone1(Var& dst, const Var& srcOrig, const CloneFlags flags)
    {
        Var dstOrig = dst;
        Var src = srcOrig;
        DLOG_SCOPE("clone1 dst={} src={}", dst, src);

        // allow dst or src or both or neither to be replaced
        hkResult begin = m_callback->beginVar(dst, src);
        if (begin.isFailure())
        {
            if ( const hkReflect::Type* asType = Detail::asType(src) )
            {
                HK_FAIL_IF(m_ignoreInvalid == false, "Cannot find native type for {} (unregistered type?)", asType->getFullName());
                // If can ignore, just skip.
                Log_Info("Skipping type {} (callback returned failure)", asType->getFullName());
            }
            else
            {
                if ( src.getType()->isForeign() )
                {
                    HK_FAIL_IF( m_ignoreInvalid == false, "Cannot allocate/construct instance of foreign type '{}'. This is likely due to the type being defined"
                        " in a missing (not loaded) WinMD file. Check that WinMD files are present on disk and correctly loaded.", src.getType()->getFullName() );
                }
                else
                {
                    HK_FAIL_IF( m_ignoreInvalid == false, "Cannot allocate/construct instance of type '{}' (unregistered type?)", src.getType()->getFullName() );
                }
                // If can ignore, just skip.
                Log_Info("Skipping instance of {} (callback returned failure)", src.getType()->getFullName());
            }
            dst = hkReflect::Var();
        }

        if (src.getAddress() != srcOrig.getAddress() || src.getType() != srcOrig.getType()) DLOG("modified src={}", src);
        if (dst.getAddress() != dstOrig.getAddress() || dst.getType() != dstOrig.getType()) DLOG("modified dst={}", dst);

        // if there is a a destination and we are not cloning a Type
        if (dst.getAddress() && !Detail::asType(src))
        {
            if (dst.getAddress() == src.getAddress())
            {
                HK_FAIL_IF(dst.getType() != src.getType() && !dst.getType()->equals(src.getType()),
                    "Cannot clone on self with a different type");
            }
            else
            {
                
                void* srcEnd = hkAddByteOffset(src.getAddress(), src.getType()->getSizeOf());
                void* dstEnd = hkAddByteOffset(dst.getAddress(), dst.getType()->getSizeOf());
                HK_FAIL_IF(hkRangesOverlap(dst.getAddress(), dstEnd, src.getAddress(), srcEnd),
                    "Cannot clone on a buffer overlapping with source object");
                // Add to the map only if we created a clone (and thus dstOrig was NULL).
                if (dstOrig.getAddress() == HK_NULL)
                {
                    m_cloneFromOriginal.insert(src, dst);

                    // if the source was replaced, the both "src" and "srcIn" map to "dst"
                    if (src.getAddress() != srcOrig.getAddress())
                    {
                        m_cloneFromOriginal.insert(srcOrig, dst);
                    }
                }

                if (flags == FLAG_NEEDS_CONSTRUCTOR && !dst.getType()->isProperty())
                {
                    hkResult constructed = hkReflect::TypeDetail::reflectConstruct(dst.getAddress(), dst.getType());
                    HK_FAIL_IF(constructed.isFailure(), "Cannot construct type {}", (dst.getType() ? dst.getType()->getName() : "<No name>"));
                }

                dispatch(src, dst);
            }
        }
        m_validation.addVar(dst);
    }

    Cloner::Callback* m_callback;
    bool m_ok;
    bool m_ignoreInvalid;
    hkMap<Var, Var> m_cloneFromOriginal;
    hkArray< Pair<Var> > m_newVars;
    hkArray< Pair<Var> > m_newObjs;
    hkReflect::Cloner::AfterReflectNewHandler* m_afterReflectNewHandler;
    hkArray<hkRefPtr<hkReferencedObject>>* m_dyingObjectsSalvageArray;

    struct Validation
    {
        Validation() : m_enabled(false)
        {
        }

        struct ValidationEntry
        {
            hkReflect::Detail::ValidateFunction m_isValidFunction;
            void* m_address;
            int m_count;
            int m_stride;
            hkReflect::QualType m_type;

            void set(hkReflect::Detail::ValidateFunction fn, void* address, int count, int stride, hkReflect::QualType objType)
            {
                m_isValidFunction = fn;
                m_address = address;
                m_count = count;
                m_stride = stride;
                m_type = objType;
            }
        };

        void addVar(hkReflect::Var v)
        {
            if(m_enabled && v)
            {
                hkReflect::Detail::ValidateFunction fn = hkReflect::TypeDetail::globalGetOptional<Opt::VALIDATE>(v.getType());
                if (fn)
                {
                    m_entries.expandOne().set(fn, v.getAddress(), 1, 0, v.getType());
                }
            }
        }

        void addArray(hkReflect::Var firstElement, int count, int stride)
        {
            if(m_enabled && firstElement)
            {
                hkReflect::Detail::ValidateFunction fn = hkReflect::TypeDetail::globalGetOptional<Opt::VALIDATE>(firstElement.getType());
                if (fn)
                {
                    m_entries.expandOne().set(fn, firstElement.getAddress(), count, stride, firstElement.getType());
                }
            }
        }

        bool callAndClearValidation()
        {
            bool ok = true;
            if(m_enabled)
            {
                for (int i = 0; (i < m_entries.getSize()) && ok; i++)
                {
                    const ValidationEntry& entry = m_entries[i];
                    void* address = entry.m_address;
                    for (int elem = 0; (elem < entry.m_count) && ok; elem++, address = hkAddByteOffset(address, entry.m_stride))
                    {
                        ok = entry.m_isValidFunction(address);
#if defined(HK_DEBUG)
                        if (!ok)
                        {
                            Log_Warning("Type validation failed for {}", entry.m_type);
                        }
#endif
                    }
                }
                if(!ok)
                {
                    Log_Info("Reflect validation failed");
                }
            }

            m_entries.clearAndDeallocate();
            return ok;
        }

        void setEnabled(bool val)
        {
            m_enabled = val;
        }

        hkArray<ValidationEntry> m_entries;
        bool m_enabled;
    };

    Validation m_validation;
    hkTypeVm::Compiler m_compiler;
};


hkResult hkReflect::Cloner::Callback::atPointer(PointerVar& dst, PointerVar& src, _Out_ hkReflect::Cloner::PointerAction::Enum* action)
{
    *action = hkReflect::Cloner::PointerAction::CLONE;
    return HK_SUCCESS;
}

_Ret_maybenull_ const hkReflect::Type* hkReflect::Cloner::Callback::typeFromType(_In_ const hkReflect::Type* srcType) { return srcType; }
void hkReflect::Cloner::Callback::cloneStart() { }
void hkReflect::Cloner::Callback::cloneEnd(hkResult) { }

namespace
{
    struct AfterReflectNewLock : public hkReferencedObject
    {
        HK_DECLARE_CLASS(AfterReflectNewLock, New, Singleton);
        hkCriticalSection m_criticalSection;
    };
    HK_SINGLETON_IMPLEMENTATION(AfterReflectNewLock);
}

void hkReflect::Cloner::AfterReflectNewHandler::callAfterReflectNews()
{
    HK_TIME_CODE_BLOCK( "Reflect.CallAfterReflectNew", HK_NULL );
    HK_MONITOR_ADD_VALUE( "NumVars", (float)m_vars.getSize(), HK_MONITOR_TYPE_INT );
    if (m_vars.getSize())
    {
        hkCriticalSectionLock lock(&AfterReflectNewLock::getInstance().m_criticalSection);
        for (int i = m_vars.getSize() - 1; i >= 0; i--)
        {
            const hkSerialize::VarN& v = m_vars[i];
            hkReflect::Detail::callAfterReflectNew(v.getAddress(), v.getType(), v.getCount());
        }
        m_vars.clear();
    }
}

void hkReflect::Cloner::AfterReflectNewHandler::addVar(const hkReflect::Var& v)
{
    m_vars.pushBack(hkSerialize::VarN::fromSingle(v));
}

void hkReflect::Cloner::AfterReflectNewHandler::addArray(const hkReflect::ArrayValue& v)
{
    m_vars.pushBack(hkSerialize::VarN::fromArray(v.getAddress(), v.getSubType(), v.getCount()));
}

hkReflect::Cloner::Cloner()
    : m_impl(new CloneImpl())
{
}

hkReflect::Cloner::~Cloner()
{
    delete m_impl;
}

void hkReflect::Cloner::setIgnoreInvalid(bool ignore)
{
    m_impl->setIgnoreInvalid(ignore);
}

void hkReflect::Cloner::setAfterReflectNewHandler(_In_ hkReflect::Cloner::AfterReflectNewHandler* handler)
{
    m_impl->setAfterReflectNewHandler(handler);
}

void hkReflect::Cloner::setDyingObjectsSalvageArray(_In_ hkArray<hkRefPtr<hkReferencedObject>>* array)
{
    m_impl->setDyingObjectsSalvageArray(array);
}

void hkReflect::Cloner::setPerformValidation(bool val)
{
    m_impl->setPerformValidation(val);
}

hkReflect::Var hkReflect::Cloner::getCachedClone(const hkReflect::Var& orig) const
{
    return m_impl->getCachedClone(orig);
}

void hkReflect::Cloner::clearObjectCache()
{
    m_impl->clearObjectCache();
}

hkReflect::Var hkReflect::Cloner::cloneVar(const hkReflect::Var& topVar, Callback& callback)
{
    if( topVar )
    {
        Var vdst;
        m_impl->clone(vdst, topVar, callback);
        return vdst;
    }
    else
    {
        Log_Warning("Attempting to clone a null var");
        return Var();
    }
}

hkReflect::Var hkReflect::Cloner::cloneVarRecursive(const hkReflect::Var& topVar)
{
    hkReflect::Detail::CloneOnHeap cb;
    return cloneVar(topVar, cb);
}

hkReflect::Var hkReflect::Detail::cloneVarInto(hkReflect::Var& dst, const hkReflect::Var& src, Cloner::Callback& callback)
{
    Cloner::CloneImpl cloner;
    cloner.cloneInto(dst, src, callback);
    return dst;
}

void hkReflect::Detail::ClonerAllocatedVars::start()
{
    m_vars.clear();
}

namespace
{
    inline void clearAllocs(const hkReflect::Var& var)
    {
        if (var.getType()->asRecord())
        {
            // Clear allocations in fields.
            for (DeclIter<DataFieldDecl> it(var.getType()); it.advance();)
            {
                clearAllocs(var[it.current()]);
            }
        }
        else if (hkReflect::StringVar str = var)
        {
            // Clear string buffer.
            str.getType()->getImpl()->clearAllocs(var.getAddress(), str.getType());
        }
        else if (hkReflect::ArrayVar arr = var)
        {
            // Clear allocations in elements.
            // This might be quite slow, but should not be performance-critical.
            hkReflect::ArrayValue val = arr.getValue();
            if (val.getSubType() &&
                (val.getSubType()->asRecord() || val.getSubType()->asString() || val.getSubType()->asArray()))
            {
                for (int i = 0; i < val.getCount(); ++i)
                {
                    clearAllocs(val[i]);
                }
            }

            // Clear array buffer.
            arr.getType()->getImpl()->clearAllocs(var.getAddress(), arr.getType());
        }
    }
}

void hkReflect::Detail::ClonerAllocatedVars::end(hkResult r)
{
    if( r.isFailure() )
    {
        for (int i = 0; i < m_vars.getSize(); i++)
        {
            if (Var v = m_vars[i])
            {
                clearAllocs(v);

                // Var.destroy won't work when we have already called the destructor
                TypeDetail::deallocate(v.getAddress(), v.getType());
            }
        }
    }
}


hkResult hkReflect::Detail::CloneOnHeap::beginVar(Var& dst, Var& src)
{
    
    
    if (!src)
    {
        Log_Info( "Cannot clone null Var" );
        return HK_FAILURE;
    }

    if ( const hkReflect::Type* t = asType(src) )
    {
        if (hkReflect::Decl asDecl = hkReflect::QualType(t))
        {
            const hkReflect::Type* context = typeFromType(asDecl.getDeclContext());
            if ( context == nullptr )
            {
                Log_Info( "Cannot find native type for '{}' (context of '{}'). ", asDecl.getDeclContext(), asDecl );
                return HK_FAILURE;
            }
            hkReflect::Decl nativeDecl = context->findDecl(asDecl.getName(), false);
            if ( !nativeDecl )
            {
                Log_Info( "Cannot find decl '{}' in type '{}'.", asDecl.getName(), context );
                return HK_FAILURE;
            }

            dst = hkReflect::Var(nativeDecl.getType());
        }
        else
        {
            // Map the referenced type to the native equivalent.
            const hkReflect::Type* d = typeFromType(t);
            if (d == HK_NULL)
            {
                Log_Info( "Cannot find native type for '{}'. ", t );
                return HK_FAILURE;
            }

            dst = hkReflect::Var(d);
        }
        return HK_SUCCESS;
    }
    else if (!dst)
    {
        // Allocate a new object on heap.
        const hkReflect::Type* d = typeFromType(src.getType());
        if (d == HK_NULL)
        {
            Log_Info( "Cannot find native type for '{}'. ", t );
            return HK_FAILURE;
        }

        // Check if the type can be constructed.
        
        if (TypeDetail::getReflectConstructionFunction(d) == HK_NULL)
        {
            Log_Info( "Type '{}' cannot be constructed for cloning. ", d );
            return HK_FAILURE;
        }

        // Try to allocate an instance.
        dst = hkReflect::TypeDetail::allocateForClone(src, d);
        if (!dst.isValid())
        {
            Log_Info( "Type '{}' cannot be allocated for cloning. ", d );
            return HK_FAILURE;
        }
        m_vars.pushBack(dst);
    }
    return HK_SUCCESS;
}


void hkReflect::Detail::CloneOnHeap::cloneStart()
{
    m_vars.start();
}

void hkReflect::Detail::CloneOnHeap::cloneEnd(hkResult r)
{
    m_vars.end(r);
}

namespace
{
    void warnAboutPreallocatedDstArray(const hkReflect::Var& dst)
    {
#if defined(HK_DEBUG_SLOW)
        // The dst array has been initialized before cloning, warn the user.
        hkReflect::TypeName className;
        hkStringBuf instanceName;

        if(hkReflect::FieldDecl field = dst.getType()->isField())
        {
            className = field.getDeclContext()->getFullName();
            instanceName.format("{}::{}", className, field.getName());

            if(hkReflect::Var declCtx = hkReflectUtil::getDeclContext(dst))
            {
                className = declCtx.getType()->getFullName();
            }
        }
        else
        {
            className = dst.getType()->getFullName();
            className.toString(instanceName);
        }

        Log_Warning("An instance of {} was initialized non-empty during cloning, possibly by the default"
            "constructor of {}. This initialization is wasteful because the allocations and the default elements "
            "in {} will be discarded and overwritten by the cloned values. Add BypassCtor to {} to prevent the "
            "superfluous initialization.", instanceName.cString(), className, instanceName.cString(), className);
#endif
    }
}

hkResult hkReflect::Detail::CloneOnHeap::beginArrayElements(hkReflect::ArrayVar& dst, hkReflect::ArrayVar& src)
{
    ArrayVar plainDst = getPlain(dst);

    // The ArrayValue may own parts of the impl etc
    hkReflect::ArrayValue srcValue;
    if (src.getValue(&srcValue).isFailure()) { return HK_FAILURE; }

    hkReflect::ArrayValue dstValue;
    if (plainDst.getValue(&dstValue).isFailure()) { return HK_FAILURE; }

    const hkReflect::Type* srcSubType = srcValue.getSubType();
    const hkReflect::Type* dstSubType = dstValue.getSubType();
    bool isVariantArray = false;

    if (dstSubType == HK_NULL && srcSubType != HK_NULL)
    {
        isVariantArray = true;
        // Ensure variant arrays subtype is set.
        dstSubType = typeFromType(srcSubType);
        if (dstSubType == HK_NULL)
        {
            return HK_FAILURE;
        }
    }

    int srcCount = src.getCount();

    if (srcCount != 0 || isVariantArray)
    {
        hkReflect::Detail::ArrayImpl::AllocResult res =
            plainDst.getImpl()->allocateElements(plainDst.getAddress(), plainDst.getType(), dstSubType, srcCount);
        if (res == hkReflect::Detail::ArrayImpl::ALLOC_FAILURE)
        {
            return HK_FAILURE;
        }
        else if (res == hkReflect::Detail::ArrayImpl::ALLOC_NOT_EMPTY)
        {
            if(isVariantArray && dstSubType == HK_NULL)
            {
                // We rely on allocateElements setting the subtype anyway since there is no easy way
                // we can do it at this point.
                return HK_FAILURE;
            }
            warnAboutPreallocatedDstArray(dst);
            return plainDst.setArraySize(srcCount);
        }
    }
    else if (srcCount == 0)
    {
        if (plainDst.getCount() > 0)
        {
            warnAboutPreallocatedDstArray(dst);
            return plainDst.setArraySize(0);
        }
    }
    return HK_SUCCESS;
}
namespace
{
    // Use the same allocation scheme as recursive clone, but don't follow pointers.
    struct ShallowCopyCallback : public hkReflect::Detail::CloneOnHeap
    {
        virtual hkResult atPointer(hkReflect::PointerVar& dst, hkReflect::PointerVar& src, _Out_ hkReflect::Cloner::PointerAction::Enum* action) HK_OVERRIDE
        {
            // Don't clone the pointer target.
            *action = hkReflect::Cloner::PointerAction::SKIP;
            // Share the pointer value.
            hkReflect::Var v;
            if (src.getValue(&v).isFailure()) { return HK_FAILURE; }
            return hkReflect::Detail::getPlain(dst).setValue(v);
        }
    };
}

hkReflect::Var hkReflect::Cloner::cloneOne(const Var& src, _In_opt_ AfterReflectNewHandler* afrnHandler, _In_opt_ hkArray<hkRefPtr<hkReferencedObject>>* salvageArray)
{
    ShallowCopyCallback clonerCallback;

    if (afrnHandler)
    {
        setAfterReflectNewHandler(afrnHandler);
    }
    if (salvageArray)
    {
        setDyingObjectsSalvageArray(salvageArray);
    }

    return cloneVar( src, clonerCallback );
}

hkReflect::Cloner::AfterReflectNewHandler::AfterReflectNewHandler()
{
}

hkReflect::Cloner::AfterReflectNewHandler::~AfterReflectNewHandler()
{
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
