// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : COMMON
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

namespace hkReflect
{
    namespace Detail
    {
        struct ArraySemantics;
    }
}

namespace hkReflect {

    /// Container for a generic reflected object, with value semantic.
    /// Used to store a value of any type. Internally stores a (value, type) pair.
    /// Uses an internal buffer for small types, and heap-allocated one otherwise.
    /// Owns the contained the object, and destroys it when deleted.
class HK_EXPORT_COMMON Any
{
    public:
        HK_DECLARE_CLASS(Any, New, Reflect);
        HK_REFLECT_AS_VARIANT_ARRAY(&hkReflect::Any::s_impl);

    public:

            /// Create a Any containing a default-constructed instance of the type.
            /// This cannot be used with Var or a class derived from Var; use fromVar() instead.
        template <typename T>
        static Any fromDefaultObj();

            /// Create a Any containing a copy of the given object.
            /// This cannot be used with Var or a class derived from Var; use fromVar() instead.
        template <typename T>
        static Any fromObj( const T& obj );

            /// Create a Any containing a copy of object the Var is pointing to.
        static Any fromVar( const Var& var );

            /// Advanced use. Create a Any containing a copy of another Any considered as object.
            /// This means the constructed Any will contain the input Any, which in turns
            /// can contain anything. So this is rarely what you want.
            /// This is equivalent to fromObj<Var>( const Var& ), but with an explicit
            /// separate helper to prevent any mistake.
        static Any asAnyOfAnyCopy( const Any& obj );

    public:

            /// Construct an empty Any.
        Any();

            /// Copy the content of another Any.
        Any(const Any& rhs);

            /// Copy the content of another Any.
        void operator=(const Any& rhs);

            /// Equality operator.
        bool operator==(const Any& rhs) const;

            /// Destroy the object stored in the Any.
        ~Any();

            /// Get a Var pointing to the contained object.
            /// Keep ownership of the contained object.
        Var var() const;

            /// Return true if the Any contains something.
        bool containsObject() const;

            /// Return true if the Any stores the value internally
        bool storesInplace() const;
            /// Set the contents to a default-constructed instance of the given type.
        void init(const QualType type);

            /// Set the content to a copy of the given object.
            /// This is equivalent to (*this) = Any::fromObj(obj).
            /// This cannot be used with Var or a class derived from Var; use setFromVar() instead.
        template <typename T>
        void setFromObj(const T& obj);

            /// Set the content to a copy of the content of the given Var.
            /// This is equivalent to (*this) = Any::fromVar(var).
        void setFromVar(const Var& var);

            /// Clear the content, destroying the contained object.
        void clear();

            /// Advanced use. Take ownership of the provided object without making a copy.
            /// The object will be deleted on destruction.
        void stealOwnership(const Var& var);

            /// Advanced use. Give away ownership of the contained object to the caller.
            /// Sets this Any to empty. If the object is stored externally, it is not deleted,
            /// if it is stored internally, a copy is made and the original is deleted.
        Var yieldOwnership();

            /// Advanced use. Steal ownership from another Any.
            /// Sets the other Any to empty.
            /// Ownership is stolen without making a copy if the object is stored externally,
            /// otherwise it is copied to the thief and deleted in the victim.
        void stealFrom(Any& victim);

            /// Advanced use. Override the type of the contained object.
            /// Will ASSERT if this Any's current type is not 'hkReflect::Type::equals' to 'newType'
            /// This method was motivated by a need to keep an Any's value the same, while replacing its type with an equivalent type that has different attributes
        void forceType(QualType newType);

            /// Advanced use. Allocate but do not construct an instance of the provided type.
            /// The instance must be manually constructed immediately after this method is called. Any other operations
            /// on the Any other than var() and deallocateOnly(), including clearing/destructing the Any, has undefined
            /// behavior.
        void allocateOnly(QualType type);

            /// Advanced use. Deallocate but do not destruct the contained object.
            /// Must be called after allocateOnly() if it is not possible to construct the object. Leaves this Any
            /// empty.
        void deallocateOnly();

    public:

        struct ArrayImpl;
        static const ArrayImpl s_impl;

        /// Specialize how anys are diffed.
        /// If "HK_BUILDING_WITH_ENGINE" is defined, this is implemented in a NewBase cpp.
        static _Ret_notnull_ const hkReflect::Detail::ArraySemantics* anyArraySemantics();
        HK_RECORD_ATTR(hk::DiffArraySemantics(anyArraySemantics));

    private:

        _Ret_notnull_ const void* addr() const;
        _Ret_notnull_ void* addr();

        template<typename T, typename AnyType>
        static T addr2(AnyType& a);

        void initInternal();
        void* allocateBuffer(const hkReflect::Var& rhs = hkReflect::Var());
        void* allocateBufferImpl(const hkReflect::Var& rhs);
        void setFromVarContent(const Var& rhs);
        void copyVarContent(const Var& rhs);
        bool canUseLocalBufferFor(const hkReflect::Type* type);
        void storeExternal(_Inout_ void* ptr, bool mustDel);
        bool mustDelete() const;

    private:

        enum Status { STATUS_EMPTY, STATUS_INPLACE, STATUS_EXTERNAL };
        QualType m_type;
        hkEnum<Status, hkUint8> m_status HK_ATTR(hk::Type(hkUint8), hk::Validate(false));
        // note: some systems use raw pointers instead of (ptr,type) pairs and
        // will break if the address of the buffer is the same as the address of the Any.
        HK_ALIGN_REAL(hkUintReal m_buf[4]);
};

} // hkReflect

#include <Common/Base/Reflect/Util/hkReflectAny.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
