// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Util/hkReflectAny.h>
#include <Common/Base/Reflect/Util/hkReflectClone.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>

hkReflect::Var hkReflect::Any::var() const
{
    return Var(addr(), m_type);
}

//
//  Equality operator

bool hkReflect::Any::operator==(const Any& rhs) const
{
    HK_ASSERT_NO_MSG(0x46023f3, false); // No two Anys can be equal
    return false;
}

bool hkReflect::Any::mustDelete() const
{
    return m_status == STATUS_EXTERNAL && reinterpret_cast<void* const*>(&m_buf)[1];
}

void hkReflect::Any::storeExternal(_Inout_ void* ptr, bool mustDel)
{
    HK_COMPILE_TIME_ASSERT(sizeof(m_buf) >= sizeof(void*)*2);

    hkMemUtil::memZeroOne(&m_buf);
    void** buf = reinterpret_cast<void**>(m_buf);
    buf[0] = ptr;
    buf[1] = (void*)mustDel;
    m_status = STATUS_EXTERNAL;
}

bool hkReflect::Any::canUseLocalBufferFor(_In_ const hkReflect::Type* type)
{
    return type->getSizeOf() <= hkSizeOf(m_buf)
        && type->getAlignOf() <= 16
        && TypeDetail::isInPlaceConstructible(type);
}

void* hkReflect::Any::allocateBufferImpl(const hkReflect::Var& rhs)
{
    void* addr;
    if (canUseLocalBufferFor(m_type))
    {
        addr = &m_buf[0];
        m_status = STATUS_INPLACE;
        hkMemUtil::memZeroOne(&m_buf);
    }
    else
    {
        if (rhs.isValid())
        {
            addr = TypeDetail::allocateForClone(rhs).getAddress();
        }
        else
        {
            addr = TypeDetail::allocate(m_type).getAddress();
        }
        if (addr)
        {
            storeExternal(addr, true);
            m_status = STATUS_EXTERNAL;
        }
    }
    return addr;
}

void* hkReflect::Any::allocateBuffer(const hkReflect::Var& rhs /* = hkReflect::Var() */)
{
    void* addr = allocateBufferImpl(rhs);
    HK_ASSERTV(0x96b9cb, addr, "Cannot allocate type {}. Ensure the type is allocatable "
        "(not ReflectIdentity).", rhs.getType() ? rhs.getType()->getFullName() : hkReflect::TypeName());
    return addr;
}

void hkReflect::Any::initInternal()
{
    HK_ASSERT_NO_MSG(0x7583426, m_type);

    // Use the underlying type if field.
    if (m_type->isField())
    {
        m_type = m_type->getParent();
    }

    void* addr = allocateBuffer();

    HK_ON_DEBUG(hkResult res = )TypeDetail::defaultConstruct(addr,m_type);
    HK_ASSERT_NO_MSG(0x4aa573a6, res.isSuccess()); // todo.nt4m fail explicitly?
}

void hkReflect::Any::copyVarContent(const hkReflect::Var& orignalRhs)
{
    HK_ASSERT_NO_MSG(0x37e32a03, addr());
    HK_ASSERT_NO_MSG(0x5bff97a1, m_type);
    HK_ASSERT_NO_MSG(0x28a303d2, orignalRhs);

    // Use the underlying type if field.
    if (m_type->isField())
    {
        m_type = m_type->getParent();
    }

    hkReflect::Var rhs = orignalRhs;
    if(rhs.getType()->isField() && !rhs.getType()->isProperty())
    {
        rhs = hkReflect::Var(rhs.getAddress(), rhs.getType()->getParent());
    }

    HK_ON_DEBUG(hkResult res = )Detail::copyObject(var(), rhs);
    HK_ASSERT_NO_MSG(0xe79e65c, res.isSuccess());
}

void hkReflect::Any::setFromVarContent(const Var& rhs)
{
    if (!rhs.isValid())
    {
        clear();
    }
    else if (!containsObject())
    {
        m_type = rhs.getType();
        allocateBuffer(rhs);
        copyVarContent(rhs);
    }
    else if (rhs.getAddress() == addr())
    {
        // if the address of 'rhs' matches our address, avoid clearing and copy constructing (which would destroy the value in 'rhs')
        // this only makes sense (for now, without clearer conversion behavior) if the types exactly match, so assert
        HK_ASSERT_NO_MSG(0x6b02fe6b, m_type == rhs.getType());
    }
    else
    {
        if (m_type->getSizeOf() != rhs.getType()->getSizeOf()
            || m_type->getAlignOf() < rhs.getType()->getAlignOf()
            || !TypeDetail::isInPlaceConstructible(rhs.getType())
            || !TypeDetail::isInPlaceConstructible(m_type)) 
        {
            // Free the current buffer and allocate a new one.
            clear();
            m_type = rhs.getType();
            allocateBuffer(rhs);
        }
        else
        {
            // Reuse the current buffer.
            TypeDetail::destruct(addr(), m_type);
            m_type = rhs.getType();
        }

        // Copy over the instance.
        copyVarContent(rhs);
    }
}

void hkReflect::Any::forceType(QualType newType)
{
    HK_ASSERT_NO_MSG(0x2256d794, m_type->equals(newType));
    HK_ASSERT_NO_MSG(0x6d70bcb9, !m_type->isField());
    m_type = newType;
}

void hkReflect::Any::clear()
{
    if (containsObject())
    {
        var().destroy(hkReflect::Var::FLAG_DESTRUCT);
    }
    deallocateOnly();
}

void hkReflect::Any::stealOwnership(const hkReflect::Var& var)
{
    HK_ASSERT_NO_MSG(0x7b21b88f, !var.getType()->isField());

    clear();

    if ( var )
    {
        m_type = var.getType();
        storeExternal( var.getAddress(), true ); // delete object when done (owned)
    }
}

hkReflect::Var hkReflect::Any::yieldOwnership()
{
    if (m_status == STATUS_EXTERNAL)
    {
        Var res = var();
        m_status = STATUS_EMPTY;
        m_type = nullptr;
        return res;
    }
    else if (m_status == STATUS_INPLACE)
    {
        // The object is in the internal buffer, must make a copy
        Var local = var();
        Var resVar = local.clone();
        HK_ASSERT_NO_MSG(0x27921732, resVar.isValid());
        TypeDetail::destruct(local.getAddress(), m_type);
        m_status = STATUS_EMPTY;
        m_type = nullptr;
        return resVar;
    }

    // empty
    return Var();
}

void hkReflect::Any::stealFrom(hkReflect::Any& victim)
{
    clear();

    if (victim.m_status == STATUS_EXTERNAL)
    {
        // Equivalent to stealOwnership(victim.yieldOwnership())
        Var victimVar = victim.var();
        victim.m_status = STATUS_EMPTY;
        victim.m_type = nullptr;
        stealOwnership(victimVar);
    }
    else if (victim.m_status == STATUS_INPLACE)
    {
        // Copy from victim internal buffer to this internal buffer
        Var victimVar = victim.var();
        m_type = victimVar.getType();
        m_status = STATUS_INPLACE;
        Detail::copyObject(var(), victimVar);
        TypeDetail::destruct( victimVar.getAddress(), m_type );
        victim.m_status = STATUS_EMPTY;
        victim.m_type = nullptr;
    }
}

void hkReflect::Any::allocateOnly( QualType type )
{
    clear();
    m_type = type;

    // Use the underlying type if field.
    if ( m_type->isField() )
    {
        m_type = m_type->getParent();
    }

    allocateBuffer();
}

void hkReflect::Any::deallocateOnly()
{
    if ( mustDelete() )
    {
        TypeDetail::deallocate(addr(), m_type);
    }
    m_status = STATUS_EMPTY;
    m_type = HK_NULL;
}

// used just for serialization

struct hkReflect::Any::ArrayImpl : public hkReflect::Detail::ArrayImpl
{
    ArrayImpl() {}
    virtual hkResult getValue(_In_ const void* arrAddr, _In_ const ArrayType* arrType, _Out_ ArrayValue* val) const HK_OVERRIDE
    {
        const Any* any = static_cast<const Any*>(arrAddr);
        const void* addr = any->addr();
        *val = ArrayValue(const_cast<void*>(addr), any->m_type ? 1 : 0, any->m_type);
    return HK_SUCCESS;
    }

    virtual hkReflect::Detail::ArrayImpl::AllocResult allocateElements(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, QualType elemType, int len) const HK_OVERRIDE
    {
        HK_ASSERT_NO_MSG(0x5bc97fe8, arrType->extendsOrEquals<Any>());

        Any* any = static_cast<Any*>(arrAddr);

        if (any->containsObject())
        {
            return ALLOC_NOT_EMPTY;
        }

        if (len == 0)
        {
            // do nothing
            return ALLOC_SUCCESS;
        }
        else if (len == 1)
        {
            any->m_type = elemType;

            if (void* addr = any->allocateBufferImpl(hkReflect::Var()))
            {
                // Cannot use TypeDetail::reflectConstruct because we don't want its hkRefObj hack.
                if (Detail::UnaryFunction func = TypeDetail::getReflectConstructionFunction(elemType))
                {
                    func(addr, elemType, 1);
                    return ALLOC_SUCCESS;
                }
            }
        }
        return ALLOC_FAILURE;
    }

    virtual hkResult setNumElements(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, int len) const HK_OVERRIDE
    {
        Any* any = static_cast<Any*>(arrAddr);
        if (len == 0)
        {
            // Clear the Any.
            any->clear();
            return HK_SUCCESS;
        }
        else if (len == 1 && any->containsObject())
        {
            // Do nothing.
            return HK_SUCCESS;
        }

        // All other cases are not allowed.
        return HK_FAILURE;
    }

    virtual hkResult spliceInto(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, int index, int numToDel, const ArrayValue& val) const HK_OVERRIDE
    {
        // Any is array of max size 1
        if (index != 0 || numToDel > 1 || val.getCount() > 1)
        {
            return HK_FAILURE;
        }

        Any* any = static_cast<Any*>(arrAddr);

        if (numToDel == 1 && !any->containsObject())
        {
            return HK_FAILURE;
        }

        if (val.getCount() == 1)
        {
            any->setFromVar( val[0] );
        }
        else if (numToDel == 1)
        {
            any->clear();
        }

        return HK_SUCCESS;
    }

    virtual hkResult inplaceFixup(_Inout_ void* self, _In_ const hkReflect::Type* selfType, _Inout_ void* target, _In_ const hkReflect::Type* targetType, int count) const HK_OVERRIDE
    {
        if ( target )
        {
            Any* any = static_cast<Any*>( self );
            any->storeExternal( target, false );
            any->m_status = STATUS_EXTERNAL;
            any->m_type = targetType;
        }
        return HK_SUCCESS;
    }

    virtual void clearAllocs(_Inout_ void* self, _In_ const ArrayType* type) const HK_OVERRIDE
    {
        Any* any = static_cast<Any*>(self);
        if (any->mustDelete())
        {
            // Var.destroy won't work if the constructor has not been called.
            TypeDetail::deallocate(any->addr(), any->m_type);
        }
    }
};

const hkReflect::Any::ArrayImpl hkReflect::Any::s_impl;

#ifndef HK_BUILDING_WITH_ENGINE
const hkReflect::Detail::ArraySemantics* hkReflect::Any::anyArraySemantics()
{
    return HK_NULL;
}
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
