// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Util/hkFieldSetterDetail.h>
#include <Common/Base/Reflect/Util/hkReflectAny.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>

namespace
{
    // Helper class for Impls of fields of a Record with a setter.
    struct RecordParentRef
    {
        typedef hkReflect::RecordVar VarType;

        HK_ALWAYS_INLINE RecordParentRef(_In_ const hkReflect::Detail::RecordImpl* parent, hkReflect::FieldDecl fieldDecl)
            : m_parentImpl(parent, true)
            , m_wrapped(fieldDecl)
        {
        }

        HK_INLINE hkReflect::RecordVar getParent(_In_ const void* addr) const
        {
            
            return hkReflect::RecordVar(hkAddByteOffset(addr, -m_wrapped.getOffset()),
                m_wrapped.getDeclContext()->asRecord(),
                static_cast<const hkReflect::Detail::RecordImpl*>(m_parentImpl.val()));
        }

        HK_INLINE hkReflect::Var getChild(const hkReflect::Var& newParent, _In_opt_ const void* origChild) const
        {
            return newParent[m_wrapped];
        }

        hkReflect::Detail::ImplPtr m_parentImpl;
        hkReflect::FieldDecl m_wrapped;
    };

    // Helper class for Impls of elements of an Array with a setter.
    struct ArrayParentRef
    {
        typedef hkReflect::ArrayVar VarType;

        HK_ALWAYS_INLINE ArrayParentRef(const hkReflect::ArrayVar& parent, _In_ const void* begin)
            : m_parent(parent)
            , m_begin(begin)
        {
        }

        HK_INLINE hkReflect::ArrayVar getParent(_In_opt_ const void* addr) const
        {
            
            return m_parent;
        }

        HK_INLINE hkReflect::Var getChild(const hkReflect::Var& newParent, _In_ const void* origChild) const
        {
            int index = hkLosslessCast<int>(hkGetByteOffset(m_begin, origChild) / m_parent.getSubType()->getSizeOf());
            return hkReflect::ArrayVar(newParent)[index];
        }

        hkReflect::ArrayVar m_parent;
        // Cached.
        const void* m_begin;
    };

    // Generic Impl for an element in a compound with a setter.
    template<typename MyType, typename ParentRef>
    struct ChildImpl : public hkReflect::ImplFromType<MyType>::Type
    {
        HK_DECLARE_CLASS(ChildImpl, New);

        
        HK_ALWAYS_INLINE ChildImpl(const ParentRef& pRef) : m_parentRef(pRef)
        {
            // These impls will be allocated dynamically so need to be managed through refcounting.
            this->m_refCount = 0;
        }

        typedef typename hkReflect::ImplFromType<MyType>::Type MyImpl;
        typedef typename hkReflect::ValueFromType<MyType>::Type MyValue;
        typedef typename hkTrait::If<sizeof(MyValue) <= 8, MyValue, const MyValue&>::Type MyValueParam;
        typedef typename hkReflect::VarFromType<MyType>::Type MyVar;
        typedef typename ParentRef::VarType ParentVar;

        virtual hkResult setValue(_In_ void* addr, _In_ const MyType* myType, MyValueParam value) const HK_OVERRIDE
        {
            // Makes a copy of the parent.
            ParentVar parentObj = m_parentRef.getParent(addr);
            hkReflect::Any copy = hkReflect::Any::fromVar(parentObj);

            MyVar fieldCopy = m_parentRef.getChild(copy.var(), addr);
            HK_RETURN_IF_FAILED(fieldCopy.setValue(value));
            HK_RETURN_IF_FAILED(parentObj.setValue(ParentVar(copy.var()).getValue()));
            return HK_SUCCESS;
        }

        virtual hkResult getValue(_In_ const void* addr, _In_ const MyType* myType, _Out_ MyValue* val) const HK_OVERRIDE
        {
            
            const MyImpl* baseImpl = myType->getImpl();
            HK_ASSERT(0xce8c940, baseImpl != this, "Infinite recursion");
            return baseImpl->getValue(addr, myType, val);
        }

        ParentRef m_parentRef;
    };

    // Impl for a Record element in a compound with a setter.
    template<typename ParentRef>
    struct ChildRecordImpl : ChildImpl<hkReflect::RecordType, ParentRef>
    {
        ChildRecordImpl(const ParentRef& pr) : ChildImpl<hkReflect::RecordType, ParentRef>(pr) {}

        // Definition below.
        virtual const hkReflect::Detail::Impl* getFieldImpl(const hkReflect::Var& parent, hkReflect::FieldDecl field) const HK_OVERRIDE;
    };

    // Impl for an Array element in a compound with a setter.
    template<typename ParentRef>
    struct ChildArrayImpl : ChildImpl<hkReflect::ArrayType, ParentRef>
    {
        using typename ChildImpl<hkReflect::ArrayType, ParentRef>::ParentVar;
        ChildArrayImpl(const ParentRef& pr) : ChildImpl<hkReflect::ArrayType, ParentRef>(pr) {}

        // Definition below.
        virtual hkResult getValue(_In_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const HK_OVERRIDE;

        virtual hkResult setNumElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int nelem) const HK_OVERRIDE
        {
            // Makes a copy of the parent.
            ParentVar parentObj = this->m_parentRef.getParent(arrAddr);
            hkReflect::Any copy = hkReflect::Any::fromVar(parentObj);

            hkReflect::ArrayVar fieldCopy = this->m_parentRef.getChild(copy.var(), arrAddr);
            HK_RETURN_IF_FAILED(fieldCopy.setArraySize(nelem));

            return parentObj.setValue(ParentVar(copy.var()).getValue());
        }

        virtual hkResult spliceInto(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, int index, int numToDel, const hkReflect::ArrayValue& val) const HK_OVERRIDE
        {
            // Makes a copy of the parent.
            ParentVar parentObj = this->m_parentRef.getParent(arrAddr);
            hkReflect::Any copy = hkReflect::Any::fromVar(parentObj);

            hkReflect::ArrayVar fieldCopy = this->m_parentRef.getChild(copy.var(), arrAddr);
            HK_RETURN_IF_FAILED(fieldCopy.spliceInto(index, numToDel, val));

            return parentObj.setValue(ParentVar(copy.var()).getValue());
        }

        virtual hkReflect::Detail::ArrayImpl::AllocResult allocateElements(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* arrType, hkReflect::QualType elemType, int n) const HK_OVERRIDE
        {
            HK_UNREACHABLE(0x7650ea3d, "This Impl should never be used for deserialization/cloning" );
#if defined(HK_COMPILER_GCC) && defined(HK_DEBUG)
            return hkReflect::Detail::ArrayImpl::ALLOC_FAILURE;
#endif
        }

        virtual void clearAllocs(_Inout_ void* arrAddr, _In_ const hkReflect::ArrayType* type) const HK_OVERRIDE
        {
            HK_UNREACHABLE(0x5da2cd4d, "This Impl should never be used for deserialization/cloning" );
        }
    };

    struct SetterRecordVisitor : hkReflect::TypeVisitor<SetterRecordVisitor, hkReflect::Detail::Impl*>
    {
        SetterRecordVisitor(const hkReflect::Detail::RecordImpl* tp) : thisPtr(tp) {}
        const hkReflect::Detail::RecordImpl* thisPtr;
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::VoidType*)
        {
            HK_UNREACHABLE( 0x147d7221, "Found void field" );
#if defined(HK_COMPILER_GCC) && defined(HK_DEBUG)
            return nullptr;
#endif
        }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::BoolType* t) { return new ChildImpl<hkReflect::BoolType, RecordParentRef>(RecordParentRef(thisPtr, t->isField())); }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::IntType* t) { return new ChildImpl<hkReflect::IntType, RecordParentRef>(RecordParentRef(thisPtr, t->isField())); }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::FloatType* t) { return new ChildImpl<hkReflect::FloatType, RecordParentRef>(RecordParentRef(thisPtr, t->isField())); }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::StringType* t) { return new ChildImpl<hkReflect::StringType, RecordParentRef>(RecordParentRef(thisPtr, t->isField())); }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::PointerType* t) { return new ChildImpl<hkReflect::PointerType, RecordParentRef>(RecordParentRef(thisPtr, t->isField())); }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::RecordType* t) { return new ChildRecordImpl<RecordParentRef>(RecordParentRef(thisPtr, t->isField())); }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::ArrayType* t) { return new ChildArrayImpl<RecordParentRef>(RecordParentRef(thisPtr, t->isField())); }
    };

    template<typename ParentRef>
    inline const hkReflect::Detail::Impl* ChildRecordImpl<ParentRef>::getFieldImpl( const hkReflect::Var& parent, hkReflect::FieldDecl field ) const
    {
        return SetterRecordVisitor(this).dispatch(field.getType());
    }

    struct SetterArrayVisitor : hkReflect::TypeVisitor<SetterArrayVisitor, hkReflect::Detail::Impl*>
    {
        SetterArrayVisitor(const hkReflect::ArrayVar& tv, const void* b) : thisVar(tv), begin(b) {}
        const hkReflect::ArrayVar& thisVar;
        const void* begin;
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::VoidType*)
        {
            HK_UNREACHABLE(0x3352f8ed, "Found void field");
#if defined(HK_COMPILER_GCC) && defined(HK_DEBUG)
            return nullptr;
#endif
        }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::BoolType* t) { return new ChildImpl<hkReflect::BoolType, ArrayParentRef>(ArrayParentRef(thisVar, begin)); }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::IntType* t) { return new ChildImpl<hkReflect::IntType, ArrayParentRef>(ArrayParentRef(thisVar, begin)); }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::FloatType* t) { return new ChildImpl<hkReflect::FloatType, ArrayParentRef>(ArrayParentRef(thisVar, begin)); }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::StringType* t) { return new ChildImpl<hkReflect::StringType, ArrayParentRef>(ArrayParentRef(thisVar, begin)); }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::PointerType* t) { return new ChildImpl<hkReflect::PointerType, ArrayParentRef>(ArrayParentRef(thisVar, begin)); }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::RecordType* t) { return new ChildRecordImpl<ArrayParentRef>(ArrayParentRef(thisVar, begin)); }
        HK_INLINE _Ret_notnull_ hkReflect::Detail::Impl* visit(_In_ const hkReflect::ArrayType* t) { return new ChildArrayImpl<ArrayParentRef>(ArrayParentRef(thisVar, begin)); ; }
    };

    template<typename ParentRef>
    hkResult ChildArrayImpl<ParentRef>::getValue(_In_ const void* arrAddr, _In_ const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const
    {
        const hkReflect::Detail::ArrayImpl* baseImpl = arrType->getImpl();
        HK_ASSERT(0xb571778, baseImpl != this, "Infinite recursion");
        hkReflect::ArrayValue base;
        baseImpl->getValue(arrAddr, arrType, &base);
        hkReflect::ArrayVar thisVar(arrAddr, arrType, this, true);
        *val = hkReflect::ArrayValue(base.getAddress(), base.getCount(), base.getSubType(),
            SetterArrayVisitor(thisVar, base.getAddress()).dispatch(base.getSubType()), true);
        return HK_SUCCESS;
    }

    template<typename MyType>
    HK_INLINE const typename hkReflect::ImplFromType<MyType>::Type* getBaseImpl(
        _In_ const MyType* type, _In_ const typename hkReflect::ImplFromType<MyType>::Type* current)
    {
        HK_ASSERT_NO_MSG(0x1ac56499, type->isField());
        HK_ASSERT_NO_MSG(0x1eb467ee, type->getImpl() == current);
        // Skip at least one impl or we will loop forever
        while (type && !hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::IMPL>(type))
        {
            const hkReflect::Type* parentType = type->getParent();
            HK_ASSERT_NO_MSG(0x5bd1f7a0, !parentType || (parentType->getKind() == hkReflect::KindFromType<MyType>::kind));
            type = reinterpret_cast<const MyType*>(parentType);
        }
        HK_ASSERT_NO_MSG(0x5bd1f7a1, type);
        const MyType* baseType = static_cast<const MyType*>(type->getParent());
        HK_ASSERT_NO_MSG(0x5bd1f7a2, baseType);
        HK_ASSERT_NO_MSG(0x5bd9055c, baseType->getKind() == hkReflect::KindFromType<MyType>::kind);
        return baseType->getImpl();
    }
}

template<typename MyType>
hkResult hkReflect::Detail::SetterValueImpl<MyType>::setValue(_Inout_ void* addr, _In_ const MyType* type, MyValueParam value) const
{
    
    Any temp; temp.init(type);
    MyVar other = temp.var();
    other.setValue(value);
    return m_setter(hkAddByteOffset(addr, -m_offset), other.getAddress()) ? HK_SUCCESS : HK_FAILURE;
}

template<typename MyType>
hkResult hkReflect::Detail::SetterValueImpl<MyType>::getValue(_In_ const void* addr, _In_ const MyType* type, _Out_ typename hkReflect::Detail::SetterValueImpl<MyType>::MyValue* val) const
{
    const typename hkReflect::ImplFromType<MyType>::Type* impl = getBaseImpl(type, this);
    HK_ASSERT(0x34283bf0, impl != this, "Infinite recursion");
    return impl->getValue(addr, type, val);
}

template<typename MyType>
hk::FieldSetter::Fn hkReflect::Detail::SetterValueImpl<MyType>::getSetter() const
{
    return m_setter;
}

template HK_EXPORT_COMMON struct hkReflect::Detail::SetterValueImpl<hkReflect::BoolType>;
template HK_EXPORT_COMMON struct hkReflect::Detail::SetterValueImpl<hkReflect::IntType>;
template HK_EXPORT_COMMON struct hkReflect::Detail::SetterValueImpl<hkReflect::FloatType>;
template HK_EXPORT_COMMON struct hkReflect::Detail::SetterValueImpl<hkReflect::StringType>;
template HK_EXPORT_COMMON struct hkReflect::Detail::SetterValueImpl<hkReflect::PointerType>;
template HK_EXPORT_COMMON struct hkReflect::Detail::SetterValueImpl<hkReflect::RecordType>;
template HK_EXPORT_COMMON struct hkReflect::Detail::SetterValueImpl<hkReflect::ArrayType>;

_Ret_maybenull_ const hkReflect::Detail::Impl* hkReflect::Detail::SetterRecordImpl::getFieldImpl(const Var& parent, hkReflect::FieldDecl field) const
{
    
    return SetterRecordVisitor(this).dispatch(field.getType());
}

hkResult hkReflect::Detail::SetterRecordImpl::setValue(_Inout_ void* addr, _In_ const RecordType* type, const RecordVar& other) const
{
    return m_setter(hkAddByteOffset(addr, -m_offset), other.getAddress()) ? HK_SUCCESS : HK_FAILURE;
};

hkResult hkReflect::Detail::SetterArrayImpl::getValue(_In_ const void* arrAddr, _In_ const ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const
{
    ArrayValue base;
    getBaseImpl(arrType, this)->getValue(arrAddr, arrType, &base);
    if (base.getCount() == 0)
    {
        *val = base;
        return HK_SUCCESS;
    }
    ArrayVar thisVar(arrAddr, arrType);
    *val = ArrayValue(base.getAddress(), base.getCount(), base.getSubType(),
        SetterArrayVisitor(thisVar, base.getAddress()).dispatch(base.getSubType()), true);
    return HK_SUCCESS;
}

hkResult hkReflect::Detail::SetterArrayImpl::setValue(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, const ArrayValue& val) const
{
    // Optimization, could go through spliceInto but that would do redundant copies.
    void* parentAddress = hkAddByteOffset(arrAddr, -m_offset);
    if (arrType->getFixedCount() != 0)
    {
        // Run setter on buffer directly
        return m_setter(parentAddress, val.getAddress()) ? HK_SUCCESS : HK_FAILURE;
    }
    else
    {
        Any temp; temp.init(arrType);

        // Creating an Any should resolve the type to the plain version.
        HK_ASSERT(0x125b12f4, temp.var().getType()->getImpl() != this, "Plain impl is the same as current impl, infinite recursion");

        ArrayVar(temp.var()).setValue(val);
        return m_setter(parentAddress, temp.var().getAddress()) ? HK_SUCCESS : HK_FAILURE;
    }
}

hkResult hkReflect::Detail::SetterArrayImpl::setNumElements(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, int nelem) const
{
    Any copy = Any::fromVar( Var(arrAddr, arrType) );
    getBaseImpl(arrType, this)->setNumElements(copy.var().getAddress(), arrType, nelem);
    void* parentAddress = hkAddByteOffset(arrAddr, -m_offset);
    return m_setter(parentAddress, copy.var().getAddress()) ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkReflect::Detail::SetterArrayImpl::spliceInto(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, int index, int numToDel, const ArrayValue& val) const
{
    Any copy = Any::fromVar( Var(arrAddr, arrType) );
    getBaseImpl(arrType, this)->spliceInto(copy.var().getAddress(), arrType, index, numToDel, val);
    void* parentAddress = hkAddByteOffset(arrAddr, -m_offset);
    return m_setter(parentAddress, copy.var().getAddress()) ? HK_SUCCESS : HK_FAILURE;
}

hkReflect::Detail::ArrayImpl::AllocResult hkReflect::Detail::SetterArrayImpl::allocateElements(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, QualType elemType, int n) const
{
    HK_UNREACHABLE(0x6f5f6963, "Deserialization/cloning should call allocateElements on the base Impl, never on this" );
#if defined(HK_COMPILER_GCC) && defined(HK_DEBUG)
    return hkReflect::Detail::ArrayImpl::ALLOC_FAILURE;
#endif
}

void hkReflect::Detail::SetterArrayImpl::clearAllocs(_Inout_ void* arrAddr, _In_ const ArrayType* arrType) const
{
    HK_ASSERT_NO_MSG(0x7c2af2d7, arrType->isField() );
    const ArrayType* baseType = arrType->getParent()->asArray();
    HK_ASSERT_NO_MSG(0x2cc84d5e, baseType );
    baseType->getImpl()->clearAllocs( arrAddr, baseType );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
