// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL !REFLECT
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0


#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Types/hkEndian.h>

namespace hkString
{
    HK_EXPORT_COMMON void HK_CALL memCpy(_Out_writes_bytes_all_(n) void* dst, _In_reads_bytes_(n) const void* src, _In_range_(>, 0) int n);
}

#if defined(DEBUG_INSTRUCTION)
#undef DEBUG_INSTRUCTION
#endif

//#define DEBUG_DISPLAY_INSTRUCTIONS
#if defined(DEBUG_DISPLAY_INSTRUCTIONS)
#include <stdio.h>
#include <string.h>
#define DEBUG_INSTRUCTION(x) printf(#x "\n"); s_instructionCounter.count(hkTypeVm::Program::x)
#else
#define DEBUG_INSTRUCTION(x) /*Detail::s_instructionCounter.count(hkTypeVm::Program::x)*/
#endif

#if defined(HK_DEBUG_SLOW)
#define DEBUG_SIZEOF(TYPE) ((TYPE)->isProperty() ? 0 : (TYPE)->getSizeOf())
#else
#define DEBUG_SIZEOF(TYPE) 0
#endif


#define HK_INTERPRETER_RETURN_IF_FAILED(COND) \
    while ( ( COND ).isFailure() ) { return HK_FAILURE; }

namespace hkTypeVm
{
    namespace Detail
    {
        struct InstructionExecCounter
        {
            hkUint64 m_instructionCounts[hkTypeVm::Program::NUM_INSTRUCTIONS];

            void reset()
            {
                for(int i = 0; i < hkTypeVm::Program::NUM_INSTRUCTIONS; i++) { m_instructionCounts[i] = 0; }
            }

            void printAll(int numSteps)
            {
#if defined(DEBUG_DISPLAY_INSTRUCTIONS)
                double totalCount = 0;
                for(int i = 0; i < hkTypeVm::Program::NUM_INSTRUCTIONS; i++) { totalCount += double(m_instructionCounts[i]); }
                hkStringBuf b;

                b.printf("%lf instructions, counts:\n", totalCount / numSteps);
                printf("%s", b.cString());
                //OutputDebugStringA(b.cString());

#define PRINT_INSTRUCTION(INST) b.printf(#INST ": %llu(%lf)\n", m_instructionCounts[int(hkTypeVm::Program::INST)] / numSteps, double(m_instructionCounts[int(hkTypeVm::Program::INST)]) / totalCount); printf("%s", b.cString()); /*OutputDebugStringA(b.cString())*/
                PRINT_INSTRUCTION(INST_ARRAY);
                PRINT_INSTRUCTION(INST_POINTER);
                PRINT_INSTRUCTION(INST_STRING);
                PRINT_INSTRUCTION(INST_BLOCK);
                PRINT_INSTRUCTION(INST_CONVERT_INTEGER);
                PRINT_INSTRUCTION(INST_CONVERT_FLOAT);
                PRINT_INSTRUCTION(INST_RECORD);
#undef PRINT_INSTRUCTION
                printf("-----\n");
                //OutputDebugStringA("-----\n");
#endif
            }

            void count(int instr)
            {
                m_instructionCounts[instr]++;
            }
        };

        extern struct InstructionExecCounter s_instructionCounter;
    }
}

namespace hkTypeVm
{
    namespace Detail
    {
        HK_INLINE void byteSwapN(_Out_writes_bytes_all_(numBytes) void* dst, _In_reads_(numBytes) const void* src, _In_range_(1, 8) int numBytes)
        {
            switch(numBytes)
            {
                case 1:
                    *reinterpret_cast<hkUint8*>(dst) = *reinterpret_cast<const hkUint8*>(src);
                    break;
                case 2:
                    *reinterpret_cast<hkUint16*>(dst) = hkEndian::swap(*reinterpret_cast<const hkUint16*>(src));
                    break;
                case 4:
                    *reinterpret_cast<hkUint32*>(dst) = hkEndian::swap(*reinterpret_cast<const hkUint32*>(src));
                    break;
                case 8:
                    *reinterpret_cast<hkUint64*>(dst) = hkEndian::swap(*reinterpret_cast<const hkUint64*>(src));
                    break;
                default:
                    HK_ASSERT_NO_MSG(0x4decf79d, 0); // Invalid numBytes
            }
        }

        HK_INLINE void byteSwapRepeated(_Out_writes_bytes_all_(dstStride * repeat) void* dst, _In_reads_bytes_(srcStride * repeat) const void* src,
            _In_range_(1, 8) int numBytes, _In_range_(>=, 0) int repeat, _In_range_(>=, numBytes) int dstStride, _In_range_(>= , numBytes) int srcStride)
        {
            switch(numBytes)
            {
                case 1:
                    for(int i = 0; i < repeat; i++, src = hkAddByteOffsetConst(src, srcStride), dst = hkAddByteOffset(dst, dstStride))
                    {
                        *reinterpret_cast<hkUint8*>(dst) = *reinterpret_cast<const hkUint8*>(src);
                    }
                    break;
                case 2:
                    for(int i = 0; i < repeat; i++, src = hkAddByteOffsetConst(src, srcStride), dst = hkAddByteOffset(dst, dstStride))
                    {
                        *reinterpret_cast<hkUint16*>(dst) = hkEndian::swap(*reinterpret_cast<const hkUint16*>(src));
                    }
                    break;
                case 4:
                    for(int i = 0; i < repeat; i++, src = hkAddByteOffsetConst(src, srcStride), dst = hkAddByteOffset(dst, dstStride))
                    {
                        *reinterpret_cast<hkUint32*>(dst) = hkEndian::swap(*reinterpret_cast<const hkUint32*>(src));
                    }
                    break;
                case 8:
                    for(int i = 0; i < repeat; i++, src = hkAddByteOffsetConst(src, srcStride), dst = hkAddByteOffset(dst, dstStride))
                    {
                        *reinterpret_cast<hkUint64*>(dst) = hkEndian::swap(*reinterpret_cast<const hkUint64*>(src));
                    }
                    break;
                default:
                    HK_ASSERT_NO_MSG(0x72248286, 0); // Invalid numBytes
            }
        }
    }

    HK_INLINE void FastCopyInterpreter::execCopy(_Out_writes_bytes_(dstStride * repeat) void* HK_RESTRICT vdst, _In_reads_bytes_(srcStride * repeat) const void* HK_RESTRICT vsrc, _In_range_(>=, 0) int numBytes, _In_range_(>=, 1) int dstStride, _In_range_(>=, 1) int srcStride, _In_range_(>=, 0) int repeat)
    {
        char* HK_RESTRICT dst = static_cast<char*>(vdst);
        const char* HK_RESTRICT src = static_cast<const char*>(vsrc);

        if(numBytes == dstStride && numBytes == srcStride)
        {
            hkString::memCpy(vdst, vsrc, repeat*numBytes);
        }
        else
        {
            for(int r = 0; r < repeat; ++r)
            {
                for(int i = 0; i < numBytes; i++) { dst[i] = src[i]; }

                dst += dstStride;
                src += srcStride;
            }
        }
    }

    HK_INLINE void FastCopyInterpreter::execCopyBlock(_Out_writes_bytes_all_(numBytes * repeat) void* dst, _In_reads_bytes_(numBytes * repeat) const void* src, _In_range_(>= , 0) int numBytes, _In_range_(>= , 0) int repeat) // numBytes is expected to be large here
    {
        hkString::memCpy(dst, src, numBytes * repeat);
    }

    HK_INLINE hkResult FastCopyInterpreter::execPointer(hkReflect::PointerVar dst, hkReflect::PointerVar src, int dstStride, int srcStride, int repeat)
    {
        HK_ASSERT_NOT_IMPLEMENTED(0x2ecb65d7 );
        return HK_FAILURE;
    }

    HK_INLINE void FastCopyInterpreter::execConvertInteger(_Out_writes_bytes_(dstStride * repeat) void* dst, _In_reads_bytes_(srcStride * repeat) const void* src, hkReflect::Format::Value dFormat, hkReflect::Format::Value sFormat, int dstStride, int srcStride, _In_range_(>= , 0) int repeat)
    {
        hkReflect::Format::IntValue srcFmt(sFormat);
        hkReflect::Format::IntValue dstFmt(dFormat);

        const int srcNumBytes = srcFmt.sizeOf();
        const int dstNumBytes = dstFmt.sizeOf();

        bool needsSwap = srcFmt.isBigEndian() != dstFmt.isBigEndian();

        const int bytesDiff = dstNumBytes - srcNumBytes;
        if(bytesDiff <= 0)
        {
            // truncating
            const void* srcStart = src;

            if(srcFmt.isBigEndian())
            {
                // copy only the last source bytes
                srcStart = hkAddByteOffsetConst(src, -bytesDiff);
            }

            if(needsSwap)
            {
                for(int i = 0; i < repeat; i++)
                {
                    Detail::byteSwapN( hkAddByteOffset(dst, dstStride * i), hkAddByteOffset(srcStart, srcStride * i), dstNumBytes);
                }
            }
            else
            {
                execCopy(dst, srcStart, dstNumBytes, dstStride, srcStride, repeat);
            }
        }
        else
        {
            hkLong dstOffset = 0;
            hkLong signByteOffset = srcNumBytes - 1;
            hkLong firstExtendedByteOffset = 1;

            if(dstFmt.isBigEndian())
            {
                // copy all the source bytes at the end of the destination
                dstOffset = bytesDiff;
                signByteOffset = srcNumBytes;
                firstExtendedByteOffset = -bytesDiff;
            }

            for(int i = 0; i < repeat; i++, dst = hkAddByteOffset(dst, dstStride), src = hkAddByteOffsetConst(src, srcStride))
            {
                void* dstStart = hkAddByteOffset(dst, dstOffset);
                char* signByte = reinterpret_cast<char*>(hkAddByteOffset(dst, signByteOffset));
                char* firstExtendByte = signByte + firstExtendedByteOffset;

                // extending - copy all the source bytes
                if(needsSwap)
                {
                    Detail::byteSwapN(dstStart, src, srcNumBytes);
                }
                else
                {
                    execCopy(dstStart, src, srcNumBytes, 0, 0, 1);
                }

                // fills the remaining bytes
                if(srcFmt.isSigned())
                {
                    const int valToFill = (*signByte & 0x80 ? 0xff : 0x00);
                    hkString::memSet(firstExtendByte, valToFill, bytesDiff);
                }
            }
        }
    }

    HK_INLINE void FastCopyInterpreter::execConvertFloat(_Out_writes_bytes_(dstStride * repeat) void* dst, _In_reads_bytes_(srcStride * repeat) const void* srcPtr,
        hkReflect::Format::Value dstFormat, hkReflect::Format::Value srcFormat, int dstStride, int srcStride, _In_range_(0, MAX_ARRAY_CHUNK_SIZE / srcStride) int repeat)
    {
        hkReflect::Format::FloatValue srcFmt(srcFormat);
        hkReflect::Format::FloatValue dstFmt(dstFormat);

        hkDouble64 byteSwapBuffer[MAX_ARRAY_CHUNK_SIZE];

        const char* srcValueLocation = HK_NULL;
        int srcBufferStride = 0;

        // Fast path!
        // In order to do a conversion, we need to be in native endianness
        const bool srcNeedsSwap = srcFmt.isBigEndian() != HK_ENDIAN_BIG;
        srcValueLocation = srcNeedsSwap ? reinterpret_cast<const char*>(byteSwapBuffer) : reinterpret_cast<const char*>(srcPtr);
        srcBufferStride = srcNeedsSwap ? hkSizeOf(hkDouble64) : srcStride;

        if(srcNeedsSwap)
        {
            const unsigned srcNumBytes = srcFmt.sizeOf();

            Detail::byteSwapRepeated(&byteSwapBuffer[0], srcPtr, srcNumBytes, repeat, hkSizeOf(byteSwapBuffer[0]), srcStride);
            srcFmt.setBigEndian(HK_ENDIAN_BIG);
        }

        // The source is now native endianness

        // This must be either hkHalf16, hkFloat32 or hkDouble64
        const bool dstNeedsSwap = dstFmt.isBigEndian() != HK_ENDIAN_BIG;
        char* dstValueLocation = dstNeedsSwap ? reinterpret_cast<char*>(byteSwapBuffer) : reinterpret_cast<char*>(dst);
        int dstBufferStride = dstNeedsSwap ? hkSizeOf(hkDouble64) : dstStride;

        // In order to do a conversion, we need to be in native endianness
        hkReflect::Format::FloatValue dstFmtUnswapped = dstFmt;
        dstFmtUnswapped.setBigEndian(HK_ENDIAN_BIG);

        switch(srcFmt.get())
        {
            case hkReflect::Format::OfFloat<hkHalf16>::Value:
            {
                switch(dstFmtUnswapped.get())
                {
                    case hkReflect::Format::OfFloat<hkHalf16>::Value:
                    {
                        for(int i = 0; i < repeat; i++, srcValueLocation = hkAddByteOffsetConst(srcValueLocation, srcBufferStride), dstValueLocation = hkAddByteOffset(dstValueLocation, dstBufferStride))
                        {
                            *reinterpret_cast<hkHalf16*>(dstValueLocation) = *reinterpret_cast<const hkHalf16*>(srcValueLocation);
                        }
                    }; break;
                    case hkReflect::Format::OfFloat<hkFloat32>::Value:
                    {
                        for(int i = 0; i < repeat; i++, srcValueLocation = hkAddByteOffsetConst(srcValueLocation, srcBufferStride), dstValueLocation = hkAddByteOffset(dstValueLocation, dstBufferStride))
                        {
                            *reinterpret_cast<hkFloat32*>(dstValueLocation) = reinterpret_cast<const hkHalf16*>(srcValueLocation)->getFloat32();
                        }
                    }; break;
                    case hkReflect::Format::OfFloat<hkDouble64>::Value:
                    {
                        for(int i = 0; i < repeat; i++, srcValueLocation = hkAddByteOffsetConst(srcValueLocation, srcBufferStride), dstValueLocation = hkAddByteOffset(dstValueLocation, dstBufferStride))
                        {
                            *reinterpret_cast<hkDouble64*>(dstValueLocation) = reinterpret_cast<const hkHalf16*>(srcValueLocation)->getDouble();
                        }
                    }; break;
                    default: HK_ASSERT_NO_MSG(0xd7a9a78, 0); // Unhandled conversion
                }
            }
            break;
            case hkReflect::Format::OfFloat<hkFloat32>::Value:
            {
                switch(dstFmtUnswapped.get())
                {
                    case hkReflect::Format::OfFloat<hkHalf16>::Value:
                    {
                        for(int i = 0; i < repeat; i++, srcValueLocation = hkAddByteOffsetConst(srcValueLocation, srcBufferStride), dstValueLocation = hkAddByteOffset(dstValueLocation, dstBufferStride))
                        {
                            reinterpret_cast<hkHalf16*>(dstValueLocation)->setReal<true>(hkReal(*reinterpret_cast<const hkFloat32*>(srcValueLocation)));
                        }
                    }; break;
                    case hkReflect::Format::OfFloat<hkFloat32>::Value:
                    {
                        for(int i = 0; i < repeat; i++, srcValueLocation = hkAddByteOffsetConst(srcValueLocation, srcBufferStride), dstValueLocation = hkAddByteOffset(dstValueLocation, dstBufferStride))
                        {
                            *reinterpret_cast<hkFloat32*>(dstValueLocation) = *reinterpret_cast<const hkFloat32*>(srcValueLocation);
                        }
                    }; break;
                    case hkReflect::Format::OfFloat<hkDouble64>::Value:
                    {
                        for(int i = 0; i < repeat; i++, srcValueLocation = hkAddByteOffsetConst(srcValueLocation, srcBufferStride), dstValueLocation = hkAddByteOffset(dstValueLocation, dstBufferStride))
                        {
                            *reinterpret_cast<hkDouble64*>(dstValueLocation) = *reinterpret_cast<const hkFloat32*>(srcValueLocation);
                        }
                    }; break;
                    default: HK_ASSERT_NO_MSG(0x3625366c, 0); // Unhandled conversion
                }
            }
            break;
            case hkReflect::Format::OfFloat<hkDouble64>::Value:
            {
                switch(dstFmtUnswapped.get())
                {
                    case hkReflect::Format::OfFloat<hkHalf16>::Value:
                    {
                        for(int i = 0; i < repeat; i++, srcValueLocation = hkAddByteOffsetConst(srcValueLocation, srcBufferStride), dstValueLocation = hkAddByteOffset(dstValueLocation, dstBufferStride))
                        {
                            reinterpret_cast<hkHalf16*>(dstValueLocation)->setReal<true>(hkReal(*reinterpret_cast<const hkDouble64*>(srcValueLocation)));
                        }
                    }; break;
                    case hkReflect::Format::OfFloat<hkFloat32>::Value:
                    {
                        for(int i = 0; i < repeat; i++, srcValueLocation = hkAddByteOffsetConst(srcValueLocation, srcBufferStride), dstValueLocation = hkAddByteOffset(dstValueLocation, dstBufferStride))
                        {
                            *reinterpret_cast<hkFloat32*>(dstValueLocation) = hkFloat32(*reinterpret_cast<const hkDouble64*>(srcValueLocation));
                        }
                    }; break;
                    case hkReflect::Format::OfFloat<hkDouble64>::Value:
                    {
                        for(int i = 0; i < repeat; i++, srcValueLocation = hkAddByteOffsetConst(srcValueLocation, srcBufferStride), dstValueLocation = hkAddByteOffset(dstValueLocation, dstBufferStride))
                        {
                            *reinterpret_cast<hkDouble64*>(dstValueLocation) = *reinterpret_cast<const hkDouble64*>(srcValueLocation);
                        }
                    }; break;
                    default: HK_ASSERT_NO_MSG(0x6f3cd4b1, 0); // Unhandled conversion
                }
            }
            break;
        }

        if(dstNeedsSwap)
        {
            Detail::byteSwapRepeated(dst, &byteSwapBuffer[0], dstFmt.sizeOf(), repeat, dstStride, hkSizeOf(byteSwapBuffer[0]));
        }
    }

    HK_INLINE void hkTypeVm::FastCopyInterpreter::execConvertBool(_Out_writes_bytes_(dstStride * numElems) void* dst, _In_reads_bytes_(srcStride * numElems) const void* src, const hkReflect::Type* dstType, const hkReflect::Type* srcType, const int dstStride, const int srcStride, _In_range_(>= , 0) const int numElems)
    {
        HK_ASSERT(0xbbd9018, srcType->asInteger() || srcType->asBool(), "Can only convert between int and bool" );
        HK_ASSERT(0x4f98b1f1, dstType->asInteger() || dstType->asBool(), "Can only convert between int and bool" );
        const int srcBoolSize = srcType->getSizeOf();
        const int dstBoolSize = dstType->getSizeOf();
        for(int i = 0; i < numElems; i++, dst = hkAddByteOffset(dst, dstStride), src = hkAddByteOffsetConst(src, srcStride))
        {
            bool srcBoolValue = false;
            switch ( srcBoolSize )
            {
                HK_NO_DEFAULT_CASE(0x34cf970a, "Unsupported bool type (> 8 bytes)" );
            case 1:
                srcBoolValue = bool( *reinterpret_cast<const hkUint8*>( src ) );
                break;

            case 2:
                srcBoolValue = bool( *reinterpret_cast<const hkUint16*>( src ) );
                break;

            case 4:
                srcBoolValue = bool( *reinterpret_cast<const hkUint32*>( src ) );
                break;

            case 8:
                srcBoolValue = bool( *reinterpret_cast<const hkUint64*>( src ) );
                break;
            }

            switch ( dstBoolSize )
            {
                HK_NO_DEFAULT_CASE( 0x5f1bc119, "Unsupported bool type (> 8 bytes)" );
            case 1:
                *reinterpret_cast<hkUint8*>( dst ) = srcBoolValue;
                break;

            case 2:
                *reinterpret_cast<hkUint16*>( dst ) = srcBoolValue;
                break;

            case 4:
                *reinterpret_cast<hkUint32*>( dst ) = srcBoolValue;
                break;

            case 8:
                *reinterpret_cast<hkUint64*>( dst ) = srcBoolValue;
                break;
            }
        }
    }

    HK_INLINE bool FastCopyInterpreter::shouldHandleString(hkReflect::StringVar dst, hkReflect::StringVar src)
    {
        // Default is to add all strings
        return true;
    }

    HK_INLINE hkResult FastCopyInterpreter::execString(hkReflect::StringVar dst, hkReflect::StringVar src)
    {
        return dst.setValue(src.getValue());
    }

    HK_INLINE void FastCopyInterpreter::beginArrayOfArrays( int numElements )
    {
    }

    HK_INLINE hkResult FastCopyInterpreter::execArray(hkReflect::ArrayVar dst, hkReflect::ArrayVar src)
    {
        HK_ASSERT_NOT_IMPLEMENTED( 0x3d485f34 );
        return HK_FAILURE;
    }

    HK_INLINE hkResult FastCopyInterpreter::execBool(const hkReflect::BoolVar& dst, const hkReflect::BoolVar& src)
    {
        return dst.setValue(src.getValue());
    }
    HK_INLINE hkResult FastCopyInterpreter::execInt(const hkReflect::IntVar& dst, const hkReflect::IntVar& src)
    {
        return dst.setValue(src.getValue());
    }
    HK_INLINE hkResult FastCopyInterpreter::execFloat(const hkReflect::FloatVar& dst, const hkReflect::FloatVar& src)
    {
        return dst.setValue(src.getValue());
    }
}

namespace hkTypeVm
{
    namespace Detail
    {
        struct StringTodo
        {
            hkReflect::StringVar m_dst;
            hkReflect::StringVar m_src;
            void set(hkReflect::StringVar const& d, hkReflect::StringVar const& s) { m_dst = d; m_src = s; }
        };
    }
}

template<typename Interp>
hkResult hkTypeVm::FastCopyInterpreter::execN(Interp& interp, _In_ const Program* program, _Out_writes_bytes_(dstSize) void* dstBase, int dstSize, _In_reads_bytes_(srcSize) const void* srcBase, int srcSize, int dstStride, int srcStride, int totalNumElems)
{
    HK_ASSERT_NO_MSG(0x69160086, totalNumElems);
    hkInplaceArray<Detail::StringTodo, 64> cstringTodos;

    int elemsOffset = 0;

    while(elemsOffset < totalNumElems)
    {
        const int dstBytesAlreadyConsumed = elemsOffset * dstStride;
        const int srcBytesAlreadyConsumed = elemsOffset * srcStride;

        FastCopyInterpreter::DestBlock dstPointer(hkAddByteOffset(dstBase, dstBytesAlreadyConsumed), dstSize - dstBytesAlreadyConsumed);
        FastCopyInterpreter::SourceBlock srcPointer(hkAddByteOffsetConst(srcBase, srcBytesAlreadyConsumed), srcSize - srcBytesAlreadyConsumed);

        const int numElems = ((totalNumElems - elemsOffset) > Interp::MAX_ARRAY_CHUNK_SIZE) ? Interp::MAX_ARRAY_CHUNK_SIZE : (totalNumElems - elemsOffset);
        Program::ConstIterator it = program->begin();
        for(; it != program->end(); ++it)
        {
            const hkTypeVm::Program::InstructionBase& baseInstr = *it;
            FastCopyInterpreter::DestBlock loopDst = dstPointer.offset(baseInstr.getDstOffset());
            FastCopyInterpreter::SourceBlock loopSrc = srcPointer.offset(baseInstr.getSrcOffset());

            const int srcLoopBytesConsumed = (numElems >= 1) ? ((numElems - 1) * srcStride) : 0;
            const int dstLoopBytesConsumed = (numElems >= 1) ? ((numElems - 1) * dstStride) : 0;

            switch(baseInstr.getInstructionType())
            {
                case hkTypeVm::Program::INST_ARRAY: // Array begin -> array execution mode
                {
                    DEBUG_INSTRUCTION(INST_ARRAY);

                    const Program::ArrayInstruction& instr = static_cast<const Program::ArrayInstruction&>(baseInstr);

                    // Add todo for this array
                    hkReflect::ArrayVar dstArr = instr.getDstVarAs<hkReflect::ArrayVar>(loopDst.get(numElems * DEBUG_SIZEOF(instr.getDstType())));
                    hkReflect::ArrayVar srcArr = instr.getSrcVarAs<hkReflect::ArrayVar>(loopSrc.get(numElems * DEBUG_SIZEOF(instr.getSrcType())));
                    interp.beginArrayOfArrays( numElems );
                    for(int i = 0; i < numElems; i++)
                    {
                        HK_INTERPRETER_RETURN_IF_FAILED( interp.execArray( dstArr, srcArr ) );
                        dstArr.setAddress(hkAddByteOffset(dstArr.getAddress(), dstStride));
                        srcArr.setAddress(hkAddByteOffset(srcArr.getAddress(), srcStride));
                    }
                    break;
                }

                case hkTypeVm::Program::INST_BOOL:
                {
                    DEBUG_INSTRUCTION(INST_BOOL);

                    const Program::DefaultInstruction& instr = static_cast<const Program::DefaultInstruction&>(baseInstr);

                    hkReflect::BoolVar dst = instr.getDstVarAs<hkReflect::BoolVar>(loopDst.get(numElems * DEBUG_SIZEOF(instr.getDstType())));
                    hkReflect::BoolVar src = instr.getSrcVarAs<hkReflect::BoolVar>(loopSrc.get(numElems * DEBUG_SIZEOF(instr.getSrcType())));
                    for(int i = 0; i < numElems; i++)
                    {
                        HK_INTERPRETER_RETURN_IF_FAILED( interp.execBool( dst, src ) );
                        dst.setAddress(hkAddByteOffset(dst.getAddress(), dstStride));
                        src.setAddress(hkAddByteOffset(src.getAddress(), srcStride));
                    }
                    break;
                }

                case hkTypeVm::Program::INST_INT:
                {
                    DEBUG_INSTRUCTION(INST_INT);

                    const Program::DefaultInstruction& instr = static_cast<const Program::DefaultInstruction&>(baseInstr);

                    hkReflect::IntVar dst = instr.getDstVarAs<hkReflect::IntVar>(loopDst.get(numElems * DEBUG_SIZEOF(instr.getDstType())));
                    hkReflect::IntVar src = instr.getSrcVarAs<hkReflect::IntVar>(loopSrc.get(numElems * DEBUG_SIZEOF(instr.getSrcType())));
                    for(int i = 0; i < numElems; i++)
                    {
                        HK_INTERPRETER_RETURN_IF_FAILED( interp.execInt( dst, src ) );
                        dst.setAddress(hkAddByteOffset(dst.getAddress(), dstStride));
                        src.setAddress(hkAddByteOffset(src.getAddress(), srcStride));
                    }
                    break;
                }

                case hkTypeVm::Program::INST_FLOAT:
                {
                    DEBUG_INSTRUCTION(INST_FLOAT);

                    const Program::DefaultInstruction& instr = static_cast<const Program::DefaultInstruction&>(baseInstr);

                    hkReflect::FloatVar dst = instr.getDstVarAs<hkReflect::FloatVar>(loopDst.get(numElems * DEBUG_SIZEOF(instr.getDstType())));
                    hkReflect::FloatVar src = instr.getSrcVarAs<hkReflect::FloatVar>(loopSrc.get(numElems * DEBUG_SIZEOF(instr.getSrcType())));
                    for(int i = 0; i < numElems; i++)
                    {
                        HK_INTERPRETER_RETURN_IF_FAILED( interp.execFloat( dst, src ) );
                        dst.setAddress(hkAddByteOffset(dst.getAddress(), dstStride));
                        src.setAddress(hkAddByteOffset(src.getAddress(), srcStride));
                    }
                    break;
                }

                case hkTypeVm::Program::INST_CONVERT_INTEGER:
                {
                    DEBUG_INSTRUCTION(INST_CONVERT_INTEGER);

                    const Program::DefaultInstruction& instr = static_cast<const Program::DefaultInstruction&>(baseInstr);

                    const hkReflect::Format::Value dstFormat = instr.getDstTypeAs<hkReflect::IntType>()->getFormat();
                    const hkReflect::Format::Value srcFormat = instr.getSrcTypeAs<hkReflect::IntType>()->getFormat();

                    interp.execConvertInteger(loopDst.get(dstLoopBytesConsumed + hkReflect::Format::IntValue(dstFormat).sizeOf())
                                                , loopSrc.get(srcLoopBytesConsumed + hkReflect::Format::IntValue(srcFormat).sizeOf())
                                                , dstFormat, srcFormat, dstStride, srcStride, numElems);
                    break;
                }

                case hkTypeVm::Program::INST_CONVERT_FLOAT:
                {
                    DEBUG_INSTRUCTION(INST_CONVERT_FLOAT);

                    const Program::DefaultInstruction& instr = static_cast<const Program::DefaultInstruction&>(baseInstr);

                    const hkReflect::Format::Value dstFormat = instr.getDstTypeAs<hkReflect::FloatType>()->getFormat();
                    const hkReflect::Format::Value srcFormat = instr.getSrcTypeAs<hkReflect::FloatType>()->getFormat();

                    interp.execConvertFloat(loopDst.get(dstLoopBytesConsumed + hkReflect::Format::FloatValue(dstFormat).sizeOf()),
                                            loopSrc.get(srcLoopBytesConsumed + hkReflect::Format::FloatValue(srcFormat).sizeOf())
                                            , dstFormat, srcFormat, dstStride, srcStride, numElems);
                    break;
                }

                case hkTypeVm::Program::INST_CONVERT_BOOL:
                {
                    DEBUG_INSTRUCTION(INST_CONVERT_BOOL);

                    const Program::DefaultInstruction& instr = static_cast<const Program::DefaultInstruction&>(baseInstr);

                    interp.execConvertBool(loopDst.get(dstLoopBytesConsumed + instr.getDstType()->getSizeOf())
                                            , loopSrc.get(srcLoopBytesConsumed + instr.getSrcType()->getSizeOf())
                                            , instr.getDstType(), instr.getSrcType(), dstStride, srcStride, numElems);
                    break;
                }

                case hkTypeVm::Program::INST_BLOCK:
                {
                    DEBUG_INSTRUCTION(INST_BLOCK);

                    const Program::BlockInstruction& instr = static_cast<const Program::BlockInstruction&>(baseInstr);

                    interp.execCopy(loopDst.get(dstLoopBytesConsumed + instr.getNumBytes()), loopSrc.get(srcLoopBytesConsumed + instr.getNumBytes()), instr.getNumBytes(), dstStride, srcStride, numElems);
                    break;
                }

                case hkTypeVm::Program::INST_POINTER:
                {
                    DEBUG_INSTRUCTION(INST_POINTER);

                    const Program::DefaultInstruction& instr = static_cast<const Program::DefaultInstruction&>(baseInstr);

                    hkReflect::PointerVar dstp = instr.getDstVarAs<hkReflect::PointerVar>(loopDst.get(dstLoopBytesConsumed + DEBUG_SIZEOF(instr.getDstType())));
                    hkReflect::PointerVar srcp = instr.getSrcVarAs<hkReflect::PointerVar>(loopSrc.get(srcLoopBytesConsumed + DEBUG_SIZEOF(instr.getSrcType())));
                    HK_INTERPRETER_RETURN_IF_FAILED( interp.execPointer( dstp, srcp, dstStride, srcStride, numElems ) );
                    break;
                }

                case hkTypeVm::Program::INST_STRING:
                {
                    DEBUG_INSTRUCTION(INST_STRING);

                    const Program::DefaultInstruction& instr = static_cast<const Program::DefaultInstruction&>(baseInstr);

                    cstringTodos.reserve( cstringTodos.getSize() + numElems );
                    for(int i = 0; i < numElems; i++)
                    {
                        hkReflect::StringVar dstVar = instr.getDstVarAs<hkReflect::StringVar>(loopDst.get(DEBUG_SIZEOF(instr.getDstType()), i*dstStride));
                        hkReflect::StringVar srcVar = instr.getSrcVarAs<hkReflect::StringVar>(loopSrc.get(DEBUG_SIZEOF(instr.getSrcType()), i*srcStride));
                        if(interp.shouldHandleString(dstVar, srcVar))
                        {
                            cstringTodos.expandOne().set(dstVar, srcVar);
                        }
                    }
                    break;
                }

                case hkTypeVm::Program::INST_RECORD:
                {
                    DEBUG_INSTRUCTION(INST_CALL);

                    const Program::RecordInstruction& instr = static_cast<const Program::RecordInstruction&>(baseInstr);

                    Interp recursiveInterp;
                    HK_INTERPRETER_RETURN_IF_FAILED(
                        Interp::exec1( recursiveInterp, instr.getProgram(),
                            dstPointer.get( instr.getDstType()->getSizeOf() ), instr.getDstType()->getSizeOf(),
                            srcPointer.get( instr.getSrcType()->getSizeOf() ), instr.getSrcType()->getSizeOf() ) );

                    break;
                }

                default:
                    HK_ASSERT_NO_MSG(0x412978cf, 0); break;
            }
        }

        // Move to the next chunk.
        elemsOffset += numElems;
    }

    // Strings are out of the loop so that object sizes are known in advance
    for(int i = 0; i < cstringTodos.getSize(); i++)
    {
        Detail::StringTodo& todo = cstringTodos[i];
        HK_INTERPRETER_RETURN_IF_FAILED( interp.execString( todo.m_dst, todo.m_src ) );
    }
    return HK_SUCCESS;
}

#undef DEBUG_INSTRUCTION
#undef HK_INTERPRETER_RETURN_IF_FAILED

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
