// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Reflect/TypeVm/hkTypeVmCompiler.h>
#include <Common/Base/Container/Array/hkFixedInplaceArray.h>

namespace hkTypeVm
{
    /// Adds the default set of passes for efficient cloning.
    HK_EXPORT_COMMON void HK_CALL addDefaultPasses(hkTypeVm::Compiler& compiler);

    class HK_EXPORT_COMMON BlockPass : public Pass
    {
        public:

            HK_DECLARE_CLASS(BlockPass, New);

            virtual hkResult apply(Program& program, hkReflect::QualType srcType, hkReflect::QualType dstType) HK_OVERRIDE;
    };

    /// Inlines arrays with a fixed number of elements.
    class HK_EXPORT_COMMON InlineFixedArrayPass : public Pass
    {
        public:

            HK_DECLARE_CLASS(InlineFixedArrayPass, New);

            virtual hkResult apply(Program& program, hkReflect::QualType srcType, hkReflect::QualType dstType) HK_OVERRIDE;
    };

    /// Inlines embedded record.
    /// By default it will inline all the record instructions which have a program but you can use the option
    /// to avoid inline instructions for which the src type is not a record.
    /// Instructions which contain user data are never inlined.
    class HK_EXPORT_COMMON InlineRecordPass : public Pass
    {
        public:

            HK_DECLARE_CLASS(InlineRecordPass, New);

            InlineRecordPass() : m_preserveInstrIfSrcIsNotRecord(false) {}
            InlineRecordPass(bool preserveInstrIfSrcIsNotRecord) : m_preserveInstrIfSrcIsNotRecord(preserveInstrIfSrcIsNotRecord) {}

            virtual hkResult apply(Program& program, hkReflect::QualType srcType, hkReflect::QualType dstType) HK_OVERRIDE;

        private:

            bool m_preserveInstrIfSrcIsNotRecord;
    };

    /// Filters programs to keep only a subset of instructions. By default no
    /// instructions are allowed so it will filter out the entire program.
    class HK_EXPORT_COMMON InstructionFilterPass : public Pass
    {
        public:

            HK_DECLARE_CLASS(InstructionFilterPass, New);

            /// Default ctor: no instruction types are allowed.
            InstructionFilterPass() {}

            /// Construct a filter allowing instructions of instrType type.
            InstructionFilterPass(hkTypeVm::Program::InstructionType instrType);

            virtual hkResult apply(Program& program, hkReflect::QualType srcType, hkReflect::QualType dstType) HK_OVERRIDE;

            /// Add instrType to the list of allowed instructions.
            void allow(hkTypeVm::Program::InstructionType instrType);

        private:

            bool isInstrAllowed(const hkTypeVm::Program::InstructionBase& instr) const;

            typedef hkFixedInplaceArray<hkTypeVm::Program::InstructionType, hkTypeVm::Program::NUM_INSTRUCTIONS> InstrTypeArray;

            InstrTypeArray m_allowedInstr;
    };

    /// Filters programs to remove instructions which have a type with specific attributes on them.
    class HK_EXPORT_COMMON AttributeFilterPass : public Pass
    {
        public:

            HK_DECLARE_CLASS(AttributeFilterPass, New);

            /// Default ctor: all types are allowed (no filtering).
            AttributeFilterPass() {}

            /// Construct a filter forbidding instructions which have this attribute on their type.
            AttributeFilterPass(_In_ const hkReflect::Type* attrType) { forbid(attrType); }

            virtual hkResult apply(Program& program, hkReflect::QualType srcType, hkReflect::QualType dstType) HK_OVERRIDE;

            /// Adds attrType to the list of forbidden attributes.
            void forbid(_In_ const hkReflect::Type* attrType);

            /// Adds T to the list of forbidden attributes.
            template<typename T>
            void forbid() { forbid(hkReflect::getType<T>()); }

        private:

            bool isInstrAllowed(const hkTypeVm::Program::InstructionBase& instr) const;

            hkInplaceArray<const hkReflect::Type*, 4> m_forbiddenAttributes;
    };

    /// Replaces any INST_INT/FLOAT requiring a conversion with a corresponding
    /// INST_CONVERT_INT/FLOAT.
    class HK_EXPORT_COMMON IntAndFloatConversionPass : public Pass
    {
        public:

            HK_DECLARE_CLASS(IntAndFloatConversionPass, New);

            virtual hkResult apply(Program& program, hkReflect::QualType srcType, hkReflect::QualType dstType) HK_OVERRIDE;
    };

    /// Pass that will check that the srcType and dstType on instructions have the same kind and trigger a compilation
    /// error if they don't.
    class HK_EXPORT_COMMON CheckTypeKindsPass : public Pass
    {
        public:

            HK_DECLARE_CLASS(CheckTypeKindsPass, New);

            virtual hkResult apply(Program& program, hkReflect::QualType srcType, hkReflect::QualType dstType) HK_OVERRIDE;
    };

    /// Filter pass that removes any instruction from the program that doesn't have a hkRefPtr in either
    /// its source or destination type.
    class HK_EXPORT_COMMON RefPtrFilterPass : public Pass
    {
        public:
            HK_DECLARE_CLASS(RefPtrFilterPass, New);

            RefPtrFilterPass();

            virtual hkResult apply(Program& program, hkReflect::QualType srcType, hkReflect::QualType dstType) HK_OVERRIDE;

        private:
            bool isInstrAllowed(const hkTypeVm::Program::InstructionBase& instr) const;
            bool isRefPtrType(const hkReflect::Type* type) const;

        private:
            hkStringPtr m_refPtrTypeName;
    };
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
