// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompilerPasses.h>

hkLog::RegisteredOrigin hkTypeVm::origin("reflect.TypeVm");

void HK_CALL hkTypeVm::addDefaultPasses(Compiler& compiler)
{
    compiler.addPass(new InlineFixedArrayPass);
    compiler.addPass(new InlineRecordPass);
    compiler.addPass(new BlockPass);
}

hkResult hkTypeVm::BlockPass::apply(Program& program, hkReflect::QualType, hkReflect::QualType)
{
    // Replace single instructions by INST_BLOCK whenever applicable.
    for(Program::Iterator it = program.begin(); it != program.end(); ++it)
    {
        switch(it->getInstructionType())
        {
            case Program::INST_BOOL:
            case Program::INST_INT:
            case Program::INST_FLOAT:
            {
                Program::DefaultInstruction& instr = static_cast<Program::DefaultInstruction&>(*it);

                if((instr.getDstType()->isProperty() == false) && instr.getSrcType() && (instr.getSrcType()->isProperty() == false) && (instr.getUserData() == 0))
                {
                    const hkReflect::Type* srcType = instr.getSrcType();
                    const hkReflect::Type* dstType = instr.getDstType();

                    const int srcSizeof = srcType->getSizeOf();
                    const int dstSizeof = dstType->getSizeOf();

                    if(dstSizeof == srcSizeof)
                    {
                        if(dstType->getFormat() == srcType->getFormat() ||
                           
                           
                           
                           (srcType->asInteger() && dstType->asBool())
                           )
                        {
                            Program::BlockInstruction blockInstr(instr.getDstOffset(), instr.getSrcOffset(), dstSizeof);
                            program.replace(it, it + 1, blockInstr);
                        }
                    }
                }
            }

            default:
                /*nop*/;
        }
    }

    // Attempt to merge INST_BLOCK instructions.
    for(Program::Iterator it = program.begin(); it != program.end(); ++it)
    {
        if(it->getInstructionType() == Program::INST_BLOCK)
        {
            Program::BlockInstruction& curBlock = static_cast<Program::BlockInstruction&>(*it);

            Program::Iterator it2 = it + 1;
            while(it2 != program.end() && it2->getInstructionType() == Program::INST_BLOCK)
            {
                const Program::BlockInstruction& nextBlock = static_cast<const Program::BlockInstruction&>(*it2);

                // If the two memory blocks are adjacent on the source and destination side we can safely merge them.
                if(curBlock.getSrcOffset() + curBlock.getNumBytes() == nextBlock.getSrcOffset() &&
                   curBlock.getDstOffset() + curBlock.getNumBytes() == nextBlock.getDstOffset())
                {
                    curBlock.setNumBytes(curBlock.getNumBytes() + nextBlock.getNumBytes());
                    program.remove(it2); // After this it2 will either point to the new instruction or the end of the program.
                }
                else
                {
                    break;
                }
            }
        }
    }

    return HK_SUCCESS;
}

hkResult hkTypeVm::InlineFixedArrayPass::apply(Program& program, hkReflect::QualType, hkReflect::QualType)
{
    for(Program::Iterator it = program.begin(); it != program.end(); ++it)
    {
        if(it->getInstructionType() == Program::INST_ARRAY)
        {
            // Copy the ArrayInstruction because edits to the program will invalidate/move it.
            const hkTypeVm::Program::ArrayInstruction instr = static_cast<const hkTypeVm::Program::ArrayInstruction&>(*it);
            if(instr.getProgram() && instr.getSrcType()->asArray() && (instr.getUserData() == 0))
            {
                const hkReflect::ArrayType* srcType = instr.getSrcTypeAs<hkReflect::ArrayType>();
                const hkReflect::ArrayType* dstType = instr.getDstTypeAs<hkReflect::ArrayType>();

                const int srcFixedSize = (srcType && srcType->isProperty()==false) ? srcType->getFixedCount() : 0;
                const int dstFixedSize = (dstType->isProperty()==false) ? dstType->getFixedCount() : 0;
                if ((srcFixedSize > 0) && (dstFixedSize > 0))
                {
                    if(const Program* elemProgram = instr.getProgram())
                    {
                        // Check if the element program has instructions to inline.
                        if (elemProgram->isEmpty() == false)
                        {
                            const int srcElemSize = srcType ? srcType->getSubType()->getSizeOf() : 0;
                            const int dstElemSize = dstType->getSubType()->getSizeOf();

                            const int numToCopy = hkMath::min2(srcFixedSize, dstFixedSize);

                            const int elemProgramCount = elemProgram->end() - elemProgram->begin();
                            for(int i = 0; i < numToCopy; ++i)
                            {
                                // Inline the instructions of the element type program into the main program.
                                const Program::Iterator newIt = it + i * elemProgramCount;
                                program.insertAfter(newIt, elemProgram->begin(), elemProgram->end());

                                // Fix the offsets of those instructions (which are relative to the array's start) to
                                // be relative to the program's start.
                                const int srcArrayOffset = instr.getSrcOffset() + i * srcElemSize;
                                const int dstArrayOffset = instr.getDstOffset() + i * dstElemSize;
                                for(int j = 0; j < elemProgramCount; ++j)
                                {
                                    Program::InstructionBase& elemInstr = *(newIt + 1 + j);

                                    elemInstr.setSrcOffset(elemInstr.getSrcOffset() + srcArrayOffset);
                                    elemInstr.setDstOffset(elemInstr.getDstOffset() + dstArrayOffset);
                                }
                            }
                        }

                        // Remove the INST_ARRAY
                        program.remove(it);
                        --it;
                    }
                }
            }
        }
    }

    return HK_SUCCESS;
}

hkResult hkTypeVm::InlineRecordPass::apply(Program& program, hkReflect::QualType, hkReflect::QualType)
{
    for (Program::Iterator it = program.begin(); it != program.end(); ++it)
    {
        if (it->getInstructionType() == Program::INST_RECORD)
        {
            // Copy the instruction because edits to the program will invalidate/move it.
            const hkTypeVm::Program::RecordInstruction instr = static_cast<const hkTypeVm::Program::RecordInstruction&>(*it);
            if (instr.getUserData() == 0 && (m_preserveInstrIfSrcIsNotRecord == false || instr.getSrcType()->asRecord()))
            {
                if(const Program* elemProgram = instr.getProgram())
                {
                    // Check if the embedded program has instructions to inline.
                    if (elemProgram->isEmpty() == false)
                    {
                        const int elemProgramCount = elemProgram->end() - elemProgram->begin();

                        // Inline the instructions of the element type program into the main program.
                        const Program::Iterator newIt = it;
                        program.insertAfter(newIt, elemProgram->begin(), elemProgram->end());

                        // Fix the offsets of those instructions (which are relative to the record start) to
                        // be relative to the program start.
                        const int srcArrayOffset = instr.getSrcOffset();
                        const int dstArrayOffset = instr.getDstOffset();
                        for (int j = 0; j < elemProgramCount; ++j)
                        {
                            Program::InstructionBase& elemInstr = *(newIt + 1 + j);

                            elemInstr.setSrcOffset(elemInstr.getSrcOffset() + srcArrayOffset);
                            elemInstr.setDstOffset(elemInstr.getDstOffset() + dstArrayOffset);
                        }
                    }

                    // Remove the INST_RECORD
                    program.remove(it);
                    --it;
                }
            }
        }
    }

    return HK_SUCCESS;
}

hkTypeVm::InstructionFilterPass::InstructionFilterPass(hkTypeVm::Program::InstructionType instrType)
{
    allow(instrType);
}

hkResult hkTypeVm::InstructionFilterPass::apply(Program& program, hkReflect::QualType, hkReflect::QualType)
{
    for(Program::Iterator it = program.begin(); it != program.end(); ++it)
    {
        Program::Iterator jt = it;

        while(jt != program.end() && isInstrAllowed(*jt) == false)
        {
            ++jt;
        }

        program.remove(it, jt);

        if(it == program.end())
        {
            break;
        }
    }

    return HK_SUCCESS;
}

void hkTypeVm::InstructionFilterPass::allow(hkTypeVm::Program::InstructionType instrType)
{
    if(m_allowedInstr.indexOf(instrType) == -1)
    {
        m_allowedInstr.pushBack(instrType);
    }
}

bool hkTypeVm::InstructionFilterPass::isInstrAllowed(const Program::InstructionBase& instr) const
{
    if(instr.getInstructionType() == Program::INST_ARRAY || instr.getInstructionType() == Program::INST_RECORD)
    {
        const Program::ProgramInstruction& progInstr = static_cast<const Program::ProgramInstruction&>(instr);

        if(hkViewPtr<const Program> eltProg = progInstr.getProgram())
        {
            // If the element program is not empty then it means it contains allowed instructions and
            // therefore we want to allow the array/record instr itself as well.
            return eltProg->isEmpty() == false;
        }
        else
        {
            // If the element type is null it means it is dynamically typed and we need to accept the instruction
            // by default.
            return true;
        }
    }
    else
    {
        return m_allowedInstr.indexOf(instr.getInstructionType()) != -1;
    }
}

hkResult hkTypeVm::AttributeFilterPass::apply(hkTypeVm::Program& program, hkReflect::QualType, hkReflect::QualType)
{
    for(hkTypeVm::Program::Iterator it = program.begin(); it != program.end(); ++it)
    {
        hkTypeVm::Program::Iterator jt = it;

        while(jt != program.end() && isInstrAllowed(*jt) == false)
        {
            ++jt;
        }

        program.remove(it, jt);

        if(it == program.end())
        {
            break;
        }
    }

    return HK_SUCCESS;
}

void hkTypeVm::AttributeFilterPass::forbid(_In_ const hkReflect::Type* type)
{
    m_forbiddenAttributes.pushBack(type);
}

bool hkTypeVm::AttributeFilterPass::isInstrAllowed(const hkTypeVm::Program::InstructionBase& instr) const
{
    if(const hkTypeVm::Program::DefaultInstruction* defInstr = hkTypeVm::Program::asDefaultInstruction(instr))
    {
        for(int i = 0; i < m_forbiddenAttributes.getSize(); ++i)
        {
            if(defInstr->getSrcType()->findAttribute(m_forbiddenAttributes[i]))
            {
                return false;
            }
        }

        if(defInstr->getSrcType()->equals(defInstr->getDstType()) == false)
        {
            for(int i = 0; i < m_forbiddenAttributes.getSize(); ++i)
            {
                if(defInstr->getDstType()->findAttribute(m_forbiddenAttributes[i]))
                {
                    return false;
                }
            }
        }
    }

    return true;
}

namespace
{
    bool isKnownFloat( const hkReflect::FloatType* type )
    {
        hkReflect::Format::FloatValue format = type->getFormat();
        if ( (int)format.sizeOf() == type->getSizeOf() )
        {
            if ( ( format.get() == hkReflect::Format::OfFloat<hkFloat32>::Value ) ||
                ( format.get() == hkReflect::Format::OfFloat<hkDouble64>::Value ) ||
                ( format.get() == hkReflect::Format::OfFloat<hkHalf16>::Value ) )
            {
                return true;
            }
            hkUint32 formatSwapped = format.getSwapEndianness();
            if ( ( formatSwapped == hkReflect::Format::OfFloat<hkFloat32>::Value ) ||
                ( formatSwapped == hkReflect::Format::OfFloat<hkDouble64>::Value ) ||
                ( formatSwapped == hkReflect::Format::OfFloat<hkHalf16>::Value ) )
            {
                return true;
            }
        }
        return false;
    }

    bool isKnownIntOrBool( const hkReflect::Type* type )
    {
        hkReflect::Format::Value format = type->getFormat();
        if ( format.getKind() == hkReflect::KIND_INT || format.getKind() == hkReflect::KIND_BOOL )
        {
            hkReflect::Format::IntValue iFormat = format;
            return iFormat.numBits() % 8 == 0 && iFormat.sizeOf() >= 1 && iFormat.sizeOf() <= 8 &&
                (int)iFormat.sizeOf() == type->getSizeOf();
        }
        return false;
    }
}

hkResult hkTypeVm::IntAndFloatConversionPass::apply(Program& program, hkReflect::QualType srcType, hkReflect::QualType dstType)
{
    // No conversion needed if the src and dst types are the same or if there is no srcType.
    if(srcType.get() == HK_NULL || srcType == dstType)
    {
        return HK_SUCCESS;
    }

    // Leave the generic INT/FLOAT/BOOL instruction if either src or dst are properties.
    if ( srcType->isProperty() || dstType->isProperty() )
    {
        return HK_SUCCESS;
    }

    for(hkTypeVm::Program::Iterator it = program.begin(); it != program.end(); ++it)
    {
        if ( const hkTypeVm::Program::DefaultInstruction* instrPtr = it->isDefaultInstruction() )
        {
            auto& instr = const_cast<hkTypeVm::Program::DefaultInstruction&>( *instrPtr );

            const hkReflect::Format::Value dstFormat = instr.getDstType()->getFormat();
            const hkReflect::Format::Value srcFormat = instr.getSrcType()->getFormat();

            if ( srcFormat == dstFormat )
            {
                continue; // formats same, no conversion needed
            }
            const auto srcKind = srcFormat.getKind();

            switch ( it->getInstructionType() )
            {
                default:
                    break;
                case Program::INST_FLOAT:
                {
                    if ( srcKind == hkReflect::KIND_FLOAT &&
                        isKnownFloat( instr.getSrcTypeAs<hkReflect::FloatType>() ) &&
                        isKnownFloat( instr.getDstTypeAs<hkReflect::FloatType>() ) )
                    {
                        instr.set( Program::INST_CONVERT_FLOAT, instr.getDstOffset(), instr.getDstType(), instr.getSrcOffset(), instr.getSrcType() );
                    }
                    break;
                }
                case Program::INST_INT:
                {
                    if ( isKnownIntOrBool( instr.getSrcType() ) && isKnownIntOrBool( instr.getDstType() ) )
                    {
                        if ( srcKind == hkReflect::KIND_BOOL )
                        {
                            instr.set( Program::INST_CONVERT_BOOL, instr.getDstOffset(), instr.getDstType(), instr.getSrcOffset(), instr.getSrcType() );
                        }
                        else
                        {
                            HK_ASSERT_NO_MSG(0x3aa2299d, srcKind == hkReflect::KIND_INT );
                            // If the formats are different switch the instruction type to INST_CONVERT_FLOAT
                            instr.set( Program::INST_CONVERT_INTEGER, instr.getDstOffset(), instr.getDstType(), instr.getSrcOffset(), instr.getSrcType() );
                        }
                    }
                    break;
                }
                case Program::INST_BOOL:
                {
                    if ( isKnownIntOrBool( instr.getSrcType() ) && isKnownIntOrBool( instr.getDstType() ) )
                    {
                        instr.set( Program::INST_CONVERT_BOOL, instr.getDstOffset(), instr.getDstType(), instr.getSrcOffset(), instr.getSrcType() );
                    }
                    break;
                }
            }
        }
    }

    return HK_SUCCESS;
}

hkResult hkTypeVm::CheckTypeKindsPass::apply(Program& program, hkReflect::QualType srcType, hkReflect::QualType dstType)
{
    for(Program::Iterator it = program.begin(); it != program.end(); ++it)
    {
        if(const Program::DefaultInstruction* instr = it->isDefaultInstruction())
        {
            const hkReflect::Type* instrSrcType = instr->getSrcType();
            const hkReflect::Type* instrDstType = instr->getDstType();

            if(instr->getSrcType() && instrSrcType->getKind() != instrDstType->getKind())
            {
                hkLog_Warning(origin, "Cannot convert {} (from {}) to {} (from {})", instrSrcType->getFullName(), hkReflect::FieldDecl(instrSrcType), instrDstType->getFullName(), hkReflect::FieldDecl(instrDstType));
                return HK_FAILURE;
            }
        }
    }

    return HK_SUCCESS;
}


hkTypeVm::RefPtrFilterPass::RefPtrFilterPass()
{
    // The type name of a template instance type is just the template type name, so we can
    // use any instance of the hkRefPtr to get the name we need.
    m_refPtrTypeName = hkReflect::getType< hkRefPtr<hkReferencedObject> >()->getName();
}

hkResult hkTypeVm::RefPtrFilterPass::apply(hkTypeVm::Program& program, hkReflect::QualType, hkReflect::QualType)
{
    for (hkTypeVm::Program::Iterator it = program.begin(); it != program.end(); ++it)
    {
        hkTypeVm::Program::Iterator jt = it;

        while (jt != program.end() && isInstrAllowed(*jt) == false)
        {
            ++jt;
        }

        program.remove(it, jt);

        if (it == program.end())
        {
            break;
        }
    }

    return HK_SUCCESS;
}

bool hkTypeVm::RefPtrFilterPass::isInstrAllowed(const hkTypeVm::Program::InstructionBase& instr) const
{
    if (const hkTypeVm::Program::DefaultInstruction* defInstr = hkTypeVm::Program::asDefaultInstruction(instr))
    {
        if (const hkReflect::Type* srcType = defInstr->getSrcType())
        {
            if (isRefPtrType(srcType))
            {
                return true;
            }
        }

        if (const hkReflect::Type* dstType = defInstr->getDstType())
        {
            if (isRefPtrType(dstType))
            {
                return true;
            }
        }
    }

    return false;
}

bool hkTypeVm::RefPtrFilterPass::isRefPtrType(_In_ const hkReflect::Type* type) const
{
    while (type && type->isDecorator())
    {
        type = type->getParent();
    }

    return (type && (m_refPtrTypeName == type->getName()));
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
