// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkScopedPtr.h>
#include <Common/Base/Container/PointerMap/hkMap.h>
#include <Common/Base/System/Log/hkLog.h>

namespace hkTypeVm
{
    extern hkLog::RegisteredOrigin origin;

    class HK_EXPORT_COMMON Program
    {
        public:

            HK_DECLARE_CLASS(Program, New);

            enum InstructionType
            {
                INST_NONE,

                INST_ARRAY,
                INST_BOOL,
                //INST_CONTAINER,
                INST_INT,
                INST_FLOAT,
                INST_POINTER,
                INST_RECORD,
                INST_STRING,

                INST_BLOCK, // typeless mem copiable block (may contain several coalesced types)

                INST_CONVERT_INTEGER, // convert between 2 integer types
                INST_CONVERT_FLOAT, // convert between 2 float types
                INST_CONVERT_BOOL, // convert between 2 bool types

                //INST_WRITE_CONSTANT,

                //BEGIN_RECORD,
                //END_RECORD,

                //BEGIN_ARRAY,
                //END_ARRAY,

                NUM_INSTRUCTIONS
            };

            class InstructionBase;
                class LargestInstruction;

                class DefaultInstruction;
                    class ProgramInstruction;
                        class RecordInstruction;
                        class ArrayInstruction;

                class BlockInstruction;
                //class WriteConstantInstruction;
                //class BeginEndRecordInstruction;

            // If you add an instruction type that inherits from DefaultInstruction please fix this function.
            static _Ret_maybenull_ const DefaultInstruction* asDefaultInstruction(const InstructionBase& instr);
            static _Ret_maybenull_ DefaultInstruction* asDefaultInstruction(InstructionBase& instr);


            template<typename P>
            class IteratorImpl;

            typedef IteratorImpl<Program> Iterator;
            typedef IteratorImpl<const Program> ConstIterator;

            HK_DECLARE_HANDLE(Label, hkUint16, hkUint16(-1));

            /// Creates a new instruction of InstrType type, append it to the program and
            /// returns a reference to it.
            template<typename InstrType>
            InstrType& addInstr();

            /// Appends the instruction to the program and returns a reference to the new instruction.
            template<typename InstrType>
            InstrType& addInstr(const InstrType& instr);

            Iterator begin();
            Iterator end();

            ConstIterator begin() const;
            ConstIterator end() const;

            /// Returns true if the program contains no instructions.
            bool isEmpty() const;

            /// Replaces the instructions in the range [b,e) by the instruction instr.
            template<typename InstrType>
            void replace(Iterator b, Iterator e, const InstrType& instr);

            /// Inserts the instruction instr after the iterator it. After the call "it" is still
            /// valid and still points to the same instruction as before the call.
            template<typename InstrType>
            void insertAfter(Iterator it, const InstrType& instr);

            /// Inserts the range [b, e) after the iterator it. After the call "it" is still
            /// valid and still points to the same instruction as before the call.
            void insertAfter(Iterator it, ConstIterator b, ConstIterator e);

            /// Removes the instruction referenced by the iterator. After the call "it" references
            /// the next instruction or is the end() iterator if there are none left.
            void remove(ConstIterator it);

            /// Removes the range [b, e). After the call "b" and "e" will reference the same
            /// element (including end() if e was end()).
            void remove(ConstIterator b, ConstIterator e);

        private:

            hkArray<LargestInstruction> m_code;
    };

    /// Program iterators are designed to be a bit more robust than your average iterator.
    /// The iterator is really an index so it will resist insertions/removals as long as
    /// that index is still valid in the program.
    /// On the other hand rather than pointing to an element it points to a slot in the
    /// program. So if an insertion happens in that slot the iterator will reference the
    /// new element instead of still pointing to the old one.
    template<typename P>
    class Program::IteratorImpl
    {
        public:

            HK_DECLARE_CLASS(IteratorImpl, New);

            typedef typename hkTrait::ConstIfConst<P, InstructionBase>::type ValueType;

            IteratorImpl() : m_prog(HK_NULL), m_idx(0) {}
            IteratorImpl(const IteratorImpl<Program>& rhs) : m_prog(rhs.m_prog), m_idx(rhs.m_idx) {}

            IteratorImpl& operator++();
            IteratorImpl operator++(int);

            IteratorImpl& operator--();
            IteratorImpl operator--(int);

            bool operator!=(const IteratorImpl& rhs) const;
            bool operator==(const IteratorImpl& rhs) const;

            /// Returns an iterator stepped forward v times.
            IteratorImpl operator+(int v) const;

            /// Returns an iterator stepped backwards v times.
            IteratorImpl operator-(int v) const;

            /// Returns the distance between this iterator and rhs.
            int operator-(const IteratorImpl& rhs) const;

            /// Steps the iterator forward v times.
            IteratorImpl& operator+=(int v);

            ValueType& operator*();
            ValueType* operator->();

        private:

            friend class Program;
            IteratorImpl(P* prog, int idx) : m_prog(prog), m_idx(idx) {}

            P* m_prog;
            int m_idx;
    };

    /// Base instruction type for all instructions.
    class HK_EXPORT_COMMON Program::InstructionBase
    {
        public:

            HK_DECLARE_CLASS(InstructionBase, New);

            InstructionBase() : m_instrType(INST_NONE), m_srcOffset(0), m_dstOffset(0), m_userData(0) {}
            InstructionBase(InstructionType instrType, int dstOff = 0, int srcOff = 0, hkUlong userData = 0)
                : m_instrType(instrType), m_srcOffset(hkLosslessCast<hkUint16>(srcOff)),
                m_dstOffset(hkLosslessCast<hkUint16>(dstOff)), m_userData(userData) {}

            InstructionType getInstructionType() const { return m_instrType; }

            void set(InstructionType instrType, int dstOffset, int srcOffset)
            {
                m_instrType = instrType;
                m_dstOffset = hkLosslessCast<hkUint16>(dstOffset);
                m_srcOffset = hkLosslessCast<hkUint16>(srcOffset);
            }

            int getSrcOffset() const { return m_srcOffset; }
            int getDstOffset() const { return m_dstOffset; }

            void setSrcOffset(int srcOffset) { m_srcOffset = hkLosslessCast<hkUint16>(srcOffset); }
            void setDstOffset(int dstOffset) { m_dstOffset = hkLosslessCast<hkUint16>(dstOffset); }

            hkUlong getUserData() const { return m_userData; }
            void setUserData(hkUlong userData) { m_userData = userData; }

            _Ret_maybenull_ const DefaultInstruction* isDefaultInstruction() const;
            _Ret_maybenull_ const ProgramInstruction* isProgramInstruction() const;
            _Ret_maybenull_ const RecordInstruction* isRecordInstruction() const;
            _Ret_maybenull_ const ArrayInstruction* isArrayInstruction() const;
            _Ret_maybenull_ const BlockInstruction* isBlockInstruction() const;

        protected:

            hkEnum<InstructionType, hkUint8> m_instrType;
            hkUint16 m_srcOffset;
            hkUint16 m_dstOffset;
            hkUlong m_userData;
    };

    /// Largest instruction type, no instruction should be bigger than this!
    class HK_EXPORT_COMMON Program::LargestInstruction : public Program::InstructionBase
    {
        // Unused private members trigger a warning on some compilers
        protected:

            void* m_data[4];
    };

    /// Instruction type for all the basic instructions except Array, Record and Container.
    class HK_EXPORT_COMMON Program::DefaultInstruction : public Program::InstructionBase
    {
        public:

            HK_DECLARE_CLASS(DefaultInstruction, New);

            DefaultInstruction()
                : m_srcType(HK_NULL), m_dstType(HK_NULL)
            {}

            DefaultInstruction(InstructionType instrType)
                : InstructionBase(instrType, 0, 0), m_srcType(HK_NULL), m_dstType(HK_NULL)
            {}

            void set(InstructionType instrType, int dstOffset, _In_ const hkReflect::Type* dstType, int srcOffset, _In_ const hkReflect::Type* srcType)
            {
                HK_ASSERT_NO_MSG(0x653e7251, dstType != HK_NULL);

                InstructionBase::set(instrType, dstOffset, srcOffset);
                m_srcType = srcType;
                m_dstType = dstType;
            }

            _Ret_notnull_ const hkReflect::Type* getSrcType() const { return m_srcType; }
            _Ret_notnull_ const hkReflect::Type* getDstType() const { return m_dstType; }

            /// Returns the src type cast as a Type if it is one, HK_NULL otherwise.
            template<typename Type>
            _Ret_maybenull_ const Type* getSrcTypeAs() const;

            /// Returns the dst type cast as a Type if it is one, HK_NULL otherwise.
            template<typename Type>
            _Ret_maybenull_ const Type* getDstTypeAs() const;

            /// Returns the Var using the src type.
            hkReflect::Var getSrcVar(_In_opt_ void* srcAddr) const;

            /// Returns the Var using the dst type.
            hkReflect::Var getDstVar(_In_opt_ void* dstAddr) const;

            /// Returns the Var using the src type.
            template<typename VarType>
            VarType getSrcVarAs(_In_ const void* srcAddr) const;

            /// Returns the Var using the dst type.
            template<typename VarType>
            VarType getDstVarAs(_In_ const void* dstAddr) const;

        private:

            const hkReflect::Type* m_srcType;
            const hkReflect::Type* m_dstType;
    };
    HK_COMPILE_TIME_ASSERT(sizeof(Program::DefaultInstruction) <= sizeof(Program::LargestInstruction));

    /// Base instruction type for types containing a subtype (array, records).
    class HK_EXPORT_COMMON Program::ProgramInstruction : public Program::DefaultInstruction
    {
        public:

            HK_DECLARE_CLASS(ProgramInstruction, New);

            ProgramInstruction(InstructionType instrType)
                : DefaultInstruction(instrType), m_program(HK_NULL)
            {}

            void set(int dstOffset, _In_ const hkReflect::Type* dstType, int srcOffset, _In_ const hkReflect::Type* srcType, _In_opt_ const Program* program)
            {
                DefaultInstruction::set(getInstructionType(), dstOffset, dstType, srcOffset, srcType);
                m_program = program;
            }

            _Ret_maybenull_ const Program* getProgram() const { return m_program; }

        private:

            const Program* m_program;
    };

    /// Instruction type for records.
    class HK_EXPORT_COMMON Program::RecordInstruction : public Program::ProgramInstruction
    {
        public:

            HK_DECLARE_CLASS(RecordInstruction, New);

            RecordInstruction()
                : ProgramInstruction(INST_RECORD)
            {}

            _Ret_maybenull_ const hkReflect::RecordType* getDstType() const;
            hkReflect::RecordVar getDstVar(_In_ void* dstAddr) const;

    };
    HK_COMPILE_TIME_ASSERT(sizeof(Program::RecordInstruction) <= sizeof(Program::LargestInstruction));


    /// Instruction type for arrays.
    class HK_EXPORT_COMMON Program::ArrayInstruction : public Program::ProgramInstruction
    {
        public:

            HK_DECLARE_CLASS(ArrayInstruction, New);

            ArrayInstruction()
                : ProgramInstruction(INST_ARRAY)
            {}

            _Ret_notnull_ const hkReflect::ArrayType* getDstType() const;
            hkReflect::ArrayVar getDstVar(_In_ void* dstAddr) const;
    };
    HK_COMPILE_TIME_ASSERT(sizeof(Program::ArrayInstruction) <= sizeof(Program::LargestInstruction));


    //class Program::BeginEndRecordInstruction : public Program::InstructionBase
    //{
    //  public:
    //
    //      HK_DECLARE_CLASS(BeginEndRecordInstruction, New);
    //
    //      void set(InstructionType instrType, const hkReflect::RecordType* recType)
    //      {
    //          InstructionBase::set(instrType);
    //          m_recordType = recType;
    //      }
    //
    //  private:
    //
    //      const hkReflect::RecordType* m_recordType;
    //      Label m_after;
    //};

    /// Instruction representing a memory block as created by the BlockPass pass.
    class HK_EXPORT_COMMON Program::BlockInstruction : public Program::InstructionBase
    {
        public:

            HK_DECLARE_CLASS(BlockInstruction, New);

            BlockInstruction(int dstOff, int srcOff, int numBytes)
                : m_numBytes(hkLosslessCast<hkUint32>(numBytes))
            {
                set(INST_BLOCK, dstOff, srcOff);
            }

            int getNumBytes() const { return m_numBytes; }
            void setNumBytes(int numBytes) { m_numBytes = hkLosslessCast<hkUint32>(numBytes); }

        private:

            hkUint32 m_numBytes;
    };
    HK_COMPILE_TIME_ASSERT(sizeof(Program::BlockInstruction) <= sizeof(Program::LargestInstruction));

    //class Program::WriteConstantInstruction : public Program::InstructionBase
    //{
    //  public:

    //      HK_DECLARE_CLASS(WriteConstantInstruction, New);

    //      WriteConstantInstruction()
    //          : InstructionBase(INST_WRITE_CONSTANT)
    //      {
    //      }

    //  private:

    //      const hkReflect::Type* m_dstType;
    //      hkReflect::Var m_constant;
    //      hkUint16 m_dstOffset;
    //};
    //HK_COMPILE_TIME_ASSERT(sizeof(Program::WriteConstantInstruction) <= sizeof(Program::LargestInstruction));

    /// Base type for compiler passes. A pass may apply any sort of transformations to a Program.
    class HK_EXPORT_COMMON Pass
    {
        public:

            HK_DECLARE_CLASS(Pass, New);

            virtual ~Pass() {}

            /// Applies the transformation to the input program. If the pass returns HK_FAILURE then the
            /// compilation fails and the compiler will return a null program.
            virtual hkResult apply(Program& program, hkReflect::QualType srcType, hkReflect::QualType dstType) = 0;
    };

    /// The type compiler builds an efficient Program out of a type or a pair of types (mapping one to the other).
    /// The base compiler won't optimize the Program and simply produce a flattened representation of the types.
    /// On top of that you can add passes to enabled certain optimizations (group adjacent fields into blocks,
    /// inline static arrays, ...).
    class HK_EXPORT_COMMON Compiler
    {
        public:

            HK_DECLARE_CLASS(Compiler, New, NonCopyable);

            Compiler();
            ~Compiler();

            /// Creates a new pass of type P and adds it to the compiler.
            template<typename P>
            void addPass() { addPass(new P); }

            /// Adds the pass to the compiler and takes ownership of the 'pass' object.
            void addPass(_In_ Pass* pass);

            /// Compiles the type into a program.
            hkViewPtr<Program> compile(hkReflect::QualType type);

            /// Creates a program mapping srcType to dstType.
            hkViewPtr<Program> compile(hkReflect::QualType srcType, hkReflect::QualType dstType);

            /// Clear any programs whose types are marked as temporary
            void clearTemporaryTypes();

            /// Include BEGIN_* / END_* instructions in the program.
            /// true by default.
            bool addBeginEndInstr;

            /// Emits instructions for fields which don't have a source (new fields). Typically used to write default values.
            /// true by default.
            bool addInstrWithNoSource;

            /// Exclude non-serializable fields from the program.
            /// false by default.
            bool excludeNonSerializableFields;

            /// Exclude properties from the program.
            /// false by default.
            bool excludeProperties;

            /// Case is significant when comparing fields
            /// true by default
            bool caseSensitive;

            /// Recurse into parent records
            /// true by default
            bool recurseIntoParentRecords;

        private:

            typedef hkMap< hkTuple<const hkReflect::Type*, const hkReflect::Type*>, Program*> ProgramMap;

            hkArray<Pass*> m_pass;
            ProgramMap m_programs;
            // These programs are considered temporary and can be cleared
            hkArray< hkTuple<const hkReflect::Type*, const hkReflect::Type*> > m_temporaryKeys;
    };
}

HK_REFLECT_ENUM(HK_EXPORT_COMMON, hkTypeVm::Program::InstructionType);

#include <Common/Base/Reflect/TypeVm/hkTypeVmCompiler.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
