// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompiler.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>

#include <Common/Base/Container/PointerMap/hkMap.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
#include <Common/Base/Container/Tuple/hkTuple.h>
#include <Common/Base/Container/Tuple/hkTupleMapOperations.cxx>

#define DEBUG_LOG_IDENTIFIER "Type.Compiler"
#include <Common/Base/System/Log/hkLog.hxx>

template class hkMap< hkTuple<const hkReflect::Type*, const hkReflect::Type*>, hkTypeVm::Program*>;

namespace
{
    using namespace hkTypeVm;

    template<typename T>
    struct InstrFromType;

    template<>
    struct InstrFromType<hkReflect::ArrayType>
    {
        static const Program::InstructionType instrType = Program::INST_ARRAY;
    };

    template<>
    struct InstrFromType<hkReflect::BoolType>
    {
        static const Program::InstructionType instrType = Program::INST_BOOL;
    };

    //template<>
    //struct InstrFromType<hkReflect::ContainerType>
    //{
    //  static const Program::InstructionType instrType = Program::INST_CONTAINER;
    //};

    template<>
    struct InstrFromType<hkReflect::IntType>
    {
        static const Program::InstructionType instrType = Program::INST_INT;
    };

    template<>
    struct InstrFromType<hkReflect::FloatType>
    {
        static const Program::InstructionType instrType = Program::INST_FLOAT;
    };

    template<>
    struct InstrFromType<hkReflect::PointerType>
    {
        static const Program::InstructionType instrType = Program::INST_POINTER;
    };

    template<>
    struct InstrFromType<hkReflect::RecordType>
    {
        static const Program::InstructionType instrType = Program::INST_RECORD;
    };

    template<>
    struct InstrFromType<hkReflect::StringType>
    {
        static const Program::InstructionType instrType = Program::INST_STRING;
    };

    class CompilerVisitor : public hkReflect::TypeVisitor
        <CompilerVisitor,
        hkResult,
        const int,
        const hkReflect::Type*,
        const int>
    {
        public:

            HK_DECLARE_CLASS(CompilerVisitor, New);

            CompilerVisitor(Compiler& compiler, Program& prog, _In_ const hkReflect::Type* topLevelDstType, _In_ const hkReflect::Type* topLevelSrcType)
                : m_compiler(compiler), m_prog(prog), m_topLevelSrcType(topLevelSrcType), m_topLevelDstType(topLevelDstType)
            {}

            hkResult dispatch( const hkReflect::Type* type, const int dstOff, const hkReflect::Type* srcType, const int srcOff )
            {
                if ( m_compiler.excludeNonSerializableFields && !type->isSerializable() )
                {
                    
                    // HK_ASSERT(0x25448d11, false, "Found non-serializable type {} ({}) while compiling a serializable hierarchy",
                    //     type, type->isField());
                }
                // Call parent method.
                return hkReflect::TypeVisitor<CompilerVisitor, hkResult, const int, const hkReflect::Type*, const int>::
                    dispatch( type, dstOff, srcType, srcOff );
            }

            hkResult visit(_In_ const hkReflect::VoidType* dstType, const int dstOff, _In_opt_ const hkReflect::Type* srcType, const int)
            {
                HK_UNREACHABLE(0x4523a895, "Found field with VOID format" );
#if defined(HK_COMPILER_GCC) && defined(HK_DEBUG)
                return HK_FAILURE;
#endif
            }

            hkResult visit(_In_ const hkReflect::OpaqueType* dstType, const int dstOff, _In_opt_ const hkReflect::Type* srcType, const int srcOff)
            {
                HK_ASSERT_NO_MSG(0x67446404, m_compiler.excludeNonSerializableFields == false);

                // Handle opaque types as Records.
                
                
                if (srcType != m_topLevelSrcType)
                {
                    auto program = m_compiler.compile(srcType, dstType);
                    if (!program)
                    {
                        return HK_FAILURE;
                    }
                    m_prog.addInstr<Program::RecordInstruction>().set(dstOff, dstType, srcOff, srcType, program);
                }
                else
                {
                    hkReflect::DeclIter<hkReflect::DataFieldDecl> it(dstType, m_compiler.recurseIntoParentRecords ? HK_NULL : dstType->getParent());
                    while (it.advance())
                    {
                        const int curOff = it.current().getOffset();
                        const hkReflect::Type* curType = it.current().getType();

                        HK_RETURN_IF_FAILED( dispatch(curType, dstOff + curOff, curType, srcOff + curOff) );
                    }
                }
                return HK_SUCCESS;
            }

            template<typename T>
            hkResult visit(_In_ const T* dstType, const int dstOff, _In_opt_ const hkReflect::Type* srcType, const int srcOff)
            {
                if(srcType != HK_NULL || m_compiler.addInstrWithNoSource)
                {
                    m_prog.addInstr<Program::DefaultInstruction>().set(InstrFromType<T>::instrType, dstOff, dstType, srcOff, srcType);
                }
                return HK_SUCCESS;
            }

            hkResult visit(_In_ const hkReflect::RecordType* dstType, const int dstOff, _In_opt_ const hkReflect::Type* srcType, const int srcOff)
            {
                if(srcType == HK_NULL && m_compiler.addInstrWithNoSource == false)
                {
                    return HK_SUCCESS;
                }

                if(dstType != m_topLevelDstType && srcType != m_topLevelSrcType)
                {
                    auto program = m_compiler.compile(srcType, dstType);
                    if( !program )
                    {
                        return HK_FAILURE;
                    }
                    m_prog.addInstr<Program::RecordInstruction>().set(dstOff, dstType, srcOff, srcType, program);
                }
                else
                {
                    //if(m_compiler.addBeginEndInstr)
                    //{
                    //  m_prog.addInstr<hkTypeCompiler::Program::BeginEndRecordInstruction>().set//(hkTypeCompiler::Program::BEGIN_RECORD, dstType);
                    //}

                    if(srcType == dstType) // fast path if types are identical
                    {
                        hkReflect::DeclIter<hkReflect::FieldDecl> it(dstType, m_compiler.recurseIntoParentRecords ? HK_NULL : dstType->getParent());
                        while(it.advance())
                        {
                            if((m_compiler.excludeNonSerializableFields == false || it.current().isSerializable()) &&
                               (m_compiler.excludeProperties == false || it.current().getType()->isProperty() == false))
                            {
                                const int curOff = it.current().getOffset();
                                const hkReflect::Type* curType = it.current().getType();

                                HK_RETURN_IF_FAILED( dispatch(curType, dstOff + curOff, curType, srcOff + curOff) );
                            }
                        }
                    }
                    else // slow path: have to do name matching
                    {
                        const hkReflect::RecordType* srcRecType = srcType ? srcType->asRecord() : HK_NULL;
                        if(srcType == srcRecType) // either both are null or both are the same type
                        {

                            hkReflect::DeclIter<hkReflect::FieldDecl> it(dstType, m_compiler.recurseIntoParentRecords ? HK_NULL : dstType->getParent());
                            while (it.advance())
                            {
                                if (m_compiler.excludeNonSerializableFields == false || it.current().isSerializable())
                                {
                                    bool fieldMatched = false;
                                    if (srcRecType)
                                    {
                                        hkReflect::FieldDecl srcField;
                                        if (m_compiler.caseSensitive)
                                        {
                                            srcField = srcRecType->findField(it.current().getName(), m_compiler.recurseIntoParentRecords);
                                        }
                                        else
                                        {
                                            srcField = srcRecType->findFieldNoCase(it.current().getName(), m_compiler.recurseIntoParentRecords);
                                        }

                                        if (srcField)
                                        {
                                            if ((m_compiler.excludeNonSerializableFields == false || srcField.isSerializable()) &&
                                                (m_compiler.excludeProperties == false || srcField.getType()->isProperty() == false))
                                            {
                                                HK_RETURN_IF_FAILED( dispatch(it.current().getType(), dstOff + it.current().getOffset(), srcField.getType(), srcOff + srcField.getOffset()) );
                                                fieldMatched = true;
                                            }
                                        }
                                    }
                                    if (!fieldMatched)
                                    {
                                        HK_RETURN_IF_FAILED( dispatch(it.current().getType(), dstOff + it.current().getOffset(), HK_NULL, 0) );
                                    }
                                }
                            }
                        }
                        else
                        {
                            HK_RETURN_IF_FAILED( dispatch(dstType, dstOff, HK_NULL, 0) );
                        }
                    }

                    //if(m_compiler.addBeginEndInstr)
                    //{
                    //  m_prog.addInstr<hkTypeCompiler::Program::BeginEndRecordInstruction>().set/(hkTypeCompiler::Program::END_RECORD, /dstType);
                    //}
                }
                return HK_SUCCESS;
            }

            hkResult visit(_In_ const hkReflect::ArrayType* dstType, const int dstOff, _In_opt_ const hkReflect::Type* srcType, const int srcOff)
            {
                if(srcType == HK_NULL && m_compiler.addInstrWithNoSource == false)
                {
                    return HK_SUCCESS;
                }

                if(srcType)
                {
                    const Program* elemTypeProg = HK_NULL;

                    if(const hkReflect::ArrayType* srcArrType = srcType->asArray())
                    {
                        if(dstType->getSubType() != HK_NULL && srcArrType->getSubType() != HK_NULL)
                        {
                            elemTypeProg = m_compiler.compile(srcArrType->getSubType(), dstType->getSubType());
                            if (!elemTypeProg)
                            {
                                return HK_FAILURE;
                            }
                        }
                    }

                    m_prog.addInstr<Program::ArrayInstruction>().set(dstOff, dstType, srcOff, srcType, elemTypeProg);
                }
                else
                {
                    // ArrayInstruction with no source, the VM may write a default value.
                    m_prog.addInstr<Program::ArrayInstruction>().set(dstOff, dstType, srcOff, srcType, HK_NULL);
                }
                return HK_SUCCESS;
            }

        private:

            Compiler& m_compiler;
            Program& m_prog;
            const hkReflect::Type* m_topLevelSrcType;
            const hkReflect::Type* m_topLevelDstType;
    };
}

hkTypeVm::Compiler::Compiler()
    : addBeginEndInstr(true)
    , addInstrWithNoSource(true)
    , excludeNonSerializableFields(false)
    , excludeProperties(false)
    , caseSensitive(true)
    , recurseIntoParentRecords(true)
{

}

hkTypeVm::Compiler::~Compiler()
{
    for(ProgramMap::Iterator it = m_programs.getIterator(); m_programs.isValid(it); it = m_programs.getNext(it))
    {
        delete m_programs.getValue(it);
    }

    for(int i = 0; i<m_pass.getSize(); ++i)
    {
        delete m_pass[i];
    }
}

void hkTypeVm::Compiler::addPass(_In_ Pass* pass)
{
    m_pass.pushBack(pass);
}

hkViewPtr<Program> hkTypeVm::Compiler::compile(hkReflect::QualType type)
{
    const hkTuple<const hkReflect::Type*, const hkReflect::Type*> progKey(type.get(), type.get());

    if(Program* cachedProg = m_programs.getWithDefault(progKey, HK_NULL))
    {
        return cachedProg;
    }
    else
    {
        hkScopedPtr<Program> newProg(new Program());

        const bool isTemporaryProgram = type->isTemporary();

        m_programs.insert(progKey, newProg.get());

        CompilerVisitor(*this, *newProg, type, type).dispatch(type, 0, type, 0);

        for(int i = 0; i<m_pass.getSize(); ++i)
        {
            if(m_pass[i]->apply(*newProg, type, type).isFailure())
            {
                m_programs.remove(progKey);
                return HK_NULL;
            }
        }

        if (isTemporaryProgram)
        {
            m_temporaryKeys.pushBack(progKey);
        }

        return newProg.steal();
    }
}

hkViewPtr<Program> hkTypeVm::Compiler::compile(hkReflect::QualType srcType, hkReflect::QualType dstType)
{
    const hkTuple<const hkReflect::Type*, const hkReflect::Type*> progKey(srcType.get(), dstType.get());

    if(Program* cachedProg = m_programs.getWithDefault(progKey, HK_NULL))
    {
        Log_Debug("{}, Type returned from cache from {}->{}, p = {}", this, srcType.get(), dstType.get(), cachedProg);
        return cachedProg;
    }
    else
    {
        hkScopedPtr<Program> newProg(new Program());

        const bool isTemporaryProgram = srcType->isTemporary() || dstType->isTemporary();

        m_programs.insert(progKey, newProg.get());

        if (CompilerVisitor(*this, *newProg, dstType, srcType).dispatch(dstType, 0, srcType, 0).isFailure())
        {
            m_programs.remove(progKey);
            return nullptr;
        }

        for(int i = 0; i<m_pass.getSize(); ++i)
        {
            if(m_pass[i]->apply(*newProg, srcType, dstType).isFailure())
            {
                m_programs.remove(progKey);
                Log_Warning("{}, Compiler could not convert types {}->{}, p = {}", this, srcType.get(), dstType.get(), newProg.get());
                return HK_NULL;
            }
        }

        if (isTemporaryProgram)
        {
            m_temporaryKeys.pushBack(progKey);
        }

        Log_Debug("{} Type newly compiled {}->{}, p = {}", this, srcType.get(), dstType.get(), newProg.get());

        return newProg.steal();
    }
}

_Ret_maybenull_ const hkTypeVm::Program::DefaultInstruction* hkTypeVm::Program::asDefaultInstruction(const InstructionBase& instr)
{
    switch(instr.getInstructionType())
    {
        case INST_ARRAY:
        case INST_BOOL:
        //case INST_CONTAINER:
        case INST_INT:
        case INST_FLOAT:
        case INST_POINTER:
        case INST_RECORD:
        case INST_STRING:
        case INST_CONVERT_INTEGER:
        case INST_CONVERT_FLOAT:
        case INST_CONVERT_BOOL:
            return static_cast<const DefaultInstruction*>(&instr);

        default:
            return HK_NULL;
    }
}

_Ret_maybenull_ hkTypeVm::Program::DefaultInstruction* hkTypeVm::Program::asDefaultInstruction( InstructionBase& instr )
{
    switch ( instr.getInstructionType() )
    {
        case INST_ARRAY:
        case INST_BOOL:
        //case INST_CONTAINER:
        case INST_INT:
        case INST_FLOAT:
        case INST_POINTER:
        case INST_RECORD:
        case INST_STRING:
            return static_cast<DefaultInstruction*>( &instr );

        default:
            return HK_NULL;
    }
}

void hkTypeVm::Program::insertAfter(Iterator it, ConstIterator b, ConstIterator e)
{
    m_code.insertAt(it.m_idx + 1, static_cast<const LargestInstruction*>(&*b), e - b);
}

void hkTypeVm::Program::remove(ConstIterator it)
{
    m_code.removeAtAndCopy(it.m_idx);
}

void hkTypeVm::Program::remove(ConstIterator b, ConstIterator e)
{
    if(int numToRemove = e - b)
    {
        m_code.removeAtAndCopy(b.m_idx, numToRemove);
    }
}

void hkTypeVm::Compiler::clearTemporaryTypes()
{
    for (auto& key : m_temporaryKeys)
    {
        ProgramMap::Iterator it = m_programs.findKey(key);
        hkTypeVm::Program* prog = m_programs.getValue(it);
        delete prog;
        m_programs.remove(it);
    }
    m_temporaryKeys.clear();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
