// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

namespace hkReflect
{

    class MutableTypeReg;

    /// Registered types iterator
    class HK_EXPORT_COMMON TypeRegIterator
    {
    public:

        HK_DECLARE_CLASS(TypeRegIterator, New, NonCopyable);
        TypeRegIterator(_In_ const MutableTypeReg* reg);
        TypeRegIterator();

        inline bool advance()
        {
            if (m_idx + 1 < m_types.getSize())
            {
                m_idx += 1;
                return true;
            }
            return false;
        }
        inline _Ret_notnull_ const hkReflect::Type* current() const { return m_types[m_idx]; }

        void push(_In_ const Type* t) { m_types.pushBack(t); }
        void append(_In_reads_(n) const Type*const* t, int n) { m_types.append(t, n); }

    protected:

        int m_idx;
        hkArray<const Type*> m_types;
    };

    /// Interface to lookup types.
    class HK_EXPORT_COMMON TypeReg : public hkReferencedObject
    {
    public:
        HK_DECLARE_CLASS(TypeReg, New);

        virtual ~TypeReg();

        /// Get the type from an object name
        virtual _Ret_maybenull_ const Type* typeFromName(_In_z_ const char* name) const = 0;

        /// Find a type given a named type (which may have template parameters)
        /// This lookup is based on type identity (type name and name/value of any template parameters)
        virtual _Ret_maybenull_ const Type* typeFromType(_In_ const Type* type) const = 0;
    };

    /// Type Registry which can have types added and removed.
    class HK_EXPORT_COMMON MutableTypeReg : public TypeReg
    {
    public:

        HK_DECLARE_CLASS(MutableTypeReg, New);

        /// Add a type.
        /// If the type cannot be added HK_FAILURE is returned and ownership is not transferred.
        /// If the type is added, HK_SUCCESS is returned and the registry assumes ownership of the type.
        /// If ownership is not reclaimed (by calling remove()), the type will be deleted when
        /// the registry is destructed.
        virtual hkResult add(_Inout_ Type* type) = 0;

        /// Remove a type.
        /// Returns HK_FAILURE if the type is not added.
        virtual hkResult remove(_Inout_ Type* type) = 0;

        /// Describes a type replacement in the \ref TypesChangedFuncArgs struct.
        /// m_old is the old type, m_new the type it's being replaced with.
        /// Both pointers are guaranteed to point to valid types.
        struct TypeReplacement
        {
            const hkReflect::Type* m_old;
            const hkReflect::Type* m_new;
        };

        /// The arguments to a TypesChangedFunc callback.
        struct TypesChangedFuncArgs
        {
            /// All types which got added. The callback function is allowed to
            /// keep a pointer to an added type, until the type is removed again
            /// or replaced by a different one, in which case the callback is
            /// called as well, with the type in either the \ref m_removed or
            /// \ref m_replaced list.
            hkArrayView<const Type*const> m_added;

            /// All types which got removed. While the callback is called, these
            /// types are still guaranteed to be valid but they will become
            /// invalid afterwards, which means that the called code should drop
            /// all references to types in this list.
            hkArrayView<const Type*const> m_removed;

            /// All types which got replaced by a new type. Both types in the
            /// \ref TypeReplacement are valid, but those pointed to by m_old
            /// will become invalid afterwards, which means that the called code
            /// should drop all references to them.
            hkArrayView<TypeReplacement> m_replaced;
        };

        /// Helper typedef for subscribeForChange
        typedef void (HK_CALL * TypesChangedFunc)(const TypesChangedFuncArgs& args, void* cbData);

        class Subscription
        {
        public:
            Subscription() {}
            Subscription(_In_ hkReferencedObject* impl) : m_impl(impl) {}
            void unsubscribe() { m_impl = HK_NULL; }
        private:
            hkRefPtr<hkReferencedObject> m_impl;
        };

        /// Receive a callback when types are added and removed.
        /// When the subscription is added, the handler is fired for all currently registered types.
        virtual Subscription subscribeForChange(TypesChangedFunc cbFunc, void* cbData) = 0;

        /// Notify the registry that the data in its types has been mutated.
        /// The built in registry uses this notification to synchronize across DLLs.
        virtual void notifyTypesMutated();

        /// Append all types in this registry to the given array.
        virtual void appendAllTypesToArray(hkArray<const Type*>& types) const = 0;
    };


} //namespace hkReflect

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
