// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/TypeReg/Detail/hkTypeRegDetail.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Types/hkEndian.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
#include <Common/Base/Fwd/hkcstring.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/Object/hkSingleton.h>

#define DEBUG_LOG_IDENTIFIER "reflect.TypeReg"
#include <Common/Base/System/Log/hkLog.hxx>

typedef hkReflect::Detail::BuiltinTypeReg BuiltinTypeReg;
HK_SINGLETON_IMPLEMENTATION(BuiltinTypeReg);

//
// The builtin registry has two parts. One part is all the types generated
// in the _Types.cxx and _Types.inl files. These are all statically allocated.
// Each DLL has its own list, and we store a list of lists (see addBatch()).
// When the list changes, we rebuildEverything() which essentially means
// recalculating opt inheritance, finding duplicate template instantiations and
// building the name lookup tree.
//
// The second part is the dynamically allocated types. Probably they come from the
// hkDynamicTypeManager, but anybody can add types.
//
// Note on locking - we don't want to lock on every call. Even if we did, it wouldn't
// make sense for the statically allocated ones. When a DLL is unloaded the statically
// allocated returned type pointer would become invalid regardless. So then, we assume
// mutation via addBatch/removeBatch are done at carefully chosen times. However
// read-only access to the static part doesn't need a lock. Only the m_extraTypes array
// must be locked since that's truly mutable.
//


//bool m_openForBatch;

hkReflect::Detail::BuiltinTypeReg::BuiltinTypeReg()
    : m_nameTreeRoot(HK_NULL)
//  : m_openForBatch(true)
{
    // Add a fixed set of supported formats.
#define BUILTIN(NAME) { \
    const hkReflect::Type* t = hkReflect::getType<NAME>(); \
    m_typeFromFormat.insert( t->getFormat().get(), t);  }

    // Integers
    BUILTIN(hkInt8Le);
    BUILTIN(hkInt16Le);
    BUILTIN(hkInt32Le);
    BUILTIN(hkInt64Le);
    BUILTIN(hkUint8Le);
    BUILTIN(hkUint16Le);
    BUILTIN(hkUint32Le);
    BUILTIN(hkUint64Le);
    BUILTIN(hkInt8Be);
    BUILTIN(hkInt16Be);
    BUILTIN(hkInt32Be);
    BUILTIN(hkInt64Be);
    BUILTIN(hkUint8Be);
    BUILTIN(hkUint16Be);
    BUILTIN(hkUint32Be);
    BUILTIN(hkUint64Be);

    // Floats.
    BUILTIN(hkFloat32Le);
    BUILTIN(hkFloat64Le);
    BUILTIN(hkFloat32Be);
    BUILTIN(hkFloat64Be);
    BUILTIN(hkHalf16Le);
    BUILTIN(hkHalf16Be);
    //BUILTIN(hkUfloat8); 

    // Bools.
    BUILTIN( hkBool8Le );
    BUILTIN( hkBool8Be );
    BUILTIN( hkBool32Le );
    BUILTIN( hkBool32Be );
    // 'bool' must be one of these
    static_assert(sizeof( bool ) == 1 || sizeof( bool ) == 4, "Invalid size of 'bool', add an item to the list above");

    BUILTIN(void);
#undef BUILTIN

    hkReflect::Detail::CheckUtil::Config config;
    config.m_isTypeReachableCallback = nullptr; // bug in type registration from static reflection data; cannot check at the moment
    m_checker.setConfig( config );
}

hkReflect::Detail::BuiltinTypeReg::~BuiltinTypeReg()
{
    // any extra types remaining need to be deleted
    for( int i = m_extraTypes.getSize()-1; i >= 0; --i )
    {
        Type* t = m_extraTypes[i];
        HK_ON_DEBUG(bool ok=)hk::DeleteTypeInfo::deleteType(t);
        HK_ASSERT(0x488e92c3,ok,"Added type needs an hk::DeleteTypeInfo attribute or to be removed before registry destruction");
    }
}

static bool s_inheritanceOk(_In_ const hkReflect::Type* type)
{
    const hkReflect::Detail::InheritanceInfo* inheritance = hkReflect::TypeDetail::getInheritance(type);
    if (inheritance == nullptr)
    {
        return true; // no opt inheritance is OK
    }
    if( inheritance->asUlong()==0 )
    {
        return false; // uninitialized opt inheritance is not OK
    }

    // synthetic types will have the same inheritance as their parent
    const hkReflect::Type* parent = type->getParent();
    if( parent == HK_NULL )
    {
        return false;
    }
    return inheritance->equals( *hkReflect::TypeDetail::getInheritance(parent) );
}

hkResult hkReflect::Detail::BuiltinTypeReg::add(_Inout_ hkReflect::Type* type)
{
//  HK_ASSERT_NO_MSG(0x7c32a3dd,m_openForBatch==false);
    hkCriticalSectionLock lock(&m_lock);
    if( treeTypeFromType(type) == HK_NULL )
    {
        
        HK_ASSERT(0x5c476ba3, type->getParent() == HK_NULL || typeFromType(TypeDetail::skipDecorators(type)->getParent()), "parent must be added first.");
        HK_ASSERT_NO_MSG(0x7959875d, type->getAlignOf() != -1);
        HK_ASSERT(0x3567d163, type->getTypeWorld() == HK_NULL, "Type must be built with TypeWorld == NULL to be used as a native type");

        for(int i = 0; i < m_extraTypes.getSize(); i++)
        {
            if ( type == m_extraTypes[i] )
            {
                return HK_SUCCESS;
            }

            if(hkReflect::Type::compareName(m_extraTypes[i], type) == 0)
            {
                TypeReplacement replaced;
                replaced.m_old = m_extraTypes[i];
                replaced.m_new = type;

                TypesChangedFuncArgs args;
                args.m_replaced = hkArrayViewT::fromSingleObject(replaced);
                fireCallbacks(args);

                hk::DeleteTypeInfo::deleteType(m_extraTypes[i]);
                m_extraTypes[i] = type;

                return HK_SUCCESS;
            }
        }

        m_extraTypes.pushBack(type);

        TypesChangedFuncArgs args;
        args.m_added = hkArrayViewT::fromSingleObject(type);
        fireCallbacks(args);
        return HK_SUCCESS;
    }
    return HK_FAILURE;
}

hkResult hkReflect::Detail::BuiltinTypeReg::remove(_Inout_ hkReflect::Type* type)
{
//  HK_ASSERT_NO_MSG(0x47fc8a7f,m_openForBatch==false);
    hkCriticalSectionLock lock(&m_lock);
    int idx = m_extraTypes.indexOf(type);
    if( idx >= 0)
    {
        m_extraTypes.removeAt(idx);

        TypesChangedFuncArgs args;
        args.m_removed = hkArrayViewT::fromSingleObject(type);
        fireCallbacks(args);
        return HK_SUCCESS;
    }
    return HK_FAILURE;
}

namespace
{
    void addAllNameMatchingTypes(_In_ const hkReflect::Detail::TypeRegNode* it, hkArray<const hkReflect::Type*>& types, _In_z_ const char* name, bool isTemplate)
    {
        if (hkString::strCmp(it->getType()->getName(), name) == 0)
        {
            if ( ( it->getType()->getTemplate() != nullptr ) == isTemplate )
            {
                // Both or none are templates.
                types.pushBack( it->getType() );
            }
        }
        if (it->getLeft())
        {
            addAllNameMatchingTypes(it->getLeft(), types, name, isTemplate);
        }
        if (it->getRight())
        {
            addAllNameMatchingTypes(it->getRight(), types, name, isTemplate);
        }
    }


    void extractParameter(_In_z_ const char* currentStart, _Inout_z_ char* currentEnd, hkInplaceArray<const char*, 8>::Temp& paramStrings)
    {
        while ((*currentStart == ' ') && (currentStart < currentEnd))
        {
            currentStart++;
        }
        currentEnd--;
        while ((*currentEnd == ' ') && (currentStart <= currentEnd))
        {
            currentEnd--;
        }

        if (currentStart <= currentEnd)
        {
            *(currentEnd + 1) = 0;
            paramStrings.pushBack(currentStart);
        }
    }

    // Stores root name in rootOut, param strings in paramBuf.
    hkResult extractRootAndParameters(const char* nameIn, hkStringBuf& paramBuf, hkStringBuf& rootOut, hkInplaceArray<const char*, 8>::Temp& paramStringsOut)
    {
        // Initialize rootOut with full name.
        rootOut = nameIn;

        const int indexOfOpen = rootOut.indexOf( '<' );
        if (indexOfOpen == 0)
        {
            Log_Warning("Invalid type name: '{}' (empty root)", nameIn);
            return HK_FAILURE;
        }
        else if (indexOfOpen >= 0)
        {
            paramBuf = rootOut;

            // Remove args from rootOut.
            rootOut.chompEnd(rootOut.getLength() - indexOfOpen);

            // Remove root and brackets from buf.
            const int indexOfClose = paramBuf.lastIndexOf(">");
            if ( indexOfClose < indexOfOpen )
            {
                Log_Warning("Invalid type name: '{}' (invalid arguments)", nameIn);
                return HK_FAILURE;
            }

            paramBuf.slice(indexOfOpen + 1, indexOfClose - indexOfOpen - 1);

            // Split string in buf into one string per argument.

            // Unfortunately this isn't clever enough, we can have nested templates etc
            //buf.split(',', paramStrings); // These still have spaces etc

            // We start with a<b>, c, d<e, f>

            int bracketNesting = 0;
            char* currentStart = const_cast<char*>(paramBuf.cString());
            char* currentEnd = currentStart + 1;

            while (*currentEnd)
            {
                if (*currentEnd == '<')
                {
                    bracketNesting++;
                }
                else
                {
                    if (bracketNesting)
                    {
                        if (*currentEnd == '>')
                        {
                            bracketNesting--;
                        }
                    }
                    else if (*currentEnd == ',')
                    {
                        extractParameter(currentStart, currentEnd, paramStringsOut);
                        currentStart = currentEnd + 1;
                        currentEnd = currentStart;
                    }
                    else if (*currentEnd == '>')
                    {
                        Log_Warning("Invalid type name: {} (invalid brackets nesting)", nameIn);
                        return HK_FAILURE;
                    }
                }
                currentEnd++;
            }
            if (bracketNesting)
            {
                Log_Warning("Invalid type name: {} (invalid brackets nesting)", nameIn);
                return HK_FAILURE;
            }
            extractParameter(currentStart, currentEnd, paramStringsOut);
        }
        else if ( rootOut.getLength() && rootOut[rootOut.getLength() - 1] == '*' )
        {
            // Name is in the form "SubType*", consider it as being "T*<SubType>".
            paramBuf.set( rootOut.cString(), rootOut.getLength() - 1 );
            char* start = const_cast<char*>( paramBuf.cString() );
            char* end = start + paramBuf.getLength();
            extractParameter( start, end, paramStringsOut );
            rootOut = "T*";
        }
        

        return HK_SUCCESS;
    }

    bool typeMatchesParameters(_In_ const hkReflect::Type* thisType, hkInplaceArray<const char*, 8>::Temp& paramStrings);

    bool typeMatchesRootAndParameters(_In_ const hkReflect::Type* thisType, _In_z_ const char* string)
    {
        hkStringBuf root;
        hkStringBuf nameStr;
        hkInplaceArray<const char*, 8>::Temp templateParamStrings;

        if (extractRootAndParameters(string, nameStr, root, templateParamStrings).isFailure())
        {
            return false;
        }

        if (hkString::strCmp(thisType->getName(), root) == 0)
        {
            return typeMatchesParameters(thisType, templateParamStrings);
        }

        return false;
    }

    bool typeMatchesParameters(_In_ const hkReflect::Type* thisType, hkInplaceArray<const char*, 8>::Temp& paramStrings)
    {
        if(const hkReflect::Template* params = thisType->getTemplate())
        {
            if (params->getNumParams() == paramStrings.getSize())
            {
                for (int paramIdx = 0; paramIdx < params->getNumParams(); paramIdx++)
                {
                    const hkReflect::Template::Parameter* thisParam = params->getParam(paramIdx);
                    if (thisParam->getKind() == hkReflect::Template::KIND_TYPE)
                    {
                        if (!typeMatchesRootAndParameters(thisParam->getAsType(), paramStrings[paramIdx]))
                        {
                            return false;
                        }
                    }
                    else
                    {
                        hkUlong paramVal = hkUlong(hkString::atoull(paramStrings[paramIdx]));
                        if (paramVal != thisParam->getAsValue())
                        {
                            return false;
                        }
                    }
                }
                return true; // No failed matches -> all matches
            }
            else
            {
                return false;
            }
        }
        else
        {
            return paramStrings.isEmpty(); // No params
        }
    }
}


_Ret_maybenull_ const hkReflect::Type* hkReflect::Detail::BuiltinTypeReg::typeFromName(_In_z_ const char* name) const
{
//  HK_ASSERT_NO_MSG(0x2e3af7cf,m_openForBatch==false);

    

    const TypeRegNode* it = m_nameTreeRoot;
    hkStringBuf root;
    hkStringBuf nameStr;
    hkInplaceArray<const char*, 8>::Temp templateParamStrings;
    hkInplaceArray<const hkReflect::Type*, 32> allMatchingTypes;

    if (extractRootAndParameters(name, nameStr, root, templateParamStrings).isFailure())
    {
        return nullptr;
    }

    while(it)
    {
        int cmp = hkString::strCmp(it->getType()->getName(), root);

        if (cmp == 0)
        {
            //if(mark)
            {
                const_cast<TypeRegNode*>(it)->markTypeUsed();
            }
            break;
        }
        else if(cmp < 0)
        {
            it = it->getRight();
        }
        else
        {
            it = it->getLeft();
        }
    }

    if (it)
    {
        addAllNameMatchingTypes(it, allMatchingTypes, it->getType()->getName(), templateParamStrings.getSize());
    }

    hkCriticalSectionLock lock(&m_lock);
    for( int i = 0; i < m_extraTypes.getSize(); ++i )
    {
        const hkReflect::Type* t = m_extraTypes[i];
        if (hkString::strCmp(t->getName(), root) == 0)
        {
            allMatchingTypes.pushBack(t);
        }
    }

    if (templateParamStrings.getSize() > 0)
    {
        // Test string is a template, we need to see which one matches
        for (int i = 0; i < allMatchingTypes.getSize(); i++)
        {
            const hkReflect::Type* thisType = allMatchingTypes[i];
            // try to match all of the params
            if (typeMatchesParameters(thisType, templateParamStrings))
            {
                return thisType;
            }
        }
    }
    else
    {
        HK_ASSERT(0x760717c, allMatchingTypes.getSize() <= 1, "Registry cannot contain more that one type with the same name");
        if (allMatchingTypes.getSize())
        {
            return allMatchingTypes[0];
        }
    }

    return HK_NULL;
}

_Ret_maybenull_ const hkReflect::Type* hkReflect::Detail::BuiltinTypeReg::typeFromType(_In_opt_ const hkReflect::Type* type) const
{
    if( type == HK_NULL )
    {
        return HK_NULL;
    }
//  HK_ASSERT_NO_MSG(0xd0b57ea,m_openForBatch==false);

    const hkReflect::Type* ret = treeTypeFromType(type);
    if(ret)
    {
        return ret;
    }

    hkCriticalSectionLock lock(&m_lock);
    for( int i = 0; i < m_extraTypes.getSize(); ++i )
    {
        const hkReflect::Type* t = m_extraTypes[i];
        if( hkReflect::Type::compareName(t,type) == 0)
        {
            return t;
        }
    }

    return HK_NULL;
}

_Ret_maybenull_ const hkReflect::Type* hkReflect::Detail::BuiltinTypeReg::treeTypeFromType(_In_ const hkReflect::Type* type) const
{
    const TypeRegNode* it = m_nameTreeRoot;

    while(it)
    {
        int cmp = hkReflect::Type::compareName( it->getType(), type );

        if (cmp == 0)
        {
            //if(mark)
            {
                const_cast<TypeRegNode*>(it)->markTypeUsed();
            }
            return it->getType();
        }
        else if(cmp < 0)
        {
            it = it->getRight();
        }
        else
        {
            it = it->getLeft();
        }
    }

    return HK_NULL;
}

_Ret_maybenull_ const hkReflect::Type* hkReflect::Detail::BuiltinTypeReg::builtinFromFormat(hkUint32 format) const
{
    return m_typeFromFormat.getWithDefault(format, HK_NULL);
}

struct hkReflect::Detail::BuiltinTypeReg::SubscriptionImpl : public hkReferencedObject
{
    public:

        SubscriptionImpl(_In_ BuiltinTypeReg* r, int i, TypesChangedFunc cbFunc, _In_ void* cbData) : m_reg(r), m_idx(i), m_func(cbFunc), m_data(cbData) {}
        ~SubscriptionImpl()
        {
            if(m_reg)
            {
                m_reg->m_subscriptions[ m_idx ] = HK_NULL;
            }
        }
        BuiltinTypeReg* m_reg;
        int m_idx; // index into BuiltinTypeReg.m_subscriptions
        TypesChangedFunc m_func;
        void* m_data;
};

hkReflect::MutableTypeReg::Subscription hkReflect::Detail::BuiltinTypeReg::subscribeForChange(TypesChangedFunc cbFunc, _In_ void* cbData)
{
    hkCriticalSectionLock lock(&m_lock);

    int idx = m_subscriptions.indexOf(HK_NULL);
    if( idx == -1 )
    {
        idx = m_subscriptions.getSize();
        m_subscriptions.pushBack(HK_NULL);
    }

    SubscriptionImpl* impl = new SubscriptionImpl(this, idx, cbFunc, cbData);
    m_subscriptions[idx] = impl;

    hkArray<const Type*> added;
    appendAllTypesToArray(added);

    TypesChangedFuncArgs args;
    args.m_added = added;
    (*cbFunc)(args, cbData);

    Subscription s(impl);
    impl->removeReference();
    return s;
}


void hkReflect::Detail::BuiltinTypeReg::appendAllTypesToArray(hkArray<const Type*>& types) const
{
    for( int i = 0; i < m_lists.getSize(); ++i )
    {
        for( const TypeRegNode* e = m_lists[i]; e; e = e->getNext() )
        {
            if( e->typeIsDuplicate() == false )
            {
                types.pushBack(e->getType());
            }
        }
    }

    // assume that only m_extraTypes can change
    hkCriticalSectionLock lock(&m_lock);

    // Last one is the extras. There can be no duplicates becuase we checked in add().
    types.append( m_extraTypes.begin(), m_extraTypes.getSize() );
}

static int s_updateAlignment(_In_opt_ const hkReflect::Type* typeIn)
{
    using namespace hkReflect;
    const hkReflect::Type* type = typeIn;
    // find a type with a size/align
    while( type && TypeDetail::localHasOptional(type, Opt::SIZE_ALIGN) == false )
    {
        type = type->getParent();
    }
    if( type == HK_NULL )
    {
        return 1;
    }

    hkUlong* optPtr = TypeDetail::localAddressOptional<Opt::SIZE_ALIGN>(type);
    Detail::SizeAlign sa(*optPtr);
    if( sa.m_alignOf == Detail::ALIGN_UNSET )
    {
        // parents first
        int align = s_updateAlignment( const_cast<hkReflect::Type*>(type->getParent()) );
        if(type->hasOwnVtable())
        {
            align = hkMath::max2(align, hkSizeOf(void*));
        }
        // include max alignment of our fields if they are higher
        if( const RecordType* rt = type->asRecord() )
        {
            for( int i = 0; i < rt->getNumDataFields(); ++i )
            {
                const hkReflect::Type* fieldType = rt->getField(i).getType();
                int fieldAlign = fieldType->getAlignOf();
                HK_ASSERT_NO_MSG(0x1aa09091, fieldAlign > 0);
                align = hkMath::max2(align, fieldAlign);
            }
        }
        sa.m_alignOf = align;
        *optPtr = sa.asUlong();
    }
    return sa.m_alignOf;
}

void hkReflect::Detail::BuiltinTypeReg::addBatch(_Inout_ TypeRegNode* head)
{
//  HK_ASSERT_NO_MSG(0x324d1add,m_openForBatch);
    HK_ASSERT_NO_MSG(0x41a9c642,m_lists.indexOf(head)==-1);
    hkCriticalSectionLock lock(&m_lock);
    DLOG_SCOPE("Adding batch #{} {}", m_lists.getSize(), head);
    m_lists.pushBack(head);

    // Some compilers don't allow alignof(T) where T is abstract (e.g. MSVC).
    // So let's figure out the alignment for those ones now.
    for( const TypeRegNode* e = head; e; e = e->getNext() )
    {
        s_updateAlignment( const_cast<hkReflect::Type*>(e->getType()) );
    }

    // Rebuild before firing so that added types are still valid in the callee
    rebuildEverything();

    for (const TypeRegNode* e = head; e; e = e->getNext())
    {
        TypeDetail::fixupUnknownSpecialMethods( const_cast<hkReflect::Type*>(e->getType()) );
    }

#ifdef HK_DEBUG_SLOW
    {
        const hkResult ok = m_checker.addAndCheckTypes( head );
        HK_ASSERT(0x69fe165d, ok.isSuccess(), "Type checks failed. See " DEBUG_LOG_IDENTIFIER " logs for details.");
    }
#endif

    {
        hkArray<const Type*> added;
        for( const TypeRegNode* e = head; e; e = e->getNext() )
        {
            if( e->typeIsDuplicate() == false )
            {
                added.pushBack(e->getType());
            }
        }

        TypesChangedFuncArgs args;
        args.m_added = added;
        fireCallbacks(args);
    }
}

void hkReflect::Detail::BuiltinTypeReg::removeBatch(_Inout_ TypeRegNode* head)
{
//  HK_ASSERT_NO_MSG(0x6d505f1a,m_openForBatch);
    int idx = m_lists.indexOf(head);
    if( idx >= 0)
    {
        hkArray<const Type*> added;
        hkArray<const Type*> removed;
        for( TypeRegNode* e = m_lists[idx]; e; e = e->getNext() )
        {
            if( e->typeIsDuplicate() == false )
            {
                removed.pushBack(e->getType());

                
                
                

                // If the type had duplicates, promote the first one (added as a new type).
                int firstDup = m_duplicates.getFirstIndex(e->getType());
                if (firstDup != -1)
                {
                    added.pushBack(m_duplicates.getValue(firstDup));
                }
            }
        }
        m_lists.removeAtAndCopy(idx);

#ifdef HK_DEBUG_SLOW
        m_checker.removeTypes( removed );
#endif

        TypesChangedFuncArgs args;
        args.m_added = added;
        args.m_removed = removed;
        fireCallbacks(args);

        // Rebuild after firing so that removed types are still valid in the callee
        rebuildEverything();

        TypesChangedFuncArgs emptyArgs;
        fireCallbacks(emptyArgs);
    }
}

bool hkReflect::Detail::BuiltinTypeReg::isRegistered(_In_ TypeRegNode* head) const
{
    return m_lists.indexOf(head) != -1;
}


namespace
{
    // note first & last - not the usual "one past the end"
    static hkReflect::Detail::TypeRegNode* s_arrayToTree( const hkArrayBase<hkReflect::Detail::TypeRegNode*>& entries, int first, int last)
    {
        if (first > last)
        {
            return HK_NULL;
        }

        //   --- 3 ---
        //   |       |
        // - 1 -   - 5 -
        // |   |   |   |
        // 0 . 2 . 4 . 6  ...

        const int middle = (first + last) / 2;
        hkReflect::Detail::TypeRegNode* leftChild = s_arrayToTree(entries, first, middle - 1); //build left tree
        hkReflect::Detail::TypeRegNode* rightChild = s_arrayToTree(entries, middle + 1, last);  //build right tree

        hkReflect::Detail::TypeRegNode* parent = entries[middle];
        parent->setChildren(leftChild, rightChild);

        return parent; // == the entry at "middle"
    }

    struct IsaInfo
    {
        const hkReflect::Type* m_type;
        int m_parent;       // parent index if available, -1 otherwise
        int m_firstChild;   // child, -1 otherwise
        int m_nextSibling;  // nextSibling if available, -1 otherwise
    };

    static inline _Ret_maybenull_ const hkReflect::Type* s_getKeyType(_In_ const hkReflect::Type* type,
        _In_ const hkPointerMap<const hkReflect::Type*, const hkReflect::Type*>::Temp& origFromDup )
    {
        // Use the canonical type for the undecorated type as key.
        const hkReflect::Type* undecorated = hkReflect::TypeDetail::getUndecorated( type );
        const hkReflect::Type* canonical = origFromDup.getWithDefault( undecorated, undecorated );
        return canonical;
    }

    static int addRecursively(
        _In_ const hkReflect::Type* typeIn,
        hkArray<IsaInfo>::Temp& infos,
        hkPointerMap<const void*, int>::Temp& indexFromType,
        _In_ const hkPointerMap<const hkReflect::Type*, const hkReflect::Type*>::Temp& origFromDup)
    {
        const hkReflect::Type* type = s_getKeyType( typeIn, origFromDup );

        // Key type should be itself undecorated.
        HK_ASSERT_NO_MSG(0x2130201, hkReflect::TypeDetail::getUndecorated( type ) == type );

        // done already?
        {
            int myIndex = indexFromType.getWithDefault(type,-1);
            if( myIndex >= 0 )
            {
                return myIndex;
            }
        }

        // add parent first - this simplifies traversals later
        int parentIndex = -1;
        if( const hkReflect::Type* parent = type->getParent() )
        {
            parent = hkReflect::TypeDetail::skipDecorators(parent);
            parentIndex = addRecursively(parent, infos, indexFromType, origFromDup);
            HK_ASSERT_NO_MSG(0x1ca15c3b, parentIndex >= 0);
        }

        // now us
        int myIndex = infos.getSize();
        {
            IsaInfo myInfo;
            myInfo.m_type = type;
            myInfo.m_parent = parentIndex;
            myInfo.m_firstChild = -1;
            if( parentIndex != -1 )
            {
                IsaInfo& par = infos[parentIndex];
                myInfo.m_nextSibling = par.m_firstChild;
                par.m_firstChild = myIndex;
            }
            else
            {
                myInfo.m_nextSibling = -1;
            }

            infos.pushBack(myInfo);
            indexFromType.insert(type, myIndex);
        }

        return myIndex;
    }

    static int buildPreOrderAndRightMost( int cur, const hkArray<IsaInfo>::Temp& infos, int lastUsedId )
    {
        hkReflect::Type* type = const_cast<hkReflect::Type*>(infos[cur].m_type);

        int preOrder = lastUsedId + 1;
        int rightMost = preOrder;
        for( int i = infos[cur].m_firstChild; i >= 0; i = infos[i].m_nextSibling )
        {
            rightMost = buildPreOrderAndRightMost( i, infos, rightMost );
        }

        HK_ASSERT_NO_MSG(0x13ad007d, hkReflect::TypeDetail::localHasOptional(type, hkReflect::Opt::INHERITANCE));
        if( hkReflect::Detail::InheritanceInfo* info = hkReflect::TypeDetail::localAddressOptional<hkReflect::Opt::INHERITANCE>(type) )
        {
            info->set(preOrder, rightMost);
        }
        return rightMost;
    }


    static void s_rebuildOptIsA(hkArrayBase<hkReflect::Detail::TypeRegNode*>& entries,
        _In_ const hkPointerMap<const hkReflect::Type*, const hkReflect::Type*>::Temp& origFromDup)
    {
        hkArray<IsaInfo>::Temp infos; infos.reserve( entries.getSize() );
        hkPointerMap<const void*, int>::Temp indexFromType; indexFromType.reserve( entries.getSize() );

        // build types into a tree based on inheritance
        for( int i = 0; i < entries.getSize(); ++i )
        {
            const hkReflect::Type* type = entries[i]->getType();
            HK_ASSERT_NO_MSG(0x17beed26,type);

            addRecursively(type, infos, indexFromType, origFromDup);
        }

        for (int i = 0; i < infos.getSize(); ++i )
        {
            // we rely on parents being seen before the child to simplify traversal of preorder and rightmost
            HK_ASSERT_NO_MSG(0x7b30dcea, infos[i].m_parent < i );
        }
        //debugPrint(infos);

        int lastUsedId = 0; // reserve 0 for "no inheritance" so that hkReflect::Type::findOptional(Opt::INHERITANCE,0) works
        for (int i = 0; i < infos.getSize(); ++i )
        {
            if( infos[i].m_parent == -1 ) // only for type roots to ensure contiguous ranges
            {
                lastUsedId = buildPreOrderAndRightMost( i, infos, lastUsedId );
            }
        }
        //debugPrint(infos);

        // Fixup inheritance of decorated types (templates with typedef arguments).
        for ( auto entry : entries )
        {
            hkReflect::Type* type = entry->getType();

            if ( hkReflect::Detail::InheritanceInfo* inherit = hkReflect::TypeDetail::localAddressOptional<hkReflect::Opt::INHERITANCE>( type ) )
            {
                if ( inherit->asUlong() == 0 )
                {
                    const hkReflect::Type* keyType = s_getKeyType( type, origFromDup );

                    // Fixup the opt.
                    hkReflect::Detail::InheritanceInfo keyOpt = hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::INHERITANCE>(keyType);
                    HK_ASSERT_NO_MSG(0x1114a933, keyOpt );
                    *inherit = keyOpt;
                }
            }
        }
    }
}

namespace hkReflect { namespace Detail { namespace
{
    struct Entry
    {
        TypeRegNode* node;
        int listIndex;
    };

    struct CompareEntries
    {
        bool operator()(const Entry& a, const Entry& b)
        {
            // Compare the type name.
            int diff = hkReflect::Type::compareName(a.node->getType(), b.node->getType());
            if ( diff )
            {
                return diff < 0;
            }

            // Take the type from the older list first.
            if ( a.listIndex != b.listIndex )
            {
                return a.listIndex < b.listIndex;
            }

            // If 'a' is the resolved type of 'b', put 'a' first so that it gets selected as canonical type. This
            // avoids having a canonical type which resolves to its duplicate.
            if ( a.node->getType() != b.node->getType() &&
                a.node->getType() == hkReflect::TypeDetail::getUndecorated( b.node->getType() ) )
            {
                return true;
            }

            return false; // hkUlong(&a) - hkUlong(&b) < 0;
        }
    };
} } }

void hkReflect::Detail::BuiltinTypeReg::rebuildEverything()
{
    hkCriticalSectionLock lock(&m_lock);
    m_duplicates.clear();

    // Gather all entries and sort globally
    // Types in older lists are taken first.
    hkArray<Entry> allEntries;
    for( int i = 0; i < m_lists.getSize(); ++i )
    {
        for( TypeRegNode* e = m_lists[i]; e; e = e->getNext() )
        {
            if(e->getType())
            {
                Entry entry = { e, i };
                allEntries.pushBack(entry);
                if (hkReflect::TypeDetail::localHasOptional(e->getType(), Opt::NAME) == 0)
                {
                    HK_BREAKPOINT(0);
                }

                // Clear inheritance opt. Needed later to check which types need to copy inheritance value from others.
                if ( hkReflect::Detail::InheritanceInfo* inherit = TypeDetail::localAddressOptional<Opt::INHERITANCE>( e->getType() ) )
                {
                    *inherit = hkReflect::Detail::InheritanceInfo();
                }
            }
        }
    }
    hkSort( allEntries.begin(), allEntries.getSize(), CompareEntries() );


    // Find unique entries

    // Note on duplicates:
    // Info on which types are duplicates is stored in two places.
    // * In the type registration entry typeIsDuplicate()
    //   This is used when walking the entry lists for iterators and the like
    // * In a map m_duplicates
    //   This is used to push changes to duplicates (for instance after setting an opt on the master)
    // Each original type comes from the oldest list which contained it. This way the original in each set always
    // stays the same until it is removed from the registry.
    hkArray<TypeRegNode*>::Temp uniqueEntries;
    uniqueEntries.reserve(allEntries.getSize());
    hkPointerMap<const hkReflect::Type*, const hkReflect::Type*>::Temp origFromDup;
    origFromDup.reserve(allEntries.getSize()/2);
    if( allEntries.getSize() )
    {
        DLOG_SCOPE("Uniquify");
        const hkReflect::Type* key = allEntries[0].node->getType();
        uniqueEntries.pushBack(allEntries[0].node);
        DLOG("Type {}", key->getFullName());
        for( int i = 1; i < allEntries.getSize(); ++i )
        {
            TypeRegNode* cur = allEntries[i].node;
            bool dup = hkReflect::Type::compareName(cur->getType(), key) == 0;
            cur->markTypeDuplicate(dup);
            if( dup )
            {
                DLOG("Dupe {}", cur->getType()->getName());
                m_duplicates.insert(key, cur->getType());
                origFromDup.insert(cur->getType(), key);
                // keep key for next iteration
            }
            else
            {
                DLOG("Type {}", cur->getType()->getFullName());
                uniqueEntries.pushBack(cur);
                key = cur->getType();
            }
        }
    }

    // Rebuild tree of names
    m_nameTreeRoot = s_arrayToTree( uniqueEntries, 0, uniqueEntries.getSize()-1);

    // And the inheritance keys
    s_rebuildOptIsA(uniqueEntries, origFromDup);

    // We changed some Opt::inheritance values so...
    notifyTypesMutated();
}


void hkReflect::Detail::BuiltinTypeReg::notifyTypesMutated()
{
    for( hkSerializeMultiMap<const Type*,const Type*>::Iterator it = m_duplicates.getIterator();
        m_duplicates.isValid(it);
        it = m_duplicates.getNext(it) )
    {
        const Type* src = m_duplicates.getKey( it );
        const hkReflect::Detail::InheritanceInfo* srcInherit = TypeDetail::localAddressOptional<Opt::INHERITANCE>( src );
        for( int i = m_duplicates.getFirstIndex(src); i != -1; i = m_duplicates.getNextIndex(i) )
        {
            Type* dst = m_duplicates.getValue(i);
            HK_ASSERT_NO_MSG(0x1572efd7, hkReflect::Type::compareName(src, dst)==0);
            if( hkReflect::Detail::InheritanceInfo* dstInherit = TypeDetail::localAddressOptional<Opt::INHERITANCE>( dst ) )
            {
                HK_ASSERT_NO_MSG( 0x76b380db, srcInherit && *srcInherit);
                *dstInherit = *srcInherit;
            }
        }
    }
}

void hkReflect::Detail::BuiltinTypeReg::fireCallbacks(const TypesChangedFuncArgs& args)
{
    hkArray<SubscriptionImpl*> subscriptions;
    subscriptions = m_subscriptions;
    // copy so we can add/remove subscriptions from callbacks
    for( int i = 0; i < subscriptions.getSize(); ++i )
    {
        if( SubscriptionImpl* impl = subscriptions[i] )
        {
            (*impl->m_func)(args, impl->m_data);
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
