// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/ReflectFile/hkSimpleTokenizer.h>

namespace
{
    bool isTokenSeparator(char chr)
    {
        return chr == ';' || chr == '(' || chr == ')' || chr == '{' || chr == '}' || chr == ',' || chr == '=' || chr == '[' || chr == ']' || chr == '#';
    }

    bool isWhiteSpace(char chr)
    {
        return chr == ' ' || chr == '\t' || chr == '\r' || chr == '\v';
    }

    bool isEof(char chr)
    {
        return chr == '\0';
    }
}

hkSimpleTokenizer::hkSimpleTokenizer(_In_z_ const char* str, _In_opt_z_ const char* originalFileName /*= HK_NULL*/, hkUint32 lineOffset /*= 0*/)
    : m_originalFileName(originalFileName), m_str(str), m_offset(0), m_lineNumber(1 + lineOffset), m_preprocessorStyleStrings(false)
{
    HK_ASSERT_NO_MSG(0x666a900, m_str);
    m_strLen = hkString::strLen(str);
    // Does the string start with a utf-8 BOM?
    if (m_strLen >= 3 && hkString::strNcmp(m_str, "\xef\xbb\xbf", 3) == 0)
    {
        m_str = m_str + 3; // skip the BOM
        m_strLen -= 3;
    }
}

hkSimpleTokenizer::hkSimpleTokenizer(const hkStringView& str, _In_opt_z_ const char* originalFileName /*= HK_NULL*/, hkUint32 lineOffset /*= 0*/)
    : m_originalFileName(originalFileName), m_str(str.begin()), m_offset(0), m_lineNumber(1 + lineOffset), m_preprocessorStyleStrings(false)
{
    HK_ASSERT_NO_MSG(0x666a901, m_str);
    m_strLen = str.getSize();
    // Does the string start with a utf-8 BOM?
    if (m_strLen >= 3 && hkString::strNcmp(m_str, "\xef\xbb\xbf", 3) == 0)
    {
        m_str = m_str + 3; // skip the BOM
        m_strLen -= 3;
    }
}

bool hkSimpleTokenizer::getNext(hkSimpleToken& token)
{
    token.m_stringLiteralType = hkSimpleToken::StringType::NONE;
    token.m_length = 0;
    token.m_start = HK_NULL;

    hkUint32 length = 0;

    // This loop skips empty tokens
    do
    {

        // End of file?
        if (m_offset >= m_strLen)
            return false;

        const char* start = hkAddByteOffsetConst(m_str, m_offset);

        // If the token starts as a string we want to parse it
        if (hkSimpleToken::StringType::Enum stringType = getStringBeginDelimiter(*start))
        {
            start++;
            m_offset++;

            char previous = '\0';

            const char* current = start;

            // Find end of string
            while (!((getStringEndDelimiter(*current) == stringType) && previous != '\\') && !isEnd(current))
            {
                // Line break in string!
                if (isNewLine(*current))
                {
                    HK_WARN(0xa324424, "Malformed string literal (newline detected) in file " << m_originalFileName << " in line " << m_lineNumber);
                    return false;
                    break;
                }

                previous = *current;

                length++;
                current++;
            }

            // Token will contain the found string without the start or end delimiter characters
            token.m_start = start;
            token.m_length = length;
            token.m_lineNumber = m_lineNumber;
            token.m_stringLiteralType = stringType;

            m_offset += 1; // Skip the closing delimiter character when parsing for the next valid token
        }
        else
        {
            // Scan to next whitespace character for end of token
            const char* current = start;

            if (isTokenSeparator(*current))
            {
                token.m_start = start;
                token.m_length = 1;
                token.m_lineNumber = m_lineNumber;

                m_offset += 1;

                // Ignore #line statements (for now, actually we should be able to handle them)
                if (*current == '#')
                {
                    hkSimpleToken lineToken;
                    if (peek(lineToken) && lineToken == "line")
                    {
                        while (!isNewLine(*current) && !isEnd(current))
                        {
                            current++;
                            m_offset++;
                        }

                        length = 0;
                        continue;
                    }
                }

                return true;
            }

            while (!isWhiteSpace(*current) && !isNewLine(*current) && !isEnd(current) && !getStringBeginDelimiter(*current))
            {
                // Comment start character
                if (*current == '/')
                {
                    // Line comment - skip to next new line or end and finish
                    if (current[1] == '/')
                    {
                        hkUint32 skipped = 1; // End character of comment

                        while (!isNewLine(*current) && !isEnd(current))
                        {
                            current++;
                            skipped++;
                        }

                        m_offset += skipped;

                        break;
                    }
                    else if (current[1] == '*')
                    {
                        hkUint32 skipped = 2;
                        current += 2;

                        while (*current != '*' || current[1] != '/')
                        {
                            if (*current == '\n')
                                m_lineNumber++;

                            current++;
                            skipped++;
                        }

                        skipped += 2;
                        m_offset += skipped;

                        break;
                    }
                }

                current++;
                length++;

                if (isTokenSeparator(*current))
                    break;
            }

            token.m_length = length;
            token.m_start = start;
            token.m_lineNumber = m_lineNumber;
        }

        // And then to next non-whitespace character for offset of next token
        m_offset += length;

        if (m_offset >= m_strLen)
            continue;

        const char* current = hkAddByteOffsetConst(m_str, m_offset);

        // Skip a new line character if it was the reason we left the loop above
        // (isNewLine already counted it)
        if (*current == '\n')
        {
            current++;
            m_offset++;
        }

        while (isWhiteSpace(*current) || isNewLine(*current))
        {
            m_offset++;
            current++;
        }
    }
    while (length == 0);

    return true;
}

bool hkSimpleTokenizer::peek(hkSimpleToken& token)
{
    // Save current state
    hkUint32 offset = m_offset;
    hkUint32 lineNumber = m_lineNumber;

    // Get next token
    bool result = getNext(token);

    // Restore state
    m_offset = offset;
    m_lineNumber = lineNumber;

    return result;
}


_Ret_opt_z_ const char* hkSimpleTokenizer::getOriginalFileName() const
{
    return m_originalFileName;
}


hkSimpleToken::StringType::Enum hkSimpleTokenizer::getStringBeginDelimiter(char chr) const
{
    switch (chr)
    {
    case '\"':
        return hkSimpleToken::StringType::QUOTED;
    case '<':
        return m_preprocessorStyleStrings ? hkSimpleToken::StringType::ANGLE_BRACKETED : hkSimpleToken::StringType::NONE;
    default:
        return hkSimpleToken::StringType::NONE;
    }
}

hkSimpleToken::StringType::Enum hkSimpleTokenizer::getStringEndDelimiter(char chr) const
{
    switch (chr)
    {
    case '\"':
        return hkSimpleToken::StringType::QUOTED;
    case '>':
        return m_preprocessorStyleStrings ? hkSimpleToken::StringType::ANGLE_BRACKETED : hkSimpleToken::StringType::NONE;
    default:
        return hkSimpleToken::StringType::NONE;
    }
}

bool hkSimpleTokenizer::isEnd(_In_z_ const char* chr) const
{
    return (hkLosslessCast<hkUint32>(chr - m_str) == m_strLen) || isEof(*chr);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
