// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/ReflectFile/hkSimpleToken.h>


bool hkSimpleToken::copyToBuffer(_Out_writes_opt_(maxLength) char* buffer, hkUint32 maxLength) const
{
    if(!buffer || maxLength <= m_length)
        return false;

    hkMemUtil::memCpy(buffer, m_start, m_length);
    buffer[m_length] = '\0';

    return true;
}

bool hkSimpleToken::isValidIdentifier() const
{
    if(m_length == 0)
        return false;

    for(hkUint32 i = 0; i < m_length; i++)
    {
        char current = m_start[i];

        if(!((current >= 'A' && current <= 'Z') || (current >= 'a' && current <= 'z') || (current >= '0' && current <= '9' && i > 0) || current == '_'))
        {
            return false;
        }
    }

    return true;
}

bool hkSimpleToken::isValidInteger( hkInt32& result ) const
{
    // Hex
    bool hex = startsWith("0x"), negHex = (!hex && startsWith("-0x"));
    if(hex || negHex)
    {
        hkInt32 resultValue = 0;

        const hkUint32 prefixLength = hex ? 2 : 3;

        for(hkUint32 i = prefixLength; i < m_length; i++)
        {
            const char current = m_start[i];

            resultValue *= 16;

            if(current >= '0' && current <= '9')
            {
                resultValue += (current - '0');
            }
            else if(current >= 'A' && current <= 'F')
            {
                resultValue += (current - 'A' + 10);
            }
            else if(current >= 'a' && current <= 'f')
            {
                resultValue += (current - 'a' + 10);
            }
            else
            {
                return false;
            }
        }

        if(negHex)
            result = -resultValue;
        else
            result = resultValue;

        return true;
    }

    bool bin = startsWith("0b"), negBin = (!bin && startsWith("-0b"));

    if(bin || negBin)
    {
        hkInt32 resultValue = 0;

        const hkUint32 prefixLength = bin ? 2 : 3;

        for(hkUint32 i = prefixLength; i < m_length; i++)
        {
            const char current = m_start[i];

            resultValue = resultValue << 1;

            if(current == '0' || current == '1')
            {
                resultValue |= (current - '0');
            }
            else
            {
                return false;
            }
        }

        if(negBin)
            result = -resultValue;
        else
            result = resultValue;

        return true;
    }
    // Generic integers
    else
    {
        bool negInt = (*m_start == '-');

        hkInt32 resultValue = 0;

        for(hkUint32 i = negInt ? 1 : 0; i < m_length; i++)
        {
            const char current = m_start[i];

            resultValue *= 10;

            if(current >= '0' && current <= '9')
            {
                resultValue += (current - '0');
            }
            else
            {
                return false;
            }
        }

        if(negInt)
            result = -resultValue;
        else
            result = resultValue;

        return true;
    }
}

// This function is a bit more complicated since atof / strtod don't return errors when parsing float values
// It therefore implements a small state machine to parse the float value for correctness but then uses atof
// for the actual conversion to guarantee precision
bool hkSimpleToken::isValidFloat( hkReal& result ) const
{
    bool negative = false;

    enum FloatPart
    {
        Sign,
        Integer,
        Fraction,
        Exponent
    };

    FloatPart currentPart = Sign;

    hkUint32 numberStart = 0;

    hkUint32 charIdx = 0;
    while(charIdx < m_length)
    {
        char current = m_start[charIdx];

        switch(currentPart)
        {
            case Sign:

                if(current == '-')
                {
                    negative = !negative;
                    charIdx++;
                    numberStart++;
                }
                else if(current == '+')
                {
                    charIdx++; // No sign change
                    numberStart++;
                }
                else if(current == '.')
                {
                    charIdx++;
                    currentPart = Fraction;
                    continue;
                }
                else if(current >= '0' && current <= '9')
                {
                    currentPart = Integer;
                    continue;
                }
                else
                {
                    return false;
                }

                break;

            case Integer:

                if(current >= '0' && current <= '9')
                {
                    charIdx++;
                }
                else if(current == '.')
                {
                    currentPart = Fraction;
                    charIdx++;
                    continue;
                }
                else if(current == 'e' || current == 'E')
                {
                    currentPart = Exponent;
                    charIdx++;
                    continue;
                }
                else if(current == 'f' && charIdx == (m_length - 1))
                {
                    charIdx++;
                    continue;
                }
                else
                {
                    return false;
                }

                break;

            case Fraction:

                if(current >= '0' && current <= '9')
                {
                    charIdx++;
                }
                else if(current == 'e' || current == 'E')
                {
                    currentPart = Exponent;
                    charIdx++;
                    continue;
                }
                else if(current == 'f' && charIdx == (m_length - 1))
                {
                    charIdx++;
                    continue;
                }
                else
                {
                    break;
                }

            case Exponent:

                if(current >= '0' && current <= '9')
                {
                    charIdx++;
                }
                else if(current == '-' || current == '+')
                {
                    charIdx++;
                }
                else if(current == 'f' && charIdx == (m_length - 1))
                {
                    charIdx++;
                    continue;
                }
                else
                {
                    return false;
                }

                break;
        }
    }

    // At the end there may be a f
    if(charIdx == m_length)
    {
        char numberBuf[64];
        copyToBuffer(numberBuf);

        if(numberBuf[m_length] == 'f')
            numberBuf[m_length] = ' ';

        hkReal temp = hkString::atof(numberBuf + numberStart);

        result = negative ? -temp : temp;

        return true;
    }

    return false;
}

bool hkSimpleToken::isValidBool( bool& result ) const
{
    const hkSimpleToken& thisToken = *this;

    if(thisToken == "true" || thisToken == "True")
    {
        result = true;
        return true;
    }
    else if(thisToken == "false" || thisToken == "False")
    {
        result = false;
        return true;
    }

    return false;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
