// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Reflect/ReflectFile/hkSimpleTokenizer.h>

/// This class implements the recursive descent parser for .reflect files.
/// A reflect file might look like this:
///
/*
hkAdditionalAttributesFor(someClass)
{
    someField hk::Ui_Expert;

    group hk::Ui_Group("Example Group")
    {
        someOtherField;
        someMoreField hk::AbsMin(0), hkAbsMax(1);

        group hk::Ui_Expert
        {
            yetAnotherField;
            yetYetAnotherField;
        }
    }
}
*/
class HK_EXPORT_COMMON hkReflectFileParser
{
    public:

        /// Input string to parse and the original file name. The file name is only used for error reporting.
        /// The result output references the inputs through string views.
        /// As a result the lifetime of the output should be coupled to the lifetime of the input string.
        hkReflectFileParser(_In_z_ const char* input, _In_z_ const char* originalFileName, hkUint32 lineOffset = 0 );
        hkReflectFileParser( const hkStringView& input, _In_z_ const char* originalFileName, hkUint32 lineOffset = 0 );
        ~hkReflectFileParser();

        /// A attribute on a property
        struct Attribute
        {
            hkStringView m_name;
            hkStringView m_arguments;
            hkStringView m_wholeString;
        };

        /// A property of a class / struct
        struct Property
        {
            hkStringView m_name;
            hkArray<Attribute> m_attributes;
        };

        /// A class / struct declaration
        struct HK_EXPORT_COMMON ClassDeclaration
        {
            hkStringView m_name;
            hkArray<Attribute> m_attributes;
            hkArray<Property> m_properties;
        };

        struct HK_EXPORT_COMMON Error
        {
            Error();
            void set(_In_z_ const char* fileName, int line, _Printf_format_string_ const char* format, ...);
            void get(hkStringBuf& buf) const;
            void clear();
            bool isSet() const;

            const char* m_fileName;
            int m_line;
            hkStringBuf m_message;
        };

        /// Parse the given input.
        /// Outputs an error via the given stringBuf on HK_FAILURE.
        /// Returns HK_SUCCESS otherwise.
        hkResult parse( Error& errorMessage );

        /// Getter for the parsing result. The lifetime of the results is coupled to the lifetime of the parser.
        const hkArray<ClassDeclaration>& getClassDeclarations() const;

    private:

        hkResult parseClassSection( Error& errorMessage );
        hkResult parseGroup( Error& errorMessage, ClassDeclaration& classDeclaration, hkArray<Attribute>& additionalAttribute );
        hkResult parseVar( Error& errorMessage, ClassDeclaration& classDeclaration, hkArray<Attribute>& additionalAttributes );
        hkResult parseSingleAttribute( Error& errorMessage, hkArray<Attribute>& appendTo );
        hkResult parseAttributeList( Error& errorMessage, _In_z_ const char* terminator, hkArray<Attribute>& appendTo );
        hkResult parseScope( Error& errorMessage, ClassDeclaration& classDeclaration, hkArray<Attribute>& additionalAttributes );
        hkStringView addAdditionalStringData(_In_z_ const char* additionalStringData );

        bool accept(_In_z_ const char* token );
        bool skip();

        hkArray<ClassDeclaration> m_classDeclarations;
        hkSimpleTokenizer m_tokenizer;
        hkSimpleToken m_currentToken;


        // this holds strings that have been generated during parsing so someone owns the data.
        // this is only needed in rare cases, mostly just string views of the input data are used.
        // it needs to be a const char* because otherwise it won't work in hkPreBuild
        hkArray<const char*> m_additionalStringData;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
