// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/ReflectFile/hkReflectFileParser.h>
#if !defined(HK_INCLUDED_FROM_PREBUILD)
#include <Common/Base/Fwd/hkcstdarg.h>
#endif

namespace
{
    struct ScopedAttributeCleanup
    {
        private:
        hkArray<hkReflectFileParser::Attribute>& m_attributeArray;
        int m_startSize;

        public:
        ScopedAttributeCleanup( hkArray<hkReflectFileParser::Attribute>& attributeArray )
            : m_attributeArray( attributeArray )
        {
            m_startSize = attributeArray.getSize();
        }

        ~ScopedAttributeCleanup()
        {
            HK_ASSERT_NO_MSG(0x3613b5fe, m_attributeArray.getSize() >= m_startSize );
            m_attributeArray.setSize( m_startSize );
        }
    };

    bool _isValidCppIdentifier( const hkSimpleToken& token )
    {
        const hkUint32 tokenLength = token.getLength();
        const char* tokenStr = token.getStart();

        if (tokenLength == 0)
            return false;

        if (!((tokenStr[0] >= 'a' && tokenStr[0] <= 'z')
            || (tokenStr[0] >= 'A' && tokenStr[0] <= 'Z')
            || (tokenStr[0] == '_')))
        {
            return false;
        }

        for (hkUint32 i = 1; i < tokenLength; i++)
        {
            if (!((tokenStr[0] >= 'a' && tokenStr[0] <= 'z')
                || (tokenStr[0] >= 'A' && tokenStr[0] <= 'Z')
                || (tokenStr[0] == '_')
                || (tokenStr[i] >= '0' && tokenStr[i] <= '9')))
            {
                return false;
            }
        }

        return true;
    }
}

hkReflectFileParser::hkReflectFileParser( _In_z_ const char* input, _In_z_ const char* originalFileName, hkUint32 lineOffset )
    : m_tokenizer(input, originalFileName, lineOffset )
{

}

hkReflectFileParser::hkReflectFileParser( const hkStringView& input, _In_z_ const char* originalFileName, hkUint32 lineOffset )
    : m_tokenizer(input, originalFileName, lineOffset )
{

}

hkReflectFileParser::~hkReflectFileParser()
{
    const int numAdditionalStrings = m_additionalStringData.getSize();
    for (int i = 0; i < numAdditionalStrings; i++)
    {
        delete[] m_additionalStringData[i];
    }
}

hkResult hkReflectFileParser::parse( Error& errorMessage )
{
    if (!m_tokenizer.getNext( m_currentToken ))
    {
        // empty input file
        return HK_SUCCESS;
    }

    while (m_currentToken.isValid())
    {
        HK_RETURN_IF_FAILED(parseClassSection(errorMessage));
    }

    return HK_SUCCESS;
}

const hkArray<hkReflectFileParser::ClassDeclaration>& hkReflectFileParser::getClassDeclarations() const
{
    return m_classDeclarations;
}

hkResult hkReflectFileParser::parseClassSection( Error& errorMessage )
{
    if (!accept( "hkAdditionalAttributesFor" ))
    {
        errorMessage.set(m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Expected 'hkAdditionalAttributesFor'" );
        return HK_FAILURE;
    }

    hkSimpleToken startingParanthesis = m_currentToken;
    if (!accept( "(" ))
    {
        errorMessage.set(m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Expected '(' after 'hkAdditionalAttributesFor'" );
        return HK_FAILURE;
    }

    if (!_isValidCppIdentifier(m_currentToken))
    {
        errorMessage.set(m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Expected identifier after 'hkAdditionalAttributesFor('" );
        return HK_FAILURE;
    }

    ClassDeclaration& classDeclaration = m_classDeclarations.expandOne();
    classDeclaration.m_name = hkStringView( m_currentToken.getStart(), m_currentToken.getLength() );
    skip();

    if (!accept( ")" ))
    {
        errorMessage.set(m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Missing ')' for '(' opened in line %d.", startingParanthesis.getLineNumber() );
        return HK_FAILURE;
    }

    if ( accept( ":" ) )
    {
        parseAttributeList( errorMessage, "{", classDeclaration.m_attributes );
    }

    hkSimpleToken startingBrace = m_currentToken;
    if (!accept( "{" ))
    {
        errorMessage.set(m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Expected '{' instead of '%.*s'", m_currentToken.getLength(), m_currentToken.getStart() );
        return HK_FAILURE;
    }

    hkArray<Attribute> additionalAttributes;

    HK_RETURN_IF_FAILED(parseScope(errorMessage, classDeclaration, additionalAttributes));

    if (!accept( "}" ))
    {
        errorMessage.set(m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Missing '}' for '{' opened in line %d.", startingBrace.getLineNumber() );
        return HK_FAILURE;
    }

    return HK_SUCCESS;
}

hkResult hkReflectFileParser::parseGroup( Error& errorMessage, ClassDeclaration& classDeclaration, hkArray<Attribute>& additionalAttributes )
{
    ScopedAttributeCleanup cleanup( additionalAttributes );

    hkSimpleToken groupStart = m_currentToken;
    if (!accept( "group" ))
    {
        errorMessage.set(m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Expected 'group'." );
        return HK_FAILURE;
    }

    while (m_currentToken.isValid() && m_currentToken != "{")
    {
        HK_RETURN_IF_FAILED(parseSingleAttribute(errorMessage, additionalAttributes));

        if (m_currentToken == ",")
        {
            skip();
        }
        else if (m_currentToken != "{")
        {
            Attribute& lastAttribute = additionalAttributes.back();
            errorMessage.set(m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Expected ',' or '{' after '%.*s'.", lastAttribute.m_wholeString.getSize(), lastAttribute.m_wholeString.begin() );
            return HK_FAILURE;
        }
    }

    hkSimpleToken startBrace = m_currentToken;
    if (!accept( "{" ))
    {
        errorMessage.set( m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Missing '{' for group started in line %d.", groupStart.getLineNumber() );
        return HK_FAILURE;
    }

    HK_RETURN_IF_FAILED(parseScope(errorMessage, classDeclaration, additionalAttributes));

    if (!accept( "}" ))
    {
        errorMessage.set(m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Missing closing '}' for '{' opened in line %d.", startBrace.getLineNumber() );
        return HK_FAILURE;
    }

    return HK_SUCCESS;
}

hkResult hkReflectFileParser::parseVar( Error& errorMessage, ClassDeclaration& classDeclaration, hkArray<Attribute>& additionalAttributes )
{
    if (!m_currentToken.isValidIdentifier())
    {
        errorMessage.set(m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Expected identifier." );
        return HK_FAILURE;
    }

    hkSimpleToken propertyName = m_currentToken;
    Property& prop = classDeclaration.m_properties.expandOne();
    prop.m_name = hkStringView( propertyName.getStart(), propertyName.getLength() );
    skip();

    parseAttributeList( errorMessage, ";", prop.m_attributes );

    prop.m_attributes.append( additionalAttributes.begin(), additionalAttributes.getSize() );

    // if we don't have manual ordering automatically add a correct order attribute.
    bool hasOrderAttribute = false;
    const int numAttributes = prop.m_attributes.getSize();
    for (int i = 0; i < numAttributes; i++)
    {
        if (prop.m_attributes[i].m_name == "hk::Ui_Order")
        {
            hasOrderAttribute = true;
            break;
        }
    }

    if (!hasOrderAttribute)
    {
        Attribute& orderAttribute = prop.m_attributes.expandOne();

        hkStringBuf order;
        order.printf("hk::Ui_Order");
        int nameLength = order.getLength();
        order.appendPrintf( "(%d)", classDeclaration.m_properties.getSize() );

        orderAttribute.m_wholeString = addAdditionalStringData( order.cString() );
        orderAttribute.m_name = orderAttribute.m_wholeString.slice(0, nameLength);
        orderAttribute.m_arguments = orderAttribute.m_wholeString.ltrim(nameLength + 1).rtrim(1);
    }

    if (!accept( ";" ))
    {
        errorMessage.set(m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Expected ';' to finish property '%.*s' started in line %d", propertyName.getLength(), propertyName.getStart(), propertyName.getLineNumber() );
        return HK_FAILURE;
    }

    return HK_SUCCESS;
}

hkResult hkReflectFileParser::parseAttributeList( Error& errorMessage, _In_z_ const char* terminator, hkArray<Attribute>& appendTo )
{
    while ( m_currentToken.isValid() && m_currentToken != terminator )
    {
        HK_RETURN_IF_FAILED(parseSingleAttribute(errorMessage, appendTo));
        Attribute& lastAttribute = appendTo.back();

        if ( !accept( "," ) )
        {
            if ( !m_currentToken.isValid() || m_currentToken != terminator )
            {
                errorMessage.set( m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Expected ',' or '%c' after '%.*s'.", terminator, lastAttribute.m_wholeString.getSize(), lastAttribute.m_wholeString.begin() );
                return HK_FAILURE;
            }
        }
    }
    return HK_SUCCESS;
}

hkResult hkReflectFileParser::parseSingleAttribute( Error& errorMessage, hkArray<Attribute>& appendTo )
{
    if (!_isValidCppIdentifier(m_currentToken))
    {
        errorMessage.set(m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Expected identifier. '%.*s' is not a valid identifier.", m_currentToken.getLength(), m_currentToken.getStart() );
        return HK_FAILURE;
    }
    hkSimpleToken nameToken = m_currentToken;

    Attribute& attr = appendTo.expandOne();
    attr.m_name = hkStringView(m_currentToken.getStart(), m_currentToken.getLength());
    attr.m_wholeString = attr.m_name;
    skip();

    hkSimpleToken startParenthesis = m_currentToken;
    if (!accept( "(" ))
    {
        return HK_SUCCESS;
    }

    hkUint32 parenthesesCount = 1;
    hkSimpleToken endParenthesis;
    hkSimpleToken lastToken = startParenthesis;
    hkStringBuf escapedAttribute;
    while (m_currentToken.isValid() && parenthesesCount > 0)
    {
        if (m_currentToken == "(")
        {
            parenthesesCount++;
        }
        else if (m_currentToken == ")")
        {
            endParenthesis = m_currentToken;
            parenthesesCount--;
        }
        else if(m_currentToken.getStringLiteralType() == hkSimpleToken::StringType::QUOTED)
        {
            // Check if we have any non ascii characters inside a string literal and convert them to escaped utf-8
            const hkArrayView<const hkUint8> stringLiteral( reinterpret_cast<const hkUint8*>(m_currentToken.getStart()), m_currentToken.getLength() );
            int firstNonAsciiChar = 0;
            for (; firstNonAsciiChar < stringLiteral.getSize(); firstNonAsciiChar++)
            {
                if (stringLiteral[firstNonAsciiChar] > 127)
                {
                    break;
                }
            }

            if (firstNonAsciiChar < stringLiteral.getSize())
            {
                // We found the first thing we need to escape
                if (escapedAttribute.getLength() == 0)
                {
                    escapedAttribute.append( attr.m_name.begin(), (int)(m_currentToken.getStart() - attr.m_name.begin() - 1) );
                }
                escapedAttribute.append( lastToken.getEnd(), (int)((m_currentToken.getStart() + firstNonAsciiChar) - lastToken.getEnd()) );
                bool lastWasNonAscii = false;
                for (int i = firstNonAsciiChar; i < stringLiteral.getSize(); i++)
                {
                    if (stringLiteral[i] > 127)
                    {
                        escapedAttribute.appendPrintf( "\\x%X", stringLiteral[i] );
                        lastWasNonAscii = true;
                    }
                    else
                    {
                        // If we have a escaped character in a string we need to end the string otherwise some compilers gets confused. Example
                        // "\x0bb" should be two characters \x0b and 'b' but this does not work in msvc
                        // so we replace it with "\x0b""b" which does work.
                        if (lastWasNonAscii)
                            escapedAttribute.append( "\"\"", 2 );
                        escapedAttribute.append( m_currentToken.getStart() + i, 1 );
                        lastWasNonAscii = false;
                    }
                }
            }
        }

        if (escapedAttribute.getLength() > 0 && m_currentToken.getStringLiteralType() != hkSimpleToken::StringType::QUOTED)
        {
            escapedAttribute.append( lastToken.getEnd(), (int)(m_currentToken.getEnd() - lastToken.getEnd()) );
        }

        lastToken = m_currentToken;
        skip();
    }

    if (parenthesesCount > 0)
    {
        errorMessage.set( m_tokenizer.getOriginalFileName(), m_currentToken.getLineNumber(), "Unbalanced parenthesis. Missing ')' for '(' started in line %d.", startParenthesis.getLineNumber() );
        return HK_FAILURE;
    }

    if (escapedAttribute.getLength() > 0)
    {
        attr.m_wholeString = addAdditionalStringData( escapedAttribute.cString() );
        hkLong startOfArguments = startParenthesis.getStart() + 1 - nameToken.getStart();
        attr.m_arguments = hkStringView( attr.m_wholeString.begin() + startOfArguments, attr.m_wholeString.end() - 1 );
    }
    else
    {
        attr.m_arguments = hkStringView( startParenthesis.getStart() + 1, endParenthesis.getStart() );
        attr.m_wholeString = hkStringView( attr.m_name.begin(), endParenthesis.getStart() + 1 );
    }

    return HK_SUCCESS;
}

hkResult hkReflectFileParser::parseScope( Error& errorMessage, ClassDeclaration& classDeclaration, hkArray<Attribute>& additionalAttributes )
{
    while (m_currentToken.isValid() && m_currentToken != "}")
    {
        if (m_currentToken == "group")
        {
            HK_RETURN_IF_FAILED(parseGroup(errorMessage, classDeclaration, additionalAttributes));
        }
        else
        {
            HK_RETURN_IF_FAILED(parseVar(errorMessage, classDeclaration, additionalAttributes));
        }
    }

    return HK_SUCCESS;
}

bool hkReflectFileParser::accept(_In_z_ const char* token )
{
    if (!m_currentToken.isValid())
        return false;

    if (m_currentToken == token)
    {
        m_tokenizer.getNext( m_currentToken );
        return true;
    }

    return false;
}

bool hkReflectFileParser::skip()
{
    if (!m_currentToken.isValid())
        return false;

    m_tokenizer.getNext( m_currentToken );
    return true;
}

hkReflectFileParser::Error::Error()
{
    clear();
}

void hkReflectFileParser::Error::clear()
{
    m_fileName = HK_NULL;
    m_line = -1;
    m_message.clear();
}

void hkReflectFileParser::Error::set(_In_z_ const char* fileName, int line, _Printf_format_string_ const char* format, ...)
{
    m_fileName = fileName;
    m_line = line;
    va_list args;
    va_start(args, format);
    m_message.vprintf(format, args);
    va_end(args);
}

void hkReflectFileParser::Error::get(hkStringBuf& buf) const
{
    HK_ASSERT_NO_MSG(0x405f886f, m_fileName != HK_NULL);
    buf.printf("%s(%d): %s", m_fileName, m_line, m_message.cString());
}

bool hkReflectFileParser::Error::isSet() const
{
    return m_message.getLength() > 0;
}

hkStringView hkReflectFileParser::addAdditionalStringData(_In_z_ const char* additionalStringData)
{
    int len = hkString::strLen( additionalStringData );
    HK_ASSERT_NO_MSG(0x215f7db7, len > 0 );
    char* data = new char[len + 1];
    hkMemUtil::memCpy( data, additionalStringData, len + 1 );
    m_additionalStringData.pushBack( data );
    return hkStringView( data, (hkUlong)len );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
