// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

namespace hkReflect
{
namespace Detail
{
    /// Implements allocation/deallocation through the Havok heap allocator.
    struct HK_EXPORT_COMMON HeapAllocImpl : public AllocationImpl
    {
        HK_DECLARE_CLASS(HeapAllocImpl, New);

        HeapAllocImpl() {}

        virtual Var allocate(_In_ const hkReflect::Type* type) const HK_OVERRIDE;
        virtual Var allocateForClone(const Var& src, _In_ const Type* type) const HK_OVERRIDE;
        virtual void deallocate(_Inout_ void* target, _In_ const hkReflect::Type* type) const HK_OVERRIDE;

        static const HeapAllocImpl s_instance;
    };

    //
    // Simple values
    //

    template<typename STORAGE>
    struct HK_EXPORT_COMMON BoolImplN : public BoolImpl
    {
        BoolImplN() { /*keep clang happy*/ }
        virtual ~BoolImplN();
        virtual hkResult setValue(_Inout_ void* addr, _In_ const BoolType* type, BoolValue val) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* addr, _In_ const BoolType* type, _Out_ BoolValue* val) const HK_OVERRIDE;
        static const BoolImplN s_instance;
    };

    template<typename STORAGE>
    struct HK_EXPORT_COMMON IntImplN : public IntImpl
    {
        IntImplN() {/* keep clang happy*/ }
        virtual ~IntImplN();
        virtual hkResult setValue(_Inout_ void* addr, _In_ const hkReflect::IntType* type, const IntValue& val) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* addr, _In_ const hkReflect::IntType* type, _Out_ IntValue* val) const HK_OVERRIDE;
        static const IntImplN s_instance;
    };

        // The implementation is a bit special in that it allows values from -127 to +255 and converts to the
        // unsigned char representation. This is to allow interchange of signed and unsigned chars
    template<>
    struct HK_EXPORT_COMMON IntImplN<char> : public IntImpl
    {
        IntImplN() {/* keep clang happy*/ }
        virtual ~IntImplN();
        virtual hkResult setValue(_Inout_ void* addr, _In_ const hkReflect::IntType* type, const hkReflect::IntValue& val) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* addr, _In_ const hkReflect::IntType* type, _Out_ IntValue* val) const HK_OVERRIDE;
        static const IntImplN s_instance;
    };

    template<typename STORAGE>
    struct HK_EXPORT_COMMON FloatImplN : public FloatImpl
    {
        FloatImplN() {/*keep clang happy*/}
        virtual ~FloatImplN();
        virtual hkResult setValue(_Inout_ void* addr, _In_ const hkReflect::FloatType* type, hkReflect::FloatValue val) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* addr, _In_ const FloatType* type, _Out_ FloatValue* val) const HK_OVERRIDE;
        static const FloatImplN s_instance;
    };

    template<>
    struct HK_EXPORT_COMMON FloatImplN<hkHalf16> : public FloatImpl
    {
        FloatImplN() { /* keep clang happy */ }
        virtual ~FloatImplN();
        virtual hkResult setValue(_Inout_ void* addr, _In_ const hkReflect::FloatType* type, hkReflect::FloatValue val) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* addr, _In_ const FloatType* type, _Out_ FloatValue* val) const HK_OVERRIDE;
        static const FloatImplN<hkHalf16> s_instance;
    };

    //
    // C arrays
    //

        /// Used for fixed size C-style arrays such as "int x[10];"
    struct HK_EXPORT_COMMON RepeatImpl : public ArrayImpl
    {
        HK_DECLARE_CLASS(RepeatImpl, New);
        RepeatImpl();

        hkResult getValue(_In_ const void* arrAddr, _In_ const ArrayType* arrType, _Out_ ArrayValue* val) const HK_OVERRIDE;
        hkResult setNumElements(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, int len) const HK_OVERRIDE;
        AllocResult allocateElements(_Inout_ void* arrAddr, _In_ const ArrayType* arrType, QualType elemType, int len) const HK_OVERRIDE;

        static const RepeatImpl s_instance;
    };

    //
    // String
    //

        // Plain old char*. The lifetime of the string is assumed to be managed elsewhere (usually static const)
    struct HK_EXPORT_COMMON StaticStringImpl : public StringImpl
    {
        StaticStringImpl() { /* keep clang happy */ }
        static const StaticStringImpl s_instance;

        virtual hkResult setValue(_Inout_ void* addr, _In_ const hkReflect::StringType* type, StringValue newValue) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* addr, _In_ const hkReflect::StringType* type, _Out_ StringValue* val) const HK_OVERRIDE;
    };

        // Plain old char[N]. The lifetime of the string is assumed to be managed elsewhere (usually static const)
    struct HK_EXPORT_COMMON CharBufferImpl : public StringImpl
    {
        CharBufferImpl() { /* keep clang happy */ }
        static const CharBufferImpl s_instance;

        virtual hkResult setValue(_Inout_ void* addr, _In_ const hkReflect::StringType* type, StringValue newValue) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* addr, _In_ const hkReflect::StringType* type, _Out_ StringValue* val) const HK_OVERRIDE;
    };

    //
    // Pointer
    //

    struct HK_EXPORT_COMMON RawPointerImpl : public PointerImpl
    {
        RawPointerImpl() { /* keep clang happy */ }
        static const RawPointerImpl s_instance;

        static HK_INLINE _Ret_notnull_ void* getInline(_In_ const void* self) { return *(void* const*)self; }

        virtual hkResult setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* type, const Var& var) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* self, _In_ const hkReflect::PointerType* type, _Out_ Var* val) const HK_OVERRIDE;
        virtual hkResult inplaceFixup(_Inout_ void* self, _In_ const Type* selfType, _Inout_ void* target, _In_ const Type* targetType, int count) const HK_OVERRIDE;
    };

    //
    // Properties
    //

    
    template<typename T>
    struct GetterReturnType
    {
        typedef T Type;
    };

    template<typename T>
    struct GetterReturnType<const T&>
    {
        typedef T Type;
    };

    template<typename T>
    struct GetterReturnType < T* >
    {
        typedef const T* Type;
    };


    template<typename T>
    struct GetterSetterType
    {
        typedef const T& Type;
    };

    template<typename T>
    struct GetterSetterType < T& >
    {
        typedef T& Type;
    };

    template<typename T>
    struct GetterSetterType < const T& >
    {
        typedef const T& Type;
    };

    template<typename T>
    struct GetterSetterType < T* >
    {
        typedef T* Type;
    };

    template<typename T>
    struct GetterSetterType < const T*>
    {
        typedef T* Type;
    };

    template<>
    struct GetterSetterType < const char* >
    {
        typedef const char* Type;
    };

    // Small utility functions to have a type-safe call to an erased type's method.
    template<typename MethodClass, typename ReturnType, ReturnType (MethodClass::*Getter)() const>
    static typename GetterReturnType<ReturnType>::Type HK_CALL erasedGetterCall(_In_ const void* obj)
    {
        return (static_cast<const MethodClass*>(obj)->*Getter)();
    }

    template<typename ReturnType>
    struct erasedSetterCall;

    template<>
    struct erasedSetterCall<bool>
    {
        template<typename MethodClass, typename ValueType, bool (MethodClass::*Setter)(ValueType)>
        static bool HK_CALL call(_Inout_ void* obj, typename GetterSetterType<ValueType>::Type v)
        {
            return (static_cast<MethodClass*>(obj)->*Setter)(v);
        }
    };

    template<>
    struct erasedSetterCall<void>
    {
        template<typename MethodClass, typename ValueType, void (MethodClass::*Setter)(ValueType)>
        static bool HK_CALL call(_Inout_ void* obj, typename GetterSetterType<ValueType>::Type v)
        {
            (static_cast<MethodClass*>(obj)->*Setter)(v);
            return true;
        }
    };

    // Simple trait to avoid repeating these types
    template<typename T> struct MethodTypes
    {
        typedef typename GetterReturnType<T>::Type (*GetterType)(const void*);
        typedef bool(*SetterType)(void*, typename GetterSetterType<T>::Type);
    };

    template<typename INT_TYPE>
    struct IntegerPropertyImpl : IntImpl
    {
        typedef INT_TYPE ValueType;
        typedef typename MethodTypes<ValueType>::GetterType GetterType;
        typedef typename MethodTypes<ValueType>::SetterType SetterType;

        GetterType m_get;
        SetterType m_set;

        IntegerPropertyImpl(GetterType mg, SetterType ms) : m_get(mg), m_set(ms) {}

        virtual hkResult setValue(_Inout_ void* addr, _In_ const hkReflect::IntType* type, const hkReflect::IntValue& val) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* addr, _In_ const hkReflect::IntType* type, _Out_ IntValue* val) const HK_OVERRIDE;
    };

    template<typename ENUM_TYPE>
    struct EnumPropertyImpl : IntegerPropertyImpl< typename hkTrait::CorrespondingIntType<ENUM_TYPE>::Type >
    {
        typedef IntegerPropertyImpl< typename hkTrait::CorrespondingIntType<ENUM_TYPE>::Type > Parent;
        typedef ENUM_TYPE ValueType;
        typedef typename MethodTypes<ValueType>::GetterType GetterTypeEnum;
        typedef typename MethodTypes<ValueType>::SetterType SetterTypeEnum;

        EnumPropertyImpl(GetterTypeEnum mg, SetterTypeEnum ms)
            : Parent(reinterpret_cast<typename Parent::GetterType>(mg), reinterpret_cast<typename Parent::SetterType>(ms)) {}
    };

    struct HK_EXPORT_COMMON StringPropertyImpl : public StringImpl
    {
        typedef const char* ValueType;
        typedef MethodTypes<ValueType>::GetterType GetterType;
        typedef MethodTypes<ValueType>::SetterType SetterType;

        GetterType m_get;
        SetterType m_set;

        StringPropertyImpl(GetterType mg, SetterType ms) : m_get(mg), m_set(ms) {}

        virtual hkResult setValue(_Inout_ void* addr, _In_ const hkReflect::StringType* type, hkReflect::StringValue newValue) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* addr, _In_ const hkReflect::StringType* type, _Out_ StringValue* val) const HK_OVERRIDE;
    };

    template<typename FP_TYPE>
    struct FloatPropertyImpl : public FloatImpl
    {
        typedef FP_TYPE ValueType;
        typedef typename MethodTypes<ValueType>::GetterType GetterType;
        typedef typename MethodTypes<ValueType>::SetterType SetterType;

        GetterType m_get;
        SetterType m_set;

        FloatPropertyImpl(GetterType mg, SetterType ms) : m_get(mg), m_set(ms) {}

        virtual hkResult setValue(_Inout_ void* addr, _In_ const hkReflect::FloatType* type, hkReflect::FloatValue val) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* addr, _In_ const FloatType* type, _Out_ FloatValue* val) const HK_OVERRIDE;
    };

    struct HK_EXPORT_COMMON BoolPropertyImpl : public BoolImpl
    {
        typedef bool ValueType;
        typedef MethodTypes<ValueType>::GetterType GetterType;
        typedef MethodTypes<ValueType>::SetterType SetterType;

        GetterType m_get;
        SetterType m_set;

        BoolPropertyImpl(GetterType mg, SetterType ms) : m_get(mg), m_set(ms) {}

        virtual hkResult setValue(_Inout_ void* addr, _In_ const hkReflect::BoolType* type, hkReflect::BoolValue) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* addr, _In_ const BoolType* type, _Out_ BoolValue* val) const HK_OVERRIDE;
    };

    template<typename PointerTypeC, typename PointeeTypeC>
    struct PointerPropertyImpl : public PointerImpl
    {
        typedef typename hkTrait::RemoveConstPtr<PointerTypeC>::Type PointerType;
        typedef typename hkTrait::RemoveConst<PointeeTypeC>::type PointeeType;
        typedef PointerType ValueType;

        typedef typename MethodTypes<ValueType>::GetterType GetterType;
        typedef typename MethodTypes<ValueType>::SetterType SetterType;

        GetterType m_get;
        SetterType m_set;

        PointerPropertyImpl(GetterType mg, SetterType ms) : m_get(mg), m_set(ms) {}

        
        //         A better way would be to list all possible prototypes somehow...
        template<typename G, typename S>
            PointerPropertyImpl(G mg, S ms) : m_get(reinterpret_cast<GetterType>(mg)), m_set(reinterpret_cast<SetterType>(ms)) {}

        virtual hkResult setValue(_Inout_ void* addr, _In_ const hkReflect::PointerType* type, const Var& var) const HK_OVERRIDE;
        virtual hkResult getValue(_In_ const void* self, _In_ const hkReflect::PointerType* type, _Out_ Var* val) const HK_OVERRIDE;
    };


    //
    // Utility functions.
    //

    /// Allocates a block of type->getSizeOf() bytes.
    Var allocateDefault(_In_ const Type* type);
    /// Deallocates a block of var.getType()->getSizeOf() bytes.
    void deallocateDefault(_Inout_ void* target, _In_ const hkReflect::Type* type);

    /// Allocates a block of type->getSizeOf() bytes.
    Var allocateDefault(_In_ const Type* type, hkMemoryAllocator& mem);
    /// Deallocates a block of var.getType()->getSizeOf() bytes.
    void deallocateDefault(_Inout_ void* target, _In_ const hkReflect::Type* type, hkMemoryAllocator& mem);

    HK_INLINE bool canSetPtr(_In_ const hkReflect::Type* pointedType, const hkReflect::Var& value);
    HK_INLINE bool canSetPtr(_In_ const hkReflect::Type* pointedType, _In_ const hkReflect::Type* valueType);

    //
    // Constructors and special methods trampolines
    //

    // Hack in order to fix "LNK2020 unresolved token" link errors in Havok.Cloth.Setup
    // addDefine("HK_DECLSPEC_DLLEXPORT") has been added to Source/ContentTools/Common/ClothSetup/UI/settings.build
    // Add the dllexport flag to these methods generates an "External" symbol in the .obj file which silences the linker.
    
#ifdef HK_DECLSPEC_DLLEXPORT
#undef HK_DECLSPEC_DLLEXPORT
#define HK_DECLSPEC_DLLEXPORT __declspec(dllexport)
#else
#define HK_DECLSPEC_DLLEXPORT
#endif

    /// Type trait to detect presence of the bypass Ctor.
    /// In C++03 there is no portable SFINAE check to detect the presence of a generic Ctor, but the bypass
    /// constructor has the very special shape T(hkReflect::BypassCtorFlag), which means that it is sufficient
    /// to check whether the hkReflect::BypassCtorFlag can be converted to T.
    template< typename Type >
    struct HasReflectCloneCtor : public hkTrait::IsConvertibleTo < hkReflect::BypassCtorFlag, Type > {};


    // Call wrappers for explicit special methods.

    template <Optional OPT, typename T> struct ExplicitWrapper;

    template<typename T>
    struct ExplicitWrapper<Opt::DEF_CONSTRUCTOR, T>
    {
        HK_DECLSPEC_DLLEXPORT static void HK_CALL func(_Inout_updates_bytes_(sizeof(T) * num) void* p, _In_opt_ const hkReflect::Type*, int num)
        {
            for (int i = 0; i < num; ++i)
            {
                new (reinterpret_cast<hkPlacementNewArg*>(static_cast<T*>(p)+i)) T;
            }
        }
    };

    template<typename T>
    struct ExplicitWrapper<Opt::COPY_CONSTRUCTOR, T>
    {
        HK_DECLSPEC_DLLEXPORT static void HK_CALL func(_Inout_updates_bytes_(sizeof(T) * num) void* target, _In_reads_bytes_(sizeof(T) * num) const void* source, _In_opt_ const hkReflect::Type*, int num)
        {
            for (int i = 0; i < num; ++i)
            {
                new (static_cast<T*>(target)+i) T(*(static_cast<const T*>(source)+i));
            }
        }
    };

    template<typename T>
    struct ReflectConstructionWrapper
    {
        HK_DECLSPEC_DLLEXPORT static void HK_CALL func(_Inout_updates_bytes_(sizeof(T) * num) void* p, _In_reads_bytes_opt_(sizeof(T) * num) const hkReflect::Type*, int num)
        {
            hkReflect::BypassCtorFlag f;
            for (int i = 0; i < num; ++i)
            {
                new (static_cast<T*>(p)+i) T(f);
            }
        }
    };

    template<typename T>
    struct ExplicitWrapper<Opt::DESTRUCTOR, T>
    {
        HK_DECLSPEC_DLLEXPORT static void HK_CALL func(_Inout_updates_bytes_(sizeof(T) * num) void* p, _In_reads_bytes_opt_(sizeof(T) * num) const hkReflect::Type*, int num)
        {
            for (int i = num-1; i >= 0; --i)
            {
                (static_cast<T*>(p)+i)->~T();
            }
        }
    };

    template<typename T>
    struct ExplicitWrapper<Opt::COPY_ASSIGNMENT, T>
    {
        HK_DECLSPEC_DLLEXPORT static void HK_CALL func(_Inout_updates_bytes_(sizeof(T) * num) void* target, _In_reads_bytes_(sizeof(T) * num) const void* source, _In_opt_ const hkReflect::Type*, int num)
        {
            for (int i = 0; i < num; ++i)
            {
                *(static_cast<T*>(target)+i) = *(static_cast<const T*>(source)+i);
            }
        }
    };


    // Trivial special methods implementations --------

    template <Optional OPT> struct Trivial;
    template<> struct HK_EXPORT_COMMON Trivial<Opt::DEF_CONSTRUCTOR> { static void HK_CALL func(_Inout_ void* target, _In_ const hkReflect::Type* type, int num); };
    template<> struct HK_EXPORT_COMMON Trivial<Opt::COPY_CONSTRUCTOR> { static void HK_CALL func(_Inout_ void* target, _In_ const void* source, _In_ const hkReflect::Type* type, int num); };
    template<> struct HK_EXPORT_COMMON Trivial<Opt::COPY_ASSIGNMENT> { static void HK_CALL func( _Inout_ void* target, _In_ const void* source, _In_ const hkReflect::Type* type, int num ); };
    template<> struct HK_EXPORT_COMMON Trivial<Opt::DESTRUCTOR> { static void HK_CALL func( _Inout_ void* target, _In_ const hkReflect::Type* type, int num ); };

    HK_EXPORT_COMMON void HK_CALL assertOnInvalidDestruction(_Inout_ void* target, _In_ const hkReflect::Type*, int num);


    // Implicit special methods implementations --------

    template <Optional OPT> struct Implicit;
    template<> struct HK_EXPORT_COMMON Implicit<Opt::DEF_CONSTRUCTOR> { static void HK_CALL func(_Inout_ void* target, _In_ const hkReflect::Type* type, int num); };
    template<> struct HK_EXPORT_COMMON Implicit<Opt::COPY_CONSTRUCTOR> { static void HK_CALL func(_Inout_ void* target, _In_ const void* source, _In_ const hkReflect::Type* type, int num); };
    template<> struct HK_EXPORT_COMMON Implicit<Opt::COPY_ASSIGNMENT> { static void HK_CALL func(_Inout_ void* target, _In_ const void* source, _In_ const hkReflect::Type* type, int num); };
    template<> struct HK_EXPORT_COMMON Implicit<Opt::DESTRUCTOR> { static void HK_CALL func(_Inout_ void* target, _In_ const hkReflect::Type* type, int num); };


    // Unknown special methods placeholders ------------

    template <Optional OPT> struct Unknown;
    template<> struct HK_EXPORT_COMMON Unknown<Opt::DEF_CONSTRUCTOR> { static void HK_CALL func(_Inout_ void*, _In_ const hkReflect::Type*, int); };
    template<> struct HK_EXPORT_COMMON Unknown<Opt::COPY_CONSTRUCTOR> { static void HK_CALL func(_Inout_ void*, _In_ const void*, _In_ const hkReflect::Type*, int); };
    template<> struct HK_EXPORT_COMMON Unknown<Opt::COPY_ASSIGNMENT> { static void HK_CALL func(_Inout_ void*, _In_ const void*, _In_ const hkReflect::Type*, int); };

    // Helper function which calls obj->getExactType().
    template <typename T> AddrAndType callGetExactType(_In_ const T* obj);
}
}

namespace hkReflect
{
    //
    // ReflectionOf T[N]
    //
    namespace Detail
    {
        template<typename T, int N> struct ReflectedKindOfImpl<T[N]> { typedef ArrayType Type; };

        template<typename T, int N>
        struct RepeatReflection
        {
            typedef RepeatReflection<T, N> Holder;
            static TypeData typeData;
            static const TemplatePod<1 + 1> params;
            static TypeRegNode typeRegNode;
        };

        template <typename T, int N>
        const TemplatePod<1 + 1> RepeatReflection<T, N>::params = {
            HK_REFLECT_GET_TYPE(HK_REFLECT_RESOLVE(T[N])), { (1 + 1) & 0xffff, {
            { hkUlong(HK_REFLECT_GET_TYPE(T)), "tT" },
            { hkUlong(N), "vN" } } } };

        HK_EXPORT_COMMON extern const FixedArrayStorage<hkUlong, AttributeItem, 1> s_repeatReflectionAttributes;

        template<typename T, int N>
        TypeData RepeatReflection<T, N>::typeData =
        {
            hkReflect::Opt::FORMAT | hkReflect::Opt::SUBTYPE | hkReflect::Opt::IMPL | hkReflect::Opt::NAME | hkReflect::Opt::ATTRIBUTES |
            hkReflect::Opt::DEF_CONSTRUCTOR | hkReflect::Opt::COPY_CONSTRUCTOR | hkReflect::Opt::DESTRUCTOR |
            hkReflect::Opt::COPY_ASSIGNMENT | hkReflect::Opt::INHERITANCE | hkReflect::Opt::TEMPLATE |
            hkReflect::Opt::ALLOC_IMPL | hkReflect::Opt::SIZE_ALIGN,
            0, // no parent
            HK_REFLECT_TYPE_OPTIONAL(Opt::FORMAT, hkReflect::Format::OfArray<N>::Value),
            HK_REFLECT_TYPE_OPTIONAL(Opt::SUBTYPE, HK_REFLECT_GET_TYPE(T)),
            HK_REFLECT_TYPE_OPTIONAL(Opt::IMPL, &RepeatImpl::s_instance),
            HK_REFLECT_TYPE_OPTIONAL(Opt::NAME, "T[N]"),
            HK_REFLECT_TYPE_OPTIONAL(Opt::ATTRIBUTES, reinterpret_cast<const AttributeArray*>(&s_repeatReflectionAttributes)),
            HK_REFLECT_TYPE_OPTIONAL(Opt::DEF_CONSTRUCTOR, &Unknown<Opt::DEF_CONSTRUCTOR>::func),
            HK_REFLECT_TYPE_OPTIONAL(Opt::COPY_CONSTRUCTOR, &Unknown<Opt::COPY_CONSTRUCTOR>::func),
            HK_REFLECT_TYPE_OPTIONAL(Opt::DESTRUCTOR, HK_NULL), 
            HK_REFLECT_TYPE_OPTIONAL(Opt::COPY_ASSIGNMENT, &Unknown<Opt::COPY_ASSIGNMENT>::func),
            HK_REFLECT_TYPE_OPTIONAL(Opt::INHERITANCE, 0),
            HK_REFLECT_TYPE_OPTIONAL(Opt::TEMPLATE, reinterpret_cast<const TemplateParameterArray*>(&RepeatReflection<T, N>::params)),
            HK_REFLECT_TYPE_OPTIONAL(Opt::ALLOC_IMPL, &HeapAllocImpl::s_instance),
            HK_REFLECT_TYPE_OPTIONAL_SIZE_ALIGN(sizeof(T[N]), HK_ALIGN_OF(T), 0),
            hkUlong(&RepeatReflection<T, N>::typeRegNode)
        };

        template <typename T, int N>
        TypeRegNode RepeatReflection<T, N>::typeRegNode(reinterpret_cast<Type*>(&RepeatReflection<T, N>::typeData));


        // char[N]

        template<typename CHAR_N, bool IMMUTABLE>
        struct ReflectionOfCharArray
        {
            typedef ReflectionOf<CHAR_N> Holder;
            static Detail::TypeData typeData;
            static const Detail::TemplatePod<1> params;
            static Detail::TypeRegNode typeRegNode;
        };

        template <typename CHAR_N, bool IMMUTABLE>
        const Detail::TemplatePod<1> ReflectionOfCharArray<CHAR_N,IMMUTABLE>::params = {
            HK_REFLECT_GET_TYPE(CHAR_N),{ 1 & 0xffff,{ { hkUlong(sizeof(CHAR_N)), "vN" } } } };

        template <typename CHAR_N, bool IMMUTABLE>
        Detail::TypeData ReflectionOfCharArray<CHAR_N, IMMUTABLE>::typeData =
        {
            hkReflect::Opt::FORMAT | hkReflect::Opt::IMPL | hkReflect::Opt::NAME |
            hkReflect::Opt::DEF_CONSTRUCTOR | hkReflect::Opt::COPY_CONSTRUCTOR | hkReflect::Opt::COPY_ASSIGNMENT |
            hkReflect::Opt::INHERITANCE | Opt::TEMPLATE | hkReflect::Opt::ALLOC_IMPL | hkReflect::Opt::SIZE_ALIGN,
            0, // parent
            HK_REFLECT_TYPE_OPTIONAL(Opt::FORMAT, hkReflect::Format::OfStringUtf8<sizeof(CHAR_N), IMMUTABLE>::Value),
            HK_REFLECT_TYPE_OPTIONAL(Opt::IMPL, &Detail::CharBufferImpl::s_instance),
            HK_REFLECT_TYPE_OPTIONAL(Opt::NAME, (IMMUTABLE ? "const char[N]" : "char[N]")),
            HK_REFLECT_TYPE_OPTIONAL(Opt::DEF_CONSTRUCTOR, &Detail::Trivial<Opt::DEF_CONSTRUCTOR>::func),
            HK_REFLECT_TYPE_OPTIONAL(Opt::COPY_CONSTRUCTOR, &Detail::Trivial<Opt::COPY_CONSTRUCTOR>::func),
            HK_REFLECT_TYPE_OPTIONAL(Opt::COPY_ASSIGNMENT, &Detail::Trivial<Opt::COPY_ASSIGNMENT>::func),
            HK_REFLECT_TYPE_OPTIONAL(Opt::INHERITANCE, 0),
            HK_REFLECT_TYPE_OPTIONAL(Opt::TEMPLATE, reinterpret_cast<const Detail::TemplateParameterArray*>(&ReflectionOfCharArray<CHAR_N,IMMUTABLE>::params)),
            HK_REFLECT_TYPE_OPTIONAL(Opt::ALLOC_IMPL, &Detail::HeapAllocImpl::s_instance),
            HK_REFLECT_TYPE_OPTIONAL_SIZE_ALIGN(sizeof(CHAR_N), 1, 0),
            hkUlong(&ReflectionOfCharArray<CHAR_N,IMMUTABLE>::typeRegNode)
        };

        template <typename CHAR_N, bool IMMUTABLE>
        Detail::TypeRegNode ReflectionOfCharArray<CHAR_N,IMMUTABLE>::typeRegNode(reinterpret_cast<Type*>(&ReflectionOfCharArray<CHAR_N,IMMUTABLE>::typeData));
    }

    template<typename T, int N>
    struct ReflectionOf< T[N] > : public Detail::RepeatReflection<T, N>
    {
        typedef typename hkTrait::If<Detail::IsReflectedBase<T>::result, ReflectionOf<T[N]>, void>::Type Holder;
    };

    // Need to specialize these so that there is no ambiguity with ReflectionOf<const T>
    template<typename T, int N> struct ReflectionOf<const T[N]> : public ReflectionOf <T[N]> {};

    template<int N> struct ReflectionOf<      char[N]> : public Detail::ReflectionOfCharArray<      char[N], false> {};
    template<int N> struct ReflectionOf<const char[N]> : public Detail::ReflectionOfCharArray<const char[N], true> {};

    namespace Detail
    {
        template<int N> struct ReflectedKindOfImpl<char[N]> { typedef StringType Type; };
        template<int N> struct ReflectedKindOfImpl<const char[N]> { typedef StringType Type; };
    }


#if defined(HK_MEMORY_TRACKER_ENABLE)
    namespace Detail
    {
        // Track C arrays only if the element type is a pointer or has a tracker handle.
        template < typename T, int N, bool TRACKED >
        struct RepeatTracker;

        template<typename T, int N> struct RepeatTracker<T, N, false> { typedef HandleHelper<NotTracked> Holder; };

        template< typename T, int N>
        struct RepeatTracker<T, N, true>
        {
            // Use the reflected type if T is reflected, otherwise use define a basic one.
            typedef RepeatTracker<T, N, true> Holder;
            static const hkUint16 trackerHandle;
            static const Detail::TrackerRegNode trackerRegNode;
            static TypeData typeData;
        };

        template< typename T, int N>
        const hkUint16 RepeatTracker<T, N, true>::trackerHandle = HK_TRACKER_CAST_TO_HANDLE(&RepeatTracker<T, N, true>::trackerRegNode);

        template< typename T, int N >
        const Detail::TrackerRegNode RepeatTracker<T, N, true>::trackerRegNode(typeData, &trackerHandle, true);
        template<typename T, int N>
        TypeData RepeatTracker<T, N, true>::typeData =
        {
            hkReflect::Opt::FORMAT | hkReflect::Opt::SUBTYPE | hkReflect::Opt::IMPL | hkReflect::Opt::SIZE_ALIGN,
            0, // parent
            HK_REFLECT_TYPE_OPTIONAL(Opt::FORMAT, hkReflect::Format::OfArray<N>::Value),
            HK_REFLECT_TYPE_OPTIONAL(Opt::SUBTYPE, (const Type*)HK_TRACKER_GET_HANDLE(T)),
            HK_REFLECT_TYPE_OPTIONAL(Opt::IMPL, &Detail::RepeatImpl::s_instance),
            HK_REFLECT_TYPE_OPTIONAL_SIZE_ALIGN(sizeof(T[N]), HK_ALIGN_OF(T[N]), 0),
        };

        template<typename T, int N> struct TrackerOf<T[N]> : public RepeatTracker<T, N, hkTrait::TypesAreDifferent<typename TrackerOf<T>::Holder, HandleHelper<NotTracked> >::result> {};
        template<typename T, int N> struct TrackerOf<const T[N]> : public TrackerOf<T[N]> {};
    }
#endif

    //
    // ReflectionOf T*
    //

    namespace Detail
    {
        
        struct HK_EXPORT_COMMON ReferencedObjectPointerImpl : public RawPointerImpl
        {
            ReferencedObjectPointerImpl() { /* keep clang happy */ }
            static const ReferencedObjectPointerImpl s_instance;

            virtual hkResult setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* type, const Var& var) const HK_OVERRIDE;
        };

        // Trait which selects the behavior of T* depending on whether T is referenced or not.
        template <typename T, bool IS_REFCOUNTED = hkTrait::InheritsFrom<T, hkReferencedObject>::result>
        struct RawPointerImplSelect;

        template <typename T>
        struct RawPointerImplSelect < T, true >
        {
            // T is referenced, T* behaves as hkRefPtr<hkReferencedObject>.
            typedef ReferencedObjectPointerImpl Impl;

            template<hkReflect::Opt::Values OPT>
            struct SpecialMethodsHelper : public ExplicitWrapper< OPT, hkRefPtr<hkReferencedObject> > {};
        };

        template <typename T>
        struct RawPointerImplSelect <T, false >
        {
            // T is not referenced, T* behaves normally.
            typedef RawPointerImpl Impl;

            template<hkReflect::Opt::Values OPT>
            struct SpecialMethodsHelper : public Trivial<OPT> {};
        };

        template<typename T> struct ReflectedKindOfImpl<T*> { typedef PointerType Type; };

        template<typename T>
        struct PtrReflection
        {
            typedef PointerPropertyImpl<T*, T> PropertyImpl;
            static TypeRegNode typeRegNode;
            static const TemplatePod<1 + 0> params;
            static TypeData typeData;

            enum
            {
                // Raw pointers are weak except if:
                // - they point to referenced objects, as they keep a reference;
                // - they point to Types, as Types are considered to be always alive.
                IS_WEAK = (!hkTrait::InheritsFrom<T, hkReferencedObject>::result && !hkTrait::InheritsFrom<T, Type>::result)
            };
        };

        template <typename T>
        TypeRegNode PtrReflection<T>::typeRegNode(HK_REFLECT_GET_NON_CONST_TYPE(T*));

        template <typename T>
        const TemplatePod<1 + 0> PtrReflection<T>::params =
        { HK_REFLECT_GET_TYPE(HK_REFLECT_RESOLVE(T*)), { (1 + 0) & 0xffff, { { hkUlong(HK_REFLECT_GET_TYPE(T)), "tT" } } } };

        template<typename T>
        TypeData PtrReflection<T>::typeData =
        {
            hkReflect::Opt::FORMAT | hkReflect::Opt::SUBTYPE | hkReflect::Opt::IMPL | hkReflect::Opt::NAME | hkReflect::Opt::DEF_CONSTRUCTOR |
            hkReflect::Opt::COPY_CONSTRUCTOR | hkReflect::Opt::DESTRUCTOR | hkReflect::Opt::COPY_ASSIGNMENT |
            hkReflect::Opt::INHERITANCE | hkReflect::Opt::TEMPLATE | hkReflect::Opt::ALLOC_IMPL |
            hkReflect::Opt::SIZE_ALIGN | hkReflect::Opt::FLAGS,
            0, // no parent
            HK_REFLECT_TYPE_OPTIONAL(Opt::FORMAT, hkReflect::Format::OfPointer::Value),
            HK_REFLECT_TYPE_OPTIONAL(Opt::SUBTYPE, HK_REFLECT_GET_TYPE(T)),
            HK_REFLECT_TYPE_OPTIONAL(Opt::IMPL, &RawPointerImplSelect<T>::Impl::s_instance),
            HK_REFLECT_TYPE_OPTIONAL(Opt::NAME, "T*"),
            HK_REFLECT_TYPE_OPTIONAL(Opt::DEF_CONSTRUCTOR, &Trivial<Opt::DEF_CONSTRUCTOR>::func),
            HK_REFLECT_TYPE_OPTIONAL(Opt::COPY_CONSTRUCTOR, &RawPointerImplSelect<T>::template SpecialMethodsHelper<Opt::COPY_CONSTRUCTOR>::func),
            HK_REFLECT_TYPE_OPTIONAL(Opt::DESTRUCTOR, &RawPointerImplSelect<T>::template SpecialMethodsHelper<Opt::DESTRUCTOR>::func),
            HK_REFLECT_TYPE_OPTIONAL(Opt::COPY_ASSIGNMENT, &RawPointerImplSelect<T>::template SpecialMethodsHelper<Opt::COPY_ASSIGNMENT>::func),
            HK_REFLECT_TYPE_OPTIONAL(Opt::INHERITANCE, 0),
            HK_REFLECT_TYPE_OPTIONAL(Opt::TEMPLATE, reinterpret_cast<const TemplateParameterArray*>(&PtrReflection<T>::params)),
            HK_REFLECT_TYPE_OPTIONAL(Opt::ALLOC_IMPL, &HeapAllocImpl::s_instance),
            HK_REFLECT_TYPE_OPTIONAL_SIZE_ALIGN(sizeof(T*), HK_ALIGN_OF(T*), 0),
            HK_REFLECT_TYPE_OPTIONAL(Opt::FLAGS, IS_WEAK ? Type::TYPE_POINTER_WEAK : 0),
            hkUlong(&PtrReflection<T>::typeRegNode)
        };

        // Deduplicate these, will have exactly the same content.
        template <typename T> struct PtrReflection<const T> : public PtrReflection<T> {};
    }

    template<typename T>
    struct ReflectionOf<T*> : public Detail::PtrReflection<T>
    {
        typedef typename hkTrait::If<Detail::IsReflectedBase<T>::result, ReflectionOf<T*>, void>::Type Holder;
    };

    
    namespace Detail
    {
        template<>
        struct HK_EXPORT_COMMON PtrReflection<void>
        {
            static TypeRegNode typeRegNode;
            static const TemplatePod<1 + 0> params;
            static TypeData typeData;
        };
    }

    //
    // ReflectionOf MarkAsOpaque<T*>
    //

#if defined(HK_MEMORY_TRACKER_ENABLE)
    namespace Detail
    {
        // Defines a void* field type with a parameterized offset.
        // Used in tracker reflection for pointer fields to generate less code and save memory.
        template <int OFFSET>
        struct PtrFieldHelper
        {
            static const TypeData typeData;
        };

        template <int OFFSET>
        const hkReflect::Detail::TypeData PtrFieldHelper<OFFSET>::typeData =
        {
            hkReflect::Opt::DECL_FORMAT,//optionals
            (hkUlong)HK_REFLECT_GET_TYPE(void*),//parent
            HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::DECL_FORMAT, ((hkReflect::Decl::DECL_DATA_FIELD) << 16) | OFFSET)
        };

        // The PlayStation(R)Vita compiler has issues with the extra cast to char&, so we need a simplified macro.
        // The PlayStation(R)4 compiler has issues *without* the extra cast, so we use the normal macro.
#ifdef HK_PLATFORM_PSVITA
    #define HK_DETAIL_SIMPLE_OFFSET_OF(FIELD, ...) int(hkUlong(&(reinterpret_cast<__VA_ARGS__*>(32)->FIELD))-32)
#else
    #define HK_DETAIL_SIMPLE_OFFSET_OF(FIELD, ...) HK_ROFFSET_OF(FIELD, __VA_ARGS__)
#endif
        // Dummy type used in MarkAsOpaque fields to wrap tracker types.
        // Reflected types cannot point directly to tracker types because tracker types are not visible outside cxxs.
        // Tracker handles are visible, but they cannot be used directly in reflected types because there is no way to
        // find them and fix them up. So we make reflected types point to Opaqueize wrappers, which in turn point to the
        // tracker handle of the non-reflected wrapped type, and we fix up the wrappers at runtime.
        template <typename T> struct Opaqueize {};
    }

    template<typename T>
    struct ReflectionOf< Detail::Opaqueize<T> >
    {
        typedef ReflectionOf< Detail::Opaqueize<T> > Holder;
        static Detail::TypeData typeData;
        static const Detail::TrackerRegNode trackerRegNode;
    };

    template<typename T>
    Detail::TypeData ReflectionOf< Detail::Opaqueize<T> >::typeData =
    {
        hkReflect::Opt::FORMAT | hkReflect::Opt::NAME | hkReflect::Opt::SIZE_ALIGN | hkReflect::Opt::FLAGS,
        (hkUlong)HK_TRACKER_GET_HANDLE(T),
        HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::FORMAT, hkReflect::Format::OfOpaque::Value),
        HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::NAME, "hkReflect::Detail::Opaque"),
        HK_REFLECT_TYPE_OPTIONAL_SIZE_ALIGN(sizeof(T), 1, 0),
        HK_REFLECT_TYPE_OPTIONAL(hkReflect::Opt::FLAGS, hkReflect::Type::TYPE_NOT_SERIALIZABLE),
        (hkUlong)&trackerRegNode
    };

    // Define a trackerRegNode so that the type is registered for fixup.
    template<typename T>
    const Detail::TrackerRegNode ReflectionOf< Detail::Opaqueize<T> >::trackerRegNode(typeData, HK_NULL, true);
#endif

    template< typename T >
    struct ReflectionOf< Detail::MarkAsOpaque<T*> > : public ReflectionOf < Detail::Opaque* > {};

    
    namespace Detail
    {
        // This class is just a placeholder to force the ReflectionOf<T*>
        // to use a raw implementation instead of e.g. the SmartPointerImpl for ReferencedObject*.
        // Used in function parameters reflection.
        template< typename T >
        struct RawPointer
        {
            // Mark as POD explicitly because PreBuild generates an explicit ctor for templates.
            HK_DECLARE_CLASS( RawPointer, New, Reflect, Pod );

            HK_REFLECT_AS_POINTER( &hkReflect::Detail::RawPointerImpl::s_instance, T );
            HK_RECORD_ATTR( hk::Rename( T*(raw) ), hk::AddFlags( Type::TYPE_POINTER_WEAK ) );

            T* m_val;
        };
    }

    // Reference types are not reflected in general...
    template <typename T> struct ReflectionOf < T& > {};

    // ...but when appearing as function parameters they are reflected as non-null pointers.
    namespace Detail
    {
        template <typename T>
        struct Reference : public hkReferenceWrapper<T>
        {
            HK_DECLARE_CLASS( Reference, New, Reflect, ReflectDetails( parents = false ), BypassCtorTrivial );
            HK_REFLECT_AS_POINTER( &hkReflect::Detail::ReferenceImpl::s_instance, T );
            HK_RECORD_ATTR( hk::Rename( T& ), hk::AddFlags( Type::TYPE_POINTER_WEAK ) );

            HK_ALWAYS_INLINE Reference( T& ref ) : hkReferenceWrapper<T>( ref ) {}
        };

        struct HK_EXPORT_COMMON ReferenceImpl : public RawPointerImpl
        {
            ReferenceImpl() { /* keep clang happy */ }

            virtual hkResult setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* type, const Var& var) const HK_OVERRIDE;
            virtual bool canBeNull() const HK_OVERRIDE;

            static const ReferenceImpl s_instance;
        };
    }
}

#include <Common/Base/_Auto/TemplateTypes/hkBuiltinImpl_Types.inl>

// Deduplicate these, will have exactly the same content.
namespace hkReflect
{
    template <typename T> struct ReflectionOf< Detail::RawPointer<const T> > : public ReflectionOf< Detail::RawPointer<T> > {};
    template <typename T> struct ReflectionOf< Detail::Reference<const T> > : public ReflectionOf< Detail::Reference<T> > {};
}

#include <Common/Base/Reflect/Impl/hkBuiltinImpl.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
