// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Detail/hkTypeHasher.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>

#define DEBUG_LOG_IDENTIFIER "s11n.TypeHasher"
#include <Common/Base/System/Log/hkLog.hxx>

// NOTE : only update sig using one of the _add* methods.
// This makes logging very much easier.

hkReflect::TypeHasher::TypeHasher(hkUlong optMask /* = hkLong(-1) */) : m_optMask(optMask)
{
}

hkReflect::TypeHasher::~TypeHasher()
{
}

static hkUint32 _addU32(hkUint32 sig, hkUint32 u32)
{
    hkUint32Le u = u32;
    hkUint32 snew = hkHash::appendCrc32(sig, &u, sizeof(u));
    DLOG("0x{:x} = addU32(0x{:x}, 0x{2:x}(={2}))", snew, sig, u32);
    return snew;
}

static hkUint32 _addStr(hkUint32 sig, _In_z_ const char* s)
{
    hkUint32 snew = hkHash::appendCrc32(sig, s, hkString::strLen(s));
    DLOG("0x{:x} = addStr(0x{:x}, '{}')", snew, sig, s);
    return snew;
}


hkUint32 hkReflect::TypeHasher::_addTypeName(hkUint32 sig, _In_ const hkReflect::Type* type)
{
    if (const char* name = TypeDetail::localGetOptional<Opt::NAME>(type))
    {
        sig = _addStr(sig, name);
    }
    return sig;
}

hkUint32 hkReflect::TypeHasher::_addTypeTemplate(hkUint32 sig, _In_ const hkReflect::Type* type)
{
    if (const hkReflect::Template* params = type->getTemplate())
    {
        if (params->getNumParams())
        {
            for (int i = 0; i < params->getNumParams(); i++)
            {
                const hkReflect::Template::Parameter* p = params->getParam(i);
                if (p->isType())
                {
                    sig = _addTypeName(sig, p->getAsType());
                    sig = _addTypeTemplate(sig, p->getAsType());
                }
                else
                {
                    sig = _addU32(sig, hkLosslessCast<hkUint32>(p->getAsValue()));
                }
            }
        }
    }
    return sig;
}


hkUint32 hkReflect::TypeHasher::_addType(hkUint32 sig, _In_ const hkReflect::Type* type)
{
    DLOG_SCOPE("AddType {} //@{}", type->getName(), type);
    // Components contributing to the signature
    // parent hash
    // format
    // subtype (hash)
    // name
    // version
    // template
    // sizealign
    // flags
    // decls (name, flags, offset, type(hash))
    // attribute string

    hkUlong optbits = TypeDetail::localGetOptionalMask(type) & m_optMask;
    if(optbits & hkReflect::Opt::INTERFACES)
    {
        if( const Detail::InterfaceArray* interfaceArray = TypeDetail::localGetOptional<hkReflect::Opt::INTERFACES>(type) )
        {
            hkArrayView<const Detail::Interface> ifaces = interfaceArray->items();
            for(int i = 0; i < ifaces.getSize(); ++i)
            {
                sig = _addTypeName(sig, ifaces[i].m_interfaceType);
                sig = _addU32(sig, hkUint32(ifaces[i].m_offset));
            }
        }
    }
    if(optbits & hkReflect::Opt::FORMAT)
    {
        hkUlong format = TypeDetail::localGetOptional<hkReflect::Opt::FORMAT>(type);
        sig = _addU32(sig, hkLosslessCast<hkUint32>(format));
        // C-style arrays contribute to the hash also. Other arrays do not.
        if( (format & 0xf) == KIND_ARRAY)
        {
            switch((format >> 5) & 0x3) // array sub type
            {
                case 0: break;
                case 1:
                {
                    HK_ASSERT_NO_MSG(0x23b2a9bb, optbits & hkReflect::Opt::SUBTYPE);
                    hkUint32 h = calc(TypeDetail::localGetOptional<Opt::SUBTYPE>(type));
                    sig = _addU32(sig, h);
                    break;
                }
                default:
                    HK_ASSERT(0x76a30d21, 0, "unhandled array format");
            }
        }
    }
    if(optbits & hkReflect::Opt::SUBTYPE)
    {
        //sig = _addType(sig, TypeDetail::localGetOptional<Opt::SUBTYPE>(type));
    }
    if(optbits & hkReflect::Opt::NAME)
    {
        sig = _addTypeName(sig, type);
    }
    if(optbits & hkReflect::Opt::TEMPLATE)
    {
        sig = _addTypeTemplate(sig, type);
    }
    if(optbits & hkReflect::Opt::VERSION)
    {
        sig = _addU32(sig, hkLosslessCast<hkUint32>(TypeDetail::localGetOptional<Opt::VERSION>(type)));
    }

    if(optbits & hkReflect::Opt::SIZE_ALIGN)
    {
        sig = _addU32(sig, hkLosslessCast<hkUint32>(TypeDetail::localGetOptional<Opt::SIZE_ALIGN>(type)));
    }
    if(optbits & hkReflect::Opt::FLAGS)
    {
        sig = _addU32(sig, hkLosslessCast<hkUint32>(TypeDetail::localGetOptional<Opt::FLAGS>(type)));
    }
    if(optbits & hkReflect::Opt::DECLS)
    {
        for(hkReflect::DeclIter<hkReflect::DataFieldDecl> it(type); it.advance();)
        {
            hkReflect::DataFieldDecl fd = it.current();
            // decls (name, flags, offset, type(hash))
            sig = _addStr(sig, fd.getName());
            sig = _addU32(sig, fd.getOffset());
            hkUint32 hf = calc(fd.getType()->getParent());
            sig = _addU32(sig, hf);

            //flags?
        }
    }
    if (optbits & hkReflect::Opt::ATTRIBUTE_STRING)
    {
        sig = _addStr(sig, TypeDetail::localGetOptional<Opt::ATTRIBUTE_STRING>(type));
    }
    return sig;
}

hkUint32 hkReflect::TypeHasher::calc(_In_ const hkReflect::Type* type)
{
    if(type == HK_NULL)
    {
        return 0;
    }
    DLOG_SCOPE("TypeHash {} //@{}", type->getName(), type);

    const hkUint64 IN_PROGRESS = hkUint64(1) << 32;
    using namespace hkReflect;
    if(hkUint64 prev = m_map.getWithDefault(type, 0))
    {
        HK_ASSERT_NO_MSG(0x2996757, prev != IN_PROGRESS);
        DLOG("Cached 0x{:x}", prev);
        // if the type is in progress, truncation means we (arbitrarily) return 0
        return hkUint32(prev);
    }
    m_map.insert(type, IN_PROGRESS);

    hkUint32 sig = 0;
    if(const Type* parent = type->getParent())
    {
        sig = calc(parent);
    }
    sig = _addType(sig, type);

    m_map.insert(type, sig); // clears the IN_PROGRESS bit
    DLOG("Final : 0x{:x}", sig);

    return sig;
}

void hkReflect::TypeHasher::clear()
{
    m_map.clearAndDeallocate();
}

hkReflect::PropertyHasher::PropertyHasher()
{
}

hkReflect::PropertyHasher::~PropertyHasher()
{
}


hkUint32 hkReflect::PropertyHasher::calc( const hkReflect::Type* type )
{
    bool dummy;
    return calc( type, dummy );
}

hkUint32 hkReflect::PropertyHasher::calc( const hkReflect::Type* type, bool& isNew )
{
    isNew = false;

    if ( type == HK_NULL )
    {
        return 0;
    }
    DLOG_SCOPE( "PropertyHash {} //@{}", type->getName(), type );

    if ( hkUint32 prev = m_map.getWithDefault( type, 0 ) )
    {
        DLOG( "Cached 0x{:x}", prev );
        return prev;
    }

    hkUint32 sig = 0;

    if ( auto decls = TypeDetail::localGetOptional<Opt::DECLS>( type ) )
    {
        for ( int i = decls->getNumDataFields(); i < decls->getNumDecls(); ++i )
        {
            sig = _addStr( sig, decls->getDecl( i ).getName() );
        }
    }
    else if ( const hkReflect::Type* parent = type->getParent() )
    {
        sig = calc( parent );
    }

    m_map.insert( type, sig );
    DLOG( "Final : 0x{:x}", sig );

    isNew = true;
    return sig;
}

void hkReflect::PropertyHasher::clear()
{
    m_map.clearAndDeallocate();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
