// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>

namespace hkReflect
{
namespace Detail
{
    using namespace hkReflect;

    template<typename Visitor>
    struct VarsCompareVisitorTemplate : public hkReflect::VarVisitor < Visitor, int, const Var& >
    {
        HK_DECLARE_PLACEMENT_ALLOCATOR();

        int dispatchNoKindCheck(const Var& var, const Var& v)
        {
            if ((var.getAddress() == v.getAddress()) && (var.getType() == v.getType()))
            {
                return 0; // shortcut!
            }
            return hkReflect::VarVisitor<Visitor, int, const Var&>::dispatch(var, v);
        }

        int compareKinds(_In_ const Type* aType, _In_ const Type* bType)
        {
            return aType->getKind() - bType->getKind();
        }

        int dispatch(const Var& var, const Var& v)
        {
            if (const int kindDiff = compareKinds(var.getType(), v.getType()))
            {
                return kindDiff;
            }
            return dispatchNoKindCheck(var, v);
        }

        int visit(const BoolVar& va, const Var& v)
        {
            BoolVar vb = v;
            return va.getValue().compare(vb.getValue());
        }

        int visit(const IntVar& va, const Var& v)
        {
            IntVar vb = v;
            return va.getValue().compare(vb.getValue());
        }

        int visit(const FloatVar& va, const Var& v)
        {
            FloatVar vb = v;
            return va.getValue().compare(vb.getValue());
        }

        int visit(const StringVar& va, const Var& v)
        {
            StringVar vb = v;
            return va.getValue().compare(vb.getValue());
        }

        int visit(const VoidVar& va, const Var& v)
        {
            HK_ASSERT_NO_MSG(0x478a98e9, 0); // Should never be encountered.
            return false;
        }

        int visit(const OpaqueVar& va, const Var& v)
        {
            HK_ASSERT_NO_MSG(0x35fd3f29, false); // Should never be encountered.
            return false;
        }

        int visit(const ArrayVar& va, const Var& v)
        {
            //compare giving more priority to the first elements
            ArrayVar vb = v;
            ArrayValue aValue = va.getValue();
            ArrayValue bValue = vb.getValue();

            int na = aValue.getCount();
            int nb = bValue.getCount();
            if (na != nb)
            {
                return na - nb;
            }

            if (na && nb)
            {
                // Only do the kind check once for the array.
                if (const int kindDiff = compareKinds(aValue.getSubType(), bValue.getSubType()))
                {
                    return kindDiff;
                }
            }

            for (int i = 0; i < na; ++i)
            {
                if (const int elementCmp = dispatchNoKindCheck(aValue[i], bValue[i]))
                {
                    // First non-zero elementCmp is the result.
                    return elementCmp;
                }
            }

            // All of the elements are == to the other elements -> a ==  b
            return 0;
        }

        int visit(const ContainerVar& va, const Var& v)
        {
            HK_ASSERT_NO_MSG(0x590a2e97, 0);
            return false;
        }

        int visit(const RecordVar& va, const Var& v)
        {
            RecordVar vb = v;
            DeclIter<DataFieldDecl> ia(va.getType());
            DeclIter<DataFieldDecl> ib(vb.getType());

            while (ia.advance() && ib.advance())
            {
                if (const int elementCmp = dispatch(va[ia.current()], v[ib.current()]))
                {
                    // First non-zero elementCmp is the result.
                    return elementCmp;
                }
            }

            if (ib.advance())
            {
                //a has got fewer fields than b
                return -1;
            }
            else if (ia.advance())
            {
                // b has got fewer fields than a
                return 1;
            }
            else
            {
                return 0;
            }
        }

        int visit(const PointerVar& va, const Var& v)
        {
            PointerVar vb = v;
            hkLong diff = reinterpret_cast<hkLong>(va.getValue().getAddress()) - reinterpret_cast<hkLong>(vb.getValue().getAddress());
            return (0l < diff) - (diff < 0l);
        }
    };

    struct HK_EXPORT_COMMON VarsCompareVisitor : public VarsCompareVisitorTemplate<VarsCompareVisitor>
    {
        typedef VarsCompareVisitorTemplate<VarsCompareVisitor> Super;

        HK_DECLARE_PLACEMENT_ALLOCATOR();

        template<typename T>
        HK_ALWAYS_INLINE int visit(const T& va, const Var& v) { return Super::visit(va, v); }
    };
}

}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
