// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Algorithm/Hash/hkHash.h>

namespace hkReflect
{
    namespace Detail
    {
        /// Prevents the Var implicit constructor to be called if the parameter type is "convertible to" (meaning in our case: if T
        /// inherits from) hkReflect::Type, Var or if it is void.
        template<typename T, bool B =
            hkTrait::IsConvertibleTo<T*, const hkReflect::Type*>::result ||
            hkTrait::IsConvertibleTo<T*, const hkReflect::Var*>::result>
            struct DisableIfSpecial
        {
            typedef hkTrait::UnusedType Type;
        };
        template<typename T> struct DisableIfSpecial < T, true > {};

        /// Prevents the Var implicit constructor to be called with a void*.
        template<bool B> struct DisableIfSpecial < const void, B > {};
        template<bool B> struct DisableIfSpecial < void, B > {};

        struct VarPropertyImpl;
    }

    /// Interface for manipulating instances of reflected types.
    /// A Var is basically a pointer to a reflected object. Var provides a set of methods for reading from/writing
    /// to the pointed object through reflection. Like a raw pointer, var does not imply ownership or lifetime management.
    /// A Var can point to an object in memory or it can be empty (pointing to no object).
    /// Subclasses of Var follow a parallel hierarchy to subclasses of Type (e.g. IntVar, FloatVar, etc.). Each
    /// subclass provides additional methods specific to its kind. You can cast a generic Var to a specific kind by
    /// copy if the Type of the object has the correct kind, otherwise the copy will return an empty Var. E.g.:
    /// hkReflect::Var var(...); if (hkReflect::IntVar asIntVar = var) { /* var points to an int, do something with it */ }
    struct HK_EXPORT_COMMON Var
    {
    public:

        HK_DECLARE_CLASS(Var, New, Reflect);
        HK_REFLECT_AS_VARIANT(&hkReflect::Var::s_impl);
        HK_RECORD_ATTR(hk::ToString(&hkReflect::Var::toString));
        static void HK_CALL toString(const hkReflect::Var& var, hkStringBuf& sb, const hkStringView& extra);
        typedef hkReflect::Detail::VarPropertyImpl PropertyImpl;

        typedef void ParentType;

        /// Construct an empty var
        Var();

        /// Create a Var from a T*. This is the constructor you should use 90% of the time.
        /// In order to avoid common mistakes, it cannot be called with a pointer to
        /// Type, Var or void. If you really want a Var holding an hkReflect::Type/Var as its *data* pointer,
        /// or a pointer to an object with unknown type, use the explicit ctors provided below.
        /// Note that there is a second hidden parameter with a default value and should never be explicitly specified.
        template<typename T>
        Var(_In_ T* addr, typename hkReflect::Detail::DisableIfSpecial<T>::Type = hkTrait::UnusedType());

        /// This constructor considers the type pointer as *data*. Meaning that you will get a Var
        /// whose data pointer is addr and it's type is addr->getExactType().
        explicit Var(_In_ const Type* addr);

        /// Create a Var pointing to another Var.
        explicit Var(_In_ const Var* addr);

        /// Create a Var from an object with an unknown type.
        static Var fromUnknownObject(_In_ const void* addr);

        /// Create a Var from an object and a member pointer.
        template<typename Type, typename FieldType>
        static Var fromField(_In_ Type* addr, _In_ FieldType Type::*memberPointer);

        template<typename T>
        static Var fromRef(T& t);

        /// Returns true if the Var holds a valid object
        operator hkBoolOperator::BoolType() const { return hkBoolOperator::cast(isValid()); }

        /// Access the i'th var if this is an indexable type, otherwise returns a null var.
        Var operator[](int i) const;

        /// Access the decl named 'f' if it exists, otherwise returns a null var.
        Var operator[](_In_z_ const char* f) const;

        /// Access a previously found decl 'f'.
        Var operator[](Decl f) const;

        template<typename T, typename F>
        Var operator[](F T::*f) const;

        /// Assigns the value pointed by rhs to the object pointed by this Var (does not change the Var).
        /// rhs must have the same kind as this Var.
        hkResult assign(const Var& rhs) const;

        /// Formats this var and replaces the contents of the given stringbuf.
        /// Extra formatting hints may be given in the "extra" parameter.
        /// Currently, only integer hints are supported such as "x,i,u,hi,hu"
        void toString(hkStringBuf& buf) const;
        void toString(hkStringBuf& buf, const hkStringView& extra) const;

        hkResult fromString(const hkStringView& str) const;

        /// Returns true if the Var holds a valid object
        HK_ALWAYS_INLINE bool isValid() const { return m_addr; }

        /// Performs a kind dependent comparison between the values of this var and v. The types are not required to be identical.
        /// The result is negative when this is less than v, 0 when they are equal and positive when v is less than this.
        /// NOTE: the result may not be consistent between different runs of a program, because pointerVars are compared by target address.
        int compare(const Var& v) const;

        /// Returns true if this is equal to v. Convenience function which uses compare.
        HK_INLINE bool equals(const Var& v) const { return compare(v) == 0; }

        /// Returns true if this is less than v. Convenience function which uses compare.
        HK_INLINE bool lessThan(const Var& v) const { return compare(v) < 0; }

        /// Returns true if this var and rhs point to the same object, false otherwise.
        bool operator==(const Var& rhs) const;
        bool operator!=(const Var& rhs) const { return !(*this == rhs); }

        /// Returns a typed pointer to the object if it can be cast to T, HK_NULL otherwise.
        template<typename T>
        _Ret_maybenull_ T* dynCast() const;

        /// Returns a typed pointer to the object the type of the Var can be cast to T, HK_NULL otherwise.
        /// Does not check the exact type of the object, so it can be used on inexact Var (e.g. pointing to
        /// interfaces).
        template<typename T>
        _Ret_maybenull_ T* upCast() const;

        /// Returns true if the object is an instance of the type. Do *not* use this to cast the object pointer,
        /// use dynCast/hkDynCast instead.
        bool isInstanceOf(_In_ const Type* t) const;

        /// Resets the value of the Var to it's default value if it has an hk::Default attribute
        void writeDefault() const;

        /// Casts the Var to interfaceType if the cast is valid (might shift the pointer) or HK_NULL otherwise.
        template<typename T> _Ret_maybenull_ T* castToInterface() const;
        hkReflect::Var castToInterface(_In_ const hkReflect::Type* interfaceType) const;

        
        HK_ALWAYS_INLINE _Ret_maybenull_ const Detail::Impl* getImpl() const { return m_impl; }
        HK_ALWAYS_INLINE QualType getType() const { return m_type; }
        HK_ALWAYS_INLINE _Ret_maybenull_ void* getAddress() const { return m_addr; }

        /// Returns a copy of the object pointed by this Var.
        hkReflect::Var clone() const;

        enum DestroyFlagsValues
        {
            FLAG_DESTRUCT = 0x1,
            FLAG_DEALLOCATE = 0x2
        };
        typedef hkFlags<DestroyFlagsValues, hkUint8> DestroyFlags;

        /// Destroys the pointed object.
        /// If FLAG_DESTRUCT is specified, the type destructor is called. If the type has no accessible
        /// destructor, the called wrapper will assert.
        /// If FLAG_DEALLOCATE is specified, the pointed buffer will be deleted using the type AllocationImpl and
        /// the Var will be reset.
        void destroy(DestroyFlags flags = FLAG_DESTRUCT | FLAG_DEALLOCATE, bool notifyTracker = true);


        /// Internal method
        void setAddress(void* addr) { m_addr = addr; m_type = m_type.withoutConst(); HK_ASSERT_NO_MSG(0x2b3c4d41, typeIsExact()); }
        void setAddress(const void* addr) { m_addr = const_cast<void*>(addr); m_type = m_type.withConst(); HK_ASSERT_NO_MSG(0x73deb3c7, typeIsExact()); }

        /// Check if the Var type is exactly the same as the one of the pointed instance.
        /// Asserts that the Var is not a property.
        bool typeIsExact() const;

        /// Internal use. Create a var given an address and base type.
        /// If possible, use the 1-argument template constructor instead.
        Var(_In_ const void* addr, QualType type);

        /// Internal use. Create a var given an address and base type.
        /// If possible, use the 1-argument template constructor instead.
        Var(const Detail::AddrAndType& addrAndType);

        // internal
        Var(void* addr, QualType type, const Detail::Impl* impl, bool isDynamic = false);
        // internal
        Var(const void* addr, QualType type, const Detail::Impl* impl, bool isDynamic = false);

        // internal
        bool hasDynamicImpl() const;

        struct VarImpl;
        static const VarImpl s_impl;

        void afterReflectNew();

        
        void* getData() const { return getAddress(); }
        void clear() { m_addr = HK_NULL; m_type.clear(); m_impl.reset(); }

    protected:

        void* m_addr;
        QualType m_type;

        Detail::ImplPtr m_impl HK_ATTR(hk::Type(hkReflect::Detail::Opaque*), hk::Serialize(false));//HK_ATTR( hk::OpaqueType );
    };

#define HK_VAR_COMMON(STEM, PARENT_VAR) \
    typedef PARENT_VAR ParentType; \
    typedef STEM##Type MyType; \
    typedef QualifiedType<MyType> MyQualType; \
    typedef Detail::STEM##Impl MyImpl; \
    HK_DECLARE_CLASS(STEM##Var, New, Reflect); \
    HK_REFLECT_AS_VARIANT(&hkReflect::Var::s_impl); \
    HK_ALWAYS_INLINE const MyImpl* getImpl() const { return static_cast<const MyImpl*>(Var::getImpl()); } \
    HK_ALWAYS_INLINE MyQualType getType() const { return m_type.reinterpret<MyType>(); } \
    HK_ALWAYS_INLINE STEM##Var() {} \
    HK_ALWAYS_INLINE STEM##Var(void* p, MyQualType t, const MyImpl* i, bool d = false) : ParentType(p, t, i, d) { } \
    HK_ALWAYS_INLINE STEM##Var(const void* p, MyQualType t, const MyImpl* i, bool d = false) : ParentType(p, t, i, d) { } \
    HK_ALWAYS_INLINE STEM##Var(void* p, MyQualType t) : ParentType(p, t) { } \
    HK_ALWAYS_INLINE STEM##Var(const void* p, MyQualType t) : ParentType(p, t) { } \
    HK_ALWAYS_INLINE STEM##Var(const Var& v) \
    { \
        if( v.getType() && hkReflect::Detail::kindMatches<STEM##Type>(v.getType()->getKind()) ) \
        { \
            Var::operator=(v); \
        } \
    }

    /// Var pointing to an instance of a ValueType.
    struct HK_EXPORT_COMMON ValueVar : public Var
    {
        HK_VAR_COMMON(Value, Var);
    };

#define HK_VALUE_VAR_COMMON(STEM) \
    HK_VAR_COMMON(STEM, ValueVar) \
    typedef STEM##Value ValueType; \

    /// Var pointing to an instance of a BoolType.
    /// Gives access to the bool value of the pointed object.
    struct HK_EXPORT_COMMON BoolVar : public ValueVar
    {
        HK_VALUE_VAR_COMMON(Bool);

        BoolValue getValue() const;//deprecated
        hkResult setValue(BoolValue val) const;
        hkResult getValue(_Out_ BoolValue* val) const;
    };

    /// Var pointing to an instance of a IntType.
    /// Gives access to the integer value of the pointed object.
    struct HK_EXPORT_COMMON IntVar : public ValueVar
    {
        HK_VALUE_VAR_COMMON(Int);

        bool isSigned() const;
        IntValue getValue() const;// deprecated
        hkResult getValue(_Out_ IntValue* val) const;
        hkResult setValue(IntValue val) const;
    };

    /// Var pointing to an instance of a FloatType.
    /// Gives access to the floating-point value of the pointed object.
    struct HK_EXPORT_COMMON FloatVar : public ValueVar
    {
        HK_VALUE_VAR_COMMON(Float);
        FloatValue getValue() const; //deprecated
        hkResult getValue(_Out_ FloatValue* val) const;
        hkResult setValue(FloatValue val) const;
    };

    /// Var pointing to an instance of a StringType.
    /// Gives access to the string value of the pointed object.
    struct HK_EXPORT_COMMON StringVar : public ValueVar
    {
        HK_VALUE_VAR_COMMON(String);
        StringValue getValue() const;//deprecated
        hkResult getValue(_Out_ StringValue* val) const;
        hkResult setValue(StringValue val) const;
    };

    /// Var pointing to an instance of a CompoundType.
    /// Provides an interface to iterate on the contents of the pointed object.
    struct HK_EXPORT_COMMON CompoundVar : public Var
    {
        HK_VAR_COMMON(Compound, Var);
        class Iterator;

        /// Find the given var.
        /// The type of var depends on the underlying compound item.
        /// For records, var should be a string fieldname. For array vars, it should
        /// be something which is comparable to the array items.
        /// For map types var may be either the key type or the key/value pair.
        VarIter find(Var var) const;
        /// Find the given var, starting from one past "prev"
        VarIter find(Var var, const VarIter& prev) const;
        /// Get an iterator to the beginning of this var
        VarIter begin() const;
        /// Get an iterator to the end of this var
        VarIter end() const;
    };


    /// Iterator for hkReflect::Vars
    struct HK_EXPORT_COMMON VarIter
    {
        /// Create an invalid variter.
        VarIter() : m_data(hkUlong(-1)) {}
        /// Create, given a compound and element of the compound and extra data.
        VarIter(CompoundVar comp, Var cur, hkUlong d) : m_compound(comp), m_cur(cur), m_data(d) {}
        /// Create an invalid iter for the given container.
        explicit VarIter(CompoundVar c) : m_compound(c), m_data(hkUlong(-1)) {}

        /// Check iterator validity
        bool isValid() const { return m_data != hkUlong(-1); }
        /// Invalidate this iterator
        void setInvalid() { m_data = hkUlong(-1); }
        /// Cast to bool
        operator hkBoolOperator::BoolType() const { return hkBoolOperator::cast(isValid()); }
        /// Access the current var
        Var operator*() const { HK_ASSERT_NO_MSG(0x7f6acf5f, isValid()); return m_cur; }
        /// Access the current var
        _Ret_notnull_ Var* operator->() { HK_ASSERT_NO_MSG(0x74f5d4b5, isValid()); return &m_cur; }
        /// Advance the iterator
        void operator++() { HK_ASSERT_NO_MSG(0x6ec0001, isValid()); m_compound.getImpl()->iterNext(*this); }
        /// Advance the iterator
        void operator++(int) { HK_ASSERT_NO_MSG(0x4ac3f525, isValid()); m_compound.getImpl()->iterNext(*this); }

        // The compound we reference.
        CompoundVar m_compound;
        // The item within the container.
        Var m_cur;
        // Internal data used by the impl.
        hkUlong m_data;
    };

    inline VarIter CompoundVar::find(Var var) const { return getImpl()->find(VarIter(*this), var); }
    inline VarIter CompoundVar::find(Var var, const VarIter& prev) const { return getImpl()->find(prev, var); }
    inline VarIter CompoundVar::begin() const { return getImpl()->iterBegin(m_addr, getType()); }
    inline VarIter CompoundVar::end() const { return VarIter(*this); }

    /// Var pointing to an instance of a PointerType.
    /// The pointed object is itself a pointer. This class provides methods to get/set the value of the pointer as
    /// a Var.
    struct HK_EXPORT_COMMON PointerVar : public CompoundVar
    {
        HK_VAR_COMMON(Pointer, CompoundVar);

        hkResult setValue(const Var& v) const;
        hkResult getValue(_Out_ Var* val) const;
        Var getValue() const;//deprecated
        bool isNull() const;
        const hkReflect::QualType getSubType() const;
        hkResult copyFrom(const PointerVar& other) const;
    };

    /// Var pointing to an instance of a ContainerType. Internal use, use ArrayVar instead.
    struct HK_EXPORT_COMMON ContainerVar : public CompoundVar
    {
        HK_VAR_COMMON(Container, CompoundVar);

        /// Get the number of elements in this container.
        inline int getCount() const { return getImpl()->getContainerValue(m_addr, getType()).getCount(); }
        /// Get the type of all elements in this container.
        inline QualType getSubType() const { return getImpl()->getContainerValue(m_addr, getType()).getSubType(); }
        /// Get the count and subtype in a single call.
        inline ContainerValue getContainerValue() const { return getImpl()->getContainerValue(m_addr, getType()); }

        /// Insert the given var, return true if successful.
        bool insert(Var var) const { return getImpl()->insert(var, VarIter(*this)); }
        /// Insert the given var with an insertion position hint.
        bool insert(Var var, const VarIter& hint) const { return getImpl()->insert(var, hint); }
        /// Remove the value at the iterator and return true on success.
        bool remove(const VarIter& it) const { return getImpl()->remove(it); }
    };

    /// Var pointing to an instance of an ArrayType.
    /// Provides methods to manipulate the pointed array and to get the contained elements as Vars.
    struct HK_EXPORT_COMMON ArrayVar : public ContainerVar
    {
        HK_VAR_COMMON(Array, ContainerVar);

        class Iterator;

        /// Efficiently get all the properties at once. It is invalidated when the array is mutated.
        hkResult getValue(_Out_ ArrayValue* val) const;
        ArrayValue getValue() const; //deprecated

        /// Get the number of elements in this container.
        int getCount() const;
        /// Get the type of all elements in this container.
        QualType getSubType() const;
        /// Access the i'th element.
        Var operator[](int i) const;
        /// Set the number of elements in this array.
        hkResult setArraySize(int len) const;
        /// Get the base element pointer.
        _Ret_maybenull_ void* getDataPointer() const;
        /// Insert an array at the given index after deleting numToDel elements.
        hkResult spliceInto(int index, int numToDel, const ArrayValue& _insert) const;
        /// Insert one item at the given index.
        hkResult insertAt(int index, const Var& var) const;
        /// Insert one item at the end of the array.
        hkResult pushBack(const Var& var) const;
        /// Remove the element at the given index.
        hkResult removeAt(int index) const; 
            /// Append the array to this one.
        hkResult append(ArrayValue h) const;
        /// Replace the contents of this array with another.
        hkResult setValue(ArrayValue h) const;
        /// Remove all elements from this array
        hkResult clear() const;
    };

    /// Var pointing to an instance of a RecordType (C++ struct/class). Internal use.
    /// Use Var::operator[] to access class fields.
    struct HK_EXPORT_COMMON RecordVar : public CompoundVar
    {
        HK_VAR_COMMON(Record, CompoundVar);

        /// Returns this Var.
        inline hkResult getValue(_Out_ RecordVar* val) const;
        inline RecordVar getValue() const; //deprecated

            /// Copy the object pointed by rhs if the type is compatible.
        hkResult setValue(const hkReflect::RecordVar& rhs) const;
    };

    /// Var pointing to an instance of an Opaque type. Internal use.
    struct HK_EXPORT_COMMON OpaqueVar : public Var
    {
        HK_VAR_COMMON(Opaque, Var);
    };

    /// Var pointing to an instance of a Void type. Internal use.
    struct HK_EXPORT_COMMON VoidVar : public Var
    {
        HK_VAR_COMMON(Void, Var);
    };

    /// Hashes a Var. It will only take the address into account so do not use as an absolutely unique key since
    /// trivial collisions may happen between Vars having the same address but different types (first field, properties).
    HK_ALWAYS_INLINE hkUint32 hkHashValue(const Var& var)
    {
        // Not hashing the type means more collisions but it is quite rare to have Vars with the same addr
        // and different types. It is an acceptable trade off allowing a faster hash function. The map will
        // perform an equality test anyway so the results will be correct.
        return hkHash::hkHashValue(var.getAddress());
    }

    namespace Detail
    {
        struct IndirectVar
        {
            // We need overloads on all the Var types because we don't want to introduce two
            // user conversions (IntVar -> Var -> IndirectVar).

            IndirectVar(const hkReflect::Var& v) : m_var(v) {}
                IndirectVar(const hkReflect::ValueVar& v) : m_var(v) {}
                    IndirectVar(const hkReflect::FloatVar& v) : m_var(v) {}
                    IndirectVar(const hkReflect::IntVar& v) : m_var(v) {}
                    IndirectVar(const hkReflect::BoolVar& v) : m_var(v) {}
                    IndirectVar(const hkReflect::StringVar& v) : m_var(v) {}
                IndirectVar(const hkReflect::CompoundVar& v) : m_var(v) {}
                    IndirectVar(const hkReflect::PointerVar& v) : m_var(v) {}
                    IndirectVar(const hkReflect::RecordVar& v) : m_var(v) {}
                    IndirectVar(const hkReflect::ContainerVar& v) : m_var(v) {}
                        IndirectVar(const hkReflect::ArrayVar& v) : m_var(v) {}
                IndirectVar(const hkReflect::OpaqueVar& v) : m_var(v) {}
                IndirectVar(const hkReflect::VoidVar& v) : m_var(v) {}

            const hkReflect::Var& m_var;
        };
    }
}

template <typename To>
HK_ALWAYS_INLINE _Ret_maybenull_ To* hkDynCast(const hkReflect::Var& var)
{
    return var.dynCast<To>();
}

// We can't use a Var as an argument for the AutoCast overload otherwise it creates an ambiguity with
// hkDynCast(T*) since Var is implicitly constructible from T*. Thus we introduce a user-conversion
// through IndirectVar to select this overload only if a Var type is used.
HK_ALWAYS_INLINE hkReflect::Detail::AutoCast hkDynCast(const hkReflect::Detail::IndirectVar& var);

namespace hkReflect
{
    /// Advanced use. Use hkDynCast to cast a Var to a typed pointer.
    template <typename To>
    _Ret_maybenull_ To* exactMatchDynCast(const hkReflect::Var& var);

    template <typename To>
    _Ret_maybenull_ To* upCast(const hkReflect::Var& var);
}

/// Keeps compatibility with old code.
typedef hkReflect::Var hkVariant;

namespace hkDebug
{
    void HK_EXPORT_COMMON dump(_In_opt_ const hkReflect::Var* obj);
    void HK_EXPORT_COMMON dump(const hkReflect::Var& obj);
}

#if !defined(__HAVOK_PARSER__)
#include <Common/Base/Reflect/Core/hkReflectVar.inl>
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
