// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>
#include <Common/Base/Reflect/Util/hkVarCoerce.h>
#include <Common/Base/System/Io/Util/hkFloatParseUtil.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>
#include <Common/Base/Reflect/Core/hkReflectVarDetail.h>

#define DEBUG_LOG_IDENTIFIER "base.reflect.var"
#include <Common/Base/System/Log/hkLog.hxx>

namespace hkReflect
{
    HK_COMPILE_TIME_ASSERT( sizeof(Var) == sizeof(ValueVar) );
    HK_COMPILE_TIME_ASSERT( sizeof(Var) == sizeof(StringVar) );
    HK_COMPILE_TIME_ASSERT( sizeof(Var) == sizeof(FloatVar) );
    HK_COMPILE_TIME_ASSERT( sizeof(Var) == sizeof(IntVar) );
    HK_COMPILE_TIME_ASSERT( sizeof(Var) == sizeof(CompoundVar) );
    HK_COMPILE_TIME_ASSERT( sizeof(Var) == sizeof(ContainerVar) );
    HK_COMPILE_TIME_ASSERT( sizeof(Var) == sizeof(ArrayVar) );
    HK_COMPILE_TIME_ASSERT( sizeof(Var) == sizeof(RecordVar) );
}

hkReflect::Var hkReflect::Var::operator[](int i) const
{
    if( ArrayVar a = *this )
    {
        return a[i];
    }
    return Var();
}


hkReflect::Var hkReflect::Var::operator[](_In_z_ const char* name) const
{
    if( *this )
    {
        if( hkReflect::Decl d = m_type->findDecl(name, true) )
        {
            return (*this)[d];
        }
    }
    return Var();
}

hkReflect::Var hkReflect::Var::operator[](hkReflect::Decl decl) const
{
    if( *this )
    {
        if( FieldDecl field = decl.asField() )
        {
            HK_ASSERT(0x167e9843, m_type->extendsOrEquals(field.getDeclContext()), "Decl does not belong to this type");

            const Detail::Impl* impl;
            bool isDynamic;
            if (m_impl.isDynamic() || m_type->hasCallbacks())
            {
                // Field in a property
                impl = m_impl->getFieldImpl(*this, field);
                isDynamic = true;
            }
            else
            {
                impl = field.getType()->getImpl();
                isDynamic = false;
            }

            return Var(hkAddByteOffset(m_addr, field.getOffset()), field.getType(), impl, isDynamic);
        }
    }
    return Var();
}


void hkReflect::Var::toString(hkStringBuf& buf) const
{
    toString(buf, hkStringView());
}

namespace
{
    /// Convert a hkLog-style integer format into a printf-style format.
    ///   'D' => '%lld' or '%llu'
    ///   'd' => '%lld' or '%llu'
    ///   'X' => '%llX'
    ///   'x' => '%llx'
    ///   'D<n>' => '%0<n>lld' or '%0<n>llu'
    ///   'd<n>' => '%0<n>lld' or '%0<n>llu'
    ///   'X<n>' => '%0<n>llX'
    ///   'x<n>' => '%0<n>llx'
    ///   (replace 'll' with 'I64' on WIN32 platform)
    void HK_CALL convertIntegerFormatToPrintf( hkReflect::IntValue value, hkStringView extra, hkStringBuf& fmtOut  )
    {
        const bool neg = value.isNegative();
        const char defaultFmt = ( neg ? 'd' : 'u' );

        char fmtChar[2] = "?";
        int padLen = 0;

        if ( extra )
        {
            const char c = extra[0];
            if ( ( c == 'D' ) || ( c == 'd' ) || ( c == 'I' ) || ( c == 'i' ) || ( c == 'u' ) || ( c == 'U' ) )
            {
                // Decimal 'D' doesn't make a difference between signed and unsigned.
                // Eventually the value passed to printf() is a hkInt64 or hkUint64 depending on the actual value.
                fmtChar[0] = defaultFmt;

                // Expect a hkLog format, but support other printf-like legacy ones for convenience (namely 'i', 'u' and for completeness the invalid 'U', 'I')
                if ( ( c != 'D' ) && ( c != 'd' ) )
                {
                    char cc[2] = " "; cc[0] = c;
                    Log_Warning( "Deprecated integer extra format specififer '{}'; use 'D' (decimal) instead.", cc );
                }
            }
            else if ( ( c == 'x' ) || ( c == 'X' ) )
            {
                fmtChar[0] = c;
                if ( neg )
                {
                    Log_Error( "Cannot format negative integer '{}' with hexadecimal specifier '{}'. Falling back to decimal 'D'.", -hkInt64( value.absValue() ) /* this is incorrect for INT64_MIN */, fmtChar );
                    fmtChar[0] = defaultFmt;
                }
            }
            else
            {
                char cc[2] = " "; cc[0] = c;
                Log_Error( "Invalid integer extra format specififer '{}'. Expected a character among [DdXx] or the deprecated [IiUu].", cc );
                fmtChar[0] = defaultFmt;
            }

            if ( extra.getSize() > 1 )
            {
                hkStringBuf str( extra.begin() + 1, extra.getSize() - 1 );
                padLen = hkString::atoi( str.cString(), 10 );
            }
        }
        else
        {
            fmtChar[0] = defaultFmt;
        }

        fmtOut = "%";
        if ( padLen > 0 )
        {
            fmtOut.appendPrintf( "0%d", padLen );
        }
#if defined(HK_PLATFORM_WIN32)
        fmtOut += "I64";
#else
        fmtOut += "ll";
#endif
        fmtOut += fmtChar;
    }

    /// Convert a hkLog-style floating-point format into a printf-style format.
    ///   'F' => '%f' and 'f' => '%f'
    ///   'G' => '%g' and 'g' => '%g'
    ///   'E' => '%E'
    ///   'e' => '%e'
    ///   'F<n>' => '%.<n>f' and 'f<n>' => '%.<n>f'
    ///   'G<n>' => '%.<n>g' and 'g<n>' => '%.<n>g'
    ///   'E<n>' => '%.<n>E'
    ///   'e<n>' => '%.<n>e'
    void HK_CALL convertFloatingFormatToPrintf( hkReflect::FloatVar fp, hkStringView extra, hkStringBuf& fmtOut )
    {
        const char defaultFmt = 'g';

        // See https://randomascii.wordpress.com/2012/03/08/float-precisionfrom-zero-to-100-digits-2/:
        // 9 resp. 17 digits are enough to guarantee round-tripping; that is, we can re-parse
        // and receive the exact same floating point value as before (provided that printing and parsing
        // are implemented correctly on the current platform).
        int defaultPrecision;
        if ( fp.getType()->getSizeOf() == sizeof( float ) )
        {
            defaultPrecision = 9;
        }
        else
        {
            defaultPrecision = 17;
        }

        char fmtChar[2] = "?";
        int userPrecision = defaultPrecision;

        if ( extra )
        {
            const char c = extra[0];
            if ( ( c == 'F' ) || ( c == 'f' ) )
            {
                fmtChar[0] = 'f';
            }
            else if ( ( c == 'G' ) || ( c == 'g' ) || ( c == 'E' ) || ( c == 'e' ) )
            {
                fmtChar[0] = c;
            }
            else
            {
                char cc[2] = " "; cc[0] = c;
                Log_Error( "Invalid floating-point extra format specififer '{}'. Expected a character among [EfFfGg].", cc );
                fmtChar[0] = defaultFmt;
            }

            if ( extra.getSize() > 1 )
            {
                hkStringBuf str( extra.begin() + 1, extra.getSize() - 1 );
                userPrecision = hkString::atoi( str.cString(), 10 );
            }
        }
        else
        {
            fmtChar[0] = defaultFmt;
            userPrecision = defaultPrecision;
        }

        fmtOut = "%";
        if ( userPrecision >= 0 )
        {
            fmtOut.appendPrintf( ".%d", userPrecision );
        }
        fmtOut += fmtChar;
    }
}

void hkReflect::Var::toString(hkStringBuf& buf, const hkStringView& extra) const
{
    // first part: fill out "buf"
    const Var& var = *this;
    if( const hk::ToString* toString = var.getType()->findAttribute<hk::ToString>() )
    {
        (*toString->m_func)( var, buf, extra );
        return;
    }
    else if(hkViewPtr<const hk::Presets> presets = var.getType()->findAttribute<hk::Presets>())
    {
        if(const char* name = presets->getNameByPreset(var))
        {
            buf.printf(name);
            return;
        }
    }

    if( hkReflect::IntVar integer = var )
    {
        const hkReflect::IntValue val = integer.getValue();
        hkStringBuf fmt;
        convertIntegerFormatToPrintf( val, extra, fmt );
        buf.printf( fmt.cString(), val.isNegative() ? -hkInt64(val.absValue()) : val.absValue() );
    }
    else if( hkReflect::StringVar str = var )
    {
        buf = str.getValue();
    }
    else if( hkReflect::FloatVar fp = var )
    {
        hkStringBuf fmt;
        convertFloatingFormatToPrintf( fp, extra, fmt );
        buf.printf( fmt.cString(), fp.getValue().m_value );

        // Work around differences in MVSCRT 2013 and 2015 - 2013 generates 3.40282002e+038 with an extra 0 in the exponent
        // (it generates always 3 digits of exponent) vs 3.40282002e+38 with 2015 (it generates the minimum amount required).
        // This somewhat unrelated, but determinism is nice to have for use cases that require round tripping.
        int index = buf.indexOfCase( "e+" );
        if ( index >= 0 )
        {
            index += 2; // first digit after "+e" or "+E"
            const int lastIndex = buf.getLength() - 1; // last digit, which we must always keep
            if ( index < lastIndex ) // more than 1 digit
            {
                // Find leading zero(s) (excluding last digit, if exponent is == 0)
                const char* const first = buf.cString() + index;
                const char* const last = buf.cString() + lastIndex;
                const char* p = first;
                while ( ( p < last ) && ( *p == '0' ) )
                {
                    ++p;
                }

                // Remove leading zero(s)
                const int numToRemove = hkGetByteOffsetInt( first, p );
                if ( numToRemove > 0 )
                {
                    char* dst = const_cast<char*>( first ); // will overwrite in-place, safe since always shorter
                    while ( p <= last )
                    {
                        *dst++ = *p++;
                    }
                    buf.setLength( buf.getLength() - numToRemove );
                }
            }
        }
    }
    else if( hkReflect::ArrayVar arr = var )
    {
        buf = "[ ";
        const char* sep = "";
        int first = 0;
        int last = arr.getCount();
        if ( extra )
        {
            // Parse extra format "offset,length"
            const int comma = extra.find( ',' );
            if ( comma != hkStringView::npos )
            {
                if ( comma > 0 )
                {
                    hkStringBuf firstStr( extra.begin(), comma );
                    const int userOffset = hkString::atoi( firstStr.cString(), 10 );
                    HK_ASSERT(0x37b36557, userOffset >= 0, "Invalid negative offset passed to extra string for formatting array. Extra syntax is ':offset,length'." );
                    first = hkMath::clamp( userOffset, first, last );
                }
                const int lengthPos = comma + 1;
                if ( lengthPos < extra.getSize() )
                {
                    hkStringBuf lengthStr( extra.begin() + lengthPos, extra.getSize() - lengthPos );
                    const int userLength = hkString::atoi( lengthStr.cString(), 10 );
                    HK_ASSERT(0x5e67968d, userLength >= 0, "Invalid negative size passed to extra string for formatting array. Extra syntax is ':offset,length'." );
                    last = hkMath::clamp( first + userLength, first, last );
                }
            }
            else
            {
                hkStringBuf firstStr( extra.begin(), extra.getSize() );
                const int userOffset = hkString::atoi( firstStr.cString(), 10 );
                HK_ASSERT(0x269ce7d0, userOffset >= 0, "Invalid negative offset passed to extra string for formatting array. Extra syntax is ':offset,length'." );
                first = hkMath::clamp( userOffset, 0, last );
            }
        }
        for ( int i = first; i < last; ++i )
        {
            buf.append(sep); sep = ", ";
            hkStringBuf s;
            arr[i].toString(s);
            buf += s;
        }
        buf.append(" ]");
    }
    else if( hkReflect::RecordVar rec = var )
    {
        buf.setJoin( rec.getType()->getName(), "{ " );
        const char* sep = "";
        for( hkReflect::VarIter cur=rec.begin(), end=rec.end(); cur!=end; ++cur )
        {
            buf.append(sep); sep = ", ";
            buf.append( (*cur).getType()->isField().getName() );
            buf.append("=");
            hkStringBuf s;
            (*cur).toString(s);
            buf += s;
        }
        buf.append(" }");
    }
    else if( hkReflect::PointerVar ptr = var )
    {
        hkReflect::Var ptd = ptr.getValue();
        if ( auto asType = Detail::asType( ptd ) )
        {
            // We usually refer to types by pointers, so print the type directly
            asType->getFullName( buf );
        }
        else
        {
            buf.printf( "%p", ptd.getAddress() );
        }
    }
    else if( hkReflect::BoolVar b = var )
    {
        buf = b.getValue() ? "true" : "false";
    }
    else if( hkReflect::VoidVar v = var )
    {
        buf = "void";
    }
    else if( hkReflect::OpaqueVar o = var )
    {
        buf = "incomplete";
    }
    else
    {
        HK_ASSERT_NO_MSG(0x6f72404,0);
    }
}

// try to get some kind of descriptive name, even for unnamed types (e.g. int[4], int*)
static void s_appendNameOf(hkReflect::QualType type, hkStringBuf& buf)
{
    if( const char* name = type->getName() )
    {
        if( const char* lt = hkString::strChr(name, '<') )
        {
            buf.append(name, int(lt-name));
        }
        else
        {
            buf.append(name);
        }
    }
    else
    {
        // no name, get the kind "Unnamed Pointer", "Unnamed Array"
        buf = "<Unnamed";
        const hkReflect::Type* t = hkReflect::typeFromKind(type->getKind());
        if(const char* kind = t->getName())
        {
            //name is like hkReflect::XxxxType - extract just Xxxx
            int n = hkString::strLen(kind);
            int prefix = HK_COUNT_OF("hkReflect::")-1;
            int suffix = HK_COUNT_OF("Type")-1;
            if( n > prefix+suffix )
            {
                buf.append(" ");
                buf.append(kind +prefix, n-(prefix+suffix)); // strip "hkReflect::" and "Type"
            }
        }
        buf.append(">");
    }
}
/*static*/ void hkReflect::Var::toString(const hkReflect::Var& arg, hkStringBuf& sb, const hkStringView& extra)
{
    // Format a var like this ((TypeOfTheThing*)0x1234)
    // The format is chosen so that it can be copy/pasted into the debugger

    hkReflect::Var var = *static_cast<const Var*>(arg.getAddress());

    if( extra == "*")
    {
        if( var.getAddress()  )
        {
            var.toString(sb);
        }
        else
        {
            sb = "(null)";
        }
        return;
    }
    sb = "((";

    if( const hkReflect::Type* type = var.getType() )
    {
        hkStringBuf stars = "*";
        while( const hkReflect::PointerType* ptype = type->asPointer() )
        {
            if( ptype->getName() == HK_NULL )
            {
                if( const hkReflect::Type* t = ptype->getSubType() )
                {
                    type = t;
                    stars += "*";
                    continue;
                }
            }
            break;
        }
        s_appendNameOf(type, sb);
        if( const Template* tmplate = type->getTemplate() )
        {
            sb.append("<");
            const char* sep = "";
            for( int i = 0; i < tmplate->getNumParams(); ++i )
            {
                sb.append(sep); sep = ", ";
                const hkReflect::Template::Parameter* p = tmplate->getParam(i);
                switch(p->getKind())
                {
                    case hkReflect::Template::KIND_TYPE:
                        s_appendNameOf(p->getAsType(), sb);
                        break;
                    case hkReflect::Template::KIND_VALUE:
                        sb.appendPrintf(HK_PRINTF_FORMAT_ULONG, p->getAsValue());
                        break;
                    default:
                        HK_ASSERT_NO_MSG(0x6f5baab7, 0);
                }
            }
            sb.append(">");
        }
        sb.append(stars);
    }
    else
    {
        sb.append("void*");
    }
    sb.append(")");
    hkUlong addr = (hkUlong)var.getAddress();
    sb.appendPrintf("0x%" HK_PRINTF_FORMAT_INT64_SIZE "x)", hkUint64(addr));
}


namespace
{
    struct VarFromStringVisitor : public hkReflect::VarVisitor<VarFromStringVisitor, hkResult, const hkStringView&>
    {
        hkResult visit(const hkReflect::VoidVar&, const hkStringView&) { return HK_FAILURE; }
        hkResult visit(const hkReflect::PointerVar& target, const hkStringView& value) { return HK_FAILURE; }

        hkResult visit(const hkReflect::RecordVar& target, const hkStringView& value)
        {
            
            return HK_FAILURE;
        }

        hkResult visit(const hkReflect::ArrayVar& target, const hkStringView& value)
        {
            
            return HK_FAILURE;
        }

        hkResult visit(const hkReflect::BoolVar& target, const hkStringView& value)
        {
            if (value == "true")
            {
                return target.setValue(true);
            }
            else if (value == "false")
            {
                return target.setValue(false);
            }
            else
            {
                return HK_FAILURE;
            }
        }

        hkResult visit(const hkReflect::IntVar& target, const hkStringView& value)
        {
            char buf[128];
            if( value.copy(buf, sizeof(buf)).isSuccess() )
            {
                const char* s = buf;
                if( hkReflect::VarCoerce::intFromString(target, hkReflect::StringVar(&s)) )
                {
                    return HK_SUCCESS;
                }
            }
            return HK_FAILURE;
        }

        hkResult visit(const hkReflect::FloatVar& target, const hkStringView& value)
        {
            double num;
            hkResult res = hkFloatParseUtil::parseFloat(hkStringView(value.begin(), value.end()), num);
            if (res.isSuccess())
            {
                return target.setValue(num);
            }
            return res;
        }

        hkResult visit(const hkReflect::StringVar& target, const hkStringView& value)
        {
            // check that the string is quoted
            if (!value || value[0] != '"' || value[value.getSize() - 1] != '"')
            {
                return HK_FAILURE;
            }
            hkStringBuf buf(value.slice(1, value.getSize() - 2));
            return target.setValue(buf.cString());
        }
    };
}

hkResult hkReflect::Var::fromString(const hkStringView& str) const
{
    // check if the value is a preset name
    if (hk::Presets::assign(*this, str).isSuccess())
    {
        return HK_SUCCESS;
    }

    // parse the value
    return VarFromStringVisitor().dispatch(*this, str);
}

hkReflect::Var hkReflect::Var::clone() const
{
    if (!isValid())
    {
        return hkReflect::Var();
    }
    const hkReflect::Type* type = m_type->isProperty() ? m_type->getParent() : m_type.get();
    hkReflect::Var res = TypeDetail::allocateForClone(*this, type);
    HK_ASSERTV(0x105703e5, res, "Cannot allocate for clone type {}. Ensure the "
        "type is allocatable (not ReflectIdentity).", type->getFullName());
    if (Detail::copyObject(res, *this).isSuccess())
    {
        return res;
    }
    return Var();
}

bool hkReflect::Var::typeIsExact() const
{
    if (m_addr)
    {
        HK_ASSERT_NO_MSG(0x64f4108c, m_type);
        if (!m_type->isProperty())
        {
            Detail::AddrAndType exact = exactObj(m_addr, m_type);
            const bool addrIsEqual = m_addr == exact.m_addr;
            const bool typeIsEqual = m_type->equals(exact.m_type);
            if (addrIsEqual == false)
            {
                HK_WARN_ALWAYS(0xabba3b92,
                    "Var has an inexact address\n"
                    "Address in Var is: " << m_addr << "\n"
                    "Exact address is:  " << exact.m_addr << "\n");

            }
            if (typeIsEqual == false)
            {
                const char* typeInVarName = m_type->getName();
                const char* exactTypeName = exact.m_type->getName();
                HK_WARN_ALWAYS(0x5432fd9a,
                    "Var has an inexact type\n"
                    "Type in Var is: " << (typeInVarName ? typeInVarName : "(no name)") << "\n"
                    "Exact type is:  " << (exactTypeName ? exactTypeName : "(no name)") << "\n");
            }
            return addrIsEqual && typeIsEqual;
        }
    }
    return true;
}

void hkReflect::Var::destroy(DestroyFlags flags /* = FLAG_DESTRUCT|FLAG_DEALLOCATE */, bool notifyTracker /* = true */)
{
    if(m_addr)
    {
        HK_ASSERT_NO_MSG(0x6cb61103,typeIsExact());
        if (flags.anyIsSet(FLAG_DESTRUCT))
        {
            TypeDetail::destruct(m_addr, m_type);
        }

        if(flags.anyIsSet(FLAG_DEALLOCATE))
        {
            TypeDetail::deallocate(m_addr, m_type);
            m_addr = HK_NULL;
            m_type = HK_NULL;
            m_impl = HK_NULL;
        }
    }
}

namespace
{

    struct ApplyDefaultsVisitor : public hkReflect::VarVisitor<ApplyDefaultsVisitor>
    {
        void visit(const hkReflect::VoidVar& b) {}

        void visit(const hkReflect::ValueVar& v)
        {
            if(hkReflect::Var value = v.getType()->getDefault())
            {
                HK_ASSERT_NO_MSG(0x4f55e3bc, value.getType()->getSizeOf() == v.getType()->getSizeOf());
                hkMemUtil::memCpy(v.getAddress(), value.getAddress(), v.getType()->getSizeOf());
            }
        }

        void visit(const hkReflect::StringVar& v)
        {
            if(hkReflect::Var dVal = v.getType()->getDefault())
            {
                // Some string types have NULL defaults
                v.assign(dVal);
            }
        }

        void visit(const hkReflect::PointerVar& r)
        {
        }

        void visit(const hkReflect::ArrayVar& h)
        {
            // Arrays are empty on construction and cannot have a default behaviour, unless they are a fixed size array
            HK_ASSERT_NO_MSG(0x6a2d5c7e, (!h.getCount()) || (h.getType()->getFixedCount() > 0));
            if(hkReflect::Var value = h.getType()->getDefault())
            {
                HK_ASSERT_NO_MSG(0x5a7ee8a7, value.getType()->getSizeOf() == h.getType()->getSizeOf());
                hkMemUtil::memCpy(h.getAddress(), value.getAddress(), h.getType()->getSizeOf());
            }
            else
            {
                for(int i = 0; i < h.getType()->getFixedCount(); ++i)
                {
                    dispatch(h[i]);
                }
            }
        }

        void visit(const hkReflect::RecordVar& rec)
        {
            for( hkReflect::DeclIter<hkReflect::DataFieldDecl> it(rec.getType()); it.advance(); )
            {
                dispatch( rec[it.current()] );
            }
        }
    };
}

void hkReflect::Var::writeDefault() const
{
    ApplyDefaultsVisitor().dispatch(*this);
}

int hkReflect::Var::compare(const Var& v) const
{
    if (this->isValid() && v.isValid())
    {
        return hkReflect::Detail::VarsCompareVisitor().dispatch(*this, v);
    }
    else
    {
        return v.isValid() - this->isValid();
    }
}

hkResult hkReflect::RecordVar::setValue(const hkReflect::RecordVar& var) const
{
    if( var.getType()->extendsOrEquals(getType()) )
    {
        return getImpl()->setValue( m_addr, getType(), var );
    }
    return HK_FAILURE;
}

namespace
{
    struct AssignVisitor : public hkReflect::VarVisitor<AssignVisitor, hkResult>
    {
        AssignVisitor(const hkReflect::Var& var) : m_var(var) {}
        const hkReflect::Var& m_var;

        template<typename T>
        hkResult assign(const T& other)
        {
            T var = m_var;
            HK_ASSERT_NO_MSG(0x77a5f187, var.isValid());
            return var.setValue(other.getValue());
        }

        HK_ALWAYS_INLINE hkResult visit(const hkReflect::VoidVar& t)    { return HK_FAILURE; }
        HK_ALWAYS_INLINE hkResult visit(const hkReflect::BoolVar& t) { return assign(t); }
        HK_ALWAYS_INLINE hkResult visit(const hkReflect::IntVar& t) { return assign(t); }
        HK_ALWAYS_INLINE hkResult visit(const hkReflect::FloatVar& t) { return assign(t); }
        HK_ALWAYS_INLINE hkResult visit(const hkReflect::StringVar& t) { return assign(t); }
        HK_ALWAYS_INLINE hkResult visit(const hkReflect::PointerVar& t) { return assign(t); }
        HK_ALWAYS_INLINE hkResult visit(const hkReflect::RecordVar& t) { return assign(t); }
        HK_ALWAYS_INLINE hkResult visit(const hkReflect::ArrayVar& t) { return assign(t); }
    };
}

hkResult hkReflect::Var::assign(const Var& rhs) const
{
    if (!isValid() || !rhs.isValid() || getType()->getKind() != rhs.getType()->getKind())
    {
        return HK_FAILURE;
    }

    // Try direct assignment.
    if (rhs.isInstanceOf(m_type) && !m_impl.isDynamic() && !m_type->hasCallbacks() && !rhs.getType()->isProperty())
    {
        if (TypeDetail::copyAssign(m_addr, rhs.getAddress(), m_type).isSuccess())
        {
            return HK_SUCCESS;
        }
    }

    // Try impl-based copy.
    return AssignVisitor(*this).dispatch(rhs);
}

hkReflect::Var hkReflect::Var::castToInterface(_In_ const hkReflect::Type* interfaceType) const
{
    // Check main inheritance branch.
    if (hkReflect::upCast(m_addr, m_type, interfaceType))
    {
        return *this;
    }

    // can't use the builtin optional logic since we want to find them all
    for (const Type* type = m_type; type != HK_NULL; type = type->getParent())
    {
        if(const Detail::InterfaceArray* ifacesPtr = TypeDetail::localGetOptional<Opt::INTERFACES>(type))
        {
            hkArrayView<const Detail::Interface> ifaces = ifacesPtr->items();
            for(int i = 0; i < ifaces.getSize(); ++i)
            {
                hkReflect::Var asCurrentIface(hkAddByteOffset(m_addr, ifaces[i].m_offset), ifaces[i].m_interfaceType);
                if (hkReflect::Var cast = asCurrentIface.castToInterface(interfaceType))
                {
                    return cast;
                }
            }
        }
    }

    return hkReflect::Var();
}

void hkReflect::Var::afterReflectNew()
{
    if ( getType() )
    {
        m_impl = getType()->getImpl();
    }
}

struct hkReflect::Var::VarImpl : public hkReflect::Detail::PointerImpl
{
    VarImpl() { /* keep clang happy */ }

    virtual hkResult setValue(_Inout_ void* self, _In_ const hkReflect::PointerType* type, const hkReflect::Var& var) const HK_OVERRIDE
    {
        hkReflect::Var& v = *static_cast<hkReflect::Var*>(self);
        v = var;
        return HK_SUCCESS;
    }

    virtual hkResult getValue(_In_ const void* self, _In_ const hkReflect::PointerType* type, _Out_ hkReflect::Var* val) const HK_OVERRIDE
    {
        *val = *static_cast<const hkReflect::Var*>(self);
        return HK_SUCCESS;
    }

    virtual hkResult inplaceFixup(_Inout_ void* self, _In_ const Type* selfType, _Inout_ void* target, _In_ const Type* targetType, int count) const HK_OVERRIDE
    {
        HK_ASSERT_NO_MSG(0x44c68836, count==1);
        hkReflect::Var& var = *static_cast<hkReflect::Var*>(self);
        var = hkReflect::Var(target, targetType);
        return HK_SUCCESS;
    }
};

const hkReflect::Var::VarImpl hkReflect::Var::s_impl;

void hkDebug::dump(const hkReflect::Var& obj)
{
    hkStringBuf buffer;
    obj.toString(buffer);
    buffer.append("\n");
    hkMemoryInitUtil::outputDebugString(buffer, HK_NULL);
}

void hkDebug::dump(_In_opt_ const hkReflect::Var* obj)
{
    if(obj)
    {
        dump(*obj);
    }
    else
    {
        hkMemoryInitUtil::outputDebugString("(null hkReflect::Var)\n", HK_NULL);
    }
}

hkResult hkReflect::ArrayVar::getValue(_Out_ hkReflect::ArrayValue* val) const
{
    return getImpl()->getValue(m_addr, getType(), val);
}

hkReflect::ArrayValue hkReflect::ArrayVar::getValue() const
{
    ArrayValue v;
    getValue(&v); //TODO.SECURITY
    return v;
}

int hkReflect::ArrayVar::getCount() const
{
    ArrayValue v;
    getValue(&v); //TODO.SECURITY
    return v.getCount();
}

hkReflect::QualType hkReflect::ArrayVar::getSubType() const
{
    ArrayValue v;
    getValue(&v); //TODO.SECURITY
    return v.getSubType();
}

hkReflect::Var hkReflect::ArrayVar::operator[](int i) const
{
    return getValue()[i];
}

hkResult hkReflect::ArrayVar::setValue(hkReflect::ArrayValue h) const
{
    return getImpl()->setValue(m_addr, getType(), h);
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
